// drD3D.dll
// Direct3D driver for the Doomsday Graphics Library
//
// list.c : List rendering

#include "drD3D.h"


// This is only for solid, non-masked primitives.
void renderList(rendlist_t *rl)
{
	int			i;
	float		tcleft, tcright, tctop, tcbottom;
	rendquad_t	*cq;
	vertex_t	vtx[4];

	if(!rl->numquads) return;	// The list is empty.

	// Bind the right texture.
	Bind(rl->tex);

	BeginScene();

	// Check what kind of primitives there are on the list.
	// There can only be one kind on each list.
	if(rl->quads->flags & RQF_FLAT)	// Check the first primitive.
	{
		// There's only triangles here, I see.				
		for(i=0; i<rl->numquads; i++)
		{
			cq = rl->quads+i;
			if(cq->flags & RQF_MISSING_WALL)
			{
				// This triangle is REALLY a quad that originally had no 
				// texture. We have to render it as two triangles.
				tcright = (tcleft=0) + cq->len/cq->texw;
				tcbottom = (tctop=0) + (cq->top - cq->bottom)/cq->texh;
				
				vtx[0].rgba = MAKERGB_A(cq->color[0].rgb, 0xff);
				vtx[0].s = tcleft;
				vtx[0].t = tctop;
				vtx[0].x = cq->v1[VX];
				vtx[0].y = cq->top;
				vtx[0].z = cq->v1[VY];

				vtx[1].rgba = MAKERGB_A(cq->color[1].rgb, 0xff);
				vtx[1].s = tcright;
				vtx[1].t = tctop;
				vtx[1].x = cq->v2[VX];
				vtx[1].y = cq->top;
				vtx[1].z = cq->v2[VY];

				vtx[2].rgba = vtx[1].rgba;
				vtx[2].s = tcright;
				vtx[2].t = tcbottom;
				vtx[2].x = cq->v2[VX];
				vtx[2].y = cq->bottom;
				vtx[2].z = cq->v2[VY];

				vtx[3].rgba = vtx[0].rgba;
				vtx[3].s = tcleft;
				vtx[3].t = tcbottom;
				vtx[3].x = cq->v1[VX];
				vtx[3].y = cq->bottom;
				vtx[3].z = cq->v1[VY];

				IDirect3DDevice3_DrawPrimitive(d3dDevice,
					D3DPT_TRIANGLEFAN, VERTEX_FORMAT,
					vtx, 4, D3DDP_DONOTLIGHT | D3DDP_DONOTUPDATEEXTENTS);

				continue;
			}

			// The vertices.
			vtx[0].rgba = MAKERGB_A(cq->color[0].rgb, 0xff);
			vtx[0].s = (cq->v1[VX]+cq->texoffx) / cq->texw;
			vtx[0].t = (cq->v1[VY]+cq->texoffy) / cq->texh;
			vtx[0].x = cq->v1[VX];
			vtx[0].y = cq->top;
			vtx[0].z = cq->v1[VY];

			vtx[1].rgba = MAKERGB_A(cq->color[1].rgb, 0xff);
			vtx[1].s = (cq->v2[VX]+cq->texoffx) / cq->texw;
			vtx[1].t = (cq->v2[VY]+cq->texoffy) / cq->texh;
			vtx[1].x = cq->v2[VX];
			vtx[1].y = cq->top;
			vtx[1].z = cq->v2[VY];

			vtx[2].rgba = MAKERGB_A(cq->color[2].rgb, 0xff);
			vtx[2].s = (cq->v3[VX]+cq->texoffx) / cq->texw;
			vtx[2].t = (cq->v3[VY]+cq->texoffy) / cq->texh;
			vtx[2].x = cq->v3[VX];
			vtx[2].y = cq->top;
			vtx[2].z = cq->v3[VY];

			IDirect3DDevice3_DrawPrimitive(d3dDevice,
				D3DPT_TRIANGLELIST, VERTEX_FORMAT,
				vtx, 3, D3DDP_DONOTLIGHT | D3DDP_DONOTUPDATEEXTENTS);
		}
	}
	else
	{
		// Render quads.
		for(i=0; i<rl->numquads; i++)
		{
			cq = rl->quads+i;

			// Calculate relative texture coordinates.
			tcright = (tcleft=cq->texoffx/(float)cq->texw) + cq->len/cq->texw;
			tcbottom = (tctop=cq->texoffy/cq->texh) + (cq->top - cq->bottom)/cq->texh;

			// The vertices.
			vtx[0].rgba = MAKERGB_A(cq->color[0].rgb, 0xff);
			vtx[0].s = tcleft;
			vtx[0].t = tctop;
			vtx[0].x = cq->v1[VX];
			vtx[0].y = cq->top;
			vtx[0].z = cq->v1[VY];

			vtx[1].rgba = MAKERGB_A(cq->color[1].rgb, 0xff);
			vtx[1].s = tcright;
			vtx[1].t = tctop;
			vtx[1].x = cq->v2[VX];
			vtx[1].y = cq->top;
			vtx[1].z = cq->v2[VY];

			vtx[2].rgba = vtx[1].rgba; 
			vtx[2].s = tcright;
			vtx[2].t = tcbottom;
			vtx[2].x = cq->v2[VX];
			vtx[2].y = cq->bottom;
			vtx[2].z = cq->v2[VY];

			vtx[3].rgba = vtx[0].rgba; 
			vtx[3].s = tcleft;
			vtx[3].t = tcbottom;
			vtx[3].x = cq->v1[VX];
			vtx[3].y = cq->bottom;
			vtx[3].z = cq->v1[VY];

			IDirect3DDevice3_DrawPrimitive(d3dDevice,
				D3DPT_TRIANGLEFAN, VERTEX_FORMAT,
				vtx, 4, D3DDP_DONOTLIGHT | D3DDP_DONOTUPDATEEXTENTS);
		}
	}
	EndScene();
}

// Masked lists only include quads.
void renderMaskedList(rendlist_t *mrl)
{
	int			i;
	float		tcleft, tcright, tctop, tcbottom;
	rendquad_t	*cq;
	vertex_t	vtx[4];
	
	if(!mrl->numquads) return;	// No quads to render, I'm afraid.

	// Curtex is used to keep track of the current texture.
	// Zero also denotes that no glBegin() has yet been called.
	currentTexName = 0;

	BeginScene();

	// Render quads.
	for(i=mrl->numquads-1; i>=0; i--)	// Render back to front.
	{
		cq = mrl->quads+i;

		// Calculate relative texture coordinates.
		tcright = (tcleft=cq->texoffx/cq->texw) + cq->len/cq->texw;
		tcbottom = (tctop=cq->texoffy/cq->texh) + (cq->top - cq->bottom)/cq->texh;

		// Is there a need to change the texture?
		if(currentTexName != cq->masktex)	
		{
			Bind(cq->masktex);
		}
		
		// The vertices.
		vtx[0].rgba = MAKERGB_A(cq->color[0].rgb, 0xff);
		vtx[0].s = tcleft;
		vtx[0].t = tctop;
		vtx[0].x = cq->v1[VX];
		vtx[0].y = cq->top;
		vtx[0].z = cq->v1[VY];

		vtx[1].rgba = MAKERGB_A(cq->color[1].rgb, 0xff);
		vtx[1].s = tcright;
		vtx[1].t = tctop;
		vtx[1].x = cq->v2[VX];
		vtx[1].y = cq->top;
		vtx[1].z = cq->v2[VY];

		vtx[2].rgba = vtx[1].rgba; 
		vtx[2].s = tcright;
		vtx[2].t = tcbottom;
		vtx[2].x = cq->v2[VX];
		vtx[2].y = cq->bottom;
		vtx[2].z = cq->v2[VY];

		vtx[3].rgba = vtx[0].rgba; 
		vtx[3].s = tcleft;
		vtx[3].t = tcbottom;
		vtx[3].x = cq->v1[VX];
		vtx[3].y = cq->bottom;
		vtx[3].z = cq->v1[VY];

		IDirect3DDevice3_DrawPrimitive(d3dDevice,
			D3DPT_TRIANGLEFAN, VERTEX_FORMAT,
			vtx, 4, D3DDP_DONOTLIGHT | D3DDP_DONOTUPDATEEXTENTS);
	}
	EndScene();
}

void renderSkyMaskLists(rendlist_t *smrl, rendlist_t *skyw)
{
	int			i;
	rendquad_t	*cq;
	vertex_t	vtx[4];

	if(!smrl->numquads && !skyw->numquads) return; // Nothing to do here.

	// Nothing gets written to the color buffer.
	// Instead of setting texture 0 we'll temporarily disable texturing.
	Disable(DGL_TEXTURING);
	// This will effectively disable color buffer writes.
	SetRS(D3DRENDERSTATE_SRCBLEND, D3DBLEND_ZERO);
	SetRS(D3DRENDERSTATE_DESTBLEND, D3DBLEND_ONE);

	for(i=0; i<4; i++) vtx[i].rgba = 0xffffffff;

	BeginScene();

	if(smrl->numquads)
	{
		for(i=0; i<smrl->numquads; i++)
		{
			cq = smrl->quads+i;
			// ONLY the vertices, please.
			vtx[0].x = cq->v1[VX];
			vtx[0].y = cq->top;
			vtx[0].z = cq->v1[VY];
			
			vtx[1].x = cq->v2[VX];
			vtx[1].y = cq->top;
			vtx[1].z = cq->v2[VY];

			vtx[2].x = cq->v3[VX];
			vtx[2].y = cq->top;
			vtx[2].z = cq->v3[VY];

			IDirect3DDevice3_DrawPrimitive(d3dDevice,
				D3DPT_TRIANGLELIST, VERTEX_FORMAT,
				vtx, 3, D3DDP_DONOTLIGHT | D3DDP_DONOTUPDATEEXTENTS);
		}
	}

	// Then the walls.
	if(skyw->numquads)
	{
		for(i=0; i<skyw->numquads; i++)
		{
			cq = skyw->quads + i;
			// Only the verts.
			vtx[0].x = cq->v1[VX];
			vtx[0].y = cq->bottom;
			vtx[0].z = cq->v1[VY];
			
			vtx[1].x = cq->v1[VX];
			vtx[1].y = cq->top;
			vtx[1].z = cq->v1[VY];

			vtx[2].x = cq->v2[VX];
			vtx[2].y = cq->top;
			vtx[2].z = cq->v2[VY];

			vtx[3].x = cq->v2[VX];
			vtx[3].y = cq->bottom;
			vtx[3].z = cq->v2[VY];

			IDirect3DDevice3_DrawPrimitive(d3dDevice,
				D3DPT_TRIANGLEFAN, VERTEX_FORMAT,
				vtx, 4, D3DDP_DONOTLIGHT | D3DDP_DONOTUPDATEEXTENTS);
		}
	}

	EndScene();

	SetRS(D3DRENDERSTATE_SRCBLEND, D3DBLEND_SRCALPHA);
	SetRS(D3DRENDERSTATE_DESTBLEND, D3DBLEND_INVSRCALPHA);
	Enable(DGL_TEXTURING);
}	

void renderDynLightLists(rendlist_t *frl, rendlist_t *wrl)
{
	int			i;
	rendquad_t	*cq;
	vertex_t	vtx[4];

	if(!frl->numquads && !wrl->numquads) return; // Nothing to do.

	// Use a Z bias.
	MatrixMode(DGL_PROJECTION);
	PushMatrix();
	Scalef(.99f, .99f, .99f);

	// Setup the correct rendering state.	
	// Disable fog.
	if(usefog) SetRS(D3DRENDERSTATE_FOGENABLE, FALSE);

	// This'll allow multiple light quads to be rendered on top of each other.
	SetRS(D3DRENDERSTATE_ZWRITEENABLE, FALSE);
	SetRS(D3DRENDERSTATE_ZFUNC, D3DCMP_LESSEQUAL);

	// Set up addition blending. The source is added to the destination.
	if(!dlBlend) // The normal setting.
	{
		SetRS(D3DRENDERSTATE_SRCBLEND, D3DBLEND_DESTCOLOR);
		SetRS(D3DRENDERSTATE_DESTBLEND, D3DBLEND_ONE);
	}
	else if(dlBlend == 1)
	{
		SetRS(D3DRENDERSTATE_SRCBLEND, D3DBLEND_ONE);
		SetRS(D3DRENDERSTATE_DESTBLEND, D3DBLEND_ONE);
	}

	// The light texture.
	Bind(lightTex);

	BeginScene();

	// The flats.
	if(frl->numquads)
	{
		for(i=0; i<frl->numquads; i++)
		{
			cq = frl->quads + i;
			// Set the color.
			vtx[0].rgba = vtx[1].rgba = vtx[2].rgba = MAKERGB_A(cq->color[0].rgb, 0xff);
						
			// The vertices.			
			vtx[0].s = (cq->texoffx - cq->v1[VX]) / cq->texw;
			vtx[0].t = (cq->texoffy - cq->v1[VY]) / cq->texh;
			vtx[0].x = cq->v1[VX];
			vtx[0].y = cq->top;
			vtx[0].z = cq->v1[VY];

			vtx[1].s = (cq->texoffx - cq->v2[VX]) / cq->texw;
			vtx[1].t = (cq->texoffy - cq->v2[VY]) / cq->texh;
			vtx[1].x = cq->v2[VX];
			vtx[1].y = cq->top;
			vtx[1].z = cq->v2[VY];

			vtx[2].s = (cq->texoffx - cq->v3[VX]) / cq->texw;
			vtx[2].t = (cq->texoffy - cq->v3[VY]) / cq->texh;
			vtx[2].x = cq->v3[VX];
			vtx[2].y = cq->top;
			vtx[2].z = cq->v3[VY];

			IDirect3DDevice3_DrawPrimitive(d3dDevice,
				D3DPT_TRIANGLELIST, VERTEX_FORMAT,
				vtx, 3, D3DDP_DONOTLIGHT | D3DDP_DONOTUPDATEEXTENTS);
		}
	}	

	// The walls.
	if(wrl->numquads)
	{
		float tctl[2], tcbr[2];	// Top left and bottom right.
		for(i=0; i<wrl->numquads; i++)
		{
			cq = wrl->quads + i;
			// Set the color.
			vtx[0].rgba = vtx[1].rgba = vtx[2].rgba = vtx[3].rgba = 
				MAKERGB_A(cq->color[0].rgb, 0xff);

			// Calculate the texture coordinates.
			tcbr[VX] = (tctl[VX]=-cq->texoffx/cq->texw) + cq->len/cq->texw;
			tcbr[VY] = (tctl[VY]=cq->texoffy/cq->texh) + (cq->top - cq->bottom)/cq->texh;

			// The vertices.
			vtx[0].s = tctl[VX];
			vtx[0].t = tcbr[VY];
			vtx[0].x = cq->v1[VX];
			vtx[0].y = cq->bottom;
			vtx[0].z = cq->v1[VY];

			vtx[1].s = tctl[VX];
			vtx[1].t = tctl[VY];
			vtx[1].x = cq->v1[VX];
			vtx[1].y = cq->top;
			vtx[1].z = cq->v1[VY];

			vtx[2].s = tcbr[VX];
			vtx[2].t = tctl[VY];
			vtx[2].x = cq->v2[VX];
			vtx[2].y = cq->top;
			vtx[2].z = cq->v2[VY];

			vtx[3].s = tcbr[VX];
			vtx[3].t = tcbr[VY];
			vtx[3].x = cq->v2[VX];
			vtx[3].y = cq->bottom;
			vtx[3].z = cq->v2[VY];

			IDirect3DDevice3_DrawPrimitive(d3dDevice,
				D3DPT_TRIANGLEFAN, VERTEX_FORMAT,
				vtx, 4, D3DDP_DONOTLIGHT | D3DDP_DONOTUPDATEEXTENTS);
		}
	}

	EndScene();

	// Restore the normal rendering state.
	if(usefog) SetRS(D3DRENDERSTATE_FOGENABLE, TRUE);
	SetRS(D3DRENDERSTATE_ZWRITEENABLE, TRUE);
	SetRS(D3DRENDERSTATE_ZFUNC, D3DCMP_LESS);
	SetRS(D3DRENDERSTATE_SRCBLEND, D3DBLEND_SRCALPHA);
	SetRS(D3DRENDERSTATE_DESTBLEND, D3DBLEND_INVSRCALPHA);

	PopMatrix();
}


void RenderList(int format, void *data)
{
	rendlist_t *rl = data;
	rendlist_t **many = data;

	switch(format)
	{
	case DGL_NORMAL_LIST:
		renderList(rl);
		break;

	case DGL_MASKED_LIST:
		renderMaskedList(rl);
		break;

	case DGL_SKYMASK_LISTS:
		renderSkyMaskLists(many[0], many[1]);
		break;

	case DGL_LIGHT_LISTS:
		renderDynLightLists(many[0], many[1]);
		break;
	}		
}



