
//**************************************************************************
//**
//** OGL_RL.C
//**
//**************************************************************************

// HEADER FILES ------------------------------------------------------------

#include <stdlib.h>
#include "dd_def.h"
#include "i_win32.h"
#include "gl_def.h"
#include "gl_rl.h"

// MACROS ------------------------------------------------------------------

#define GREATER(x,y) ((x)>(y) ? (x) : (y))

// TYPES -------------------------------------------------------------------

// EXTERNAL FUNCTION PROTOTYPES --------------------------------------------

// PUBLIC FUNCTION PROTOTYPES ----------------------------------------------

// PRIVATE FUNCTION PROTOTYPES ---------------------------------------------

// EXTERNAL DATA DECLARATIONS ----------------------------------------------

extern int			skyhemispheres;
extern int			dlMaxRad; // Dynamic lights maximum radius.
extern int			useDynLights, dlBlend, simpleSky;
extern boolean		whitefog;

// PUBLIC DATA DEFINITIONS -------------------------------------------------

int numrlists=0;			// Number of rendering lists.
boolean				renderTextures = true;

// PRIVATE DATA DEFINITIONS ------------------------------------------------

static rendlist_t *rlists=0;	// The list of rendering lists.
static rendlist_t masked_rlist;	// Rendering list for masked textures.
static rendlist_t invsky_rlist;	// List for invisible sky triangles.
static rendlist_t invskywall_rlist; // List for invisible sky walls (w/skyfix).
static rendlist_t dlwall_rlist, dlflat_rlist; // Dynamic lighting for walls/flats.

// CODE --------------------------------------------------------------------

void RL_Init()
{
	numrlists = 0;
	rlists = 0;
	memset(&masked_rlist, 0, sizeof(masked_rlist));
	memset(&invsky_rlist, 0, sizeof(invsky_rlist));
	memset(&invskywall_rlist, 0, sizeof(invskywall_rlist));
	memset(&dlwall_rlist, 0, sizeof(dlwall_rlist));
	memset(&dlflat_rlist, 0, sizeof(dlflat_rlist));
}

void RL_DestroyList(rendlist_t *rl)
{
	// All the list data will be destroyed.
	free(rl->quads);
	memset(rl, 0, sizeof(rendlist_t));
}

void RL_DeleteLists()
{
	int		i;

	// Delete all lists.
	for(i=0; i<numrlists; i++) RL_DestroyList(rlists+i);
	RL_DestroyList(&masked_rlist);
	RL_DestroyList(&invsky_rlist);
	RL_DestroyList(&invskywall_rlist);
	RL_DestroyList(&dlwall_rlist);
	RL_DestroyList(&dlflat_rlist);

	free(rlists);
	rlists = 0;
	numrlists = 0;
}

// Reset the indices.
void RL_ClearLists()
{
	int		i;
	for(i=0; i<numrlists; i++) rlists[i].numquads = 0;
	masked_rlist.numquads = 0;
	invsky_rlist.numquads = 0;
	invskywall_rlist.numquads = 0;
	dlwall_rlist.numquads = 0;
	dlflat_rlist.numquads = 0;
	skyhemispheres = 0;
}

// This is only called for flat rendquads w/dynlights in use.
static void RL_DynLightQuad(rendquad_t *quad, lumobj_t *lum)
{
	quad->texw = lum->radius*2;
	quad->texh = lum->radius*2;

	// The texture offset is what actually determines where the
	// dynamic light map will be rendered. For light quads the
	// texture offset is global.
	quad->texoffx = FIX2FLT(lum->thing->x) + lum->radius;
	quad->texoffy = FIX2FLT(lum->thing->y) + lum->radius;
}

static rendlist_t *RL_FindList(DGLuint tex)
{
	int	i;
	rendlist_t *dest;

	for(i=0; i<numrlists; i++)
		if(rlists[i].tex == tex)
			return rlists+i;

	// Then create a new list.
	rlists = (rendlist_t*)realloc(rlists, sizeof(rendlist_t) * ++numrlists);
	dest = rlists + numrlists-1;
	memset(dest, 0, sizeof(rendlist_t));
	dest->tex = tex;
	return dest;
}

// Here we will add the quad to the correct rendering list, creating 
// a new list if necessary.
void RL_AddQuad(rendquad_t *quad, DGLuint quadtex)
{
	rendlist_t	*dest = 0;	// The destination list.
	rendquad_t	*dq;		// Quad in the dest list.
	
	// Masked quads go to the masked list.
	if(quad->flags & RQF_MASKED) 
		dest = &masked_rlist;
	else if(quad->flags & RQF_SKY_MASK_WALL)
		dest = &invskywall_rlist;
	else if(quad->flags & RQF_LIGHT) // Dynamic lights?
		dest = &dlwall_rlist;
	else
	{
		// Find a suitable list. This can get a bit redundant for large
		// numbers of primitives of the same texture (oh yeah, this is 
		// a real cycle-eater).
		dest = RL_FindList(quadtex);
	}
	// Now we have a destination list. This is the only place where
	// quads are added. 
	if(++dest->numquads > dest->listsize)		// See if we have to allocate more memory.
	{
		dest->listsize = dest->numquads+10;		
		dest->quads = (rendquad_t*)realloc(dest->quads, 
			sizeof(rendquad_t) * dest->listsize);
	}
	dq = dest->quads + dest->numquads-1;
	memcpy(dq, quad, sizeof(rendquad_t));
	
	// Let a masked quad know its texture.
	if(quad->flags & RQF_MASKED) 
	{
		dq->masktex = quadtex;
		if(!quadtex) I_Error("RL_AddQuad: There can't be a masked quad with no texture.\n");
	}
}

// Uh, this is a pretty ugly hack.
subsector_t *currentssec;

// Adds a series of flat quads (triangles) as a fan.
void RL_AddFlatQuads(rendquad_t *base, DGLuint quadtex, int numvrts, 
					 fvertex_t *origvrts, int dir)
{
	fvertex_t	*vtx;
	rendlist_t	*dest;
	rendquad_t	*qi;
	int			i, firstquad;
	float		distances[MAX_CC_SIDES], middist;
	fvertex_t	vrts[MAX_CC_SIDES];
	int			lightlevel = base->color[0].rgb[0];

	if(!numvrts) return;	// No data, can't do anything.

	if(base->flags & RQF_SKY_MASK)
		dest = &invsky_rlist;
	else if(base->flags & RQF_LIGHT)
		dest = &dlflat_rlist;
	else
		// First find the right list.
		dest = RL_FindList(quadtex);
	
	// Check that there's enough room.
	firstquad = dest->numquads;
	dest->numquads += numvrts-2;
	if(dest->numquads > dest->listsize)
	{
		// Allocate more memory.
		dest->listsize = dest->numquads + 20;
		dest->quads = (rendquad_t*)realloc(dest->quads, 
			sizeof(rendquad_t) * dest->listsize);	
	}

	// Calculate the distance to each vertex.
	if(!(base->flags & RQF_LIGHT)) // Lights don't need distance info.
		for(i=0; i<numvrts; i++) distances[i] = PointDist2D(&origvrts[i].x);
	
	// Make a distance modification. This is a bit of a hack...
	// But a useful one, it hides many of the tiny gaps between polygons.
	memcpy(vrts, origvrts, sizeof(fvertex_t)*numvrts);	
	
	if(!(base->flags & RQF_LIGHT))
	{
		middist = PointDist2D(&currentssec->midpoint.x);
		if(middist > 256)
			for(i=0; i<numvrts; i++)
			{
				float mod = (middist-256)/128;
				if(mod > 5) mod = 5;
				vrts[i].x += mod * currentssec->diffverts[i].x;
				vrts[i].y += mod * currentssec->diffverts[i].y;
			}
	}

	// Add them as a fan.
	if(dir == 0)	// Normal direction.
	{
		// All triangles share the first vertex.
		base->v1[VX] = vrts->x;	
		base->v1[VY] = vrts->y;
		base->dist[0] = distances[0];

		for(i=2, qi=dest->quads+firstquad; i<numvrts; i++, qi++)
		{
			memcpy(qi, base, sizeof(rendquad_t));
			// The second vertex is the previous from this one.
			vtx = vrts+i-1;
			qi->v2[VX] = vtx->x;
			qi->v2[VY] = vtx->y;
			qi->dist[1] = distances[i-1];
			// The third vertex is naturally the current one.
			vtx = vrts+i;
			qi->v3[VX] = vtx->x;
			qi->v3[VY] = vtx->y;
			qi->dist[2] = distances[i];
			
			if(base->flags & RQF_LIGHT) 
				RL_DynLightQuad(qi, (lumobj_t*)quadtex);
			else
				VertexColors(lightlevel, qi, 3);
		}
	}
	else	// Reverse direction?
	{
		// All triangles share the last vertex.
		vtx = vrts + numvrts-1;
		base->v1[VX] = vtx->x;
		base->v1[VY] = vtx->y;
		base->dist[0] = distances[numvrts-1];

		for(i=numvrts-3, qi=dest->quads+firstquad; i>=0; i--, qi++)
		{
			memcpy(qi, base, sizeof(rendquad_t));
			// The second vertex is the next from this one.
			vtx = vrts+i+1;
			qi->v2[VX] = vtx->x;
			qi->v2[VY] = vtx->y;
			qi->dist[1] = distances[i+1];
			// The third vertex is naturally the current one.
			vtx = vrts+i;
			qi->v3[VX] = vtx->x;
			qi->v3[VY] = vtx->y;
			qi->dist[2] = distances[i];

			if(base->flags & RQF_LIGHT) 
				RL_DynLightQuad(qi, (lumobj_t*)quadtex);
			else
				VertexColors(lightlevel, qi, 3);
		}
	}	
}

void RL_RenderAllLists()
{
	int			i;
	rendlist_t	*lists[2];

	if(!renderTextures) gl.Disable(DGL_TEXTURING);

	// The sky might be visible. Render the needed hemispheres.
	R_RenderSky(skyhemispheres);				

	lists[0] = &invsky_rlist;
	lists[1] = &invskywall_rlist;
	gl.RenderList(DGL_SKYMASK_LISTS, lists);

	if(!renderTextures) gl.Disable(DGL_TEXTURING);

	for(i=0; i<numrlists; i++) 
		gl.RenderList(DGL_NORMAL_LIST, rlists+i);

	if(dlBlend != 3)
	{
		// Update the light blending mode.
		gl.SetInteger(DGL_DL_BLEND_MODE, dlBlend);

		lists[0] = &dlflat_rlist;
		lists[1] = &dlwall_rlist;
		gl.RenderList(DGL_LIGHT_LISTS, lists);	
	}

	gl.RenderList(DGL_MASKED_LIST, &masked_rlist);

	if(!renderTextures) gl.Enable(DGL_TEXTURING);
}

