
//**************************************************************************
//**
//** I_MUS.C
//**
//** Version:		1.0
//** Last Build:	-?-
//** Author:		jk
//**
//** Playing MUS songs using Windows MIDI output.
//**
//**************************************************************************

// HEADER FILES ------------------------------------------------------------

#define WIN32_LEAN_AND_MEAN
#include <stdio.h>
#include <windows.h>
#include <mmsystem.h>
#include <malloc.h>
#include <math.h>
#include "st_start.h"
#include "i_mus.h"

// MACROS ------------------------------------------------------------------

// TYPES -------------------------------------------------------------------

typedef	unsigned short int UWORD;

typedef struct {
	char    ID[4];          // identifier "MUS" 0x1A
    UWORD	scoreLen;
    UWORD	scoreStart;
	UWORD	channels;		// number of primary channels
	UWORD	sec_channels;	// number of secondary channels
	UWORD	instrCnt;
	UWORD	dummy;
	// The instrument list begins here.
} MUSHeader_t;

typedef struct {
	unsigned char	channel : 4;
	unsigned char	event : 3;
	unsigned char	last : 1;
} MUSEventDesc_t;

enum // MUS event types.
{
	MUS_EV_RELEASE_NOTE,
	MUS_EV_PLAY_NOTE,
	MUS_EV_PITCH_WHEEL,
	MUS_EV_SYSTEM,			// Valueless controller.
	MUS_EV_CONTROLLER,		
	MUS_EV_FIVE,			// ?
	MUS_EV_SCORE_END,
	MUS_EV_SEVEN			// ?
};

enum // MUS controllers.
{
	MUS_CTRL_INSTRUMENT,
	MUS_CTRL_BANK,
	MUS_CTRL_MODULATION,
	MUS_CTRL_VOLUME,
	MUS_CTRL_PAN,
	MUS_CTRL_EXPRESSION,
	MUS_CTRL_REVERB,
	MUS_CTRL_CHORUS,
	MUS_CTRL_SUSTAIN_PEDAL,
	MUS_CTRL_SOFT_PEDAL,
	// The valueless controllers.
	MUS_CTRL_SOUNDS_OFF,
	MUS_CTRL_NOTES_OFF,
	MUS_CTRL_MONO,
	MUS_CTRL_POLY,
	MUS_CTRL_RESET_ALL,
	NUM_MUS_CTRLS
};

// EXTERNAL FUNCTION PROTOTYPES --------------------------------------------

void I_Error (char *error, ...);

// PUBLIC FUNCTION PROTOTYPES ----------------------------------------------

// PRIVATE FUNCTION PROTOTYPES ---------------------------------------------

// EXTERNAL DATA DECLARATIONS ----------------------------------------------

// PUBLIC DATA DEFINITIONS -------------------------------------------------

// PRIVATE DATA DEFINITIONS ------------------------------------------------

static int midiAvail = 0, disabled = 0;
static HMIDIOUT midiOut;
static int origVol;				// The original MIDI volume.
static MUSHeader_t *song = 0;	// The song;
static unsigned char *playPos;	// The current play position.
static int playing = 0, looping = 0;	// The song is playing/looping.
static int waitTicks = 0;		
static unsigned char chanVols[16];	// Last volume for each channel.

static char ctrl_mus2midi[NUM_MUS_CTRLS] =
{
	0,		// Not used.
	0,		// Bank select.
	1,		// Modulation.
	7,		// Volume.
	10,		// Pan.
	11,		// Expression.
	91,		// Reverb.
	93,		// Chorus.
	64,		// Sustain pedal.
	67,		// Soft pedal.
	// The valueless controllers.
	120,	// All sounds off.
	123,	// All notes off.
	126,	// Mono.
	127,	// Poly.
	121		// Reset all controllers.
};

// CODE --------------------------------------------------------------------

/*static void MUS_CheckResult(MMRESULT mmres,char *inWhichFunc)
{
	if(mmres == MMSYSERR_NOERROR) return;	// No error.

	I_Error("Bad MIDI result: %s returns %d (0x%x).\n",inWhichFunc,mmres,mmres);
}*/

//==========================================================================
//
// MUS_SongStartAddress
//
//==========================================================================

static unsigned char *MUS_SongStartAddress()
{
	if(!song) return 0;
	return (unsigned char*)song + song->scoreStart;
}

//==========================================================================
//
// MUS_PauseSong
//
//==========================================================================

void MUS_PauseSong()
{
	playing = 0;
	// Stop all the notes.
	midiOutReset(midiOut);
}

//==========================================================================
//
// MUS_ResumeSong
//
//==========================================================================

void MUS_ResumeSong()
{
	playing = 1;
}

//==========================================================================
//
// MUS_SetMasterVolume
//
// Vol is from 0 to 255.
//
//==========================================================================

void MUS_SetMasterVolume(int vol)
{
	// Clamp it to a byte.
	if(vol < 0) vol = 0;
	if(vol > 255) vol = 255;
	// Straighen the volume curve.
	vol <<= 8;	// Make it a word.
	vol = (int) (255.9980469 * sqrt(vol));
	midiOutSetVolume(midiOut,vol+(vol<<16));	// Expand to a dword.
}

//==========================================================================
//
// MUS_SongPlayer
//
// Called at 140 Hz, interprets the MUS data to MIDI events.
//
//==========================================================================

void MUS_SongPlayer()
{
	MUSEventDesc_t	*evDesc;
	unsigned char	midiStatus,midiChan,midiParm1,midiParm2;
	int				scoreEnd = 0;

	// Get immediately out of here if we have nothing to do.
	if(disabled || !midiAvail || !song || !playing) return;	

	// This is the wait counter.
	if(--waitTicks > 0) return;
	
	// We assume playPos is OK. We'll keep playing events until the 
	// 'last' bit is set, and then the waitTicks is set.
	for(;;)	// Infinite loop in an interrupt handler. Good call.
	{
		evDesc = (MUSEventDesc_t*)playPos++;
		midiStatus = midiChan = midiParm1 = midiParm2 = 0;
		// Construct the MIDI event.
		switch(evDesc->event)
		{
		case MUS_EV_RELEASE_NOTE:
			midiStatus = 0x80;
			// Which note?
			midiParm1 = *playPos++;
			break;

		case MUS_EV_PLAY_NOTE:
			midiStatus = 0x90;
			// Which note?
			midiParm1 = *playPos++;
			// Is the volume there, too?
			if(midiParm1 & 0x80)	
				chanVols[evDesc->channel] = *playPos++;
			midiParm1 &= 0x7f;
			midiParm2 = chanVols[evDesc->channel];
			break;				

		case MUS_EV_CONTROLLER:
			midiStatus = 0xb0;
			midiParm1 = *playPos++;
			midiParm2 = *playPos++;
			// The instrument control is mapped to another kind of MIDI event.
			if(midiParm1 == MUS_CTRL_INSTRUMENT)
			{
				midiStatus = 0xc0;
				midiParm1 = midiParm2;
				midiParm2 = 0;
			}
			else
			{
				// Use the conversion table.
				midiParm1 = ctrl_mus2midi[midiParm1];
			}
			break;

		case MUS_EV_PITCH_WHEEL:
			midiStatus = 0xe0;
			midiParm1 = *playPos++;
			//printf( "pitch wheel: ch %d, %d\n", evDesc->channel, midiParm1);
			break;

		case MUS_EV_SYSTEM:
			midiStatus = 0xb0;
			midiParm1 = ctrl_mus2midi[*playPos++];
			break;

		case MUS_EV_SCORE_END:
			// We're done.
			scoreEnd = 1;
			break;

		default:
			I_Error("MUS_SongPlayer: Unhandled MUS event %d.\n",evDesc->event);
		}
		if(scoreEnd) break;
		// Choose the channel.
		midiChan = evDesc->channel;
		// Redirect MUS channel 16 to MIDI channel 10 (drums).
		if(midiChan == 15) midiChan = 9; else if(midiChan == 9) midiChan = 15;
		//ST_Message("MIDI event/%d: %x %d %d\n",evDesc->channel,midiStatus,midiParm1,midiParm2);

		// Send out the MIDI event.
		midiOutShortMsg(midiOut, midiChan | midiStatus | (midiParm1<<8) | (midiParm2<<16));
		
		// Check if this was the last event in a group.
		if(evDesc->last) break;
	}
	waitTicks = 0;
	// Check for end of score.
	if(scoreEnd)
	{
		playPos = MUS_SongStartAddress();
		if(!looping) playing = 0;
		// Reset the MIDI output so no notes are left playing when 
		// the song ends.
		midiOutReset(midiOut);
	}
	else
	{
		// Read the number of ticks to wait.
		for(;;)
		{
			midiParm1 = *playPos++;
			waitTicks = waitTicks*128 + (midiParm1 & 0x7f);
			if(!(midiParm1 & 0x80)) break;
		}
	}
}

//==========================================================================
//
// MUS_RegisterSong
//
//==========================================================================

int MUS_RegisterSong(void *data, int length)
{
	if(!midiAvail) return 0;

	// First unregister the previous song.
	if(song) MUS_UnregisterSong();	

	song = malloc(length);
	memcpy(song, data, length);
	playPos = MUS_SongStartAddress();	// Go to the beginning of the song.
	playing = 0;	// We aren't playing this song.
	return 1;
}

//==========================================================================
//
// MUS_UnregisterSong
//
//==========================================================================

void MUS_UnregisterSong(void)
{
	if(!song || !midiAvail) return;
/*	MUS_CheckResult(midiOutUnprepareHeader((HMIDIOUT)midiStrm,&midiHdr,sizeof(MIDIHDR)),
		"midiOutUnprepareHeader");
	// Reset the data.
	midiHdr.dwBytesRecorded = 0;*/
	
	// First stop the song.
	MUS_StopSong();
	// This is the actual unregistration.
	free(song);
	song = 0;
}

//==========================================================================
//
// MUS_IsSongPlaying
//
//==========================================================================

int MUS_IsSongPlaying()
{
	return playing;
}

//==========================================================================
//
// MUS_Reset
//
//==========================================================================

void MUS_Reset()
{
	int		i;

	// Reset channel settings.
	for(i=0; i<=0xf; i++) // All channels.
		midiOutShortMsg(midiOut, 0xe0+i); // Pitch bend.
	
	midiOutReset(midiOut);	
}

//==========================================================================
//
// MUS_StopSong
//
//==========================================================================

void MUS_StopSong()
{
//	MUS_CheckResult(midiStreamStop(midiStrm),"strmStop");
	if(!midiAvail) return;
	playing = 0;
	playPos = MUS_SongStartAddress();
	MUS_Reset();
}

//==========================================================================
//
// MUS_PlaySong
//
//==========================================================================

void MUS_PlaySong(int loopy)
{
/*	ST_Message("MUS_PlaySong: Starting song (looping %d). %d bytes.\n",looping,midiHdr.dwBytesRecorded);
	MUS_CheckResult(midiStreamOut(midiStrm,&midiHdr,sizeof(midiHdr)),"strmOut");
	MUS_CheckResult(midiStreamRestart(midiStrm),"strmRestart");*/
	playing = 1;
	playPos = MUS_SongStartAddress();
	looping = loopy;
}

//==========================================================================
//
// MUS_Init
//
// Returns 0 if no problems.
//
//==========================================================================

int MUS_Init()
{
	MMRESULT mmres;

	if(midiAvail) return 0;	// Already initialized.

	//ST_Message("MUS_Init: %d MIDI out devices present.\n",midiOutGetNumDevs());
	// Open the midi stream.
	if((mmres = midiOutOpen(&midiOut,MIDI_MAPPER,0,0,CALLBACK_NULL)) != MMSYSERR_NOERROR)
	{
		ST_Message("MUS_Init: midiOutOpen: (code %d).\n",mmres);
		return -1;
	}
	// Now the MIDI is available.
	ST_Message("MUS_Init: MIDI available and OK.\n");
	// Get the original MIDI volume (restored in shutdown).
	midiOutGetVolume(midiOut, &origVol);
	midiAvail = 1;
	disabled = 0;
	song = 0;
	playPos = 0;
	playing = 0;
	return 0;
}

//==========================================================================
//
// MUS_Shutdown
//
//==========================================================================

void MUS_Shutdown()
{
	if(!midiAvail) return;

	// Restore the original volume.
	midiOutSetVolume(midiOut, origVol);
	midiOutClose(midiOut);
	// If there is a registered song, unregister it.
/*	if(midiHdr.dwBytesRecorded) MUS_UnregisterSong();
	// Free the event buffer.
	free(midiHdr.lpData);*/
	MUS_UnregisterSong();
	midiAvail = 0;
	song = 0;
	playPos = 0;
	playing = 0;
}

void MUS_Disable(int really)
{
	// Make sure no notes are left playing.
	// Only if disabled is true.
	if((disabled=really)) midiOutReset(midiOut);
}