//**************************************************************************
//**
//** GL_DRAW.C
//**
//** Version:		1.0
//** Last Build:	-?-
//** Author:		jk
//**
//** Rendering lists and other rendering.
//**
//**************************************************************************

// HEADER FILES ------------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "dd_def.h"
#include "m_bams.h"
#include "gl_def.h"
#include "gl_rl.h"
#include "gl_dyn.h"
#include "console.h"
#include "settings.h"

// MACROS ------------------------------------------------------------------

// TYPES -------------------------------------------------------------------

// EXTERNAL FUNCTION PROTOTYPES --------------------------------------------

// PUBLIC FUNCTION PROTOTYPES ----------------------------------------------

// PRIVATE FUNCTION PROTOTYPES ---------------------------------------------

void GL_ProjectionMatrix();
void RL_DynLightQuad(rendquad_t *quad, lumobj_t *lum);

// EXTERNAL DATA DECLARATIONS ----------------------------------------------

extern int			useDynLights, translucentIceCorpse;
//extern fadeout_t	fadeOut[2];		// For both skies.
extern int			skyhemispheres, haloMode;
extern int			dlMaxRad;

// PUBLIC DATA DEFINITIONS -------------------------------------------------

boolean		whitefog = false;		// Is the white fog in use?
float		fieldOfView = 90.0f;
float		maxLightDist = 1024;

float		vx, vy, vz, vang, vpitch;

boolean		willRenderSprites = true, freezeRLs = false;
int			missileBlend;
int			litSprites = 1;
int			r_ambient = 0;

float		nearClip, farClip;

int viewpw, viewph;	// Viewport size, in pixels.
int viewpx, viewpy; // Viewpoint top left corner, in pixels.

float		maxSpriteAngle = 60;

// PRIVATE DATA DEFINITIONS ------------------------------------------------

static float yfov;
static float viewsidex, viewsidey;	// For the black fog.

static boolean firstsubsector;		// No range checking for the first one.

// CODE --------------------------------------------------------------------

// How far the point is from the viewside plane?
float PointDist2D(float c[2])
{
/*          (YA-YC)(XB-XA)-(XA-XC)(YB-YA)
        s = -----------------------------
                        L**2
	Luckily, L**2 is one. dist = s*L. Even more luckily, L is also one.
*/
	float dist = (vz-c[VY])*viewsidex - (vx-c[VX])*viewsidey;
	if(dist < 0) return -dist;	// Always return positive.
	return dist;
}

// ---------------------------------------------------

void GL_InitData()
{
	GL_TexInit();		// OpenGL texture manager.
	bamsInit();			// Binary angle calculations.
	C_Init();			// Clipper.
	RL_Init();			// Rendering lists.
	R_SkyInit();		// The sky.
}

void GL_ResetData(void)	// Called before starting a new level.
{
	GL_TexReset();		// Textures are deleted (at least skies need this???).
	RL_DeleteLists();	// The rendering lists are destroyed.

	// We'll delete the sky textures. New ones will be generated for each level.
	//glDeleteTextures(2, skynames);
	//skynames[0] = skynames[1] = 0;
	
	// Ready for new fadeout colors.
//	fadeOut[0].set = fadeOut[1].set = 0;

	DL_Clear();
	H_Clear();
}

void GL_InitRenderer()	// Initializes the renderer to 2D state.
{
	//GLfloat fogcol[4] = { .7f, .7f, .7f, 1 };
	//GLfloat fogcol[4] = { .54f, .54f, .54f, 1 };
	byte fogcol[4] = { 138, 138, 138, 1 };

	// The variables.
	nearClip = 5;
	farClip = 16500;//8000;	

	// Here we configure the OpenGL state and set the projection matrix.
	gl.Disable(DGL_CULL_FACE);
	gl.Disable(DGL_DEPTH_TEST);
	gl.Enable(DGL_TEXTURING);

	// The projection matrix.
	gl.MatrixMode(DGL_PROJECTION);
	gl.LoadIdentity();
	//gluOrtho2D(0, 320, 200, 0);
	gl.Ortho(0, 0, 320, 200, -1, 1);

	// Default state for the white fog is off.
	whitefog = false;
	gl.Disable(DGL_FOG);
	gl.Fog(DGL_FOG_MODE, DGL_LINEAR);
	gl.Fog(DGL_FOG_END, 2100);	// This should be tweaked a bit.
	gl.Fogv(DGL_FOG_COLOR, fogcol);

	/*glEnable(GL_POLYGON_SMOOTH);
	glHint(GL_POLYGON_SMOOTH_HINT, GL_NICEST);*/
}

void GL_UseWhiteFog(int yes)
{
	if(!whitefog && yes)
	{
		// White fog is turned on.
		whitefog = true;
		gl.Enable(DGL_FOG);
	}
	else if(whitefog && !yes)
	{
		// White fog must be turned off.
		whitefog = false;
		gl.Disable(DGL_FOG);
	}
	// Otherwise we won't do a thing.
}

void GL_SwitchTo3DState()
{
	// Push the 2D state on the stack.
//	glPushAttrib(/*GL_VIEWPORT_BIT | */GL_ENABLE_BIT);
	gl.MatrixMode(DGL_PROJECTION);
	gl.PushMatrix();
	gl.MatrixMode(DGL_MODELVIEW);
	gl.PushMatrix();

	// Enable some... things.
	gl.Enable(DGL_CULL_FACE);
	gl.Enable(DGL_DEPTH_TEST);

	viewpx = viewwindowx * screenWidth/320,
	viewpy = viewwindowy * screenHeight/200;
	// Set the viewport.
	if(viewheight != SCREENHEIGHT)
	{
		viewpw = viewwidth * screenWidth/320;
		viewph = viewheight * screenHeight/200 + 1;
		//glViewport(viewpx, screenHeight-viewpy-viewph, viewpw, viewph);
		gl.Viewport(viewpx, viewpy, viewpw, viewph);
	}
	else
	{
		viewpw = screenWidth;
		viewph = screenHeight;
	}

	// The 3D projection matrix.
	GL_ProjectionMatrix();
}

void GL_Restore2DState(int step)
{
	if(step == 1)
	{
//		extern int setblocks;
		gl.MatrixMode(DGL_PROJECTION);
		gl.LoadIdentity();
		//gluOrtho2D(0, 320, (setblocks<11)?(SCREENHEIGHT-SBARHEIGHT*sbarscale/20):200, 0);
		//gluOrtho2D(0, 320, (320*viewheight)/viewwidth, 0);
		gl.Ortho(0, 0, 320, (320*viewheight)/viewwidth, -1, 1);
		gl.MatrixMode(DGL_MODELVIEW);
		gl.LoadIdentity();
	}
	if(step == 2)
	{
		gl.Viewport(0, 0, screenWidth, screenHeight);
	}
	if(step == 3)
	{
		gl.MatrixMode(DGL_PROJECTION);
		gl.PopMatrix();
		gl.MatrixMode(DGL_MODELVIEW);
		gl.PopMatrix();
				
		gl.Disable(DGL_CULL_FACE);
		gl.Disable(DGL_DEPTH_TEST);
	}
}

static void GL_ProjectionMatrix()
{
	// We're assuming pixels are squares... well, they are nowadays.
	float aspect = viewpw/(float)viewph;

	gl.MatrixMode(DGL_PROJECTION);
	gl.LoadIdentity();
	gl.Perspective(yfov=fieldOfView/aspect, aspect, nearClip, farClip);
	
	// We'd like to have a left-handed coordinate system.
	gl.Scalef(1,1,-1);
}

static void GL_ModelViewMatrix()
{
	vx = FIX2FLT(viewx);
	vy = FIX2FLT(viewz);
	vz = FIX2FLT(viewy);
	vang = viewangle / (float)ANGLE_MAX * 360 - 90;

	gl.MatrixMode(DGL_MODELVIEW);
	gl.LoadIdentity();
	gl.Rotatef(vpitch=viewpitch*85.0/110.0, 1, 0, 0);
	gl.Rotatef(vang, 0, 1, 0);
	gl.Scalef(1, 1.2f, 1);	// This is the aspect correction.
	gl.Translatef(-vx,-vy,-vz);
}


static int SegFacingDir(float v1[2], float v2[2])
{
	float nx = v1[VY]-v2[VY], ny = v2[VX]-v1[VX];
	float vvx = v1[VX]-vx, vvy = v1[VY]-vz;

	// The dot product.
	if(nx*vvx+ny*vvy > 0) return 1;	// Facing front.
	return 0;	// Facing away.
}

void VertexColors(int lightlevel, rendquad_t *quad, int num)
{
	int		i;
	float	light = lightlevel / 255.0f, real, minimum;

	for(i=0; i<num; i++)
	{
		real = light - (quad->dist[i] - 32)/maxLightDist * (1-light);
		minimum = light*light + (light - .63f) / 2;
		if(real < minimum) real = minimum; // Clamp it.

		// Add extra light.
		real += extralight/16.0f;

		// Check for torch.
		if(viewplayer->fixedcolormap)
		{
			// Colormap 1 is the brightest. I'm guessing 16 would be the darkest.
			int ll = 16 - viewplayer->fixedcolormap;
			float d = (1024 - quad->dist[i]) / 512.0f;
			float newmin = d*ll / 15.0f;
			if(real < newmin) real = newmin;
		}

		// Clamp the final light.
		if(real < 0) real = 0;
		if(real > 1) real = 1;

		quad->color[i].rgb[0] = quad->color[i].rgb[1] = quad->color[i].rgb[2] = 
			(DGLubyte) 0xff * real;
	}
}

// The sector height should've been checked by now.
void R_RenderWallSeg(seg_t *seg, sector_t *frontsec, boolean accurate)
{
	sector_t		*backsec = seg->backsector;
	side_t			*sid = seg->sidedef;
	line_t			*ldef = seg->linedef;
	float			ffloor = FIX2FLT(frontsec->floorheight);
	float			fceil = FIX2FLT(frontsec->ceilingheight);
	float			bfloor, bceil, fsh = fceil-ffloor, bsh;
	float			tcyoff;
	rendquad_t		quad;
	float			origv1[2], origv2[2];	// The original end points, for dynlights.
	int				sectorlight;

	memset(&quad, 0, sizeof(quad));		// Clear the quad.

	// Get the start and end points. Full floating point conversion is
	// actually only necessary for polyobjs.
	if(accurate)
	{
		quad.v1[VX] = FIX2FLT(seg->v1->x);
		quad.v1[VY] = FIX2FLT(seg->v1->y);
		quad.v2[VX] = FIX2FLT(seg->v2->x);
		quad.v2[VY] = FIX2FLT(seg->v2->y);
		// These are the original vertices, copy them.
		memcpy(origv1, quad.v1, sizeof(origv1));
		memcpy(origv2, quad.v2, sizeof(origv2));
	}
	else
	{
		float dx, dy;
		// Not-so-accurate.
		quad.v1[VX] = Q_FIX2FLT(seg->v1->x);
		quad.v1[VY] = Q_FIX2FLT(seg->v1->y);
		quad.v2[VX] = Q_FIX2FLT(seg->v2->x);
		quad.v2[VY] = Q_FIX2FLT(seg->v2->y);
		// The original vertex. For dlights.
		memcpy(origv1, quad.v1, sizeof(origv1));
		memcpy(origv2, quad.v2, sizeof(origv2));
		// Make the very small gap-hiding adjustment.
		dx = quad.v2[VX] - quad.v1[VX];
		dy = quad.v2[VY] - quad.v1[VY];
		quad.v2[VX] += dx/seg->length/4;
		quad.v2[VY] += dy/seg->length/4;
	}

	// Let's first check which way this seg is facing.
	if(!SegFacingDir(quad.v1, quad.v2)) return;	// The wrong direction?

	// Calculate the distances.
	quad.dist[0] = PointDist2D(quad.v1);
	quad.dist[1] = PointDist2D(quad.v2);

	// Calculate the color at both vertices.
	sectorlight = LevelFullBright? 255 : frontsec->lightlevel;
	if(sectorlight < r_ambient) sectorlight = r_ambient;
	VertexColors(sectorlight, &quad, 2);

	// This line is now seen in the map.
	ldef->flags |= ML_MAPPED;

	// Some texture coordinates.
	quad.texoffx = (sid->textureoffset>>FRACBITS)+(seg->offset>>FRACBITS);
	quad.len = seg->length;
	tcyoff = Q_FIX2FLT(sid->rowoffset);

	// The middle texture, single sided.
	if(sid->midtexture && !backsec)
	{
		curtex = GL_PrepareTexture(sid->midtexture);		
		quad.texoffy = tcyoff;
		if(ldef->flags & ML_DONTPEGBOTTOM)
			quad.texoffy += texh-fsh;

		// Fill in the remaining quad data.
		quad.flags = 0;
		quad.top = fceil;
		quad.bottom = ffloor;
		quad.texw = texw;
		quad.texh = texh;
		RL_AddQuad(&quad, curtex);
		//if(useDynLights) DL_ProcessWall(&quad, origv1, origv2);

		//printf( "Solid segment in sector %p.\n", frontsec);
		// This is guaranteed to be a solid segment.
		C_AddViewRelSeg(quad.v1[VX], quad.v1[VY], quad.v2[VX], quad.v2[VY]);
	}
	// The skyfix?
	if(frontsec->skyfix)
	{
		if(!backsec || (backsec && (backsec->ceilingheight+(backsec->skyfix<<FRACBITS) < 
			frontsec->ceilingheight+(frontsec->skyfix<<FRACBITS))))
		{
			quad.flags = RQF_SKY_MASK_WALL;
			quad.top = fceil + frontsec->skyfix;
			quad.bottom = fceil;
			RL_AddQuad(&quad, curtex);
			//if(useDynLights) DL_ProcessWall(&quad, origv1, origv2);
		}
	}
	// If there is a back sector we may need upper and lower walls.
	if(backsec)	// A twosided seg?
	{
		bfloor = FIX2FLT(backsec->floorheight);
		bceil = FIX2FLT(backsec->ceilingheight);
		bsh = bceil - bfloor;
		if(bsh <= 0 || bceil <= ffloor || bfloor >= fceil)
		{
			//printf( "Solid segment in sector %p (backvol=0).\n", frontsec);
			// The backsector has no space. This is a solid segment.
			C_AddViewRelSeg(quad.v1[VX],quad.v1[VY],quad.v2[VX],quad.v2[VY]);
		}
		if(sid->midtexture)	// Quite probably a masked texture.
		{
			float mceil = (bceil<fceil)?bceil:fceil,
				mfloor = (bfloor>ffloor)?bfloor:ffloor,
				msh = mceil - mfloor;
			if(msh > 0)
			{
				curtex = GL_PrepareTexture(sid->midtexture);
				// Calculate the texture coordinates. Also restrict
				// vertical tiling (if masked) by adjusting mceil and mfloor.
				/*if(texmask)	// A masked texture?
				{
					quad.flags = RQF_MASKED;
					quad.texoffy = 0;
					// We don't allow vertical tiling.
					if(ldef->flags & ML_DONTPEGBOTTOM)
					{
						mfloor += tcyoff;
						mceil = mfloor + texh;
					}
					else
					{
						mceil += tcyoff;
						mfloor = mceil - texh;
					}
				}
				else // Normal texture.
				{
					quad.flags = 0;
					quad.texoffy = tcyoff;
					if(ldef->flags & ML_DONTPEGBOTTOM) // Lower unpegged. Align bottom.
						quad.texoffy += texh-msh;
				}*/
				if(texmask)
				{
					quad.flags = RQF_MASKED;
					quad.texoffy = 0 ;
				}
				else
				{
					quad.flags = 0;
					quad.texoffy = tcyoff;
				}
				// We don't allow vertical tiling.
				if(ldef->flags & ML_DONTPEGBOTTOM)
				{
					mfloor += tcyoff;
					mceil = mfloor + texh;
				}
				else
				{
					mceil += tcyoff;
					mfloor = mceil - texh;
				}
				// Clip it.
				if(mfloor < bfloor) mfloor = bfloor;
				if(mfloor < ffloor) mfloor = ffloor;
				
				// Fill in the remainder of the data.
				quad.top = mceil;
				quad.bottom = mfloor;
				quad.texw = texw;
				quad.texh = texh;
				RL_AddQuad(&quad, curtex);
				//if(useDynLights) DL_ProcessWall(&quad, origv1, origv2);
			}
		}
		// Upper wall.
		if(bceil < fceil && !(frontsec->ceilingpic == skyflatnum && backsec->ceilingpic == skyflatnum))
		{
			float topwh = fceil - bceil;
			if(sid->toptexture)	// A texture present?
			{
				curtex = GL_PrepareTexture(sid->toptexture);
				// Calculate texture coordinates.
				quad.texoffy = tcyoff;
				if(!(ldef->flags & ML_DONTPEGTOP))
				{
					// Normal alignment to bottom.
					quad.texoffy += texh-topwh;
				}								
				quad.flags = 0;				
			}
			else
			{
				// No texture? Bad thing. You don't deserve texture 
				// coordinates. Take the ceiling texture.
				curtex = GL_PrepareFlat(frontsec->ceilingpic);				
				quad.flags = RQF_FLAT | RQF_MISSING_WALL;
			}
			quad.top = fceil;
			quad.bottom = bceil;
			quad.texw = texw;
			quad.texh = texh;
			RL_AddQuad(&quad, curtex);
			//if(useDynLights) DL_ProcessWall(&quad, origv1, origv2);
		}
		// Lower wall.
		if(bfloor > ffloor && !(frontsec->floorpic == skyflatnum && backsec->floorpic == skyflatnum))
		{
			if(sid->bottomtexture)	// There is a texture?
			{
				curtex = GL_PrepareTexture(sid->bottomtexture);
				// Calculate texture coordinates.
				quad.texoffy = tcyoff;
				if(ldef->flags & ML_DONTPEGBOTTOM)
				{
					// Lower unpegged. Align with normal middle texture.
					//quad.texoffy += fsh-texh;
					quad.texoffy += fceil-bfloor;
				}
				quad.flags = 0;
			}
			else
			{
				// No texture? Again!
				curtex = GL_PrepareFlat(frontsec->floorpic);
				quad.flags = RQF_FLAT | RQF_MISSING_WALL;
			}
			quad.top = bfloor;
			quad.bottom = ffloor;
			quad.texw = texw;
			quad.texh = texh;
			RL_AddQuad(&quad, curtex);
			//if(useDynLights) DL_ProcessWall(&quad, origv1, origv2);
		}
	}
}
	
void R_RenderSubsector(int ssecidx)
{
	subsector_t		*ssec = SUBSECTOR_PTR(ssecidx);
	extern subsector_t *currentssec;
	int				i;
	byte			*seg;
	sector_t		*sect = ssec->sector;
	float			ffloor = FIX2FLT(sect->floorheight);
	float			fceil = FIX2FLT(sect->ceilingheight);
	rendquad_t		triangle;
	int				sectorlight;
	
	if(fceil-ffloor <= 0) 
	{
		return;	// Skip this, no volume.
	}

	if(!firstsubsector)
	{
		if(!C_CheckSubsector(ssec)) return;	// This isn't visible.
	}
	else
		firstsubsector = false;

//	printf( "*** Rendering subsector %d (sector %p)\n", ssecidx, sect);

	// Set some global current subsector data (aagh).
	currentssec = ssec;

	// Sprites for this sector have to be drawn. This must be done before
	// the segments of this subsector are added to the clipper. Otherwise
	// the sprites would get clipped by them, and that wouldn't be right.
	R_AddSprites(sect);

	// Dynamic lights or flares?
//	if(useDynLights || haloMode) DL_MarkForSubsector(ssec);

	// Draw the walls.
	for(i=0, seg=segs+SEGIDX(ssec->firstline); i<ssec->numlines; i++, seg+=SEGSIZE)
		if(((seg_t*)seg)->linedef)	// "minisegs" have no linedefs.
			R_RenderWallSeg((seg_t*)seg, sect, false);

	// Is there a polyobj on board?
	if(ssec->poly)
		for(i=0; i<ssec->poly->numsegs; i++)
			R_RenderWallSeg(ssec->poly->segs[i], sect, true);

	sectorlight = LevelFullBright? 255 : sect->lightlevel;
	if(sectorlight < r_ambient) sectorlight = r_ambient;

	// The floor.
	memset(&triangle, 0, sizeof(triangle));
	triangle.flags = RQF_FLAT;// | RQF_FLOOR_FACING;	// This is a flat floor triangle.
	triangle.color[0].rgb[0] = sectorlight;
	if(viewz > sect->floorheight) // && vpitch < yfov)
	{
		// Check for sky... in the floor?
		if(sect->floorpic == skyflatnum) 
		{
			triangle.flags |= RQF_SKY_MASK;
			skyhemispheres |= SKYHEMI_LOWER;
//			if(sect->special == 200) special200 = true;
		}
		curtex = GL_PrepareFlat(sect->floorpic);
		triangle.texw = texw;
		triangle.texh = texh;
		triangle.texoffx = sect->flatoffx;
		triangle.texoffy = sect->flatoffy;
		triangle.top = ffloor;
		// The first vertex is always the first in the whole list.
		RL_AddFlatQuads(&triangle, curtex, ssec->numedgeverts, ssec->edgeverts, 0);
	}
	// And the roof.
	triangle.flags = RQF_FLAT;
	triangle.color[0].rgb[0] = sectorlight;
	if(viewz < sect->ceilingheight) //&& vpitch > -yfov)
	{
		// Check for sky.
		if(sect->ceilingpic == skyflatnum) 
		{
			triangle.flags |= RQF_SKY_MASK;
			skyhemispheres |= SKYHEMI_UPPER;
		}
		curtex = GL_PrepareFlat(sect->ceilingpic);
		triangle.texw = texw;
		triangle.texh = texh;
		triangle.texoffx = 0;
		triangle.texoffy = 0;
		triangle.top = fceil + sect->skyfix;
		// The first vertex is always the last in the whole list.
		RL_AddFlatQuads(&triangle, curtex, ssec->numedgeverts, ssec->edgeverts, 1);
	}
	// Dynamic lights. Processes both the ceiling and the floor, and all
	// visible wall segments.
	if(useDynLights) 
		DL_ProcessSubsector(&triangle, ssec);
	else if(haloMode)
	{
		// Flares are on, but dynamic lights aren't processed. We must
		// mark the rendered luminous objects ourselves.
		for(i=0; i<numLuminous; i++)
		{
			if(luminousList[i].thing->subsector == ssec)
				luminousList[i].flags |= LUMF_RENDERED;
		}
	}
}

void R_RenderNode(int bspnum)
{
	node_t          *bsp;
	int             side;

	// If the clipper is full we're pretty much done.
	if(cliphead)
		if(cliphead->start == 0 && cliphead->end == BANG_MAX)
			return;

	if (bspnum & NF_SUBSECTOR)
	{
		if (bspnum == -1)
			R_RenderSubsector(0);
		else
			R_RenderSubsector(bspnum&(~NF_SUBSECTOR));
		return;
	}

	//bsp = &nodes[bspnum];
	bsp = NODE_PTR(bspnum);

//
// decide which side the view point is on
//
	side = R_PointOnSide (viewx, viewy, bsp);
	
	R_RenderNode (bsp->children[side]); // recursively divide front space

/*	if(cliphead->start == 0 && cliphead->end == BANG_MAX)
		return;	// We can stop rendering.*/

	//if (R_CheckBBox (bsp->bbox[side^1]))    // possibly divide back space
	// We can't use that, unfortunately.
	R_RenderNode (bsp->children[side^1]);
	
}

void R_RenderMap()
{
	binangle_t	viewside;

	// This is all the clearing we'll do.
	gl.Clear(DGL_DEPTH_BUFFER_BIT);

	// Setup the modelview matrix.
	GL_ModelViewMatrix();

	if(!freezeRLs)
	{
		RL_ClearLists();	// Clear the lists for new quads.
		C_ClearRanges();	// Clear the clipper.

		if(useDynLights || haloMode || litSprites) DL_InitForNewFrame(); // Maintain luminous objects.

		// Add the backside clipping range, vpitch allowing.
		if(vpitch <= 90-yfov/2 && vpitch >= -90+yfov/2)
		{
			float a = fabs(vpitch) / (90-yfov/2);
			//binangle_t startAngle = (binangle_t) BANG_45*(1+a);
			binangle_t startAngle = (binangle_t) (BANG_45*fieldOfView/90)*(1+a);
			binangle_t angLen = BANG_180 - startAngle;																					
			viewside = (viewangle>>16) + startAngle;
			C_SafeAddRange(viewside, viewside+angLen);
			C_SafeAddRange(viewside+angLen, viewside+2*angLen);
		}
		// The viewside line for the depth cue.
		viewsidex = -FIX2FLT(viewsin);
		viewsidey = FIX2FLT(viewcos);

		// We don't want subsector clipchecking for the first subsector.
		firstsubsector = true;
		R_RenderNode(numnodes-1);
		
	}
	RL_RenderAllLists();

	// The halos.
	if(haloMode) H_InitForNewFrame();

	// Clipping fragmentation happens when there are holes in the walls.
	/*if(cliphead->next)
	{
		clipnode_t *ci;
		printf("\nTic: %d, Clipnodes are fragmented:\n", gametic);
		for(ci=cliphead; ci; ci=ci->next)
			printf( "range %p: %4x => %4x (%d)\n",ci,ci->start,ci->end,ci->used);
		I_Error("---Fragmented clipper---\n");
	}*/
}

// Global variables... agh.
static byte *slRGB1, *slRGB2; // The colors to modify when sprites are lit.
static fvertex_t slViewVec;
static vissprite_t *slSpr;

static boolean spriteLighter(lumobj_t *lum, fixed_t dist)
{
	int			i, temp;
	float		fdist = FIX2FLT(dist);
	fvertex_t	lightVec = { FIX2FLT(slSpr->gx - lum->thing->x), FIX2FLT(slSpr->gy - lum->thing->y) };
	float		directness, side, inleft, inright, zfactor;
	
	if(!fdist) return true;
	if(slRGB1[0] == 0xff && slRGB1[1] == 0xff && slRGB1[2] == 0xff 
		&& slRGB2[0] == 0xff && slRGB2[1] == 0xff && slRGB2[2] == 0xff)
		return false; // No point in continuing, light is already white.

	zfactor = (FIX2FLT(slSpr->gz + slSpr->gzt)/2 - (FIX2FLT(lum->thing->z) + lum->center)) / dlMaxRad;
	if(zfactor < 0) zfactor = -zfactor;
	if(zfactor > 1) return true; // Too high or low.
	zfactor = 1-zfactor;
	// Enlarge the full-lit area.
	zfactor *= 2;
	if(zfactor > 1) zfactor = 1;
	
	lightVec.x /= fdist;
	lightVec.y /= fdist;

	// Also include the effect of the distance to zfactor.
	fdist /= dlMaxRad;
	fdist = 1-fdist;
	fdist *= 2;
	if(fdist > 1) fdist = 1;
	zfactor *= fdist;

	// Now the view vector and light vector are normalized.
	directness = slViewVec.x * lightVec.x + slViewVec.y * lightVec.y; // Dot product.
	side = -slViewVec.y * lightVec.x + slViewVec.x * lightVec.y;
	// If side > 0, the light comes from the right.
	if(directness > 0)
	{
		// The light comes from the front.
		if(side > 0)
		{
			inright = 1;
			inleft = directness;
		}
		else 
		{
			inleft = 1;
			inright = directness;
		}
	}
	else
	{
		// The light comes from the back.
		if(side > 0)
		{
			inright = side;
			inleft = 0;
		}
		else 
		{
			inleft = -side;
			inright = 0;
		}
	}
	inright *= zfactor;
	inleft *= zfactor;
	if(inleft > 0)
	{
		for(i=0; i<3; i++)
		{
			temp = slRGB1[i] + inleft*lum->rgb[i];
			if(temp > 0xff) temp = 0xff;
			slRGB1[i] = temp;
		}
	}
	if(inright > 0)
	{
		for(i=0; i<3; i++)
		{
			temp = slRGB2[i] + inright*lum->rgb[i];
			if(temp > 0xff) temp = 0xff;
			slRGB2[i] = temp;
		}
	}
	return true;
}

void R_RenderSprite(vissprite_t *spr)
{
	float		bot,top;
	float		off = FIX2FLT(spriteoffset[spr->patch]);
	float		w = FIX2FLT(spritewidth[spr->patch]);
	int			sprh;
	float		v1[2];
	DGLubyte	alpha;
	boolean		additiveBlending = false, flip, restoreMatrix = false;
	rendquad_t	tempquad;

	// Do we need to translate any of the colors?
	if(spr->mobjflags & DDMF_TRANSLATION)
	{
		dc_translation = translationtables-256 + spr->class*((MAXPLAYERS-1)*256) +
			((spr->mobjflags & DDMF_TRANSLATION) >> (DDMF_TRANSSHIFT-8));
		// We need to prepare a translated version of the sprite.
		GL_SetTranslatedSprite(spr->patch, dc_translation);
	}
	else
	{
		// Set the texture. No translation required.
		GL_SetSprite(spr->patch);
	}
	sprh = spriteheights[spr->patch];

	// Set the lighting and alpha.
	if(missileBlend && spr->mobjflags & DDMF_BRIGHTSHADOW)
	{
		alpha = 204;	// 80 %.
		additiveBlending = true;
	}
	else if(spr->mobjflags & DDMF_SHADOW)
		alpha = 85;		// One third.
	else if(spr->mobjflags & DDMF_ALTSHADOW)
		alpha = 170;	// Two thirds.
	else
		alpha = 255;

	if(spr->lightlevel < 0)
		gl.Color4ub(255, 255, 255, alpha);
	else
	{
		v1[VX] = Q_FIX2FLT(spr->gx);
		v1[VY] = Q_FIX2FLT(spr->gy);
		tempquad.dist[0] = PointDist2D(v1);
		VertexColors(r_ambient > spr->lightlevel? r_ambient : spr->lightlevel, 
			&tempquad, 1);

		// Add extra light using dynamic lights.
		if(litSprites)
		{
			float len;
			memcpy(tempquad.color[1].rgb, tempquad.color[0].rgb, 3);
			slSpr = spr;
			slRGB1 = tempquad.color[0].rgb;
			slRGB2 = tempquad.color[1].rgb;
			slViewVec.x = FIX2FLT(spr->gx - viewx);
			slViewVec.y = FIX2FLT(spr->gy - viewy);
			len = sqrt(slViewVec.x*slViewVec.x + slViewVec.y*slViewVec.y);
			if(len)
			{
				slViewVec.x /= len;
				slViewVec.y /= len;
				DL_RadiusIterator(spr->gx, spr->gy, dlMaxRad << FRACBITS, spriteLighter);
			}
		}
		gl.Color4ub(tempquad.color[0].rgb[0],
			tempquad.color[0].rgb[1],
			tempquad.color[0].rgb[2],
			alpha);
	}

	// We must find the correct positioning using the sector floor and ceiling
	// heights as an aid.
	top = FIX2FLT(spr->gzt);
	if(sprh < spr->secceil-spr->secfloor)	// Sprite fits in, adjustment possible?
	{
		// Check top.
//		if(top > spr->secceil) top = spr->secceil;
		// Check bottom.
		if(top-sprh < spr->secfloor)
			top = spr->secfloor+sprh;
	}
	// Adjust by the floor clip.
	top -= FIX2FLT(spr->floorclip);
	bot = top - sprh;

	// Should the texture be flipped?
	flip = spr->xiscale < 0;

	// Do we need to do some aligning?
	if(spr->viewAligned || alwaysAlign == 2)
	{
		float centerx = FIX2FLT(spr->gx), centery = FIX2FLT(spr->gy);
		float centerz = (top+bot)/2;
		// We must set up a modelview transformation matrix.
		restoreMatrix = true;
		gl.MatrixMode(DGL_MODELVIEW);
		gl.PushMatrix();
		// Rotate around the center of the sprite.
		gl.Translatef(centerx, centerz, centery);
		if(!spr->viewAligned)
		{
			float dx = centerx - vx, dy = centery - vz;
			float spriteAngle = BANG2DEG(bamsAtan2(centerz-vy, sqrt(dx*dx + dy*dy)));
			if(spriteAngle > 180) spriteAngle -= 360;
			//ST_Message( "%.0f\n", spriteAngle);
			if(fabs(spriteAngle) > maxSpriteAngle)
			{
				// Rotate along the sprite edge.
				dx = spr->v1[VX] - spr->v2[VX];
				dy = spr->v1[VY] - spr->v2[VY];
				gl.Rotatef(/*spriteAngle + (spriteAngle>0? -maxSpriteAngle : maxSpriteAngle)*/
					spriteAngle>0? spriteAngle-maxSpriteAngle : spriteAngle+maxSpriteAngle,
					dx, 0, dy);			
			}
		}
		else
		{
			// Normal rotation perpendicular to the view plane.
			gl.Rotatef(vpitch, viewsidex, 0, viewsidey);
		}
		gl.Translatef(-centerx, -centerz, -centery);
	}

	if(additiveBlending)
	{
		// Change the blending mode.
		gl.Func(DGL_BLENDING, DGL_SRC_ALPHA, DGL_ONE);
	}

	// Render the sprite.
	gl.Begin(DGL_QUADS);
	gl.TexCoord2f(flip, 1);
	gl.Vertex3f(spr->v1[VX], bot, spr->v1[VY]);
	gl.TexCoord2f(flip, 0);
	gl.Vertex3f(spr->v1[VX], top, spr->v1[VY]);

	if(litSprites && spr->lightlevel >= 0)
	{
		gl.Color4ub(tempquad.color[1].rgb[0],
			tempquad.color[1].rgb[1],
			tempquad.color[1].rgb[2],
			alpha);
	}
	gl.TexCoord2f(!flip, 0);
	gl.Vertex3f(spr->v2[VX], top, spr->v2[VY]);
	gl.TexCoord2f(!flip, 1);
	gl.Vertex3f(spr->v2[VX], bot, spr->v2[VY]);
	gl.End();

	if(restoreMatrix)
	{
		// Restore the original modelview matrix.
		gl.PopMatrix();
	}

	if(additiveBlending)
	{
		// Change to normal blending.
		gl.Func(DGL_BLENDING, DGL_SRC_ALPHA, DGL_ONE_MINUS_SRC_ALPHA);
	}
}

void GL_DrawPSprite(int x, int y, float scale, int flip, int lump)
{
	int		w, h;

	GL_SetSprite(lump);
	w = spritewidth[lump] >> FRACBITS;
	h = spriteheights[lump];

	if(flip) flip = 1; // Ensure it's zero or one.

	gl.Begin(DGL_QUADS);
	gl.TexCoord2f(flip, 0);
	gl.Vertex2f(x, y);
	gl.TexCoord2f(!flip, 0);
	gl.Vertex2f(x + w*scale, y);
	gl.TexCoord2f(!flip, 1);
	gl.Vertex2f(x + w*scale, y + h*scale);
	gl.TexCoord2f(flip, 1);
	gl.Vertex2f(x, y+h*scale);
	gl.End();
}



// Console commands.
int CCmdFog(int argc, char **argv)
{
	int		i;

	if(argc == 1)
	{
		CON_Printf( "Usage: %s (cmd) (args)\n", argv[0]);
		CON_Printf( "Commands: on, off, mode, color, start, end, density.\n");
		CON_Printf( "Modes: linear, exp, exp2.\n");
		//CON_Printf( "Hints: fastest, nicest, dontcare.\n");
		CON_Printf( "Color is given as RGB (0-255).\n");
		CON_Printf( "Start and end are for linear fog, density for exponential.\n");
		return true;		
	}
	if(!stricmp(argv[1], "on"))
	{
		GL_UseWhiteFog(true);
		CON_Printf( "Fog is now active.\n");
	}
	else if(!stricmp(argv[1], "off"))
	{
		GL_UseWhiteFog(false);
		CON_Printf( "Fog is now disabled.\n");
	}
	else if(!stricmp(argv[1], "mode") && argc == 3)
	{
		if(!stricmp(argv[2], "linear"))	
		{
			gl.Fog(DGL_FOG_MODE, DGL_LINEAR);
			CON_Printf( "Fog mode set to linear.\n");
		}
		else if(!stricmp(argv[2], "exp")) 
		{
			gl.Fog(DGL_FOG_MODE, DGL_EXP);
			CON_Printf( "Fog mode set to exp.\n");
		}
		else if(!stricmp(argv[2], "exp2")) 
		{
			gl.Fog(DGL_FOG_MODE, DGL_EXP2);
			CON_Printf( "Fog mode set to exp2.\n");
		}
		else return false;
	}
/*	else if(!stricmp(argv[1], "hint") && argc == 3)
	{
		if(!stricmp(argv[2], "fastest")) 
		{
			glHint(GL_FOG_HINT, GL_FASTEST);
			CON_Printf( "Fog quality set to fastest.\n");
		}
		else if(!stricmp(argv[2], "nicest")) 
		{
			glHint(GL_FOG_HINT, GL_NICEST);
			CON_Printf( "Fog quality set to nicest.\n");
		}
		else if(!stricmp(argv[2], "dontcare")) 
		{
			glHint(GL_FOG_HINT, GL_DONT_CARE);
			CON_Printf( "Fog quality set to \"don't care\".\n");
		}
		else return false;
	}*/
	else if(!stricmp(argv[1], "color") && argc == 5)
	{
		byte col[4];
		for(i=0; i<3; i++)
			col[i] = strtol(argv[2+i], NULL, 0)/*/255.0f*/;
		col[3] = 1;
		gl.Fogv(DGL_FOG_COLOR, col);
		CON_Printf( "Fog color set.\n");
	}	
	else if(!stricmp(argv[1], "start") && argc == 3)
	{
		gl.Fog(DGL_FOG_START, strtod(argv[2], NULL));
		CON_Printf( "Fog start distance set.\n");
	}
	else if(!stricmp(argv[1], "end") && argc == 3)
	{
		gl.Fog(DGL_FOG_END, strtod(argv[2], NULL));
		CON_Printf( "Fog end distance set.\n");
	}
	else if(!stricmp(argv[1], "density") && argc == 3)
	{
		gl.Fog(DGL_FOG_DENSITY, strtod(argv[2], NULL));
		CON_Printf( "Fog density set.\n");
	}
	else return false;
	// Exit with a success.
	return true;
}
