/*
 * Copyright (C) 1996 by Raphael Quinet.  All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation.  If more than a few
 * lines of this code are used in a program which displays a copyright
 * notice or credit notice, the following acknowledgment must also be
 * displayed on the same screen: "This product includes software
 * developed by Raphael Quinet for use in the Quake Editing Utilities
 * project."  THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR
 * IMPLIED WARRANTY.
 *
 * More information about the QEU project can be found on the WWW:
 * "http://www.montefiore.ulg.ac.be/~quinet/games/editing.html" or by
 * mail: Raphael Quinet, 9 rue des Martyrs, B-4550 Nandrin, Belgium.
 */

/*
 * UNENT.C - Writing entity lumps to files, by tiglari@hexenworld.com
 *
 * This file was derived by by from various bits and pieces
 * of the QEU suite, which was partially derived from code
 * written by the DEU Team: Raphael Quinet, Brandon Wyber, Ted
 * Vessenes and others.  Apologies for the C++-ifications, but
 * I was trying to get better at writing iterators when I
 * did this.
 *
 * The unent.cpp code will only compile in DOS, due to the use of
 * findfirst() to manage wildcards, and fnsplit for drive/directory
 * paths, so you'll have to make some changes for non-mess systems.
 * I've also put some stuff into Q_FILES & \Q_BSP, hopefully not such
 * as to trash portability, but I don't know for sure.
 */

#include <dir.h>
#include <iostreams.h>

#include "qeu.h"
#include "q_files.h"
#include "q_misc.h"
#include "f_pack.h"
#include "f_bsp.h"

/*
 *  Raphael's code reads in the entire directory; since the Q2 directory
 *  is too big for a TC3 program, it is read from the .pak entry-by-entry
 *
 */

class PACKEntryIterator {
public:
  PACKEntryIterator(FILE *file, UInt32 offset, UInt32 dirsize, char *ext);
  Bool More(PACKDirectory &);  // true if more, fills in structure
protected:
  UInt32 currpos;    // current directory position in packfile.
  char extension[4];
  Int16 size;
  FILE *packfile;
};

PACKEntryIterator::PACKEntryIterator(FILE *file,UInt32 offset, UInt32 dirsize, char *ext)
{
  currpos = offset;
  size = (UInt16)(dirsize / sizeof(struct PACKDirectory));
  strcpy(extension,ext);
  packfile = file;
}


Bool PACKEntryIterator::More(PACKDirectory &current)
{
  while (size) {
    if (fseek(packfile, currpos, SEEK_SET)<0) {
      ProgError("fseek failed in PACKEntryIterator::More (unent.cpp)");
      return FALSE;
    }
    if (ReadBytes(packfile,&current,sizeof(struct PACKDirectory)) == FALSE) {
      ProgError("ReadBytes failed in PACKEntryIterator::More (unent.cpp)");
      return FALSE;
    }
    char *name = GetBaseName(current.name);
    char *ext =  GetFileExtension(name);
    currpos = currpos+sizeof(struct PACKDirectory);
    size --;
    if (!strcmp(ext,"bsp")) {
      return TRUE;
    }
  }
  return FALSE;
}


/* writes entities in bsp to entfilename; adapted from UnBSPFile */
Bool UnEntitiesBSP(FILE *bspfile, UInt32 offset, BSPDirPtr dir,
		char *entfilename)
{
  FILE   *newfile;

  if (bspfile == NULL || dir == NULL) {
    return FALSE;
  }
  newfile = fopen(entfilename, "wb");
  if (newfile == NULL) {
    return FALSE;
  }

  if ((fseek(bspfile, offset + dir[0].offset, SEEK_SET) < 0)
	  || (CopyBytes(newfile, bspfile, dir[0].size-1) == FALSE))
	  // -1 above tiglari; prevents extra char from being written.
  {
	  fclose(newfile);
	  return FALSE;
  }
  fclose(newfile);
  return TRUE;
}

//  rips entities from a bsp-file, or section of PACK starting at offset
void RipBSPEntities(ostream &out, char *name, UInt32 offset, FILE *file)
{
    int magic;
    char *entname = ChangeFileExtension(name,NULL,"ent");
    out << "   writing to: " << entname;
    if ((fseek(file, offset, SEEK_SET) < 0)
	|| ((magic = ReadMagic(file)) != FTYPE_BSP && magic != FTYPE_Q2BSP))
      out << "  urrk, not BSP \n";
    else
      out << "\n";
    UInt16 bspdirsize;
    BSPDirPtr bspdir = ReadBSPDirectory(file, offset, &bspdirsize);
    UnEntitiesBSP(file, offset, bspdir, entname);
    free(entname);
}

// rips entities from PACK file
void RipPACKEntities(ostream &out, UInt32 diroffset, UInt32 dirsize, FILE *file)
{
  char buf[57];
  PACKEntryIterator Iter(file, diroffset, dirsize, "bsp");
  PACKDirectory entry;

  while (Iter.More(entry)) {
    strncpy(buf, entry.name, 56);
    buf[56] = '\0';
    char *name = GetBaseName(buf);
    RipBSPEntities(out, name, entry.offset, file);
  }
}

char *CopyFilePath(char *filename)
{
  char drive[3], path[64];
  char *ret;
  fnsplit(filename,drive,path,NULL,NULL);
  ret = (char *)QMalloc(strlen(path) + strlen(drive) + 1);
  strcpy(ret,drive);
  strcat(ret,path);
  return ret;
}


void main(int argc, char *argv[])
{
  char       *filename = NULL;
  char       *ext = NULL;
  FILE       *file;
  int         ftype;
  PACKDirPtr  dir;
  UInt16      dirsize, n;
  Bool        pak = FALSE;
  Bool        bsp = FALSE;
  char       *dirname = NULL;


  cout << "Entity Extractor by tiglari@hexenworld.com\n";
  cout << "  Code almost entirely based on Raphael Quinet's QEU\n";

  if (argc < 2) {
    cout << "\n  Usage:  unent <name>.pak|bsp\n";
    cout << "    Writes entities to files bspname.ent for each bspfile\n";
    cout << "    Drive, path and wildcards OK in <name>\n";
    cout << "    Currently supports Quake, Quake2 and Hexen2\n";

  }
  else {
   // check extensions
    filename = argv[1];
    ext = GetFileExtension(filename);
    if (!(pak = !strcmp(ext,"pak"))) {
      if (!(bsp = !strcmp(ext,"bsp"))) {
	ProgError("extension .pak or .bsp is required", NULL);
      }
    }
  }

  char *filepath = CopyFilePath(filename);
  struct ffblk ffblk;
  Bool done = findfirst(filename,&ffblk,0);
  while (!done) {
  /* open the PACK/BSP file... */
    char currfile[128];
    strcpy(currfile,filepath);
    strcat(currfile,ffblk.ff_name);
    cout << "\n  Extracting entities from:  " << currfile << "\n";
    file = OpenFileReadMagic(currfile, &ftype);
    if (file == NULL)
      ProgError("File not found (%s)", currfile);

    if (pak) {
      if (ftype != FTYPE_PACK)
	ProgError("File is not a PACK file (%s)", currfile);

  /* Read the PACK directory and go get the entities */
      UInt32 diroffset, dirsize;
      ReadInt32(file, &diroffset);
      ReadInt32(file, &dirsize);

      RipPACKEntities(cout, diroffset, dirsize, file);
    } else if (bsp) {

      if (ftype != FTYPE_BSP && ftype != FTYPE_Q2BSP)
	ProgError("File is not a BSP file (%s)", currfile);

    // rip the entities

      RipBSPEntities(cout, currfile, 0, file);
    }
    done = findnext(&ffblk);

  }
  exit(0);
}

