//
//	Extract motion capture data from a Heretic II FlexModel file
//
//	By Chris Burke
//	serotonin@earthlink.net
//
//	Date			Who		Description
//	----------		---		-------------------------------------------------------
//	07/01/1999		CJB		Coded
//


//	System includes
#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<ctype.h>
#include	<fcntl.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<io.h>

//	Local includes
#include	"CondComp.h"
#include	"FlexModel.h"
#include	"MatrixMath.h"

#if	(__MODEL_IS_CORVUS__ || __MODEL_IS_KIERA__)
#define	kPlayerHeight		64
#define	kPlayerScale		0.4
#endif

//
//	Externs
//

#if	__FIND_SKEL_PARTS__
extern void
FindSkelGroups(
	fmheader_t* pModelHeader, int nFrames, frame_t* pFirstFrame
);
#endif

#if	__EXPORT_DXF__
extern char*
DXF_WriteFile(
	char* pName, char* pComment, char* pFrameName, unsigned long meshnodemask,
	fmheader_t* pModelHeader, meshnode_t* pMeshNodes,
	glcmd_t* pGLCmds, frame_t* pFirstFrame
);
#endif

#if	__EXPORT_OBJ__
extern char*
OBJ_WriteFile(
	char* pName, char* pComment, char* pFrameName, unsigned long meshnodemask,
	fmheader_t* pModelHeader, meshnode_t* pMeshNodes,
	glcmd_t* pGLCmds, frame_t* pFirstFrame
);
#endif

#if	__CAPTURE_MOTION__
extern void
CaptureMotion(
	fmheader_t* pModelHeader, int nFrames,
	frame_t* pFirstFrame, refskel_t* pFirstRef
);
#endif

#if	__FIND_BONE_LENGTHS__
extern void
FindBoneLengths(
	fmheader_t* pModelHeader, int nFrames,
	frame_t* pFirstFrame, refskel_t* pFirstRef
);
#endif

#if	__NEW_SWORD_GLCMDS__
extern int
FixSwordGLCmds(
	fmheader_t* pModelHeader, glcmd_t* pGLCmds, int glcmdstart, int glcmdcnt
);
#endif

//	Globals
int		gModelFID;
FILE*	gModelFile;
char*	gModelFileName;
char*	gProgramName;
long	gModelSize, gNewModelSize;
int		gNumFooFrames;

meshnode_t*	gpMeshNodes;
glcmd_t*	gpGLCmds;
frame_t*	gpFirstFrame;
fmheader_t*	gpModelHeader;
refskel_t*	gpFirstRef;

unsigned char*	gpModelData;
unsigned char*	gpNewModelData;
char	gScratchString[128];


#if	(__MODEL_IS_CORVUS__ || __MODEL_IS_KIERA__)
char*	gNodeNames[] =
{
	"Front_Torso",		"Back_Torso",	"Staff_Away",		"Bow_Away",
	"Armor",			"Right_Arm",	"Right_Hand_Open",	"Right_Hand_Staff",
	"Staff",			"Hellstaff",	"Left_Arm",			"Left_Hand_Open",
	"Left_Hand_Bow",	"Right_Leg",	"Left_Leg",			"Head"
};
#else
char*	gNodeNames[] =
{
	"Dummy"				//	Dummy value
};
#endif

//
//	Utils
//

void quit( char* pMsg, int errCode )
{
	if ( pMsg && *pMsg )
		printf( "\nFATAL: %s\n", pMsg );
	exit( errCode );
}

void usage( char* pMsg, int errCode )
{
	printf(
		"\n"
		"Program: %s\n"
		"Purpose: Generate motion capture data from Heretic II FlexModels\n"
		"\n"
		"Usage:\n"
		"%s file.fm\n",
		gProgramName, gProgramName
	);
	quit( pMsg, errCode );
}

main ( int argc, char **argv )
{

	//	Trivial check for right # of arguments
	gProgramName = argv[0];
	printf( "Executing %s....\n", gProgramName );
	if ( argc != 2 )
		usage("No model file specified",-1);

	//	Make sure the model file exists
	gModelFileName = argv[1];
	printf("Model File: %s\n", gModelFileName);
	if ( !( gModelFile = fopen( gModelFileName, "rb" ) ) )
		usage("Can't open model file", errno);

	//	Get size of model file and suck it into allocated memory
	fseek( gModelFile, 0L, SEEK_END );
	gModelSize = ftell( gModelFile );
	if ( !( gpModelData = malloc( gModelSize ) ) )
		quit("Can't allocate memory to read model", -1);
#if	__USE_NATIVE_IO__
	fclose( gModelFile );
	gModelFID = _open( gModelFileName, O_RDONLY+O_BINARY );
	bytesRead = _read( gModelFID, gpModelData, gModelSize );
	_close( gModelFID );
	if ( bytesRead != gModelSize )
		quit("Couldn't read model data", -1);
#else
	fseek( gModelFile, 0L, SEEK_SET );
	fread( gpModelData, gModelSize, 1L, gModelFile );
	fclose( gModelFile );
#endif

#if	__ADD_SKIN_NAME__
	gNewModelSize = gModelSize + sizeof(skinpath_t);
	if ( !( gpNewModelData = malloc( gNewModelSize ) ) )
		quit("Can't allocate memory for expanded model", -1);
#endif

#if	1
	//	Perform consistency check on model
	{
		header_t	theHeader;
		fmheader_t	theModelHeader;
		byte*		pIntoModel = gpModelData;
		byte*		pIntoNewModel = gpNewModelData;

		while ( pIntoModel < gpModelData + gModelSize )
		{
			//	Read the header for the next chunk
			memcpy( theHeader.ident, pIntoModel, FM_MAXCHUNKIDENT );
			theHeader.ident[FM_MAXCHUNKIDENT-1] = '\0';
			theHeader.version	= FM_GETINT( pIntoModel + FM_MAXCHUNKIDENT + 0 );
			theHeader.size		= FM_GETINT( pIntoModel + FM_MAXCHUNKIDENT + 4 );
			printf(
				"%-32s version %d, length %d bytes.\n",
				theHeader.ident, theHeader.version, theHeader.size );
			if ( theHeader.size <= 0 )
				quit("Invalid (<=0) chunk size in model",-1);
#if	__ADD_SKIN_NAME__
			//	Copy each chunk from old model to new model
			memcpy( pIntoNewModel, pIntoModel, FM_HEADERSIZE + theHeader.size );
			pIntoNewModel += (FM_HEADERSIZE + theHeader.size);
#endif
			//	Process by chunk type
			if ( !strcmp( theHeader.ident, FM_HEADERCHUNKNAME ) )
			{
				//	A HEADER chunk
				if ( theHeader.version != FM_HEADERCHUNKVER )
					quit("Invalid HEADER chunk version number",-1);

				gpModelHeader = (fmheader_t*)(pIntoModel + FM_HEADERSIZE);

				theModelHeader.skinwidth	= FM_GETINT( pIntoModel + FM_HEADERSIZE + 0);	// in pixels
				theModelHeader.skinheight	= FM_GETINT( pIntoModel + FM_HEADERSIZE + 4);	// in pixels
				theModelHeader.framesize	= FM_GETINT( pIntoModel + FM_HEADERSIZE + 8);	// byte count of each frame
				theModelHeader.skincnt		= FM_GETINT( pIntoModel + FM_HEADERSIZE + 12);	// number of skins
				theModelHeader.vert3dcnt	= FM_GETINT( pIntoModel + FM_HEADERSIZE + 16);	// number of unique vertices
				theModelHeader.vertuvcnt	= FM_GETINT( pIntoModel + FM_HEADERSIZE + 20);	// greater than num_xyz for seams
				theModelHeader.tricnt		= FM_GETINT( pIntoModel + FM_HEADERSIZE + 24);	// number of triangles
				theModelHeader.glcmdcnt		= FM_GETINT( pIntoModel + FM_HEADERSIZE + 28);	// dwords in strip/fan command list
				theModelHeader.framecnt		= FM_GETINT( pIntoModel + FM_HEADERSIZE + 32);	// number of animation frames
				theModelHeader.meshnodecnt	= FM_GETINT( pIntoModel + FM_HEADERSIZE + 36);	// number of mesh nodes
				//	Dump all the data
				printf(
					"    Skins are %d x %d pixels.\n"
					"    %d skins identified.\n"
					"    %d animation frames, size %d each.\n"
					"    %d 3D vertices, %d texture verteces, and %d glcmds.\n"
					"    %d triangles, %d mesh nodes.\n",
					theModelHeader.skinwidth, theModelHeader.skinheight,
					theModelHeader.skincnt,
					theModelHeader.framecnt, theModelHeader.framesize,
					theModelHeader.vert3dcnt, theModelHeader.vertuvcnt,
					theModelHeader.glcmdcnt,
					theModelHeader.tricnt, theModelHeader.meshnodecnt
				);
			}
			else
			if ( !strcmp( theHeader.ident, FM_SKINCHUNKNAME ) )
			{
				//	A SKIN chunk
				int		i;
				char*	pSkinName = pIntoModel + FM_HEADERSIZE;
				skinpath_t theSkinPath;
				if ( theHeader.version != FM_SKINCHUNKVER )
					quit("Invalid SKIN chunk version number",-1);
				//	Dump all the data
				for ( i = 0; i < theModelHeader.skincnt; i++ )
				{
					memcpy( theSkinPath.path, pSkinName, FM_MAXPATHLENGTH );
					theSkinPath.path[FM_MAXPATHLENGTH-1] = '\0';
					printf("    %s\n", theSkinPath.path);
					pSkinName += FM_SKINPATHSIZE;
				}
#if	__ADD_SKIN_NAME__
				{
					//	Extend the SKIN chunk of the new model,
					//	and adjust the count in the model header.
					//	WARNING: THIS USES DIRECT ACCESS TO MODEL
					//	IN MEMORY, MIGHT NOT BE ENDIAN-SAFE ON
					//	OTHER THAN INTEL PLATFORM.
					skinpath_t	newName;
					fmheader_t*	pNewHeader;
					header_t*	pNewChunkHeader;
					printf( "    Adding #%d skin name '%s'.\n", i+1, __NEW_SKIN_PATH__ );
					//	Back up the new model pointer to header of SKIN chunk
					pNewChunkHeader = (header_t*)( (byte*)pIntoNewModel - (FM_HEADERSIZE + theHeader.size) );
					//	Copy new name to end of SKIN chunk
					//	NOTE: This assumes that the SKIN chunk is exactly
					//	the size needed to store the specified number of
					//	skin paths.
					memset( newName.path, 0, FM_MAXPATHLENGTH );
					strcpy( newName.path, __NEW_SKIN_PATH__ );
					memcpy( pIntoNewModel, &newName, sizeof(newName) );
					pIntoNewModel += sizeof(newName);
					//	Revise chunk size and skin count
					pNewChunkHeader->size += sizeof(newName);
					pNewHeader = (fmheader_t*)(gpNewModelData + FM_HEADERSIZE);
					++pNewHeader->skincnt;
				}
#endif
				pSkinName = pSkinName;
			}
			else
			if ( !strcmp( theHeader.ident, FM_STCOORDCHUNKNAME ) )
			{
				//	An ST COORD chunk
				int		i, u, v;
				byte*	pCoords = pIntoModel + FM_HEADERSIZE;
				if ( theHeader.version != FM_STCOORDCHUNKVER )
					quit("Invalid ST COORD chunk version number",-1);
				//	Dump all the data
				for ( i = 0; i < theModelHeader.vertuvcnt; i++ )
				{
					u = FM_GETSHORT( pCoords + 0 );
					v = FM_GETSHORT( pCoords + 2 );
					printf("    %-4d: (%d,%d)\n", i, u, v);
					pCoords += FM_STCOORDUVSIZE;
				}
			}
			else
			if ( !strcmp( theHeader.ident, FM_TRISCHUNKNAME ) )
			{
				//	A TRIS chunk
				int			i;
				triangle_t	theTri;
				byte*		pCoords = pIntoModel + FM_HEADERSIZE;
				if ( theHeader.version != FM_TRISCHUNKVER )
					quit("Invalid TRIS chunk version number",-1);
				//	Dump all the data
				for ( i = 0; i < theModelHeader.tricnt; i++ )
				{
					theTri.index_3d[0] = FM_GETSHORT( pCoords + 0 );
					theTri.index_3d[1] = FM_GETSHORT( pCoords + 2 );
					theTri.index_3d[2] = FM_GETSHORT( pCoords + 4 );
					theTri.index_uv[0] = FM_GETSHORT( pCoords + 6 );
					theTri.index_uv[1] = FM_GETSHORT( pCoords + 8 );
					theTri.index_uv[2] = FM_GETSHORT( pCoords + 10 );
					printf(
						"    %-4d: "
						"3D Verteces (%d-%d-%d),"
						"texture verteces (%d-%d-%d)\n",
						i,
						theTri.index_3d[0],theTri.index_3d[1],theTri.index_3d[2],
						theTri.index_uv[0],theTri.index_uv[1],theTri.index_uv[2]
					);
					pCoords += FM_TRISINFOSIZE;
				}
			}
			else
			if ( !strcmp( theHeader.ident, FM_FRAMESCHUNKNAME ) )
			{
				//	A FRAMES chunk
				frame_t*	pFrame;
				if ( theHeader.version != FM_FRAMESCHUNKVER )
					quit("Invalid FRAMES chunk version number",-1);
				pFrame = (frame_t*)(pIntoModel + FM_HEADERSIZE);
				gpFirstFrame = pFrame;
				//	Can't trust frame count in model header. Based on
				//	examination of model data, the count in Corvus and
				//	Kiera models is incorrect.
				gNumFooFrames = theHeader.size /
					( sizeof(framehdr_t) + theModelHeader.vert3dcnt * sizeof(vertex3d_t) );
#if	__FIND_SKEL_PARTS__
				FindSkelGroups( &theModelHeader, gNumFooFrames, gpFirstFrame );
#else
				{
					int			i, nFrames, tally;
					char*		pName;
					frame_t		theFrame;
					char		numbuf[8], aniName[16];
					int			minLNI, maxLNI;
					if ( theHeader.version != FM_FRAMESCHUNKVER )
						quit("Invalid FRAMES chunk version number",-1);
					minLNI = 99999;
					maxLNI = -1;
					//	Dump some of the data
					tally = 0;
					//	WARNING: WE READ FLOATS DIRECTLY INTO DATA
					//	STRUCTURES HERE...THIS IS NOT ENDIAN-CLEAN
					pFrame = (frame_t*)(pIntoModel + FM_HEADERSIZE);

					gpFirstFrame = pFrame;

					*aniName = '\0';
					for ( i = 0; i < theModelHeader.framecnt; i++ )
					{
						//	If *aniName == '\0' this is the start of a
						//	new animation, otherwise it is part of the
						//	animation named in aniName
						theFrame = *pFrame;		//	WARNING: NOT ENDIAN CLEAN
#if	__MAKE_MINI_ME__
						//	WARNING: KLUDGE TO CREATE "MINI-ME" MODEL
						//	WARNING: THIS OVERWRITES THE MODEL IN RAM
						//	WARNING: THIS ASSUMES AUTO-ALIGNMENT
						pFrame->header.scale[0] *= kPlayerScale;
						pFrame->header.scale[1] *= kPlayerScale;
						pFrame->header.scale[2] *= kPlayerScale;
						pFrame->header.translate[0] *= kPlayerScale;
						pFrame->header.translate[1] *= kPlayerScale;
						pFrame->header.translate[2] *= kPlayerScale;
						pFrame->header.translate[2] -= (kPlayerHeight*(1.-kPlayerScale))/2.;
						//	END KLUDGE
#endif
						//	Isolate the name and any numeric suffix
						if ( !strcmp(theFrame.header.name,__EXPORT_FRAME_NAME__) )
							i = i;
						pName = theFrame.header.name;
						pName += strlen(pName) - 1;
						while ( isdigit(*pName) ) --pName;
						strcpy( numbuf, pName+1 );
						*(pName+1) = '\0';
						//	Does this name match previous frame?
						if ( !strcmp( theFrame.header.name, aniName ) )
						{
							//	Frame is part of current animation; revise frame count
							nFrames = atoi( numbuf );	//	So far, this is the # of frames
							if (!nFrames) nFrames = 1;
						}
						else
						{
							//	Starting a new animation; dump previous
							//	name and frame count
							if ( *aniName != '\0' )
							{
								printf(
										"    Animation: %s (%d frames)\n",
										aniName, nFrames
									  );
								tally += nFrames;
							}
							strcpy( aniName, theFrame.header.name );
							nFrames = 1;
						}
						{
							//	Check the range of "lightnormalindex"
							int		vv;
							for ( vv=0; vv<theModelHeader.vert3dcnt; vv++ )
							{
								if ( pFrame->verts[vv].lightnormalindex < minLNI )
									minLNI = pFrame->verts[vv].lightnormalindex;
								if ( pFrame->verts[vv].lightnormalindex > maxLNI )
									maxLNI = pFrame->verts[vv].lightnormalindex;

							}
						}
						pFrame = (frame_t*)((byte*)pFrame + theModelHeader.framesize);
					}
					if ( *aniName != '\0' )
					{
						printf(
								"    Animation: %s (%d frames)\n",
								aniName, nFrames
							  );
						tally += nFrames;
					}
#if	__CHECK_FRAMECNT__
					if ( tally != theModelHeader.framecnt )
						quit( "Frame count doesn't match header", -1 );
#endif
					printf( "Light normal index is in range %d - %d.\n", minLNI, maxLNI );
				}
#endif
			}
			else
			if ( !strcmp( theHeader.ident, FM_GLCMDSCHUNKNAME ) )
			{
				//	A GLCMDS chunk
				int				numTris;			//	Number of triangles processed
				int				numNodes;			//	Number of distinct sets of GLCMDS processed
				int				numLocalTris;		//	Number of triangles in this part.
				int				numGLCmds;			//	Number of GLCmd 32-bit elements processed
				int				numLocalGLCmds;		//	Number of GLCmd 32-bit elements in this node
				int				theCmd, i;
				signed char*	pCmd = pIntoModel + FM_HEADERSIZE;
				glvert_t*		pVert;
				glvert_t		theVert;
//				byte			foo[40];

				if ( theHeader.version != FM_GLCMDSCHUNKVER )
					quit("Invalid GLCMDS chunk version number",-1);

				gpGLCmds = (glcmd_t*)pCmd;

				//	Dump some of the data
				numTris = 0;
				numNodes = 0;
				numLocalTris = 0;
				numGLCmds = 0;
				numLocalGLCmds = 0;

				while ( numGLCmds < theModelHeader.glcmdcnt )
				{
					theCmd = FM_GETINT( pCmd );
					pCmd += 4;
					++numLocalGLCmds;
					if ( !theCmd )
					{
						//	End of this "node"
						if ( numLocalTris )
							printf( "    Node %3d: draws %3d triangles using %3d GLCmd elements\n", numNodes, numLocalTris, numLocalGLCmds );
						++numNodes;
						numTris += numLocalTris;
						numGLCmds += numLocalGLCmds;
						numLocalTris = 0;
						numLocalGLCmds = 0;
						//	NOTE: KEEP LOOKING UNTIL WE'VE DONE ALL THE GLCMD DATA INDICATED IN HEADER
					}
					else
					if ( theCmd > 0 )
					{
						//	Process TRISTRIP command
//						printf("-");
						pVert = (glvert_t*)pCmd;
						for ( i=0; i<theCmd; i++ )
						{
							//	WARNING: WE READ FLOATS DIRECTLY INTO DATA
							//	STRUCTURES HERE...THIS IS NOT ENDIAN-CLEAN
							theVert = *pVert++;
						}
						numLocalTris += (theCmd - 2);
						numLocalGLCmds += theCmd * 3;
						pCmd = (byte*)pVert;
					}
					else
					{
						//	Process TRIFAN command
//						printf("V");
						theCmd = -theCmd;
						pVert = (glvert_t*)pCmd;
						for ( i=0; i<theCmd; i++ )
						{
							//	WARNING: WE READ FLOATS DIRECTLY INTO DATA
							//	STRUCTURES HERE...THIS IS NOT ENDIAN-CLEAN
							theVert = *pVert++;
						}
						numLocalTris += (theCmd - 2);
						numLocalGLCmds += theCmd * 3;
						pCmd = (byte*)pVert;
					}
				}
				if ( numLocalTris )
					quit( "Unterminated GLCMDS list", -1);
			}
			else
			if ( !strcmp( theHeader.ident, FM_MESHNODESCHUNKNAME ) )
			{
				//	A MESH NODES chunk
				int		i;
				meshnode_t	theNode, *pNode;
				if ( theHeader.version != FM_MESHNODESCHUNKVER )
					quit("Invalid MESH NODES chunk version number",-1);
				pNode = (meshnode_t*)(pIntoModel + FM_HEADERSIZE);

				gpMeshNodes = pNode;

				for ( i=0; i<theModelHeader.meshnodecnt; i++ )
				{
					theNode = *pNode++;
					printf( "    GLCmds for '%s' start at index %4d, occupying %3d elements.\n",
						gNodeNames[i], theNode.glcmdstart, theNode.glcmdcnt );
					i = i;
#if	__NEW_SWORD_GLCMDS__
					if ( i == H2MODELNODE_STAFF )
						theNode.glcmdcnt =
							FixSwordGLCmds( &theModelHeader, gpGLCmds, theNode.glcmdstart, theNode.glcmdcnt );
#endif
				}
			}
			else
			if ( !strcmp( theHeader.ident, FM_SKELETONCHUNKNAME ) )
			{
				//	A SKELETON chunk
				skelhdr_t*		pSkelHeader;
				int				nJoints, nSkelVerts;
				int*			pJVerts;
				int				i, j;
				imp_point_t*	pFrameInfo;
				float			maxOrigin, minOrigin;

				if ( theHeader.version != FM_SKELETONCHUNKVER )
					quit("Invalid SKELETON chunk version number",-1);

				pSkelHeader = (skelhdr_t*)(pIntoModel + FM_HEADERSIZE);
				nJoints = pSkelHeader->numjoints;
				printf( "    Skeleton type %d with %d clusters\n", pSkelHeader->skeltype, nJoints );
				pJVerts = pSkelHeader->jointverts + nJoints;
#if	__WRITE_SKEL_DATA__
				{
					FILE*	skelFile;
					//	Find and output the frame-by-frame positioning
					//	data for the skelton
					for ( i=0; i<nJoints; i++ )
						pJVerts += pSkelHeader->jointverts[i];
					//	Now pJVerts points at the extended data flag. If the flag
					//	is zero, there is no extended data. Otherwise tehre is.
					if ( *pJVerts )
					{
						//	There is extended data. The extended data
						//	SHOULD be a size that we can calculate from the frame count
						//	at the start of the model, but the frame count is wrong in
						//	the Corvus and Kiera models so we calculate based on the
						//	original size of this chunk which is correct.
						++pJVerts;		//	Point at extended data
						if ( skelFile=fopen(__SKEL_FILE_NAME__,"wb") )
						{
							fwrite( (positioning_t*)pJVerts, theHeader.size - ((byte*)pJVerts - (byte*)pSkelHeader), 1, skelFile );
							fclose( skelFile );
							printf( "    wrote '%s'.\n", __SKEL_FILE_NAME__ );
						}
					}
				}
#else
				for ( i=0; i<nJoints; i++ )
				{
					nSkelVerts = pSkelHeader->jointverts[i];
					printf( "    Cluster %d contains %d new verteces\n", i, nSkelVerts );
					for ( j=0; j<nSkelVerts; j++ )
						printf( "%3d ", *pJVerts++ );
					printf("\n");
				}
				pFrameInfo = (void*)pJVerts;
				maxOrigin = -999999.;
				minOrigin =  999999.;
				for ( i=1; i<1000; i++ )
				{
					for ( j=0; j<3; j++ )
					{
						if ( pFrameInfo[i].origin.c[j] > maxOrigin )
							maxOrigin = pFrameInfo[i].origin.c[j];
						if ( pFrameInfo[i].origin.c[j] < minOrigin )
							minOrigin = pFrameInfo[i].origin.c[j];
					}
				}
				printf("Origin is in range (%f - %f)\n", minOrigin, maxOrigin);
#endif
				i = i;
			}
			else
			if ( !strcmp( theHeader.ident, FM_REFERENCESCHUNKNAME ) )
			{
				//	A REFERENCES chunk
				refheader_t		refHeader;
				if ( theHeader.version != FM_REFERENCESCHUNKVER )
					quit("Invalid REFERENCES chunk version number",-1);
				memcpy( &refHeader, pIntoModel + FM_HEADERSIZE, sizeof(refHeader) );
				gpFirstRef = (refskel_t*)(pIntoModel + FM_HEADERSIZE + sizeof(refHeader));
#if		__WRITE_REFS_DATA__
				{
					//	WARNING: ASSUMES WE HAVE ALREADY ENCOUNTERED FRAMES CHUNK, ETC
					//	EXPORT THE REFERENCES SECTION TO A FILE
					int		refSize;
					FILE*	fp2;
					refSize = sizeof(theHeader) + theHeader.size;
					fp2 = fopen( __REFS_FILE_NAME__, "wb" );
					fwrite( pIntoModel, refSize, 1, fp2 );
					fclose( fp2 );
					printf("    wrote '%s'.\n",__REFS_FILE_NAME__);
				}
#endif
#if		__CAPTURE_MOTION__
				{
					//	WARNING: ASSUMES WE HAVE ALREADY ENCOUNTERED FRAMES CHUNK
					CaptureMotion( &theModelHeader, gNumFooFrames, gpFirstFrame, gpFirstRef );
				}
#endif
#if	__FIND_BONE_LENGTHS__
				{
					//	WARNING: ASSUMES WE HAVE ALREADY ENCOUNTERED FRAMES CHUNK
					FindBoneLengths( &theModelHeader, gNumFooFrames, gpFirstFrame, gpFirstRef );
				}
#endif
#if (!__CAPTURE_MOTION__ && !__FIND_BONE_LENGTHS__ && !__WRITE_REFS_DATA__)
				{
					int				i, j, k;
					float			v;
					refheader_t		rHeader;
					refskel_t		theRefs[1400];
					memcpy( &rHeader, pIntoModel + FM_HEADERSIZE, sizeof(rHeader) );
					memcpy( theRefs, pIntoModel + FM_HEADERSIZE + sizeof(rHeader), sizeof(theRefs) );
					for ( i=0; i<1400; i++ )
					{
						for ( j=0; j<H2CORVUS_NUMREFPOINTS; j++ )
						{
							for ( k=0; k<3; k++ )
							{
								v = theRefs[i].refpoints[j].origin.c[k];
								if ( (v<0.0) || (v>255.0) )
									printf("  frame %d, reference %d, origin coordinate %d out of range (%8.4f)\n", i, j, k, v );
//								v = theRefs[i].refpoints[j].up.c[k];
//								if ( (v<0.0) || (v>255.0) )
//									printf("  frame %d, reference %d, up coordinate %d out of range (%8.4f)\n", i, j, k, v );
//								v = theRefs[i].refpoints[j].direction.c[k];
//								if ( (v<0.0) || (v>255.0) )
//									printf("  frame %d, reference %d, direction coordinate %d out of range (%8.4f)\n", i, j, k, v );
							}
						}
					}
					i = 0;
				}
#endif
			}
			else
				printf( "WARNING: Unrecognized chunk identifier: %s\n", theHeader.ident );

			//	Advance to next chunk
			pIntoModel += theHeader.size + FM_HEADERSIZE;
		}
		if ( pIntoModel != gpModelData + gModelSize )
			quit("Invalid (too big) chunk size in model",-1);
	}
#endif

#if	__EXPORT_DXF__
	{
		char*	pFileName;
		pFileName =
			DXF_WriteFile(
				(void*)0, (void*)0, __EXPORT_FRAME_NAME__, __EXPORT_MESHMASK__,
				gpModelHeader, gpMeshNodes, gpGLCmds, gpFirstFrame
			);
		printf("Wrote: %s\n", pFileName);
	}
#endif

#if	__EXPORT_OBJ__
	{
		//	Write out all the parts of the model to
		//	separate .OBJ files
		char*	pFileName;
#if	__EXPORT_PARTS__
		int		i;
		for ( i=1; i <65536L; i <<= 1 )
		{
			pFileName =
				OBJ_WriteFile(
					(void*)0, (void*)0, __EXPORT_FRAME_NAME__, i,
					gpModelHeader, gpMeshNodes, gpGLCmds, gpFirstFrame
				);
			printf("Wrote: %s\n", pFileName);
		}
#else
		pFileName =
			OBJ_WriteFile(
				(void*)0, (void*)0, __EXPORT_FRAME_NAME__, __EXPORT_MESHMASK__,
				gpModelHeader, gpMeshNodes, gpGLCmds, gpFirstFrame
			);
		printf("Wrote: %s\n", pFileName);
#endif
	}
#endif

#if	__WRITE_TWEAK_FILE__
	//	Kludge to write tweaked model
	{
//#if	__USE_NATIVE_IO__
//		if ( (gModelFID = _open( "tweak.fm", O_WRONLY+O_BINARY+O_TRUNC)) == -1 )
//			if ( (gModelFID = _open( "tweak.fm", O_WRONLY+O_BINARY+O_CREAT)) == -1 )
//				quit("Can't open 'tweak.fm' for writing", errno);
//		bytesRead = _write( gModelFID, gpModelData, gModelSize );
//		_close( gModelFID );
//		if ( bytesRead != gModelSize )
//			quit("Couldn't write model data", -1);
//#else
		FILE*	fout;
		if ( !(fout = fopen( "tweak.fm", "wb" )) )
			quit( "Can't open output file 'tweak.fm'", errno );
#if	__ADD_SKIN_NAME__
		fwrite( gpNewModelData, gNewModelSize, 1L, fout );
#else
		fwrite( gpModelData, gModelSize, 1L, fout );
#endif
		fclose( fout );
//#endif
		printf("Wrote: tweak.fm\n");
	}
	//	END KLUDGE
#endif

	//	Discard model
	free( gpModelData );

	//	Lint-friendly exit
	return(0);
}

