//
//	Create a heretic II player flexmodel form a set of body parts
//	and a motion capture data file.
//

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>

#include	"CondComp.h"
#include	"FlexModel.h"
#include	"bones.h"
#include	"mocap.h"
#include	"MoCapIO.h"
#include	"BodyPart.h"
#include	"SectIO.h"
#include	"glcmd.h"
#include	"h2bitmaps.h"
#include	"MatrixMath.h"
#include	"Normals.h"
#include	"ScaledFrame.h"

//	Local definitions

char*				gProgramName;
BodyContext_t		gBodyContext = { 0 };

mocapdata_t*		gpMocapData;

RawPlayerModel_t*	gpRawModel;
fmheader_t			gModelHeader;

//
//	Utils
//

void quit( char* pMsg, int errCode )
{
	if ( pMsg && *pMsg )
		printf( "\nFATAL: %s\n", pMsg );
	exit( errCode );
}

void usage( char* pMsg, int errCode )
{
	printf(
		"\n"
		"Program: %s\n"
		"Purpose: Generate Heretic II FlexModels from motion capture data\n"
		"\n"
		"Usage:\n"
		"%s\n",
		gProgramName, gProgramName
	);
	quit( pMsg, errCode );
}

//
//	Build and output a preliminary model header for
//	the flexmodel. Returns NIL pointer if error,
//	else file pointer.
//

void*
WritePMHeader1( FILE* outFile, fmheader_t* pHeader, RawPlayerModel_t* pModel, mocapdata_t* pMotion )
{
	header_t	h;
	long		rc;
	int			i;

	//	Construct and write the generic chunk header
	memset( &h, 0, sizeof(h) );
	strcpy( h.ident, FM_HEADERCHUNKNAME );
	h.version = FM_HEADERCHUNKVER;
	h.size = sizeof( fmheader_t );
	if ( !(rc = fwrite(&h, sizeof(h), 1, outFile)) )
		return( (void*)0 );

	//	Fill in what we can now about the model header
	pHeader->skinwidth = 256;
	pHeader->skinheight = 256;
	pHeader->framesize = 0;			//	WARNING: DUMMY VALUE - CALCULATE REAL VALUE FROM VERTEX COUNT (vert3dcnt * sizeof(vertex3d_t))
	pHeader->skincnt = SKIN_NUM_SKINS;
	pHeader->vert3dcnt = 0;			//	WARNING: DUMMY VALUE - CALCULATE REAL VALUE AS SUM OVER ALL BODY PARTS
	pHeader->vertuvcnt = 0;			//	WARNING: DUMMY VALUE - CALCULATE REAL VALUE AS SUM OVER ALL BODY PARTS
	pHeader->tricnt = 0;			//	WARNING: DUMMY VALUE - CALCULATE REAL VALUE AS SUM OVER ALL BODY PARTS
	pHeader->glcmdcnt = 0;			//	WARNING: DUMMY VALUE - CALCULATE REAL VALUE AS SUM OVER ALL BODY PARTS
	pHeader->framecnt = pMotion->fileHdr.framecnt;
	pHeader->meshnodecnt = H2NUM_MESHNODES;

	//	Fill in more stuff by summing over all body parts and all seams
	//	Note that body parts contribute verteces, texture verteces, and
	//	triangles, while seams contribute only triangles and texture
	//	verteces.
	for ( i=0; i<NUM_BODY_PARTS; i++ )
	{
		if ( pModel->parts[i] )
		{
			//	Fix up vert3dcnt, vertuvcnt, tricnt
			pHeader->vert3dcnt += pModel->parts[i]->vert3DCount;
			pHeader->vertuvcnt += pModel->parts[i]->vertUVCount;
			pHeader->tricnt    += pModel->parts[i]->trisCount;
		}
	}
	//	Fix up framesize
	pHeader->framesize = pHeader->vert3dcnt * sizeof(vertex3d_t) + sizeof(framehdr_t);	//	Now we know frame size
	for ( i=0; i<SEAM_NUM_SEAMS; i++ )
	{
		if ( pModel->seams[i] )
		{
			//	Fix up vert3dcnt, vertuvcnt, tricnt
			pHeader->vertuvcnt += pModel->seams[i]->vertUVCount;
			pHeader->tricnt    += pModel->seams[i]->trisCount;
		}
	}

	//	OK, now everything should be right except for the glcmdcnt
	if ( !(rc = fwrite(pHeader, sizeof(fmheader_t), 1, outFile)) )
		return( (void*)0 );

	return	outFile;

}

//
//	Output a dummy skin names chunk to the flexmodel.
//	Heretic II more or less ignores these skin names
//	on player models, but there should be some there
//	to make qmview and similar tools happy.
//
//	Returns NIL or file pointer
//

static char*
skinnames[SKIN_NUM_SKINS] =
{
	SKIN_NAME_1, SKIN_NAME_2
};

void*
WritePMDummySkins( FILE* outFile, fmheader_t* pHeader, RawPlayerModel_t* pModel, mocapdata_t* pMotion )
{
	header_t	h;
	long		rc;
	int			i;
	skinpath_t	n;

	//	Construct and write the generic chunk header
	memset( &h, 0, sizeof(h) );
	strcpy( h.ident, FM_SKINCHUNKNAME );
	h.version = FM_SKINCHUNKVER;
	h.size = SKIN_NUM_SKINS * sizeof( skinpath_t );
	if ( !(rc = fwrite(&h, sizeof(h), 1, outFile)) )
		return( (void*)0 );

	//	Make sure model header is correct
	pHeader->skincnt = SKIN_NUM_SKINS;

	//	Write the skin names
	for ( i=0; i<SKIN_NUM_SKINS; i++ )
	{
		memset( &n, 0, sizeof(n) );
		strcpy( n.path, skinnames[i] );
		if ( !(rc = fwrite(&n, sizeof(n), 1, outFile)) )
			return( (void*)0 );
	}

	return	outFile;

}

//
//	Output a UV coordinates chunk to the flexmodel.
//	The coordinates are assumed to already be normalized
//	to the range 0..255.
//
//	Returns NIL or file pointer
//

void*
WritePMSTCoords( FILE* outFile, fmheader_t* pHeader, RawPlayerModel_t* pModel, mocapdata_t* pMotion )
{
	header_t	h;
	long		rc;
	int			i, j;
	stcoord_t	c;

	//	Construct and write the generic chunk header
	memset( &h, 0, sizeof(h) );
	strcpy( h.ident, FM_STCOORDCHUNKNAME );
	h.version = FM_STCOORDCHUNKVER;
	h.size = pHeader->vertuvcnt * sizeof(stcoord_t);
	if ( !(rc = fwrite(&h, sizeof(h), 1, outFile)) )
		return( (void*)0 );

	//	Write out the texture coordinates for all
	//	body parts and seams.
	//
	//	NOTE: SHOULD BUILD TABLE OF INDEX OF FIRST
	//	STCOORD # FOR EACH BODY PART / SEAM FOR LATER
	for ( i=0; i<NUM_BODY_PARTS; i++ )
	{
		if ( pModel->parts[i] && pModel->parts[i]->pVertUV )
		{
			for ( j=0; j<pModel->parts[i]->vertUVCount; j++ )
			{
				c.u = pModel->parts[i]->pVertUV[j].c[0];
				c.v = pModel->parts[i]->pVertUV[j].c[1];
				if ( !(rc = fwrite(&c, sizeof(stcoord_t), 1, outFile)) )
					return( (void*)0 );
			}
		}
	}
	for ( i=0; i<SEAM_NUM_SEAMS; i++ )
	{
		if ( pModel->seams[i] && pModel->seams[i]->pVertUV )
		{
			for ( j=0; j<pModel->seams[i]->vertUVCount; j++ )
			{
				c.u = pModel->seams[i]->pVertUV[j].c[0];
				c.v = pModel->seams[i]->pVertUV[j].c[1];
				if ( !(rc = fwrite(&c, sizeof(stcoord_t), 1, outFile)) )
					return( (void*)0 );
			}
		}
	}

	return	outFile;

}

//
//	Output a triangles chunk to the flexmodel.
//
//	Returns NIL or file pointer
//

void*
WritePMTris( FILE* outFile, fmheader_t* pHeader, RawPlayerModel_t* pModel, mocapdata_t* pMotion )
{
	header_t	h;
	long		rc;
	int			i, j, k;
	triangle_t	t;
	int			vB, xB;

	//	Construct and write the generic chunk header
	memset( &h, 0, sizeof(h) );
	strcpy( h.ident, FM_TRISCHUNKNAME );
	h.version = FM_TRISCHUNKVER;
	h.size = pHeader->tricnt * sizeof(triangle_t);
	if ( !(rc = fwrite(&h, sizeof(h), 1, outFile)) )
		return( (void*)0 );

	//	Write out the triangles for all
	//	body parts and seams.
	for ( i=0; i<NUM_BODY_PARTS; i++ )
	{
		if ( pModel->parts[i] && pModel->parts[i]->pTris && pModel->parts[i]->pTrisTex )
		{
			vB = pModel->parts[i]->vertBase;
			xB = pModel->parts[i]->texBase;
			for ( j=0; j<pModel->parts[i]->trisCount; j++ )
			{
				for ( k=0; k<3; k++ )
				{
					t.index_3d[k] = vB + pModel->parts[i]->pTris[j].v[k];
					t.index_uv[k] = xB + pModel->parts[i]->pTrisTex[j].v[k];
				}
				if ( !(rc = fwrite(&t, sizeof(triangle_t), 1, outFile)) )
					return( (void*)0 );
			}
		}
	}
	for ( i=0; i<SEAM_NUM_SEAMS; i++ )
	{
		if ( pModel->seams[i] && pModel->seams[i]->pTris && pModel->seams[i]->pTrisTex )
		{
			vB = pModel->seams[i]->vertBase;
			xB = pModel->seams[i]->texBase;
			for ( j=0; j<pModel->seams[i]->trisCount; j++ )
			{
				for ( k=0; k<3; k++ )
				{
					t.index_3d[k] = vB + pModel->seams[i]->pTris[j].v[k];
					t.index_uv[k] = xB + pModel->seams[i]->pTrisTex[j].v[k];
				}
				if ( !(rc = fwrite(&t, sizeof(triangle_t), 1, outFile)) )
					return( (void*)0 );
			}
		}
	}

	return	outFile;

}

//
//	Build, scale and output all the animation frames of the player model,
//	The animation frames include vertex coordinates and "light normals"
//	for each 3D vertex given a pose of the model.
//
//	Each frame uses the same set of verteces, which are rotated into place
//	based on motion capture data. A set of "reference" and "skeleton" 
//	verteces are also rotated and scaled along with each frame, and output
//	to a scratch file so they can be used to build the reference and skeleton
//	chunks of the model later.
//
//	Returns NIL pointer if error, else file pointer.
//

void*
WritePMFrames( FILE* outFile, fmheader_t* pHeader, RawPlayerModel_t* pModel, mocapdata_t* pMotion )
{
	header_t		h;
	long			rc;
	frame_t*		pFrame;
	int				i, frame, part, vert, nVerts, vB, useAlt;
	vec3dbl_t		rotX, rotY, rotZ, trans;
	vec3dbl_t*		pFrameVerts;
	unsigned char*	pNormals;
	int				proglevel;
	FILE*			refFile;
	refinfo_t		refskel;
	vec3dbl_t		orgPoint, dirPoint, upPoint;
	char*			pName;
	char			progChar;

	printf("generating frame, reference and skeleton info...\n");
	//	Save reference information to a scratch file as we go
	if ( !(refFile=fopen(REFSKEL_FILE_NAME,"wb")) )
		return	(void*)0;

	//	Construct and write the generic frames chunk header
	memset( &h, 0, sizeof(h) );
	strcpy( h.ident, FM_FRAMESCHUNKNAME );
	h.version = FM_FRAMESCHUNKVER;
	h.size = pHeader->framesize * pHeader->framecnt;
	if ( !(rc = fwrite(&h, sizeof(h), 1, outFile)) )
	{
		fclose( refFile );
		return( (void*)0 );
	}

	//	Allocate frame buffer
	if ( !(pFrame=((frame_t*)(malloc(pHeader->framesize)))) )
	{
		fclose( refFile );
		return	(void*)0;
	}

	//	Allocate raw frame vertex buffer
	nVerts = pHeader->vert3dcnt;
	if ( !(pFrameVerts=malloc(nVerts*sizeof(vec3dbl_t))) )
	{
		free( pFrame );
		fclose( refFile );
		return	(void*)0;
	}

	proglevel = 1;
	for ( frame=0; frame<pHeader->framecnt; frame++ )
	{
		progChar = '.';
		//	For each frame...
		//	Set up the frame name
		memset( pFrame->header.name, 0, sizeof(pFrame->header.name) );
		strcpy( pFrame->header.name, pMotion->frames[frame].name.name );

		if ( !strcmp(pFrame->header.name,"conjure11") )
			frame = frame;

		//	Calculate frame data - PARTS AND SEAMS AND REFERENCE/SKELETON POINTS
		//	Perform all rotations and translations

		//	First, do the verteces of the mesh of each part
		//	While doing this, also do the reference / skeleton points for the part
		for ( part=0; part<BONE_NUM_BONES; part++ )
		{
			//	Check for alternate geometry
			useAlt = 0;
			for ( i=0; i<MAX_ALT_GEOMETRY_FRAMES; i++ )
			{
				pName = pModel->parts[part]->altFrames[i];
				if ( !(*pName) )
					break;		//	End of list
				if ( !strcmp(pName,pFrame->header.name) )
				{
					useAlt = 1;
					progChar = 'A';
					break;		//	Found match
				}
			}
			//	Set up common for all verteces in this part
			rotX.c[0] = pMotion->frames[frame].rotXYZ[part].c[0];
			rotX.c[1] = 0.;
			rotX.c[2] = 0.;
			rotY.c[0] = 0.;
			rotY.c[1] = pMotion->frames[frame].rotXYZ[part].c[1];
			rotY.c[2] = 0.;
			rotZ.c[0] = 0.;
			rotZ.c[1] = 0.;
			rotZ.c[2] = pMotion->frames[frame].rotXYZ[part].c[2];
			trans.c[0] = pMotion->frames[frame].transXYZ[part].c[0];
			trans.c[1] = pMotion->frames[frame].transXYZ[part].c[1];
			trans.c[2] = pMotion->frames[frame].transXYZ[part].c[2];
			//	Process body part - always true if not doing reference data
			if (pModel->parts[part])
			{
				//	Note that if a body part has not geometry, we do nothing
				nVerts = pModel->parts[part]->vert3DCount;
				vB     = pModel->parts[part]->vertBase;
				for ( vert=0; vert<nVerts; vert++ )
				{
					//	To reconstruct the motion, the order of application of the various
					//	elements to a given bone MUST be:
					//
					//	1) Apply X-axis rotation
					//	2) Apply Z-axis rotation
					//	3) Apply Y-axis rotation
					//	4) Apply translations in any order
					//	Get the vertex
					if ( useAlt )
					{
						for ( i=0; i<3; i++ )
							pFrameVerts[vB+vert].c[i] = pModel->parts[part]->pAltVert3D[vert].c[i];
					}
					else
					{
						for ( i=0; i<3; i++ )
							pFrameVerts[vB+vert].c[i] = pModel->parts[part]->pVert3D[vert].c[i];
					}
					M3DRotate( &(pFrameVerts[vB+vert]), &rotX );
					M3DRotate( &(pFrameVerts[vB+vert]), &rotZ );
					M3DRotate( &(pFrameVerts[vB+vert]), &rotY );
					M3DTranslate( &(pFrameVerts[vB+vert]), &trans );
				}
			}
			//	Process this body part's reference points
			for ( i=0; i<3; i++ )
			{
				orgPoint.c[i] = pModel->parts[part]->refPoint.origin.c[i];
				dirPoint.c[i] = pModel->parts[part]->refPoint.direction.c[i];
				upPoint.c[i]  = pModel->parts[part]->refPoint.up.c[i];
			}
			M3DRotate( &orgPoint, &rotX );	M3DRotate( &orgPoint, &rotZ );	M3DRotate( &orgPoint, &rotY );
			M3DTranslate( &orgPoint, &trans );
			//
			M3DRotate( &dirPoint, &rotX );	M3DRotate( &dirPoint, &rotZ );	M3DRotate( &dirPoint, &rotY );
			M3DTranslate( &dirPoint, &trans );
			//
			M3DRotate( &upPoint, &rotX );	M3DRotate( &upPoint, &rotZ );	M3DRotate( &upPoint, &rotY );
			M3DTranslate( &upPoint, &trans );
			for ( i=0; i<3; i++ )
			{
				refskel.refs[part].origin.c[i] = (float)(orgPoint.c[i]);
				refskel.refs[part].direction.c[i] = (float)(dirPoint.c[i]);
				refskel.refs[part].up.c[i] = (float)(upPoint.c[i]);
			}
		}

		//	Calculate light normal indeces for the resulting mesh verteces
		if ( !(pNormals = MakeLightNormals( pHeader, pModel, pFrameVerts )) )
		{
			free( pFrame );
			free( pFrameVerts );
			fclose( refFile );
			return( (void*)0 );
		}

		//	Calculate min and max coords in X, Y, Z
		//	Calculate scale and translation to make all coords 0..255
		//	Scale / translate all mesh verteces, and the reference / skeleton points
		MakeScaledFrame( pFrame, pHeader, pFrameVerts, pNormals, &refskel );	//	No reference points to factor in

		//	Output the frame
		printf("%c",progChar);
		if ( !(rc = fwrite(pFrame, pHeader->framesize, 1, outFile)) )
		{
			free( pFrame );
			free( pFrameVerts );
			free( pNormals );
			fclose( refFile );
			return( (void*)0 );
		}

		//	Output the reference / skeleton points for this frame
		if ( !(rc = fwrite(&refskel, sizeof(refskel), 1, refFile)) )
		{
			free( pFrame );
			free( pFrameVerts );
			free( pNormals );
			fclose( refFile );
			return( (void*)0 );
		}

		//	Progress bar stuff
		if ( frame >= ((pHeader->framecnt-1)*proglevel)/10 )
		{
			printf("%d%c\n",proglevel*10,'%');
			++proglevel;
		}

	}

	//	Release memory and exit
	free( pFrame );
	free( pFrameVerts );
	free( pNormals );
	fclose( refFile );

	return	outFile;

}


//
//	Output GLCMD drawing data for the model.
//	Output mesh node data for the model.
//
//	Returns NIL pointer if error, else file pointer.
//

#define	MAX_PARTS_PER_NODE	5
//	Body parts in each mesh node
static int
part2mesh[H2NUM_MESHNODES][MAX_PARTS_PER_NODE+1] =
{
	//	NOTE: The front torso and back torso are problematic, because
	//	the motion capture data does not distinguish between then and
	//	they are no considered  separate bones (maybe they should be,
	//	for more realistic breathing, etc.). Here, we have made
	//	arbitrary assignments of parts to the front and back torso.
	{	4,	BONE_HIPS_LEFT,			BONE_HIPS_RIGHT,		BONE_LOWER_TORSO,		BONE_UPPER_TORSO_SPINE },						//	0: Front torso, shoulders, pelvis
	{	5,	BONE_HIPS_STAFF,		BONE_HIPS_SPINE,		BONE_LEFT_UPPER_TORSO,	BONE_RIGHT_UPPER_TORSO,	BONE_SHOULDER_BOW },	//	1: Back torso, shoulders, pelvis
	{	1,	BONE_STORED_STAFF },																									//	2: The thingy on the right hip when the staff is stowed
	{	1,	BONE_STORED_BOW },																										//	3: The bow, slung over the shoulder
	{	4,	BONE_ARMOR_LEFT,		BONE_ARMOR_RIGHT,		BONE_RIGHT_PAD,			BONE_LEFT_PAD },								//	4: Armor
	{	2,	BONE_RIGHT_UPPER_ARM,	BONE_RIGHT_LOWER_ARM },																			//	5: The right upper and lower arm (no hand)
	{	1,	BONE_RIGHT_HAND_OPEN },																									//	6: The right hand, empty and open
	{	1,	BONE_RIGHT_HAND_CLOSED },																								//	7: The right hand, gripping the staff thingy (no staff)
	{	2,	BONE_RIGHT_STAFF_BASE, BONE_RIGHT_STAFF_TIP },																											//	8: The Durnwood Staff
	{	2,	BONE_RIGHT_HELL_BASE, BONE_RIGHT_HELL_HEAD },																										//	9: The Hellstaff
	{	2,	BONE_LEFT_UPPER_ARM,	BONE_LEFT_LOWER_ARM },																			//	A: The left upper and lower arm (no hand)
	{	1,	BONE_LEFT_HAND_OPEN },																									//	B: The left hand, empty and open
	{	2,	BONE_LEFT_HAND_CLOSED,	BONE_BOW },																						//	C: The left hand gripping the entire bow
	{	3,	BONE_RIGHT_UPPER_LEG,	BONE_RIGHT_LOWER_LEG,	BONE_RIGHT_FOOT },														//	D: The right upper leg, lower leg, and foot
	{	3,	BONE_LEFT_UPPER_LEG,	BONE_LEFT_LOWER_LEG,	BONE_LEFT_FOOT },														//	E: The left upper leg, lower leg, and bag thingy on the hip
	{	2,	BONE_NECK,				BONE_HEAD }																						//	F: The head and hair
};
//	Seams in each meshnode
//	WARNING: NEED SEAM PROCESSING HERE
void*
WritePMGLCmdsAndNodes( FILE* outFile, fmheader_t* pHeader, RawPlayerModel_t* pModel, mocapdata_t* pMotion )
{
	header_t		h;
	long			rc;
	GLCmdInfo_t*	pGLs;
	meshnode_t		mesh[H2NUM_MESHNODES];
	h2bitmap_t		triMap, vertMap;
	int				j, vcnt, node, part, cmd, n, ccnt, cmdBase;
	long			hdrLoc, endLoc;
	int				glSize, glNodeSize, glStart;

	//
	//	Write the glcmds chunk
	//

	//	Construct and write the generic chunk header for GLCMDS
	//	Use a dummy size since we haven't figured the size out yet.
	hdrLoc = ftell( outFile );							//	Remember file position to update header later
	memset( &h, 0, sizeof(h) );
	strcpy( h.ident, FM_GLCMDSCHUNKNAME );
	h.version = FM_GLCMDSCHUNKVER;
	h.size = 0;											//	WARNING: UPDATE THIS LATER
	if ( !(rc = fwrite(&h, sizeof(h), 1, outFile)) )
		return	(void*)0;

	//	Get raw GLCmds by body part
	//	WARNING: THIS DOES NOT HANDLE SEAMS CORRECTLY YET
	if ( !(pGLs = MakeGLCmds( pHeader, pModel )) )
		return	pGLs;

	//	Write out GL Cmds by mesh node, keeping track of the start and count
	//	in mesh[] and the total size of output in glSize;
	glSize = 0;
	glStart = 0;
	for ( node=0; node<H2NUM_MESHNODES; node++ )
	{
		glNodeSize = 0;
		n = part2mesh[node][0];			//	# body parts to aggregate
		for ( j=1; j<=n; j++ )
		{
			part = part2mesh[node][j];
			ccnt    = pGLs->numGLCmds[ part ];
			cmdBase = pGLs->startGLCmds[ part ];
			cmd  = 0;
			while ( cmd < ccnt )
			{
				//	Output GLCMD command value
				vcnt = pGLs->cmds[cmdBase+cmd].index_3d;
				if ( !(rc = fwrite( &vcnt, sizeof(int), 1, outFile )) )
				{
					free( pGLs );
					return( (void*)0 );
				}
				glNodeSize += ( sizeof(int) / sizeof(int) );
				++cmd;
				//	Output associated vertex data
				if ( vcnt < 0 )
					vcnt = -vcnt;
				if ( !(rc = fwrite( &(pGLs->cmds[cmdBase+cmd]), sizeof(glvert_t), vcnt, outFile )) )
				{
					free( pGLs );
					return	(void*)0;
				}
				glNodeSize += vcnt * ( sizeof(glvert_t) / sizeof(int) );
				cmd += vcnt;
			}
		}
		//	Write trailing zero on this mesh node's GLCMDS
		vcnt = 0;
		if ( !(rc = fwrite( &vcnt, sizeof(int), 1, outFile )) )
		{
			free( pGLs );
			return( (void*)0 );
		}
		glNodeSize += ( sizeof(int) / sizeof(int) );
		//	End-of-node housekeeping
		mesh[node].glcmdstart = glStart;		//	We'll need this for mesh nodes chunk
		mesh[node].glcmdcnt   = glNodeSize;		//	We'll need this for mesh nodes chunk
		glStart += glNodeSize;
		glSize  += glNodeSize;
	}

	//	Fix up the header of the GLCMDS section now that we know how
	//	many 32-bit units were generated as glcmds.
	endLoc = ftell( outFile );
	fseek( outFile, hdrLoc, SEEK_SET );
	h.size = glSize * sizeof(int);				//	Correct size of this chunk
	if ( !(rc = fwrite(&h, sizeof(h), 1, outFile)) )
	{
		free( pGLs );
		return( (void*)0 );
	}
	fseek( outFile, endLoc, SEEK_SET );

	//	Fix up the glcmdcnt field of the model header
	pHeader->glcmdcnt = glSize;

	//	Release the glcmds working memory
	free( pGLs );

	//
	//	Now write the MESHNODES chunk
	//

	//	Construct and write the generic chunk header for MESHNODES.
	memset( &h, 0, sizeof(h) );
	strcpy( h.ident, FM_MESHNODESCHUNKNAME );
	h.version = FM_MESHNODESCHUNKVER;
	h.size = sizeof(mesh);

	if ( !(rc = fwrite(&h, sizeof(h), 1, outFile)) )
		return	(void*)0;

	//	For each meshnode, construct the bitmaps for triangle and vertex membership
	triMap.numbits = MAX_FM_TRIANGLES;
	triMap.mapbytes = MAX_FM_TRIANGLES >> 3;
	vertMap.numbits = MAX_FM_VERTS;
	vertMap.mapbytes = MAX_FM_VERTS >> 3;
	for ( node=0; node<H2NUM_MESHNODES; node++ )
	{
		//	WARNING: HIGHLY IMPLEMENTATION DEPENDENT.
		//	IT IS KNOWN FOR THIS IMPLEMENTATION THAT THE
		//	FORMAT OF BITMAPS IN THE .FM FILE IS THE SAME
		//	AS THE FORMAT USED BY THE BITMAP ROUTINES HERE.
		//	ALSO, WE PEEK A LOT INTO BITMAP INTERNALS TO
		//	TRICK THE BITMAP ROUTINES HERE INTO DIRECTLY
		//	UPDATING THE BITMAPS IN MESH[]
		//	Clear out the bit maps for this mesh node
		triMap.bitdata = mesh[node].trimap;
		vertMap.bitdata = mesh[node].vertmap;
		ClrBitMap( &triMap, 0, triMap.numbits );
		ClrBitMap( &vertMap, 0, vertMap.numbits );
		n = part2mesh[node][0];			//	# body parts to aggregate
		for ( j=1; j<=n; j++ )
		{
			part = part2mesh[node][j];
			if ( pModel->parts[part] )
			{
				//	Mark the triangles used by this body part
				SetBitMap( &triMap, pModel->parts[part]->triBase, pModel->parts[part]->trisCount );
				//	Mark the verteces used by this body part
				SetBitMap( &vertMap, pModel->parts[part]->vertBase, pModel->parts[part]->vert3DCount );
			}
		}
	}

	//	Write the meshnode data
	if ( !(rc = fwrite(&mesh, sizeof(mesh), 1, outFile)) )
		return	(void*)0;

	//	All done
	return	outFile;
}

//
//	Output skeletal data for the model.
//
//	The data has been pre-computed during generation of the "frames"
//	section of the model, and stored in the file REFSKEL_FILE_NAME.
//	We read in this file, and select the necessary reference points
//	and output them.
//
//	Returns NIL pointer if error, else file pointer.
//
#define	MAX_PARTS_PER_SKEL	24
static int
skel2parts[H2CORVUS_NUMJOINTS][MAX_PARTS_PER_SKEL+1] =
{
	//	What body parts are in each skeletal node.
	//	NOTE: These are in the order in which vertex clusters are
	//	stored in the skeleton chunk
	{	1,
		BONE_HEAD },
	{	23,
		BONE_NECK,				BONE_UPPER_TORSO_SPINE,	BONE_SHOULDER_BOW,		BONE_STORED_BOW,
		BONE_LEFT_UPPER_TORSO,	BONE_LEFT_UPPER_ARM,	BONE_LEFT_LOWER_ARM,	BONE_LEFT_HAND_OPEN,
		BONE_LEFT_HAND_CLOSED,	BONE_BOW,				BONE_RIGHT_UPPER_TORSO,	BONE_RIGHT_UPPER_ARM,
		BONE_RIGHT_LOWER_ARM,	BONE_RIGHT_HAND_OPEN,	BONE_RIGHT_HAND_CLOSED,	BONE_RIGHT_STAFF_BASE,
		BONE_RIGHT_STAFF_TIP,	BONE_RIGHT_HELL_BASE,	BONE_RIGHT_HELL_HEAD,	BONE_ARMOR_LEFT,
		BONE_LEFT_PAD,			BONE_ARMOR_RIGHT,		BONE_RIGHT_PAD },
	{	1,
		BONE_LOWER_TORSO }
};
static int
skelRefBone[ H2CORVUS_NUMJOINTS ] =
{
	//	What body part's reference to use for positioning data for
	//	each skeletal node.
	//	NOTE: These are in the order in which skeletal reference
	//	points are stored in the skeleton chunk - NOT necessarily
	//	the same as the order of vertex clusters.
		BONE_HIPS_SPINE,		BONE_LOWER_TORSO,		BONE_NECK
};
void*
WritePMSkels( FILE* outFile, fmheader_t* pHeader, RawPlayerModel_t* pModel, mocapdata_t* pMotion )
{
	header_t		h;
	long			rc;
	int				sh[1+1+H2CORVUS_NUMJOINTS];
	int				joint, part, vert, element, nVerts, nParts, vB;
	int				vertIdx;
	long			startPos, endPos;
	long			skelSize;
	FILE*			skelFile;
	refinfo_t*		skelData;
	int				frame;

	//	Open the refskel data file and get its length
	printf("importing generated skeleton data...\n");
	if ( !(skelFile=fopen(REFSKEL_FILE_NAME,"rb")) )
		return	(void*)0;
	fseek(skelFile, 0, SEEK_END);
	skelSize = ftell(skelFile);
	fseek(skelFile, 0, SEEK_SET);

	//	Grab a big hunk of memory to read the reference and skeleton into
	if ( !(skelData=malloc(skelSize)) )
	{
		fclose(skelFile);
		return	(void*)0;
	}

	//	Read in the reference and skeleton data
	rc = fread( skelData, skelSize, 1, skelFile );
	fclose( skelFile );
	if ( !rc )
	{
		free( skelData );
		return	(void*)0;
	}

	//	Construct the specific header for the skeleton chunk, using
	//	standard values and a computed vertex count.
	sh[0] = FM_SKELTYPE_CORVUS;			//	Standard Corvus / Kiera skeltype
	sh[1] = H2CORVUS_NUMJOINTS;			//	Standard Corvus / Kiera joint count

	//	Figure out how many verteces are part of each joint
	for ( joint=0; joint<H2CORVUS_NUMJOINTS; joint++ )
	{
		nParts = skel2parts[joint][0];
		nVerts = 0;
		for ( element=0; element<nParts; element++ )
		{
			part = skel2parts[joint][element+1];
			if ( pModel->parts[part] )
				nVerts += pModel->parts[part]->vert3DCount;
		}
		sh[2+joint] = nVerts;
	}

	//	Construct and write the generic chunk header for skeletons,
	//	including positioning information.
	memset( &h, 0, sizeof(h) );
	strcpy( h.ident, FM_SKELETONCHUNKNAME );
	h.version = FM_SKELETONCHUNKVER;
	//	Size calculation...
	h.size = (2+H2CORVUS_NUMJOINTS)*sizeof(int);		//	Size of header without per-frame information
	for ( element=0; element<H2CORVUS_NUMJOINTS; element++ )
		h.size += sh[2+element]*sizeof(int);
	h.size += 1*sizeof(int);							//	For the trailing 1 indicating per-frame information follows
	h.size += H2CORVUS_NUMJOINTS * pHeader->framecnt * sizeof(positioning_t);	//	For the per-frame data
	//	End of size calculation
	if ( !(rc = fwrite(&h, sizeof(h), 1, outFile)) )
		return	(void*)0;

	//	DEBUGGING AID
	startPos = ftell(outFile);

	//	Write the specific header for a skeleton section
	if ( !(rc = fwrite(sh, (2+H2CORVUS_NUMJOINTS)*sizeof(int), 1, outFile)) )
		return	(void*)0;

	//	Now write out the vertex indeces for each of the joints
	for ( joint=0; joint<H2CORVUS_NUMJOINTS; joint++ )
	{
		nParts = skel2parts[joint][0];
		for ( element=0; element<nParts; element++ )
		{
			part = skel2parts[joint][element+1];
			if ( pModel->parts[part] )
			{
				//	All verteces in this part are part of the skeleton.
				//	We just need to write out the numbers from vertBase
				//	to (vertBase+vert3DCount-1)
				nVerts = pModel->parts[part]->vert3DCount;
				vB     = pModel->parts[part]->vertBase;
				for ( vertIdx=vB; vertIdx<vB+nVerts; vertIdx++ )
				{
					if ( !(rc = fwrite( &vertIdx, sizeof(int), 1, outFile)) )
						return	(void*)0;
				}
			}
		}
	}

	//	Write the sentinel value 1 to tell the game that
	//	per-frame positioning data follows.
	vert = 1;
	if ( !(rc = fwrite( &vert, sizeof(int), 1, outFile)) )
	{
		free( skelData );
		return	(void*)0;
	}

	//	Write the extended skeleton data
	for ( frame = 0; frame < pHeader->framecnt; frame++ )
	{
		for ( joint=0; joint<H2CORVUS_NUMJOINTS; joint++ )
		{
			rc = fwrite( &(skelData[frame].refs[skelRefBone[joint]]), sizeof(imp_point_t), 1, outFile );
			if ( !rc )
				return	(void*)0;
		}
	}
	free( skelData );

	//	DEBUGGING AID
	endPos = ftell(outFile);
	vert = vert;

	//	WARNING; DUMMY CODE
	return	outFile;
}


//
//	Output a references section for the model.
//
//	The reference data must be scaled and translated
//	in the same way as the frame data.
//
//	Since this tool assumes the same model bone lengths
//	and uses the same animations as Corvus, we can just
//	steal and clone the references section from Corvus..
//
//	Or, since the reference points have no associated
//	geometry, we can just rotate a single point equal
//	to the bone length at the same time as we're rotating
//	the model geometry (see frames section).
//
//	Either way, the reference data is precomputed and
//	stored in a file called REFTEMP_FILE_NAME. All
//	this routine does is load in that file, and append
//	it to the output file.
// 
//	Returns NIL pointer if error, else file pointer.
//
static int
refRefBone[ H2CORVUS_NUMREFPOINTS ] =
{
	//	What body part's reference to use for positioning data for
	//	each reference point
	BONE_RIGHT_HAND_CLOSED,	BONE_LEFT_HAND_CLOSED,		BONE_RIGHT_FOOT,			BONE_LEFT_FOOT,
	BONE_RIGHT_STAFF_BASE,	BONE_RIGHT_STAFF_TIP,		BONE_RIGHT_HELL_HEAD
};
void*
WritePMRefs( FILE* outFile, fmheader_t* pHeader, RawPlayerModel_t* pModel, mocapdata_t* pMotion )
{
	header_t		h;
	refheader_t		rh;
	refskel_t		frameRefs;
	long			rc;
	long			refSize;
	FILE*			refFile;
	refinfo_t*		refData;
	int				frame, ref;

	//	Open the reference / skeleton scratch file and get its length
	printf("importing generated reference data...\n");
	if ( !(refFile=fopen(REFSKEL_FILE_NAME,"rb")) )
		return	(void*)0;
	fseek(refFile, 0, SEEK_END);
	refSize = ftell(refFile);
	fseek(refFile, 0, SEEK_SET);

	//	Grab a big hunk of memory to read the references into
	if ( !(refData=malloc(refSize)) )
	{
		fclose(refFile);
		return	(void*)0;
	}

	//	Read in the data
	rc = fread( refData, refSize, 1, refFile );
	fclose( refFile );
	if ( !rc )
	{
		free( refData );
		return	(void*)0;
	}

	//	Construct and write the generic chunk header for references
	memset( &h, 0, sizeof(h) );
	strcpy( h.ident, FM_REFERENCESCHUNKNAME );
	h.version = FM_REFERENCESCHUNKVER;
	h.size = sizeof(rh) + sizeof(refskel_t)*pHeader->framecnt;
	if ( !(rc = fwrite(&h, sizeof(h), 1, outFile)) )
	{
		free( refData );
		return	(void*)0;
	}

	//	Construct and write the specific header for the references chunk
	rh.reftype = FM_REFTYPE_CORVUS;
	rh.idunno  = FM_IDUNNO_CORVUS;
	if ( !(rc = fwrite(&rh, sizeof(rh), 1, outFile)) )
	{
		free( refData );
		return	(void*)0;
	}

	//	Write the frame-by-frame reference data
	for ( frame = 0; frame < pHeader->framecnt; frame++ )
	{
		for ( ref=0; ref<H2CORVUS_NUMREFPOINTS; ref++ )
			frameRefs.refpoints[ref] = refData[frame].refs[refRefBone[ref]];
		rc = fwrite( &frameRefs, sizeof(refskel_t), 1, outFile );
		if ( !rc )
		{
			free( refData );
			return	(void*)0;
		}
	}
	free( refData );

	//	All done
	return	outFile;
}


//
//	Output a revised model header for he flexmodel.
//	This should be same as originally-written header,
//	except the glcmdcnt should be correct now.
//
//	Returns NIL pointer if error, else file pointer.
//

void*
WritePMHeader2( FILE* outFile, fmheader_t* pHeader, RawPlayerModel_t* pModel, mocapdata_t* pMotion )
{
	long		rc;
	long		oldPos;

	//	Save file pointer and rewind to start of header data
	oldPos = ftell( outFile );
	fseek( outFile, sizeof(header_t), SEEK_SET );

	//	Write the revised header
	if ( !(rc = fwrite(pHeader, sizeof(fmheader_t), 1, outFile)) )
		return( (void*)0 );

	//	Restore file position
	fseek( outFile, oldPos, SEEK_SET );

	return	outFile;

}


//
//	Main line
//
main( int argc, char *argv[] )
{
	FILE*	outFile;
	int		fail = 0;

	//	Trivial check for right # of arguments
	gProgramName = argv[0];
	printf( "Executing %s....\n", gProgramName );

	//	Read in the body parts of the player model
	printf( "reading body parts...\n" );
	if ( !(gpRawModel = ReadRawPlayerModel( &gBodyContext )) )
		quit( "Can't read body parts", -1);

	//	Read in the motion capture data
	printf( "reading motion capture data...\n" );
	if ( !(gpMocapData = ReadMocapDataFile( MOCAP_FILE_NAME )) )
		quit( "Can't read motion capture data file", -1 );
	if ( gpMocapData->fileHdr.version != CURRENT_MOCAP_VERSION )
		quit( "Incorrect motion capture file format version number", -1 );
	//	Create an output file for storing the model
	printf( "creating %s...\n", OUTPUT_FILE_NAME );
	if ( !(outFile=fopen(OUTPUT_FILE_NAME,"wb")) )
		quit( "Can't create output file", -1 );

	//
	//	Write out the new flexmodel, chunk by chunk
	//
	fail = 0;

	//	header		Some header information (40 bytes)
	printf("writing header chunk...\n");
	if ( !(WritePMHeader1( outFile, &gModelHeader, gpRawModel, gpMocapData )) )
		quit( "Error writing model header", -1 );

	//	skin		List of paths to skin texture files (a few kbytes)
	printf("writing skin pathnames chunk...\n");
	if ( !(WritePMDummySkins( outFile, &gModelHeader, gpRawModel, gpMocapData )) )
		quit( "Error writing skin path names section", -1 );

	//	st coord	Skin texture coordinates (a few kbytes)
	printf("writing skin texture coordinates chunk...\n");
	if ( !(WritePMSTCoords( outFile, &gModelHeader, gpRawModel, gpMocapData )) )
		quit( "Error writing skin path names", -1 );

	//	tris		Model triangle information (about 10 kbytes)
	printf("writing triangles chunk...\n");
	if ( !(WritePMTris( outFile, &gModelHeader, gpRawModel, gpMocapData )) )
		quit( "Error writing triangles section", -1 );

	//	frames		Animation frames (about 3 megabytes)
	printf("writing frames chunk...\n");
	if ( !(WritePMFrames( outFile, &gModelHeader, gpRawModel, gpMocapData )) )
		quit( "Error writing animation frames section", -1 );

	//	glcmds		Model triangle rendering information (about 20 kbytes)
	//	mesh nodes	Model node information (about 10 kbytes)
	printf("writing glcmds and mesh nodes chunks...\n");
	if ( !(WritePMGLCmdsAndNodes( outFile, &gModelHeader, gpRawModel, gpMocapData )) )
		quit( "Error writing glcmds or mesh nodes section", -1 );

	//	skeleton	Model skeleton information (about 150 kbytes)
	printf("writing skeletons chunk...\n");
	if ( !(WritePMSkels( outFile, &gModelHeader, gpRawModel, gpMocapData )) )
		quit( "Error writing skeleton section", -1 );

	//	references	Other stuff (about 350 kbytes)
	printf("writing references chunk...\n");
	if ( !(WritePMRefs( outFile, &gModelHeader, gpRawModel, gpMocapData )) )
		quit( "Error writing references section", -1 );

	//	Update the header
	printf("updating header chunk...\n");
	if ( !(WritePMHeader2( outFile, &gModelHeader, gpRawModel, gpMocapData )) )
		quit( "Error updating model header", -1 );

	//	Close the output file
	fclose( outFile );

	//	Dispose of the motion capture data
	free( gpMocapData );
	//	Dispose of the body parts
	DisposeRawPlayerModel( gpRawModel );

	//	Done
	return	0;
}
