//
//	Read in the body parts for a Heretic II Player Model
//
//	The body parts are defined in bones.h (one body part per bone,
//	except for the "reference" bones). The user provides a text
//	file that specifies the name of the mesh file and the texture
//	file containing each body part.
//
//	The following body parts are needed, in this order:
//
//	Name					Constant ID
//	-----------------------	----------------------
//	Left hip				BONE_HIPS_LEFT
//	Left thigh				BONE_LEFT_UPPER_LEG
//	Left calf				BONE_LEFT_LOWER_LEG
//	Left foot				BONE_LEFT_FOOT
//
//	Right hip				BONE_HIPS_RIGHT
//	Right thigh				BONE_RIGHT_UPPER_LEG
//	Right calf				BONE_RIGHT_LOWER_LEG
//	Right foot				BONE_RIGHT_FOOT
//
//	Lower back				BONE_HIPS_SPINE
//	Middle back				BONE_LOWER_TORSO
//	Left shoulder			BONE_LEFT_UPPER_TORSO
//	Left upper arm			BONE_LEFT_UPPER_ARM
//
//	Left lower arm			BONE_LEFT_LOWER_ARM
//	Open left hand			BONE_LEFT_HAND_OPEN
//	Closed left hand		BONE_LEFT_HAND_CLOSED
//	Bow of left hand		BONE_BOW
//
//	Right shoulder			BONE_RIGHT_UPPER_TORSO
//	Right upper arm			BONE_RIGHT_UPPER_ARM
//	Right lower arm			BONE_RIGHT_LOWER_ARM
//	Open right hand			BONE_RIGHT_HAND_OPEN
//
//	Closed right hand		BONE_RIGHT_HAND_CLOSED
//	Base of staff (rt)		BONE_RIGHT_STAFF_BASE
//	Tip of staff (rt)		BONE_RIGHT_STAFF_TIP
//	Base of hellstaff (rt)	BONE_RIGHT_HELL_BASE

//	Head of hellstaff (rt)	BONE_RIGHT_HELL_HEAD
//	Upper back				BONE_UPPER_TORSO_SPINE
//	Neck					BONE_NECK
//	Head and hair			BONE_HEAD

//	Back hanger				BONE_SHOULDER_BOW
//	Bow on shoulder			BONE_STORED_BOW
//	Left side of armor		BONE_ARMOR_LEFT
//	Left shoulder piece		BONE_LEFT_PAD

//	Right side of armor		BONE_ARMOR_RIGHT
//	Right shoulder piece	BONE_RIGHT_PAD
//	Right hip hanger		BONE_HIPS_STAFF
//	Staff stowed right hip	BONE_STORED_STAFF
//
//	The mesh for each body part must be built with only triangles.
//	Each body part must be within 5% of the length given for Corvus
//	at the end of bones.h
//
//	Each body part is initially read into memory as a list of vertex
//	coordinates, a list of texture coordinates and a list of triangles.
//

//
//	NOTE: This is a dummy version that creates each body part mathematically
//

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<memory.h>

#include	"CondComp.h"
#include	"FlexModel.h"
#include	"bones.h"
#include	"BodyPart.h"

//	The width allocated to each stick's tecture
#define	STICK_TEX_WIDTH		(1+2+2+2+2+1)

//	The required length of each bone (including reference bones)
static double
gBoneLengths[] =
{
	CORVUS_HIPS_LEFT_LEN,			//	Left hip
	CORVUS_LEFT_UPPER_LEG_LEN,		//	Left thigh
	CORVUS_LEFT_LOWER_LEG_LEN,		//	Left calf
	CORVUS_LEFT_FOOT_LEN,			//	Left foot

	CORVUS_HIPS_RIGHT_LEN,			//	Right hip
	CORVUS_RIGHT_UPPER_LEG_LEN,		//	Right thigh
	CORVUS_RIGHT_LOWER_LEG_LEN,		//	Right calf
	CORVUS_RIGHT_FOOT_LEN,			//	Right foot

	CORVUS_HIPS_SPINE_LEN,			//	Lower back
	CORVUS_LOWER_TORSO_LEN,			//	Middle back
	CORVUS_LEFT_UPPER_TORSO_LEN,	//	Left shoulder
	CORVUS_LEFT_UPPER_ARM_LEN,		//	Left upper arm

	CORVUS_LEFT_LOWER_ARM_LEN,		//	Left lower arm
	CORVUS_LEFT_HAND_OPEN_LEN,		//	Open left hand
	CORVUS_LEFT_HAND_CLOSED_LEN,	//	Closed left hand
	CORVUS_BOW_LEN,					//	Bow of left hand

	CORVUS_RIGHT_UPPER_TORSO_LEN,	//	Right shoulder
	CORVUS_RIGHT_UPPER_ARM_LEN,		//	Right upper arm
	CORVUS_RIGHT_LOWER_ARM_LEN,		//	Right lower arm
	CORVUS_RIGHT_HAND_OPEN_LEN,		//	Open right hand

	CORVUS_RIGHT_HAND_CLOSED_LEN,	//	Closed right hand
	CORVUS_RIGHT_STAFF_BASE_LEN,	//	Staff of right hand (bottom)
	CORVUS_RIGHT_STAFF_TIP_LEN,		//	Staff of right hand (top, no blade)
	CORVUS_RIGHT_HELL_BASE_LEN,		//	Hellstaff of right hand (bottom)

	CORVUS_RIGHT_HELL_HEAD_LEN,		//	Hellstaff of right hand (top)
	CORVUS_UPPER_TORSO_SPINE_LEN,	//	Upper back
	CORVUS_NECK_LEN,				//	Neck
	CORVUS_HEAD_LEN,				//	Head and hair

	CORVUS_SHOULDER_BOW_LEN,		//	Back hanger
	CORVUS_STORED_BOW_LEN,			//	Bow on shoulder
	CORVUS_ARMOR_LEFT_LEN,			//	Left side of armor
	CORVUS_LEFT_PAD_LEN,			//	Left shoulder piece

	CORVUS_ARMOR_RIGHT_LEN,			//	Right side of armor
	CORVUS_RIGHT_PAD_LEN,			//	Right shoulder piece
	CORVUS_HIPS_STAFF_LEN,			//	Right hip hanger
	CORVUS_STORED_STAFF_LEN,		//	Staff stowed right hip
};

//	The name of each bone, including reference bones
static	char*
boneNames[ BONE_NUM_BONES ] =					//	Human-readable name of each bone
{
	"hips left",				"left upper leg",				"left lower leg",				"left foot",
	"hips right",				"right upper leg",				"right lower leg",				"right foot",
	"hips spine",				"lower torso",					"left upper torso",				"left upper arm",
	"left lower arm",			"left hand open",				"left hand closed",				"bow",
	"right upper torso",		"right upper arm",				"right lower arm",				"right hand open",
	"right hand closed",		"staff base",					"staff tip",					"hellstaff base",
	"hellstaff head",			"upper torso spine",			"neck",							"head",
	"shoulder bow",				"stored bow",					"armor left",					"left pad",
	"armor right",				"right pad",					"hips staff",					"stored staff"
};

//	Canonical reference info for the "end" of each bone, based
//	on standard Corvus bone lengths. These are rotated into
//	position based on the frame animation to create reference
//	points and skeleton location points.
//
//	Note that not all bones are used as skeleton or reference
//	points. For unused bones, we story dummy data.

#define	RVL			(REFERENCE_POSITION_VECTOR_LENGTH)		//	Same length used in Corvus model
#define	SVL			(SKELETON_POSITION_VECTOR_LENGTH)		//	Same length used in Corvus model
#define	NOT_USED	1.0
static const imp_point_t
bodyPartRefs[] =
{
//		<<ORIGIN>>									<<DIRECTION>>									<<UP>>
	{	{ CORVUS_HIPS_LEFT_LEN, 0., 0. },			{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_HIPS_LEFT
	{	{ CORVUS_LEFT_UPPER_LEG_LEN, 0., 0. },		{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_LEFT_UPPER_LEG
	{	{ CORVUS_LEFT_LOWER_LEG_LEN, 0., 0. },		{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_LEFT_LOWER_LEG
	{	{ CORVUS_LEFT_FOOT_LEN/2., 0., 0. },		{ CORVUS_LEFT_FOOT_LEN/2., -RVL, 0. },			{ CORVUS_LEFT_FOOT_LEN/2, 0., RVL }				},	//	BONE_LEFT_FOOT

	{	{ CORVUS_HIPS_RIGHT_LEN, 0., 0. },			{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_HIPS_RIGHT
	{	{ CORVUS_RIGHT_UPPER_LEG_LEN, 0., 0. },		{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_RIGHT_UPPER_LEG
	{	{ CORVUS_RIGHT_LOWER_LEG_LEN, 0., 0. },		{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_RIGHT_LOWER_LEG
	{	{ CORVUS_RIGHT_FOOT_LEN/2., 0., 0. },		{ CORVUS_RIGHT_FOOT_LEN/2., -RVL, 0. },			{ CORVUS_RIGHT_FOOT_LEN/2, 0., RVL }			},	//	BONE_RIGHT_FOOT

	{	{ CORVUS_HIPS_SPINE_LEN, 0., 0. },			{ CORVUS_HIPS_SPINE_LEN+SVL, 0., 0. },			{ CORVUS_HIPS_SPINE_LEN, 0., SVL }				},	//	BONE_HIPS_SPINE
	{	{ CORVUS_LOWER_TORSO_LEN, 0., 0. },			{ CORVUS_LOWER_TORSO_LEN+SVL, 0., 0. },			{ CORVUS_LOWER_TORSO_LEN, 0., SVL }				},	//	BONE_LOWER_TORSO

	{	{ CORVUS_LEFT_UPPER_TORSO_LEN, 0., 0. },	{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_LEFT_UPPER_TORSO
	{	{ CORVUS_LEFT_UPPER_ARM_LEN, 0., 0. },		{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_LEFT_UPPER_ARM
	{	{ CORVUS_LEFT_LOWER_ARM_LEN, 0., 0. },		{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_LEFT_LOWER_ARM
	{	{ CORVUS_LEFT_HAND_OPEN_LEN, 0., 0. },		{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_LEFT_HAND_OPEN
	{	{ CORVUS_LEFT_HAND_CLOSED_LEN, 0., 0. },	{ CORVUS_LEFT_HAND_CLOSED_LEN+RVL, 0., 0. },	{ CORVUS_LEFT_HAND_CLOSED_LEN, 0., RVL }		},	//	BONE_LEFT_HAND_CLOSED
	{	{ CORVUS_BOW_LEN, 0., 0. },					{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_BOW

	{	{ CORVUS_RIGHT_UPPER_TORSO_LEN, 0., 0. },	{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_RIGHT_UPPER_TORSO
	{	{ CORVUS_RIGHT_UPPER_ARM_LEN, 0., 0. },		{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_RIGHT_UPPER_ARM
	{	{ CORVUS_RIGHT_LOWER_ARM_LEN, 0., 0. },		{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_RIGHT_LOWER_ARM
	{	{ CORVUS_RIGHT_HAND_OPEN_LEN, 0., 0. },		{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_RIGHT_HAND_OPEN
	{	{ CORVUS_RIGHT_HAND_CLOSED_LEN, 0., 0. },	{ CORVUS_RIGHT_HAND_CLOSED_LEN+RVL, 0., 0. },	{ CORVUS_RIGHT_HAND_CLOSED_LEN, 0., -RVL }		},	//	BONE_RIGHT_HAND_CLOSED

	{	{ CORVUS_RIGHT_STAFF_BASE_LEN, 0., 0. },	{ CORVUS_RIGHT_STAFF_BASE_LEN, 0., RVL },		{ CORVUS_RIGHT_STAFF_BASE_LEN, -RVL, 0. }		},	//	BONE_RIGHT_STAFF_BASE
	{	{ CORVUS_RIGHT_STAFF_TIP_LEN, 0., 0. },		{ CORVUS_RIGHT_STAFF_TIP_LEN, 0., RVL },		{ CORVUS_RIGHT_STAFF_TIP_LEN, RVL, 0. }			},	//	BONE_RIGHT_STAFF_TIP
	{	{ CORVUS_RIGHT_HELL_BASE_LEN, 0., 0. },		{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_RIGHT_HELL_BASE
	{	{ CORVUS_RIGHT_HELL_HEAD_LEN, 0., 0. },		{ CORVUS_RIGHT_HELL_HEAD_LEN, 0., RVL },		{ CORVUS_RIGHT_HELL_HEAD_LEN, RVL, 0. }			},	//	BONE_RIGHT_HELL_HEAD

	{	{ CORVUS_UPPER_TORSO_SPINE_LEN, 0., 0. },	{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_UPPER_TORSO_SPINE
	{	{ CORVUS_NECK_LEN, 0., 0. },				{ CORVUS_NECK_LEN+SVL, 0., 0. },				{ CORVUS_NECK_LEN, 0., SVL }					},	//	BONE_NECK
	{	{ CORVUS_HEAD_LEN, 0., 0. },				{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_HEAD
	{	{ CORVUS_SHOULDER_BOW_LEN, 0., 0. },		{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_SHOULDER_BOW
	{	{ CORVUS_STORED_BOW_LEN, 0., 0. },			{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_STORED_BOW

	{	{ CORVUS_ARMOR_LEFT_LEN, 0., 0. },			{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_ARMOR_LEFT
	{	{ CORVUS_LEFT_PAD_LEN, 0., 0. },			{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_LEFT_PAD
	{	{ CORVUS_ARMOR_RIGHT_LEN, 0., 0. },			{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_ARMOR_RIGHT
	{	{ CORVUS_RIGHT_PAD_LEN, 0., 0. },			{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_RIGHT_PAD

	{	{ CORVUS_HIPS_STAFF_LEN, 0., 0. },			{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							},	//	BONE_HIPS_STAFF
	{	{ CORVUS_STORED_STAFF_LEN, 0., 0. },		{ NOT_USED, 0., 0. },							{ NOT_USED, 0., 0. }							}	//	BONE_STORED_STAFF
};
//
//	Weapons data - let's make the weapons look real :-)
//	NOTE: Incomplete - dummy code for now.
//

//
//	Data for the durnwood staff - exported from .obj to .c by CROSSROADS
//	and hand-edited to match our format.
//
#include	"staffdata.h"

//
//	Data for the hellstaff - exported from .obj to .c by CROSSROADS
//	and hand-edited to match our format.
//
#include	"helldata.h"

//
//	Data for the stored bow - exported from .obj to .c by CROSSROADS
//	and hand-edited to match our format.
//
#include	"bowdata.h"

//
//	Data for the drawn bow - exported from .obj to .c by CROSSROADS
//	and hand-edited to match our format.
//
#include	"bowhanddata.h"

//
//	Create an empty body part
//

static
RawBodyPart_t*
NewBodyPart()
{
	RawBodyPart_t*	pBP;
	//	Allocate the memory
	if ( !(pBP = malloc(sizeof(RawBodyPart_t))) )
		return	pBP;
	//	Initialize the structure
	pBP->boneNumber	= 0;
	pBP->vert3DCount= 0;
	pBP->trisCount  = 0;
	pBP->pVert3D    = (void*)0;
	pBP->pTris      = (void*)0;
	pBP->vertUVCount= 0;
	pBP->pVertUV    = (void*)0;
	pBP->pTrisTex   = (void*)0;
	pBP->texWidth   = 0;
	pBP->texHeight  = 0;
	pBP->texID      = 0;
	pBP->pTexPixels = (void*)0;
	pBP->pAltVert3D	= (void*)0;
	{
		int	i;
		for ( i=0; i<=MAX_ALT_GEOMETRY_FRAMES; i++ )
			(pBP->altFrames[i])[0] = '\0';
	}
}

//
//	Dispose of a body part
//

void
DisposeBodyPart( RawBodyPart_t* pBP )
{
	if ( pBP )
	{
		//	Release memory we point at
		if ( pBP->pVert3D )
			free( pBP->pVert3D );
		if ( pBP->pTris )
			free( pBP->pTris );
		if ( pBP->pVertUV )
			free( pBP->pVertUV );
		if ( pBP->pTrisTex )
			free( pBP->pTrisTex );
		if ( pBP->pTexPixels )
			free( pBP->pTexPixels );
		if ( pBP->pAltVert3D )
			free( pBP->pAltVert3D );

		//	Release the structure itself
		free( pBP );
	}
}


//
//	Set up the reference point information for a body part
//	based on its bone number.
//
static void
SetPartRefPoint( RawBodyPart_t* pPart, int boneNumber, BodyContext_t* pContext )
{
	pPart->refPoint = bodyPartRefs[ boneNumber ];
}


//
//	Create a body part that is just a rectangular stick of the
//	correct length for a given bone number, aligned along the
//	positive X axis. The stick is 2 units on each side.
//
//	The texture coordinates for the stick are calculated from its
//	bone number and length - each texture is allocated 10 pixels
//	of width (2+2+2+2, +1 on each side for clear space). For
//	bone numbers 0..24, the vertical texture space begins at
//	Y=1; for bone numbers 25..49, the vertical texture space
//	begins at Y=129. The height allocated to the texture is
//	the next whole integer of the bone length, plus 4 for base,
//	4 for tip, and one on each side for clear space.
//
//	The texture for the base is placed at the lowest V coordinate.
//	The texture verteces are laid out as follows:
//		3'--2'
//		| \ |
//		0---1---2---3---0'
//		|  /|\  |  /|\  |
//		| | | | | | | | |
//		| | | | | | | | |
//		|/  |  \|/  |  \|
//		4---5---6---7---4'
//		| \ |
//		7'--6'
//

static
RawBodyPart_t*
GetStickBone( int boneNumber )
{
	RawBodyPart_t*	pPart;
	int				tBaseU, tBaseV;
	float			xd;
	int				xi;
	int				fail = 0;

	//	Bounds check
	if ( (boneNumber < 0) || (boneNumber >= BONE_NUM_BONES) )
		return (void*)0;
	//	Return NIL if can't allocate a body part; else init
	//	all fields.
	if ( !(pPart = NewBodyPart()) )
		return	pPart;

	//	Each stick has 8 verteces, 6 faces, and two triangles per face.
	//	A single texture is used for all of the sticks in the body.
	//	There are more texture verteces than 3D verteces, because when
	//	the stick is laid out flat two extra verteces are needed for each
	//	end of the stick and for the seam along its length. Fill in the
	//	easy fields first
	pPart->boneNumber  = boneNumber;
	pPart->vert3DCount = 8;
	pPart->trisCount   = 12;
	pPart->vertUVCount = 8 + 2 + 2 + 2;
	pPart->texWidth    = 256;
	pPart->texHeight   = 256;
	pPart->texID       = 0xABADF00D;

	//	For the 3D verteces, we need memory for eight vec3_t structures
	//	For the UV verteces, we need memory for twelve texpoint_t structures
	if ( !(pPart->pVert3D = malloc( pPart->vert3DCount * sizeof(vec3_t) )) ) fail = 1;
	if ( !(pPart->pVertUV = malloc( pPart->vertUVCount * sizeof(texpoint_t) )) ) fail = 1;

	///	If this is the bow, we also need memory for the alternate geometry
	//	This really should be motion-captured, but it's not so we hard-wire it.
	if ( boneNumber == BONE_BOW )
	{
		if ( !(pPart->pAltVert3D = malloc( pPart->vert3DCount * sizeof(vec3_t) )) ) fail = 1;
	}

	//	For the Triangle verteces, we need memory for twelve trividx_t structures
	//	We also allocate a texture triangle for each 3D triangle
	if ( !(pPart->pTris = malloc( 12 * sizeof(trividx_t) )) ) fail = 1;
	if ( !(pPart->pTrisTex = malloc( 12 * sizeof(trividx_t) )) ) fail = 1;

	//	Don't assign a texture bitmap yet
	pPart->pTexPixels  = (void*)0;

	//	If we couldn't get memory for everything, release
	//	whatever we did get and return error
	if ( fail )
	{
		if ( pPart->pVert3D )    free( pPart->pVert3D );
		if ( pPart->pVertUV )    free( pPart->pVertUV );
		if ( pPart->pTris )      free( pPart->pTris );
		if ( pPart->pTrisTex )   free( pPart->pTrisTex );
		if ( pPart->pAltVert3D ) free( pPart->pAltVert3D );
		free( pPart );
		return (void*)0;
	}

	//	Fill in the coordinates and indeces.

	//	First the 3D verteces
	xd = (float)(gBoneLengths[ pPart->boneNumber ]);		//	Bone length
	{ pPart->pVert3D[0].c[0] = 0.; pPart->pVert3D[0].c[1] = -1.; pPart->pVert3D[0].c[2] = -1.; }
	{ pPart->pVert3D[1].c[0] = 0.; pPart->pVert3D[1].c[1] = -1.; pPart->pVert3D[1].c[2] =  1.; }
	{ pPart->pVert3D[2].c[0] = 0.; pPart->pVert3D[2].c[1] =  1.; pPart->pVert3D[2].c[2] =  1.; }
	{ pPart->pVert3D[3].c[0] = 0.; pPart->pVert3D[3].c[1] =  1.; pPart->pVert3D[3].c[2] = -1.; }
	{ pPart->pVert3D[4].c[0] = xd; pPart->pVert3D[4].c[1] = -1.; pPart->pVert3D[4].c[2] = -1.; }
	{ pPart->pVert3D[5].c[0] = xd; pPart->pVert3D[5].c[1] = -1.; pPart->pVert3D[5].c[2] =  1.; }
	{ pPart->pVert3D[6].c[0] = xd; pPart->pVert3D[6].c[1] =  1.; pPart->pVert3D[6].c[2] =  1.; }
	{ pPart->pVert3D[7].c[0] = xd; pPart->pVert3D[7].c[1] =  1.; pPart->pVert3D[7].c[2] = -1.; }

	//	Optional alternate 3D verteces
	if ( boneNumber == BONE_BOW )
	{
		//	The alternate here is for the bow with a nocked arrow. For now
		//	the data is the same as the normal geometry, except double-high.
		//	It is used only in frames draw1 - draw5.
		{ pPart->pVert3D[0].c[0] = -xd; pPart->pVert3D[0].c[1] = -1.; pPart->pVert3D[0].c[2] = -1.; }
		{ pPart->pVert3D[1].c[0] = -xd; pPart->pVert3D[1].c[1] = -1.; pPart->pVert3D[1].c[2] =  1.; }
		{ pPart->pVert3D[2].c[0] = -xd; pPart->pVert3D[2].c[1] =  1.; pPart->pVert3D[2].c[2] =  1.; }
		{ pPart->pVert3D[3].c[0] = -xd; pPart->pVert3D[3].c[1] =  1.; pPart->pVert3D[3].c[2] = -1.; }
		{ pPart->pVert3D[4].c[0] =  xd; pPart->pVert3D[4].c[1] = -1.; pPart->pVert3D[4].c[2] = -1.; }
		{ pPart->pVert3D[5].c[0] =  xd; pPart->pVert3D[5].c[1] = -1.; pPart->pVert3D[5].c[2] =  1.; }
		{ pPart->pVert3D[6].c[0] =  xd; pPart->pVert3D[6].c[1] =  1.; pPart->pVert3D[6].c[2] =  1.; }
		{ pPart->pVert3D[7].c[0] =  xd; pPart->pVert3D[7].c[1] =  1.; pPart->pVert3D[7].c[2] = -1.; }
		strcpy( pPart->altFrames[0], "draw1" );
		strcpy( pPart->altFrames[1], "draw2" );
		strcpy( pPart->altFrames[2], "draw3" );
		strcpy( pPart->altFrames[3], "draw4" );
		strcpy( pPart->altFrames[4], "draw5" );
	}

	//	Next the triangles. NOTE: Vertex indeces start at zero in the
	//	FlexModel format used by Heretic II; we use zero-based too.
	//	Note also that the verteces must be defined using a consistent
	//	"left-hand rule" for calculation of normals based on X=right,
	//	Y=up, Z=away
	{ pPart->pTris[0].v[0]  = 0; pPart->pTris[0].v[1]  = 1; pPart->pTris[0].v[2]  = 3; }			//	1st base triangle
	{ pPart->pTris[1].v[0]  = 2; pPart->pTris[1].v[1]  = 3; pPart->pTris[1].v[2]  = 1; }			//	2nd base triangle
	{ pPart->pTris[2].v[0]  = 0; pPart->pTris[2].v[1]  = 4; pPart->pTris[2].v[2]  = 1; }			//	1st bottom triangle
	{ pPart->pTris[3].v[0]  = 5; pPart->pTris[3].v[1]  = 1; pPart->pTris[3].v[2]  = 4; }			//	2nd bottom triangle
	{ pPart->pTris[4].v[0]  = 5; pPart->pTris[4].v[1]  = 6; pPart->pTris[4].v[2]  = 1; }			//	1st rear triangle
	{ pPart->pTris[5].v[0]  = 2; pPart->pTris[5].v[1]  = 1; pPart->pTris[5].v[2]  = 6; }			//	2nd rear triangle
	{ pPart->pTris[6].v[0]  = 2; pPart->pTris[6].v[1]  = 6; pPart->pTris[6].v[2]  = 3; }			//	1st top triangle
	{ pPart->pTris[7].v[0]  = 7; pPart->pTris[7].v[1]  = 3; pPart->pTris[7].v[2]  = 6; }			//	2nd top triangle
	{ pPart->pTris[8].v[0]  = 0; pPart->pTris[8].v[1]  = 3; pPart->pTris[8].v[2]  = 4; }			//	1st top triangle
	{ pPart->pTris[9].v[0]  = 7; pPart->pTris[9].v[1]  = 4; pPart->pTris[9].v[2]  = 3; }			//	2nd top triangle
	{ pPart->pTris[10].v[0] = 5; pPart->pTris[10].v[1] = 4; pPart->pTris[10].v[2] = 6; }			//	1st tip triangle
	{ pPart->pTris[11].v[0] = 7; pPart->pTris[11].v[1] = 6; pPart->pTris[11].v[2] = 4; }			//	2nd tip triangle

	//	Now the texture triangles
	//	0' = 8; 2' = 9; 3' = 10; 4' = 11; 6' = 12; 7' = 13
	{ pPart->pTrisTex[0].v[0]  = 0;  pPart->pTrisTex[0].v[1]  = 1;  pPart->pTrisTex[0].v[2]  = 10; }	//	1st base triangle (primes)
	{ pPart->pTrisTex[1].v[0]  = 9;  pPart->pTrisTex[1].v[1]  = 10; pPart->pTrisTex[1].v[2]  = 1;  }	//	2nd base triangle (primes)
	{ pPart->pTrisTex[2].v[0]  = 0;  pPart->pTrisTex[2].v[1]  = 4;  pPart->pTrisTex[2].v[2]  = 1;  }	//	1st bottom triangle
	{ pPart->pTrisTex[3].v[0]  = 5;  pPart->pTrisTex[3].v[1]  = 1;  pPart->pTrisTex[3].v[2]  = 4;  }	//	2nd bottom triangle
	{ pPart->pTrisTex[4].v[0]  = 5;  pPart->pTrisTex[4].v[1]  = 6;  pPart->pTrisTex[4].v[2]  = 1;  }	//	1st rear triangle
	{ pPart->pTrisTex[5].v[0]  = 2;  pPart->pTrisTex[5].v[1]  = 1;  pPart->pTrisTex[5].v[2]  = 6;  }	//	2nd rear triangle
	{ pPart->pTrisTex[6].v[0]  = 2;  pPart->pTrisTex[6].v[1]  = 6;  pPart->pTrisTex[6].v[2]  = 3;  }	//	1st top triangle
	{ pPart->pTrisTex[7].v[0]  = 7;  pPart->pTrisTex[7].v[1]  = 3;  pPart->pTrisTex[7].v[2]  = 6;  }	//	2nd top triangle
	{ pPart->pTrisTex[8].v[0]  = 8;  pPart->pTrisTex[8].v[1]  = 3;  pPart->pTrisTex[8].v[2]  = 11; }	//	1st top triangle (primes)
	{ pPart->pTrisTex[9].v[0]  = 7;  pPart->pTrisTex[9].v[1]  = 11; pPart->pTrisTex[9].v[2]  = 3;  }	//	2nd top triangle (primes)
	{ pPart->pTrisTex[10].v[0] = 5;  pPart->pTrisTex[10].v[1] = 4;  pPart->pTrisTex[10].v[2] = 12; }	//	1st tip triangle (primes)
	{ pPart->pTrisTex[11].v[0] = 13; pPart->pTrisTex[11].v[1] = 12; pPart->pTrisTex[11].v[2] = 4;  }	//	2nd tip triangle (primes)

	//	Now the texture space coordinates
	//	0' = 8; 2' = 9; 3' = 10; 4' = 11; 6' = 12; 7' = 13
	//
	//	The texture verteces are laid out as follows:
	//		3'--2'
	//		| \ |
	//		0---1---2---3---0'
	//		|  /|\  |  /|\  |
	//		| | | | | | | | |
	//		| | | | | | | | |
	//		|/  |  \|/  |  \|
	//		4---5---6---7---4'
	//		| \ |
	//		7'--6'
	xi = (int)(xd + 0.9999);					//	Integral bone length
	if ( pPart->boneNumber >= 25 )
		{ tBaseU = (STICK_TEX_WIDTH * (pPart->boneNumber - 25)); tBaseV = 128; }
	else
		{ tBaseU = (STICK_TEX_WIDTH * pPart->boneNumber); tBaseV = 0; }
	//	Uppermost texture verteces
	{ pPart->pVertUV[10].c[0] = tBaseU + 1;					pPart->pVertUV[10].c[1] = tBaseV + 1; } 
	{ pPart->pVertUV[9].c[0]  = tBaseU + 1 + 2;				pPart->pVertUV[9].c[1]  = tBaseV + 1; } 
	//	Upper texture verteces
	{ pPart->pVertUV[0].c[0]  = tBaseU + 1;					pPart->pVertUV[0].c[1]  = tBaseV + 1 + 2; } 
	{ pPart->pVertUV[1].c[0]  = tBaseU + 1 + 2;				pPart->pVertUV[1].c[1]  = tBaseV + 1 + 2; } 
	{ pPart->pVertUV[2].c[0]  = tBaseU + 1 + 2 + 2;			pPart->pVertUV[2].c[1]  = tBaseV + 1 + 2; } 
	{ pPart->pVertUV[3].c[0]  = tBaseU + 1 + 2 + 2 + 2;		pPart->pVertUV[3].c[1]  = tBaseV + 1 + 2; } 
	{ pPart->pVertUV[8].c[0]  = tBaseU + 1 + 2 + 2 + 2 + 2;	pPart->pVertUV[8].c[1]  = tBaseV + 1 + 2; } 
	//	Lower texture verteces
	{ pPart->pVertUV[4].c[0]  = tBaseU + 1;					pPart->pVertUV[4].c[1]  = tBaseV + 1 + 2 + xi; } 
	{ pPart->pVertUV[5].c[0]  = tBaseU + 1 + 2;				pPart->pVertUV[5].c[1]  = tBaseV + 1 + 2 + xi; } 
	{ pPart->pVertUV[6].c[0]  = tBaseU + 1 + 2 + 2;			pPart->pVertUV[6].c[1]  = tBaseV + 1 + 2 + xi; } 
	{ pPart->pVertUV[7].c[0]  = tBaseU + 1 + 2 + 2 + 2;		pPart->pVertUV[7].c[1]  = tBaseV + 1 + 2 + xi; } 
	{ pPart->pVertUV[11].c[0] = tBaseU + 1 + 2 + 2 + 2 + 2;	pPart->pVertUV[11].c[1] = tBaseV + 1 + 2 + xi; } 
	//	Lowest texture verteces
	{ pPart->pVertUV[13].c[0] = tBaseU + 1;					pPart->pVertUV[13].c[1] = tBaseV + 1 + 2 + xi + 2; } 
	{ pPart->pVertUV[12].c[0] = tBaseU + 1 + 2;				pPart->pVertUV[12].c[1] = tBaseV + 1 + 2 + xi + 2; } 

	//	Give the finished part to the caller
	return	pPart;

}

//
//	Read in a specified body part
//

static
RawBodyPart_t*
ReadBodyPart( int boneNumber, BodyContext_t* pContext )
{
	RawBodyPart_t*	pPart;

	//	Read in the mesh - DUMMY FOR NOW
	pPart = GetStickBone( boneNumber );

	//	Set up the reference points from the mesh - DUMMY FOR NOW
	SetPartRefPoint( pPart, boneNumber, pContext );

	return	pPart;
}

//
//	Allocate and initialize a raw player model structure
//

static
RawPlayerModel_t*
NewRawPlayerModel()
{
	RawPlayerModel_t*	pBody;
	int					i;
	//	Allocate memory
	if ( !(pBody = malloc(sizeof(RawPlayerModel_t))) )
		return	pBody;
	//	Zero all fields
	for ( i=0; i<NUM_BODY_PARTS; i++ )
		pBody->parts[i] = (void*)0;
	for ( i=0; i<H2NUM_MESHNODES; i++ )
		pBody->nodes[i] = (void*)0;
	for ( i=0; i<SEAM_NUM_SEAMS; i++ )
		pBody->seams[i] = (void*)0;
	return	pBody;
}

//
//	Dispose of a player model structure
//

void
DisposeRawPlayerModel( RawPlayerModel_t* pBody )
{
	int					i;
	//	Allocate memory
	if ( pBody )
	{
		//	Release and zero all fields
		for ( i=0; i<NUM_BODY_PARTS; i++ )
			DisposeBodyPart( pBody->parts[i] );
		//	WARNING - DON'T KNOW HOW TO DO THESE YER
		for ( i=0; i<H2NUM_MESHNODES; i++ )
			DisposeBodyPart( pBody->nodes[i] );
		for ( i=0; i<SEAM_NUM_SEAMS; i++ )
			DisposeBodyPart( pBody->seams[i] );
	}
	free( pBody );
}

//
//	Build a stick-figure player model mathematically
//

RawPlayerModel_t*
MakeStickRawPlayerModel( BodyContext_t* pContext )
{
	RawPlayerModel_t*	pBody;
	int					fail = 0;
	int					boneNumber;
	int					tB, vB, xB;
	//	Allocate memory
	if ( !(pBody = NewRawPlayerModel()) )
		return	pBody;
	//	Get the parts
	tB = 0;
	vB = 0;
	xB = 0;
	for ( boneNumber=0; boneNumber<NUM_BODY_PARTS; boneNumber++ )
	{
		printf( "    %s...\n", boneNames[boneNumber] );
		if ( !(pBody->parts[boneNumber] = GetStickBone(boneNumber)) )
			fail = 1;
		//	Add local-to-global conversion information
		pBody->parts[boneNumber]->triBase = tB;
		pBody->parts[boneNumber]->vertBase = vB;
		pBody->parts[boneNumber]->texBase = xB;
		tB += pBody->parts[boneNumber]->trisCount;
		vB += pBody->parts[boneNumber]->vert3DCount;
		xB += pBody->parts[boneNumber]->vertUVCount;
		//	Set up the reference points from the mesh - DUMMY FOR NOW
		SetPartRefPoint( pBody->parts[boneNumber], boneNumber, pContext );

	}
	//	Make sure no errors; clean up if there are.
	if ( fail )
	{
		DisposeRawPlayerModel( pBody );
		pBody = (void*)0;
	}
	return	pBody;
}

//
//	Get a player model from the user (prompt for input or use context info)
//

RawPlayerModel_t*
ReadRawPlayerModel( BodyContext_t* pContext )
{
	//	Dummy code for now...use stick figure
	return	MakeStickRawPlayerModel( pContext );
}

