//
//	Create a frame scaled so that all vertex coordinates are in the range 0..255
//	for a Heretic II flexmodel
//

#include	<stdlib.h>
#include	<math.h>

#include	"CondComp.h"
#include	"FlexModel.h"
#include	"mocap.h"
#include	"bones.h"
#include	"BodyPart.h"
#include	"MatrixMath.h"
#include	"ScaledFrame.h"

//
//	Calculate the new frame coordinates, scale factors, etc. and
//	build a frame record with light normals, etc.
//
//	If pRef is non-NIL, we need to scale 3xBONE_NUM_BONES reference /
//	skeleton points at pRef by the resulting scale factors as well.
//

void
MakeScaledFrame( frame_t* pFrame, fmheader_t* pHeader, vec3dbl_t* pVerts, unsigned char* pNormals, refinfo_t* pRef )
{
	double		minX, maxX, minY, maxY, minZ, maxZ;
	int			vert, ref, nVerts, i;

	//	Find the range of coordinates in the frame
	nVerts = pHeader->vert3dcnt;
	minX = minY = minZ =  9999999999;
	maxX = maxY = maxZ = -9999999999;
	for ( vert=0; vert<nVerts; vert++ )
	{
		//	Minima
		if ( pVerts[vert].c[0] < minX )
			minX = pVerts[vert].c[0];
		if ( pVerts[vert].c[1] < minY )
			minY = pVerts[vert].c[1];
		if ( pVerts[vert].c[2] < minZ )
			minZ = pVerts[vert].c[2];
		//	Maxima
		if ( pVerts[vert].c[0] > maxX )
			maxX = pVerts[vert].c[0];
		if ( pVerts[vert].c[1] > maxY )
			maxY = pVerts[vert].c[1];
		if ( pVerts[vert].c[2] > maxZ )
			maxZ = pVerts[vert].c[2];
	}
	//	Set the translation and scaling factors
	pFrame->header.translate[0] = (float)minX;
	pFrame->header.translate[1] = (float)minY;
	pFrame->header.translate[2] = (float)minZ;
	pFrame->header.scale[0]     = (float)((maxX-minX) / 255.);
	pFrame->header.scale[1]     = (float)((maxY-minY) / 255.);
	pFrame->header.scale[2]     = (float)((maxZ-minZ) / 255.);
	//	Emit the scaled / translated vertex coordinates and light normals
	nVerts = pHeader->vert3dcnt;
	for ( vert=0; vert<nVerts; vert++ )
	{
		pFrame->verts[vert].lightnormalindex = pNormals[vert];
		pFrame->verts[vert].v[0] = (byte)((pVerts[vert].c[0] - minX) / pFrame->header.scale[0] + 0.5);
		pFrame->verts[vert].v[1] = (byte)((pVerts[vert].c[1] - minY) / pFrame->header.scale[1] + 0.5);
		pFrame->verts[vert].v[2] = (byte)((pVerts[vert].c[2] - minZ) / pFrame->header.scale[2] + 0.5);
	}
	//	Emit the reference data if requested.
	//	Although it is not included in the scaling and translation
	//	calculations, it is scaled and translated by the result.
	if ( pRef )
	{
		for ( ref=0; ref<BONE_NUM_BONES; ref++ )
		{
			for ( i=0; i<3; i++ )
			{
				pRef->refs[ref].origin.c[i]    = (float)((int)((pRef->refs[ref].origin.c[i] - pFrame->header.translate[i]) / pFrame->header.scale[i] + 0.5));
				pRef->refs[ref].direction.c[i] = (float)((int)((pRef->refs[ref].direction.c[i] - pFrame->header.translate[i]) / pFrame->header.scale[i] + 0.5));
				pRef->refs[ref].up.c[i]        = (float)((int)((pRef->refs[ref].up.c[i] - pFrame->header.translate[i]) / pFrame->header.scale[i] + 0.5));
			}
		}
	}
}

