//
//	Create 8-bit indexed normals for the posed verteces of a
//	Heretic II flexmodel
//

#include	<stdlib.h>
#include	<math.h>

#include	"CondComp.h"
#include	"FlexModel.h"
#include	"bones.h"
#include	"MatrixMath.h"
#include	"BodyPart.h"
#include	"Normals.h"

//
//	Array of "canned" normal vectors. These are all unit vectors.
//	To save space in the model, a single index into this table
//	is stored rather than an actual normal angle, The index
//	represents the "canned" normal closest to the actual normal
//	needed.
//
//	This is the data that occurs in the QDATA program source file,
//	anorm.h
//
//	The number of normals selected is strange - 162 is 2 x 3x3x3x3.
//	A perfect cube (like 125 or 216) makes more sense, since it
//	corresponds to equal gradations of theta for x, y, and z.
//
 
static vec3dbl_t
vNorms[FM_NUMVERTEXNORMALS] = 
{
	{-0.525731,  0.000000,  0.850651},
	{-0.442863,  0.238856,  0.864188},
	{-0.295242,  0.000000,  0.955423},
	{-0.309017,  0.500000,  0.809017}, 
	{-0.162460,  0.262866,  0.951056}, 
	{ 0.000000,  0.000000,  1.000000}, 
	{ 0.000000,  0.850651,  0.525731}, 
	{-0.147621,  0.716567,  0.681718}, 

	{ 0.147621,  0.716567,  0.681718}, 
	{ 0.000000,  0.525731,  0.850651},
	{ 0.309017,  0.500000,  0.809017}, 
	{ 0.525731,  0.000000,  0.850651},
	{ 0.295242,  0.000000,  0.955423},
	{ 0.442863,  0.238856,  0.864188},
	{ 0.162460,  0.262866,  0.951056}, 
	{-0.681718,  0.147621,  0.716567}, 

	{-0.809017,  0.309017,  0.500000}, 
	{-0.587785,  0.425325,  0.688191}, 
	{-0.850651,  0.525731,  0.000000}, 
	{-0.864188,  0.442863,  0.238856}, 
	{-0.716567,  0.681718,  0.147621}, 
	{-0.688191,  0.587785,  0.425325}, 
	{-0.500000,  0.809017,  0.309017}, 
	{-0.238856,  0.864188,  0.442863}, 

	{-0.425325,  0.688191,  0.587785}, 
	{-0.716567,  0.681718, -0.147621}, 
	{-0.500000,  0.809017, -0.309017}, 
	{-0.525731,  0.850651,  0.000000}, 
	{ 0.000000,  0.850651, -0.525731}, 
	{-0.238856,  0.864188, -0.442863}, 
	{ 0.000000,  0.955423, -0.295242}, 
	{-0.262866,  0.951056, -0.162460}, 

	{ 0.000000,  1.000000,  0.000000}, 
	{ 0.000000,  0.955423,  0.295242}, 
	{-0.262866,  0.951056,  0.162460}, 
	{ 0.238856,  0.864188,  0.442863}, 
	{ 0.262866,  0.951056,  0.162460}, 
	{ 0.500000,  0.809017,  0.309017}, 
	{ 0.238856,  0.864188, -0.442863}, 
	{ 0.262866,  0.951056, -0.162460}, 

	{ 0.500000,  0.809017, -0.309017}, 
	{ 0.850651,  0.525731,  0.000000}, 
	{ 0.716567,  0.681718,  0.147621}, 
	{ 0.716567,  0.681718, -0.147621}, 
	{ 0.525731,  0.850651,  0.000000}, 
	{ 0.425325,  0.688191,  0.587785}, 
	{ 0.864188,  0.442863,  0.238856}, 
	{ 0.688191,  0.587785,  0.425325}, 

	{ 0.809017,  0.309017,  0.500000}, 
	{ 0.681718,  0.147621,  0.716567}, 
	{ 0.587785,  0.425325,  0.688191}, 
	{ 0.955423,  0.295242,  0.000000}, 
	{ 1.000000,  0.000000,  0.000000}, 
	{ 0.951056,  0.162460,  0.262866}, 
	{ 0.850651, -0.525731,  0.000000}, 
	{ 0.955423, -0.295242,  0.000000}, 

	{ 0.864188, -0.442863,  0.238856}, 
	{ 0.951056, -0.162460,  0.262866}, 
	{ 0.809017, -0.309017,  0.500000}, 
	{ 0.681718, -0.147621,  0.716567}, 
	{ 0.850651,  0.000000,  0.525731}, 
	{ 0.864188,  0.442863, -0.238856}, 
	{ 0.809017,  0.309017, -0.500000}, 
	{ 0.951056,  0.162460, -0.262866}, 

	{ 0.525731,  0.000000, -0.850651}, 
	{ 0.681718,  0.147621, -0.716567}, 
	{ 0.681718, -0.147621, -0.716567}, 
	{ 0.850651,  0.000000, -0.525731}, 
	{ 0.809017, -0.309017, -0.500000}, 
	{ 0.864188, -0.442863, -0.238856}, 
	{ 0.951056, -0.162460, -0.262866}, 
	{ 0.147621,  0.716567, -0.681718}, 

	{ 0.309017,  0.500000, -0.809017}, 
	{ 0.425325,  0.688191, -0.587785}, 
	{ 0.442863,  0.238856, -0.864188}, 
	{ 0.587785,  0.425325, -0.688191}, 
	{ 0.688191,  0.587785, -0.425325}, 
	{-0.147621,  0.716567, -0.681718}, 
	{-0.309017,  0.500000, -0.809017}, 
	{ 0.000000,  0.525731, -0.850651}, 

	{-0.525731,  0.000000, -0.850651}, 
	{-0.442863,  0.238856, -0.864188}, 
	{-0.295242,  0.000000, -0.955423}, 
	{-0.162460,  0.262866, -0.951056}, 
	{ 0.000000,  0.000000, -1.000000}, 
	{ 0.295242,  0.000000, -0.955423}, 
	{ 0.162460,  0.262866, -0.951056}, 
	{-0.442863, -0.238856, -0.864188}, 

	{-0.309017, -0.500000, -0.809017}, 
	{-0.162460, -0.262866, -0.951056}, 
	{ 0.000000, -0.850651, -0.525731}, 
	{-0.147621, -0.716567, -0.681718}, 
	{ 0.147621, -0.716567, -0.681718}, 
	{ 0.000000, -0.525731, -0.850651}, 
	{ 0.309017, -0.500000, -0.809017}, 
	{ 0.442863, -0.238856, -0.864188}, 

	{ 0.162460, -0.262866, -0.951056}, 
	{ 0.238856, -0.864188, -0.442863}, 
	{ 0.500000, -0.809017, -0.309017}, 
	{ 0.425325, -0.688191, -0.587785}, 
	{ 0.716567, -0.681718, -0.147621}, 
	{ 0.688191, -0.587785, -0.425325}, 
	{ 0.587785, -0.425325, -0.688191}, 
	{ 0.000000, -0.955423, -0.295242}, 

	{ 0.000000, -1.000000,  0.000000}, 
	{ 0.262866, -0.951056, -0.162460}, 
	{ 0.000000, -0.850651,  0.525731}, 
	{ 0.000000, -0.955423,  0.295242}, 
	{ 0.238856, -0.864188,  0.442863}, 
	{ 0.262866, -0.951056,  0.162460}, 
	{ 0.500000, -0.809017,  0.309017}, 
	{ 0.716567, -0.681718,  0.147621}, 

	{ 0.525731, -0.850651,  0.000000}, 
	{-0.238856, -0.864188, -0.442863}, 
	{-0.500000, -0.809017, -0.309017}, 
	{-0.262866, -0.951056, -0.162460}, 
	{-0.850651, -0.525731,  0.000000}, 
	{-0.716567, -0.681718, -0.147621}, 
	{-0.716567, -0.681718,  0.147621}, 
	{-0.525731, -0.850651,  0.000000}, 

	{-0.500000, -0.809017,  0.309017}, 
	{-0.238856, -0.864188,  0.442863}, 
	{-0.262866, -0.951056,  0.162460}, 
	{-0.864188, -0.442863,  0.238856}, 
	{-0.809017, -0.309017,  0.500000}, 
	{-0.688191, -0.587785,  0.425325}, 
	{-0.681718, -0.147621,  0.716567}, 
	{-0.442863, -0.238856,  0.864188},

	{-0.587785, -0.425325,  0.688191}, 
	{-0.309017, -0.500000,  0.809017}, 
	{-0.147621, -0.716567,  0.681718}, 
	{-0.425325, -0.688191,  0.587785}, 
	{-0.162460, -0.262866,  0.951056}, 
	{ 0.442863, -0.238856,  0.864188},
	{ 0.162460, -0.262866,  0.951056}, 
	{ 0.309017, -0.500000,  0.809017}, 

	{ 0.147621, -0.716567,  0.681718}, 
	{ 0.000000, -0.525731,  0.850651},
	{ 0.425325, -0.688191,  0.587785}, 
	{ 0.587785, -0.425325,  0.688191}, 
	{ 0.688191, -0.587785,  0.425325}, 
	{-0.955423,  0.295242,  0.000000}, 
	{-0.951056,  0.162460,  0.262866}, 
	{-1.000000,  0.000000,  0.000000}, 

	{-0.850651,  0.000000,  0.525731}, 
	{-0.955423, -0.295242,  0.000000}, 
	{-0.951056, -0.162460,  0.262866}, 
	{-0.864188,  0.442863, -0.238856}, 
	{-0.951056,  0.162460, -0.262866}, 
	{-0.809017,  0.309017, -0.500000}, 
	{-0.864188, -0.442863, -0.238856}, 
	{-0.951056, -0.162460, -0.262866}, 

	{-0.809017, -0.309017, -0.500000}, 
	{-0.681718,  0.147621, -0.716567}, 
	{-0.681718, -0.147621, -0.716567}, 
	{-0.850651,  0.000000, -0.525731}, 
	{-0.688191,  0.587785, -0.425325}, 
	{-0.587785,  0.425325, -0.688191}, 
	{-0.425325,  0.688191, -0.587785}, 
	{-0.425325, -0.688191, -0.587785}, 

	{-0.587785, -0.425325, -0.688191}, 
	{-0.688191, -0.587785, -0.425325}, 
};


//
//	Generate the normals.
//
//	For each vertex, we need to find all triangles that
//	use this vertex. NOTE THAT IF WE IGNORE SEAMS, all
//	triangles using a given vertex will be in the same
//	body part, which cuts down the search.
//
//	Then we need to calculate a unit vector at (0,0,0)
//	pointing in a direction normal to each triangle.
//
//	Then we need to average all of these unit vectors
//	to get the vertex normal, and make sure it is a
//	unit vector.
//
//	Finally, we need to find the entry in vNorms that
//	is closest to the calculated normal.
//

unsigned char*
MakeLightNormals( fmheader_t* pHeader, RawPlayerModel_t* pModel, vec3dbl_t* pVerts )
{
	unsigned char*	pNorms;
	int				vert, part, inPart, tri, i, j, k, coord, vB;
	int				nVerts, nTris, nNorms;
	vec3dbl_t		norms[MAX_TRIS_PER_VERT];
	int				foundIt;
	vec3dbl_t		u, v, q;
	int				minIDX;
	double			r, minR;

	//	Allocate memory for the result
	if ( !(pNorms=malloc(pHeader->vert3dcnt*sizeof(unsigned char))) )
		return	(void*)0;

	//	Look at all verteces
	nVerts = pHeader->vert3dcnt;
	for ( vert=0; vert<nVerts; vert++ )
	{
		nNorms = 0;
		//	Find the part this vertex is in
		inPart = -1;
		for ( part=0; part<NUM_BODY_PARTS; part++ )
		{
			if ( pModel->parts[part] )
				inPart = part;
			if ( vert < pModel->parts[part]->vertBase ) 
				break;
		}
		//	If we break, the vertex is in part inPart - 1.
		//	If we don't break, the vertex is in part inPart
		if ( part != NUM_BODY_PARTS )
			--inPart;
		vB = pModel->parts[inPart]->vertBase;
		//	Find the triangles of interest in this part
		nTris = pModel->parts[inPart]->trisCount;
		for ( tri=0; tri<nTris; tri++ )
		{
			foundIt = 0;
			for ( i=0; i<3; i++ )
			{
				if ( pModel->parts[inPart]->pTris[tri].v[i] == (vert-vB) )
				{
					foundIt = 1;
					break;
				}
			}
			if ( foundIt )
			{
				//	OK this triangle contains our vertex.
				//	Calculate and store its normal. The
				//	normal is the cross product of the
				//	vectors 0->1 x 0->2
				i = pModel->parts[inPart]->pTris[tri].v[0];	//	Index of vertex #0
				j = pModel->parts[inPart]->pTris[tri].v[1];	//	Index of vertex #1
				k = pModel->parts[inPart]->pTris[tri].v[2];	//	Index of vertex #2
				for ( coord=0; coord<3; ++coord )
				{
					v.c[coord] = pModel->parts[inPart]->pVert3D[j].c[coord] - pModel->parts[inPart]->pVert3D[i].c[coord];
					u.c[coord] = pModel->parts[inPart]->pVert3D[k].c[coord] - pModel->parts[inPart]->pVert3D[i].c[coord];
				}
				M3DCrossProduct( &(norms[nNorms]), &v, &u );
				M3DUnitVector( &(norms[nNorms]), &(norms[nNorms]) );
				if ( ++nNorms >= MAX_TRIS_PER_VERT )
				{
					//	Hmmm..This vertex is too complex. This need not
					//	be a fatal error - just ignore the preceding normal
					--nNorms;
				}
			}
		}
		//	OK we have all the normals for this vertex. Average them to get the
		//	vertex normal
		q.c[0] = 0.;
		q.c[1] = 0.;
		q.c[2] = 0.;
		for ( i=0; i<nNorms; i++ )
		{
			for ( coord=0; coord<3; coord++ )
				q.c[coord] += norms[i].c[coord];
		}
		M3DUnitVector( &q, &q );
		//	Now, find the closest pre-computed normal by minimizing the distance between
		//	the endpoint of our vertex normal and the enpoint of the pre-computed normal.
		minR = 100;
		minIDX = -1;
		for ( i=0; i<FM_NUMVERTEXNORMALS; i++ )
		{
			r = M3DSqrLength( &q, &(vNorms[i]) );
			if ( r<minR )
			{
				minR = r;
				minIDX = i;
			}
		}
		if ( minIDX == -1 )
			minIDX = 0;
		//	We have a winner!
		pNorms[vert] = (unsigned char)minIDX;
	}

	//	All done
	return	pNorms;
}
