//
//	Code to create lists of GLCMDS for Heretic II player models
//

#include	<stdio.h>
#include	<stdlib.h>

#include	"CondComp.h"
#include	"FlexModel.h"
#include	"bones.h"
#include	"BodyPart.h"
#include	"glcmd.h"
#include	"h2bitmaps.h"


//
//	Given a list of triangles, a "qualification bitmap", a triangle
//	to exclude, and pair of vertex IDs defining an edge, determine
//	whether any other qualified triangle in the list shares the edge.
//
//	Note that all qualified triangles are assumed to share the other
//	endpoint of the edge being looked for.
//
//	The verteces of the matching edge may be defined in any order.
//
//	Return the triangle #, or -1 if no match
//

static int
WhoSharesEdge( trividx_t* pTris, h2bitmap_t* pQual, int notMe, int v2 )
{
	int		tri;
	int		foundIt;
	int		i;

	//	v2 is vertex number of far end of shared (?) edge.
	//	notMe is index of triangle we're trying to match
	//	Find someone else who shares this edge.
	foundIt = 0;
	tri = -1;
	while ( !foundIt && ((tri = FindSetBitMap( pQual, tri+1, 1 )) != -1) )
	{
		if ( tri != notMe )
		{
			//	tri is another v1 qualified triangle
			//	Is it also a v2 triangle?
			for ( i=0; i<3; i++ )
			{
				if ( pTris[tri].v[i] == v2 )
				{
					foundIt = 1;
					break;		//	Found other endpoint of desired edge
				}
			}
		}
	}
	if ( !foundIt )
		tri = -1;
	return	tri;
}


//
//	Given a set of body parts each made of triangles, create a
//	list of GLCmds for each body part and keep track of the
//	location and number of GLCmds in each list.
//
//	A non-optimized algorithm is used. This algorithm uses
//	ONLY trifans - no tristrips are generated.
//
//	Note that no two body parts may contain the same vertex.
//	Note that at most two triangles in a given body part may
//	share an edge.
//
//	1)	For each body part
//	2)	Mark all verteces in this part as "not used"
//	3)	For each vertex that is not used, determine how many
//		triangles that include only unused verteces (aka
//		"qualified triangles") include this vertex
//	4)	(Mark any verteces not used in any triangles, as used)
//	5)	Pick the vertex used in the most qualified triangles
//	6)	Build a list of the qualified triangles for this vertex
//	*** Steps in case the mesh is not completely closed
//	7)	Determine whether an edge exists in any qualified
//		triangle, consisting of this vertex and the vertex
//		"to the right mod 3" in the definition of the triangle,
//		which edge is not shared with any other qualified
//		triangle. WARNING: FAILS IF NORMALS ARE DEFINED WRONG
//	8)  If such an edge exists, pick the corresponding qualified
//		triangle as the starting triangle; else, pick any qualified
//		triangle as the starting triangle
//	*** End special steps
//	9)	Generate a TRIFAN GLCmd list for all the qualified
//		triangles
//	10)	Mark this vertex (the pivot of the TRIFAN) as used
//	11)	If any verteces are unused, go to step 3 (next part),
//		else done.
//

//	WARNING: THIS DOES NOT HANDLE SEAMS CORRECTLY YET


GLCmdInfo_t*
MakeGLCmds( fmheader_t* pHeader, RawPlayerModel_t* pModel )
{
	GLCmdInfo_t*	pInfo;
	int				i, j, tri, vert, part, triBase, vertBase, v2, t2, tV;
	int				glBase, glCount;
	h2bitmap_t*		pUsedMap;
	h2bitmap_t*		pQualMap;
	short*			pUseCount;
	int				numVerts, numTris, maxVert, maxUse;
	int				foundIt;

	//	Allocate a data structure to return. The size of the cmds[]
	//	array is bounded in that, worst case, there is one command
	//	struct and 3 vertex structs per triangle. Since we don't know
	//	the size up front, we allocate a safe size.
	j = sizeof(GLCmdInfo_t) + (4 * pHeader->tricnt * sizeof(glvert_t));		//	Bytes needed for GLCMDS (MAX)
//	printf("Allocating space for %d gldmds...\n", 4 * pHeader->tricnt);
	if ( !(pInfo=malloc(j)) )
		return	pInfo;
	for ( i=0; i<NUM_BODY_PARTS; i++ )
	{
		pInfo->startGLCmds[i] = 0;
		pInfo->numGLCmds[i] = 0;
	}

	//	Step 1: for each body part...
	glBase   = 0;
	for ( part=0; part<NUM_BODY_PARTS; part++ )
	{
		//	NOTE: Not all parts or seams may be present
		if ( pModel->parts[part] )
		{
			//	Keep starting GL commands index for each body part
			pInfo->startGLCmds[part] = glBase;

			//	Get number of verteces and triangles for this body part
			numVerts = pModel->parts[part]->vert3DCount;
			vertBase = pModel->parts[part]->vertBase;
			numTris  = pModel->parts[part]->trisCount;
			triBase  = pModel->parts[part]->triBase;

			//	Allocate a bitmap to keep track of the used verteces, and
			//	an array to keep track of usage count for each vertex
			if ( !(pUsedMap = NewBitMap( numVerts )) )	//	Step 2: Allocate housekeeping / mark unused
			{
				free( pInfo );
				return	(void*)0;
			}
			if ( !(pQualMap = NewBitMap( numTris )) )
			{
				free( pInfo );
				FreeBitMap( pUsedMap );
				return	(void*)0;
			}
			if ( !(pUseCount = malloc( numVerts * sizeof(short) )) )
			{
				free( pInfo );
				FreeBitMap( pUsedMap );
				FreeBitMap( pQualMap );
				return	(void*)0;
			}

			//	Main loop to process GLCmds, runs until there are no
			//	verteces left to process in this body part.
			glCount = 0;			//	Count of number of entries in cmds[] used by this body part
			while ( 1 )
			{
				//	Step 3: Get usage counts and identify qualified triangles
				for ( vert=0; vert<numVerts; vert++ )
					pUseCount[vert] = 0;
				ClrBitMap( pQualMap, 0, numTris );				//	Erase entire bit map
				//	Check each triangle in the body part;
				for ( tri=0; tri<numTris; tri++ )
				{
					//	Check each vertex of the triangle - is it a "qualified triangle"?
					foundIt = 1;
					for ( i=0; i<3; i++ )
					{
						if ( TstBitMap( pUsedMap, pModel->parts[part]->pTris[tri].v[i] ) )
						{
							foundIt = 0;
							break;
						}
					}
					//	If i gets to 3 here, we know that none of the verteces are marked used
					//	so this is a qualified triangle and we can increment each vertex' use count
					if ( foundIt )
					{
						for ( i=0; i<3; i++ )
							++pUseCount[ pModel->parts[part]->pTris[tri].v[i] ];
						SetBitMap( pQualMap, tri, 1 );			//	Mark qualified triangle
					}
					else
						i=i;
				}
				//	Step 4: Cull unused verteces
				//	If a vertex is marked as "used", it means that all triangles
				//	containing that vertex have been processed. If we can't find
				//	any qualified triangles containing a given vertex, it means
				//	that the vertex appears only in triangles where at least one
				//	vertex is marked as "used". This indicates that all triangles
				//	containing the vertex have been processed (via other verteces)
				//	and so the vertex should be marked as "used".
				//
				//	Step 5: Select highest-use vertex
				maxVert = -1;
				maxUse  = -1;
				for ( vert=0; vert<numVerts; vert++ )
				{
					if ( !pUseCount[vert] )					//	Cull unused vertex
						SetBitMap( pUsedMap, vert, 1 );
					else
					if ( pUseCount[vert] > maxUse )			//	Track highest use count
					{
						maxVert = vert;
						maxUse = pUseCount[vert];
					}
				}
				//	Part of step 11; Stop if all verteces are used
				if ( maxVert == -1 )
					break;
				//	Step 6: Cull qualified triangles to only those containing maxVert
				tri = -1;
				while ( (tri = FindSetBitMap( pQualMap, tri+1, 1 )) != -1 )
				{
					//	Tri is a qualified triangle
					foundIt = 0;
					for ( i=0; i<3; i++ )
					{
						if ( pModel->parts[part]->pTris[tri].v[i] == maxVert )
						{
							foundIt = 1;
							break;
						}
					}
					if ( !foundIt )							//	Cull triangle not having this vertex
						ClrBitMap( pQualMap, tri, 1 );
				}
				//	Step 7: Determine whether unique edges require special starting triangle
				tri = -1;
				foundIt = 0;
				while ( !foundIt && ((tri = FindSetBitMap( pQualMap, tri+1, 1 )) != -1) )
				{
					//	Tri is a qualified triangle containing our vertex
					for ( i=0; i<3; i++ )
					{
						//	WARNING: Debugger screwed up when these braces weren't here!!??
						if ( pModel->parts[part]->pTris[tri].v[i] == maxVert )
							break;
					}
					//	i is index of pivot vertex in triangle definition
					v2 = pModel->parts[part]->pTris[tri].v[ (i+1) % 3 ];
					//	v2 is vertex number of far end of shared (?) edge
					//	Find someone else who shares this edge.
					t2 = WhoSharesEdge( pModel->parts[part]->pTris, pQualMap, tri, v2 );
					if ( t2 == -1 )
					{
						//	Hmmm..found a unique edge with the correct "handedness"
						//	so tri is our starting triangle
						foundIt = 1;
					}
				}
				//	Step 8: Choose starting triangle
				if ( !foundIt )					//	Any qualified triangle will do if no unique edges
					tri = FindSetBitMap( pQualMap, 0, 1 );
				else
					tri = tri;
				//	Step 9: Generate a TRIFAN
//				printf( "TRIFAN:   %d triangles around body part %d, vertex %d\n", maxUse, part, maxVert );
				//	1) Output the GL_TRIFAN * (maxUse+2) command
				//	2) Output starting triangle verteces in proper order (maxVert first)
				//	3) while (not all triangles in this set done)
				//	4)  Find THE triangle containing maxVert AND the last vertex output
				//	5)  Output its unique vertex
				//	6)  goto 4/5
				pInfo->cmds[glBase+glCount++].index_3d = GL_TRIFAN*(maxUse+2);	//	substep 1
//				printf( "    %d", GL_TRIFAN*(maxUse+2) );
				for ( i=0; i<3; i++ )
				{
					//	WARNING: Debugger screwed up when these braces weren't here!!??
					if ( pModel->parts[part]->pTris[tri].v[i] == maxVert )
						break;
				}
				//	Now tri is the index of the matching triangle, and i is the index of
				//	the matching vertex.
				for ( j=0; j<3; j++ )									//	substep 2
				{
					//	Vertex information, adjusted by base vertex for this body part
					v2 = vertBase + pModel->parts[part]->pTris[tri].v[ (i+j)%3 ];
					pInfo->cmds[glBase+glCount].index_3d = v2;
//					printf( " %d", v2 );
					//	Texture information (absolute) scaled to 0..1 range
					tV = pModel->parts[part]->pTrisTex[tri].v[ (i+j)%3 ];
					pInfo->cmds[glBase+glCount].u = (float)( (float)(pModel->parts[part]->pVertUV[tV].c[0]) / 256. );
					pInfo->cmds[glBase+glCount].v = (float)( (float)(pModel->parts[part]->pVertUV[tV].c[1]) / 256. );
					++glCount;
				}
				//	Cleverly, v2 will be set to the last vertex output (adjusted by body part base vertex)
				while (--maxUse)										//	substep 3
				{
																		//	substep 4
					v2 -= vertBase;
					t2 = WhoSharesEdge( pModel->parts[part]->pTris, pQualMap, tri, v2 );
					if ( t2 == -1 )
					{
						//	This means that we can't find a triangle that shares an edge
						//	with us. This should NEVER occur, unless the model has dangling
						//	triangles (in which case the model is broken).
						printf( "\nNothing shares edge (%d, %d)!!\n", maxVert, v2 );
					}
					for ( i=0; i<3; i++ )								//	substep 5
					{
						//	WARNING: Debugger screwed up when these braces weren't here!!??
						if ( pModel->parts[part]->pTris[t2].v[i] == maxVert )
							break;
					}
					for ( j=0; j<3; j++ )
					{
						//	WARNING: Debugger screwed up when these braces weren't here!!??
						if ( pModel->parts[part]->pTris[t2].v[j] == v2 )
							break;
					}
					//	Vertex information, adjusted by base vertex for this body part
					if ( ((i==0)&&(j==1)) || ((j==0)&&(i==1)) )			//	substep 6
						i = 2;												//	Position of unique vertex of triangle t2
					else
					if ( ((i==0)&&(j==2)) || ((j==0)&&(i==2)) )
						i = 1;												//	Position of unique vertex of triangle t2
					else
						i = 0;												//	Position of unique vertex of triangle t2
					v2 = vertBase + pModel->parts[part]->pTris[t2].v[i];	//	Unique vertex of triangle t2
					pInfo->cmds[glBase+glCount].index_3d = v2;
//					printf( " %d", v2 );
					//	Texture information (absolute) scaled to 0..1 range
					tV = pModel->parts[part]->pTrisTex[t2].v[i];
					pInfo->cmds[glBase+glCount].u = (float)( (float)(pModel->parts[part]->pVertUV[tV].c[0]) / 256. );
					pInfo->cmds[glBase+glCount].v = (float)( (float)(pModel->parts[part]->pVertUV[tV].c[1]) / 256. );
					++glCount;

					//	t2 is index of triangle just processed
					//	v2 is unique vertex of TRIFAN just output, aadjusted by base vertex of body part
					tri = t2;
				}
//				printf("\n");
				//	Count number of cmd[] entries generated (implicit above)
				//	Mark this vertex as USED
				SetBitMap( pUsedMap, maxVert, 1 );					//	Step 10
				//	Part of step 11: Keep going
			}
			//	Done with this part
			FreeBitMap( pUsedMap );
			FreeBitMap( pQualMap );
			free( pUseCount );
		}
		else
			glCount = 0;
		//	Keep track of how many GLCmds are generated for each body part.
		//	Keep a running total of verteces (starting index for each body part).
		//	Move on to next body part
		pInfo->numGLCmds[part] = glCount;
//		printf( "%d total glcmd entries starting at offset %d for body part %d.\n", glCount, glBase, part );
		glBase += glCount;
	}

	//	All done
	return	pInfo;
}

