//
//	Definitions of data structures used to hold a body part
//	as it is read into memory.
//
//	The initial definition of a body part consists of its
//	bone number, the number of 3D verteces, the number of
//	texture verteces, the number of triangles, pointers
//	to the vertex and triangle data, a pointer to triangle
//	mapping data, the dimensions of the texture map, and
//	a pointer to the raw data (24 bits deep) of the texture
//	map.
//
//
//	There are 36 body parts, but only 16 meshnodes in a
//	player model. Two or more body parts are made part of
//	a meshnode using a "seam", a set of triangles not in
//	either body part that join the ends of two body parts.
//
//	There are some problems associated with seams. Here
//	are some of them:
//
//	1)	Texuring of seams is going to be tough if each body
//		part gets its own texture map - where is the seam
//		texture specified?
//
//	2)	The front and back torso meshnodes don't correspond
//		at all to bones / body parts.
//
//	3)	How to figure out what 3D points belong in each seam,
//		and what the triangles of the seam are?
//
//	To solve problems #1 and #3, this program requires a model
//	for each meshnode in addition to body part models. The
//	meshnode model is made by merging all the body parts in
//	the meshnode, creating all the seams, and using a single
//	texture map on the whole meshnode - all without adding
//	any new verteces to any of the body parts and without
//	creating any new verteces to make the seams. The program
//	then "subtracts" the body part models from the meshnode
//	model to find the triangles that make up the seams, and
//	takes the texture coordinates for all parts and seams
//	seams in the meshnode from the single texture map.
//
//	To solve problem #2, a complete torso model is input
//	instead of separate models for the front torso and back
//	torso meshnodes. The meshnodes and seams are then generated
//	automatically based on the Z-coordinate of each vertex.
//
//	NOTE: The legs are a separate meshnode from the torso,
//	but seams are needed for the leg-to-hip joints. Not sure
//	how to handle this other than to require a model of
//	legs, head. and torso. And if we're doing that, why not
//	a full body model?
//
//	If no meshnode models are provided, no attempt is made
//	to create seams and problem #2 is solved by dividing the
//	model into front and back torso based on Z coordinate of
//	each vertex in each body part.
//
//	The meshnode models needed for seams are:
//	Head Node = head + neck
//	Front Torso Node = front of(
//						hips left + hips right + hips spine +
//						lower torso + left upper torso +
//						right upper torso
//					   )
//	Back Torso Node = back of(
//						hips left + hips right + hips spine +
//						lower torso + left upper torso +
//						right upper torso
//					   )
//	Left Leg Node = left upper leg + left lower leg + left foot
//	Right Leg Node = right upper leg + right lower leg + right foot
//	Left Arm Node = left upper arm + left lower arm
//	Right Arm Node = right upper arm + right lower arm
//	Left Bow Node = left hand closed + bow
//	Armor Node = armor left + left pad + armor right + right pad
//
//	NOTE: The bow node has an alternate geometry in frames
//	Draw1-Draw5. This is not a motion capture issue; there is
//	no indication of this in the motion capture data. The 2nd
//	geometry is used in these frames to show the nocked arrow.
//	The generic way to handle this is to allow any body part
//	to have an alternate geometry, and include a list of frames
//	in which the alternate geometry is used.

#ifndef	__BODYPARTHINCLUDED__
#define	__BODYPARTHINCLUDED__

//	The number of body parts in a complete player model
#define	NUM_BODY_PARTS	(BONE_STORED_STAFF+1)	//	Also BONE_LH_REF

//  Names for the seams - note that these ARE NOT the same
//	as the joints, because:
//	1)	The hip position and extremities count as joints
//		but not as seams;
//	2)	The front and back torso seam does not correspond
//		to a joint;
//	3)	Seams are only used within meshnodes.
//		XYZZY WARNING: NOT TRUE, THERE SHOULD ALSO BE A
//		SEAM BETWEEN LEGS AND TORSO
//	4)	Some meshnodes do not use seams (e.g. bow in left
//		hand, hellstaff)
//	5)	Some meshnodes include seams that have no
//		corresponding joint.
//

enum
{
	SEAM_HIP_LEFT_UPPER_LEG = 0,
	SEAM_LEFT_UPPER_LOWER_LEG,
	SEAM_LEFT_LOWER_LEG_FOOT,
	SEAM_HIP_RIGHT_UPPER_LEG,
	SEAM_RIGHT_UPPER_LOWER_LEG,
	SEAM_RIGHT_LOWER_LEG_FOOT,
	SEAM_HIP_LOWER_TORSO,
	SEAM_LOW_TORSO_HIGH_TORSO,
	SEAM_TORSO_LEFT_UPPER_ARM,
	SEAM_LEFT_UPPER_LOWER_ARM,
	SEAM_LEFT_LOWER_ARM_OPEN_HAND,
	SEAM_LEFT_LOWER_ARM_CLOSED_HAND,
	SEAM_TORSO_RIGHT_UPPER_ARM,
	SEAM_RIGHT_UPPER_LOWER_ARM,
	SEAM_RIGHT_LOWER_ARM_OPEN_HAND,
	SEAM_RIGHT_LOWER_ARM_CLOSED_HAND,
	SEAM_TORSO_NECK,
	SEAM_NECK_HEAD,
	SEAM_HIP_LEFT_MIDDLE,
	SEAM_HIP_RIGHT_MIDDLE,
	SEAM_FRONT_BACK_TORSO,
	SEAM_RIGHT_LEFT_ARMOR,
	SEAM_NUM_SEAMS
};

//	Structure for storing a single body part
#define	MAX_ALT_GEOMETRY_FRAMES	32
typedef struct
{
	//	Housekeeping stuff
	int				boneNumber;				//	Which bone is this (codes from bones.h, e.g. BONE_HIPS_SPINE)
	imp_point_t		refPoint;				//	Reference or skeletal point for this body part

	//	Mesh stuff
	int				vert3DCount;			//	Number of 3D verteces in the mesh
	int				trisCount;				//	Number of triangles in the mesh
	vec3_t*			pVert3D;				//	Array of (X,Y,Z) coordinates of points in the mesh
	trividx_t*		pTris;					//	Array of 3X indeces into pVert3D giving the corners of each
											//	triangle in 3D space, ordered using left-hand normal rule
	//	Texture stuff
	int				vertUVCount;			//	Number of texture verteces in the mesh
	texpoint_t*		pVertUV;				//	Array of (U,V) coordinates of points in texture space
	trividx_t*		pTrisTex;				//	Array of 3X indeces into pVertUV giving the corners of each
											//	triangle in UV space, corresponding to verteces of 3D space
											//	triangles defined in pTris.
	int				texWidth;				//	Width (in pixels) of the texture data
	int				texHeight;				//	Height (in pixels) of the texture data
	int				texID;					//	Texture source identifier (allows several body parts to use same texture)
	unsigned char*	pTexPixels;				//	Raw pixel data for the texture (always 24 bits deep R-G-B)

	//	Overall stuff
	int				vertBase;				//	Add to local vertex # to get global vertex #
	int				triBase;				//	Add to local triangle # to get global triangle #
	int				texBase;				//	Add to local texture vertex # to get global texture vertex #

	//	Alternate geometry stuff
	vec3_t*			pAltVert3D;				//	Alternate array of (X,Y,Z) coordinates (for variable geometry)
	char			altFrames[MAX_ALT_GEOMETRY_FRAMES+1][16];	//	List of frame names in which to use alternate
}	RawBodyPart_t;

//
//	Structure for storing a seam.
//	This is the same as that for a body part, but:
//	1)	BoneNumber is replaced by a Joint Number + 1000
//		(e.g. JOINT_HIP_RIGHT_UPPER_LEG + 1000)
//	2)	vert3DCount is always zero, and pVert3D is always NULL
//	3)	Elements in pTris indicate the vertex number in the
//		associated body part model, and are offset by 1000 *
//		the bone number of the body part in which the vertex
//		is found
//	4)	texID and pTexPixels ALWAYS refer to pixel data that is
//		already in use by a body part / meshnode.
//

//
//	Structure for storing a meshnode.
//	This is the same as that for a body part, but:
//	1)	BoneNumber is replaced by a MeshNode Number + 2000
//		(e.g. H2MODELNODE_HEAD + 2000)
//	2)	texID and pTexPixels ALWAYS refer to pixel data that is
//		already in use by a body part.
//

//	Structure for storing the set of all body parts
//	(except reference bones)
typedef struct
{
	RawBodyPart_t*	parts[NUM_BODY_PARTS];	//	Every part in the body
	RawBodyPart_t*	nodes[H2NUM_MESHNODES];	//	Every meshnode in the body (optional)
	RawBodyPart_t*	seams[SEAM_NUM_SEAMS];	//	Every seam in the body (optional)
}	RawPlayerModel_t;

//
//	Structure for storing context information associated
//	with a set of body parts (don't know what, yet, but
//	probably a list of file names and some other stuff)
//

typedef struct
{
	long			fubar;					//	DUMMY
}	BodyContext_t;


//	Function prototypes

extern RawPlayerModel_t*	MakeStickRawPlayerModel( BodyContext_t* pContext );
extern RawPlayerModel_t*	ReadRawPlayerModel( BodyContext_t* pContext );
extern void					DisposeRawPlayerModel( RawPlayerModel_t* pBody );
extern void					DisposeBodyPart( RawBodyPart_t* pBP );


#endif
