//
//	Extract motion capture data from a Heretic II FlexModel file
//
//	By Chris Burke
//	serotonin@earthlink.net
//
//	Date			Who		Description
//	----------		---		-------------------------------------------------------
//	07/01/1999		CJB		Coded
//


//	System includes
#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<ctype.h>
#include	<fcntl.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<io.h>

//	Local includes
#include	"CondComp.h"
#include	"FlexModel.h"
#include	"MatrixMath.h"

//
//	Externs
//

//	Globals
int				gModelFID;
FILE*			gModelFile;
char*			gModelFileName = (char*)0;
char*			gProgramName;
long			gModelSize, gNewModelSize;
int				gNumFooFrames;

meshnode_t*		gpMeshNodes;
glcmd_t*		gpGLCmds;
frame_t*		gpFirstFrame;
fmheader_t*		gpModelHeader;
refskel_t*		gpFirstRef;

unsigned char*	gpModelData;
unsigned char*	gpNewModelData;
char			gScratchString[128];

unsigned short	optA = 0;
unsigned short	optL = 0;
char*			optO = (char*)0;

//
//	Utils
//

void quit( char* pMsg, int errCode )
{
	if ( pMsg && *pMsg )
		printf( "\nFATAL: %s\n", pMsg );
	exit( errCode );
}

void usage( char* pMsg, int errCode )
{
	printf(
		"\n"
		"Program: %s\n"
		"Purpose: Set paths to skins for Heretic II FlexModels\n"
		"\n"
		"Usage:\n"
		"%s [/A] [/L] [/O outfile.fm] infile.fm skinpath [skinpath...]\n"
		"The /A option appends paths to current list (else replace current list).\n"
		"The /L option just lists the current skin paths without changing the model.\n"
		"The /O option writes output to a new file (else input file is overwritten).\n",
		gProgramName, gProgramName
	);
	quit( pMsg, errCode );
}

main ( int argc, char **argv )
{
	FILE*		fout;
	header_t	theHeader;
	fmheader_t	theModelHeader;
	byte*		pIntoModel;
	byte*		pIntoNewModel;

	//	Parse arguments
	gProgramName = *argv++; --argc;
	printf(
		"Executing %s\n"
		"by Chris Burke (serotonin@earthlink.net)....\n",
		gProgramName
	);

	while (argc)
	{
		if ( (!strcmp(*argv,"/a")) || (!strcmp(*argv,"/A")) )
		{
			//	Append
			optA = 1;
		}
		else
		if ( (!strcmp(*argv,"/l")) || (!strcmp(*argv,"/L")) )
		{
			//	List
			optL = 1;
		}
		else
		if ( (!strcmp(*argv,"/o")) || (!strcmp(*argv,"/O")) )
		{
			//	Output file
			++argv; --argc;
			if ( !argc )
				usage( "/O requires output file name", -1 );
			//	List
			optO = *argv;
		}
		else
		{
			//	Input file
			gModelFileName = *argv;
			//	Remaining arguments are skin path names
			++argv; --argc;
			if ( !argc && !optA && !optL )
				usage( "At least one skin path must be specified", -1 );
			break;
		}
		++argv; --argc;
	}

	//	Trivial check for right # of arguments
	if ( !gModelFileName )
		usage("No model file specified",-1);
	if ( !optO )
		optO = gModelFileName;

	//	Make sure the model file exists
	printf("Input Model File: %s\n", gModelFileName);
	if ( !( gModelFile = fopen( gModelFileName, "rb" ) ) )
		usage("Can't open input model file", errno);

	//	Get size of model file and suck it into allocated memory
	fseek( gModelFile, 0L, SEEK_END );
	gModelSize = ftell( gModelFile );
	if ( !( gpModelData = malloc( gModelSize ) ) )
		quit("Can't allocate memory to read model", -1);	fseek( gModelFile, 0L, SEEK_SET );
	fread( gpModelData, gModelSize, 1L, gModelFile );
	fclose( gModelFile );
	pIntoModel = gpModelData;
	printf( "%d bytes\n", gModelSize );

	if ( !optL )
	{
		//	Allocate memory for new model. If optA is set, we
		//	keep old skins and add new ones; else we just add
		//	new ones. So the optA case is largest. Since we
		//	haven't seen the skins chunk or the model header
		//	chunk yet, we don't know how big to make the model
		//	in the non-optA case. So we allocate as if optA were
		//	set, to make sure we always have enough memory.
		gNewModelSize = gModelSize + argc*sizeof(skinpath_t);		//	As if we were always appending
		if ( !( gpNewModelData = malloc( gNewModelSize ) ) )
			quit("Can't allocate memory for modified model", -1);
		pIntoNewModel = gpNewModelData;
	}

	//	Perform consistency check on model
	while ( pIntoModel < gpModelData + gModelSize )
	{
		//	Read the header for the next chunk
		memcpy( theHeader.ident, pIntoModel, FM_MAXCHUNKIDENT );
		theHeader.ident[FM_MAXCHUNKIDENT-1] = '\0';
		theHeader.version	= FM_GETINT( pIntoModel + FM_MAXCHUNKIDENT + 0 );
		theHeader.size		= FM_GETINT( pIntoModel + FM_MAXCHUNKIDENT + 4 );
		printf(
			"%-32s version %d, length %d bytes.\n",
			theHeader.ident, theHeader.version, theHeader.size );
		if ( theHeader.size <= 0 )
			quit("Invalid (<=0) chunk size in model",-1);

		if ( !optL )
		{
			//	Copy each chunk from old model to new model
			//	Note that if this turns out to be a skin chunk,
			//	the copy in the new model may be larger or
			//	smaller in size.
			memcpy( pIntoNewModel, pIntoModel, FM_HEADERSIZE + theHeader.size );
			pIntoNewModel += (FM_HEADERSIZE + theHeader.size);
		}
		
		//	Process by chunk type
		if ( !strcmp( theHeader.ident, FM_HEADERCHUNKNAME ) )
		{
			//	A HEADER chunk
			if ( theHeader.version != FM_HEADERCHUNKVER )
				quit("Invalid HEADER chunk version number",-1);
			gpModelHeader = (fmheader_t*)(pIntoModel + FM_HEADERSIZE);
			theModelHeader.skinwidth	= FM_GETINT( pIntoModel + FM_HEADERSIZE + 0);	// in pixels
			theModelHeader.skinheight	= FM_GETINT( pIntoModel + FM_HEADERSIZE + 4);	// in pixels
			theModelHeader.framesize	= FM_GETINT( pIntoModel + FM_HEADERSIZE + 8);	// byte count of each frame
			theModelHeader.skincnt		= FM_GETINT( pIntoModel + FM_HEADERSIZE + 12);	// number of skins
			theModelHeader.vert3dcnt	= FM_GETINT( pIntoModel + FM_HEADERSIZE + 16);	// number of unique vertices
			theModelHeader.vertuvcnt	= FM_GETINT( pIntoModel + FM_HEADERSIZE + 20);	// greater than num_xyz for seams
			theModelHeader.tricnt		= FM_GETINT( pIntoModel + FM_HEADERSIZE + 24);	// number of triangles
			theModelHeader.glcmdcnt		= FM_GETINT( pIntoModel + FM_HEADERSIZE + 28);	// dwords in strip/fan command list
			theModelHeader.framecnt		= FM_GETINT( pIntoModel + FM_HEADERSIZE + 32);	// number of animation frames
			theModelHeader.meshnodecnt	= FM_GETINT( pIntoModel + FM_HEADERSIZE + 36);	// number of mesh nodes
			//	Dump all the data
			printf(
//				"    Skins are %d x %d pixels.\n"
				"    %d skins identified.\n",
//				"    %d animation frames, size %d each.\n"
//				"    %d 3D vertices, %d texture verteces, and %d glcmds.\n"
//				"    %d triangles, %d mesh nodes.\n",
//				theModelHeader.skinwidth, theModelHeader.skinheight,
				theModelHeader.skincnt
//				theModelHeader.framecnt, theModelHeader.framesize,
//				theModelHeader.vert3dcnt, theModelHeader.vertuvcnt,
//				theModelHeader.glcmdcnt,
//				theModelHeader.tricnt, theModelHeader.meshnodecnt
			);
		}
		else
		if ( !strcmp( theHeader.ident, FM_SKINCHUNKNAME ) )
		{
			//	A SKIN chunk
			int		i;
			char*	pSkinName = pIntoModel + FM_HEADERSIZE;
			skinpath_t theSkinPath;
			skinpath_t	newName;
			fmheader_t*	pNewHeader;
			header_t*	pNewChunkHeader;
			if ( theHeader.version != FM_SKINCHUNKVER )
				quit("Invalid SKIN chunk version number",-1);
			//	Find model header of new model
			pNewHeader = (fmheader_t*)(gpNewModelData + FM_HEADERSIZE);
			//	Find header of SKIN chunk in new model
			pNewChunkHeader = (header_t*)( (byte*)pIntoNewModel - (FM_HEADERSIZE + theHeader.size) );
			//	Handle the two cases
			if ( optA || optL )
				printf("    Current skin paths are:\n");
			else
				printf("    Deleting current skin paths:\n");
			//	Dump all the data (using old model)
			for ( i = 0; i < theModelHeader.skincnt; i++ )
			{
				memcpy( theSkinPath.path, pSkinName, FM_MAXPATHLENGTH );
				theSkinPath.path[FM_MAXPATHLENGTH-1] = '\0';
				printf("    %s\n", theSkinPath.path);
				pSkinName += FM_SKINPATHSIZE;
			}
			if ( !optL )
			{
				if ( optA )
				{
					//	gNewModelSize is already set to the correct size.
					//	pIntoNewModel already points at end of chunk.
					//	pNewChunkHeader->size already has correct starting size
					//	pNewHeader->skincnt already has correct starting count
				}
				else
				{
					//	Calculate actual size of new model
					gNewModelSize -= theModelHeader.skincnt*sizeof(skinpath_t);
					//	Adjust pointers and counts to delete current skin names
					pIntoNewModel = (unsigned char*)(pNewChunkHeader + 1);		//	NOTE: pointer math
					pNewChunkHeader->size = 0;
					pNewHeader->skincnt = 0;
				}
				//	Extend the SKIN chunk of the new model,
				//	and adjust the count in the model header.
				//	WARNING: THIS USES DIRECT ACCESS TO MODEL
				//	IN MEMORY, MIGHT NOT BE ENDIAN-SAFE ON
				//	OTHER THAN INTEL PLATFORM.
				while ( argc-- )
				{
					printf( "    Adding #%d skin name '%s'...\n", pNewHeader->skincnt+1, *argv );
					//	Copy new name to end of SKIN chunk
					//	NOTE: This assumes that the SKIN chunk is exactly
					//	the size needed to store the specified number of
					//	skin paths.
					memset( newName.path, 0, FM_MAXPATHLENGTH );
					strcpy( newName.path, *argv++ );
					memcpy( pIntoNewModel, &newName, sizeof(newName) );
					pIntoNewModel += sizeof(newName);
					//	Revise chunk size and skin count
					pNewChunkHeader->size += sizeof(newName);
					++pNewHeader->skincnt;
				}
			}
		}
		else
		if ( !strcmp( theHeader.ident, FM_STCOORDCHUNKNAME ) )
		{
			//	An ST COORD chunk
			byte*	pCoords = pIntoModel + FM_HEADERSIZE;
			if ( theHeader.version != FM_STCOORDCHUNKVER )
				quit("Invalid ST COORD chunk version number",-1);
		}
		else
		if ( !strcmp( theHeader.ident, FM_TRISCHUNKNAME ) )
		{
			//	A TRIS chunk
			byte*		pCoords = pIntoModel + FM_HEADERSIZE;
			if ( theHeader.version != FM_TRISCHUNKVER )
				quit("Invalid TRIS chunk version number",-1);
		}
		else
		if ( !strcmp( theHeader.ident, FM_FRAMESCHUNKNAME ) )
		{
			//	A FRAMES chunk
			frame_t*	pFrame;
			if ( theHeader.version != FM_FRAMESCHUNKVER )
				quit("Invalid FRAMES chunk version number",-1);
			pFrame = (frame_t*)(pIntoModel + FM_HEADERSIZE);
			gpFirstFrame = pFrame;
		}
		else
		if ( !strcmp( theHeader.ident, FM_GLCMDSCHUNKNAME ) )
		{
			//	A GLCMDS chunk
			signed char*	pCmd = pIntoModel + FM_HEADERSIZE;
			if ( theHeader.version != FM_GLCMDSCHUNKVER )
				quit("Invalid GLCMDS chunk version number",-1);
			gpGLCmds = (glcmd_t*)pCmd;
		}
		else
		if ( !strcmp( theHeader.ident, FM_MESHNODESCHUNKNAME ) )
		{
			//	A MESH NODES chunk
			meshnode_t*		pNode;
			if ( theHeader.version != FM_MESHNODESCHUNKVER )
				quit("Invalid MESH NODES chunk version number",-1);
			pNode = (meshnode_t*)(pIntoModel + FM_HEADERSIZE);
		}
		else
		if ( !strcmp( theHeader.ident, FM_SKELETONCHUNKNAME ) )
		{
			//	A SKELETON chunk
			skelhdr_t*	pSkelHeader;
			if ( theHeader.version != FM_SKELETONCHUNKVER )
				quit("Invalid SKELETON chunk version number",-1);

			pSkelHeader = (skelhdr_t*)(pIntoModel + FM_HEADERSIZE);
		}
		else
		if ( !strcmp( theHeader.ident, FM_REFERENCESCHUNKNAME ) )
		{
			//	A REFERENCES chunk
			refheader_t		refHeader;
			if ( theHeader.version != FM_REFERENCESCHUNKVER )
				quit("Invalid REFERENCES chunk version number",-1);
			memcpy( &refHeader, pIntoModel + FM_HEADERSIZE, sizeof(refHeader) );
			gpFirstRef = (refskel_t*)(pIntoModel + FM_HEADERSIZE + sizeof(refHeader));
		}
		else
			printf( "WARNING: Unrecognized chunk identifier: %s\n", theHeader.ident );

		//	Advance to next chunk
		pIntoModel += theHeader.size + FM_HEADERSIZE;
	}

	//	Make sure we processed the whole model
	if ( pIntoModel != gpModelData + gModelSize )
		quit("Invalid (too big) chunk size in model",-1);

	if ( !optL )
	{
		//	Write tweaked model
		printf("Output Model File: %s\n", optO);
		if ( !(fout = fopen( optO, "wb" )) )
			quit( "Unable to open output model file", errno );
		fwrite( gpNewModelData, gNewModelSize, 1L, fout );
		fclose( fout );
		printf("Wrote: %s, %d bytes\n", optO, gNewModelSize);

		//	Discard tweaked model
		free( gpNewModelData );
	}

	//	Discard model
	free( gpModelData );

	//	Lint-friendly exit
	return(0);
}

