//
//	Extract motion capture data from a Heretic II FlexModel file
//
//	By Chris Burke
//	serotonin@earthlink.net
//
//	Date			Who		Description
//	----------		---		-------------------------------------------------------
//	07/01/1999		CJB		Coded
//


//	System includes
#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<ctype.h>
#include	<fcntl.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<io.h>
#include	<math.h>

//	Local includes
#include	"CondComp.h"
#include	"FlexModel.h"
#include	"MatrixMath.h"

#define	__DUMMY_OUT__	":-)"
#define	__DUMMY_FRAME__	":-)"
#ifndef	FALSE
#define	FALSE				0
#define	TRUE				(!FALSE)
#endif

#define	x_(n)	((n)*gpOutputFrame->header.scale[0]+gpOutputFrame->header.translate[0])
#define	y_(n)	((n)*gpOutputFrame->header.scale[1]+gpOutputFrame->header.translate[1])
#define	z_(n)	((n)*gpOutputFrame->header.scale[2]+gpOutputFrame->header.translate[2])


//
//	Externs
//

extern char*
OBJ_WriteFile(
	char* pName, char* pComment, char* pFrameName, unsigned long meshnodemask,
	fmheader_t* pModelHeader, meshnode_t* pMeshNodes,
	glcmd_t* pGLCmds, frame_t* pFirstFrame
);

//	Globals
int				gModelFID;
FILE*			gModelFile;
char*			gModelFileName = "tris.fm";
char*			gProgramName;
long			gModelSize, gNewModelSize;
int				gNumFooFrames;
int				gNumMeshNodes;

meshnode_t*		gpMeshNodes;
glcmd_t*		gpGLCmds;
frame_t*		gpFirstFrame;
fmheader_t*		gpModelHeader;
refskel_t*		gpFirstRef;

int				gOutputFrameNumber;
frame_t*		gpOutputFrame = (void*)0;
refskel_t*		gpOutputFrameRefs = (void*)0;
skelpos_t*		gpOutputFrameSkels = (void*)0;

unsigned char*	gpModelData;
unsigned char*	gpNewModelData;
char			gScratchString[1024];
char			gOutFileName[1024];

byte			optP = FALSE;
char*			optF = __DUMMY_FRAME__;
long			optM = 0x0000FFFFL;
byte			optX = FALSE;

//	NOTE: These are generic
char*	gNodeNames[] =
{
	"Node0",	"Node1",	"Node2",	"Node3",
	"Node4",	"Node5",	"Node6",	"Node7",
	"Node8",	"Node9",	"Node10",	"Node11",
	"Node12",	"Node13",	"Node41",	"Node15"
};

//	NOTE: These are corvus / kiera specific
static char*	gRefNames[] =
{
	"left hand ref",	"right hand ref",	"right foot ref",	"left foot ref",
	"staff ref",		"blade ref",		"hellhead ref"
};

//	NOTE: These are corvus / kiera specific
static char*	gSkelNames[] =
{
	"lower back",		"upper back",		"neck"
};

//
//	Utils
//

void quit( char* pMsg, int errCode )
{
	if ( pMsg && *pMsg )
		printf( "\nFATAL: %s\n", pMsg );
	exit( errCode );
}

void usage( char* pMsg, int errCode )
{
	printf(
		"\n"
		"Program: %s\n"
		"by Chris Burke (serotonin@earthlink.net)\n"
		"Purpose: Export a single frame of a Heretic II FlexModel as Wavefront .OBJ\n"
		"\n"
		"Usage:\n"
		"%s [/f framename] [/m hexmask] [/p] [/x] file.fm\n",
		gProgramName, gProgramName
	);
	printf(
		"Options:\n"
		"    /f framename    Export the specified animation frame (else 1st frame)\n"
		"    /m hexmask      Export only the specified meshnodes (else all nodes)\n"
		"    /p              Export each meshnode to a separate file (else 1 file)\n"
		"    /x              Output extended reference and skeleton info (else just mesh)\n"
		"    file,fm         Name of the input file (else tris.fm)\n"
	);
	printf("Output file name is frame name + hexmask + .obj\n");
	quit( pMsg, errCode );
}

//
//	Calculate the angle between two vectors that share an origin.
//	The basic algorithm is to convert the vectors to unit vectors,
//	then use the facts that:
//		1)	the sum of the angles of a triangle is 180 degrees
//		2)	bisecting the angle between two unit vectors creates
//			two identical back-to-back right triangles each with
//			hypoteneuse 1 unit long and adjacent side equal in
//			length to 1/2 the distance between the points
//		3)  the cosine of an angle in a right triangle is the ratio
//			of the length of the opposite side to the length of
//			the hypoteneuse
//	In other words:
//		cos theta = ((dist between p1 and p2)/2) / 1.0
//		2 theta + phi = pi
//		phi = pi - 2 acos( ((dist between p1 and p2)/2) / 1.0 )
//
static float
AngleBetween( vec3_t* pivot, vec3_t* v1, vec3_t* v2 )
{
	vec3dbl_t	nv1, nv2;
	double	d, phi, theta, foo;

	nv1.c[0] = v1->c[0] - pivot->c[0];
	nv1.c[1] = v1->c[1] - pivot->c[1];
	nv1.c[2] = v1->c[2] - pivot->c[2];
	M3DUnitVector( &nv1, &nv1 );

	nv2.c[0] = v2->c[0] - pivot->c[0];
	nv2.c[1] = v2->c[1] - pivot->c[1];
	nv2.c[2] = v2->c[2] - pivot->c[2];
	M3DUnitVector( &nv2, &nv2 );

	d = M3DLength( &nv1, &nv2);
	theta = acos(d/2.);
	phi = PI - 2.*theta;
	foo = theta*(180./PI);
	foo = foo;
	return	(float)phi;
}


//
//	Main line
//
main ( int argc, char **argv )
{

	//	Parse arguments
	gProgramName = *argv++; --argc;
	printf(
		"Executing %s\n"
		"by Chris Burke (serotonin@earthlink.net)....\n",
		gProgramName
	);

	while (argc)
	{
		if ( (!strcmp(*argv,"/x")) || (!strcmp(*argv,"/X")) )
		{
			//	Output extended info
			optX = TRUE;
		}
		else
		if ( (!strcmp(*argv,"/p")) || (!strcmp(*argv,"/P")) )
		{
			//	Export in parts
			optP = TRUE;
		}
		else
		if ( (!strcmp(*argv,"/m")) || (!strcmp(*argv,"/M")) )
		{
			//	Meshnode mask
			if ( argc < 2 )
				usage( "/m requires hexadecimal argument", -1 );
			++argv; --argc;
			sscanf( *argv, "%lx", &optM );
		}
		else
		if ( (!strcmp(*argv,"/f")) || (!strcmp(*argv,"/F")) )
		{
			//	Frame name
			if ( argc < 2 )
				usage( "/f requires string argument", -1 );
			++argv; --argc;
			optF = *argv;
		}
		else
		{
			//	Input file
			char*	p;
			char*	q;
			char*	r;
			gModelFileName = *argv;
			//	Make sure input file is last argument
			if ( argc != 1 )
				usage( "unrecognized argument", -1 );
		}
		++argv; --argc;
	}

	//	Make sure the model file exists
	printf("Opening model file: %s\n", gModelFileName);
	if ( !( gModelFile = fopen( gModelFileName, "rb" ) ) )
		usage("Can't open model file", errno);

	//	Get size of model file and suck it into allocated memory
	fseek( gModelFile, 0L, SEEK_END );
	gModelSize = ftell( gModelFile );
	if ( !( gpModelData = malloc( gModelSize ) ) )
		quit("Can't allocate memory to read model", -1);
	fseek( gModelFile, 0L, SEEK_SET );
	fread( gpModelData, gModelSize, 1L, gModelFile );
	fclose( gModelFile );

	//	Perform consistency check on model
	{
		header_t	theHeader;
		fmheader_t	theModelHeader;
		byte*		pIntoModel = gpModelData;
		byte*		pIntoNewModel = gpNewModelData;

		while ( pIntoModel < gpModelData + gModelSize )
		{
			//	Read the header for the next chunk
			memcpy( theHeader.ident, pIntoModel, FM_MAXCHUNKIDENT );
			theHeader.ident[FM_MAXCHUNKIDENT-1] = '\0';
			theHeader.version	= FM_GETINT( pIntoModel + FM_MAXCHUNKIDENT + 0 );
			theHeader.size		= FM_GETINT( pIntoModel + FM_MAXCHUNKIDENT + 4 );
			printf(
				"%-32s version %d, length %d bytes.\n",
				theHeader.ident, theHeader.version, theHeader.size );
			if ( theHeader.size <= 0 )
				quit("Invalid (<=0) chunk size in model",-1);
			//	Process by chunk type
			if ( !strcmp( theHeader.ident, FM_HEADERCHUNKNAME ) )
			{
				//	A HEADER chunk
				if ( theHeader.version != FM_HEADERCHUNKVER )
					quit("Invalid HEADER chunk version number",-1);

				gpModelHeader = (fmheader_t*)(pIntoModel + FM_HEADERSIZE);

				theModelHeader.skinwidth	= FM_GETINT( pIntoModel + FM_HEADERSIZE + 0);	// in pixels
				theModelHeader.skinheight	= FM_GETINT( pIntoModel + FM_HEADERSIZE + 4);	// in pixels
				theModelHeader.framesize	= FM_GETINT( pIntoModel + FM_HEADERSIZE + 8);	// byte count of each frame
				theModelHeader.skincnt		= FM_GETINT( pIntoModel + FM_HEADERSIZE + 12);	// number of skins
				theModelHeader.vert3dcnt	= FM_GETINT( pIntoModel + FM_HEADERSIZE + 16);	// number of unique vertices
				theModelHeader.vertuvcnt	= FM_GETINT( pIntoModel + FM_HEADERSIZE + 20);	// greater than num_xyz for seams
				theModelHeader.tricnt		= FM_GETINT( pIntoModel + FM_HEADERSIZE + 24);	// number of triangles
				theModelHeader.glcmdcnt		= FM_GETINT( pIntoModel + FM_HEADERSIZE + 28);	// dwords in strip/fan command list
				theModelHeader.framecnt		= FM_GETINT( pIntoModel + FM_HEADERSIZE + 32);	// number of animation frames
				theModelHeader.meshnodecnt	= FM_GETINT( pIntoModel + FM_HEADERSIZE + 36);	// number of mesh nodes
				//	Dump all the data
				printf(
					"    Skins are %d x %d pixels.\n"
					"    %d skins identified.\n"
					"    %d animation frames, size %d each.\n"
					"    %d 3D vertices, %d texture verteces, and %d glcmds.\n"
					"    %d triangles, %d mesh nodes.\n",
					theModelHeader.skinwidth, theModelHeader.skinheight,
					theModelHeader.skincnt,
					theModelHeader.framecnt, theModelHeader.framesize,
					theModelHeader.vert3dcnt, theModelHeader.vertuvcnt,
					theModelHeader.glcmdcnt,
					theModelHeader.tricnt, theModelHeader.meshnodecnt
				);
			}
			else
			if ( !strcmp( theHeader.ident, FM_SKINCHUNKNAME ) )
			{
				//	A SKIN chunk
				int		i;
				char*	pSkinName = pIntoModel + FM_HEADERSIZE;
				skinpath_t theSkinPath;
				if ( theHeader.version != FM_SKINCHUNKVER )
					quit("Invalid SKIN chunk version number",-1);
#if	0
				//	Dump all the data
				for ( i = 0; i < theModelHeader.skincnt; i++ )
				{
					memcpy( theSkinPath.path, pSkinName, FM_MAXPATHLENGTH );
					theSkinPath.path[FM_MAXPATHLENGTH-1] = '\0';
					printf("    %s\n", theSkinPath.path);
					pSkinName += FM_SKINPATHSIZE;
				}
#endif
				pSkinName = pSkinName;
			}
			else
			if ( !strcmp( theHeader.ident, FM_STCOORDCHUNKNAME ) )
			{
				//	An ST COORD chunk
				int		i, u, v;
				byte*	pCoords = pIntoModel + FM_HEADERSIZE;
				if ( theHeader.version != FM_STCOORDCHUNKVER )
					quit("Invalid ST COORD chunk version number",-1);
#if	0
				//	Dump all the data
				for ( i = 0; i < theModelHeader.vertuvcnt; i++ )
				{
					u = FM_GETSHORT( pCoords + 0 );
					v = FM_GETSHORT( pCoords + 2 );
					printf("    %-4d: (%d,%d)\n", i, u, v);
					pCoords += FM_STCOORDUVSIZE;
				}
#endif
			}
			else
			if ( !strcmp( theHeader.ident, FM_TRISCHUNKNAME ) )
			{
				//	A TRIS chunk
				int			i;
				triangle_t	theTri;
				byte*		pCoords = pIntoModel + FM_HEADERSIZE;
				if ( theHeader.version != FM_TRISCHUNKVER )
					quit("Invalid TRIS chunk version number",-1);
#if	0
				//	Dump all the data
				for ( i = 0; i < theModelHeader.tricnt; i++ )
				{
					theTri.index_3d[0] = FM_GETSHORT( pCoords + 0 );
					theTri.index_3d[1] = FM_GETSHORT( pCoords + 2 );
					theTri.index_3d[2] = FM_GETSHORT( pCoords + 4 );
					theTri.index_uv[0] = FM_GETSHORT( pCoords + 6 );
					theTri.index_uv[1] = FM_GETSHORT( pCoords + 8 );
					theTri.index_uv[2] = FM_GETSHORT( pCoords + 10 );
					printf(
						"    %-4d: "
						"3D Verteces (%d-%d-%d),"
						"texture verteces (%d-%d-%d)\n",
						i,
						theTri.index_3d[0],theTri.index_3d[1],theTri.index_3d[2],
						theTri.index_uv[0],theTri.index_uv[1],theTri.index_uv[2]
					);
					pCoords += FM_TRISINFOSIZE;
				}
#endif
			}
			else
			if ( !strcmp( theHeader.ident, FM_FRAMESCHUNKNAME ) )
			{
				//	A FRAMES chunk
				frame_t*	pFrame;
				if ( theHeader.version != FM_FRAMESCHUNKVER )
					quit("Invalid FRAMES chunk version number",-1);
				pFrame = (frame_t*)(pIntoModel + FM_HEADERSIZE);
				gpFirstFrame = pFrame;
				//	Can't trust frame count in model header. Based on
				//	examination of model data, the count in Corvus and
				//	Kiera models is incorrect.
				gNumFooFrames = theHeader.size /
					( sizeof(framehdr_t) + theModelHeader.vert3dcnt * sizeof(vertex3d_t) );
				{
					int			i, nFrames, tally;
					char*		pName;
					frame_t		theFrame;
					char		numbuf[8], aniName[16];
					if ( theHeader.version != FM_FRAMESCHUNKVER )
						quit("Invalid FRAMES chunk version number",-1);
					//	Dump some of the data
					tally = 0;
					//	WARNING: WE READ FLOATS DIRECTLY INTO DATA
					//	STRUCTURES HERE...THIS IS NOT ENDIAN-CLEAN
					pFrame = (frame_t*)(pIntoModel + FM_HEADERSIZE);

					gpFirstFrame = pFrame;
					//	Make sure we have a frame name to export
					if ( !strcmp(optF,__DUMMY_FRAME__) )
						optF = pFrame->header.name;

					//	Find the frame number in question
					printf( "    searching for frame '%s'...\n", optF );
					pFrame = gpFirstFrame;
					for (gOutputFrameNumber = 0; gOutputFrameNumber<gpModelHeader->framecnt; gOutputFrameNumber++)
					{
						if ( !strcmp( optF, pFrame->header.name ) )
							break;
						pFrame = (frame_t*)( (char*)pFrame + gpModelHeader->framesize );
					}
					if ( gOutputFrameNumber >= gpModelHeader->framecnt )
						quit( "No such frame name!", -1 );

					//	Save pointer to frame for later
					gpOutputFrame = pFrame;
#if	0
					//	Dump some of the data
					*aniName = '\0';
					for ( i = 0; i < theModelHeader.framecnt; i++ )
					{
						//	If *aniName == '\0' this is the start of a
						//	new animation, otherwise it is part of the
						//	animation named in aniName
						theFrame = *pFrame;		//	WARNING: NOT ENDIAN CLEAN
						//	Isolate the name and any numeric suffix
						if ( !strcmp(theFrame.header.name,__EXPORT_FRAME_NAME__) )
							i = i;
						pName = theFrame.header.name;
						pName += strlen(pName) - 1;
						while ( isdigit(*pName) ) --pName;
						strcpy( numbuf, pName+1 );
						*(pName+1) = '\0';
						//	Does this name match previous frame?
						if ( !strcmp( theFrame.header.name, aniName ) )
						{
							//	Frame is part of current animation; revise frame count
							nFrames = atoi( numbuf );	//	So far, this is the # of frames
							if (!nFrames) nFrames = 1;
						}
						else
						{
							//	Starting a new animation; dump previous
							//	name and frame count
							if ( *aniName != '\0' )
							{
								printf(
										"    Animation: %s (%d frames)\n",
										aniName, nFrames
									  );
								tally += nFrames;
							}
							strcpy( aniName, theFrame.header.name );
							nFrames = 1;
						}
						pFrame = (frame_t*)((byte*)pFrame + theModelHeader.framesize);
					}
					if ( *aniName != '\0' )
					{
						printf(
								"    Animation: %s (%d frames)\n",
								aniName, nFrames
							  );
						tally += nFrames;
					}
#if	__CHECK_FRAMECNT__
					if ( tally != theModelHeader.framecnt )
						quit( "Frame count doesn't match header", -1 );
#endif
#endif
				}
			}
			else
			if ( !strcmp( theHeader.ident, FM_GLCMDSCHUNKNAME ) )
			{
				//	A GLCMDS chunk
				int				numTris;			//	Number of triangles processed
				int				numNodes;			//	Number of distinct sets of GLCMDS processed
				int				numLocalTris;		//	Number of triangles in this part.
				int				numGLCmds;			//	Number of GLCmd 32-bit elements processed
				int				numLocalGLCmds;		//	Number of GLCmd 32-bit elements in this node
				int				theCmd, i;
				signed char*	pCmd = pIntoModel + FM_HEADERSIZE;
				glvert_t*		pVert;
				glvert_t		theVert;
//				byte			foo[40];

				if ( theHeader.version != FM_GLCMDSCHUNKVER )
					quit("Invalid GLCMDS chunk version number",-1);

				gpGLCmds = (glcmd_t*)pCmd;

#if	0
				//	Dump some of the data
				numTris = 0;
				numNodes = 0;
				numLocalTris = 0;
				numGLCmds = 0;
				numLocalGLCmds = 0;

				while ( numGLCmds < theModelHeader.glcmdcnt )
				{
					theCmd = FM_GETINT( pCmd );
					pCmd += 4;
					++numLocalGLCmds;
					if ( !theCmd )
					{
						//	End of this "node"
						if ( numLocalTris )
							printf( "    Node %3d: draws %3d triangles using %3d GLCmd elements\n", numNodes, numLocalTris, numLocalGLCmds );
						++numNodes;
						numTris += numLocalTris;
						numGLCmds += numLocalGLCmds;
						numLocalTris = 0;
						numLocalGLCmds = 0;
						//	NOTE: KEEP LOOKING UNTIL WE'VE DONE ALL THE GLCMD DATA INDICATED IN HEADER
					}
					else
					if ( theCmd > 0 )
					{
						//	Process TRISTRIP command
//						printf("-");
						pVert = (glvert_t*)pCmd;
						for ( i=0; i<theCmd; i++ )
						{
							//	WARNING: WE READ FLOATS DIRECTLY INTO DATA
							//	STRUCTURES HERE...THIS IS NOT ENDIAN-CLEAN
							theVert = *pVert++;
						}
						numLocalTris += (theCmd - 2);
						numLocalGLCmds += theCmd * 3;
						pCmd = (byte*)pVert;
					}
					else
					{
						//	Process TRIFAN command
//						printf("V");
						theCmd = -theCmd;
						pVert = (glvert_t*)pCmd;
						for ( i=0; i<theCmd; i++ )
						{
							//	WARNING: WE READ FLOATS DIRECTLY INTO DATA
							//	STRUCTURES HERE...THIS IS NOT ENDIAN-CLEAN
							theVert = *pVert++;
						}
						numLocalTris += (theCmd - 2);
						numLocalGLCmds += theCmd * 3;
						pCmd = (byte*)pVert;
					}
				}
				if ( numLocalTris )
					quit( "Unterminated GLCMDS list", -1);
#endif
			}
			else
			if ( !strcmp( theHeader.ident, FM_MESHNODESCHUNKNAME ) )
			{
				//	A MESH NODES chunk
				int		i;
				meshnode_t	theNode, *pNode;
				if ( theHeader.version != FM_MESHNODESCHUNKVER )
					quit("Invalid MESH NODES chunk version number",-1);
				pNode = (meshnode_t*)(pIntoModel + FM_HEADERSIZE);

				gpMeshNodes = pNode;

#if	0
				//	Dump some of the data
				for ( i=0; i<theModelHeader.meshnodecnt; i++ )
				{
					theNode = *pNode++;
					printf( "    GLCmds for '%s' start at index %4d, occupying %3d elements.\n",
						gNodeNames[i], theNode.glcmdstart, theNode.glcmdcnt );
					i = i;
				}
#endif
			}
			else
			if ( !strcmp( theHeader.ident, FM_SKELETONCHUNKNAME ) )
			{
				//	A SKELETON chunk
				skelhdr_t*	pSkelHeader;
				int			nJoints, nSkelVerts;
				int*		pJVerts;
				int			i, j;

				if ( theHeader.version != FM_SKELETONCHUNKVER )
					quit("Invalid SKELETON chunk version number",-1);

				//	Find the skels info for the output frame
				nSkelVerts = 0;
				pSkelHeader = (skelhdr_t*)(pIntoModel + FM_HEADERSIZE);
				nJoints = pSkelHeader->numjoints;
				pJVerts = pSkelHeader->jointverts + nJoints;
				for ( i=0; i<nJoints; i++ )
				{
					nSkelVerts += pSkelHeader->jointverts[i];
				}
				pJVerts += nSkelVerts;		//	Point past last vertex index
				if ( *pJVerts )
				{
					//	Save pointer to output frame skels info for later
					gpOutputFrameSkels = (void*)(pJVerts+1);
					gpOutputFrameSkels += gOutputFrameNumber;
				}

#if	0
				//	Dump some of the data
				pSkelHeader = (skelhdr_t*)(pIntoModel + FM_HEADERSIZE);
				nJoints = pSkelHeader->numjoints;
				printf( "    Skeleton type %d with %d clusters\n", pSkelHeader->skeltype, nJoints );
				pJVerts = pSkelHeader->jointverts + nJoints;
				for ( i=0; i<nJoints; i++ )
				{
					nSkelVerts = pSkelHeader->jointverts[i];
					printf( "    Cluster %d contains %d new verteces\n", i, nSkelVerts );
					for ( j=0; j<nSkelVerts; j++ )
						printf( "%3d ", *pJVerts++ );
					printf("\n");
				}
#endif
				i = i;
			}
			else
			if ( !strcmp( theHeader.ident, FM_REFERENCESCHUNKNAME ) )
			{
				//	A REFERENCES chunk
				refheader_t		refHeader;
				if ( theHeader.version != FM_REFERENCESCHUNKVER )
					quit("Invalid REFERENCES chunk version number",-1);
				memcpy( &refHeader, pIntoModel + FM_HEADERSIZE, sizeof(refHeader) );
				gpFirstRef = (refskel_t*)(pIntoModel + FM_HEADERSIZE + sizeof(refHeader));

				//	Save pointer to output frame refs info for later
				gpOutputFrameRefs = gpFirstRef + gOutputFrameNumber;

#if	0
				//	Dump some of the data
				{
					int				i, j, k;
					float			v;
					refheader_t		theHeader;
					refskel_t		theRefs[1400];
					memcpy( &theHeader, pIntoModel + FM_HEADERSIZE, sizeof(theHeader) );
					memcpy( theRefs, pIntoModel + FM_HEADERSIZE + sizeof(theHeader), sizeof(theRefs) );
					for ( i=0; i<1400; i++ )
					{
						for ( j=0; j<H2CORVUS_NUMREFPOINTS; j++ )
						{
							for ( k=0; k<3; k++ )
							{
								v = theRefs[i].refpoints[j].origin.c[k];
								if ( (v<0.0) || (v>255.0) )
									printf("  frame %d, reference %d, coordinate %d out of range (%8.4f)\n", i, j, k, v );
							}
						}
					}
					i = 0;
				}
#endif
			}
			else
				printf( "WARNING: Unrecognized chunk identifier: %s\n", theHeader.ident );

			//	Advance to next chunk
			pIntoModel += theHeader.size + FM_HEADERSIZE;
		}
		if ( pIntoModel != gpModelData + gModelSize )
			quit("Invalid (too big) chunk size in model",-1);
	}

	{
		//	Write out the model to .OBJ file(s)
		char*	pFileName;
		printf("\n");
		if ( optP )
		{
			//	Export parts
			int		i;
			long	m;
			char*	p;
			for ( i = 0; i < gpModelHeader->meshnodecnt; i++ )
			{
				m = 1L << i;
				if ( m & optM )
				{
					pFileName =
						OBJ_WriteFile(
							(void*)0, (void*)0, optF, m,
							gpModelHeader, gpMeshNodes, gpGLCmds, gpFirstFrame
						);
					if ( pFileName )
						printf("Wrote: %s\n", pFileName);
				}
			}
		}
		else
		{
			//	Export whole thing
			pFileName =
				OBJ_WriteFile(
					(void*)0, (void*)0, optF, optM,
					gpModelHeader, gpMeshNodes, gpGLCmds, gpFirstFrame
				);
			if ( pFileName )
				printf("Wrote: %s\n", pFileName);
		}
	}

	//	Write extended info if requested
	if ( optX )
	{
		int			i;
		double		ang, l1, l2;
		vec3_t		piv, v1, v2;
		vec3dbl_t	pivv, vv1, vv2;

		//	First the reference data
		if ( gpOutputFrameRefs )
			for ( i=0; i<H2CORVUS_NUMREFPOINTS; i++ )
			{
				piv.c[0] = x_(gpOutputFrameRefs->refpoints[i].origin.c[0]);
				piv.c[1] = y_(gpOutputFrameRefs->refpoints[i].origin.c[1]);
				piv.c[2] = z_(gpOutputFrameRefs->refpoints[i].origin.c[2]), 
				v1.c[0]  = x_(gpOutputFrameRefs->refpoints[i].direction.c[0]);
				v1.c[1]  = y_(gpOutputFrameRefs->refpoints[i].direction.c[1]);
				v1.c[2]  = z_(gpOutputFrameRefs->refpoints[i].direction.c[2]), 
				v2.c[0]  = x_(gpOutputFrameRefs->refpoints[i].up.c[0]);
				v2.c[1]  = y_(gpOutputFrameRefs->refpoints[i].up.c[1]);
				v2.c[2]  = z_(gpOutputFrameRefs->refpoints[i].up.c[2]);
				printf(
					"%-14s: org=(%4.0f,%4.0f,%4.0f) dir=(%4.0f,%4.0f,%4.0f) up=(%4.0f,%4.0f,%4.0f)\n",
					gRefNames[i],
					piv.c[0], piv.c[1], piv.c[2],
					v1.c[0], v1.c[1], v1.c[2],
					v2.c[0], v2.c[1], v2.c[2]
				);
				pivv.c[0] = piv.c[0];
				pivv.c[1] = piv.c[1];
				pivv.c[2] = piv.c[2];
				vv1.c[0] = v1.c[0];
				vv1.c[1] = v1.c[1];
				vv1.c[2] = v1.c[2];
				vv2.c[0] = v2.c[0];
				vv2.c[1] = v2.c[1];
				vv2.c[2] = v2.c[2];
				l1 = M3DLength( &pivv, &vv1 );
				l2 = M3DLength( &pivv, &vv2 );
				ang = AngleBetween( &piv, &v1, &v2 );
				printf( "    l1 = %6.2f; l2 = %6.2f; angle is %6.4f radians.\n", l1, l2, ang );
			}
		//	Then the skeletal data
		if ( gpOutputFrameSkels )
			for ( i=0; i<H2CORVUS_NUMJOINTS; i++ )
			{
				piv.c[0] = x_(gpOutputFrameSkels->pos[i].origin.c[0]);
				piv.c[1] = y_(gpOutputFrameSkels->pos[i].origin.c[1]);
				piv.c[2] = z_(gpOutputFrameSkels->pos[i].origin.c[2]), 
				v1.c[0]  = x_(gpOutputFrameSkels->pos[i].direction.c[0]);
				v1.c[1]  = y_(gpOutputFrameSkels->pos[i].direction.c[1]);
				v1.c[2]  = z_(gpOutputFrameSkels->pos[i].direction.c[2]), 
				v2.c[0]  = x_(gpOutputFrameSkels->pos[i].up.c[0]);
				v2.c[1]  = y_(gpOutputFrameSkels->pos[i].up.c[1]);
				v2.c[2]  = z_(gpOutputFrameSkels->pos[i].up.c[2]);
				printf(
					"%-14s: org=(%4.0f,%4.0f,%4.0f) dir=(%4.0f,%4.0f,%4.0f) up=(%4.0f,%4.0f,%4.0f)\n",
					gSkelNames[i],
					piv.c[0], piv.c[1], piv.c[2],
					v1.c[0], v1.c[1], v1.c[2],
					v2.c[0], v2.c[1], v2.c[2]
				);
				pivv.c[0] = piv.c[0];
				pivv.c[1] = piv.c[1];
				pivv.c[2] = piv.c[2];
				vv1.c[0] = v1.c[0];
				vv1.c[1] = v1.c[1];
				vv1.c[2] = v1.c[2];
				vv2.c[0] = v2.c[0];
				vv2.c[1] = v2.c[1];
				vv2.c[2] = v2.c[2];
				l1 = M3DLength( &pivv, &vv1 );
				l2 = M3DLength( &pivv, &vv2 );
				ang = AngleBetween( &piv, &v1, &v2 );
				printf( "    l1 = %6.2f; l2 = %6.2f; angle is %6.4f radians.\n", l1, l2, ang );
			}
	}

	//	Discard model
	free( gpModelData );

	//	Lint-friendly exit
	return(0);
}


