//
//	OBJ File Output Routines
//
//	By Chris Burke
//	serotonin@earthlink.net
//
//	Date			Who		Description
//	----------		---		-------------------------------------------------------
//	07/01/1999		CJB		Coded
//

#include	<stdio.h>
#include	<string.h>
#include	<stdlib.h>

#include	"CondComp.h"
#include	"FlexModel.h"

extern char*	gNodeNames[];

static char	gfnbuf[256];

//
//	Output a face ("f") entity of an OBJ file, with given
//	3dspace and texture space vertex indeces.
//
//	NOTE: Verteces in an OBJ file are based at 1, unlike
//	verteces in a FlexModel file which are based at 0.
//
int
OBJ_WriteFace( FILE* pOut, trividx_t verts, trividx_t texture, int reverse )
{
	if ( reverse )
	{
		fprintf(
			pOut, "f %d/%d %d/%d %d/%d\n",
			verts.v[0], texture.v[0], 
			verts.v[1], texture.v[1], 
			verts.v[2], texture.v[2]
		);
	}
	else
	{
		fprintf(
			pOut, "f %d/%d %d/%d %d/%d\n",
			verts.v[2], texture.v[2], 
			verts.v[1], texture.v[1], 
			verts.v[0], texture.v[0]
		);
	}
	return	errno;
}

//
//	Output a 3D vertex ("v") entity of an OBJ file, with given coordinates
//
int
OBJ_Write3DVert( FILE* pOut, vec3_t vert )
{
	//	Tried writing vert[0], vert[1], vert[2], but this causes the Y and Z
	//	coordinates to be swapped when the model is imported into TrueSpace. This
	//	version pre-swaps Y and Z coordinates.

#if		__TRUESPACE2_OBJS__
	//	Swap Y and Z coordinates to get TrueSpace2 to read model correctly
	fprintf( pOut, "v  %8.4f %8.4f %8.4f\n", vert.c[0], vert.c[2], vert.c[1]  );	//	Swap Y and Z
#elif	__POSER4_OBJS__
	//	Rotate order of X, Y, Z to get Poser4 to read model correctly
	fprintf( pOut, "v  %8.4f %8.4f %8.4f\n", vert.c[1], vert.c[2], vert.c[0]  );	//	Swap for Poser 4
#else
	//	Output X, Y, Z in standard order
	fprintf( pOut, "v  %8.4f %8.4f %8.4f\n", vert.c[0], vert.c[1], vert.c[2]  );	//	No Swap
#endif

	return	errno;
}

//
//	Output a texture vertex ("vt") entity of an OBJ file, with given coordinates
//
int
OBJ_WriteTexVert( FILE* pOut, ftexpoint_t vert )
{
#if		__TRUESPACE2_OBJS__
	//	TrueSpace 2: texture coordinates in range 0 .. 255, Flip X
	fprintf( pOut, "vt %8.4f %8.4f\n", 255. - (int)(vert.c[0] * 256.), vert.c[1] * 256.  );
#elif	__POSER4_OBJS__
	//	Poser 4: Texture coordinates in range 0.0 .. 1.0
	fprintf( pOut, "vt %8.4f %8.4f\n", vert.c[0], (0.99999 - vert.c[1])  );
#else
	//	Default / Heretic II: texture coordinates in range 0.0 ... 1.0
	fprintf( pOut, "vt %8.4f %8.4f\n", vert.c[0], vert.c[1]  );
#endif
	return	errno;
}



//
//	Write .OBJ file for specified frame and mesh node mask
//

char*
OBJ_WriteFile(
	char* pName, char* pComment, char* pFrameName, unsigned long meshnodemask,
	fmheader_t* pModelHeader, meshnode_t* pMeshNodes,
	glcmd_t* pGLCmds, frame_t* pFirstFrame
)
{
	FILE*		pOut;
	frame_t*	pFrame;
	int			node, vert, coord, i;
	int			numGLs, glsDone, vertsDone, theCmd;
	vec3_t		theVert;
	trividx_t	tri3d, tritex;
	ftexpoint_t	tp;

	glvert_t*	pVert;
	glcmd_t*	pGLs;

//	pFrame = (frame_t*)( (byte*)pFirstFrame + framenumber*(pModelHeader->framesize) );
	pFrame = pFirstFrame;
	for (i = 0; i<pModelHeader->framecnt; i++)
	{
		if ( !strcmp( pFrameName, pFrame->header.name ) )
			break;
		pFrame = (frame_t*)( (char*)pFrame + pModelHeader->framesize );
	}
	if ( i >= pModelHeader->framecnt )
		return (void*)0;

	if ( !pName )
	{
		pName = pFrame->header.name;
		strcpy(gfnbuf,pName);
		sprintf( gfnbuf+strlen(gfnbuf), "%04X\0", (meshnodemask & 0x0000FFFFL) );
		strcat(gfnbuf,".obj");
		pName = gfnbuf;
	}

	if ( !(pOut = fopen( pName, "w" )) )
		return	(void*)0;

	//
	//	Write object name
	//
	if ( !pComment )
		pComment = pName;
	fprintf( pOut, "o %s\n", pComment );

	//
	//	Specify a (bogus) texture map library
	//
//	fprintf( pOut, "maplib %s\n", pFrame->header.name );

	//
	//	Write the list of 3d space verteces
	//
	//	NOTE: All verteces are written, even though triangles
	//	will be written only for selected meshnodes. The 3d space
	//	vertex indeces written by this code are indeces into the
	//	frame.
	//
	for ( vert=0; vert<pModelHeader->vert3dcnt; vert++ )
	{
		//	Copy vertex data to theVert
		for ( coord=0; coord<3; coord++ )
			theVert.c[coord] =
				pFrame->verts[vert].v[coord] * pFrame->header.scale[coord] +
					pFrame->header.translate[coord];
		//	Emit the vertex
		fprintf(
			pOut, "# vertex %d (tris.fm vertex %d, unscaled [%d,%d,%d])\n",
			vert+1, vert,
			pFrame->verts[vert].v[0], pFrame->verts[vert].v[1],
			pFrame->verts[vert].v[2]
		);
		OBJ_Write3DVert( pOut, theVert );
	}

	//
	//	Write the list of texture space verteces
	//
	//	NOTE: Verteces are written only for selected meshnodes. The
	//	texture space verteces written by this code are indexed
	//	differently than in the original model: they are numbered
	//	using vertex index into the GLCMDS list.
	//
	//	NOTE: We write the texture coordinates by scaling the values
	//	in the GLCMDS chunk, rather than using the triangle - vertex
	//	mapping in the TRIS and ST COORD chunks.
	//
	for ( node=0; node<pModelHeader->meshnodecnt; node++ )
	{
		if ( meshnodemask & (1L<<node) )
		{
			//	Export texture verteces for this node
			pGLs = pGLCmds + pMeshNodes[node].glcmdstart;
			numGLs = pMeshNodes[node].glcmdcnt;
			glsDone = 0;
			while ( glsDone < numGLs )
			{
				//	Process this list of GLCMDS
				theCmd = pGLs->theCmd;
				++glsDone;
				if ( theCmd < 0 )
				{
					//	TRIFAN
					theCmd = -theCmd;
					glsDone += theCmd * 3;
					pVert = (glvert_t*)(pGLs+1);
					while ( theCmd-- )
					{
						//	note that these are all in range 0.0 .. 1,0
						tp.c[0] = pVert->u;
						tp.c[1] = pVert->v;
						fprintf( pOut, "# for 3D vertex %d (tris.fm 3D vertex %d)\n", pVert->index_3d+1, pVert->index_3d );
						OBJ_WriteTexVert( pOut, tp );
						++pVert;
					}
					pGLs= (glcmd_t*)pVert;
				}
				else
				if ( theCmd > 0 )
				{
					glsDone += theCmd * 3;
					pVert = (glvert_t*)(pGLs+1);
					while ( theCmd-- )
					{
						//	note that these are all in range 0.0 .. 1,0
						tp.c[0] = pVert->u;
						tp.c[1] = pVert->v;
						fprintf( pOut, "# for 3D vertex %d (tris.fm 3D vertex %d)\n", pVert->index_3d+1, pVert->index_3d );
						OBJ_WriteTexVert( pOut, tp );
						++pVert;
					}
					pGLs= (glcmd_t*)pVert;
				}
				else
				{
					//	End of list
					glsDone = glsDone;
				}
			}
		}
	}

	//
	//	Write the list of triangles, giving each meshnode
	//	its own named group
	//
	fprintf( pOut, "g %s\n", pFrame->header.name );
//	//	Use textures on this group
//	fprintf( pOut, "usemtl x\n" );
	for ( node=0; node<pModelHeader->meshnodecnt; node++ )
	{
		if ( meshnodemask & (1L<<node) )
		{
#if	1
			//	Create a group for this node
			fprintf( pOut, "g %s %s\n", gNodeNames[node], pFrame->header.name );
			//	Use textures on this group
			fprintf( pOut, "usemtl x\n" );
#endif
			//	Remind the gentle reader...
			fprintf( pOut, "# NOTE: Vertex indeces are based at 1 for .OBJ\n" );
			fprintf( pOut, "# NOTE: This differs from zero-based tris.fm\n" );
			//	Export this node
			pGLs = pGLCmds + pMeshNodes[node].glcmdstart;
			numGLs = pMeshNodes[node].glcmdcnt;
			glsDone = 0;
			vertsDone = 0;
			while ( glsDone < numGLs )
			{
				//	Process this list of GLCMDS
				theCmd = pGLs->theCmd;
				++glsDone;
				if ( theCmd < 0 )
				{
					//	TRIFAN
					theCmd = -theCmd;
					glsDone += theCmd * 3;
					pVert = (glvert_t*)(pGLs+1);
					//	Load up the first triangle edge into v[0] and v[2]
					tri3d.v[0]  = pVert[0].index_3d + 1;
					tri3d.v[2]  = pVert[1].index_3d + 1;
					tritex.v[0] = vertsDone + 0 + 1;
					tritex.v[2] = vertsDone + 1 + 1;
					pVert += 2;
					vertsDone += 2;
					theCmd -= 2;
					if ( theCmd) do
					{
						//	Copy 3rd vertex of previous triangle back to 2nd
						//	to preserve normal orientation, then get new vertex
						tri3d.v[1]  = tri3d.v[2];
						tri3d.v[2]  = pVert[0].index_3d + 1;
						tritex.v[1] = tritex.v[2];
						tritex.v[2] = vertsDone + 0 + 1;
						//	Emit face
						OBJ_WriteFace( pOut, tri3d, tritex, 0 );
						//	Advance pointers
						++pVert;
						++vertsDone;
						--theCmd;
					} while (theCmd );
					pGLs= (glcmd_t*)pVert;
				}
				else
				if ( theCmd > 0 )
				{
					int		phase = 0;
					//	TRISTRIP
					glsDone += theCmd * 3;
					pVert = (glvert_t*)(pGLs+1);
					//	Load up the first triangle
					for ( vert=0; vert<3; vert++ )
					{
						tri3d.v[vert]  = pVert[vert].index_3d + 1;
						tritex.v[vert] = vertsDone + vert + 1;
					}
					pVert += 3;
					vertsDone += 3;
					theCmd -= 3;
					//	Emit 1st face
					OBJ_WriteFace( pOut, tri3d, tritex, phase );
					phase = !phase;
					if ( theCmd) do
					{
						//	Copy 2nd and 3rd vertex from previous triangle
						//	to 1st and 2nd vertex of new triangle, then get
						//	new vertex
						for ( vert=1; vert<3; vert++ )
						{
							tri3d.v[vert-1]  = tri3d.v[vert];
							tritex.v[vert-1] = tritex.v[vert];
						}
						tri3d.v[2]  = pVert[0].index_3d + 1;
						tritex.v[2] = vertsDone + 1;
						//	Emit face
						OBJ_WriteFace( pOut, tri3d, tritex, phase );
						//	Advance pointers
						++pVert;
						++vertsDone;
						--theCmd;
						phase = !phase;
					} while (theCmd );
					pGLs= (glcmd_t*)pVert;
				}
				else
				{
					//	End of list
					glsDone = glsDone;
				}
			}
		}
	}

	fclose( pOut );

	//	Cheap error report
	return	pName;

}