//
//	Matrix-based 3D math (C version).
//	Based on C3dMatrix by Nigel Thomson
//
//	By Chris Burke
//	serotonin@earthlink.net
//
//	Date			Who		Description
//	----------		---		-------------------------------------------------------
//	07/09/1999		CJB		Coded
//

#include	<math.h>

#include	"CondComp.h"
#include	"FlexModel.h"
#include	"MatrixMath.h"

static mat3d_t
gIdentityMatrix =
{
	1.0,	0.0,	0.0,	0.0,
	0.0,	1.0,	0.0,	0.0,
	0.0,	0.0,	1.0,	0.0,
	0.0,	0.0,	0.0,	1.0
};

//
//	Make identity matrix
//
void
M3DIdent( mat3d_t* pM )
{
	*pM = gIdentityMatrix;
}

//
//	Copy matrix at pS to pD
//
void
M3DAssign( mat3d_t* pD, mat3d_t* pS )
{
	*pD = *pS;
}

//
//	Initialize the members of patrix pM
//
void
M3DInit(
	mat3d_t* pM,
	double v00, double v01, double v02, double v03,
	double v10, double v11, double v12, double v13,
	double v20, double v21, double v22, double v23,
	double v30, double v31, double v32, double v33
)
{
    pM->m00=v00; pM->m01=v01; pM->m02=v02; pM->m03=v03; 
    pM->m10=v10; pM->m11=v11; pM->m12=v12; pM->m13=v13; 
    pM->m20=v20; pM->m21=v21; pM->m22=v22; pM->m23=v23; 
    pM->m30=v30; pM->m31=v31; pM->m32=v32; pM->m33=v33; 
}

//
//	Add matrix at p1 and matrix at p2,
//	store result in matrix at pM
//
void
M3DAdd( mat3d_t* pM, mat3d_t* p1, mat3d_t* p2 )
{
	M3DInit(
		pM,
		p1->m00 + p2->m00,
		p1->m01 + p2->m01,
		p1->m02 + p2->m02,
		p1->m03 + p2->m03,
		p1->m10 + p2->m10,
		p1->m11 + p2->m11,
		p1->m12 + p2->m12,
		p1->m13 + p2->m13,
		p1->m20 + p2->m20,
		p1->m21 + p2->m21,
		p1->m22 + p2->m22,
		p1->m23 + p2->m23,
		p1->m30 + p2->m30,
		p1->m31 + p2->m31,
		p1->m32 + p2->m32,
		p1->m33 + p2->m33
	);
}

//
//	Add matrix at pS to matrix at pD
//
void
M3DAddAssign( mat3d_t* pD, mat3d_t* pS )
{
	M3DInit(
		pD,
		pD->m00 + pS->m00,
		pD->m01 + pS->m01,
		pD->m02 + pS->m02,
		pD->m03 + pS->m03,
		pD->m10 + pS->m10,
		pD->m11 + pS->m11,
		pD->m12 + pS->m12,
		pD->m13 + pS->m13,
		pD->m20 + pS->m20,
		pD->m21 + pS->m21,
		pD->m22 + pS->m22,
		pD->m23 + pS->m23,
		pD->m30 + pS->m30,
		pD->m31 + pS->m31,
		pD->m32 + pS->m32,
		pD->m33 + pS->m33
	);
}

//
//	Multiply matrix p1 and matrix p2 together,
//	storing result at pM
//
void
M3DMul( mat3d_t* pM, mat3d_t* p1, mat3d_t* p2 )
{
	M3DInit(
		pM,
		p1->m00*p2->m00 + p1->m01*p2->m10 + p1->m02*p2->m20 + p1->m03*p2->m30,
		p1->m00*p2->m01 + p1->m01*p2->m11 + p1->m02*p2->m21 + p1->m03*p2->m31,
		p1->m00*p2->m02 + p1->m01*p2->m12 + p1->m02*p2->m22 + p1->m03*p2->m32,
		p1->m00*p2->m03 + p1->m01*p2->m13 + p1->m02*p2->m23 + p1->m03*p2->m33,
		p1->m10*p2->m00 + p1->m11*p2->m10 + p1->m12*p2->m20 + p1->m13*p2->m30,
		p1->m10*p2->m01 + p1->m11*p2->m11 + p1->m12*p2->m21 + p1->m13*p2->m31,
		p1->m10*p2->m02 + p1->m11*p2->m12 + p1->m12*p2->m22 + p1->m13*p2->m32,
		p1->m10*p2->m03 + p1->m11*p2->m13 + p1->m12*p2->m23 + p1->m13*p2->m33,
		p1->m20*p2->m00 + p1->m21*p2->m10 + p1->m22*p2->m20 + p1->m23*p2->m30,
		p1->m20*p2->m01 + p1->m21*p2->m11 + p1->m22*p2->m21 + p1->m23*p2->m31,
		p1->m20*p2->m02 + p1->m21*p2->m12 + p1->m22*p2->m22 + p1->m23*p2->m32,
		p1->m20*p2->m03 + p1->m21*p2->m13 + p1->m22*p2->m23 + p1->m23*p2->m33,
		p1->m30*p2->m00 + p1->m31*p2->m10 + p1->m32*p2->m20 + p1->m33*p2->m30,
		p1->m30*p2->m01 + p1->m31*p2->m11 + p1->m32*p2->m21 + p1->m33*p2->m31,
		p1->m30*p2->m02 + p1->m31*p2->m12 + p1->m32*p2->m22 + p1->m33*p2->m32,
		p1->m30*p2->m03 + p1->m31*p2->m13 + p1->m32*p2->m23 + p1->m33*p2->m33
	);
}

//
//	Multiply matrix p1 and matrix p2 together,
//	storing result at p1
//
void
M3DMulAssign( mat3d_t* p1, mat3d_t* p2 )
{
	M3DInit(
		p1,
		p1->m00*p2->m00 + p1->m01*p2->m10 + p1->m02*p2->m20 + p1->m03*p2->m30,
		p1->m00*p2->m01 + p1->m01*p2->m11 + p1->m02*p2->m21 + p1->m03*p2->m31,
		p1->m00*p2->m02 + p1->m01*p2->m12 + p1->m02*p2->m22 + p1->m03*p2->m32,
		p1->m00*p2->m03 + p1->m01*p2->m13 + p1->m02*p2->m23 + p1->m03*p2->m33,
		p1->m10*p2->m00 + p1->m11*p2->m10 + p1->m12*p2->m20 + p1->m13*p2->m30,
		p1->m10*p2->m01 + p1->m11*p2->m11 + p1->m12*p2->m21 + p1->m13*p2->m31,
		p1->m10*p2->m02 + p1->m11*p2->m12 + p1->m12*p2->m22 + p1->m13*p2->m32,
		p1->m10*p2->m03 + p1->m11*p2->m13 + p1->m12*p2->m23 + p1->m13*p2->m33,
		p1->m20*p2->m00 + p1->m21*p2->m10 + p1->m22*p2->m20 + p1->m23*p2->m30,
		p1->m20*p2->m01 + p1->m21*p2->m11 + p1->m22*p2->m21 + p1->m23*p2->m31,
		p1->m20*p2->m02 + p1->m21*p2->m12 + p1->m22*p2->m22 + p1->m23*p2->m32,
		p1->m20*p2->m03 + p1->m21*p2->m13 + p1->m22*p2->m23 + p1->m23*p2->m33,
		p1->m30*p2->m00 + p1->m31*p2->m10 + p1->m32*p2->m20 + p1->m33*p2->m30,
		p1->m30*p2->m01 + p1->m31*p2->m11 + p1->m32*p2->m21 + p1->m33*p2->m31,
		p1->m30*p2->m02 + p1->m31*p2->m12 + p1->m32*p2->m22 + p1->m33*p2->m32,
		p1->m30*p2->m03 + p1->m31*p2->m13 + p1->m32*p2->m23 + p1->m33*p2->m33
	);
}

//
//	Multiply a point (in a vec3dbl_t with coordinates ordered X, Y, Z)
//	by a 3x3 matrix. Essentially this rotates the point around axes.
//
void
M3DPointMul( vec3dbl_t* pD, mat3d_t* pM, vec3dbl_t* pS )
{
	vec3dbl_t	p;
	p = *pS;		//	So that source and destination can be same
	pD->c[0] = pM->m00 * p.c[0] + pM->m01 * p.c[1] + pM->m02 * p.c[2];
	pD->c[1] = pM->m10 * p.c[0] + pM->m11 * p.c[1] + pM->m12 * p.c[2];
	pD->c[2] = pM->m20 * p.c[0] + pM->m21 * p.c[1] + pM->m22 * p.c[2];
}

//
//	Multiply a point (in a vec4dbl_t with coordinates ordered X, Y, Z, 1)
//	by a 4x4 matrix. Essentially this rotates the point around axes
//	and translates. The multiplication is configured so that translation
//	information is stored in the right column of pS.
//
static void
M3DPoint4Mul( vec4dbl_t* pD, mat3d_t* pM, vec4dbl_t* pS )
{
	vec4dbl_t	p;
	p = *pS;		//	So that source and destination can be same
	pD->c[0] = pM->m00 * p.c[0] + pM->m01 * p.c[1] + pM->m02 * p.c[2] + pM->m03 * p.c[3];
	pD->c[1] = pM->m10 * p.c[0] + pM->m11 * p.c[1] + pM->m12 * p.c[2] + pM->m13 * p.c[3];
	pD->c[2] = pM->m20 * p.c[0] + pM->m21 * p.c[1] + pM->m22 * p.c[2] + pM->m23 * p.c[3];
	pD->c[3] = pM->m30 * p.c[0] + pM->m31 * p.c[1] + pM->m32 * p.c[2] + pM->m33 * p.c[3];
}

//
//	Assign a vec3dbl_t point to a vec4dbl_t point
//
static void
M3DSetPoint4( vec4dbl_t* pD, vec3dbl_t* pS )
{
	pD->c[0] = pS->c[0];
	pD->c[1] = pS->c[1];
	pD->c[2] = pS->c[2];
	pD->c[3] = 1.;
}

//
//	Assign a vec4dbl_t point to a vec3dbl_t point
//
static void
M3DSetPoint( vec3dbl_t* pD, vec4dbl_t* pS )
{
	pD->c[0] = pS->c[0];
	pD->c[1] = pS->c[1];
	pD->c[2] = pS->c[2];
}

//
//	Rotate a point (in a vec3dbl_t with coordinates ordered X, Y, Z)
//	n radians around each axis (specified in a vec3dbl_t with
//	rotations ordered rx, ry, rz).
//
void
M3DRotate( vec3dbl_t* pD, vec3dbl_t* pR )
{
	double	sinx, cosx, siny, cosy, sinz, cosz;
	mat3d_t	mx, my, mz;
	vec4dbl_t	p;

	sinx = sin(pR->c[0]);
	cosx = cos(pR->c[0]);
	siny = sin(pR->c[1]);
	cosy = cos(pR->c[1]);
	sinz = sin(pR->c[2]);
	cosz = cos(pR->c[2]);

	M3DSetPoint4( &p, pD );

	M3DInit(
		&mx,
		1., 0.,		0.,		0.,
		0.,	cosx,	sinx,	0.,
		0., -sinx,	cosx,	0.,
		0.,	0.,		0.,		1.
	);

    M3DInit(
		&my,
		cosy,	0.,	-siny,	0.,
		0.,		1.,	0.,		0.,
		siny,	0.,	cosy,	0.,
		0.,		0.,	0.,		1.
	);

    M3DInit(
		&mz,
		cosz,	sinz,	0.,	0.,
		-sinz,	cosz,	0.,	0.,
		0.,		0.,		1.,	0.,
		0.,		0.,		0.,	1.
	);

	//	NOTE: I THINK ORDER MIGHT BE VERY IMPORTANT HERE.
	//	WHEN DOING MULTIPLE ROTATIONS IN OTHER PARTS OF
	//	THE CODE, THE ROTATIONS ARE APPLIED IN THIS ORDER:
	//	X-AXIS FIRST, THEN Z-AXIS, THEN Y-AXIS.
//	M3DMulAssign( &my, &mz );
//	M3DMulAssign( &mx, &my );
//	M3DPoint4Mul( &p,  &mx, &p );
	M3DMulAssign( &mz, &my );
	M3DMulAssign( &mx, &mz );
	M3DPoint4Mul( &p,  &mx, &p );
	M3DSetPoint(  pD,  &p  );
}

//
//	Rotate, using degrees instead of radians
//
void
M3DRotateDeg( vec3dbl_t* pD, vec3dbl_t* pR )
{
	vec3dbl_t	r;
	r.c[0] = pR->c[0] * kD2R;
	r.c[1] = pR->c[1] * kD2R;
	r.c[2] = pR->c[2] * kD2R;
	M3DRotate( pD, &r );
}

//
//	Translate a point (in a vec3dbl_t with coordinates ordered X, Y, Z)
//	by a given vector.
//
void
M3DTranslate( vec3dbl_t* pD, vec3dbl_t* pT )
{
	mat3d_t	mt;
	vec4dbl_t	p;

	M3DSetPoint4( &p, pD );

	M3DInit(
		&mt,
		1.,			0.,			0.,			pT->c[0],
		0.,			1.,			0.,			pT->c[1],
		0.,			0.,			1.,			pT->c[2],
		0.,			0.,			0.,			1.
	);

	M3DPoint4Mul( &p,  &mt, &p );
	M3DSetPoint(  pD,  &p  );
}

//
//	Untranslate a point (in a vec3dbl_t with coordinates ordered X, Y, Z)
//	by a given vector.
//
void
M3DUnTranslate( vec3dbl_t* pD, vec3dbl_t* pT )
{
	mat3d_t	mt;
	vec4dbl_t	p;

	M3DSetPoint4( &p, pD );

	M3DInit(
		&mt,
		1.,			0.,			0.,			-pT->c[0],
		0.,			1.,			0.,			-pT->c[1],
		0.,			0.,			1.,			-pT->c[2],
		0.,			0.,			0.,			1.
	);

	M3DPoint4Mul( &p,  &mt, &p );
	M3DSetPoint(  pD,  &p  );
}

//
//	Scale a point (in a vec3dbl_t with coordinates ordered X, Y, Z)
//	by a given vector.
//
void
M3DScale( vec3dbl_t* pD, vec3dbl_t* pN )
{
	mat3d_t	ms;
	vec4dbl_t	p;

	M3DSetPoint4( &p, pD );

	M3DInit(
		&ms,
		pN->c[0],	0.,			0.,			0.,
		0.,			pN->c[1],	0.,			0.,
		0.,			0.,			pN->c[2],	0.,
		0.,			0.,			0.,			1.
	);

	M3DPoint4Mul( &p,  &ms, &p );
	M3DSetPoint(  pD,  &p  );
}

//
//	Return the length of a line segment bounded by two
//	points in 3D space, each in a vec3dbl_t
//
double
M3DLength( vec3dbl_t* p1, vec3dbl_t*p2 )
{
	return	sqrt( M3DSqrLength( p1, p2 ) );
}

//
//	Return the squared length of a line segment bounded
//	by two points in 3D space, each in a vec3dbl_t. This
//	is faster than M3DLength when only relative distance
//	is needed.
//
double
M3DSqrLength( vec3dbl_t* p1, vec3dbl_t*p2 )
{
	double	a, b, c;
	a =	(p1->c[0] - p2->c[0]);
	b = (p1->c[1] - p2->c[1]);
	c = (p1->c[2] - p2->c[2]);
	return	(a*a)+(b*b)+(c*c);
}

//
//	Return the magnitude of a vector in a vec3dbl_t
//
static vec3dbl_t gOrigin = { 0., 0., 0. };
double
M3DMagnitude( vec3dbl_t* pV )
{
	return M3DLength( pV, &gOrigin );
}

//
//	Convert a given vector to a unit vector
//
void
M3DUnitVector( vec3dbl_t* pD, vec3dbl_t* pS )
{
	double	vm;
	vm = M3DMagnitude( pS );
	pD->c[0] = pS->c[0] / vm;
	pD->c[1] = pS->c[1] / vm;
	pD->c[2] = pS->c[2] / vm;
}

//
//	Calculate the cross product UxV of two vectors ordered X, Y, Z,
//	storing the result in a given vector.
//
void
M3DCrossProduct( vec3dbl_t* pD, vec3dbl_t* pU, vec3dbl_t* pV )
{
	vec3dbl_t	vR;
	vR.c[0] = (pV->c[2]*pU->c[1]) - (pV->c[1]*pU->c[2]);
	vR.c[1] = (pV->c[0]*pU->c[2]) - (pV->c[2]*pU->c[0]);
	vR.c[2] = (pV->c[1]*pU->c[0]) - (pV->c[0]*pU->c[1]);
	*pD = vR;
}
