//
//	Part of H2MoCap
//
//	This code examines all animation frames of a Heretic 2 player model,
//	to find groups of points that keep more-or-less constant distance
//	from each other. This information is used to deduce the "skeleton"
//	of the player model.
//
//	NOTE: It turns out this isn't really an effective technique due to
//	the way frames are compressed and other factors that cause the actual
//	coordinates of lots of points to move around from frame to frame.
//	Instead, a complex skeleton was assumed, and joint locations were
//	also assumed; these were used to analyze the model in terms of bone
//	lengths and a standard deviation was calculated over all frames. If
//	the standard deviation of calculated bone length was very low, the
//	joint location was assumed to be accurate.
//
//	By Chris Burke
//	serotonin@earthlink.net
//
//	Date			Who		Description
//	----------		---		-------------------------------------------------------
//	07/01/1999		CJB		Coded
//

#include	<stdio.h>
#include	<stdlib.h>
#include	<math.h>

#include	"CondComp.h"
#include	"FlexModel.h"
#include	"h2BitMaps.h"

#define		kMaxSkeletonMembers	100
#define		kIgnoreDistance		65536.0
#define		kMaxBuddyRatio		1.1			//	Determined by trial and error
#define		kMinBellQ			22			//	Determined by trial and error

h2bitmap_t*	gpUsedVerteces;
h2bitmap_t*	gpSkelMembers[ kMaxSkeletonMembers ];
int			gNumSkelMembers;

extern void	quit( char*, int);

void
FindSkelGroups( fmheader_t* pModelHeader, int nFrames, frame_t* pFirstFrame )
{
	//	For each vertex in the model, we want to calculate its distance from
	//	all other verteces during every animation frame. Then we want to look
	//	for sets of verteces that keep constant distance within some delta.
	//	Each such set is one member of the model skeleton. We keep a set of
	//	bitmaps to indicate which verteces are part of which members.

	int			i;
	int			lastvertexchecked = -1;
	int			partnervertex;
	double		mindist, maxdist, dist;
	double*		pDistances;
	frame_t*	pFrame;
	int			ok;

	gNumSkelMembers = -1;

	//	Create a bitmap indicating which verteces have been assigned to skeleton
	if ( !(gpUsedVerteces = NewBitMap( pModelHeader->vert3dcnt )) )
		quit( "(FindSkelGroups) Can't allocate global vertex tracking bitmap", -1 );

	//	Create table for saving distance between points for every frame
	if ( !(pDistances = malloc( nFrames*sizeof(float) )) )
		quit( "(FindSkelGroups) Can't allocate distance table", -1 );

	//	Find a vertex that hasn't been assigned, and check it out
	while ( (lastvertexchecked = FindClrBitMap( gpUsedVerteces, lastvertexchecked+1, 1 )) != -1 )
	{
		//	lastvertexchecked is now the index of a vertex that belongs to nobody.
		//	We need only check verteces after this one in the model, but for
		//	simplicity we keep the bitmap for each skeleton member the same size

		printf( "New vertex partner group: " );

		//	Create a bitmap for tracking members of this skeleton
		if ( ++gNumSkelMembers >= kMaxSkeletonMembers )
			quit( "(FindSkelGroups) Too many elements in skeleton", -1 );
		if ( !(gpSkelMembers[gNumSkelMembers] = NewBitMap( pModelHeader->vert3dcnt )) )
			quit( "(FindSkelGroups) Can't allocate member vertex tracking bitmap", -1 );

		//	To save memory, we pick a point to compare to, check the distance in all
		//	frames, then if the distance stays relatively constant we mark the point
		//	used and add it to the bitmap for this skeletal element, NOTE that we
		//	check against ourself as well, for simplicity. This code won't work right
		//	if we don't test against ourself.

		partnervertex = lastvertexchecked;
		while ( (partnervertex = FindClrBitMap( gpUsedVerteces, partnervertex, 1 )) != -1 )
		{

			//	Calculate distance between lastvertexchecked and partnervertex in every frame
			//	Point at the header of the first frame
			//	WARNING: WE READ FLOATS DIRECTLY INTO DATA
			//	STRUCTURES HERE...THIS IS NOT ENDIAN-CLEAN
			pFrame = pFirstFrame;
			for ( i = 0; i < nFrames; i++ )
			{
				//	Calculate distance using frame pointer.
				//	WARNING: NOT ENDIAN-CLEAN
				dist =
					sqrt(
						pow(
							pFrame->verts[lastvertexchecked].v[0] * pFrame->header.scale[0] -
							pFrame->verts[partnervertex].v[0] * pFrame->header.scale[0],
							2.0
						) +
						pow(
							pFrame->verts[lastvertexchecked].v[1] * pFrame->header.scale[1] -
							pFrame->verts[partnervertex].v[1] * pFrame->header.scale[1],
							2.0
						) +
						pow(
							pFrame->verts[lastvertexchecked].v[2] * pFrame->header.scale[2] -
							pFrame->verts[partnervertex].v[2] * pFrame->header.scale[2],
							2.0
						)
					);
				pDistances[i] = dist;
//				printf( "%6.3f ", dist );
//				if ( (lastvertexchecked >= 47) && (lastvertexchecked <= 54) )
//					if ( (partnervertex >= 47) && (partnervertex <= 54) )
//						printf( "distance [%d]-[%d]: %6.3f\n", lastvertexchecked, partnervertex, dist );

				//	Advance to next frame
				pFrame = (frame_t*)((byte*)pFrame + pModelHeader->framesize);
			}

			//	See if distance stayed within allowable delta
			ok = 0;
			mindist = kIgnoreDistance;
			maxdist = 0.0;
			for ( i=0; i<nFrames; i++ )
			{
				dist = pDistances[i];
				if ( dist < mindist ) mindist = dist;
				if ( dist > maxdist ) maxdist = dist;
			}
			if ( !mindist ) mindist = 1.E-20;
			dist = maxdist / mindist;
			if ( dist <= kMaxBuddyRatio )
				ok = 1;
			else
			{
				//	2nd stage scan to eliminate anomalous frames
				double	mean, stddev;
				//	Calculate mean
				mean = 0;
				for ( i=0; i<nFrames; i++ )
					mean += pDistances[i];
				mean /= nFrames;
				//	Calculate standard deviation
				stddev = 0;
				for ( i=0; i<nFrames; i++ )
					stddev += pow( pDistances[i] - mean, 2 );
				stddev = sqrt(stddev / nFrames);
				if ( (mean / stddev) >= kMinBellQ )
					ok = 1;
			}
//			if ( (lastvertexchecked >= 47) && (lastvertexchecked <= 54) )
//				if ( (partnervertex >= 47) && (partnervertex <= 54) )
//					dist = dist;
			if ( ok )
			{
				//	We have a winner! Mark it used in global list, and
				//	as a member of the current skeleton element
				SetBitMap( gpUsedVerteces, partnervertex, 1 );
				SetBitMap( gpSkelMembers[gNumSkelMembers], partnervertex, 1 );
				printf( "%4d", partnervertex );
			}
			++partnervertex;
		}
		//	If there are no more partner verteces, this skeletal element is done!
		printf("\n");
	}

}
