//
//	DXF File Output Routines
//
//	By Chris Burke
//	serotonin@earthlink.net
//
//	Date			Who		Description
//	----------		---		-------------------------------------------------------
//	07/01/1999		CJB		Coded
//

#include	<stdio.h>
#include	<string.h>
#include	<stdlib.h>

#include	"CondComp.h"
#include	"FlexModel.h"

static char	gfnbuf[256];

//
//	Output a 3DFACE entity of a DXF file, on a given layer
//	with a given index color
//
int
DXF_Write3DFace( FILE* pOut, vec3_t* pVerts, int layer, int indexcolor, int reverse )
{
	int		vert, coord;
	fprintf( pOut, "0\n3DFACE\n8\n%d\n62\n%d\n", layer, indexcolor );
	if ( !reverse )
	{
		for ( vert=0; vert<3; vert++ )
			for ( coord=0; coord<3; coord++ )
				fprintf( pOut, "%d\n%f\n", 10*(coord+1)+vert, (pVerts[vert]).c[coord] );
		//	Last vertex is written twice for triangles
		for ( coord=0; coord<3; coord++ )
			fprintf( pOut, "%d\n%f\n", 10*(coord+1)+3, (pVerts[2]).c[coord] );
	}
	else
	{
		for ( vert=2; vert>=0; vert-- )
			for ( coord=0; coord<3; coord++ )
				fprintf( pOut, "%d\n%f\n", 10*(coord+1)+vert, (pVerts[vert]).c[coord] );
		//	Last vertex is written twice for triangles
		for ( coord=0; coord<3; coord++ )
			fprintf( pOut, "%d\n%f\n", 10*(coord+1)+3, (pVerts[0]).c[coord] );
	}

	return	errno;
}


//
//	Write .DXF file for specified frame and mesh node mask.
//	This according to Paul Rourke's document,
//	"Minimum Requirements for Creating a DXF File of a 3D Model"
//

char*
DXF_WriteFile(
	char* pName, char* pComment, char* pFrameName, unsigned long meshnodemask,
	fmheader_t* pModelHeader, meshnode_t* pMeshNodes,
	glcmd_t* pGLCmds, frame_t* pFirstFrame
)
{
	FILE*		pOut;
	frame_t*	pFrame;
	int			i;

//	pFrame = (frame_t*)( (byte*)pFirstFrame + framenumber*(pModelHeader->framesize) );
	pFrame = pFirstFrame;
	for (i = 0; i<pModelHeader->framecnt; i++)
	{
		if ( !strcmp( pFrameName, pFrame->header.name ) )
			break;
		pFrame = (frame_t*)( (char*)pFrame + pModelHeader->framesize );
	}
	if ( i >= pModelHeader->framecnt )
		return (void*)0;

	if ( !pName )
	{
		pName = pFrame->header.name;
		strcpy(gfnbuf,pName);
		sprintf( gfnbuf+strlen(gfnbuf), "%04X\0", (meshnodemask & 0x0000FFFFL) );
		strcat(gfnbuf,".dxf");
		pName = gfnbuf;
	}

	if ( !(pOut = fopen( pName, "w" )) )
		return	(void*)0;

	//
	//	Write initial comment
	//
	if ( !pComment )
		pComment = pName;
	fprintf( pOut, "999\n%s\n", pComment );
	
	
	//
	//	Write canned header section
	//
	fprintf(
		pOut,
		"0\nSECTION\n2\nHEADER\n9\n$ACADVER\n1\nAC1006\n"
		"9\n$INSBASE\n10\n0.0\n20\n0.0\n30\n0.0\n"
		"9\n$EXTMIN\n10\n-500.0\n20\n-500.0\n30\n-500.0\n"
		"9\n$EXTMAX\n10\n500.0\n20\n500.0\n30\n500.0\n"
		"0\nENDSEC\n"
	);

	//
	//	Write canned tables section
	//
	fprintf(
		pOut,
		"0\nSECTION\n2\nTABLES\n"
		"0\nTABLE\n2\nLTYPE\n70\n1\n0\nLTYPE\n2\nCONTINUOUS\n"
		"70\n64\n3\nSolid line\n72\n65\n73\n0\n40\n0.000000\n"
		"0\nENDTAB\n"
		"0\nTABLE\n2\nLAYER\n70\n6\n0\nLAYER\n2\n1\n70\n"
		"64\n62\n7\n6\nCONTINUOUS\n"
		"0\nLAYER\n2\n2\n70\n64\n62\n7\n6\nCONTINUOUS\n"
		"0\nENDTAB\n"
		"0\nTABLE\n2\nSTYLE\n70\n0\n0\nENDTAB\n"
		"0\nENDSEC\n"
	);
	
	//
	//	Write canned blocks section
	//
	fprintf(
		pOut,
		"0\nSECTION\n2\nBLOCKS\n0\nENDSEC\n"
	);

	//
	//	Write entities section - header
	//
	fprintf(
		pOut,
		"0\nSECTION\n2\nENTITIES\n"
	);

	//
	//	3D entity data goes here.
	//
	//	We write it as a series of "facets" (triangles) on
	//	layer 1 with index color 3, using the GLCMDS of the
	//	model and the vertex coordinates of the selected
	//	animation frame.
	//
	//	We take care to always specify the verteces of the
	//	triangles in the same order (match the order of the 1st
	//	triangle in the GLCMD, assuming that it is properly
	//	oriented).
	//
	{
		int			node, vert, coord;
		int			numGLs, glsDone, theCmd;
		vec3_t		triverts[3];
		glvert_t*	pVert;
		glcmd_t*	pGLs;

		for ( node=0; node<H2NUM_MESHNODES; node++ )
		{
			if ( meshnodemask & (1L<<node) )
			{
				//	Export this node
				pGLs = pGLCmds + pMeshNodes[node].glcmdstart;
				numGLs = pMeshNodes[node].glcmdcnt;
				glsDone = 0;
				while ( glsDone < numGLs )
				{
					//	Process this list of GLCMDS
					theCmd = pGLs->theCmd;
					++glsDone;
					if ( theCmd < 0 )
					{
						//	TRIFAN
						theCmd = -theCmd;
						glsDone += theCmd * 3;
						pVert = (glvert_t*)(pGLs+1);
						//	Load up the first triangle edge into v[0] and v[2]
						for ( coord=0; coord<3; coord++ )
							(triverts[0]).c[coord] =
								pFrame->verts[pVert[0].index_3d].v[coord] *
									pFrame->header.scale[coord] +
										pFrame->header.translate[coord];
						for ( coord=0; coord<3; coord++ )
							(triverts[2]).c[coord] =
								pFrame->verts[pVert[1].index_3d].v[coord] *
									pFrame->header.scale[coord] +
										pFrame->header.translate[coord];
						pVert += 2;
						theCmd -= 2;
						if ( theCmd) do
						{
							//	Copy 3rd vertex of previous triangle back to 2nd
							//	to preserve normal orientation
							for ( coord=0; coord<3; coord++ )
								(triverts[1]).c[coord] = (triverts[2]).c[coord];
							//	Load 3rd vertex of new triangle
							for ( coord=0; coord<3; coord++ )
								(triverts[2]).c[coord] =
									pFrame->verts[pVert[0].index_3d].v[coord] *
										pFrame->header.scale[coord] +
											pFrame->header.translate[coord];
							//	Emit face
							DXF_Write3DFace( pOut, triverts, 1, 3, 0 );
							//	Advance pointers
							++pVert;
							--theCmd;
						} while (theCmd );
						pGLs= (glcmd_t*)pVert;
					}
					else
					if ( theCmd > 0 )
					{
						int		phase = 0;
						//	TRISTRIP
						glsDone += theCmd * 3;
						pVert = (glvert_t*)(pGLs+1);
						//	Load up the first triangle
						for ( vert=0; vert<3; vert++ )
							for ( coord=0; coord<3; coord++ )
								(triverts[vert]).c[coord] =
									pFrame->verts[pVert[vert].index_3d].v[coord] *
										pFrame->header.scale[coord] +
											pFrame->header.translate[coord];
						pVert += 3;
						theCmd -= 3;
						//	Emit 1st face
						DXF_Write3DFace( pOut, triverts, 1, 3, phase );
						phase = !phase;
						if ( theCmd) do
						{
							//	Copy 2nd and 3rd vertex from previous triangle
							//	to 1st and 2nd vertex of new triangle
							for ( vert=1; vert <3; vert++ )
								for ( coord=0; coord<3; coord++ )
									(triverts[vert-1]).c[coord] = (triverts[vert]).c[coord];
							//	Load 3rd vertex of new triangle
							for ( coord=0; coord<3; coord++ )
								(triverts[2]).c[coord] =
									pFrame->verts[pVert[0].index_3d].v[coord] *
										pFrame->header.scale[coord] +
											pFrame->header.translate[coord];
							//	Emit face, reversed on odd phase
							DXF_Write3DFace( pOut, triverts, 1, 3, phase );
							//	Advance pointers
							++pVert;
							--theCmd;
							phase = !phase;
						} while (theCmd );
						pGLs= (glcmd_t*)pVert;
					}
					else
					{
						//	End of list
						glsDone = glsDone;
					}
				}
			}
		}
	}

	//
	//	Write entities section - footer
	//
	fprintf(
		pOut,
		"0\nENDSEC\n"
	);

	//
	//	Write EOF indication
	//
	fprintf(
		pOut,
		"0\nEOF\n"
	);

	fclose( pOut );

	//	Cheap error report
	return	pName;

}