//
//	Generic bitmap manipulation routines.
//	The format is chosen to match the bitmaps used
//	in Heretic 2 player models
//
//	By Chris Burke
//	serotonin@earthlink.net
//
//	Date			Who		Description
//	----------		---		-------------------------------------------------------
//	07/01/1999		CJB		Coded
//

#include	<stdlib.h>
#include	<string.h>

#include	"h2BitMaps.h"

//	Create and zero a bitmap of a given size
h2bitmap_t*
NewBitMap( int numbits )
{
	h2bitmap_t*	rv;
	if ( rv = malloc(sizeof(h2bitmap_t)) )
	{
		rv->mapbytes = (numbits+7) >> 3;
		if ( rv->bitdata = malloc( rv->mapbytes ) )
		{
			rv->numbits = numbits;
			ClrBitMap( rv, 0, numbits );
		}
		else
		{
			free( rv );
			rv = (h2bitmap_t*)0;
		}
	}
	return	rv;
}

//	Dispose of a bitmap
h2bitmap_t*
FreeBitMap( h2bitmap_t* pMap )
{
	free( pMap->bitdata );
	free( pMap );
	return	(h2bitmap_t*) 0;
}

//	Get size of a bitmap
int
SizeBitMap( h2bitmap_t* pMap )
{
	return	pMap->numbits;
}

//	Set bit(s) in a bitmap
void
SetBitMap( h2bitmap_t* pMap, int startbit, int numbits )
{
	int		startbyte, bitinbyte;
	int		endbyte, bitend;
	int		midbytes;
	//	Clip inputs
	if ( !numbits )
		return;
	if ( numbits < 0 )
	{
		startbit -= numbits;
		numbits = -numbits;
	}
	if ( startbit > pMap->numbits )
		return;
	if ( startbit < 0 )
		startbit = 0;
	if ( startbit + numbits > pMap->numbits )
		numbits = pMap->numbits - startbit;
	//	Process request in 3 parts: header, body, footer
	startbyte = startbit >> 3;
	bitinbyte = startbit & 0x07;
	endbyte   = (startbit + numbits - 1) >> 3;
	bitend    = (startbit + numbits - 1) & 0x07;
	midbytes  = (endbyte - startbyte) - 1;
	if ( midbytes == -1 )
	{
		//	Simple case - all bits in same byte
		pMap->bitdata[startbyte] |= ( (0x00FF >> (7-bitend)) & (0x00FF << bitinbyte) );
	}
	else
	{
		//	Request spans multiple bytes. Process in 3 parts: high bits of startbyte,
		//	fully-set bytes, and low bits of endbyte
		pMap->bitdata[startbyte] |= (0x00FF << bitinbyte);
		if ( midbytes )
			memset( pMap->bitdata + startbyte + 1, 0xff, midbytes );
		pMap->bitdata[endbyte] |= 0x00FF >> (7-bitend);
	}
}

//	Clear bit(s) in a bitmap
void
ClrBitMap( h2bitmap_t* pMap, int startbit, int numbits )
{
	int		startbyte, bitinbyte;
	int		endbyte, bitend;
	int		midbytes;
	//	Clip inputs
	if ( !numbits )
		return;
	if ( numbits < 0 )
	{
		startbit -= numbits;
		numbits = -numbits;
	}
	if ( startbit > pMap->numbits )
		return;
	if ( startbit < 0 )
		startbit = 0;
	if ( startbit + numbits > pMap->numbits )
		numbits = pMap->numbits - startbit;
	//	Process request in 3 parts: header, body, footer
	startbyte = startbit >> 3;
	bitinbyte = startbit & 0x07;
	endbyte   = (startbit + numbits - 1) >> 3;
	bitend    = (startbit + numbits - 1) & 0x07;
	midbytes  = (endbyte - startbyte) - 1;
	if ( midbytes == -1 )
	{
		//	Simple case - all bits in same byte
		pMap->bitdata[startbyte] &= ~( ( (0x00FF >> (7-bitend)) & (0x00FF << bitinbyte) ) );
	}
	else
	{
		//	Request spans multiple bytes. Process in 3 parts: high bits of startbyte,
		//	fully-set bytes, and low bits of endbyte
		pMap->bitdata[startbyte] &= ~(0x00FF << bitinbyte);
		if ( midbytes )
			memset( pMap->bitdata + startbyte + 1, 0x00, midbytes );
		pMap->bitdata[endbyte] &= ~(0x00FF >> (7-bitend));
	}
}

//	Test a given bit in a bitmap
int
TstBitMap( h2bitmap_t* pMap, int startbit )
{
	int		startbyte, bitmask;
	//	Clip inputs
	if ( (startbit < 0) || (startbit > pMap->numbits) )
		return	1;
	startbyte = startbit >> 3;
	bitmask   = 0x01 << (startbit & 0x07);
	return	( (pMap->bitdata[startbyte] & bitmask) != 0 );
}

//	Find a given number of free bits in a bitmap
//	This is a very slow implementation
int
FindClrBitMap( h2bitmap_t* pMap, int startbit, int numbits )
{
	int		begins;
	int		foundbits = 0;
	while ( startbit < pMap->numbits )
	{
		//	Search for start of free area
		while ( (startbit < pMap->numbits) && TstBitMap(pMap, startbit) )
			++startbit;
		foundbits = 1;
		begins = startbit;
		++startbit;
		while ( (startbit < pMap->numbits) && (foundbits < numbits) && !TstBitMap(pMap, startbit) )
		{
			++startbit;
			++foundbits;
		}
		if ( foundbits == numbits )
			return	begins;
	}
	return	-1;
}

//	Find a given number of set bits in a bitmap
//	This is a very slow implementation
int
FindSetBitMap( h2bitmap_t* pMap, int startbit, int numbits )
{
	int		begins;
	int		foundbits = 0;
	while ( startbit < pMap->numbits )
	{
		//	Search for start of set area
		while ( (startbit < pMap->numbits) && !TstBitMap(pMap, startbit) )
			++startbit;
		foundbits = 1;
		begins = startbit;
		++startbit;
		while ( (startbit < pMap->numbits) && (foundbits < numbits) && TstBitMap(pMap, startbit) )
		{
			++startbit;
			++foundbits;
		}
		if ( foundbits == numbits )
			return	begins;
	}
	return	-1;
}
