//
//	Overwrite the data points in a given frame that correspond to
//	the Corvus staff, replacing the staff with a sword.
//

#include	<stdio.h>
#include	<memory.h>
#include	<string.h>

#include	"CondComp.h"
#include	"FlexModel.h"
#include	"bones.h"
#include	"MatrixMath.h"

#if		__MODEL_IS_CORVUS__
#define		kFirstStaffVertex	205
#define		kFXAdjust			5
#elif	__MODEL_IS_KIERA__
#define		kFirstStaffVertex	234
#define		kFXAdjust			4
#endif

#define		kNumStaffVerteces	14
#define		kNumStaffGLCmds		124

#if		__DO_LIGHT_SABRE__
#define		kTipVertex			13		//	Vertex at tip of light sabre
#else
#define		kTipVertex			8		//	Vertex at tip of sword
#endif

#if	__NEW_SWORD_REFS__
extern void
GetFrameKeyPoints(
	frame_t* pFrame, refskel_t* pRef,
	vec3dbl_t* pJoints, vec3dbl_t* pYReference, vec3dbl_t* pZReference
);
#endif

#if	__DO_LIGHT_SABRE__
//
//	Lightsaber data
//
static vec3dbl_t
gSwordData[kNumStaffVerteces] =
{
	{	0.5,	0.0,	0.0		},	//	0:	pre-base (to use up vertex)
	{	0.0,	0.0,	0.6		},	//	1;	Base
	{	0.0,	0.5,	0.3		},	//	2:
	{	0.0,	0.5,	-0.3	},	//	3:
	{	0.0,	0.0,	-0.6	},	//	4:
	{	0.0,	-0.5,	-0.3	},	//	5:
	{	0.0,	-0.5,	0.3		},	//	6:
	{	33.0,	-0.3,	0.2		},	//	7:	Base
	{	33.0,	-0.3,	-0.2	},	//	8:
	{	33.0,	0.0,	-0.4	},	//	9:
	{	33.0,	0.3,	-0.2	},	//	10:
	{	33.0,	0.3,	0.2		},	//	11:
	{	33.0,	0.0,	0.4		},	//	12:
	{	33.5,	0.0,	0.0		}	//	13:	tip
};

//	The original staff GLCmds occupy 124 32-bit words.
//	A command (1st number on each line) occupies 1 word.
//	A vertex (remaining numbers on each line) occupies
//	3 words. To make the replacement the same length,
//	we repeat verteces in some of the fans below.
static int
gSwordGLCmds[] =
{
#if	__ZERO_LIGHT_SABRE__
	//	Here to make the light sabre invisible
#else
	10,	7,	8,	13,	9,	10,	13,	11,	12,	13,	7,
	14,	1,	12,	2,	11,	3,	10,	4,	9,	5,	8,	6,	7,	1,	12,
	-4,	11,	12,	13,	10,
#endif
	0
};

//	These are quick-and dirty, using an unused
//	part of the skins for the lightsabre coloring.
static texpoint_t
gSwordTexture[kNumStaffVerteces] =
{
#if	__MODEL_IS_CORVUS__
	//	WARNING: DUMMY DATA FOR CORVUS RIGHT NOW.
	//	Map Corvus light saber to an unused part of the skin
	{	235,	147	},
	{	230,	142	},
	{	240,	142	},
	{	230,	142	},
	{	240,	142	},
	{	230,	142	},
	{	240,	142	},
	{	240,	162	},
	{	230,	162	},
	{	240,	162	},
	{	230,	162	},
	{	240,	162	},
	{	230,	162	},
	{	235,	157	}
#elif	__MODEL_IS_KIERA__
	//	Map Kiera light saber to an unused part of the skin
	{	85,	45	},
	{	80,	40	},
	{	90,	40	},
	{	80,	40	},
	{	90,	40	},
	{	80,	40	},
	{	90,	40	},
	{	90,	60	},
	{	80,	60	},
	{	90,	60	},
	{	80,	60	},
	{	90,	60	},
	{	80,	60	},
	{	85,	55	}
#endif
};

#else
//
//	Sword data
//
static vec3dbl_t
gSwordData[kNumStaffVerteces] =
{
	{	1.0,	-4.0,	0.0		},	//	 0 or 205 (right lower)
	{	2.0,	-4.0,	1.0		},	//	 1 or 206 (right upper)
	{	2.0,	-4.0,	-1.0	},	//	 2 or 207 (right upper)
	{	1.0,	4.0,	0.0		},	//	 3 or 208 (left lower)
	{	2.0,	4.0,	-1.0	},	//	 4 or 209 (left upper)
	{	1.5,	5.0,	0.0		},	//	 5 or 210 (left hilt)
	{	30.0,	1.5,	0.0		},	//	 6 or 211 (left tip)
	{	26.0,	0.0,	-1.0	},	//	 7 or 212 (rear bevel)
	{	34.5,	0.0,	0.0		},	//	 8 or 213 (tip)
	{	2.0,	4.0,	1.0		},	//	 9 or 214 (left upper)
	{	-4.5,	0.0,	0.0		},	//	10 or 215 (base of blade)
	{	30.0,	-1.5,	0.0		},	//	11 or 216 (right tip)
	{	1.5,	-5.0,	0.0		},	//	12 or 217 (right hilt)
	{	26.0,	0.0,	1.0		}	//	13 or 218 (front bevel)
};

//	The original staff GLCmds occupy 124 32-bit words.
//	A command (1st number on each line) occupies 1 word.
//	A vertex (remaining numbers on each line) occupies
//	3 words. To make the replacement the same length,
//	we repeat verteces in some of the fans below.
static int
gSwordGLCmds[] =
{
	//	Draw sword triangles
	-4, 11,  8,  7, 10,
	-4,  7,  8,  6, 10,
	-4,  6,  8, 13, 10,
	-4, 13,  8, 11, 10,

//	-4,  8, 13,  11, 7,
	-4,  8, 13,   6, 7,

	//	Draw upper surface of hilt
	6,   5,  4,  9,  2,  1, 12,
	//	Draw side surface of hilt
	6,   5,  9,  3,  1,  0, 12,
	//	Draw other side of hilt
	6,   5,  4,  3,  2,  0, 12,

	//	End of GLCmds list
	0
};

//	These are quick-and dirty, using existing blade
//	for the blade of the sword, and existing staff
//	for the hilt of the sword.
static texpoint_t
gSwordTexture[kNumStaffVerteces] =
{
#if	__MODEL_IS_CORVUS__
#if	__MAP_HILT_TO_STAFF__
	{	166,	178	},	//	0 or 205
	{	164,	178	},	//	1 or 206
	{	170,	178	},	//	2 or 207
	{	166,	230	},	//	3 or 208
	{	170,	230	},	//	4 or 209
	{	166,	182	},	//	5 or 210
#else
	{	7,		184	},	//	0 or 205
	{	4,		184	},	//	1 or 206
	{	10,		184	},	//	2 or 207
	{	7,		203	},	//	3 or 208
	{	10,		203	},	//	4 or 209
	{	7,		181	},	//	5 or 210
#endif
	{	61,		175	},	//	6 or 211
	{	66,		180	},	//	7 or 212
	{	66,		163	},	//	8 or 213
#if	__MAP_HILT_TO_STAFF__
	{	164,	230	},	//	9 or 214
#else
	{	4,		203	},	//	9 or 214
#endif
	{	70,		199	},	//	10 or 215
	{	72,		207	},	//	11 or 216
#if	__MAP_HILT_TO_STAFF__
	{	166,	234	},	//	12 or 217
#else
	{	7,		206	},	//	12 or 217
#endif
	{	66,		180	}	//	13 or 218
#elif	__MODEL_IS_KIERA__
	//	Data for Kiera - always maps hilt to staff
	{	5,	112	},		//	Hilt
	{	3,	112	},
	{	9,	112	},
	{	5,	164	},
	{	9,	164	},
	{	5,	116	},
	{	29,	173	},		//	Blade
	{	34,	182	},
	{	31,	161	},		//	.....tip of blade
	{	3,	164	},		//	Hilt
	{	38,	180	},		//	Blade
	{	35,	237	},
	{	5,	114	},		//	Hilt
	{	34,	182	}		//	Blade
#endif
};
#endif

//
//	Overwrite the GLCmds for the staff, replacing them
//	with commands to draw the sword.
//

int
FixSwordGLCmds(
	fmheader_t* pModelHeader, glcmd_t* pGLCmds, int glcmdstart, int glcmdcnt
)
{
	glcmd_t		theCmd;
	glvert_t	theVert;
	int			i, j, v, nVerts, glUsed;

	glUsed = 0;
	i = 0;
	pGLCmds += glcmdstart;
	while ( gSwordGLCmds[i] )
	{
		theCmd.theCmd = gSwordGLCmds[i++];
		*pGLCmds++ = theCmd;
		++glUsed;
		nVerts = (theCmd.theCmd > 0 )? theCmd.theCmd : -theCmd.theCmd;
		for ( j=0; j<nVerts; j++ )
		{
			v = gSwordGLCmds[i++];
			theVert.index_3d = v + kFirstStaffVertex;
			theVert.u = gSwordTexture[v].c[0] / 256.;
			theVert.v = gSwordTexture[v].c[1] / 256.;
			*((glvert_t*)pGLCmds) = theVert;
			pGLCmds += 3;
			glUsed  += 3;
		}
	}
	theCmd.theCmd = 0;
	while ( glUsed < glcmdcnt )
	{
		*pGLCmds++ = theCmd;
		++glUsed;
	}
	return	glUsed;
}

#if	(__MODEL_IS_CORVUS__ || __MODEL_IS_KIERA__)
//
//	Rotate the sword properly for this frame, and overwrite the model
//	data for this frame.
//
//	When the sword is substituted for the staff, and motion capture
//	data applied, there is a chance that the resulting frame will
//	not fit in a 256x256x256 grid using the frame's current translate
//	and rotate values. This can be handled in two ways: by clipping
//	points outside the allowed range, or by re-translating and
//	re-scaling the frame. __ENABLE_RESCALE__ controls which way is
//	used.
//
//	NOTE: if __ENABLE_RESCALE__ is non-zero, the re-translate /
//	re-scale code is used. This code is BROKEN. Currently it tries
//	to calculate the change in frame dimensions and "tweak" points
//	in the frame based on result. What it needs to do, is expand
//	out all points in the frame, calculate new minima and maxima
//	in all directions, and from this calculate new translation and
//	scale factors and re-compress the frame. The frame compression
//	code is needed anyway to build player models from parts.
//
void
StaffToSword(
	fmheader_t* pModelHeader, frame_t* pFrame, refskel_t* pRef,
	vec3dbl_t* pTranslate, vec3dbl_t* pRotate
)
{
	int			i, rescale=0;
	vec3dbl_t	sword[ kNumStaffVerteces ];
	vec3dbl_t	rot, tran;
	vec3dbl_t	maxs, mins;

	//	Copy the sword
	memcpy( sword, gSwordData, sizeof(gSwordData) );

	//	Tweak the sword geometry based on the frame name
	//	(mainly so pole-vault looks right)
#if	__TWEAK_FRAMES__
	if ( strncmp( "vault", pFrame->header.name, 5 ) )
#endif
	{
		for ( i=0; i<kNumStaffVerteces; i++ )
		{
			sword[i].c[0] = -sword[i].c[0];
		}
	}

#if	__NEW_SWORD_REFS__
	{
		//	Adjust the STAFF reference point to be above the hand
		//	(above the hand-staff joint), so that certain special
		//	effects match the new length of the staff. Note that
		//	we must do this BEFORE overwriting the frame, since
		//	the key point loci depend on the original geomettry.
		vec3dbl_t	jointPos[ JOINT_NUM_JOINTS ];
		vec3dbl_t	boneY[ BONE_NUM_BONES ];
		vec3dbl_t	boneZ[ BONE_NUM_BONES ];
		vec3dbl_t	ref, away;
		GetFrameKeyPoints( pFrame, pRef, &jointPos[0], &boneY[0], &boneZ[0] );
		ref = jointPos[ JOINT_RIGHT_CLOSED_HAND_STAFF ];
		away = jointPos[ JOINT_RIGHT_STAFF_END ];
		//	Calculate a point that is 8 units away from
		//	the hand joint along the axis of the staff.
		away.c[0] -= ref.c[0];
		away.c[1] -= ref.c[1];
		away.c[2] -= ref.c[2];
		M3DUnitVector( &away, &away );
		away.c[0] *= -kFXAdjust;
		away.c[1] *= -kFXAdjust;
		away.c[2] *= -kFXAdjust;
		ref.c[0]  += away.c[0];
		ref.c[1]  += away.c[1];
		ref.c[2]  += away.c[2];
		//	Store as new reference point.
		ref.c[0] = (ref.c[0] - pFrame->header.translate[0]) / pFrame->header.scale[0];
		ref.c[1] = (ref.c[1] - pFrame->header.translate[1]) / pFrame->header.scale[1];
		ref.c[2] = (ref.c[2] - pFrame->header.translate[2]) / pFrame->header.scale[2];
		pRef->refpoints[ H2CORVUS_STAFF ].origin.c[0] = ref.c[0];
		pRef->refpoints[ H2CORVUS_STAFF ].origin.c[1] = ref.c[1];
		pRef->refpoints[ H2CORVUS_STAFF ].origin.c[2] = ref.c[2];
	}
#endif

	//	Do X, Y, and Z rotations separately to make
	//	sure they happen in right order.

	//	X rotation
	rot.c[0] = pRotate[ BONE_STAFF ].c[0];
	rot.c[1] = 0.;
	rot.c[2] = 0.;
#if	__TWEAK_FRAMES__
	if ( !strncmp( "spining", pFrame->header.name, 7 ) )
		rot.c[0] += PI/4.;
#endif
	for ( i=0; i<kNumStaffVerteces; i++ )
	{
		M3DRotate( &sword[i], &rot );
	}
	//	Z rotation
	rot.c[0] = 0.;
	rot.c[1] = 0.;
	rot.c[2] = pRotate[ BONE_STAFF ].c[2];
	for ( i=0; i<kNumStaffVerteces; i++ )
	{
		M3DRotate( &sword[i], &rot );
	}
	//	Y rotation
	rot.c[0] = 0.;
	rot.c[1] = pRotate[ BONE_STAFF ].c[1];
	rot.c[2] = 0.;
	for ( i=0; i<kNumStaffVerteces; i++ )
	{
		M3DRotate( &sword[i], &rot );
	}

	//	Translate the sword to the correct location
	//	(that is, in the closed fist)
	for ( i=0; i<kNumStaffVerteces; i++ )
	{
		M3DTranslate( &sword[i], &pTranslate[ BONE_STAFF ] );
	}

	//	OK, now the sword array has the correct 3D space coordinates
	//	for the sword geometry in this frame. We need to store these
	//	using the scale and translation factors for this frame.

	//	Translate
	tran.c[0] = -pFrame->header.translate[0];
	tran.c[1] = -pFrame->header.translate[1];
	tran.c[2] = -pFrame->header.translate[2];
	for ( i=0; i<kNumStaffVerteces; i++ )
	{
		M3DTranslate( &sword[i], &tran );
	}
	//	Scale & bounds check
	tran.c[0] = 1. / pFrame->header.scale[0];
	tran.c[1] = 1. / pFrame->header.scale[1];
	tran.c[2] = 1. / pFrame->header.scale[2];
	mins.c[0] = mins.c[1] = mins.c[2] = 0.;
	maxs.c[0] = maxs.c[1] = maxs.c[2] = 255.;
	for ( i=0; i<kNumStaffVerteces; i++ )
	{
		int		j;
		M3DScale( &sword[i], &tran );
		//	DANGER: After scaling, all coordinates must be in the
		//	range 0..255. If anything is bigger, we need to re-scale
		//	and / or re-translate the entire frame.
		for ( j=0; j<3; j++ )
		{
			if ( sword[i].c[j] < 0. )
			{
#if	__ENABLE_RESCALE__

				//	WARNING: KLUDGE TO MAKE RECURSION TERMINATE
				if ( sword[i].c[j] > -0.1 )
					sword[i].c[j] = 0.;
				else
					rescale = 1;
				if ( sword[i].c[j] < mins.c[j] )
					mins.c[j] = sword[i].c[j];
#else
				printf("    Clipping new object in frame '%s'.\n", pFrame->header.name );
				sword[i].c[j] = 0.;
#endif
			}
			if ( sword[i].c[j] >= 256. )
			{
#if	__ENABLE_RESCALE__
				rescale = 1;
				if ( sword[i].c[j] > maxs.c[j] )
					maxs.c[j] = sword[i].c[j];
#else
				printf("    Clipping new object in frame '%s'.\n", pFrame->header.name );
				sword[i].c[j] = 255;
#endif
			}
		}
	}

#if	__ENABLE_RESCALE__
	//	See if we have to rescale the whole frame.
	if ( rescale )
	{
		vec3dbl_t	vert;
		vec3dbl_t	newScale, newTrans;
		int		j;
		printf("    Re-scaling frame '%s'...\n", pFrame->header.name );
		for ( j=0; j<3; j++ )
		{
			//	OK, we know that all the points in the original
			//	frame fit in the 0..255 range; the only points
			//	that create new minima or maxima are in the sword.
			//	If a new minimum is created, and it's less than
			//	zero, we need to change the translation factor
			//	to make the minimum coordinate zero. If this,
			//	or anything else, leads to any coordinate being
			//	greater than 255, we need to change the scale to
			//	make the maximum coordinate 255.
			newTrans.c[j] = pFrame->header.translate[j];	//	Original
			newScale.c[j] = pFrame->header.scale[j];		//	Original
			if ( mins.c[j] < 0. )
				newTrans.c[j] += newScale.c[j] * mins.c[j];	//	Adjust translation
			newScale.c[j] *= (maxs.c[j]-mins.c[j]) / 255.;	//	Adjust scale
		}
		//	Re-process all current verteces
		for ( i=0; i<pModelHeader->vert3dcnt; i++ )
		{
			vert.c[0] =
				pFrame->verts[i].v[0] * pFrame->header.scale[0] +
					pFrame->header.scale[0];
			vert.c[1] =
				pFrame->verts[i].v[1] * pFrame->header.scale[1] +
					pFrame->header.scale[1];
			vert.c[2] =
				pFrame->verts[i].v[2] * pFrame->header.scale[2] +
					pFrame->header.scale[2];
			//	NOTE: OVERWRITE OLD MODEL DATA!
			//	WARNING: NOT ENDIAN CLEAN
			pFrame->verts[i].v[0] = (vert.c[0] - newTrans.c[0]) / newScale.c[0];
			pFrame->verts[i].v[1] = (vert.c[1] - newTrans.c[1]) / newScale.c[1];
			pFrame->verts[i].v[2] = (vert.c[2] - newTrans.c[2]) / newScale.c[2];
		}
		//	Update scale and translation factors
		//	NOTE: OVERWRITE OLD MODEL DATA
		//	WARNING: NOT ENDIAN CLEAN
		for ( i=0; i<3; i++ )
		{
			pFrame->header.scale[i] = newScale.c[i];
			pFrame->header.translate[i] = newTrans.c[i];
		}
		//	Recurse to calculate correct sword position
		StaffToSword( pModelHeader, pFrame, pTranslate, pRotate );
	}
	else
#endif
	{
		//	OK, now we can overwrite the data in the frame
		for ( i=0; i<kNumStaffVerteces; i++ )
		{
			pFrame->verts[kFirstStaffVertex+i].v[0] = (int)(sword[i].c[0]);
			pFrame->verts[kFirstStaffVertex+i].v[1] = (int)(sword[i].c[1]);
			pFrame->verts[kFirstStaffVertex+i].v[2] = (int)(sword[i].c[2]);
		}
	}

#if	__NEW_SWORD_REFS__
	{
		//	Adjust the BLADE reference point to match the
		//	tip of the sword. Note that the vertex data is
		//	already normalized to 0..255 range. Note that
		//	this must be done AFTER the new sword data has
		//	been written.
		vec3dbl_t	ref;
		ref.c[0] = pFrame->verts[ kFirstStaffVertex+kTipVertex ].v[0];
		ref.c[1] = pFrame->verts[ kFirstStaffVertex+kTipVertex ].v[1];
		ref.c[2] = pFrame->verts[ kFirstStaffVertex+kTipVertex ].v[2];
		pRef->refpoints[ H2CORVUS_BLADE ].origin.c[0] = ref.c[0];
		pRef->refpoints[ H2CORVUS_BLADE ].origin.c[1] = ref.c[1];
		pRef->refpoints[ H2CORVUS_BLADE ].origin.c[2] = ref.c[2];
	}
#endif

}
#endif

