#include <stdio.h>
#include <stdlib.h>

#define	MAX_FM_TRIANGLES	2048
#define MAX_FM_VERTS		2048
#define MAX_FM_FRAMES		2048
#define MAX_FM_SKINS		64
#define	MAX_FM_SKINNAME		64
#define MAX_FM_MESH_NODES	16

typedef unsigned char byte;

typedef struct
{
	char ident[32];
	int version;
	int size;
} header_t;

typedef struct
{
	int ident;
	int version;

	int skinwidth;
	int skinheight;
	int framesize;	// byte size of each frame

	int num_skins;
	int num_xyz;
	int num_st;		// greater than num_xyz for seams
	int num_tris;
	int num_glcmds;	// dwords in strip/fan command list
	int num_frames;

	int ofs_skins;	// each skin is a MAX_SKINNAME string
	int ofs_st;		// byte offset from start for stverts
	int ofs_tris;		// offset for dtriangles
	int ofs_frames;	// offset for first frame
	int ofs_glcmds;	
	int ofs_end;		// end of file
} dmdl_t;

typedef struct
{
	int skinwidth;
	int skinheight;
	int framesize;		// byte size of each frame
	int num_skins;
	int num_xyz;
	int num_st;			// greater than num_xyz for seams
	int num_tris;
	int num_glcmds;		// dwords in strip/fan command list
	int num_frames;
	int num_mesh_nodes;
} fmheader_t;

typedef struct 
{
	short	index_xyz[3];
	short	index_st[3];
} fmtriangle_t;

typedef struct
{
	byte	tris[MAX_FM_TRIANGLES>>3];
	byte	verts[MAX_FM_VERTS>>3];
	short	start_glcmds, num_glcmds;
} fmmeshnode_t;

void write_section(FILE *fp, char *ident, int version,  void *data, int size)
{
	header_t header;

	memset(&header, 0, sizeof(header));
	strcpy(header.ident, ident);
	header.version = version;
	header.size = size;

	fwrite(&header, 1, sizeof(header), fp);

	if (data != NULL)
		fwrite(data, 1, size, fp);
}

void main(int argc, char **argv)
{
	FILE *fp;
	long filesize;
	char *buffer;
	fmheader_t header;
	char skin[MAX_FM_SKINNAME];
	fmmeshnode_t meshnode;
	dmdl_t *qheader;
	fmtriangle_t *triangles;
	int i, j, n;

	if (argc != 3) {
		fprintf(stderr, "usage: %s infile outfile\n", argv[0]);
		exit(1);
	}

	if ((fp = fopen(argv[1], "rb")) == NULL) {
		fprintf(stderr, "can't open %s\n", argv[1]);
		exit(1);
	}

	fseek(fp, 0, SEEK_END);
	filesize = ftell(fp);
	fseek(fp, 0, SEEK_SET);

	if ((buffer = malloc(filesize)) == NULL) {
		fprintf(stderr, "out of memory\n");
		exit(1);
	}

	fread(buffer, 1, filesize, fp);
	fclose(fp);

	if ((fp = fopen(argv[2], "wb")) == NULL) {
		fprintf(stderr, "can't create %s\n", argv[2]);
		exit(1);
	}

	qheader = (dmdl_t *)buffer;

	header.skinwidth = qheader->skinwidth;
	header.skinheight = qheader->skinheight;
	header.framesize = qheader->framesize;
	header.num_skins = 1;
	header.num_xyz = qheader->num_xyz;
	header.num_st = qheader->num_st;
	header.num_tris = qheader->num_tris;
	header.num_glcmds = qheader->num_glcmds;
	header.num_frames = qheader->num_frames;
	header.num_mesh_nodes = 1;

	write_section(fp, "header", 2, &header, sizeof(fmheader_t));

	memset(skin, 0, MAX_FM_SKINNAME);
	strcpy(skin, "!skin.m8");

	write_section(fp, "skin", 1, skin, MAX_FM_SKINNAME);

	write_section(fp, "st coord", 1, buffer + qheader->ofs_st, qheader->ofs_tris - qheader->ofs_st);

	write_section(fp, "tris", 1, buffer + qheader->ofs_tris, qheader->ofs_frames - qheader->ofs_tris);

	write_section(fp, "frames", 1, buffer + qheader->ofs_frames, qheader->ofs_glcmds - qheader->ofs_frames);

	write_section(fp, "glcmds", 1, buffer + qheader->ofs_glcmds, qheader->ofs_end - qheader->ofs_glcmds);

	memset(&meshnode, 0, sizeof(fmmeshnode_t));

	triangles = (fmtriangle_t *)(buffer + qheader->ofs_tris);

	for(i = 0; i < header.num_tris; i++)
		meshnode.tris[i >> 3] |= 1 << (i & 7 );
	
	for (i = 0 ; i < header.num_tris ; i++) {
		for (j = 0 ; j < 3 ; j++) {	
			n = triangles[i].index_xyz[j];

			if ((meshnode.tris[i >> 3] & (1 << (i & 7 ))) != 0)
				meshnode.verts[n >> 3] |= 1 << (n & 7 );
		}
	}

	meshnode.start_glcmds = 0;
	meshnode.num_glcmds = header.num_glcmds;

	write_section(fp, "mesh nodes", 3, &meshnode, sizeof(fmmeshnode_t));

	fclose(fp);
	exit(0);
}
