/*******************************
PCX2M8 -  Converts .pcx graphics files to
  the .m8 format for use as heretic2 textures.

  Author:  Ted Mielczarek (luser)
  			tam4@lehigh.edu
            http://www.lehigh.edu/~tam4

  Yes, this is written for windows.  It'd be incredibly easy to make
  	a commandline version.  Just have it convert(filename).  Why didn't
    I do this?  I like windows.  It seemed a lot easier to have a little
    GUI.  If you don't like it, change it.

  Large parts of this code were taken from pcx2wal by
  Trey Harrison (Troy)
  - trey@u.washington.edu
  - http://starbase.NeoSoft.COM/~otaku/

  and wal2m8 by Brend McLeod (Drizzt)
  - mcleod@wgn.net
********************************/
#include <stdio.h>
#include <windows.h>
#include <commctrl.h>

#include "pcx2m8.rh"
#include "wal2m8.h"

// globals
HINSTANCE hinst;
HWND hwndMain;
OPENFILENAME ofn;
unsigned char pcx_pal[768];
//--------------------------------------------------------------
// Prototypes
int MbError(char *fmt, ...);
BOOL CALLBACK MainDlgProc(HWND hwnd,UINT uMsg,WPARAM wParam,LPARAM lParam);
BOOL CALLBACK ConvDlgProc(HWND hwnd,UINT uMsg,WPARAM wParam,LPARAM lParam);
void InitOFN(HWND hwnd);
BOOL GetFileName(char szName[]);
void convert(char szFile[]);
// functions taken from pcx2wal.cpp with minor modifications
__inline unsigned char remap_pixel(int ir, int ig, int ib);
unsigned char average_pixels(unsigned char *src, int miplevel, int bytesperline);
void write_mip(FILE *f,unsigned char *base_mip,int base_width, int base_height, int miplevel);
//--------------------------------------------------------------
// Code
int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
    LPSTR lpszCmdLine, int nCmdShow)
{
   UNREFERENCED_PARAMETER(lpszCmdLine);
   UNREFERENCED_PARAMETER(nCmdShow);
   UNREFERENCED_PARAMETER(hPrevInstance);


   hinst = hInstance;  // save instance handle

	// Create the main window.
	if(DialogBoxParam(hinst, "MainDlg", NULL, (DLGPROC)MainDlgProc, NULL) == -1)
    {
    	MessageBox(NULL,"Error creating Main Dialog","Error",MB_ICONERROR|MB_OK);
        return FALSE;
    }

    return TRUE;
}

/************************************
* MainDlgProc -                     *
*     Main Window Callback function *
************************************/
BOOL CALLBACK MainDlgProc(
    HWND hwnd,        // handle of window
    UINT uMsg,        // message identifier
    WPARAM wParam,    // first message parameter
    LPARAM lParam)    // second message parameter
{
	char szFile[256];

   switch (uMsg)
   {
   	case WM_INITDIALOG:
    	hwndMain = hwnd;
        InitOFN(hwnd);
   		return TRUE;

      case WM_COMMAND:
         switch(LOWORD(wParam))
         {
         	case IDC_BTNADDFILE:
            	if(GetFileName(szFile))
                	SendDlgItemMessage(hwnd,IDC_FILELIST,LB_ADDSTRING,0,LPARAM(szFile));
                break;

         	case IDC_BTNCLEAR:
            	SendDlgItemMessage(hwnd,IDC_FILELIST,LB_RESETCONTENT,0,0);
                break;
                
            case IDC_BTNCONVERT:
            	if(SendDlgItemMessage(hwnd,IDC_FILELIST,LB_GETCOUNT,0,0) <= 0)
                {
                	MbError("Nothing to convert!");
                    break;
                }

                DialogBoxParam(hinst, "ConvDlg", NULL, (DLGPROC)ConvDlgProc, LPARAM(hwnd));
                break;

            case IDC_BTNEXIT:
            	// close window
               PostMessage(hwnd,WM_CLOSE,0,0);
               break;
         }
         return TRUE;

      case WM_CLOSE:
         EndDialog(hwnd,TRUE);
         return TRUE;
   }
   return FALSE;
}
//--------------------------------------------------------------
/****************************
*  ConvDlgProc - Dialog Box Procedure
*    for the "Converting" dialog.
*   Display the filename being converted and a progress bar.
*   Do one file approx every .5 seconds.
*****************************/
BOOL CALLBACK ConvDlgProc(
    HWND hwnd,        // handle of window
    UINT uMsg,        // message identifier
    WPARAM wParam,    // first message parameter
    LPARAM lParam)    // second message parameter
{
	static int i;
    static int count;
    static HWND hwndParent;
    char szFile[256],szTitle[256];
	UNREFERENCED_PARAMETER(wParam);
	UNREFERENCED_PARAMETER(lParam);

	switch (uMsg)
	{
		case WM_INITDIALOG:
        	i = 0;
            hwndParent = HWND(lParam);
            count = SendDlgItemMessage(hwndParent,IDC_FILELIST,LB_GETCOUNT,0,0);
            SetTimer(hwnd,1,500,NULL);
            SendDlgItemMessage(hwnd,IDC_PROGRESS,PBM_SETRANGE,0,MAKELPARAM(0,count));
            SendDlgItemMessage(hwnd,IDC_PROGRESS,PBM_SETSTEP,WPARAM(1),0);
            return TRUE;

		case WM_TIMER:
        	SendDlgItemMessage(hwndParent,IDC_FILELIST,LB_GETTEXT,i,LPARAM(szFile));
            SetDlgItemText(hwnd,IDC_FILENAME,szFile);
            wsprintf(szTitle,"Converting %s",szFile);
            SetWindowText(hwnd,szFile);
            convert(szFile);
            i++;
            SendDlgItemMessage(hwnd,IDC_PROGRESS,PBM_STEPIT,0,0);
            if(i==count)
            {
            	KillTimer(hwnd,1);
            	EndDialog(hwnd,TRUE);
            }
            return TRUE;;
    }
    return FALSE;
}
/**************************
*   MbError - Useful little function that
*    creates a MessageBox from a format string and
*   variable argument list (ala printf).  Saves
*   the hassle of using wsprintf to format
*   when reporting error messages.
**************************/
int MbError(char *fmt, ...)
{
   char buffer[256];
   va_list argptr;
   int cnt;

   va_start(argptr, fmt);
   cnt = vsprintf(buffer, fmt, argptr);
   va_end(argptr);

   MessageBox(hwndMain,buffer,"Error",MB_ICONEXCLAMATION|MB_OK);

   return(cnt);
}
/**************************
*  InitOFN - initialize the OPENFILENAME struct
*    with the owner window and other required data
*   that won't change.
**************************/
void InitOFN(HWND hwnd)
{
   ofn.lStructSize = sizeof(OPENFILENAME);
   ofn.hwndOwner = hwnd;
   ofn.lpstrFilter = "PCX Files\0*.pcx\0All Files\0*.*\0\0";
   ofn.lpstrCustomFilter = NULL;
   ofn.nFilterIndex = 0;
   ofn.nMaxFile = MAX_PATH;
   ofn.lpstrFileTitle = NULL;
   ofn.lpstrInitialDir = NULL;
   ofn.lpstrDefExt = "pcx";
   ofn.Flags = OFN_FILEMUSTEXIST | OFN_HIDEREADONLY;
   ofn.lpstrTitle = "Open PCX File";
}
/**************************
*  GetFileName - get a PCX filename and return it in
*    the szName parameter.  Uses the OPENFILENAME struct
*   initialized in InitOFN.  Pops up a messagebox on
*   an error.  Returns TRUE if a filename was returned,
*   FALSE on error or cancel.
**************************/
BOOL GetFileName(char szName[])
{
	char szBuf[256];
	BOOL bResult;

   lstrcpy(szBuf,szName);
   ofn.lpstrFile = szBuf;
   ofn.lpstrFile[0] = 0;

   bResult = GetOpenFileName(&ofn);

   if(bResult)
   {
   	strcpy(szName,szBuf);
   }
   else
   {
      switch(CommDlgExtendedError())
      {
         case CDERR_FINDRESFAILURE:
            MbError("Find resource failure!");
            break;
         case CDERR_INITIALIZATION:
            MbError("Initialization error!");
            break;
         case CDERR_LOCKRESFAILURE:
            MbError("Lock resource error!");
            break;
         case CDERR_LOADRESFAILURE:
            MbError("Load resource error!");
            break;
         case CDERR_LOADSTRFAILURE:
            MbError("Load string error!");
            break;
         case CDERR_MEMALLOCFAILURE:
            MbError("Memory allocation error!");
            break;
         case CDERR_MEMLOCKFAILURE:
            MbError("Memory lock failure!");
            break;
         case CDERR_NOHINSTANCE:
            MbError("No hinstance!");
            break;
         case CDERR_NOHOOK:
            MbError("No hook!");
            break;
         case CDERR_NOTEMPLATE:
            MbError("No template!");
            break;
         case CDERR_STRUCTSIZE:
            MbError("Incorrect struct size!");
            break;
         case FNERR_BUFFERTOOSMALL:
            MbError("Filename buffer too small!");
            break;
         case FNERR_INVALIDFILENAME:
            MbError("Invalid filename!");
            break;
         case FNERR_SUBCLASSFAILURE:
            MbError("Subclassing failure!");
            break;
         default:
            // user probably pressed cancel
            break;
      }
   }

   return bResult;
}
/**************************
*  convert - The meat of the program.  Parts stolen from
*    pcx2wal.cpp and wal2m8.c.  Other parts written by myself.
*    takes a filename as parameter and attempts to convert the
*    pcx into a m8 file.
**************************/
void convert(char szFile[])
{
//  these vars from wal2m8.c
	FILE	*m8File;
	FILE	*f;
	int   i,j;
	h2miptex_t m8Header;
	char szM8Name[MAX_PATH];
//  these vars from pcx2wal.cpp
	int   width,height;
	short w,h;
	int   mipsize;

	unsigned char *mip=0;
	unsigned char cur_byte;
	int run_len;
// end of stolen vars
	char *dot,szTitle[32];
    unsigned nw,nh;
    int  numMips;

    // open pcx file
    f = fopen(szFile,"rb");
    if (!f)
    {
      MbError("Invalid input .pcx file: %s\n",szFile);
      return;
    }

    // read width/height
    fseek(f,8,SEEK_SET);
    fread(&w,sizeof(short),1,f);
    fread(&h,sizeof(short),1,f);

    width  = w+1;
    height = h+1;

    // get pic size
    mipsize = width*height;

    mip = new unsigned char[mipsize];

    fseek(f,128,SEEK_SET);
    j = 0;
    //decoding loop type thing

    while(j < mipsize)
    {
      cur_byte = fgetc(f);
      if((cur_byte & 0xC0) == 0xC0)
      {
        run_len = cur_byte & 0x3F;
        cur_byte = fgetc(f);
        for( ;(run_len>0) && (j<mipsize); run_len--, j++)
        {
          mip[j] = cur_byte;
        }
      }
      else
      {
        mip[j] = cur_byte;
        j++;
      }
    }

    // get palette
    fseek(f,-768,SEEK_END);
    fread(pcx_pal,768,1,f);

    fclose(f);

    // get m8 file name
	strcpy(szM8Name, szFile);
	dot = strchr(szM8Name,'.');
    strcpy(dot,".m8");

    GetFileTitle(szM8Name,szTitle,sizeof(szTitle));
    dot = strchr(szTitle,'.');
    if(dot != NULL)
    	dot = 0;

    // setup m8 header
    memset(&m8Header, 0, sizeof(h2miptex_t));
    m8Header.identifier = 2;// 02 00 00 00
    strcpy(m8Header.name, szTitle);

    // first mip is size of pcx, and offset 1040
    // which is after the m8 header
    m8Header.width[0] = width;
    m8Header.height[0] = height;
    m8Header.offsets[0] = 1040;
    // calc width, height, and offset for each additional mip
    for (i = 1; i < 16; i++)
    {
        // new width and height are old dimension / miplevel
    	nw = width / (1<<i);
        nh = height / (1<<i);
        if(nw < 1)
	       	nw = 1;
        if(nh < 1)
	       	nh = 1;

        m8Header.width[i] = nw;
        m8Header.height[i] = nh;

    	m8Header.offsets[i] = m8Header.offsets[i-1] + nw*nh;
        // if width and height both 1, we've hit the smallest mip
        // no need to make any more.  h2 appears to ignore mips
        // that have width, height, and offset == 0
        if(nw == 1 && nh == 1)
			break;
    }

    numMips = i+1;

    // no anim comes after this.  maybe change this in the future?
    memset(m8Header.animname,0,sizeof(m8Header.animname));

    // set palette to pcx's palette
    for (i = 0; i < 768; i++)
    {
    	m8Header.palette[i] = pcx_pal[i];
    }

    m8Header.flags = 0;
    m8Header.contents = 0;
    m8Header.value = 0;

    m8File = fopen(szM8Name, "wb");
    if (m8File == NULL)
    {
    	MbError("Could not open %s for writing.", szM8Name);
        return;
    }

    // write header
    fseek(m8File, 0, SEEK_SET);
    fwrite(&m8Header, sizeof(h2miptex_t), 1, m8File);

    fseek(m8File, sizeof(h2miptex_t), SEEK_SET);

    // write out 16 mips
    for(i=0;i<numMips;i++)
    	write_mip(m8File,mip,width,height,i);

   	fclose(m8File);

    delete [mipsize] mip;
}
/**************************
*  write_mip - scales down the largest mip based on its original
*    dimensions and the miplevel (0-15)
**************************/
void write_mip(FILE *f,unsigned char *base_mip,int base_width, int base_height, int miplevel)
{
  int mipsize = base_width/(1<<miplevel) * base_height/(1<<miplevel);
  int i,j;
  int w,h;

  unsigned char *new_mip=0;
  unsigned char *dst=0;

  int s,t,mipstep;

  printf("Writing mip %d...\n",miplevel+1);

  //simpler case
  if (miplevel==0)
  {
    //simple case, 1x1 pixel ratio
	fwrite(base_mip,mipsize,1,f);
      return;
  }

  w = base_width  >> miplevel;
  h = base_height >> miplevel;

  mipstep = 1<<miplevel;

  new_mip = new unsigned char [mipsize];
  dst     = new_mip;

  for (j=0,t=0; j<h; j++,t += mipstep)
  for (i=0,s=0; i<w; i++,s += mipstep)
  {
    *dst = average_pixels(&base_mip[t*base_width+s],mipstep,base_width);
    dst++;
  }

  fwrite(new_mip,mipsize,1,f);
  delete [mipsize] new_mip;
}
/**************************
*  I didn't write this function, but it averages the pixels
*   to get an ok-looking color
**************************/
unsigned char average_pixels(unsigned char *src, int miplevel, int bytesperline)
{
  unsigned char ir,ig,ib;
  double r=0,g=0,b=0;
  int    i,j;
  int    total_pixels=0;

  double oot;

  unsigned char *texel=src;
  int texeloffs;

  for (j=0;j<miplevel;j++,texel += (bytesperline-miplevel))
  for (i=0;i<miplevel;i++)
  {
    texeloffs = (*texel) * 3;

    r += (double)pcx_pal[texeloffs+0];
    g += (double)pcx_pal[texeloffs+1];
    b += (double)pcx_pal[texeloffs+2];

    total_pixels++;
    texel++;
  }

  oot = 1.0 / (double)(total_pixels);
  r *= oot;
  g *= oot;
  b *= oot;

  ir = (unsigned char)r;
  ig = (unsigned char)g;
  ib = (unsigned char)b;

  return remap_pixel(ir,ig,ib);
}

#define ABS(x) (x)>(0)?(x):(-(x))
/**************************
*  remap_pixel -  get the closest color to the color specified.
**************************/
__inline unsigned char remap_pixel(int ir, int ig, int ib)
{
  int i;
  int diff,min_diff=99999;
  unsigned char match=0;

  unsigned char *pal;

  pal = pcx_pal;

  for (i=0;i<256;i++,pal+=3)
  {
    diff = 0;

    diff += ABS(pal[0]-ir);
    diff += ABS(pal[1]-ig);
    diff += ABS(pal[2]-ib);
    
    if (diff<min_diff)
    {
      min_diff = diff;
      match = i;
    }
  }
  
  return match;
}
