/*
Copyright (C) Matthew 'pagan' Baranowski & Sander 'FireStorm' van Rossen

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "system.h"
#include "ndictionary.h"
#include "md3gl.h"
#include "md3view.h"
#include "DiskIO.h"

bool	loadmesh( gl_mesh& newMesh , FILE* F )
{
	char	ID[5];

	fread(ID, 4,1,F);ID[4]=0;
	if (strcmp(ID,"IDP3")==0)
	{
		char		name[69];
		UINT32		triangleStart;
		UINT32		headerSize;
		UINT32		texvecStart;
		UINT32		vertexStart;
		UINT32		meshSize;

		fread(name,68,1,F);name[65]=0; //65 chars, 32-bit aligned == 68 chars in file
		strcpy(newMesh.name,name);

		newMesh.meshFrameNum	= get32(F);
		newMesh.skinNum			= get32(F);
		newMesh.vertexNum		= get32(F);
		newMesh.triangleNum		= get32(F);
		triangleStart			= get32(F);
		headerSize				= get32(F);
		texvecStart				= get32(F);
		vertexStart				= get32(F);
		meshSize				= get32(F);

		if ( ( headerSize == 108           ) && //header should be 108 bytes long
			 ( meshSize   >  triangleStart ) &&
			 ( meshSize   >  texvecStart   ) &&
			 ( meshSize   >  vertexStart   ) )
		{
			unsigned int i,j;
			char	texName[68];

			newMesh.meshFrames	  = new FMeshFrame[newMesh.meshFrameNum];
			newMesh.triangles	  = new TriVec	  [newMesh.triangleNum ];
			newMesh.textureCoord  = new TexVec    [newMesh.vertexNum   ];
			newMesh.iterMesh	  = new Vec3	  [newMesh.vertexNum   ];
			newMesh.bindings	  = new GLuint    [newMesh.skinNum     ];

			for (i=0;i<newMesh.skinNum;i++)
			{
				fread(texName,68,1,F);texName[65]=0;
				//65 chars, 32-bit aligned == 68 chars in file

				newMesh.bindings[ i ] =
					loadTexture(texName);
			}

			for (i=0;i<newMesh.triangleNum;i++)
			{
				newMesh.triangles[ i ][ 0 ] = get32(F);
				newMesh.triangles[ i ][ 1 ] = get32(F);
				newMesh.triangles[ i ][ 2 ] = get32(F);
			}

			for (i=0;i<newMesh.vertexNum;i++)
			{
				newMesh.textureCoord[ i ][ 0 ] = getFloat(F);
				newMesh.textureCoord[ i ][ 1 ] = getFloat(F);
			}

			UINT16 unknown;

			for (i=0;i<newMesh.meshFrameNum;i++)
			{

				newMesh.meshFrames[ i ] = 
					new Vec3[newMesh.vertexNum];

				for (j=0;j<newMesh.vertexNum;j++)
				{
					// big hack, change later!
					newMesh.meshFrames[ i ][ j ][0]  = ((float)get16(F)) / 64;
					newMesh.meshFrames[ i ][ j ][1]  = ((float)get16(F)) / 64;
					newMesh.meshFrames[ i ][ j ][2]  = ((float)get16(F)) / 64;
					unknown = get16(F);
				}
			}
			
			return true;
		} else
			return false;
	} else 
		return false;
}

bool	loadmodel( gl_model& newModel , FILE* F )
{
	char	ID[5];

	ID[0] = fgetc(F);
	ID[1] = fgetc(F);
	ID[2] = fgetc(F);
	ID[3] = fgetc(F);
	ID[4] = 0;
	if (strcmp(ID,"IDP3")==0)
	{
		char		name[69];
		UINT32		maxskinNum;
		UINT32		headerSize;
		UINT32		tagStart;
		UINT32		tagEnd;
		UINT32		fileSize;
		UINT32		version;

		version					= get32(F);

		fread(name,68,1,F);name[65]=0; //65 chars, 32-bit aligned == 68 chars in file

		newModel.frameNum		= get32(F);
		newModel.tagNum			= get32(F);
		newModel.meshNum		= get32(F);
		maxskinNum				= get32(F);
		headerSize				= get32(F);
		tagStart				= get32(F);
		tagEnd					= get32(F);
		fileSize				= get32(F);

		if ( ( headerSize == 108           ) && //header should be 108 bytes long
			 ( fileSize   >  tagStart      ) &&
			 ( fileSize   >  tagEnd        ) &&
			 ( version    == 15            ) )
		{
			unsigned int i,j;

			//initialize matrix to identity matrix
			for (i=0;i<16;i++) 
				newModel.CTM_matrix[ i ] = 0;
			newModel.CTM_matrix[  1 ] = 1;
			newModel.CTM_matrix[  6 ] = 1;
			newModel.CTM_matrix[ 11 ] = 1;
			newModel.CTM_matrix[ 16 ] = 1;

			newModel.modelListPosition = NULL;
			newModel.parent			= NULL;
			newModel.currentFrame	= 0;
			newModel.isBlended		= false;
			newModel.blendParam1	= GL_ONE;
			newModel.blendParam2	= GL_ONE;
			
			newModel.boneFrames = 
				new BoneFrame[newModel.frameNum];

			for (i=0;i<newModel.frameNum;i++)
			{
				newModel.boneFrames[ i ].Mins[ 0 ]		= getFloat(F);
				newModel.boneFrames[ i ].Mins[ 1 ]		= getFloat(F);
				newModel.boneFrames[ i ].Mins[ 2 ]		= getFloat(F);
				newModel.boneFrames[ i ].Maxs[ 0 ]		= getFloat(F);
				newModel.boneFrames[ i ].Maxs[ 1 ]		= getFloat(F);
				newModel.boneFrames[ i ].Maxs[ 2 ]		= getFloat(F);
				newModel.boneFrames[ i ].Position[ 0 ]	= getFloat(F);
				newModel.boneFrames[ i ].Position[ 1 ]	= getFloat(F);
				newModel.boneFrames[ i ].Position[ 2 ]	= getFloat(F);
				newModel.boneFrames[ i ].Scale			= getFloat(F);
				
				fread(newModel.boneFrames[ i ].Creator,16,1,F);
			}
			if (newModel.tagNum!=0)
			{
				newModel.linkedModels = new gl_model*[newModel.tagNum];
				for (i=0;i<newModel.tagNum;i++)
					newModel.linkedModels[ i ] = NULL;

				newModel.tags = 
					new TagFrame[newModel.frameNum];

				for (i=0;i<newModel.frameNum;i++)
				{
					newModel.tags[ i ] = 
						new Tag[newModel.tagNum];

					for (j=0;j<newModel.tagNum;j++)
					{
						fread(newModel.tags[ i ][ j ].Name   ,12,1,F);
						fread(newModel.tags[ i ][ j ].unknown,52,1,F);
						newModel.tags[ i ][ j ].Position[ 0 ]		= getFloat(F);
						newModel.tags[ i ][ j ].Position[ 1 ]		= getFloat(F);
						newModel.tags[ i ][ j ].Position[ 2 ]		= getFloat(F);
						newModel.tags[ i ][ j ].Matrix[ 0 ][ 0 ]	= getFloat(F);
						newModel.tags[ i ][ j ].Matrix[ 1 ][ 0 ]	= getFloat(F);
						newModel.tags[ i ][ j ].Matrix[ 2 ][ 0 ]	= getFloat(F);
						newModel.tags[ i ][ j ].Matrix[ 0 ][ 1 ]	= getFloat(F);
						newModel.tags[ i ][ j ].Matrix[ 1 ][ 1 ]	= getFloat(F);
						newModel.tags[ i ][ j ].Matrix[ 2 ][ 1 ]	= getFloat(F);
						newModel.tags[ i ][ j ].Matrix[ 0 ][ 2 ]	= getFloat(F);
						newModel.tags[ i ][ j ].Matrix[ 1 ][ 2 ]	= getFloat(F);
						newModel.tags[ i ][ j ].Matrix[ 2 ][ 2 ]	= getFloat(F);
					}
				}
			} else
			{
				newModel.linkedModels	= NULL;
				newModel.tags			= NULL;
			}

			newModel.meshes =
				new gl_mesh[newModel.meshNum];

			for (i=0;i<newModel.meshNum;i++)
			{
				if (!loadmesh(newModel.meshes[ i ],F))
					return false;
			}

			return true;
		} else
			return false;
	} else 
		return false;
}

/*
cleans up allocated model data
*/
void delete_gl_model( gl_model *model )
{
	unsigned int i, j, t;
	gl_mesh *mesh;	
	gl_model *linkModel, *parent;

	if (model == NULL) return;
		
	// delete tag and bone frame data
	if (model->tags) {
		delete [] model->tags; 
		model->tags = NULL;
	}

	if (model->boneFrames) {
		delete [] model->boneFrames; 
		model->boneFrames = NULL;
	}

	// remove all meshes
	for (i=0 ; i<model->meshNum ; i++) {
		mesh = &model->meshes[i];

		if (mesh->triangles) {
			delete [] mesh->triangles; 
			mesh->triangles = NULL;
		}

		if (mesh->textureCoord) {
			delete [] mesh->textureCoord; mesh->textureCoord = NULL;
		}

		if (mesh->iterMesh) { 
			delete [] mesh->iterMesh; mesh->iterMesh = NULL;
		}

		// free textures from the mesh
		for (t=0 ; t<mesh->skinNum ; t++) 
			freeTexture( mesh->bindings[t] );

		if (mesh->bindings) {
			delete [] mesh->bindings; mesh->bindings = NULL;
		}
		
		for (j=0 ; j<mesh->meshFrameNum ; j++) {
			if (mesh->meshFrames[j]) {
				delete [] mesh->meshFrames[j]; mesh->meshFrames[j] = NULL;
			}
		}

		if (mesh->meshFrames) {
			delete [] mesh->meshFrames;	mesh->meshFrames = NULL;
		}
	}

	if (model->meshes) {
		delete [] model->meshes; model->meshes = NULL;
	}
	

	// remove references from parent
	parent = model->parent;
	if (parent != NULL) {
		for (i=0 ; i<parent->tagNum ; i++) {
			if (parent->linkedModels[i] == model) {
				parent->linkedModels[i] = NULL;
			}
		}
		model->parent = NULL;
	}	
	
	
	if (model->linkedModels) {
		// remove references from children
		for (i=0 ; i<model->tagNum ; i++) {
			linkModel = model->linkedModels[i];
			if (linkModel != NULL) {
			  	//linkModel->parent = NULL;
				delete_gl_model( linkModel );
			}
		}
	}

	// remove the model from mdview.modelList
	if (model->modelListPosition) {
		mdview.modelList->remove( model->modelListPosition );
		if (model == mdview.baseModel) {
			mdview.baseModel = NULL;
		}
	}

	// remove all entries in the tag menu
	for (i=0 ; i<model->tagNum ; i++) {
		tagMenu_remove( (GLMODEL_DBLPTR)&model->linkedModels[i] );
	}

	delete model;
}

bool	loadmd3(gl_model& newModel, char* filename)
{
	FILE* F;
	if ( ( F = fopen( filename , "rb" ) ) != NULL ) 
	{
		if (loadmodel(newModel,F))
		{
			fclose(F);
			return true;
		} else
			return false;
	} else
		return false;
}


/*
writes out the mesh to a raw file
*/
void write_baseModelToRaw( char *fname )
{

	gl_model *model = mdview.baseModel;
	gl_mesh		*mesh;
	int i, mesh_num;
	Vec3 *vecs;
	TriVec	*tris;	
	int tri_num, t, v, n;

	
	// open up file for writing
	FILE *rawfile = fopen( fname, "wt" );
	if (!rawfile) { 
		Debug("Could not write to raw file %s", fname);
		return;
	}

	// write out header
	fprintf(rawfile, "Object1\n");		
	mesh_num = model->meshNum;		

	// for all the meshes in a model
	for (i=0 ; i<mesh_num; i++) 
	{		
		mesh	  = &model->meshes[i];				
	
		vecs = mesh->meshFrames[model->currentFrame];	
		tris	= mesh->triangles;		
		tri_num = mesh->triangleNum;				
				
		// write out all the triangles as 3 vertices
		for (t=0 ; t<tri_num ; t++) {
			for (v=0 ; v<3 ; v++) {
				for (n=0 ; n<3 ; n++) {
					fprintf(rawfile, "%f ", vecs[tris[t][v]][n] );
				}				
			}
			fprintf(rawfile, "\n");
		}
	}

	fclose(rawfile);
}


