/**
 * e_game.c
 */ 

#include "g_local.h"
#include "m_player.h"

props_t *gProperties = NULL;

/**
 * Called from InitGame at game DLL load time.  Set up cvars used by Expert, etc
 */
void ExpertGameInits() {

	// Expert: Custom cvars

	// options bitvectors
	sv_expflags = gi.cvar("expflags", "0", CVAR_SERVERINFO);
	sv_utilflags = gi.cvar("utilflags", "0", CVAR_SERVERINFO|CVAR_LATCH);
	sv_arenaflags = gi.cvar("arenaflags", "0", CVAR_SERVERINFO);
	
	// lethality: scales amount of damage everything in the game does
	sv_lethality = gi.cvar("lethality", "1", CVAR_SERVERINFO);
	
	// pace: scales the speeds of the player, projectiles, and other velocities
	sv_pace = gi.cvar("pace", "1", CVAR_SERVERINFO);

	// number of requested teams
	sv_numteams = gi.cvar("numteams", "0", CVAR_SERVERINFO);

	// gibstats
	sv_giblog = gi.cvar("giblog", EXPERT_GIB_FILENAME, CVAR_SERVERINFO);

	// obit settings
	sv_movethresh = gi.cvar("movethresh", EXPS_MOVE_THRESH_MIN, CVAR_SERVERINFO);
	sv_spree = gi.cvar("spree", EXPS_KILLING_SPREE, CVAR_SERVERINFO);
	sv_skunkval = gi.cvar("skunkval", EXPS_SKUNK_MIN, CVAR_SERVERINFO);
	sv_pbrange = gi.cvar("pbrange", EXPS_RANGE_POINTBLANK, CVAR_SERVERINFO);
	sv_longrange = gi.cvar("longrange", EXPS_RANGE_EXTREME, CVAR_SERVERINFO);

	// unsettable 
	version = gi.cvar("expert", EXPERT_VERSION, CVAR_SERVERINFO|CVAR_NOSET);
	gamedir = gi.cvar("game", "", CVAR_NOSET);
	
	// CTF
	ctf = gi.cvar("ctf", "0", CVAR_SERVERINFO);
	capturelimit = gi.cvar ("capturelimit", "7", CVAR_SERVERINFO);

	// FlagTrack
	flagtrack = gi.cvar("flagtrack", "0", CVAR_SERVERINFO);

	// global properties table
	gProperties = newProps();
}

/**
 * Called by SpawnEntities at the beginning of each level, after 
 * tagmalloc'd memory from the previous level has been freed.
 * SpawnEntities is called by the engine.
 */
void ExpertLevelInits() {

	// To allow players more air time after hook releases,
	// and allow longer range arcing projectiles.
	// FIXME : causes an engine crash
	if (expflags & EXPERT_WEAPONS) {
		//gi.cvar_set("sv_gravity", 650);
	}

	// Init CTF
	if (ctf->value) {
		CTFInit();
	}
	
	// Initialize FlagTrack
	if (flagtrack->value) {
		FlagTrackInit();
	}

	// Initialize Arena
	if (arenaflags & EXPERT_ENABLE_ARENA) {
		arenaInitLevel();
	}

	// Initialize the teamplay system
	if (teamplayEnabled()) {
		loadTeams();
	}

	// Initialize MOTD. (can't be called from InitGame because tags are freed above)
	if ( !(utilflags & EXPERT_DISABLE_MOTD) ) {
		InitMOTD();
	}

	// Initialize Obituary (can't be called from InitGame for same reason)
	if ( !(utilflags & EXPERT_DISABLE_CLIENT_OBITUARIES) ) {
		InitExpertObituary();
	} 

	// Start GibStat logging if applicable.
	if (utilflags & EXPERT_ENABLE_GIBSTAT_LOGGING)
	{
	/*
		gsStartLogging(sv_giblog->string);
		gsLogLevelStart();
		gsEnumConnectedClients();
	*/
	}

}

// Delayed init that occurs every level, 5 seconds after completed connection (ClientBegin)
void ExpertPlayerDelayedInits(edict_t *player) 
{
	// These stuffcmds are huge enough to cause unreliable dumps or ever overflows
	// if sent during connect.

	// Some likely attempts at getting help
	StuffCmd(player, "alias help \"cmd motd\";alias settings \"cmd settings\";\n");
	StuffCmd(player, "alias motd \"cmd motd\";alias serverhelp \"cmd motd\";\n");

	// SwitchFire aliases
	StuffCmd(player, "alias back \"-attack;cmd weaplast\";\n");
	StuffCmd(player, "alias +sg \"use shotgun;+attack\";alias -sg back;\n");
	StuffCmd(player, "alias +ssg \"use super shotgun;+attack\";alias -ssg back;\n");
	StuffCmd(player, "alias +mg \"use machinegun;+attack\";alias -mg back;\n");
	StuffCmd(player, "alias +cg \"use chaingun;+attack\";alias -cg back;\n");
	StuffCmd(player, "alias +gl \"use grenade launcher;+attack\";alias -gl back;\n");
	StuffCmd(player, "alias +rl \"use rocket launcher;+attack\";alias -rl back;\n");
	StuffCmd(player, "alias +hb \"use hyperblaster;+attack\";alias -hb back;\n");
	StuffCmd(player, "alias +rg \"use railgun;+attack\";alias -rg back;\n");
	StuffCmd(player, "alias +bfg \"use bfg10k;+attack\";alias -bfg back;\n");
	
	player->client->resp.delayedInit = true;
}

// InitCmds: Sends commands to the client every level load.
void InitCmds(edict_t *player)
{
	// Add aliases for the grappling hook and for pogo
	StuffCmd(player, "alias +hook +use; alias -hook -use;\n");
	StuffCmd(player, "alias +pogo +use; alias -pogo -use;\n");
}

// Called during ClientBegin, which is called for each player
// every level change and on initial connect.
void ExpertPlayerLevelInits(edict_t *player) {

	// Expert: Display MOTD
	if ( !(utilflags & EXPERT_DISABLE_MOTD) )
		DisplayMOTD(player);

	// Expert: Send any necessary commands to the client
	InitCmds(player);

	// Expert: If teamplay is enabled, assign the player to a team
	if (teamplayEnabled()) { 
		assignTeam(player);
	}

	//EXPERT: GibStats Logging
	//if ( utilflags & EXPERT_ENABLE_GIBSTAT_LOGGING)
	//	gsLogClientConnect(player);

}

// called on player disconnect
void ExpertPlayerDisconnect(edict_t *player) {

	//EXPERT: GibStats Logging
//	if (utilflags & EXPERT_ENABLE_GIBSTAT_LOGGING)
//		gsLogClientDisconnect(player);

	// Expert Arena: remove the player from the arena queue
	if (arenaflags & EXPERT_ENABLE_ARENA) {
		arenaDisconnect(player);
	}

	// Expert Matrix: Remove the player
	//ContractMatrix(player);

	// Expert CTF
	if (ctf->value)
		CTFDeadDropFlag(player);

	// Expert FlagTrack
	if (flagtrack->value) {
		tossFlag(player);
	}

	// Expert Teamplay
	if (teamplayEnabled()) {
		teamDisconnect(player);
	}

}

/**
 *  Inhibit additional entities in various modes.  Returning
 *  true indicates the entity should be freed.
 */
qboolean ExpertInhibit(edict_t *ent) {

	// Inhibit all weapons in free gear mode
	if ( expflags & EXPERT_FREE_GEAR &&
	    strstr(ent->classname, "weapon")) {
		return true;
	}

	// Inhibit critical all powerups in no powerups mode
	if (expflags & EXPERT_NO_POWERUPS && 
	    (strstr(ent->classname, "item_power") ||
	     strcmp(ent->classname, "item_quad") == 0 ||
	     strcmp(ent->classname, "item_invulnerability") == 0)) 
	{
		return true;
	}

	// All health and armor restoration removed in Alternate Restore
	if (expflags & EXPERT_ALTERNATE_RESTORE &&
	    (strstr(ent->classname, "item_armor") ||
		 strstr(ent->classname, "item_health") ||
	     strcmp(ent->classname, "item_ancient_head") == 0 ||
	     strcmp(ent->classname, "item_adrenaline") == 0)) 
	{
		return true;
	}

	// Inhibit plats in No Plats mode.  This is intended for games
	// with the hook or pogo, where plats aren't necessary for getting
	// around, and instead get in the way.  Also useful to alter level
	// dynamics such as in Capture Showdown (q2ctf5).  Should generally 
	// be enabled on a level-by-level basis. 
	// Note: we could remove doors, too, but some doors are areaportals.
	// We could leave the areaportals open all the time at a performance penalty.
	if (expflags & EXPERT_NO_PLATS &&
	    Q_stricmp(ent->classname, "func_plat") == 0)
	{
		return true;
	}

	// Other on-the-fly changes that are not inhibits

	// Under Expert weapons, transmogrify BFG into an SSG.  
	// BFG is ok for one on one, but too effective in larger games
	if (expflags & EXPERT_WEAPONS) {
		if (strcmp(ent->classname, "weapon_bfg") == 0) {
			ent->classname = "weapon_supershotgun";
		}
	}

	return false;

}

/**
 * Various changes to the itemlist. Must be done on the fly at
 * level load time in order to support the changes as optional.  We do
 * this once, immediately before "worldspawn" is spawned, since worldspawn
 * sets up configstrings using the itemlist.
 */
void ExpertItemListChanges() {

	// Expert: Switch pickup names when Expert powerups are used
	if (expflags & EXPERT_POWERUPS)
	{
		itemlist[ITEM_INDEX(FindItemByClassname("item_quad"))].pickup_name = "Vampirism";
		itemlist[ITEM_INDEX(FindItemByClassname("item_invulnerability"))].pickup_name = "Mutant Jump";
	}

	// Expert: Switch pickup names when balanced items is in effect
	if (expflags & EXPERT_BALANCED_ITEMS)
	{
		itemlist[ITEM_INDEX(FindItemByClassname("item_power_shield"))].pickup_name = "Blood Armor";
	}
	
	// Expert: Alternate armor stats
	if (expflags & EXPERT_BALANCED_ITEMS)
	{
		itemlist[ITEM_INDEX(FindItem("Body Armor"))].info = &balanced_bodyarmor_info;
		itemlist[ITEM_INDEX(FindItem("Combat Armor"))].info = &balanced_combatarmor_info;
		itemlist[ITEM_INDEX(FindItem("Jacket Armor"))].info = &balanced_jacketarmor_info;
	}

}

// ShardPoints: Returns how many armor points to add to a player's armor type.
int ShardPoints(int armor_index)
{
	gitem_armor_t *armor = (gitem_armor_t *) itemlist[armor_index].info;

	if (armor == NULL || !(itemlist[armor_index].flags & IT_ARMOR))
	{
		gi.error ("Bad armor given to ShardPoints\n");
		return 2;
	}

	if (expflags & EXPERT_BALANCED_ITEMS)
	{
		// God bless floating-point division rounding errors
		return (BALANCED_SHARD_POINTS / armor->normal_protection) + 0.6;
	}
	else
		return 2;
}

/**
 * Give a client the "free gear" set of weapons and ammo.
 * Returns the weapon to start with.
 */
gitem_t *giveFreeGear(gclient_t *client) {

	gitem_t		*item;

	// give weapons 
	item = FindItem("Shotgun");
	client->pers.inventory[ITEM_INDEX(item)] = 1;

	item = FindItem("Super Shotgun");
	client->pers.inventory[ITEM_INDEX(item)] = 1;

	item = FindItem("Machinegun");
	client->pers.inventory[ITEM_INDEX(item)] = 1;
	
	item = FindItem("Chaingun");
	client->pers.inventory[ITEM_INDEX(item)] = 1;
	
	item = FindItem("Grenade Launcher");
	client->pers.inventory[ITEM_INDEX(item)] = 1;
	
	item = FindItem("Rocket Launcher");
	client->pers.inventory[ITEM_INDEX(item)] = 1;
	
	item = FindItem("HyperBlaster");
	client->pers.inventory[ITEM_INDEX(item)] = 1;

	item = FindItem("Railgun");
	client->pers.inventory[ITEM_INDEX(item)] = 1;

	// give ammo
	item = FindItem("Shells");
	client->pers.inventory[ITEM_INDEX(item)] = 25;

	item = FindItem("Bullets");
	client->pers.inventory[ITEM_INDEX(item)] = 100;

	item = FindItem("Cells");
	client->pers.inventory[ITEM_INDEX(item)] = 40;

	item = FindItem("Grenades");
	client->pers.inventory[ITEM_INDEX(item)] = 5;

	item = FindItem("Slugs");
	// Expert : more ammo for balanced version of railgun
	if (expflags & EXPERT_WEAPONS) {
		client->pers.inventory[ITEM_INDEX(item)] = 15;
	} else {
		client->pers.inventory[ITEM_INDEX(item)] = 5;
	}
	
	item = FindItem("Rockets");
	client->pers.inventory[ITEM_INDEX(item)] = 10;

	// weapon to start with
	return FindItem("Rocket Launcher");
		
}

// In Alternate Restore mode, there is no health or armor on the map, so players regenerate.
// Players who are in combat or have recently been in combat regenerate faster.
void alternateRestoreRegen(edict_t *player) {

	int armorAmount, healthAmount;

	if (player->health >= 0 &&
	    level.time - player->client->lastRestore > EXPERT_REGEN_DELAY)
	{
		// get max armor from itemlist
		int maxArmor = ((gitem_armor_t *)itemlist[jacket_armor_index].info)->max_count;

		if (player->client->pers.inventory[jacket_armor_index] >= maxArmor &&
		    player->health >= player->max_health)
		{
			// doesn't need regen yet
			// note player will get regen'd immediately as soon as damaged
			return;
		}
	
		// regen faster if in combat or recently in combat
	    if (level.time - player->client->lastCombat < EXPERT_COMBAT_REGEN_TIMEOUT)
		{
			armorAmount = EXPERT_REGEN_COMBAT_ARMOR_AMOUNT;
			healthAmount = EXPERT_REGEN_COMBAT_HEALTH_AMOUNT;
		}
		else
		{
			armorAmount = EXPERT_REGEN_ARMOR_AMOUNT;
			healthAmount = EXPERT_REGEN_HEALTH_AMOUNT;
		}

		// give armor
		player->client->pers.inventory[jacket_armor_index] += armorAmount;
		if (player->client->pers.inventory[jacket_armor_index] >= maxArmor) 
		{
			player->client->pers.inventory[jacket_armor_index] = maxArmor;
		}

		// give health
		if (player->health < player->max_health) {
			player->health += healthAmount;
			if (player->health > player->max_health) {		
				player->health = player->max_health;
			}
			// play a regen sound if regaining health, only for the player being regen'd
			unicastSound(player, gi.soundindex ("items/n_health.wav"), 1);
		}
		player->client->lastRestore = level.time;
	}

}

// utility: return the index into the itemlist for the gitem_t of an armor 
// of type "armor". #defines of armor types are in g_local.h, and are
// stored in the gitem_armor_t->armor field
int indexForArmor(int armor) {

	switch (armor)
	{
		case ARMOR_JACKET: return jacket_armor_index; break;
		case ARMOR_COMBAT: return combat_armor_index; break;
		case ARMOR_BODY: return body_armor_index; break;
		default: return jacket_armor_index; break;
	}
}

// ---------- Modified pickup routines

void ExpertAddToDroppedWeapon(edict_t *drop, edict_t *self) {

	// If Expert Powerups AND balanced items, powerups are placed into the 
	// dropped weapon, and the weapon glows according to the powerups it holds
	// (balanced items needs to be set since only then is item dropping
	// guaranteed on every death)
	if (expflags & EXPERT_BALANCED_ITEMS &&
	    expflags & EXPERT_POWERUPS)
	{
		drop->included_invincibility = 0;
		drop->included_quad = 0;

		if (self->client->quad_framenum > level.framenum)
		{
			drop->included_quad = self->client->quad_framenum;
			drop->s.effects |= EF_QUAD;
		}

		if (self->client->invincible_framenum > level.framenum)
		{
			drop->included_invincibility = self->client->invincible_framenum;
			drop->s.effects |= EF_PENT;
		}
	}

	// Expert: Don't rotate dropped items to 
	// help distinguish them from normal items.
	if (expflags & EXPERT_NO_HACKS) {
		drop->s.effects &= ~EF_ROTATE;
	}
}

void ExpertPickupDroppedWeapon(edict_t *ent, edict_t *other) {

	// If balanced items, then give the player the health, 
	// armor and powerups stored in the dropped weapon
	if (expflags & EXPERT_BALANCED_ITEMS) {

		int armor_index;
		gitem_armor_t *armorinfo;

		// 30 restorative health in all dropped weapons
		if (other->health < other->max_health) {
			// In alternate restore, packs are the primary source of restorative health
			if (expflags & EXPERT_ALTERNATE_RESTORE) {
				other->health += 50;
			} else {
				other->health += 30;
			}
			if (other->health > other->max_health) {		
				other->health = other->max_health;
			}
		}

		// give the player the equivalent of an armor shard
		armor_index = ArmorIndex (other);
		if (!armor_index) {
			armor_index = jacket_armor_index;
		}
		armorinfo = (gitem_armor_t *) itemlist[armor_index].info;
		// in alternate restore mode, packs are the primary source of restorative health
		if (expflags & EXPERT_ALTERNATE_RESTORE) {
			other->client->pers.inventory[armor_index] += floor(0.5 * armorinfo->max_count);
		} else {
			// roughly 1/3 of a full armor
			other->client->pers.inventory[armor_index] += ShardPoints(armor_index);
		}

		if (other->client->pers.inventory[armor_index] > armorinfo->max_count) {
			other->client->pers.inventory[armor_index] = armorinfo->max_count;
		}

		// If both Balanced Items and Expert powerups,
		// powerups are included in the dropped weapon
		if (expflags & EXPERT_POWERUPS) {
			if (ent->included_quad > level.framenum)
			{
				if (other->client->quad_framenum < ent->included_quad)
				{
					gi.centerprintf(other, "You got the Vampire Artifact!\n\nYou receive as life\n"
						"points half the health\ndamage you do!\n");
					other->client->quad_framenum = ent->included_quad;
				}
				else // player had powerup already
					other->client->quad_framenum += ent->included_quad - level.framenum;
			}
		
			if (ent->included_invincibility > level.framenum)
			{
				if (other->client->invincible_framenum < ent->included_invincibility)
				{
					gi.centerprintf(other, "You got the Mutant Jump!\n\nNow you can jump like a Mutant!\n"
						"You are also invulnerable to slime,\nlava, and falling!");
					other->client->invincible_framenum = ent->included_invincibility;
				}
				else
					other->client->invincible_framenum += ent->included_invincibility - level.framenum;
			}
		}	
	}
}

// Ark of Life restores both full armor and full health.
// Only if you have both full armor and full health will you
// not pick up the Ark.
// FIXME avoid constants
qboolean canPickupArkOfLife(edict_t *player) {

	if (player->health >= player->max_health) 
	{
		if (player->client->pers.inventory[jacket_armor_index] >= 160 || 
		    player->client->pers.inventory[combat_armor_index] >= 96 ||
		    player->client->pers.inventory[body_armor_index] >= 80)
		{
			return false;
		}
	}
}

void pickupArkOfLife(edict_t *ark, edict_t *player) {

	// Code ripped from Pickup_Armor
	int				armor_index;
	gitem_armor_t	*armorinfo;

	// cap health.  The caller (Pickup_Health in g_items.c) has already
	// added the MegaHealth's 100 count to the player's health
	if (player->health > player->max_health) {
		player->health = player->max_health;
	}
		
	// get info on armor player is currently wearing
	armor_index = ArmorIndex (player);

	// if player has no armor, give him a new set of combat armor
	if (!armor_index)
	{
		armor_index = jacket_armor_index;
	}

	// restore player's armor to full
	armorinfo = itemlist[armor_index].info;
	player->client->pers.inventory[armor_index] = armorinfo->base_count;

	gi.centerprintf(player, "You got the Ark of Life!\n\nYour health and armor are\n"
				"fully replenished!\n");

	// respawn like a normal health pack
	SetRespawn (ark, 30);
}

// ---------- Dropped powerup management

void ItemEffects(edict_t *dropped)
{
//	gi.bprintf(PRINT_HIGH, "Called ItemEffects\n");
	
	// If either powerup has expired, remove it's effect
	if (dropped->s.effects & EF_QUAD && dropped->included_quad <= level.framenum)
	{
//		gi.bprintf(PRINT_HIGH, "Turning off quad..\n");
		dropped->s.effects &= ~EF_QUAD;
	}

	if (dropped->s.effects & EF_PENT && dropped->included_invincibility <= level.framenum)
	{
//		gi.bprintf(PRINT_HIGH, "Turning off wings..\n");
		dropped->s.effects &= ~EF_PENT;
	}

	// If either powerup remains, set a timer for when the earliest
	// powerup will expire.
	if (dropped->s.effects & EF_QUAD || dropped->s.effects & EF_PENT) 
	{
		if (dropped->s.effects & EF_QUAD) {
//			gi.bprintf(PRINT_HIGH, "Setting nextthink to remove quad effect...\n");
			// quad effect should be removed when the player can no
			// longer get quad from the weapon.  One second buffer
			// to avoid a bunch of extra thinks
			dropped->nextthink = dropped->included_quad/10 + 1;
		}
		if (dropped->s.effects & EF_PENT)
		{
			float pentTimeout;

//			gi.bprintf(PRINT_HIGH, "Setting nextthink to remove pent effect..\n");
	
			pentTimeout = dropped->included_invincibility/10 + 1;
			// pent will expire before quad
			if (pentTimeout < dropped->nextthink) {
				dropped->nextthink = pentTimeout;
			}
		}
	}
	else
	{
//		gi.bprintf(PRINT_HIGH, "Setting nextthink to free item..\n");
		// note: this code becomes wrong if powerups last more than 30 seconds
		dropped->nextthink = dropped->drop_time + BALANCED_DROPPED_ITEM_TIME;
		dropped->think = G_FreeEdict;
	}
}


void wave(edict_t *ent, int waveNum) {

	// can't wave when ducked
	if (ent->client->ps.pmove.pm_flags & PMF_DUCKED)
		return;

	if (ent->client->anim_priority > ANIM_WAVE)
		return;

	ent->client->anim_priority = ANIM_WAVE;

	switch (waveNum)
	{
	case 0:
		gi.cprintf (ent, PRINT_HIGH, "flipoff\n");
		ent->s.frame = FRAME_flip01-1;
		ent->client->anim_end = FRAME_flip12;
		break;
	case 1:
		gi.cprintf (ent, PRINT_HIGH, "salute\n");
		ent->s.frame = FRAME_salute01-1;
		ent->client->anim_end = FRAME_salute11;
		break;
	case 2:
		gi.cprintf (ent, PRINT_HIGH, "taunt\n");
		ent->s.frame = FRAME_taunt01-1;
		ent->client->anim_end = FRAME_taunt17;
		break;
	case 3:
		gi.cprintf (ent, PRINT_HIGH, "wave\n");
		ent->s.frame = FRAME_wave01-1;
		ent->client->anim_end = FRAME_wave11;
		break;
	case 4:
	default:
		gi.cprintf (ent, PRINT_HIGH, "point\n");
		ent->s.frame = FRAME_point01-1;
		ent->client->anim_end = FRAME_point12;
		break;
	}
}

