// e_util.c
//
// Misc. functions dependant only on the original game src.

#include "g_local.h"

#define LOG_FILENAME				"debuglog.txt"

FILE *logfile = NULL;

// numchr: Find out how many of a given character are in a given string.
// Useful for finding out how many returns to pad for centerprinting..
int numchr(char *str, int c)
{
	int number = 0;
	char *left = str;

	if (str == NULL)
		return 0;
	
	while ((left = strchr(left, c)) != NULL)
	{
		number++;
		left++;
	}

	return number;
}

// StuffCmd: Sends arbitrary console commands to a client.
// Mucho thanks to James Abbatiello for this code!
void StuffCmd(edict_t *player, char *cmd)
{
	gi.WriteByte(11);
	gi.WriteString(cmd);
	gi.unicast (player, true);
}

// BootPlayer: Boots the given player with the given (1st) error message.
// 2nd message is for global messages.
void BootPlayer(edict_t *player, char *error, char *global)
{
	char *buf;

	// Create command buffer and stuff it
	buf = malloc(strlen(error) + 10);
	buf = strcpy(buf, "error \"");
	buf = strcat(buf, error);
	buf = strcat(buf, "\"\n");

	StuffCmd(player, buf);
	free(buf);

	// Make an announcement
	gi.bprintf(PRINT_HIGH, "%s kicked: %s\n", player->client->pers.netname, global);

	// Kick the player for good measure
	buf = malloc(7 + strlen(player->client->pers.netname));
	buf = strcpy(buf, "kick ");
	buf = strcat(buf, player->client->pers.netname);
	buf = strcat(buf, "\n");
	gi.AddCommandString(buf);
	free(buf);
}

void E_LogAppend(char *desc, char *fmt, ...)
{
	va_list argptr;

	if (logfile == NULL)
	{
		logfile = OpenGamedirFile(gamedir->string, LOG_FILENAME, "a");

		if (logfile == NULL)
		{
			gi.dprintf("ERROR: Unable to open log file. Log message cancelled.\n");
			return;
		}

//		char filename[256];
//		strcpy(filename, gamedir->string);
//		strcat(filename, "/");
//		strcat(filename, LOG_FILENAME);
//		logfile = fopen(filename, "a");
	}

	fprintf(logfile, "Log message at time %.2f, frame %d. Description: %s\n"
					 "Message:\n", level.time, level.framenum, desc);

	va_start(argptr, fmt);
	vfprintf(logfile, fmt, argptr);
	va_end(argptr);

	fprintf(logfile, "\n----- End of message (line preceeded by \\n -----\n\n");
}

void E_LogClose(void)
{
	if (logfile != NULL)
	{
		fprintf(logfile, "Closing logfile at time %.2f seconds elapsed in game\n", level.time);
		fclose(logfile);
	}
	else
		gi.dprintf("ERROR in E_LogClose: Attempted to close an already-closed logfile\n");
}

qboolean nearToGround(edict_t *ent)
{
	vec3_t		startPos, traceTo, dist;
	trace_t		trace;

	VectorCopy(ent->s.origin, startPos);

	dist[0] = 0;
	dist[1] = 0;
	dist[2] = EXP_AIRBORNE_THRESH;

	VectorAdd(startPos, dist, traceTo);
	trace = gi.trace(ent->s.origin, ent->mins, ent->maxs, traceTo, ent, MASK_PLAYERSOLID|MASK_WATER);

	if (trace.fraction == 1.0)
		return (false);

	return (true);
}

void trimWhitespace(char *szString)
{
	char *temp;
	int	 len = 0;
	int  pos = 0;
	int  i = 0;
	int  curChar = 0;

	if (szString == NULL)
		return;

	len = strlen(szString);

	if (len == 0)
		return;

	temp = calloc(len + 1, sizeof(char));
	strcpy(temp, szString);

	// Start from the tail of the string and replace spaces with null.
	for (i = (len - 1); i > 0 ; i--)
	{
		curChar = *(temp + i);
		if ((curChar == ' ') ||		// space
		    (curChar == '\t') ||	// tab
		    (curChar == '\n') ||	// CR
		    (curChar == '\r'))		// LF
			temp[i] = 0;		// Change to null
		else
			break;
	}

	// Start from the front of the string and move the pointer up one.
	for (i = 0 ; i < len ; i++)
	{
		curChar = *(temp + i);
		if ((curChar == ' ') ||		
		    (curChar == '\t') ||
		    (curChar == '\n') ||
		    (curChar == '\r'))
			pos++;
		else
			break;
	}

	// Copy the string and exit
	strcpy(szString, temp + pos);
	free(temp);

}

float playerDistance(edict_t *plyr1, edict_t *plyr2)
{
	vec3_t	v;
	float	distance = 0;

	VectorSubtract (plyr1->s.origin, plyr2->s.origin, v);
	
	// fabs function is HORRIBLY slow...

	distance = VectorLength(v);

	if (distance >= 0)
		return (distance);

	return -(distance);
}

/**
 * Takes a string and bitshifts all characters that Quake2 can
 * render as green so that the string will render as green when
 * printed.
 *
 * For greenCopy(), szDest must be at least as large as szSrc.
 * For convenience szDest is returned.
 */
char *greenText(char *s)
{
	return greenCopy(s, s);
}

char *greenCopy(char *szDest, const char *szSrc)
{

	int curChar = 0;
	int len = 0;
	int i = 0;

	len = strlen(szSrc);

	if (len == 0)
		return NULL;

	for (i = 0 ; i < len ; i++)
	{
		curChar = *(szSrc + i);
		if (curChar > 32 && curChar < 128) {
			szDest[i] = curChar | 128;
		} else {
			szDest[i] = curChar;
		}
	}
	szDest[i] = 0;

	return szDest;
}

int strBeginsWith (char *prefix, char *s2)
{
	int 	c1, c2, max = 999999;
	
	do
	{
		c1 = *prefix++;
		c2 = *s2++;
		if (!c1) return(1); // Reached end of search string
		if (!max--)
			return 1;		// strings are equal until end point
		
		if (c1 != c2)
		{
			if (c1 >= 'a' && c1 <= 'z')
				c1 -= ('a' - 'A');
			if (c2 >= 'a' && c2 <= 'z')
				c2 -= ('a' - 'A');
			if (c1 != c2)
				return 0;		// strings not equal
		}
	} while (c1);
	
	return 1;		// strings are equal
}

int insertValue(char *mess, char *format_str, char *field, char *value, int max_size)
{
	int x=0, y=0, matches = 0;
	mess[0] = 0;
	while (format_str[x] && (x < max_size))
	{
		if (strBeginsWith(field, format_str + x))
		{
			if (x + strlen(value) < max_size)
			{
				//PRINT2("Found match on '%s' for field '%s\n", format_str, field);
				matches++;
				strcpy(mess+y, value);
				y = y + strlen(value);
				x = x + strlen(field);
			}
			else {
				// Buffer overflow would occur.
				return(-1);
			}
		}
		else {
			mess[y++] = format_str[x++];
		}
	}
	if (y < max_size)
		mess[y] = 0;
	else mess[max_size - 1] = 0;
	return(matches);
}

int StrToInt(char *pszFrom, int Default)
{
	char *psz = pszFrom; //for incrementing through the string
	signed char sign = 1;
	int value = 0;

	while(*psz == ' ' || *psz == '\t' || *psz == '\n')
		psz++;

	if (*psz == '+')
		psz++;
	else if (*psz == '-') {
		psz++;
		sign = -1;
	}

	if (*psz < '0' || *psz > '9')
		return Default;

	while (*psz >= '0' && *psz <= '9') {
		value = (10 * value) + *psz - '0';
		psz++;
	}

	return (value * sign);
}

void ResizeLevelMemory(void** ppMem, size_t sizeNew, size_t sizeOld)
{
	byte* pOld = (byte *)*ppMem;
	byte* pNew = gi.TagMalloc(sizeNew, TAG_LEVEL);

	assert(ppMem != NULL && sizeNew > 0 && sizeOld > 0);
	if (sizeNew == sizeOld) {
		return;
	}

	memcpy(pNew, pOld, sizeOld);

	#ifdef _DEBUG
	{
		if (sizeNew > sizeOld) {
			memset(pNew+sizeNew, 0xCC, sizeNew - sizeOld);
		} else if (sizeOld > sizeNew) {
			memset(pOld+sizeNew, 0xCC, sizeOld - sizeNew);
		}
	}
	#endif

	gi.TagFree(pOld);
	*ppMem = (void *)pNew;
}

// OpenGamedirFile: Opens a filename from a given basedir.
// Returns a file handle if successful, NULL otherwise.
// NOTE: No checking is done on the mode.
FILE *OpenGamedirFile(const char *basedir, const char *filename, char *mode)
{
	char tempname[MAX_QPATH] = {0};
	FILE *file;

	strcpy(tempname, basedir);
	strcat(tempname, "/");
	strcat(tempname, filename);

	file = fopen(tempname, mode);

	if (file == NULL)
	{
		gi.dprintf("Unable to load file %s\n",
			tempname);
	}

	return file;
}
