// e_util.c
//
// Misc. functions

#include "g_local.h"

FILE *logfile = NULL;

// numchr: Find out how many of a given character are in a given string.
// Useful for finding out how many returns to pad for centerprinting..
int numchr(char *str, int c)
{
	int number = 0;
	char *left = str;

	if (str == NULL)
		return 0;
	
	while ((left = strchr(left, c)) != NULL)
	{
		number++;
		left++;
	}

	return number;
}

// StuffCmd: Sends arbitrary console commands to a client.
// Mucho thanks to James Abbatiello for this code!
void StuffCmd(edict_t *player, char *cmd)
{
	gi.WriteByte(11);
	gi.WriteString(cmd);
	gi.unicast (player, true);
}

// InitCmds: Sends commands to the client every level load.
void InitCmds(edict_t *player)
{
	// Add aliases for the grappling hook
	StuffCmd(player, "alias +hook +use; alias -hook -use;\n");
	// Some likely attempts at getting help
	StuffCmd(player, "alias help \"cmd motd\";alias settings \"cmd settings\";\n");
	StuffCmd(player, "alias motd \"cmd motd\";alias serverhelp \"cmd motd\";\n");

	// Expert: Make sure timescale is a userinfo var so we can look at it
//	StuffCmd(player, "set timescale 1 u;\n");
//	StuffCmd(player, "set timescale 1 u;\n");
}

void E_ClearItemEffect(edict_t *dropped)
{
//	gi.dprintf("Called E_ClearItemEffect\n");
	
	if (dropped->s.effects & EF_QUAD && dropped->included_quad <= level.framenum)
	{
//		gi.dprintf("Turning off quad..\n");
		dropped->s.effects &= ~EF_QUAD;
	}

	if (dropped->s.effects & EF_PENT && dropped->included_invincibility <= level.framenum)
	{
//		gi.dprintf("Turning off wings..\n");
		dropped->s.effects &= ~EF_PENT;
	}

	if (dropped->s.effects & EF_QUAD)
	{
//		gi.dprintf("Setting nextthink to add included_quad...\n");
		dropped->nextthink = dropped->included_quad/10;
	}
	else if (dropped->s.effects & EF_PENT)
	{
//		gi.dprintf("Setting nextthink to add included pent..\n");
		dropped->nextthink = dropped->included_invincibility/10;
	}
	else
	{
//		gi.dprintf("Setting nextthink to free item..\n");
		dropped->nextthink = dropped->drop_time + BALANCED_DROPPED_ITEM_TIME;
		dropped->think = G_FreeEdict;
	}
}

// Clear Quad/Pent effect on items dropped by players,
// then set the time when they'll disappear
void E_ClearQuadEffect(edict_t *dropped)
{
	dropped->s.effects = dropped->s.effects - (dropped->s.effects | EF_QUAD);
	dropped->nextthink = dropped->drop_time + BALANCED_DROPPED_ITEM_TIME;
	dropped->think = G_FreeEdict;
}

void E_ClearPentEffect(edict_t *dropped)
{
	dropped->s.effects = dropped->s.effects - (dropped->s.effects | EF_PENT);
	dropped->nextthink = dropped->drop_time + BALANCED_DROPPED_ITEM_TIME;
	dropped->think = G_FreeEdict;
}

// BootPlayer: Boots the given player with the given (1st) error message.
// 2nd message is for global messages.
void BootPlayer(edict_t *player, char *error, char *global)
{
	char *buf;

	// Create command buffer and stuff it
	buf = malloc(strlen(error) + 10);
	buf = strcpy(buf, "error \"");
	buf = strcat(buf, error);
	buf = strcat(buf, "\"\n");

	StuffCmd(player, buf);
	free(buf);

	// Make an announcement
	gi.bprintf(PRINT_HIGH, "%s kicked: %s\n", player->client->pers.netname, global);

	// Kick the player for good measure
	buf = malloc(7 + strlen(player->client->pers.netname));
	buf = strcpy(buf, "kick ");
	buf = strcat(buf, player->client->pers.netname);
	buf = strcat(buf, "\n");
	gi.AddCommandString(buf);
	free(buf);
}

// ShardPoints: Returns how many armor points to add to a player's armor type.
int ShardPoints(int armor_index)
{
	gitem_armor_t *armor = (gitem_armor_t *) itemlist[armor_index].info;

	if (armor == NULL || !(itemlist[armor_index].flags & IT_ARMOR))
	{
		gi.error ("Bad armor given to ShardPoints\n");
		return 2;
	}

	if ((int)sv_expflags->value & EXPERT_BALANCED_ITEMS)
	{
		// God bless floating-point division rounding errors
		return (BALANCED_SHARD_POINTS / armor->normal_protection) + 0.6;
	}
	else
		return 2;
}

void E_LogAppend(char *desc, char *fmt, ...)
{
	va_list argptr;

	if (logfile == NULL)
	{
		char filename[256];
		strcpy(filename, gamedir->string);
		strcat(filename, "/");
		strcat(filename, EXPERT_LOG_FILENAME);
		logfile = fopen(filename, "a");
	}

	fprintf(logfile, "Log message at time %.2f, frame %d. Description: %s\n"
					 "Message:\n", level.time, level.framenum, desc);

	va_start(argptr, fmt);
	vfprintf(logfile, fmt, argptr);
	va_end(argptr);

	fprintf(logfile, "\n----- End of message (line preceeded by \\n -----\n\n");
}

void E_LogClose(void)
{
	if (logfile != NULL)
	{
		fprintf(logfile, "Closing logfile at time %.2f seconds elapsed in game\n", level.time);
		fclose(logfile);
	}
	else
		gi.dprintf("ERROR in E_LogClose: Attempted to close an already-closed logfile\n");
}

qboolean NearToGround(edict_t *ent)
{
	vec3_t		startPos, traceTo, dist;
	trace_t		trace;

	VectorCopy(ent->s.origin, startPos);

	dist[0] = 0;
	dist[1] = 0;
	dist[2] = EXP_AIRBORNE_THRESH;

	VectorAdd(startPos, dist, traceTo);
	trace = gi.trace(ent->s.origin, ent->mins, ent->maxs, traceTo, ent, MASK_PLAYERSOLID|MASK_WATER);

	if (trace.fraction == 1.0)
		return (false);

	return (true);
}

void TrimSpaces(char *szString)
{
	char *temp;
	int	 len = 0;
	int  pos = 0;
	int  i = 0;
	int  curChar = 0;

	len = strlen(szString);

	if (len == 0)
		return;

	temp = calloc(len + 1, sizeof(char));
	strcpy(temp, szString);

	// Start from the tail of the string and replace spaces with null.
	for (i = (len - 1) ; i > 0 ; i--)
	{
		curChar = *(temp + i);
		if (curChar == 32)			// Check for space
			temp[i] = 0;			// Change to null
		else
			i = -1;
	}

	// Start from the front of the string and move the pointer up one.
	for (i = 0 ; i < len ; i++)
	{
		curChar = *(temp + i);
		if (curChar == 32)
			pos++;
		else
			i = len;
	}

	// Copy the string and exit
	strcpy(szString, temp + pos);
	free(temp);

}
