/*
	e_obit.c

	Implimentation of Expert Obituary Functions ansilary functions. e.g. Init, Parsing

	Credit is due to "SteQve" for ServObits 1.1 
		Some of the functionality within this module was modeled after the 
		code in ServObits.c code. Originally, I was going to use most of 
		ServObits as it stood for the Expert code. Unfortunately (for me), 
		the structure just wouldn't fit quite right into the Expert mod due 
		to the complexities of the "Context" system that was devised. Some of 
		the "helper" functions were slightly modified and included.
			String replacement functions.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "g_local.h"

//
// Main entry point for Expert Obituaries
//
void ExpertClientObituary (edict_t *self, edict_t *inflictor, edict_t *attacker)
{

	char			*aName = NULL;
	char			*vName = NULL;

	char			*szCause = NULL;
	unsigned long	cOD = 0;			// Cause Of Death, 0 is index of unknown.
	unsigned long	context = 0;		// Contexts death. 0 is nothing special.
	int				aGender = 0;
	int				vGender = 0;

	// The only thing we need to determine now is what worldspawn, if any, killed 
	// the player. The T_Damage routine helpingly did some classname stuff for us.

	// First take care of the scoring.
	if (attacker == self)
	{
		self->client->resp.score--;
		self->enemy = NULL;
	} else {
		self->enemy = attacker;
		if (attacker && attacker->client)
			attacker->client->resp.score++;
		else
		self->client->resp.score--;
	}
	
	if (Q_stricmp(self->client->killedBy, OBIT_CONSOLE_KILL) == 0)
	{	
		ConsoleKill(self);
		if ( (int)sv_utilflags->value & EXPERT_ENABLE_GIBSTAT_LOGGING)
			gsLogKillSelf(self, OBIT_CONSOLE_KILL);
		return;
	}

	if (Q_stricmp(attacker->classname, "worldspawn") == 0)
	{	// Victim did something stupid.
		// Check to see if victim was caught by an entity.
		if (Q_stricmp(inflictor->classname, "worldspawn") == 0)
		{	// If the inflictor->classname is Worldspawn, we know victim jumped
			// or swam in something not nice.
			if (self->watertype & CONTENTS_LAVA)
			szCause = "lava";
			else if (self->watertype & CONTENTS_SLIME)
			szCause = "slime";
			else if (self->watertype & CONTENTS_WATER)
			szCause = "water";
			else if (self->groundentity && (self->client->fall_time > level.time) )
			szCause = "fall";
		} else {
			szCause = inflictor->classname;
		}
	} else {
		szCause = self->client->killedBy;
	}

	cOD = FindCause(szCause, obitCauseMap, obits->countCause);
	context = DiscoverContexts(self, inflictor, attacker);

	if ( (int)sv_utilflags->value & EXPERT_ENABLE_GIBSTAT_LOGGING)
	{
		if ( (context & CON_KILL_SELF) )
			gsLogKillSelf(self, szCause);
		else
			gsLogFrag(self, attacker);
	}


	// Get genders and names.
	aGender = GetEntityGender(attacker);
	vGender = GetEntityGender(self);

	// Print the best client message possible.
	if (attacker->client == NULL)
	{
		DisplayBestObituaryMessage(cOD, context, aGender, vGender, 
									NULL, self->client->pers.netname);
	} else {
		DisplayBestObituaryMessage(cOD, context, aGender, vGender, 
									attacker->client->pers.netname, 
									self->client->pers.netname);
	}

}

void ConsoleKill (edict_t *self)
{
	int				vGender = 0;
	unsigned long	context;

	context = CON_SUICIDE;

	// Get genders and names.
	vGender = GetEntityGender(self);

	// The CAUSE is always going to be 0 (*) for a console kill.
	// Print the best client message possible.
	DisplayBestObituaryMessage(0, context, vGender, vGender, 
								self->client->pers.netname, 
								self->client->pers.netname);
}

void DisplayBestObituaryMessage(unsigned long cause, unsigned long context,
								int aGender, int vGender,
								char *aName, char *vName)
{
	obits_t 		*ob = NULL; 	//Store the returned obituary

	unsigned long	i;
	unsigned long	mask = 0;

	// Use the exact match context search.
	ob = FindContext(cause, context, obits);

	if (ob != NULL) 			// Exact match was found
	{
		PrintRandObitMsg(aName, vName, aGender, vGender, ob);
		return;
	}

	// Ok, that didn't work. Let's try masking from the MSB, the positions of the
	// most significant context.
	for (i = 0 ; i < (sizeof(unsigned long) * 8) ; i++)
	{
		mask = ((unsigned long)1 << 31) >> i;		// Set bit mask
		if ( ((mask & context) != context ) &&
			((mask & context) != 0 ))	// The mask has an effect.
		{
			ob = FindContext(cause, (mask & context), obits);
			if (ob != NULL) 	// Exact match was found this time.
			{
				PrintRandObitMsg(aName, vName, aGender, vGender, ob);
				return;
			}
		}
	}

	// Ok, This sucks... Nothing was found. Rather than tie up the servers
	// processor searching for random combinations of masks, let's try the "normal"
	// message under the weapon and then the "normal" message under "*"

	ob = FindContext(cause, 0, obits);
	if (ob != NULL)
	{
		PrintRandObitMsg(aName, vName, aGender, vGender, ob);
		return;
	}

	ob = FindContext(0, 0, obits);
	PrintRandObitMsg(aName, vName, aGender, vGender, ob);

}

void PrintRandObitMsg(char *aName, char *vName, 
						int aGender, int vGender, obits_t *ob)
{
	int iRnd = 0;
	char *szObit;
	char *szTemp;
	char *atkName;
	char *vicName;

	int vicSeparator = 156;
	char *pronouns[3][2] =	{	{OBIT_HE, OBIT_SHE},
								{OBIT_HIM, OBIT_HER},
								{OBIT_HIS, OBIT_HER}
							};

	if (ob == NULL)
	{	// No message passed in.
		gi.bprintf (PRINT_MEDIUM,"%s died and nobody wrote an obituary.\n", vName);
		return;
	}

	szTemp = calloc(sizeof(char), 512);
	szObit = calloc(sizeof(char), 512);

	atkName= calloc(sizeof(char), 100);
	vicName= calloc(sizeof(char), 100);
	// Make the attacker's name green if name ! null
	if (aName != NULL)
		GreenText(atkName, aName);

	// Place dot separators around the victims name
	//strcpy(vicName, &(char)vicSeparator);
	//strcat(vicName, vName);
	//strcat(vicName, &(char)vicSeparator);

	strcpy(vicName, vName);

	if (ob->msgCount > 1 )
	{
		srand(time(NULL));		// Make sure it's random.
		iRnd = rand() % ob->msgCount;
		strcpy(szObit, ob->messages[iRnd]);
	} else {
		strcpy(szObit, ob->messages[0]);
	}

	// Ok... We'll substitute the proper strings for the genders...
	InsertValue(szTemp, szObit, OTOK_ATTACKER_NAME, atkName, 512);
	strcpy(szObit, szTemp);
	InsertValue(szTemp, szObit, OTOK_ATTACKER_HE, pronouns[0][aGender], 512);
	strcpy(szObit, szTemp);
	InsertValue(szTemp, szObit, OTOK_ATTACKER_HIM, pronouns[1][aGender], 512);
	strcpy(szObit, szTemp);
	InsertValue(szTemp, szObit, OTOK_ATTACKER_HIS, pronouns[2][aGender], 512);
	strcpy(szObit, szTemp);
	InsertValue(szTemp, szObit, OTOK_VICTIM_NAME, vicName, 512);
	strcpy(szObit, szTemp);
	InsertValue(szTemp, szObit, OTOK_VICTIM_HE, pronouns[0][vGender], 512);
	strcpy(szObit, szTemp);
	InsertValue(szTemp, szObit, OTOK_VICTIM_HIM, pronouns[1][vGender], 512);
	strcpy(szObit, szTemp);
	InsertValue(szTemp, szObit, OTOK_VICTIM_HIS, pronouns[2][vGender], 512);
	strcpy(szObit, szTemp);

	gi.bprintf (PRINT_MEDIUM,"%s\n",szObit);
	free(szTemp);
	free(szObit);
}

void DisplayObituaryInfo(edict_t *ent)
{

	if (obits == NULL)
	{
		if ( (int)sv_utilflags->value & EXPERT_DISABLE_CLIENT_OBITUARIES )
			gi.cprintf (ent, PRINT_MEDIUM, "Expert Client Obituary Disabled.\n");
		else
			gi.cprintf (ent, PRINT_HIGH, "Expert Client Obituary failed to load.\n");
	} else {
		gi.cprintf (ent, PRINT_MEDIUM, "\nExpert Client Obituary Memory Stats\n");
		gi.cprintf (ent, PRINT_MEDIUM, "-----------------------------------\n");
		gi.cprintf (ent, PRINT_MEDIUM, "Total Number of Causes:   %i\n", 
							obits->countCause);
		gi.cprintf (ent, PRINT_MEDIUM, "Total Number of Contexts: %i\n", 
							obits->countContext);
		gi.cprintf (ent, PRINT_MEDIUM, "Total Number of Messages: %i\n",
							obits->countMsgs);
		gi.cprintf (ent, PRINT_MEDIUM, "Memory Used in bytes:     %i\n", 
							obits->memAllocated);
		gi.cprintf (ent, PRINT_MEDIUM, "\n");
	}
}

// This routine interrogates the victim, inflictor and attacker for context flags
// and returns any found. The routine also tries to do the minimum work possible;
// i.e. it won't run through attacker contexts if it's a self kill.
unsigned long DiscoverContexts(edict_t *vict, edict_t *infl, edict_t *attk)
{
	unsigned long c = 0;
	int			  gibVal = -40;

	// First, compute the health at which gibbing happens
	gibVal *= sv_lethality->value;

	// Check the Gender. We'll assume that most genders are male.
	if (GetEntityGender(vict) == OBIT_GENDER_FEMALE)
		c |= CON_VICTIM_FEMALE;

	// Check if victim is airborne.
	if ( !NearToGround(vict) )
		c |= CON_VICTIM_AIRBORNE;

	// Does the victim have the Quad Damage item?
	if (vict->client->quad_framenum > level.framenum)
		c |= CON_VICTIM_QUAD;

	// Does the victim have the Invunlerability item?
	if (vict->client->invincible_framenum > level.framenum)
		c |= CON_VICTIM_INVULN;

	// Is the victim's ping HORRIBLE?
	if (vict->client->ping >= EXP_PING_MERCY_MIN)
		c |= CON_MERCY_KILL;

	if (vict->health <= -40) // id Hardcoded it as well...
		c |= CON_GIBBED;

	// FIXME: Make this code real
	/*
	if ( (victim hasn't moved in a long time) &&
			( (int)sv_utilflags->value & EXPERT_MOVE_CHECKING) )
		c |= CON_VICTIM_MV_THRESH;
	*/

	// Now, so this doesn't take forever... Let's see if the victim and attacker
	// are one in the same. If they are, return current contexts and/or kill self

	if ( (vict == attk) || (Q_stricmp(attk->classname, "player")) )
		return (c | CON_KILL_SELF);

	// We've now excluded that the attacker and defender were one in the same.
	// What's left is another player and "Worldspawn". Split the decision here.

	if (attk->client == NULL)
	{	// Player did something stupid like jumping off a building, swimming in
		// lava or standing under a platform for a long time.
		// Checking for this is all taken care of in the ExpertClientObituary
		// function, we just need to return the current contexts.
		return (c);
	}

	// Ok... Attacker stuff here. Some of this will be inflictor checks followed
	// by attacker checks. Don't worry, it's all related to the attacker; some of
	// the information was cached at the time of the projectile spawning.

	// Get attacker gender.
	if (GetEntityGender(attk) == OBIT_GENDER_FEMALE)
		c |= CON_ATTACKER_FEMALE;

	// Check if attacker was airborne at the time of shooting
	if (Q_stricmp(infl->classname, "player"))
	{	// It's an instant effect weapon.
		if ( !NearToGround(attk) )
			c |= CON_ATTACKER_AIRBORNE;
	}else{
		if (infl->firedFromAir)
			c |= CON_ATTACKER_AIRBORNE;
	}

	// Does the attacker have the Quad Damage item?
	if (attk->client->quad_framenum > level.framenum)
		c |= CON_ATTACKER_QUAD;

	// Does the attacker have the Invunlerability item?
	if (attk->client->invincible_framenum > level.framenum)
		c |= CON_ATTACKER_INVULN;

	// FIXME: Make this code real
	/*
	if ( (attacker hasn't moved in a long time) &&
			( (int)sv_utilflags->value & EXPERT_MOVE_CHECKING) )
		c |= CON_ATTACKER_MV_THRESH;
	*/

	return (c);
}

//
// Obituary Helper Functions
//

obits_t* FindContext(unsigned long cause, unsigned long cFlag, obitInfo_t *obit)
{
	unsigned long i;
	obits_t** searchIn;

	// Set the search pointer. Makes this more readable.
	if (obits->causes[cause] == NULL)
		return NULL;

	searchIn = obits->causes[cause]->obituary;

	if (searchIn == NULL)	// No entries as of yet...
		return NULL;

	for (i = 0 ; i < obits->causes[cause]->entryCount ; i++)
	{
		if (searchIn[i]->cFlags == cFlag)
			return (searchIn[i]);
	}

	return (NULL);
}

unsigned long FindCause(char *szCause, char **szMap, unsigned long cCount)
{
	unsigned long i;

	if (szCause == NULL)
		return (0);

	for (i = (cCount - 1); i != 0 ; i--)
	{
		if (Q_stricmp(szCause, szMap[i]) == 0)
		return (i);
	}

	return (0);

}

//
// OBITUARY INITIALIZATION CODE
//

void InitExpertObituary(void)
{
	cFlagsMap_t **cFlagsMap ; // ** because it's going to be a dynamic array.
	unsigned long mallocSize;

	// "obits" is declared in e_defs.h
	// allocate memory for obituary information structure.
	mallocSize = sizeof(obitInfo_t);
	obits = gi.TagMalloc(mallocSize, TAG_LEVEL);
	// Let the info structure know...
	obits->memAllocated = mallocSize;
	obits->countMsgs = 0;

	gi.dprintf(OBIT_SEP);

	// Load the Cause Map.
	obitCauseMap = LoadCauseMap(E_OBIT_CAUSEFILE, obits);

	if (obitCauseMap == NULL)
	{
		gi.dprintf(ERR_OBIT_SEP);
		gi.dprintf(ERR_OBIT_NOCAUSEMAP);
		gi.dprintf(ERR_OBIT_SEP);
		obits = NULL;
		return;
	}

	// Load the context translator.
	cFlagsMap = LoadContextTranslator(E_OBIT_CONTEXTFILE, obits);
	if (cFlagsMap == NULL)
	{
		gi.dprintf(ERR_OBIT_SEP);
		gi.dprintf(ERR_OBIT_NOCAUSEMAP);
		gi.dprintf(ERR_OBIT_SEP);
		obits = NULL;
		return;
	}

	obits->causes = LoadMessageTree(E_OBIT_OBITUARYFILE, cFlagsMap, obitCauseMap, obits);

	if (obits->causes == NULL)
	{
		obits = NULL;
		gi.dprintf(ERR_OBIT_SEP);
		gi.dprintf(ERR_OBIT_MALLOC);
		gi.dprintf(ERR_OBIT_SEP);
		return;
	}

	FreeContextTranslator(cFlagsMap, obits);
	if (ReTagObitData(obits) == false)
	{	
		obits = NULL;
		gi.dprintf(ERR_OBIT_SEP);
		gi.dprintf(ERR_OBIT_MALLOC);
		gi.dprintf(ERR_OBIT_SEP);
		return;
	}

	gi.dprintf(ERR_OBIT_SUCCESS);
	gi.dprintf(OBIT_SEP);
}

obitContainer_t **LoadMessageTree(const char *szFilename, cFlagsMap_t **cFlagMap,
									char **szCauses, obitInfo_t *obit)
{
	FILE *f;
	obits_t *curObit;

	char line[OBIT_BUFF_SIZE];
	char *message;
	char *cause;
	char *context;
	char *newMessage;

	unsigned long mallocSize;
	unsigned long lIndex;
	unsigned long causeInt;
	unsigned long cFlag;
	int i = 0;

	char *filename;

	// Create the filename.
	filename = malloc(strlen(gamedir->string) + strlen(szFilename) + 2);
	strcpy(filename, gamedir->string);
	strcat(filename, "/");
	strcat(filename, szFilename);

	gi.dprintf("   Loading Message Tree...\n");

	// First, let's open the file.
	if ( (f=fopen(filename, "r")) == NULL)
	{	// Error opening the file.
		//		free(filename);
		gi.dprintf(ERR_OBIT_FILEOPEN, szFilename);
		return (NULL);
	}

	// Create the container in obits
	mallocSize = sizeof(obitContainer_t *) * (obits->countCause);
	obit->causes = gi.TagMalloc(mallocSize, TAG_LEVEL);
	obit->memAllocated += mallocSize;

	if (obit->causes == NULL)
	{	// malloc error...
		gi.dprintf(ERR_OBIT_MALLOC);
		return (NULL);
	}


	// It also prevents GPF's if you allocate the actual space, instead of just the **
	for (i = 0 ; i < obit->countCause ; i++)
	{
		mallocSize = sizeof(obitContainer_t);
		obit->causes[i] = gi.TagMalloc(mallocSize, TAG_LEVEL);
		if (obit->causes[i] == NULL)
		{	// malloc error...
			gi.dprintf(ERR_OBIT_MALLOC);
			return (NULL);
		}
		obit->memAllocated += mallocSize;
	}

	// We'll use an array of obits_t (tempObits) to store the pointers to the
	// parsed information. When all information has been parsed, we'll copy the
	// tempObits to the permentant member of obits->causes.

	while ( fgets(line, OBIT_BUFF_SIZE, f) )
	{
		// Not a comment or blank line.
		if ( (strlen(line) > 2 ) && (strcspn(line, OBIT_COMMENT) != 0) )
		{
			// Re-initialize all temporary vars...
			message = NULL;
			cause = NULL;
			context = NULL;
			causeInt = 0;
			cFlag = 0;

			// Load the message parts.
			message = strtok(line, OBIT_DELIMITER);
			cause = strtok(NULL, OBIT_DELIMITER);
			TrimSpaces(cause);
			// Translate the Cause into an ordinal into obits->causes.
			causeInt= FindCause(cause, szCauses, obits->countCause);
			// Translatet the context string into a cFlag

			// Let's process all cFlags...
			context = strtok(NULL, OBIT_DELIMITER);

			if (context == NULL)
			{
				// Normal context.
				cFlag = 0;
			} else {
				// Read the contexts from the buffer.
				while(context != NULL)
				{
					TrimSpaces(context);
					cFlag |= GetContext(cFlagMap, context, obits->countContext);
					context = strtok(NULL, OBIT_DELIMITER);
				}
			}

			// See if we can find the context...
			curObit = FindContext(causeInt, cFlag, obits);
			if (curObit == NULL)
			{	// It's not an entry in the context list yet...
				if (obit->causes[causeInt]->entryCount == 0)
				{
					obit->causes[causeInt]->obituary = malloc(sizeof(obits_t*));
				}else {

					mallocSize = sizeof(obits_t*) * 
								(obit->causes[causeInt]->entryCount + 1);
					obit->causes[causeInt]->obituary = 
						realloc(obit->causes[causeInt]->obituary, mallocSize);
				}
				obit->causes[causeInt]->entryCount++;
				/*
				tmpObitContainers = obit->causes;
				tmpObitContainer = tmpObitContainers[cause];
				tmpObitContainer->obituary = malloc(sizeof(obits_t*));
				*/
				if (obits->causes[causeInt]->obituary == NULL)
				{	// malloc error...
					gi.dprintf(ERR_OBIT_MALLOC);
					return (NULL);
				}
				// Now, tagmalloc a new obits_t
				mallocSize = sizeof(obits_t);
				curObit = gi.TagMalloc(mallocSize, TAG_LEVEL);
				if (curObit == NULL)
				{	// malloc error...
					gi.dprintf(ERR_OBIT_MALLOC);
					return (NULL);
				}
				obits->memAllocated += mallocSize;

				lIndex = (obits->causes[causeInt]->entryCount) - 1;

				obits->causes[causeInt]->obituary[lIndex] = curObit;
			} // If it was found, we don't do anything... Dont' want multiples.

			// Now that curObit is all set, let's add the message.
			if (curObit->msgCount == 0)
			{	// We have to create the message store.
				curObit->cFlags = cFlag;
				curObit->msgCount++;
				curObit->messages = malloc(sizeof(char*));
				if (curObit->messages == NULL)
				{	// malloc error...
					gi.dprintf(ERR_OBIT_MALLOC);
					return (NULL);
				}
			}else{
				// The message store exists, realloc it.
				curObit->msgCount++;
				curObit->messages = realloc(curObit->messages,
									sizeof(char*) * (curObit->msgCount));
				if (curObit->messages == NULL)
				{	// malloc error...
					gi.dprintf(ERR_OBIT_MALLOC);
					return (NULL);
				}
			}

			// Now, allocate the string buffer.
			mallocSize = ( sizeof(char *) * (strlen(message)) ) + 2;
			newMessage = gi.TagMalloc( mallocSize, TAG_LEVEL);
			if (newMessage == NULL)
			{	// malloc error...
				gi.dprintf(ERR_OBIT_MALLOC);
				return (NULL);
			}
			strcpy(newMessage, message);
			obit->memAllocated += mallocSize;
			obit->countMsgs++;
			curObit->messages[curObit->msgCount - 1] = newMessage;
			
		}
	}

	// That should do it... (I hope...)
	fclose(f);
	return (obits->causes);
}

/* This subroutine takes a look at each of the obit_t structures contained in the
obitInto_t pointer passed in and converts them to TagMalloc'd strings and 
records the memory usage.

The routine then frees the "temporary" string pointers that were originally
allocated.
*/
qboolean ReTagObitData(obitInfo_t *obit)
{
	unsigned long ulCause;
	
	// The causes member of the obitInfo structure should be the correct size.
	
	for (ulCause = 0 ; ulCause < obit->countCause ; ulCause++)
	{	// We'll loop through all of the entries in obit->causes
		// And retag the obituary. The obituary will then retag all of the messages.
		
		if (TagMallocObituary(obit->causes[ulCause], obit) == NULL)
		{	// Malloc error...
			return (false);
		}
	}
	
	return (true);
	
}

obits_t** TagMallocObituary(obitContainer_t *obitCont, obitInfo_t *obit)
{
	unsigned long i;
	unsigned long mallocSize;
	obits_t 	**newObits;
	
	// Allocate new space
	mallocSize = sizeof(obits_t*) * (obitCont->entryCount);
	newObits = gi.TagMalloc(mallocSize, TAG_LEVEL);
	if (newObits == NULL)
	{	// Malloc error...
		return (NULL);
	}
	obit->memAllocated += mallocSize;
	
	// Copy entries from old space.
	for (i = 0 ; i < obitCont->entryCount ; i++)
	{
		newObits[i] = obitCont->obituary[i];
		if (TagMallocMessages(obitCont->obituary[i], obit) == NULL)
		{	// MALLOC Problem.
			return (NULL);
		}
	}
	
	free(obitCont->obituary);
	obitCont->obituary = newObits;
	
	return (newObits);
}

char** TagMallocMessages(obits_t *obitEntry, obitInfo_t *obit)
{
	unsigned long mallocSize;
	unsigned long i;
	char **newMsgArray;
	
	
	// Allocate the new space.
	mallocSize = sizeof(char*) * (obitEntry->msgCount);
	newMsgArray = gi.TagMalloc(mallocSize, TAG_LEVEL);
	if (newMsgArray == NULL)
	{	// Malloc error
		return (NULL);
	}
	obit->memAllocated += mallocSize;
	
	// Copy entries from old space.
	for (i = 0 ; i < obitEntry->msgCount ; i++)
		newMsgArray[i] = obitEntry->messages[i];
	
	free(obitEntry->messages);
	obitEntry->messages = newMsgArray;
	
	return (newMsgArray);
}

unsigned long GetContext(cFlagsMap_t **cMap, char *szContext, unsigned long count)
{
	unsigned long l;
	
	for (l = 0; l < count; l++)
	{
		if (Q_stricmp(szContext, cMap[l]->key) == 0)
			return (cMap[l]->cFlags);
	}
	return (0);
}

void FreeContextTranslator(cFlagsMap_t **cTrans, obitInfo_t *obit)
{
	int i = 0;
	
	for (i = 0 ; i < obits->countContext ; i++)
	{
		// Loop through the cTrans and first free every key...
		free(cTrans[i]->key);
		// Then free the element itself
		free(cTrans[i]);
	}
	
	// Now free the whole damned thing.
	free(cTrans);
}

//
// This function loads the temporary Context Translator from the context file.
//
cFlagsMap_t** LoadContextTranslator(const char *szFilename, obitInfo_t *obits)
{
	FILE *f;
	cFlagsMap_t **cMap;
	cFlagsMap_t *newMap;
	
	char line[OBIT_BUFF_SIZE];
	char *temp;
	char *cKey;
	unsigned long cFlag = 0;
	unsigned long conCount = 0;
	unsigned long cFlagBits = 0;
	char *filename;
	
	// Create the filename.
	filename = malloc(strlen(gamedir->string) + strlen(szFilename) + 2);
	strcpy(filename, gamedir->string);
	strcat(filename, "/");
	strcat(filename, szFilename);
	
	// Mental note... Not initializing variables causes GPF's... DOH!
	
	gi.dprintf("   Loading Context Translator...\n");
	
	// First, let's open the file.
	if ( (f=fopen(filename, "r")) == NULL)
	{	// Error opening the file.
		//		free(filename);
		gi.dprintf(ERR_OBIT_FILEOPEN, szFilename);
		return (NULL);
	}
	//	free(filename);
	
	// Now, let's malloc the space for the translation table.
	cMap = malloc(sizeof(cFlagsMap_t*));
	
	// Now, we'll loop through it and allocate space as we go.
	while (fgets(line, OBIT_BUFF_SIZE, f))
	{
		// Check to make sure line is not a comment or blank.
		if( (strlen(line) > 1) && (strcspn(line, OBIT_COMMENT) != 0) )
		{
			temp = strtok(line, OBIT_DELIMITER);
			if (temp != NULL)
			{
				cKey = malloc( sizeof(char) * (strlen(temp) + 1) );
				if (cKey == NULL)
				{	// malloc problem... We're hosed...
					return (NULL);
				}
				// Allocate space for transferring temp to perm data.
				
				TrimSpaces(temp);
				strcpy(cKey, temp);
				temp = strtok(NULL, OBIT_DELIMITER);
				TrimSpaces(temp);
				cFlagBits = atol(temp);
				
				// Unbitify (Wow... New word) the context flag from the file.
				if (cFlagBits == 0)
					cFlag = 0;
				else
					cFlag = 1 << (cFlagBits - 1);
				
				newMap = malloc(sizeof(cFlagsMap_t));
				if (newMap == NULL)
				{	// malloc problem... We're hosed...
					return (NULL);
				}
				
				newMap->key = cKey;
				newMap->cFlags = cFlag;
				
				conCount++;
				
				cMap = realloc(cMap, sizeof(cFlagsMap_t*) * conCount);
				if (cMap == NULL)
				{	// malloc problem... We're REALLY hosed...
					return (NULL);
				}
				cMap[conCount - 1] = newMap;
			}
		}
	}
	
	fclose(f);
	obits->countContext = conCount;
	
	return (cMap);
}

//
// This function loads the Global Cause Map with information from cause file
//
char** LoadCauseMap(const char *szFilename, obitInfo_t *obituary)
{
	FILE *f;
	char **temp;
	char **causeMap;
	char *cause;
	char *token;
	int  lineCount = 0;
	int  i = 0;
	long lMemSize = 0;
	
	char line[OBIT_BUFF_SIZE];
	char *filename;
	
	// Create the filename.
	filename = malloc(strlen(gamedir->string) + strlen(szFilename) + 2);
	strcpy(filename, gamedir->string);
	strcat(filename, "/");
	strcat(filename, szFilename);
	
	gi.dprintf("   Loading obituary Cause Map...\n");
	
	// First, let's open the file.
	if ( (f=fopen(filename, "r")) == NULL)
	{	// Error opening the file.
		gi.dprintf(ERR_OBIT_FILEOPEN, szFilename);
		return (NULL);
	}
	
	temp = malloc(sizeof(char*));
	
	// Now, we'll loop through it and allocate tempspace as we go.
	while (fgets(line, OBIT_BUFF_SIZE, f))
	{
		// 1-8-97 Added comments to Cause Map
		if( (strlen(line) > 1) && (strcspn(line, OBIT_COMMENT) != 0) )
		{
			lineCount++;
			// This malloc's on the first call and resizes on the subsequent.
			temp = realloc( temp, (sizeof(char*) * lineCount));
			
			// Now, let's malloc a string and put it where it belongs.
			lMemSize = sizeof(char) * (strlen(line));
			cause = gi.TagMalloc(lMemSize, TAG_LEVEL);
			if (cause == NULL)
			{
				// Malloc error...
				gi.dprintf(ERR_OBIT_MALLOC);
				return (NULL);
			}
			
			// And how about copying the line into the new string
			token = strtok(line, "\n");
			TrimSpaces(token);
			strcpy(cause, token);
			
			// We now notify the obituary information of malloc.
			obituary->memAllocated += lMemSize;
			
			// Store the pointer in the temp array.
			temp[lineCount - 1] = cause;
			
		}
	}
	
	// Ok, file is processed. Close it.
	fclose(f);
	
	// Now that we know how many causes, lets store that and malloc the "real"
	// cause map.
	obituary->countCause = lineCount;
	
	lMemSize = sizeof(char*) * lineCount;
	causeMap = gi.TagMalloc(lMemSize, TAG_LEVEL);
	obituary->memAllocated += lMemSize;
	
	for (i = 0 ; i < (lineCount) ; i++)
	{
		causeMap[i] = temp[i];
	}
	
	// We're done with the temp structure, free it.
	free(temp);
	
	obituary->countCause = lineCount;
	return (causeMap);
}

int GetEntityGender(edict_t *ent)
{
	if (ent->client)
	{
		if (strstr(ent->client->pers.sounddir, "female") != NULL)
			return (OBIT_GENDER_FEMALE);
		if (strstr(ent->client->pers.sounddir, "male") !=NULL )
			return (OBIT_GENDER_MALE);
	}
	return (OBIT_GENDER_OTHER);
}

// Thanks to SteQve for these routines... It was 3:00 a.m. and I couldn't think...

int StrBeginsWith (char *s1, char *s2)
{
	int 	c1, c2, max = 999999;
	
	do
	{
		c1 = *s1++;
		c2 = *s2++;
		if (!c1) return(1); // Reached end of search string
		if (!max--)
			return 1;		// strings are equal until end point
		
		if (c1 != c2)
		{
			if (c1 >= 'a' && c1 <= 'z')
				c1 -= ('a' - 'A');
			if (c2 >= 'a' && c2 <= 'z')
				c2 -= ('a' - 'A');
			if (c1 != c2)
				return 0;		// strings not equal
		}
	} while (c1);
	
	return 1;		// strings are equal
}

int InsertValue(char *mess, char *format_str, char *field, char *value, int max_size)
{
	int x=0, y=0, matches = 0;
	mess[0] = 0;
	while (format_str[x] && (x < max_size))
	{
		if (StrBeginsWith(field, format_str + x))
		{
			if (x + strlen(value) < max_size)
			{
				//PRINT2("Found match on '%s' for field '%s\n", format_str, field);
				matches++;
				strcpy(mess+y, value);
				y = y + strlen(value);
				x = x + strlen(field);
			}
			else {
				// Buffer overflow would occur.
				return(-1);
			}
		}
		else {
			mess[y++] = format_str[x++];
		}
	}
	if (y < max_size)
		mess[y] = 0;
	else mess[max_size - 1] = 0;
	return(matches);
}

void GreenText(char *szDest, const char *szSrc)
{

	int curChar = 0;
	int len = 0;
	int i = 0;

	len = strlen(szSrc);

	if (len == 0)
		return;

	strcpy(szDest, szSrc);
	for (i = 0 ; i < len ; i++)
	{
		curChar = *(szDest + i);
		if( (curChar > 32) && (curChar < 128) )
			szDest[i] |= 128;
	}

}

