#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>		/* close() */
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/uio.h>
#include <netinet/in.h>
#include <netdb.h>

#define FALSE			0
#define TRUE			1
#define MAX_BUF_SIZE		2048		/* The UDP socket buf size */
#define MAX_OUTBUF_SIZE		4096		/* Output buffer size */
#define MAX_LINE_BUF_SIZE	80		/* Generic string size */
#define TIMEOUT			5		/* Timeout in seconds */
#define Q2_BUF_PADDING		7		/* # padding chars */
#define Q2_BUF_HEADER		4		/* Q2 header length */
#define Q2_STATUS		"status\n"	/* Status command */
#define Q2_STATUS_LEN		7		/* Length of status command */
#define Q2_PING			"ping\n"	/* Ping command */
#define Q2_PING_LEN		5		/* Length of ping command */
#define Q2_PING_TIMES		10		/* # ping's */
#define Q2_ACK			"ack"		/* Acknowledge reply */
#define Q2_ACK_LEN		3		/* Length of ack */
#define Q2_SEPARATOR		0x5c		/* Rule/value separator */
#define Q2_DEF_UDP_PORT		27910		/* Default UDP port */


/*
 * Prototypes for local functions
 */
static void		Usage(void);
static void		Signal_Handler(int signal);
static int		Decode_Status_Buf(unsigned char *buf,
					  int *silent_mode);
static int		Get_Status(int *sock,
				   struct sockaddr_in *remote,
				   unsigned char *buf,
				   int *silent_mode);
static int		Decode_Ack_Buf(unsigned char *buf);
static int		Get_Ping_Time(int *sock,
				      struct sockaddr_in *remote,
				      unsigned char *buf);

/*
 * Prototypes for external functions
 */
extern int		getopt(int argc,
			       char *const *argv,
			       const char *optstring);


void Usage(void)
{
    char		msg[] = {
	"\n"
	"Usage: q2ping [options] hostname\n"
	"\n"
	"Where options are:\n"
	"       -s         Skip status info\n"
	"       -S         Silent mode, no output will be printed\n"
	"                  Non-zero return code indicate a failure\n"
	"       -p n       Use UDP port n instead of default 27910\n"
	"\n"
    };
    
    
    (void)fprintf(stderr, "%s", &msg);
    
    return;
}


void Signal_Handler(int signal)
{
    switch(signal){
    case SIGALRM:
	/*
	 * Informative message printed in the interrupted function
	 */
	break;
    default:
	(void)fprintf(stdout,
		      "\nCaught signal %d\n",
		      signal);
	
	break;
    }
    
    return;
}


/*
 * Decode rule/value and possible user info
 * Return codes:
 * 0 = Success
 * 1 = Bogus header
 */
int Decode_Status_Buf(unsigned char *buf, int *silent_mode)
{
    int			rc;			/* Return code */
    int			buf_cntr;		/* buf index */
    int			buf_len;		/* Length of buf */
    int			separator_number;	/* Counts separators */
    int			rule_cntr;		/* Index for rule buf */
    int			value_cntr;		/* Index for value buf */
    int			out_cntr;		/* Index for out buf */
    int			user_info;		/* User info flag */
    char		rule_buf[MAX_LINE_BUF_SIZE];	/* Store rule str */
    char		value_buf[MAX_LINE_BUF_SIZE];	/* Store value */
    char		out_buf[MAX_OUTBUF_SIZE];	/* Output buffer */
    

    buf_len = strlen((const char *)buf);

    (void)memset((char *)&rule_buf, 0, (size_t)MAX_LINE_BUF_SIZE);
    (void)memset((char *)&value_buf, 0, (size_t)MAX_LINE_BUF_SIZE);
    (void)memset((char *)&out_buf, 0, (size_t)MAX_OUTBUF_SIZE);

    /*
     * Reset index counters
     */
    rule_cntr = 0;
    value_cntr = 0;
    out_cntr = 0;
    
    separator_number = 0;

    user_info = FALSE;

    /*
     * Loop until all status rules and values have been decoded
     */
    for(buf_cntr = 0; buf_cntr < buf_len; buf_cntr++){
	if(buf_cntr < 4){
	    /*
	     * Verify that we got a valid header (0xffffffff)
	     * Re-init the buf_cntr instead of using a special variable
	     * (not optimal, but what the heck...)
	     */
	    for(buf_cntr = 0; buf_cntr < (Q2_BUF_HEADER - 1); buf_cntr++){
		if(buf[buf_cntr] != 0xff){
		    /*
		     * Bogus header
		     */
		    return(1);
		}
	    }
	    
	    continue;
	}
	else if((buf_cntr >= 4) && (buf_cntr < 10)){
	    /*
	     * Status info seem to always contain a "print<nl>" command
	     * preceeding the actual status data. The GameSpy program
	     * does not show this, we do the same !
	     */
	    buf_cntr += 6;
	    
	    continue;
	}

	if((buf[buf_cntr] != Q2_SEPARATOR) && (separator_number == 0)){
	    /*
	     * Get and store one rule in each loop
	     */
	    rule_buf[rule_cntr] = buf[buf_cntr];
	    
	    rule_cntr++;

	    continue;
	}
	else if((buf[buf_cntr] == Q2_SEPARATOR) && (separator_number == 0)){
	    /*
	     * Skip over the Q2_SEPARATOR between the rule and the value
	     */
	    separator_number++;

	    continue;
	}
	else if((buf[buf_cntr] != Q2_SEPARATOR) && (separator_number == 1)){
	    /*
	     * Get and store the value for the rule just stored
	     */
	    if((buf[buf_cntr] == 0x0a) && (user_info == FALSE)){
		/*
		 * The rule_buf now contain the last rule
		 */
		user_info = TRUE;
		
		rc = strlen((char *)&out_buf);
		
		(void)sprintf((char *)&out_buf[rc],
			      "%s = %s\n",
			      &rule_buf,
			      &value_buf);

		/*
		 * Zero the value buffer for next loop
		 * Note: We also use value_buf to store user info
		 */
		(void)memset((char *)&value_buf,
			     0,
			     (size_t)MAX_LINE_BUF_SIZE);
	    
		/*
		 * Reset value index
		 */
		value_cntr = 0;

		if(buf_len > (buf_cntr + 1)){
		    /*
		     * There is also user info in the main buffer
		     */
		    rc = strlen((char *)&out_buf);
		    
		    (void)sprintf((char *)&out_buf[rc],
				  "\n%s\n%s\n",
				  "Kills  Pingtime  User",
				  "---------------------");
		}
		
		continue;
	    }
	    else if((buf[buf_cntr] == 0x0a) && (user_info == TRUE)){
		/*
		 * User info
		 */
		rc = strlen((char *)&out_buf);
	    
		(void)sprintf((char *)&out_buf[rc],
			      "%s\n",
			      &value_buf);

		/*
		 * Zero the value buffer for next loop
		 */
		(void)memset((char *)&value_buf,
			     0,
			     (size_t)MAX_LINE_BUF_SIZE);
	    
		/*
		 * Reset value index
		 */
		value_cntr = 0;
	    
		continue;
	    } else {
		value_buf[value_cntr] = buf[buf_cntr];
	    
		value_cntr++;

		continue;
	    }
	}
	else if((buf[buf_cntr] == Q2_SEPARATOR) && (separator_number == 1)){
	    /*
	     * Got both the rule and the value, time to store it in
	     * the output buffer
	     */
	    rc = strlen((char *)&out_buf);
	    
	    (void)sprintf((char *)&out_buf[rc],
			  "%s = %s\n",
			  &rule_buf,
			  &value_buf);

	    /*
	     * Zero the rule and the value buffers for next loop
	     */
	    (void)memset((char *)&rule_buf, 0, (size_t)MAX_LINE_BUF_SIZE);
	    (void)memset((char *)&value_buf, 0, (size_t)MAX_LINE_BUF_SIZE);
	    
	    /*
	     * Reset index and Q2_SEPARATOR counters
	     */
	    rule_cntr = 0;
	    value_cntr = 0;
	    
	    separator_number = 0;
	}
    }

    /*
     * We've loop'ed through all status info, time to print the result
     */
    if(*silent_mode == FALSE){
	(void)fprintf(stdout,
		      "\n%s\n%s\n%s\n",
		      "============= Status info ==============",
		      &out_buf,
		      "========== End of status info ==========");
    }

    /*
     * Success
     */
    return(0);
}


/*
 * Send the status command
 * Return codes:
 * 0 = Success
 * 1 = sendto() failed
 * 2 = recvfrom() failed
 * 3 = Timeout
 * 4 = Non-zero return from Decods_Status_Buf()
 */
int Get_Status(int *sock,
	       struct sockaddr_in *remote,
	       unsigned char *buf,
	       int *silent_mode)
{
    int			rc;		/* return code */
    int			remote_len;	/* Size of struct sockaddr_in */
    int			send_len;	/* Size of send buffer */
    

    (void)memset(buf,
		 0,
		 (size_t)MAX_BUF_SIZE);

    /*
     * Init send buffer with the header (0xffffffff)
     */
    buf[0] = 0xff;
    buf[1] = 0xff;
    buf[2] = 0xff;
    buf[3] = 0xff;
    
    /*
     * Add the Q2_STATUS command to the send buffer
     */
    (void)memcpy(&buf[4],
		 (char *)Q2_STATUS,
		 Q2_STATUS_LEN);
    
    remote_len = sizeof(*remote);
    
    send_len = (Q2_BUF_HEADER + Q2_STATUS_LEN + Q2_BUF_PADDING);
    
    /*
     * Send the Q2_STATUS command down the wire
     */
    rc = sendto(*sock,
		(const char *)buf,
		send_len,
		0,
		(struct sockaddr *)remote,
		remote_len);
    
    if(rc < 0){
	(void)fprintf(stderr,
		      "sendto : %s\n",
		      strerror(errno));
	
	return(1);
    }

    /*
     * Zero the buffer
     */
    (void)memset(buf,
		 0,
		 (size_t)MAX_BUF_SIZE);

    /*
     * Define the timeout signal
     */
    (void)signal(SIGALRM, &Signal_Handler);
    
    (void)alarm(TIMEOUT);
    
    /*
     * Wait for the response
     */
    rc = recvfrom(*sock,
		  (char *)buf,
		  MAX_BUF_SIZE,
		  0,
		  (struct sockaddr *)remote,
		  &remote_len);
    
    if(rc < 0){
	/*
	 * If we end up here _and_ errno == EINTR, then it's most
	 * likely due to a timeout, it is somewhat weak though since
	 * we might have been interrupted for other causes
	 */
	if(errno != EINTR){
	    /*
	     * The recvfrom() call failed for some other reason
	     */
	    (void)fprintf(stderr,
			  "recvfrom : %s\n",
			  strerror(errno));

	    return(2);
	}

	if(*silent_mode == FALSE){
	    (void)fprintf(stdout, "\nTimed out !\n");
	}
	
	return(3);
    }

    /*
     * Hopefully we received some status info
     */
    rc = Decode_Status_Buf(buf, silent_mode);
    
    /*
     * Disable the SIGALRM
     */
    (void)alarm(0);

    if(rc != 0){
	return(4);
    } else {
	return(0);
    }
}


/*
 * Ack reply verification
 * Return codes:
 * 0 = Success
 * 1 = Invalid header
 * 2 = Invalid (or no) ack
 */
int Decode_Ack_Buf(unsigned char *buf)
{
    int			rc;		/* Return code */
    int			loop_cntr;	/* Loop counter and buf index */
    
    for(loop_cntr = 0; loop_cntr < Q2_BUF_HEADER; loop_cntr++){
	/*
	 * Verify that we got a valid header (0xffffffff)
	 */
	if(buf[loop_cntr] != 0xff){
	    /*
	     * No, we didn't
	     */
	    return(1);
	}
    }

    /*
     * Verify that we got a Q2_ACK
     */
    rc = strcmp((const char *)&buf[Q2_BUF_HEADER], Q2_ACK);
    
    if(rc != 0){
	/*
	 * No, we didn't
	 */
	return(2);
    }

    /*
     * Yes, we did
     */
    return(0);
}


/*
 * Send ping command and calculate ping time
 * Return codes:
 * 0 = Success
 * 1 = gettimeofday(start) failed
 * 2 = sendto() failed
 * 3 = recvfrom() failed
 * 4 = gettimeofday(stop) failed
 */
int Get_Ping_Time(int *sock,
		  struct sockaddr_in *remote,
		  unsigned char *buf)
{
    int			rc;		/* Return code */
    int			loop_cntr;	/* Loop counter */
    int			remote_len;	/* Size of struct sockaddr_in */
    int			send_len;	/* Size of send buffer */
    int			timed_out;	/* Timeout flag */
    int			successful_ping; /* Number of successful ping's */
    int			wrong_ack;	/* Number of wrong ack's */
    long		sec_l;		/* Ping time, seconds */
    long		usec_l;		/* Ping time, microseconds */
    float		ping_time;	/* Combined ping time */
    float		total_ping_time; /* Total (aggr.) ping time */
    
    struct timeval	start;		/* Start time */
    struct timeval	stop;		/* Stop time */
    

    (void)setbuf(stdout, NULL);

    (void)fprintf(stdout, "\nRetrieveing ping statistics ");
    
    remote_len = sizeof(*remote);

    /*
     * Reset counters
     */
    successful_ping = 0;
    wrong_ack = 0;
    total_ping_time = 0;

    /*
     * Enter the loop that will send a ping Q2_PING_TIMES
     */
    for(loop_cntr = 0; loop_cntr < Q2_PING_TIMES; loop_cntr++){
	(void)memset(buf,
		     0,
		     (size_t)MAX_BUF_SIZE);

	/*
	 * Init buffer with the header (0xffffffff)
	 */
	buf[0] = 0xff;
	buf[1] = 0xff;
	buf[2] = 0xff;
	buf[3] = 0xff;

	/*
	 * Add the Q2_PING command to the buffer
	 */
	(void)memcpy(&buf[4],
		     &Q2_PING,
		     (size_t)Q2_PING_LEN);

	send_len = (Q2_BUF_HEADER + Q2_PING_LEN + Q2_BUF_PADDING);

	timed_out = FALSE;
	/*
	 * Get the start time
	 */
	rc = gettimeofday(&start, (void *)NULL);
    
	if(rc < 0){
	    (void)fprintf(stderr,
			  "gettimeofday : %s\n",
			  strerror(errno));
	
	    return(1);
	}

	/*
	 * Send the Q2_PING command down the wire
	 */
	rc = sendto(*sock,
		    (const char *)buf,
		    send_len,
		    0,
		    (struct sockaddr *)remote,
		    remote_len);
	
	if(rc < 0){
	    (void)fprintf(stderr,
			  "sendto : %s\n",
			  strerror(errno));
	    
	    return(2);
	}
	
	(void)memset(buf,
		     0,
		     (size_t)MAX_BUF_SIZE);

	/*
	 * Define the timeout signal
	 */
	(void)signal(SIGALRM, &Signal_Handler);
	
	(void)alarm(TIMEOUT);
	
	/*
	 * Wait for a response
	 */
	rc = recvfrom(*sock,
		      (char *)buf,
		      MAX_BUF_SIZE,
		      0,
		      (struct sockaddr *)remote,
		      &remote_len);
	
	if(rc < 0){
	    if(errno != EINTR){
		(void)fprintf(stderr,
			      "recvfrom : %s\n",
			      strerror(errno));

		return(3);
	    }
	    
	    timed_out = TRUE;

	    (void)fprintf(stdout, "T");
	}

	/*
	 * Get the stop time
	 */
	rc = gettimeofday(&stop, (void *)NULL);
    
	if(rc < 0){
	    (void)fprintf(stderr,
			  "gettimeofday : %s\n",
			  strerror(errno));
	
	    return(4);
	}

	/*
	 * Calculate the ping time, but only if we didn't timeout
	 */
	if(timed_out == FALSE){
	    sec_l = (stop.tv_sec - start.tv_sec);
	
	    if(start.tv_usec > stop.tv_usec){
		usec_l = (start.tv_usec - stop.tv_usec);
	    
		ping_time = ((sec_l * 1000000) - usec_l);
	    } else {
		usec_l = (stop.tv_usec - start.tv_usec);

		ping_time = ((sec_l / 1000000) + usec_l);
	    }

	    /*
	     * Verify that we received a ack
	     */
	    rc = Decode_Ack_Buf(buf);
	
	    if(rc != 0){
		/*
		 * Wrong data in ack, print letter "X" and increment
		 * the wrong_ack counter
		 */
		(void)fprintf(stdout, "X");

		wrong_ack++;
	    } else {
		/*
		 * We received a ack, print a dot and increment ping counter
		 * and finally add this ping time to the total ping time
		 */
		(void)fprintf(stdout, ".");

		successful_ping++;
	    
		total_ping_time += ping_time;
	    }
	} /* if(timed_out == FALSE) */
    } /* for(loop_cntr = 0; loop_cntr < Q2_PING_TIMES; loop_cntr++) */

    /*
     * Show the results
     */
    if(loop_cntr == successful_ping){
	(void)fprintf(stdout,
		      "\n\n%s = %d\n%s = %3.2f msec\n\n",
		      "Number of ping requests",
		      loop_cntr,
		      "Average ping time",
		      ((total_ping_time / successful_ping) / 1000));
    } else {
	(void)fprintf(stdout,
		      "\n\n%s = %d\n%s = %d\n%s = %d\n%s = %3.2f msec\n\n",
		      "Number of ping requests",
		      loop_cntr,
		      "Number of invalid ack's",
		      wrong_ack,
		      "Number of timeout's",
		      (loop_cntr - successful_ping),
		      "Average ping time",
		      ((loop_cntr - successful_ping) == 0) ?
		      (float)0 :
		      ((total_ping_time / successful_ping) / 1000));
    }
    
    return(0);
}


/*
 * Main function
 * Return codes:
 *  0 = Success
 *  1 = Help arguments (-h or -?)
 *  2 = Wrong number of arguments
 *  3 = gethostbyname() failed
 *  4 = socket() failed
 *  5 = Get_Status() failed, sendto()
 *  6 = Get_Status() failed, recvfrom()
 *  7 = Get_Status() failed, timeout
 *  8 = Get_Status() failed, bogus status reply header
 *  9 = Get_Status() failed, unknown return code
 * 10 = Get_Ping_Time() failed, gettimeofday(start)
 * 11 = Get_Ping_Time() failed, sendto()
 * 12 = Get_Ping_Time() failed, recvfrom()
 * 13 = Get_Ping_Time() failed, gettimeofday(stop)
 * 14 = Get_Ping_Time() failed, unknown return code
 */
int main(int argc, char *argv[])
{
    int			rc;			/* Return code */
    int			sock;			/* The socket fd */
    int			no_status;		/* Skip status info flag */
    int			silent_mode;		/* Silent mode flag */
    unsigned short	udp_port;		/* Holds the UDP port */
    unsigned char	buf[MAX_BUF_SIZE];	/* Send/recv buffer */

    extern int		optind;			/* See getopt(3C) */
    extern char		*optarg;		/* See getopt(3C) */
    
    struct sockaddr_in	remote;			/* Address struct */
    struct hostent	*hp;			/* hostent pointer */
    

    /*
     * Default values
     */
    no_status = FALSE;
    silent_mode = FALSE;
    udp_port = (unsigned short)Q2_DEF_UDP_PORT;
    
    while((rc = getopt(argc, argv, "?hp:sS")) != EOF){
	switch(rc){
	case 'h':
	case '?':
	    Usage();
	    
	    return(1);
	case 'p':
	    udp_port = (unsigned short)atoi(optarg);
	    
	    break;
	case 's':
	    no_status = TRUE;
	    
	    break;
	case 'S':
	    silent_mode = TRUE;

	    no_status = FALSE;
	    
	    break;
	}
    }

    if((optind == argc) || ((argc - optind) != 1)){
	Usage();
	
	return(2);
    }

    /*
     * This will make sure silent mode override "-s" argument
     * (In case "-S" was provided before "-s" on the command line)
     */
    if((silent_mode == TRUE) && (no_status == TRUE)){
	no_status = FALSE;
    }
    
    hp = gethostbyname(argv[optind]);
    
    if(hp == NULL){
	(void)fprintf(stderr,
		      "\nCould not resolve hostname \"%s\"\n\n",
		      argv[optind]);
	
	return(3);
    }

    sock = socket(AF_INET, SOCK_DGRAM, 0);
    
    if(sock < 0){
	(void)fprintf(stderr,
		      "Could not create socket : %s\n",
		      strerror(errno));
	
	return(4);
    }

    /*
     * Zero the sockaddr_in struct, then copy the hostent struct into it
     */
    (void)memset((struct sockaddr_in *)&remote,
		 0,
		 (size_t)sizeof(struct sockaddr_in));

    (void)memcpy((char *)&remote.sin_addr,
		 (char *)hp->h_addr,
		 hp->h_length);
    
    /*
     * Init sockaddr_in struct with proper family type and port info
     */
    remote.sin_family = AF_INET;
    remote.sin_port = htons(udp_port);

    if(no_status == FALSE){
	/*
	 * Get current status from remote
	 */
	rc = Get_Status(&sock,
			&remote,
			(unsigned char *)&buf,
			&silent_mode);

	switch(rc){
	case 0:
	    /*
	     * Success, continue
	     */

	    break;
	case 1:
	    /*
	     * sendto() failed
	     */
	    (void)close(sock);
	    
	    return(5);
	case 2:
	    /*
	     * recvfrom() failed
	     */
	    (void)close(sock);
	    
	    return(6);
	case 3:
	    /*
	     * Timeout
	     */
	    (void)close(sock);
	    
	    return(7);
	case 4:
	    /*
	     * Non-zero from Decode_Status_Buf(), bogus status reply header
	     */
	    (void)close(sock);
	    
	    return(8);
	default:
	    /*
	     * Unknown return code
	     */
	    (void)close(sock);
	    
	    return(9);
	}
    }

    /*
     * Skipped in silent mode
     */
    if(silent_mode == FALSE){
	rc = Get_Ping_Time(&sock,
			   &remote,
			   (unsigned char *)&buf);
    
	(void)close(sock);
	
	switch(rc){
	case 0:
	    /*
	     * Success
	     */
	    return(0);
	case 1:
	    /*
	     * gettimeofday(start) failed
	     */
	    return(10);
	case 2:
	    /*
	     * sendto() failed
	     */
	    return(11);
	case 3:
	    /*
	     * recvfrom() failed
	     */
	    return(12);
	case 4:
	    /*
	     * gettimeofday(stop) failed
	     */
	    return(13);
	default:
	    /*
	     * Unknown return code
	     */
	    return(14);
	}
    }

    /*
     * In case of success and in silent mode
     */
    (void)close(sock);
    
    return(0);
}

