/*  lmpc -- the Little Movie Processing Centre
    Copyright (C) 1994-98 Uwe Girlich

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Uwe Girlich
    Erika-von-Brockdorff-Strasse 2
    04159 Leipzig
    Deutschland / Germany
    E-mail: Uwe.Girlich@itp.uni-leipzig.de */

/****************************************************************************\
|  uqwd.c - implementation, general QWD routines                             |
\****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include "tools.h"
#include "uqwd.h"

#include "udem.h"
#include "udemy.h"

token_t QWD_token[]={
        { "sblock",              TOKEN_SBLOCK,         0 },
        { "cblock",              TOKEN_CBLOCK,         0 },

        { "time",                TOKEN_TIME,           0 },
        { "seq1",                TOKEN_SEQ1,           0 },
        { "seq2",                TOKEN_SEQ2,           0 },
        { "hidden",              TOKEN_HIDDEN,         0 },
        { "special",             TOKEN_SPECIAL,        0 },
        { "data",                TOKEN_DATA,           0 },
        { "angles",              TOKEN_ANGLES,         0 },
        { "uk_angles",           TOKEN_UK_ANGLES,      0 },
        { "speed",               TOKEN_SPEED,          0 },
        { "impulse",             TOKEN_IMPULSE,        0 },
        { "fire",                TOKEN_FIRE,           0 },
        { "jump",                TOKEN_JUMP,           0 },
        { "flag",                TOKEN_FLAG,           0 },
        { "game",                TOKEN_GAME,           0 },
        { "load",                TOKEN_LOAD,           0 },
        { "code",                TOKEN_CODE,           0 },
        { "connect",             TOKEN_CONNECT,        0 },
        { "hallo",               TOKEN_HALLO,          0 },
        { "console",             TOKEN_CONSOLE,        0 },
        { "id",                  TOKEN_ID,             0 },
        { "unknown",             TOKEN_UNKNOWN,        0 },
        { "version",             TOKEN_VERSION,        0 },
        { "client",              TOKEN_CLIENT,         0 },
        { "age",                 TOKEN_AGE,            0 },
        { "mapname",             TOKEN_MAPNAME,        0 },
        { "level",               TOKEN_LEVEL,          0 },
        { "origin",              TOKEN_ORIGIN,         0 },
        { "vol",                 TOKEN_VOL,            0 },
        { "attenuation",         TOKEN_ATTENUATION,    0 },
        { "soundnum",            TOKEN_SOUNDNUM,               0 },
        { "default_modelindex",  TOKEN_DEFAULT_MODELINDEX,     0 },
        { "default_frame",       TOKEN_DEFAULT_FRAME,          0 },
        { "default_colormap",    TOKEN_DEFAULT_COLORMAP,       0 },
        { "default_skin",        TOKEN_DEFAULT_SKIN,           0 },
        { "default_origin",      TOKEN_DEFAULT_ORIGIN,         0 },
        { "default_angles",      TOKEN_DEFAULT_ANGLES,         0 },
        { "entity",              TOKEN_ENTITY,                 0 },
        { "player",              TOKEN_PLAYER,                 0 },
        { "frags",               TOKEN_FRAGS,                  0 },
        { "ping",                TOKEN_PING,                   0 },
        { "entertime",           TOKEN_ENTERTIME,              0 },
        { "user",                TOKEN_USER,                   0 },
        { "style",               TOKEN_STYLE,                  0 },
        { "string",              TOKEN_STRING,                 0 },
        { "index",               TOKEN_INDEX,                  0 },
        { "value",               TOKEN_VALUE,                  0 },
        { "camera",              TOKEN_CAMERA,                 0 },
        { "save",                TOKEN_SAVE,                   0 },
        { "take",                TOKEN_TAKE,                   0 },
        { "cspeedx",             TOKEN_CSPEEDX,                0 },
        { "cspeedy",             TOKEN_CSPEEDY,                0 },
        { "cspeedz",             TOKEN_CSPEEDZ,                0 },
        { "byte6",               TOKEN_BYTE6,                  0 },
        { "byte7",               TOKEN_BYTE7,                  0 },
        { "byte8",               TOKEN_BYTE8,                  0 },
        { "model",               TOKEN_MODEL,                  0 },
        { "speedx",              TOKEN_SPEEDX,                 0 },
        { "speedy",              TOKEN_SPEEDY,                 0 },
        { "speedz",              TOKEN_SPEEDZ,                 0 },
        { "angles_1",            TOKEN_ANGLES_1,               0 },
        { "angles_2",            TOKEN_ANGLES_2,               0 },
        { "angles_3",            TOKEN_ANGLES_3,               0 },
        { "frame",               TOKEN_FRAME,                  0 },
        { "packetentity",        TOKEN_PACKETENTITY,           0 },
        { "modelindex",          TOKEN_MODELINDEX,             0 },
        { "colormap",            TOKEN_COLORMAP,               0 },
        { "skin",                TOKEN_SKIN,                   0 },
        { "effects",             TOKEN_EFFECTS,                0 },
        { "origin_x",            TOKEN_ORIGIN_X,               0 },
        { "origin_y",            TOKEN_ORIGIN_Y,               0 },
        { "origin_z",            TOKEN_ORIGIN_Z,               0 },
        { "entitytype",          TOKEN_ENTITYTYPE,             0 },
        { "trace_endpos",        TOKEN_TRACE_ENDPOS,           0 },
        { "count",               TOKEN_COUNT,                  0 },
        { "channel",             TOKEN_CHANNEL,                0 },
        { "nail",                TOKEN_NAIL,                   0 },
        { "weapon",              TOKEN_WEAPON,                 0 },
        { "weaponframe",         TOKEN_WEAPONFRAME,            0 },
        { "deltapacketentity",   TOKEN_DELTAPACKETENTITY,      0 },
        { "dir",                 TOKEN_DIR,                    0 },
        
        { "disconnect",          TOKEN_DISCONNECT,          0x02 },
        { "updatestat",          TOKEN_UPDATESTAT,          0x03 },
        { "sound",               TOKEN_SOUND,               0x06 },
        { "print",               TOKEN_PRINT,               0x08 },
        { "stufftext",           TOKEN_STUFFTEXT,           0x09 },
        { "setangle",            TOKEN_SETANGLE,            0x0A },
        { "serverdata",          TOKEN_SERVERDATA,          0x0B },
        { "lightstyle",          TOKEN_LIGHTSTYLE,          0x0C },
        { "updatefrags",         TOKEN_UPDATEFRAGS,         0x0E },
        { "damage",              TOKEN_DAMAGE,              0x13 },
        { "spawnstatic",         TOKEN_SPAWNSTATIC,         0x14 },
        { "spawnbaseline",       TOKEN_SPAWNBASELINE,       0x16 },
        { "temp_entity",         TOKEN_TEMP_ENTITY,         0x17 },
        { "centerprint",         TOKEN_CENTERPRINT,         0x1A },
        { "killedmonster",       TOKEN_KILLEDMONSTER,       0x1B },
        { "foundsecret",         TOKEN_FOUNDSECRET,         0x1C },
        { "spawnstaticsound",    TOKEN_SPAWNSTATICSOUND,    0x1D },
        { "intermission",        TOKEN_INTERMISSION,        0x1E },
        { "cdtrack",             TOKEN_CDTRACK,             0x20 },
        { "smallkick",           TOKEN_SMALLKICK,           0x22 },
        { "bigkick",             TOKEN_BIGKICK,             0x23 },
        { "updateping",          TOKEN_UPDATEPING,          0x24 },
        { "updateentertime",     TOKEN_UPDATEENTERTIME,     0x25 },
        { "updatestatlong",      TOKEN_UPDATESTATLONG,      0x26 },
        { "muzzleflash",         TOKEN_MUZZLEFLASH,         0x27 },
        { "updateuserinfo",      TOKEN_UPDATEUSERINFO,      0x28 },
        { "playerinfo",          TOKEN_PLAYERINFO,          0x2A },
        { "nails",               TOKEN_NAILS,               0x2B },
        { "choke",               TOKEN_CHOKE,               0x2C },
        { "modellist",           TOKEN_MODELLIST,           0x2D },
        { "soundlist",           TOKEN_SOUNDLIST,           0x2E },
        { "packetentities",      TOKEN_PACKETENTITIES,      0x2F },
        { "deltapacketentities", TOKEN_DELTAPACKETENTITIES, 0x2F },
        
        { NULL,                  0,                         0 }
};

typedef float fvector[3];
typedef short svector[3];

/* check, if file is QWD */
unsigned long isQWDbin(char *filename)
{
  FILE *file;
  struct stat buf;
  size_t filesize;
  unsigned char buffer[10];
  unsigned long blocksize;
  unsigned long tics;
  unsigned long code;
  
  if ((file=fopen(filename, "rb"))==NULL) return 0;
  if (stat(filename,&buf)==-1) return 0;
  filesize=buf.st_size;
  rewind(file);
  tics=0;
  while (ftell(file)<filesize) {
    tics++;
    /* 4 bytes float (time stamp) */
    if (fseek(file, 4, SEEK_CUR)!=0) return 0;
    /* code */
    if (fread(buffer, 1, 1, file)!=1) return 0;
    code=buffer[0];
    switch (code) {
      case 0:
        if (fseek(file, 0x24, SEEK_CUR) != 0) return 0;
      break;
      case 1:
        if (fread(buffer, 1, 4, file)!=4) return 0;
        blocksize = ((buffer[0])    ) +
                    ((buffer[1])<< 8) +
                    ((buffer[2])<<16) +
                    ((buffer[3])<<24);
        if (fseek(file, blocksize, SEEK_CUR)!=0) return 0;
      break;
      default:
        return 0;
      break;  
    }
  } 
  if (ftell(file)!=filesize) return 0;
  fclose(file);
  return tics;
}

void QWD_init(QWD_t* d, char *filename, char *mode)
{
  struct stat buf;
  
  if ((d->filename = strdup(filename))==NULL) syserror(errno,"strdup");
  if (strcmp(d->filename,"-")==0) {
    d->file=stdout;
  }
  else {
    QWD_open(d,mode);
    if (stat(d->filename,&buf)==-1) syserror(errno,d->filename);
    d->filesize=buf.st_size;
  }
  d->nummodels = 0;
  d->numsounds = 0;
  d->lastHealth = 100;
  d->lastArmor = 0;
  d->lastMinusHealth = 0;
  d->lastMinusArmor = 0;
  d->playermodel = 0;
  /* we need a rewind here because some fopen modes may seek at the end */
  rewind(d->file);
}
    
void QWD_open(QWD_t *d, char *mode)
{
  if ((d->file=fopen(d->filename, mode))==NULL) syserror(errno,d->filename);
}
  
void QWD_close(QWD_t *d)
{
  if (fclose(d->file)) syserror(errno,d->filename);
  d->file=NULL;
}
            
void QWD_done(QWD_t* d)
{
  QWD_close(d);
  free(d->filename);
}

void QWD_block_read_bin(QWD_t *d, BB_t* m)
{
  int code;
  
  m->base = ftell(d->file);
  if (fread(m->buffer,1,9,d->file)!=9) syserror(FIREAD,d->filename);
  m->p = m->buffer+4;
  m->end = &(m->buffer[9]);
  code = ReadByte(m);
  switch (code) {
    case 0: /* cblock */
      m->size = 36;
      if (fread(m->buffer+9,1,m->size - 4,d->file)!=m->size - 4) 
        syserror(FIREAD,d->filename);
      m->size += 5;      
    break;
    case 1: /* sblock */
      m->size = ReadLong(m);
      if (fread(m->buffer+9,1,m->size,d->file)!=m->size) 
        syserror(FIREAD,d->filename);
      m->size += 9;
    break;
    default:
      syserror(FIREAD,d->filename);
    break;
  }
  if (m->size>MAX_MESSAGE_SIZE) syserror(WQWD,d->filename);
  m->p = m->buffer;
  m->end = &(m->buffer[m->size]);
  m->top = (void *) d;
}

#define QWDTOP ((QWD_t*)(m->top))

node* QWD_bin_to_node(BB_t *m, int opt)
{
  node *n, *tn, *ttn;
  float time;
  int code;
  char ts[1000];
   
  /* binary in */
  time = ReadFloat(m);
  n = node_add_hint(
        node_command_init(TOKEN_TIME,V_TIME,H_SIMPLE,NODE_VALUE_TIME_dup(time),0),
        H_SIMPLE
      );
  code = ReadByte(m);
  switch (code) {
    case 0: /* cblock */ {
      int i;
      long load;
      fvector angles;
      svector speed;
      fvector uk_angles;
      long flag, impulse;
      
      /* binary in */
      load = ReadLong(m);
      for (i=0;i<3;i++) angles[i] = ReadFloat(m);
      for (i=0;i<3;i++) speed[i] = ReadShort(m);
      flag = ReadByte(m);
      impulse = ReadByte(m);
      for (i=0;i<3;i++) uk_angles[i] = ReadFloat(m);
      
      /* construct node tree */
      node_add_next(n, 
        node_command_init(TOKEN_LOAD, V_INT, H_SIMPLE, NODE_VALUE_INT_dup(load), 0)
      );
      node_add_next(n,
        node_triple_command_init(TOKEN_ANGLES, V_FLOAT, H_SIMPLE, 
                                 NODE_VALUE_FLOAT_dup(angles[0]),
                                 NODE_VALUE_FLOAT_dup(angles[1]),
                                 NODE_VALUE_FLOAT_dup(angles[2]),
                                 0)
      );
      node_add_next(n,
        node_triple_command_init(TOKEN_SPEED, V_INT, H_SIMPLE, 
                                 NODE_VALUE_INT_dup(speed[0]),
                                 NODE_VALUE_INT_dup(speed[1]),
                                 NODE_VALUE_INT_dup(speed[2]),
                                 0)
      );
      if (flag) {
        if (flag & 0x01) {
          node_add_next(n, node_init_all(TOKEN_FIRE, H_SIMPLE, NULL, 0));
        }
        if (flag & 0x02) {
          node_add_next(n, node_init_all(TOKEN_JUMP, H_SIMPLE, NULL, 0));
        }
        for (i=2,tn=NULL;i<8;i++) {
          if ((flag>>i) & 0x01) {
            tn=node_link(tn,node_init_all(V_INT, H_SIMPLE, 
                                          NODE_VALUE_INT_dup(i), 0));
          }
        }
        if (tn!=NULL) {
          node_add_next(n,node_init_all(TOKEN_FLAG,H_SIMPLE,tn,0));
        }
      }
      if (impulse) {
        node_add_next(n, node_command_init(TOKEN_IMPULSE, V_INT, H_SIMPLE, 
                                           NODE_VALUE_INT_dup(impulse),0)
                     );
      }
      node_add_next(n,
        node_triple_command_init(TOKEN_UK_ANGLES, V_FLOAT, H_SIMPLE, 
                                 NODE_VALUE_FLOAT_dup(uk_angles[0]),
                                 NODE_VALUE_FLOAT_dup(uk_angles[1]),
                                 NODE_VALUE_FLOAT_dup(uk_angles[2]),
                                 0)
      );
      n = node_init_all(TOKEN_CBLOCK,0,n,0);
      if (opt & 0x01) {
        sprintf(ts,"%i",QWDTOP->frame);
        node_add_comment(n,NODE_VALUE_STRING_dup(ts));
      }       
      return n;
    }
    break;
    case 1: /* sblock */ {
      unsigned long size;
      unsigned long seq1, seq2;
      int hid1, hid2;
      long id;
      
      size = ReadLong(m); 
      seq1 = ReadLong(m);
      if (seq1 == 0xFFFFFFFF) {
        id = ReadByte(m);
        switch (id) {
          case '\002': {
            char text[1000];
             
            ReadString(m,text);

            tn=node_command_init(TOKEN_DISCONNECT, V_STRING, H_STRING, 
                                 NODE_VALUE_STRING_dup(text),0);
          }
          break;
          case 0x42: { /* 'B' */
            char text[1000];
             
            ReadString(m,text);

            tn=node_command_init(TOKEN_STUFFTEXT, V_STRING, H_STRING, 
                                 NODE_VALUE_STRING_dup(text),0);
          }
          break;
          case 0x6a: { /* 'j' */
            tn=node_init(TOKEN_CONNECT, NULL, 0);
          }
          break;
          case 0x6b: { /* 'k' */
            tn=node_init(TOKEN_HALLO, NULL, 0);
          }
          case 0x6e: { /* 'n' */
            char text[1000];
             
            ReadString(m,text);

            tn=node_command_init(TOKEN_CONSOLE, V_STRING, H_STRING, 
                                 NODE_VALUE_STRING_dup(text),0);
          }
          default: {
            tn=node_command_init(TOKEN_CODE, V_BYTEHEX, H_SIMPLE, 
                                 NODE_VALUE_BYTEHEX_dup(id),0);
            for (ttn=NULL; m->p < m->end;) {
              ttn=node_link(ttn,
                   node_init_all(V_BYTEHEX,H_SIMPLE,
                                 NODE_VALUE_BYTEHEX_dup((unsigned char)ReadByte(m)),0));
            }
            if (ttn!=NULL) 
              node_add_next(tn,node_init_all(TOKEN_DATA,H_SIMPLE,ttn,0));
          }
          break;
        }
        n=node_link(n,
           node_init_all(TOKEN_SPECIAL,H_SIMPLE,tn,0)
          ); 
      }
      else {
        seq2 = ReadLong(m);
        hid1 = (seq1 >> 31);
        hid2 = (seq2 >> 31);
        seq1 &= 0x7FFFFFFF;          
        seq2 &= 0x7FFFFFFF;
        n = node_link(n,node_command_init(TOKEN_SEQ1, V_INT, H_LONG, 
                                        NODE_VALUE_INT_dup(seq1),0));
        n = node_link(n,node_command_init(TOKEN_SEQ2, V_INT, H_LONG, 
                                        NODE_VALUE_INT_dup(seq2),0));
        if (hid1 || hid2) {
          tn = NULL;
          if (hid1) {
            tn = node_link(tn, node_init_all(V_INT,H_SIMPLE,
                                             NODE_VALUE_INT_dup(1),0));
          }
          if (hid2) {
            tn = node_link(tn, node_init_all(V_INT,H_SIMPLE,
                                             NODE_VALUE_INT_dup(2),0));
          }
          node_add_next(n, node_init_all(TOKEN_HIDDEN, H_SIMPLE, tn, 0));
        }
        while (m->p < m->end) {
          id = ReadByte(m);
          switch(id) {
            case 0x02: { /* disconnect */
              /* construct node tree */
              n=node_link(n,node_init_all(TOKEN_DISCONNECT, H_SIMPLE, NULL, 0));
            }
            break;
            case 0x03: { /* updatestat */
              /* variables */
              long index; 
              long value;
              
              /* binary in */
              index = ReadByte(m);
              value = ReadByte(m);

              /* construct node tree */
              n=node_link(n,
                  node_init_all(TOKEN_UPDATESTAT, H_SIMPLE, 
                    node_add_next(
                      node_command_init(TOKEN_INDEX, V_INT, H_BYTE, 
                                        NODE_VALUE_INT_dup(index),0),
                      node_command_init(TOKEN_VALUE, V_INT, H_BYTE, 
                                        NODE_VALUE_INT_dup(value),0)
                    ), 
                    0)
                );
            }
            break;
            case 0x06: { /* sound */
              /* variables */
              float vol;
              float attenuation;
              long channel;
              long entity;
              long soundnum;
              fvector origin;

              int i;
     
              /* binary in */
              long mix;

              /* init */
              vol = attenuation = 0.0;

              mix = ReadShort(m);
              if (mix & 0x8000) vol = (float) ReadByte(m) /255.0;
              if (mix & 0x4000) attenuation = (float) ReadByte(m) / 64.0;
              channel = mix & 0x07;
              entity = (mix >> 3) & 0x03FF;
              soundnum = ReadByte(m);
              for (i=0 ; i<3 ; i++) origin[i] = ReadCoord(m);

              /* construct node tree */
              tn = NULL;
              tn=node_link(tn,node_command_init(TOKEN_ENTITY,V_INT,0,
                              NODE_VALUE_INT_dup(entity),0));
              tn=node_link(tn,node_triple_command_init(TOKEN_ORIGIN,V_FLOAT,H_COORD,
                              NODE_VALUE_FLOAT_dup(origin[0]),
                              NODE_VALUE_FLOAT_dup(origin[1]),
                              NODE_VALUE_FLOAT_dup(origin[2]),0));
              tn=node_link(tn,node_command_init(TOKEN_CHANNEL,V_INT,0,
                              NODE_VALUE_INT_dup(channel),0));
              ttn=node_command_init(TOKEN_SOUNDNUM,V_INT,H_BYTE,
                  NODE_VALUE_INT_dup(soundnum),0);
              if (soundnum >= 1 && soundnum <= QWDTOP->numsounds) {
                node_add_comment(ttn,NODE_VALUE_STRING_dup(QWDTOP->precache_sounds[soundnum]));
              }
              tn=node_link(tn,ttn);
              if (mix & 0x8000) {
                tn=node_link(tn,node_command_init(TOKEN_VOL,V_FLOAT,H_VOL,
                                                  NODE_VALUE_FLOAT_dup(vol),0));
              }           
              if (mix & 0x4000) {
                tn=node_link(tn,node_command_init(TOKEN_ATTENUATION,V_FLOAT,H_ATTENUATION,
                                NODE_VALUE_FLOAT_dup(attenuation),0));
              }
              n=node_link(n,node_init_all(TOKEN_SOUND,0,tn,0));
            }
            break;
            case 0x08: { /* print */
              long level; 
              char string[1000];
              
              level = ReadByte(m);;
              ReadString(m,string);
              
              n=node_link(n,
                  node_init_all(TOKEN_PRINT, H_SIMPLE, 
                    node_add_next(
                      node_command_init(TOKEN_LEVEL, V_INT, H_BYTE, 
                                        NODE_VALUE_INT_dup(level),0),
                      node_command_init(TOKEN_STRING, V_STRING, H_STRING, 
                                        NODE_VALUE_STRING_dup(string),0)
                    ), 
                    0)
                );
            }
            break;
            case 0x09: { /* stufftext */
              char text[1000];
              
              ReadString(m,text);
              
              n=node_link(n,node_command_init(TOKEN_STUFFTEXT, V_STRING, H_STRING,
                                              NODE_VALUE_STRING_dup(text),0)
                         );
            }
            break;
            case 0x0A: { /* setangle */ 
              /* variables */
              fvector angles;
              
              int i;
              
              /* binary in */
              for (i=0;i<3;i++) angles[i]=ReadAngle(m);
              
              /* construct node tree */              
              n=node_link(n,
              node_triple_command_init(TOKEN_SETANGLE, V_FLOAT, H_ANGLE, 
                                       NODE_VALUE_FLOAT_dup(angles[0]),
                                       NODE_VALUE_FLOAT_dup(angles[1]),
                                       NODE_VALUE_FLOAT_dup(angles[2]),0));
            }
            break;
            case 0x0B: { /* serverdata */
              
              long version;
              long age;
              long client;
              char dir[1000];
              char mapname[1000];
              long uk_bytes[40];
              
              int i;
              
              version = ReadLong(m);
              age = ReadLong(m);
              ReadString(m, dir);
              client = ReadByte(m);
              ReadString(m, mapname);
              if (version == 25) { /* QW 2.00 */
                for (i=0;i<40;i++) {
                  uk_bytes[i] = ReadByte(m);
                }
              }
              tn = node_command_init(TOKEN_VERSION, V_INT, H_LONG, 
                                     NODE_VALUE_INT_dup(version),0);
              tn = node_link(tn, 
                     node_command_init(TOKEN_AGE, V_INT, H_LONG, 
                                       NODE_VALUE_INT_dup(age),0)
                   );
              tn = node_link(tn, 
                     node_command_init(TOKEN_DIR, V_STRING, H_STRING,
                                       NODE_VALUE_STRING_dup(dir),0)
                   );
              tn = node_link(tn, 
                     node_command_init(TOKEN_CLIENT, V_INT, H_BYTE, 
                                       NODE_VALUE_INT_dup(client),0)
                   );
              tn = node_link(tn, 
                     node_command_init(TOKEN_MAPNAME, V_STRING, H_STRING,
                                       NODE_VALUE_STRING_dup(mapname),0)
                   );
              if (version == 25) { /* QW 2.00 */
                for (i=0,ttn=0;i<40;i++) {
                  ttn=node_link(ttn,node_init_all(V_BYTEHEX, H_SIMPLE,
                                                  NODE_VALUE_INT_dup(uk_bytes[i]),0));
                }
                tn=node_link(tn,node_init_all(TOKEN_DATA,H_SIMPLE,ttn,0));
              }
              n=node_link(n,node_init_all(TOKEN_SERVERDATA, H_SIMPLE, tn, 0));
            }
            break;
            case 0x0C: { /* lightstyle */
              /* variables */
              long style; 
              char string[1000];
              
              /* binary in */
              style = ReadByte(m);
              ReadString(m,string);

              /* construct node tree */
              n=node_link(n,
                  node_init_all(TOKEN_LIGHTSTYLE, H_SIMPLE, 
                    node_add_next(
                      node_command_init(TOKEN_STYLE, V_INT, H_BYTE, 
                                        NODE_VALUE_INT_dup(style),0),
                      node_command_init(TOKEN_STRING, V_STRING, H_STRING, 
                                        NODE_VALUE_STRING_dup(string),0)
                    ), 
                    0)
                );
            }
            break;
            case 0x0E: { /* updatefrags */  
              /* variables */
              long player;
              long frags;

              /* binary in */
              player = ReadByte(m);
              frags = ReadShort(m);

              /* construct node tree */
              n=node_link(n,
                  node_init_all(TOKEN_UPDATEFRAGS,H_SIMPLE,
                    node_add_next(  
                      node_command_init(TOKEN_PLAYER,V_INT,H_BYTE,
                                        NODE_VALUE_INT_dup(player),0),
                      node_command_init(TOKEN_FRAGS,V_INT,
                                        H_SHORT,NODE_VALUE_INT_dup(frags),0)
                    ),
                    0)
                );
            }
            break;
            case 0x13: { /* damage */
              /* variables */
              long save;
              long take;
              fvector origin;
                    
              int i;
                      
              /* binary in */
              save = ReadByte(m);
              take = ReadByte(m);
              for (i=0 ; i<3 ; i++) origin[i] = ReadCoord(m);
            
              /* add */
              QWDTOP->lastMinusArmor = save;
              QWDTOP->lastMinusHealth = take;

              /* construct node tree */
              n=node_link(n,
              node_init_all(TOKEN_DAMAGE,H_SIMPLE,
                node_link(
                  node_link(
                    node_command_init(TOKEN_SAVE, V_INT, H_BYTE, 
                                      NODE_VALUE_INT_dup(save),0)
                  ,
                    node_command_init(TOKEN_TAKE, V_INT, H_BYTE, 
                                      NODE_VALUE_INT_dup(take),0)
                  )
                ,
                  node_triple_command_init(TOKEN_ORIGIN,V_FLOAT,H_COORD,
                                           NODE_VALUE_FLOAT_dup(origin[0]),
                                           NODE_VALUE_FLOAT_dup(origin[1]),
                                           NODE_VALUE_FLOAT_dup(origin[2]),
                                           0)
                ),0
              ));
            }
            break;
            case 0x14: { /* spawnstatic */
              /* variables */
              long modelindex;
              long frame;
              long colormap;
              long skin;
              fvector origin;
              fvector angles;

              int i;

              /* binary in */
              modelindex = ReadByte(m);
              frame = ReadByte(m);
              colormap = ReadByte(m);
              skin = ReadByte(m);
              for (i=0 ; i<3 ; i++) {
                origin[i] = ReadCoord(m);
                angles[i] = ReadAngle(m); 
              }

              /* construct node tree and return the root of it */
              tn = NULL;              
              ttn=node_command_init(TOKEN_MODELINDEX,V_INT,H_BYTE,
                                    NODE_VALUE_INT_dup(modelindex),0);
              if (modelindex >= 1 &&
                  modelindex <= QWDTOP->nummodels &&
                  QWDTOP->precache_models[modelindex][0]!='*') {
                node_add_comment(ttn,NODE_VALUE_STRING_dup(QWDTOP->precache_models[modelindex]));
              }
              tn=node_link(tn,ttn);
              tn=node_link(tn,node_command_init(TOKEN_FRAME,V_INT,H_BYTE,NODE_VALUE_INT_dup(frame),0));
              tn=node_link(tn,node_command_init(TOKEN_COLORMAP,V_INT,H_BYTE,NODE_VALUE_INT_dup(colormap),0));
              tn=node_link(tn,node_command_init(TOKEN_SKIN,V_INT,H_BYTE,NODE_VALUE_INT_dup(skin),0));
              tn=node_link(tn,
                   node_triple_command_init(TOKEN_ORIGIN,V_FLOAT,H_COORD,
                     NODE_VALUE_FLOAT_dup(origin[0]),
                     NODE_VALUE_FLOAT_dup(origin[1]),
                     NODE_VALUE_FLOAT_dup(origin[2]),0));
              tn=node_link(tn,
                   node_triple_command_init(TOKEN_ANGLES,V_FLOAT,H_COORD,
                     NODE_VALUE_FLOAT_dup(angles[0]),
                     NODE_VALUE_FLOAT_dup(angles[1]),
                     NODE_VALUE_FLOAT_dup(angles[2]),0));
              n=node_link(n,node_init_all(TOKEN_SPAWNSTATIC,H_QWD_SPAWNSTATIC,tn,0));
            }
            break;
            case 0x16: { /* spawnbaseline */
              /* variables */
              long entity;
              long default_modelindex;
              long default_frame;
              long default_colormap;
              long default_skin;
              fvector default_origin;
              fvector default_angles;

              int i;

              /* binary in */
              entity = ReadEntity(m);
              default_modelindex = ReadByte(m);
              default_frame = ReadByte(m);
              default_colormap = ReadByte(m);
              default_skin = ReadByte(m);
              for (i=0 ; i<3 ; i++) {
                default_origin[i] = ReadCoord(m);
                default_angles[i] = ReadAngle(m); 
              }

              /* construct node tree and return the root of it */
              tn = NULL;
              tn=node_link(tn,node_command_init(TOKEN_ENTITY,V_INT,H_SHORT,
                              NODE_VALUE_INT_dup(entity),0));
              ttn=node_command_init(TOKEN_DEFAULT_MODELINDEX,V_INT,H_BYTE,
                                    NODE_VALUE_INT_dup(default_modelindex),0);
              if (default_modelindex >= 1 &&
                  default_modelindex <= QWDTOP->nummodels &&
                  QWDTOP->precache_models[default_modelindex][0]!='*') {
                node_add_comment(ttn,NODE_VALUE_STRING_dup(QWDTOP->precache_models[default_modelindex]));
              }
              tn=node_link(tn,ttn);
              tn=node_link(tn,node_command_init(TOKEN_DEFAULT_FRAME,V_INT,H_BYTE,NODE_VALUE_INT_dup(default_frame),0));
              tn=node_link(tn,node_command_init(TOKEN_DEFAULT_COLORMAP,V_INT,H_BYTE,NODE_VALUE_INT_dup(default_colormap),0));
              tn=node_link(tn,node_command_init(TOKEN_DEFAULT_SKIN,V_INT,H_BYTE,NODE_VALUE_INT_dup(default_skin),0));
              tn=node_link(tn,
                   node_triple_command_init(TOKEN_DEFAULT_ORIGIN,V_FLOAT,H_COORD,
                     NODE_VALUE_FLOAT_dup(default_origin[0]),
                     NODE_VALUE_FLOAT_dup(default_origin[1]),
                     NODE_VALUE_FLOAT_dup(default_origin[2]),0));
              tn=node_link(tn,
                   node_triple_command_init(TOKEN_DEFAULT_ANGLES,V_FLOAT,H_COORD,
                     NODE_VALUE_FLOAT_dup(default_angles[0]),
                     NODE_VALUE_FLOAT_dup(default_angles[1]),
                     NODE_VALUE_FLOAT_dup(default_angles[2]),0));
              n=node_link(n,node_init_all(TOKEN_SPAWNBASELINE,H_QWD_SPAWNBASELINE,tn,0));
            }
            break;
            case 0x17: { /* temp_entity */
              /* variables */
              long entitytype;
              long entity;
              vec3_t origin;
              vec3_t trace_endpos;
              long count;
              
              int i;
              
              /* init */
              entity = 0; count = 0;

              /* binary in */
              entitytype = ReadByte(m);
              switch (entitytype) {
                case 0:
                case 1:
                case 3:
                case 4:
                case 7:
                case 8:
                case 10:
                case 11:
                case 13:
                  for (i=0 ; i<3 ; i++) origin[i] = ReadCoord(m);
                break; 
                case 5:
                case 6:
                case 9:
                  entity = ReadEntity(m);
                  for (i=0 ; i<3 ; i++) origin[i] = ReadCoord(m);
                  for (i=0 ; i<3 ; i++) trace_endpos[i] = ReadCoord(m);
                break;
                case 2:
                case 12:
                  count = ReadByte(m);
                  for (i=0 ; i<3 ; i++) origin[i] = ReadCoord(m);
                break;
              }          
              
              /* construct node tree */
              tn = NULL;              
              tn = node_link(tn,node_command_init(TOKEN_ENTITYTYPE,V_INT,H_BYTE,
                                NODE_VALUE_INT_dup(entitytype),0));
              switch (entitytype) {
                case 0:
                case 1:
                case 3:
                case 4:
                case 7:
                case 8:
                case 10:
                case 11:
                case 13:
                  tn=node_link(tn,node_triple_command_init(TOKEN_ORIGIN,V_FLOAT,H_COORD,
                                  NODE_VALUE_FLOAT_dup(origin[0]),
                                  NODE_VALUE_FLOAT_dup(origin[1]),
                                  NODE_VALUE_FLOAT_dup(origin[2]),0));
                break;
                case 5:
                case 6:
                case 9:
                  tn=node_link(tn,node_command_init(TOKEN_ENTITY,V_INT,H_SHORT,
                                  NODE_VALUE_INT_dup(entity),0));
                  tn=node_link(tn,node_triple_command_init(TOKEN_ORIGIN,V_FLOAT,H_COORD,
                                  NODE_VALUE_FLOAT_dup(origin[0]),
                                  NODE_VALUE_FLOAT_dup(origin[1]),
                                  NODE_VALUE_FLOAT_dup(origin[2]),0));
                  tn=node_link(tn,node_triple_command_init(TOKEN_TRACE_ENDPOS,V_FLOAT,H_COORD,
                                  NODE_VALUE_FLOAT_dup(trace_endpos[0]),
                                  NODE_VALUE_FLOAT_dup(trace_endpos[1]),
                                  NODE_VALUE_FLOAT_dup(trace_endpos[2]),0));
                break;
                case 2:
                case 12:
                  tn=node_link(tn,node_command_init(TOKEN_COUNT,V_INT,H_BYTE,
                                  NODE_VALUE_INT_dup(count),0));
                  tn=node_link(tn,node_triple_command_init(TOKEN_ORIGIN,V_FLOAT,H_COORD,
                                  NODE_VALUE_FLOAT_dup(origin[0]),
                                  NODE_VALUE_FLOAT_dup(origin[1]),
                                  NODE_VALUE_FLOAT_dup(origin[2]),0));
                break;
              }
              n=node_link(n,node_init_all(TOKEN_TEMP_ENTITY,0,tn,0));
            }
            break;
            case 0x1A: { /* centerprint */
              char text[1000];
              
              ReadString(m,text);
              
              n=node_link(n,node_command_init(TOKEN_CENTERPRINT, V_STRING, H_STRING,
                                              NODE_VALUE_STRING_dup(text),0)
                         );
            }
            break;
            case 0x1B: { /* killedmonster */
              /* construct node tree */
              n=node_link(n,node_init_all(TOKEN_KILLEDMONSTER, H_SIMPLE, NULL,0));
            }
            break;
            case 0x1C: { /* foundsecret */
              /* construct node tree */
              n=node_link(n,node_init_all(TOKEN_FOUNDSECRET, H_SIMPLE, NULL,0));
            }
            break;
            case 0x1D: { /* spawnstaticsound */
              /* variables */
              fvector origin;
              long soundnum;
              float vol;
              float attenuation;

              int i;

              /* binary in */
              for (i=0 ; i<3 ; i++) origin[i] = ReadCoord(m);
              soundnum = ReadByte(m);
              vol = (float) ReadByte(m) / 255.0;
              attenuation = (float) ReadByte(m) / 64.0;

              /* construct node tree and return the root of it */
              tn = node_triple_command_init(TOKEN_ORIGIN,V_FLOAT,H_COORD,
                     NODE_VALUE_FLOAT_dup(origin[0]),
                     NODE_VALUE_FLOAT_dup(origin[1]),
                     NODE_VALUE_FLOAT_dup(origin[2]),0);
              ttn = node_command_init(TOKEN_SOUNDNUM,V_INT,H_BYTE,
                                      NODE_VALUE_INT_dup(soundnum),0);
              if (soundnum >= 1 && soundnum <= QWDTOP->numsounds) {
                node_add_comment(ttn,NODE_VALUE_STRING_dup(QWDTOP->precache_sounds[soundnum]));
              }
              tn=node_link(tn,ttn);
              tn=node_link(tn,node_command_init(TOKEN_VOL,V_FLOAT,H_VOL,
                              NODE_VALUE_FLOAT_dup(vol),0));
              tn=node_link(tn,node_command_init(TOKEN_ATTENUATION,V_FLOAT,
                              H_ATTENUATION,NODE_VALUE_FLOAT_dup(attenuation),0));
              n=node_link(n,node_init_all(TOKEN_SPAWNSTATICSOUND,0,tn,0));
            }
            break;
            case 0x1E: { /* intermission */
              /* variables */
              
              fvector origin;
              fvector angles;
              
              int i;
              
              /* binary in */
              for ( i=0 ; i<3 ; i++) origin[i] = ReadCoord(m);
              for ( i=0 ; i<3 ; i++) angles[i] = ReadAngle(m);
              
              /* construct node tree */
              n=node_link(n,
              node_init_all(TOKEN_INTERMISSION,H_SIMPLE,
                node_link(
                  node_triple_command_init(TOKEN_ORIGIN,V_FLOAT,H_COORD,
                                           NODE_VALUE_FLOAT_dup(origin[0]),
                                           NODE_VALUE_FLOAT_dup(origin[1]),
                                           NODE_VALUE_FLOAT_dup(origin[2]),
                                           0)
                ,
                  node_triple_command_init(TOKEN_ANGLES,V_FLOAT,H_ANGLE,
                                           NODE_VALUE_FLOAT_dup(angles[0]),
                                           NODE_VALUE_FLOAT_dup(angles[1]),
                                           NODE_VALUE_FLOAT_dup(angles[2]),
                                           0)
                ),0));                                             
            }
            break;
            case 0x20: { /* cdtrack */
              long track;
              
              track = ReadByte(m);
              
              n = node_link(n,node_command_init(TOKEN_CDTRACK, V_INT, H_BYTE,
                              NODE_VALUE_INT_dup(track), 0));
            }
            break;
            case 0x22: { /* smallkick */
              /* construct node tree */
              n=node_link(n,node_init_all(TOKEN_SMALLKICK, H_SIMPLE, NULL,0));
            }
            break;
            case 0x23: { /* bigkick */
              /* construct node tree */
              n=node_link(n,node_init_all(TOKEN_BIGKICK, H_SIMPLE, NULL,0));
            }
            break;
            case 0x24: { /* updateping */
              /* variables */
              long player;
              long ping;

              /* binary in */
              player = ReadByte(m);
              ping = ReadShort(m);

              tn=NULL;
              tn=node_link(tn,
                   node_command_init(TOKEN_PLAYER, V_INT, H_BYTE, 
                                     NODE_VALUE_INT_dup(player), 0));
              tn=node_link(tn,
                   node_command_init(TOKEN_PING, V_INT, H_SHORT, 
                                     NODE_VALUE_INT_dup(ping), 0));
              n=node_link(n,node_init_all(TOKEN_UPDATEPING, H_SIMPLE, tn, 0));
            }
            break;
            case 0x25: { /* updateentertime */
              /* variables */
              long player;
              float entertime;

              /* binary in */
              player = ReadByte(m);
              entertime = ReadFloat(m);

              /* construct node tree */
              n=node_link(n,
                  node_init_all(TOKEN_UPDATEENTERTIME, H_SIMPLE, 
                    node_add_next(
                      node_command_init(TOKEN_PLAYER, V_INT, H_BYTE, 
                                        NODE_VALUE_INT_dup(player),0),
                      node_command_init(TOKEN_ENTERTIME, V_TIME, H_SIMPLE, 
                                        NODE_VALUE_TIME_dup(entertime),0)
                    ), 
                    0)
                );
            }
            break;
            case 0x26: { /* updatestatlong */
              /* variables */
              long index; 
              long value;
              
              /* binary in */
              index = ReadByte(m);
              value = ReadLong(m);

              /* construct node tree */
              n=node_link(n,
                  node_init_all(TOKEN_UPDATESTATLONG, H_SIMPLE, 
                    node_add_next(
                      node_command_init(TOKEN_INDEX, V_INT, H_BYTE, 
                                        NODE_VALUE_INT_dup(index),0),
                      node_command_init(TOKEN_VALUE, V_INT, H_LONG, 
                                        NODE_VALUE_INT_dup(value),0)
                    ), 
                    0)
                );
            }
            break;
            case 0x27: { /* muzzleflash */
              /* variables */
              long entity;
              
              /* binary in */
              entity = ReadEntity(m);
              
              /* construct node tree */
              n = node_link(n,node_command_init(TOKEN_MUZZLEFLASH, V_INT, H_SHORT,
                              NODE_VALUE_INT_dup(entity), 0));
            }
            break;
            case 0x28: { /* updateuserinfo */
              /* variables */
              long player;
              long user;
              char string[1000];

              /* binary in */
              player = ReadByte(m);
              user = ReadLong(m);
              ReadString(m,string);
              
              tn=NULL;
              tn=node_link(tn,
                   node_command_init(TOKEN_PLAYER, V_INT, H_BYTE, 
                                     NODE_VALUE_INT_dup(player), 0));
              tn=node_link(tn,
                   node_command_init(TOKEN_USER, V_INT, H_LONG, 
                                     NODE_VALUE_INT_dup(user), 0));
              tn=node_link(tn,
                   node_command_init(TOKEN_STRING, V_STRING, H_STRING, 
                                     NODE_VALUE_STRING_dup(string), 0));
              n=node_link(n,node_init_all(TOKEN_UPDATEUSERINFO, H_SIMPLE, tn, 0));
            }
            break;
            case 0x2A: { /* playerinfo */
              /* variables */
              long player;
              long mask;
              fvector origin;
              long frame;
              float ping = 0.0;
              fvector cspeed;
              long model;
              long uk_byte6;
              long weapon;
              long weaponframe;
              
              long mask2=0;
              long load=0;
              fvector angles;
              svector speed;
              long flag=0;
              long impulse=0;
                            
              int i, j;
              
              /* binary in */
              player = ReadByte(m);
              mask = ReadShort(m);
              for (i=0;i<3;i++) origin[i] = ReadCoord(m);
              frame = ReadByte(m);
              /* bit 0 */
              if (mask & 0x0001) {
                ping=(ReadByte(m) * 0.001);
              }
              /* bit 1 */
              if (mask & 0x0002) {
                mask2 = ReadByte(m);
                
                /* bit 0 */
                angles[0] = (mask2 & 0x01) ? ReadAngle16(m) : 0;
                angles[1] = ReadAngle16(m);
                /* bit 1 */
                angles[2] = (mask2 & 0x02) ? ReadAngle16(m) : 0;
                /* bit 2 */
                speed[0] = (mask2 & 0x04) ? ReadByte(m) : 0;
                /* bit 3 */
                speed[1] = (mask2 & 0x08) ? ReadByte(m) : 0;
                /* bit 4 */
                speed[2] = (mask2 & 0x10) ? ReadByte(m) : 0;
                /* bit 5 */
                flag = (mask2 & 0x20) ? ReadByte(m): 0;
                /* bit 6 */
                impulse = (mask2 & 0x40) ? ReadByte(m): 0;
                /* bit 7 */
                load = (mask2 & 0x80) ? ReadByte(m): 0;
              }
              /* bit 2 */
              cspeed[0] = (mask & 0x0004) ? ReadCoord(m) : 0 ;
              /* bit 3 */
              cspeed[1] = (mask & 0x0008) ? ReadCoord(m) : 0 ;
              /* bit 4 */
              cspeed[2] = (mask & 0x0010) ? ReadCoord(m) : 0 ;
              /* bit 5 */
              model = (mask & 0x0020) ? ReadByte(m) : QWDTOP->playermodel;
              /* bit 6 */
              uk_byte6 = (mask & 0x0040) ? ReadByte(m) : 0;
              /* bit 7 */
              weapon = (mask & 0x0080) ? ReadByte(m) : 0;
              /* bit 8 */
              weaponframe = (mask & 0x0100) ? ReadByte(m) : 0;

              /* construct node tree */
              tn=NULL;
              tn=node_link(tn,node_command_init(TOKEN_PLAYER, V_INT, H_BYTE, 
                                                NODE_VALUE_INT_dup(player), 0));
              tn=node_link(tn,node_triple_command_init(TOKEN_ORIGIN, V_FLOAT, H_COORD,
                                          NODE_VALUE_FLOAT_dup(origin[0]),
                                          NODE_VALUE_FLOAT_dup(origin[1]),
                                          NODE_VALUE_FLOAT_dup(origin[2]),
                                          0));
              tn=node_link(tn,node_command_init(TOKEN_FRAME, V_INT, H_BYTE, 
                                                NODE_VALUE_INT_dup(frame), 0));
              /* bit 0 */
              if (mask & 0x0001) {
                tn=node_link(tn,node_command_init(TOKEN_PING, V_FLOAT, H_SIMPLE, 
                                                  NODE_VALUE_FLOAT_dup(ping), 0));
              }
              /* bit 1 */
              if (mask & 0x0002) {

                ttn=NULL;
                /* bit 0 */
                if (mask2 & 0x01) {
                  ttn=node_link(ttn,node_command_init(TOKEN_ANGLES_1, V_FLOAT, H_SIMPLE,
                                    NODE_VALUE_FLOAT_dup(angles[0]),0));
                }
                ttn=node_link(ttn,node_command_init(TOKEN_ANGLES_2, V_FLOAT, H_SIMPLE,
                                    NODE_VALUE_FLOAT_dup(angles[1]),0));
                /* bit 1 */
                if (mask2 & 0x02) {
                  ttn=node_link(ttn,node_command_init(TOKEN_ANGLES_3, V_FLOAT, H_SIMPLE,
                                    NODE_VALUE_FLOAT_dup(angles[2]),0));
                }
                /* bit 2 */
                if (mask2 & 0x04) {
                  ttn=node_link(ttn,node_command_init(TOKEN_SPEEDX, V_INT, H_SIMPLE,
                                    NODE_VALUE_INT_dup(speed[0]),0));
                }
                /* bit 3 */
                if (mask2 & 0x080) {
                  ttn=node_link(ttn,node_command_init(TOKEN_SPEEDY, V_INT, H_SIMPLE,
                                    NODE_VALUE_INT_dup(speed[1]),0));
                }
                /* bit 4 */
                if (mask2 & 0x10) {
                  ttn=node_link(ttn,node_command_init(TOKEN_SPEEDZ, V_INT, H_SIMPLE,
                                    NODE_VALUE_INT_dup(speed[2]),0));
                }
                /* bit 5 */
                if (mask2 & 0x20) {
                  if (flag & 0x01) {
                    ttn=node_link(ttn, node_init_all(TOKEN_FIRE, H_SIMPLE, NULL, 0));
                  }
                  if (flag & 0x02) {
                    ttn=node_link(ttn, node_init_all(TOKEN_JUMP, H_SIMPLE, NULL, 0));
                  }
                  { 
                    node* tttn;
                  
                    for (i=2,tttn=NULL;i<8;i++) {
                      if ((flag>>i) & 0x01) {
                        tttn=node_link(tttn,node_init_all(V_INT, H_SIMPLE, 
                                                    NODE_VALUE_INT_dup(i), 0));
                      }
                    }
                    if (tttn!=NULL) {
                      ttn=node_link(ttn,node_init_all(TOKEN_FLAG,H_SIMPLE,tttn,0));
                    }
                  }
                }
                /* bit 6 */
                if (mask2 & 0x40) {
                  ttn=node_link(ttn,
                    node_command_init(TOKEN_IMPULSE, V_INT, H_SIMPLE, 
                                      NODE_VALUE_INT_dup(impulse),0));
                }
                /* bit 7 */
                if (mask2 & 0x80) {
                  ttn=node_link(ttn, 
                    node_command_init(TOKEN_LOAD, V_INT, H_SIMPLE, 
                                      NODE_VALUE_INT_dup(load), 0));
                }
                tn=node_link(tn,node_init_all(TOKEN_PLAYER, 0, ttn, 0)); 
              }
              /* bit 2 */
              if (mask & 0x0004) {
                tn=node_link(tn,node_command_init(TOKEN_CSPEEDX, V_FLOAT, H_COORD, 
                                                  NODE_VALUE_FLOAT_dup(cspeed[0]), 0));
              }
              /* bit 3 */
              if (mask & 0x0008) {
                tn=node_link(tn,node_command_init(TOKEN_CSPEEDY, V_FLOAT, H_COORD, 
                                                  NODE_VALUE_FLOAT_dup(cspeed[1]), 0));
              }
              /* bit 4 */
              if (mask & 0x0010) {
                tn=node_link(tn,node_command_init(TOKEN_CSPEEDZ, V_FLOAT, H_COORD, 
                                                  NODE_VALUE_FLOAT_dup(cspeed[2]), 0));
              }
              /* bit 5 */
              if (mask & 0x0020) {
                tn=node_link(tn,node_command_init(TOKEN_MODEL, V_INT, H_BYTE, 
                                                  NODE_VALUE_INT_dup(model), 0));
              }
              /* bit 6 */
              if (mask & 0x0040) {
                tn=node_link(tn,node_command_init(TOKEN_BYTE6, V_INT, H_BYTE, 
                                                  NODE_VALUE_INT_dup(uk_byte6), 0));
              }
              /* bit 7 */
              if (mask & 0x0080) {
                if (weapon) {
                  for (i=2, j=weapon, ttn=NULL; j; j>>=1, i++) {
                    if (j&0x01) ttn=node_link(ttn,node_init_all(V_INT,H_BYTE,
                                                  NODE_VALUE_INT_dup(i),0));
                  }
                  tn=node_link(tn,node_init_all(TOKEN_WEAPON,H_SIMPLE,ttn,0));
                }
                else {
                  tn=node_link(tn, node_command_init(TOKEN_WEAPON,V_INT,H_BYTE,
                                                     NODE_VALUE_INT_dup(1),0));  
                }                                    
              }
              /* bit 8 */
              if (mask & 0x0100) {
                tn=node_link(tn,node_command_init(TOKEN_WEAPONFRAME, V_INT, H_BYTE, 
                                                  NODE_VALUE_INT_dup(weaponframe), 0));
              }
              n=node_link(n,node_init_all(TOKEN_PLAYERINFO, 0, tn, 0));
            }
            break;
            case 0x2B: { /* nails */
              /* variables */
              long count;
              svector origin;
              float angle_1;
              float angle_2;
              
              int i, j;
              unsigned char b[5];
              
              /* binary in and construct node tree */
              count = ReadByte(m);
              tn = NULL;
              for (j=0;j<count;j++) {
                ttn = NULL;
                for (i=0;i<5;i++) b[i] = ReadByte(m);
                origin[0] = (b[0] & 0xFF) | ((b[1] & 0x0F) << 8);
                origin[1] = ((b[1] & 0xF0) >> 4) | (b[2] << 4);
                origin[2] = (b[3] & 0xFF) | ((b[4] & 0x0F) << 8);
                for (i=0;i<3;i++) origin[i] = (origin[i] - 2048) * 2;
                angle_1 = (b[4] & 0xF0) >> 4;
                if (angle_1>=8) angle_1 = angle_1 - 16;
                angle_1 *= 180.0 / 8.0;
                angle_2 = ReadAngle(m);
                ttn = node_link(ttn, node_triple_command_init(TOKEN_ORIGIN, V_INT, H_SHORT,
                                     NODE_VALUE_INT_dup(origin[0]),
                                     NODE_VALUE_INT_dup(origin[1]),
                                     NODE_VALUE_INT_dup(origin[2]),0));
                ttn = node_link(ttn, node_command_init(TOKEN_ANGLES_1, V_FLOAT, H_SIMPLE, 
                                     NODE_VALUE_FLOAT_dup(angle_1),0));
                ttn = node_link(ttn, node_command_init(TOKEN_ANGLES_2, V_FLOAT, H_SIMPLE, 
                                     NODE_VALUE_FLOAT_dup(angle_2),0));
                tn = node_link(tn, node_init_all(TOKEN_NAIL, 0 , ttn, 0));
              }
              n = node_link(n,node_init_all(TOKEN_NAILS, 0, tn, 0)); 
            }
            break;
            case 0x2C: { /* choke */
              /* variables */
              long choke;
              
              /* binaray in */
              choke = ReadByte(m);
              
              /* construct node tree */
              n=node_link(n,node_command_init(TOKEN_CHOKE,V_INT, H_BYTE, 
                                              NODE_VALUE_INT_dup(choke), 0)); 
            }
            break;
            case 0x2D: { /* modellist */
              char text[1000];
              int i;

              /* free the old strings (multiple level recordings) */
              for (i=1;i<=QWDTOP->nummodels;i++) free(QWDTOP->precache_models[i]); 
              QWDTOP->nummodels = 0;
              
              /* binary in */
              do {
                ReadString(m, text);
                if (strlen(text)==0) break;
                if (QWDTOP->nummodels<QWD_MAX_SOUNDS)
                  QWDTOP->nummodels++;
                else
                  syserror(WQWD, QWDTOP->filename);
                QWDTOP->precache_models[QWDTOP->nummodels] = strdup(text);
                if (QWDTOP->precache_models[QWDTOP->nummodels]==NULL) 
                  syserror(errno, "strdup");
                if (strcmp(text,"progs/player.mdl")==0) {
                  QWDTOP->playermodel=QWDTOP->nummodels;
                }
              } while (*text);
              
              /* construct node tree */
              for (i=1,tn=NULL; i<=QWDTOP->nummodels; i++) {
                tn=node_link(tn,
                     node_init_all(V_STRING, H_STRINGLIST,
                       NODE_VALUE_STRING_dup(QWDTOP->precache_models[i]),0)
                   );
              }
              n=node_link(n,node_init_all(TOKEN_MODELLIST, 0, tn, 0)); 
            }
            break;
            case 0x2E: { /* soundlist */
              char text[1000];
              int i;

              /* free the old strings (multiple level recordings) */
              for (i=1;i<=QWDTOP->numsounds;i++) free(QWDTOP->precache_sounds[i]); 
              QWDTOP->numsounds = 0;
              
              /* binary in */
              do {
                ReadString(m, text);
                if (strlen(text)==0) break;
                if (QWDTOP->numsounds<QWD_MAX_SOUNDS)
                  QWDTOP->numsounds++;
                else
                  syserror(WQWD, QWDTOP->filename);
                QWDTOP->precache_sounds[QWDTOP->numsounds] = strdup(text);
                if (QWDTOP->precache_sounds[QWDTOP->numsounds]==NULL) 
                  syserror(errno, "strdup");
              } while (*text);
              
              /* construct node tree */
              for (i=1,tn=NULL; i<=QWDTOP->numsounds; i++) {
                tn=node_link(tn,
                     node_init_all(V_STRING,H_STRINGLIST,
                       NODE_VALUE_STRING_dup(QWDTOP->precache_sounds[i]),0)
                   );
              }
              n=node_link(n,node_init_all(TOKEN_SOUNDLIST, 0, tn, 0)); 
            }
            break;
            case 0x2F: { /* packetentities */
              /* variables */
              long entity;
              long mask;
              long modelindex;
              long frame;
              long colormap;
              long skin;
              long effects;
              vec3_t origin;
              vec3_t angles;
                
              node *q1;
              /*
              unsigned char *pold, *pnew;
              */
              
              /* binary in and node tree */              
              tn = NULL;
              
              /* pold=m->p; */ 
              while ((mask=ReadShort(m)) != 0x0000) {
                entity = mask & 0x01FF;
                mask &= 0xFE00;
                /*  if (mask & 0x4000) { } */
                if (mask & 0x8000) mask |= ReadByte(m);
                modelindex = mask & 0x0004 ? ReadByte(m) : 0;
                frame = mask & 0x2000 ? ReadByte(m) : 0;
                colormap = mask & 0x0008 ? ReadByte(m) : 0;
                skin = mask & 0x0010 ? ReadByte(m) : 0;
                effects = mask & 0x0020 ? ReadByte(m) : 0;
                origin[0] = mask & 0x0200 ? ReadCoord(m) : 0;
                angles[0] = mask & 0x0001 ? ReadAngle(m) : 0;
                origin[1] = mask & 0x0400 ? ReadCoord(m) : 0;
                angles[1] = mask & 0x1000 ? ReadAngle(m) : 0;
                origin[2] = mask & 0x0800 ? ReadCoord(m) : 0;
                angles[2] = mask & 0x0002 ? ReadAngle(m) : 0;

                ttn=NULL;
                
                if (entity!=0) {
                  ttn=node_link(ttn,node_command_init(TOKEN_ENTITY,V_INT,H_BYTE,
                                       NODE_VALUE_INT_dup(entity),0));
                }
                if (mask & 0x0004) {
                  q1=node_command_init(TOKEN_MODELINDEX,V_INT,H_BYTE,NODE_VALUE_INT_dup(modelindex),0);
                  if (modelindex>=1 && modelindex<=QWDTOP->nummodels &&
                      QWDTOP->precache_models[modelindex][0]!='*') {
                    node_add_comment(q1,
                    NODE_VALUE_STRING_dup(QWDTOP->precache_models[modelindex]));
                  }
                  ttn=node_link(ttn,q1);
                }
                if (mask & 0x2000) {
                  ttn=node_link(ttn,node_command_init(TOKEN_FRAME,V_INT,H_BYTE,
                                    NODE_VALUE_INT_dup(frame),0));
                }
                if (mask & 0x0008) {
                  ttn=node_link(ttn,node_command_init(TOKEN_COLORMAP,V_INT,H_BYTE,
                                    NODE_VALUE_INT_dup(colormap),0));
                }
                if (mask & 0x0010) {
                  ttn=node_link(ttn,node_command_init(TOKEN_SKIN,V_INT,H_BYTE,
                                    NODE_VALUE_INT_dup(skin),0));
                }
                if (mask & 0x0020) {
                  ttn=node_link(ttn,node_command_init(TOKEN_EFFECTS,V_INT,H_BYTE,
                                    NODE_VALUE_INT_dup(effects),0));
                }
                if (mask&0x0200) { 
                  ttn=node_link(ttn,node_command_init(TOKEN_ORIGIN_X,V_FLOAT,H_COORD,
                                    NODE_VALUE_FLOAT_dup(origin[0]),0));
                }
                if (mask&0x0400) { 
                  ttn=node_link(ttn,node_command_init(TOKEN_ORIGIN_Y,V_FLOAT,H_COORD,
                                    NODE_VALUE_FLOAT_dup(origin[1]),0));
                }
                if (mask&0x0800) { 
                  ttn=node_link(ttn,node_command_init(TOKEN_ORIGIN_Z,V_FLOAT,H_COORD,
                                    NODE_VALUE_FLOAT_dup(origin[2]),0));
                }
                if (mask&0x0001) { 
                  ttn=node_link(ttn,node_command_init(TOKEN_ANGLES_1,V_FLOAT,H_ANGLE,
                                    NODE_VALUE_FLOAT_dup(angles[0]),0));
                }
                if (mask&0x1000) { 
                  ttn=node_link(ttn,node_command_init(TOKEN_ANGLES_2,V_FLOAT,H_ANGLE,
                                    NODE_VALUE_FLOAT_dup(angles[1]),0));
                }
                if (mask&0x0002) { 
                  ttn=node_link(ttn,node_command_init(TOKEN_ANGLES_3,V_FLOAT,H_ANGLE,
                                    NODE_VALUE_FLOAT_dup(angles[2]),0));
                }
                if (ttn!=NULL)
                  tn = node_link(tn,node_init_all(TOKEN_PACKETENTITY, 0, ttn, 0));
              }
              /*
              pnew=m->p;
              m->p=pold;
              for (ttn=NULL; m->p < m->end;) {
                ttn=node_link(ttn,node_init_all(V_BYTEHEX,H_SIMPLE,
                                                NODE_VALUE_BYTEHEX_dup((unsigned char)ReadByte(m)),0));
              }
              if (ttn!=NULL)
                tn=node_link(tn,node_init_all(TOKEN_DATA,H_SIMPLE,ttn,0));
              m->p=pnew;
              */
              /* finish node tree */
              n = node_link(n,node_init_all(TOKEN_PACKETENTITIES, 0, tn, 0));
            }
            break;
            case 0x30: { /* deltapacketentities */
              /* variables */
              long index;
              long entity;
              long mask;
              long modelindex;
              long frame;
              long colormap;
              long skin;
              long effects;
              vec3_t origin;
              vec3_t angles;
                
              node *q1;
              /*
              unsigned char *pold, *pnew;
              */

              /* binary in and node tree */              
              tn = NULL;
              
              /* pold=m->p; */
              index = ReadByte(m);
              tn = node_link(tn, node_command_init(TOKEN_INDEX, V_INT, H_BYTE, 
                                 NODE_VALUE_INT_dup(index),0));
              while ((mask=ReadShort(m)) != 0x0000) {
                entity = mask & 0x01FF;
                mask &= 0xFE00;
                /*  if (mask & 0x4000) { } */
                if (mask & 0x8000) mask |= ReadByte(m);
                modelindex = mask & 0x0004 ? ReadByte(m) : 0;
                frame = mask & 0x2000 ? ReadByte(m) : 0;
                colormap = mask & 0x0008 ? ReadByte(m) : 0;
                skin = mask & 0x0010 ? ReadByte(m) : 0;
                effects = mask & 0x0020 ? ReadByte(m) : 0;
                origin[0] = mask & 0x0200 ? ReadCoord(m) : 0;
                angles[0] = mask & 0x0001 ? ReadAngle(m) : 0;
                origin[1] = mask & 0x0400 ? ReadCoord(m) : 0;
                angles[1] = mask & 0x1000 ? ReadAngle(m) : 0;
                origin[2] = mask & 0x0800 ? ReadCoord(m) : 0;
                angles[2] = mask & 0x0002 ? ReadAngle(m) : 0;

                ttn=NULL;
                
                if (entity!=0) {
                  ttn=node_link(ttn,node_command_init(TOKEN_ENTITY,V_INT,H_BYTE,
                                       NODE_VALUE_INT_dup(entity),0));
                }
                if (mask & 0x0004) {
                  q1=node_command_init(TOKEN_MODELINDEX,V_INT,H_BYTE,NODE_VALUE_INT_dup(modelindex),0);
                  if (modelindex>=1 && modelindex<=QWDTOP->nummodels &&
                      QWDTOP->precache_models[modelindex][0]!='*') {
                    node_add_comment(q1,
                    NODE_VALUE_STRING_dup(QWDTOP->precache_models[modelindex]));
                  }
                  ttn=node_link(ttn,q1);
                }
                if (mask & 0x2000) {
                  ttn=node_link(ttn,node_command_init(TOKEN_FRAME,V_INT,H_BYTE,
                                    NODE_VALUE_INT_dup(frame),0));
                }
                if (mask & 0x0008) {
                  ttn=node_link(ttn,node_command_init(TOKEN_COLORMAP,V_INT,H_BYTE,
                                    NODE_VALUE_INT_dup(colormap),0));
                }
                if (mask & 0x0010) {
                  ttn=node_link(ttn,node_command_init(TOKEN_SKIN,V_INT,H_BYTE,
                                    NODE_VALUE_INT_dup(skin),0));
                }
                if (mask & 0x0020) {
                  ttn=node_link(ttn,node_command_init(TOKEN_EFFECTS,V_INT,H_BYTE,
                                    NODE_VALUE_INT_dup(effects),0));
                }
                if (mask&0x0200) { 
                  ttn=node_link(ttn,node_command_init(TOKEN_ORIGIN_X,V_FLOAT,H_COORD,
                                    NODE_VALUE_FLOAT_dup(origin[0]),0));
                }
                if (mask&0x0400) { 
                  ttn=node_link(ttn,node_command_init(TOKEN_ORIGIN_Y,V_FLOAT,H_COORD,
                                    NODE_VALUE_FLOAT_dup(origin[1]),0));
                }
                if (mask&0x0800) { 
                  ttn=node_link(ttn,node_command_init(TOKEN_ORIGIN_Z,V_FLOAT,H_COORD,
                                    NODE_VALUE_FLOAT_dup(origin[2]),0));
                }
                if (mask&0x0001) { 
                  ttn=node_link(ttn,node_command_init(TOKEN_ANGLES_1,V_FLOAT,H_ANGLE,
                                    NODE_VALUE_FLOAT_dup(angles[0]),0));
                }
                if (mask&0x1000) { 
                  ttn=node_link(ttn,node_command_init(TOKEN_ANGLES_2,V_FLOAT,H_ANGLE,
                                    NODE_VALUE_FLOAT_dup(angles[1]),0));
                }
                if (mask&0x0002) { 
                  ttn=node_link(ttn,node_command_init(TOKEN_ANGLES_3,V_FLOAT,H_ANGLE,
                                    NODE_VALUE_FLOAT_dup(angles[2]),0));
                }
                if (ttn!=NULL)
                  tn = node_link(tn,node_init_all(TOKEN_DELTAPACKETENTITY, 0, ttn, 0));
              }
              /* 
              pnew=m->p;
              m->p=pold;
              for (ttn=NULL; m->p < m->end;) {
                ttn=node_link(ttn,node_init_all(V_BYTEHEX,H_SIMPLE,
                                                NODE_VALUE_BYTEHEX_dup((unsigned char)ReadByte(m)),0));
              }
              if (ttn!=NULL)
                tn=node_link(tn,node_init_all(TOKEN_DATA,H_SIMPLE,ttn,0));
              m->p=pnew;
              */
              /* finish node tree */
              n = node_link(n,node_init_all(TOKEN_DELTAPACKETENTITIES, 0, tn, 0));
            }
            break;
            default: {
              tn=node_command_init(TOKEN_ID,V_BYTEHEX,H_SIMPLE,
                                   NODE_VALUE_BYTEHEX_dup(id),0);
              for (ttn=NULL; m->p < m->end;) {
                ttn=node_link(ttn,node_init_all(V_BYTEHEX,H_SIMPLE,
                                                NODE_VALUE_BYTEHEX_dup((unsigned char)ReadByte(m)),0));
              }
              if (ttn!=NULL)
                tn=node_link(tn,node_init_all(TOKEN_DATA,H_SIMPLE,ttn,0));

              n=node_link(n,
                  node_add_comment(
                    node_init_all(TOKEN_UNKNOWN,H_UNKNOWN,tn,0),
                    NODE_VALUE_STRING_dup("something is wrong")
                  )
                );
            }
            break;
          }
        }
      }
      n = node_init_all(TOKEN_SBLOCK,0,n,0);
      if (opt & 0x01) {
        sprintf(ts,"%i",QWDTOP->frame);
        node_add_comment(n,NODE_VALUE_STRING_dup(ts));
      }       
      return n;
    }
    break;
    default:
      syserror(WQWD, QWDTOP->filename);
    break;
  }
  return n;
}

/*- file end uqwd.c ---------------------------------------------------------*/
