/*

.enemy
Will be world if not currently angry at anyone.

.pathcorner
The next path spot to walk toward.  If .enemy, ignore .pathcorner
When an enemy is killed, the monster will try to return to it's path.

.hunt_time
Set to time + something when the player is in sight, but movement straight for
him is blocked.  This causes the monster to use wall following code for
movement direction instead of sighting on the player.

.ideal_yaw
A yaw angle of the intended direction, which will be turned towards at up
to 45 deg / state.  If the enemy is in view and hunt_time is not active,
this will be the exact line towards the enemy.

.pausetime
A monster will leave it's stand state and head towards it's .pathcorner when
time > .pausetime.

walkmove(angle, speed) primitive is all or nothing
*/


//
// when a monster becomes angry at a player, that monster will be used
// as the sight target the next frame so that monsters near that one
// will wake up even if they wouldn't have noticed the player
//

/*
=============
range

returns the range catagorization of an entity reletive to self
0	melee range, will become hostile even if back is turned
1	visibility and infront, or visibility and show hostile
2	infront and show hostile
3	only triggered by damage
=============
*/
float(entity targ) range =
{
vector	spot1, spot2;
float		r,melee;	

	if((self.solid==SOLID_BSP||self.solid==SOLID_TRIGGER)&&self.origin=='0 0 0')
		spot1=(self.absmax+self.absmin)*0.5;
	else
		spot1 = self.origin + self.view_ofs;

	if((targ.solid==SOLID_BSP||targ.solid==SOLID_TRIGGER)&&targ.origin=='0 0 0')
		spot2=(targ.absmax+targ.absmin)*0.5;
	else
		spot2 = targ.origin + targ.view_ofs;
	
	r = vlen (spot1 - spot2);

	if (self.classname=="monster_mummy")
		melee = 50;
	else
		melee = 100;

	if (r < melee)
		return RANGE_MELEE;
	if (r < 500)
		return RANGE_NEAR;
	if (r < 1000)
		return RANGE_MID;
	return RANGE_FAR;
};

/*
=============
visible2ent

returns 1 if the entity is visible to self, even if not infront ()
=============
*/
float visible2ent (entity targ, entity forent)
{
vector	spot1, spot2;
	if((forent.solid==SOLID_BSP||forent.solid==SOLID_TRIGGER)&&forent.origin=='0 0 0')
		spot1=(forent.absmax+forent.absmin)*0.5;
	else
		spot1 = forent.origin + forent.view_ofs;
		
	if((targ.solid==SOLID_BSP||targ.solid==SOLID_TRIGGER)&&targ.origin=='0 0 0')
		spot2=(targ.absmax+targ.absmin)*0.5;
	else
		spot2 = targ.origin + targ.view_ofs;

    traceline (spot1, spot2, TRUE, forent);   // see through other monsters

	if(trace_ent.thingtype>=THINGTYPE_WEBS)
		traceline (trace_endpos, spot2, TRUE, trace_ent);
//	else if (trace_inopen && trace_inwater)//FIXME?  Translucent water?
//		return FALSE;			// sight line crossed contents

	if (trace_fraction == 1)
	{
		if(forent.flags&FL_MONSTER)
		{
			if(visibility_good(targ,0.15 - skill/20))
				return TRUE;
		}
		else
			return TRUE;
	}

	return FALSE;
}

/*
=============
infront_of_ent

returns 1 if the targ is in front (in sight) of from
=============
*/
float infront_of_ent (entity targ , entity from)
{
	vector	vec,spot1,spot2;
	float	accept,dot;

	if((from.classname=="player")||(from.classname=="bot"))
	    makevectors (from.v_angle);
	else
	    makevectors (from.angles);

	if((from.solid==SOLID_BSP||from.solid==SOLID_TRIGGER)&&from.origin=='0 0 0')
		spot1=(from.absmax+from.absmin)*0.5;
	else
		spot1 = from.origin + from.view_ofs;

	spot2=(targ.absmax+targ.absmin)*0.5;

    vec = normalize (spot2 - spot1);
	dot = vec * v_forward;

    accept = 0.3;
	
    if ( dot > accept)
		return TRUE;
	return FALSE;
}

/*
=============
visible

returns 1 if the entity is visible to self, even if not infront ()
=============
*/
float visible (entity targ)
{
	return visible2ent(targ,self);
}

/*
=============
infront

returns 1 if the entity is in front (in sight) of self
=============
*/
float infront (entity targ)
{
	return infront_of_ent(targ,self);
}


void() HuntTarget =
{
	self.goalentity = self.enemy;
	if(self.spawnflags&PLAY_DEAD)
	{
//		dprint("getting up!!!\n");
		self.think=self.th_possum_up;
		self.spawnflags(-)PLAY_DEAD;
	}
	else
		self.think = self.th_run;
//	self.ideal_yaw = vectoyaw(self.enemy.origin - self.origin);
	self.ideal_yaw = vectoyaw(self.goalentity.origin - self.origin);
	thinktime self : 0.1;
//	SUB_AttackFinished (1);	// wait a while before first attack
};

void() FoundTarget =
{
	if ((self.enemy.classname == "player")||(self.enemy.classname == "bot"))
	{	// let other monsters see this monster for a while
		sight_entity = self;
		sight_entity_time = time + 1;
	}
	
	self.show_hostile = time + 1;		// wake up other monsters

	SightSound ();

	HuntTarget ();
};

/*
===========
FindTarget

Self is currently not attacking anything, so try to find a target

Returns TRUE if an enemy was sighted

When a player fires a missile, the point of impact becomes a fakeplayer so
that monsters that see the impact will respond as if they had seen the
player.

To avoid spending too much time, only a single client (or fakeclient) is
checked each frame.  This means multi player games will have slightly
slower noticing monsters.
============
*/
float(float dont_hunt) FindTarget =
{
entity	client;
float		r;

// if the first spawnflag bit is set, the monster will only wake up on
// really seeing the player, not another monster getting angry

	if(!deathmatch&&(self.classname=="monster_imp_lord"||self.classname=="cube_of_force"))
		return FindMonsterTarget();

	if(deathmatch||coop) return BotsFindTarget(dont_hunt);

	if (sight_entity_time >= time&&sight_entity!=world)
	{
		client = sight_entity;
		if (client.enemy == self.enemy)
			return TRUE;
	}
	else
	{
		client = checkclient ();
		if (!client)
			return FALSE;	// current check entity isn't in PVS
	}

	if (client == self.enemy)
		return FALSE;

	r = range (client);
	if (r == RANGE_FAR)
		return FALSE;

	if(!visibility_good(client,5))
	{
		return FALSE;
	}

	if(self.think!=spider_onwall_wait)
		if (r == RANGE_NEAR)
		{
			if (client.show_hostile < time && !infront (client))
				return FALSE;
		}
		else if (r == RANGE_MID)
		{
			if (!infront (client))
				return FALSE;
		}
	
	if (!visible (client))
		return FALSE;

//
// got one
//
	self.enemy = client;

	if ((self.enemy.classname != "player")&&(self.enemy.classname != "bot"))
	{
		self.enemy = self.enemy.enemy;
		if ((self.enemy.classname != "player")&&(self.enemy.classname != "bot"))
		{
			self.enemy = world;
			return FALSE;
		}
	}

	if(!dont_hunt)
		FoundTarget ();
	return TRUE;
};


void(float dist) ai_forward =
{
	walkmove (self.angles_y, dist, FALSE);
};

void(float dist) ai_back =
{
	walkmove ( (self.angles_y+180), dist, FALSE);
};


/*
=============
ai_pain

stagger back a bit
=============
*/
void(float dist) ai_pain =
{
//	ai_back (dist);
float	away;
	
	away = vectoyaw (self.origin - self.enemy.origin)+90*random(0.5,-0.5);
	
	walkmove (away, dist,FALSE);
};

/*
=============
ai_painforward

stagger back a bit
=============
*/
void(float dist) ai_painforward =
{
	walkmove (self.ideal_yaw, dist, FALSE);
};

/*
=============
ai_walk

The monster is walking it's beat
=============
*/
void(float dist) ai_walk =
{
	
	MonsterCheckContents();

	movedist = dist;
	
	// check for noticing a player
	if (FindTarget (FALSE))
		return;

	if(self.classname=="monster_eidolon")
	{
		if (!self.path_current)
			riderpath_init();
		if(!eidolon_riderpath_move(dist))
		{
			if(self.think==self.th_walk)
				self.think=eidolon_guarding;
		}
		else if(self.think==eidolon_guarding)
			self.think=self.th_walk;
	}
	else
		movetogoal (dist);
};


/*
=============
ai_stand

The monster is staying in one place for a while, with slight angle turns
=============
*/
void() ai_stand =
{
	MonsterCheckContents();
	
	if (FindTarget (FALSE))
		return;
	
	if(self.spawnflags&PLAY_DEAD)
		return;

	if (time > self.pausetime)
	{
		self.th_walk ();
		return;
	}

// change angle slightly
};

/*
=============
ai_turn

don't move, but turn towards ideal_yaw
=============
*/
void() ai_turn =
{
	if (FindTarget (FALSE))
		return;
	
	ChangeYaw ();
};

//=============================================================================

/*
=============
ChooseTurn
=============
*/
void(vector dest3) ChooseTurn =
{
	local vector	dir, newdir;
	
	dir = self.origin - dest3;

	newdir_x = trace_plane_normal_y;
	newdir_y = 0 - trace_plane_normal_x;
	newdir_z = 0;
	
	if (dir * newdir > 0)
	{
		dir_x = 0 - trace_plane_normal_y;
		dir_y = trace_plane_normal_x;
	}
	else
	{
		dir_x = trace_plane_normal_y;
		dir_y = 0 - trace_plane_normal_x;
	}

	dir_z = 0;
	self.ideal_yaw = vectoyaw(dir);	
};

/*
============
FacingIdeal

Within angle to launch attack?
============
*/
float() FacingIdeal =
{
	local	float	delta;
	
	delta = anglemod(self.angles_y - self.ideal_yaw);
	if (delta > 45 && delta < 315)
		return FALSE;
	return TRUE;
};


//=============================================================================

float() CheckAnyAttack =
{
	if (self.model=="models/medusa.mdl"||self.model=="models/medusa2.mdl")
			return(MedusaCheckAttack ());

	if (!enemy_vis)
		return FALSE;

	if(self.classname=="monster_eidolon")
		if(self.goalentity==self.controller)
			return FALSE;
		else
			return eidolon_check_attack();

	if (self.model=="models/archer.mdl")
		return(ArcherCheckAttack ());

	if(self.goalentity==self.controller)
		return FALSE;

	return CheckAttack ();
};


/*
=============
ai_attack_face

Turn in place until within an angle to launch an attack
=============
*/
void() ai_attack_face =
{
	self.ideal_yaw = enemy_yaw;
	ChangeYaw ();
	if (FacingIdeal())  // Ready to go get em
	{
		if (self.attack_state == AS_MISSILE)
			self.th_missile ();
		else if (self.attack_state == AS_MELEE)
			self.th_melee ();
		self.attack_state = AS_STRAIGHT;
	}
};


/*
=============
ai_run_slide

Strafe sideways, but stay at aproximately the same range
=============
*/
void ai_run_slide ()
{
float	ofs;
	
	self.ideal_yaw = enemy_yaw;
	ChangeYaw ();
	if (self.lefty)
		ofs = 90;
	else
		ofs = -90;
	
	if (walkmove (self.ideal_yaw + ofs, movedist, FALSE))
		return;
		
	self.lefty = 1 - self.lefty;
	
	walkmove (self.ideal_yaw - ofs, movedist, FALSE);
}


/*
=============
ai_run

The monster has an enemy it is trying to kill
=============
*/
void(float dist) ai_run =
{
	
	MonsterCheckContents();
	
	movedist = dist;
// see if the enemy is dead
	if (!self.enemy.flags2&FL_ALIVE||(self.enemy.artifact_active&ARTFLAG_STONED&&self.classname!="monster_medusa"))
	{
		self.enemy = world;
	// FIXME: look all around for other targets
		if (self.oldenemy.health > 0)
		{
			self.enemy = self.oldenemy;
			HuntTarget ();
		}
		else if(coop)
		{
			if(!FindTarget(TRUE))	//Look for other enemies in the area
			{
				if (self.pathentity)
					self.th_walk ();
				else
					self.th_stand ();
				return;
			}
		}
		else
		{
			if (self.pathentity)
				self.th_walk ();
			else
				self.th_stand ();
			return;
		}
	}

	self.show_hostile = time + 1;		// wake up other monsters

// check knowledge of enemy
	enemy_vis = visible(self.enemy);
	if (enemy_vis)
	{
		self.search_time = time + 5;
		if(self.mintel)
		{
			self.goalentity=self.enemy;
		    self.wallspot=(self.enemy.absmin+self.enemy.absmax)*0.5;
		}
	}
	else
	{
		if(coop)
		{
			if(!FindTarget(TRUE))
				if(self.model=="models/spider.mdl")
				{
					if(random()<0.5)
						SetNextWaypoint();
				}
				else 
					SetNextWaypoint();
		}
		if(self.mintel)
			if(self.model=="models/spider.mdl")
			{
				if(random()<0.5)
					SetNextWaypoint();
			}
			else 
				SetNextWaypoint();
	}

	if(random()<0.5&&(!self.flags&FL_SWIM)&&(!self.flags&FL_FLY)&&(self.spawnflags&JUMP))
		CheckJump();

// look for other coop players
	if (coop && self.search_time < time)
	{
		if (FindTarget (FALSE))
			return;
	}

	enemy_infront = infront(self.enemy);
	enemy_range = range(self.enemy);
	if(self.classname!="monster_eidolon")
		enemy_yaw = vectoyaw(self.goalentity.origin - self.origin);
	
	if ((self.attack_state == AS_MISSILE) || (self.attack_state == AS_MELEE))  // turning to attack
	{
		if(self.classname!="monster_eidolon")
			ai_attack_face ();
		return;
	}

	if (CheckAnyAttack ())
		return;					// beginning an attack
		
	if (self.attack_state == AS_SLIDING)
	{
		ai_run_slide ();
		return;
	}
		
// head straight in
//	if(self.netname=="spider")
//		check_climb();

	if(self.classname=="monster_eidolon")
	{
		if(!self.path_current)
			riderpath_init();
		if(!eidolon_riderpath_move(dist))
		{
			if(self.think==self.th_run)
				eidolon_guarding();
		}
		else if(self.think==eidolon_guarding)
			self.th_run();
	}
	else
		movetogoal (dist);		// done in C code...
};

float() LocateTarget =
{
	return FindTarget(TRUE);
};

// You must perform the following call sometime before calling this function:
//    enemy_range = range (self.enemy);
float(float AttackType, float ChanceModifier) CheckMonsterAttack =
{
	local vector	spot1, spot2;	
	local entity	targ;
	local float		chance;

	targ = self.enemy;
	
	if (self.classname == "monster_hydra")
		if (self.enemy.watertype != CONTENT_WATER) 
		{
			if (self.search_time < time)
			{
				self.monster_stage = 0;
				self.enemy = world;
				return 0;
			}
			return 0;
		}
		else self.search_time = time + 5;

// see if any entities are in the way of the shot
	spot1 = self.origin + self.view_ofs;
	spot2 = targ.origin + targ.view_ofs;

	traceline (spot1, spot2, FALSE, self);

	if(trace_ent.thingtype>=THINGTYPE_WEBS)
		traceline (trace_endpos, spot2, FALSE, trace_ent);

	if (trace_ent != targ)
		if(trace_ent.health>25||!trace_ent.takedamage||(trace_ent.flags&FL_MONSTER&&trace_ent.classname!="player_sheep"))
			return MA_BLOCKED;		// don't have a clear shot
			
	if (trace_inopen && trace_inwater)
		return MA_CROSSED;			// sight line crossed contents

	if (enemy_range == RANGE_MELEE)
	{	// melee attack
	   if (AttackType & MA_SHORT_MISSILE)
		{
		   if (random() < 0.5)
			{
			   self.th_missile ();
			   return MA_SUCCESSFUL;
			}
		}
		if (self.th_melee)
		{
			if (AttackType & MA_MELEE)
			{
			   self.th_melee ();
			   return MA_SUCCESSFUL;
			}
			else 
				return MA_NOWEAPON;
		}
	}
	
// missile attack
	if (!self.th_missile || !(AttackType & (MA_MISSILE | MA_FAR_MELEE)))
	{
		return MA_NOWEAPON;
	}
	
	if (time < self.attack_finished)
		return MA_TOOSOON;
		
	if (enemy_range == RANGE_FAR)
		return MA_TOOFAR;
		
	if (enemy_range == RANGE_MELEE)
	{
		chance = 0.9;
		self.attack_finished = 0;
	}
	else if (enemy_range == RANGE_NEAR)
	{
		if (self.th_melee)
			chance = 0.2;
		else
			chance = 0.4;
	}
	else if (enemy_range == RANGE_MID)
	{
		if (self.th_melee)
			chance = 0.05;
		else
			chance = 0.1;
	}
	else
		chance = 0;

	chance = chance * ChanceModifier;
	if (chance > 0.95) chance = 0.95;

	if (random () < chance)
	{
		if (self.th_melee)
		{  // quake c wouldn't allow me to put this in on if!!!
			if	(AttackType & MA_FAR_MELEE)
			{
				self.th_melee ();
			}
			else
			{
				self.th_missile ();
			}
		}
		else
		{
			self.th_missile ();
		}
		SUB_AttackFinished (random(8,0));
		return MA_SUCCESSFUL;
	}

	return MA_NOATTACK;
};

float (vector offset, vector to_where)monster_checkpos =
{  
	// This function will trace 2 lines - the first line will go from the origin to the offset from the origin.
	// If this could be done atleast half way, then a 2nd trace is done from the end point of the first trace
	// to the final destination.  If this was mostly successful, then the function will return true, otherwise 
	// false.  You would use this function in a situation where you want to see if the enemy is visibile from
	// your right side, so you first see if you can go do the right side, then go forward from there to the enemy.
	local vector start;
	local float length;

	start = self.origin + offset;
	traceline (self.origin, start, FALSE, self);
	if (trace_fraction < 0.5) 
	{  // Couldn't get to the offset
		return FALSE;
	}
	length = vlen(self.origin-start) * trace_fraction;
	start = trace_endpos;
	traceline (start,to_where, FALSE, self);
	if (trace_ent == self.enemy || trace_fraction > 0.98) 
	{  // We found the enemy!
		length = length + vlen(start-self.enemy.origin) * trace_fraction;
		return length;
	}

	return FALSE;
};


void (float l, float r, float u, float d, float last_move, vector where) find_path =
{
	// This function will check to see if an enemy can be located from the top, bottom, left, and right sides.
	// The l, r, u, d parameters specify the order for which the search should be done.  If it couldn't find
	// the enemy, then it will try the last seen position of the enemy.  last_move indicates that the previous
	// move was successful (i.e. the monster could move forward).  where is the position to check for.
	local float length;
	local float newyaw;
	local float newz;
	local float c;
	local float retval;
//	local vector a, b;

	makevectors (self.angles);
	length = 99999;
	newyaw = self.ideal_yaw;
	newz = self.velocity_z;

	c = 0;
	while(c<=4)
	{  // We have 5 checks to do
		if (c == 0 && last_move)
		{  // Try checking forward
			retval = monster_checkpos(v_forward*300,where);
			if (retval && retval < length)
			{
				//dprint("found you to the forward\n");
				self.monster_duration = 18 + 5;
				length = retval;
			}
		}
		if (c == l)
		{  // Try checking to the left
			retval = monster_checkpos(v_right*-200,where);
			if (retval && retval < length)
			{
				//dprint("found you to the left\n");
				newyaw = self.angles_y + 90;
				self.monster_duration = 18 + 5;
				length = retval;
			}
		}
		else if (c == r)
		{  // Try checking to the right
			retval = monster_checkpos(v_right*200,where);
			if (retval && retval < length)
			{
				//dprint("found you to the right\n");
				newyaw = self.angles_y - 90;
				self.monster_duration = 18 + 5;
				length = retval;
			}
		}
		else if (c == u)
		{  // Try checking to the top
			retval = monster_checkpos(v_up*200,where);
			if (retval && retval < length)
			{
				//dprint("found you to the up\n");
  		      newz = 30;
				self.monster_duration = 18 + 5;
				length = retval;
			}
		}
		else if (c == d)
		{  // Try checking to the bottom
			retval = monster_checkpos(v_up*-200,where);
			if (retval && retval < length)
			{
				//dprint("found you to the down\n");
  		      newz = -30;
				self.monster_duration = 18 + 5;
				length = retval;
			}
		}
		c = c + 1;
	}

	if (length == 99999 && self.monster_last_seen != where)
	{  // If we didn't find a direction, and we haven't done this, try looking where the enemy
		// was last seen
		find_path(l,r,u,d,0,self.monster_last_seen);
		//dprint("Using last seen\n");
	}
	else
	{
		self.ideal_yaw = newyaw;
		self.velocity_z = newz;
	}
};

float () FindDir =
{ // Monster couldn't go in the direction it is pointed to, so find one it can go to
	local vector a,b,c;
	local float inc,step;

	if (random() < 0.5) inc = 45;
	else inc = -45;

	c = '0 0 0';
	c_y = c_y + inc;

	step = 0;
	while(step < 6)
	{  // 7 directions to check (45 degrees each)
		makevectors (self.angles + c);
		a = self.origin + self.view_ofs;
		b = a + v_forward*100;

		traceline (a, b, FALSE, self);

		if (trace_fraction > 0.9)
		{  // We can mostly go this direction
			return self.angles_y + c_y;
			//dprint("   found\n");
		}
		c_y = c_y + inc;
		step = step + 1;
	}

	return self.angles_y;
};

/*
==============================================

GENERAL

==============================================
*/
/*
=============
get_visibility

Checks for drf_translucent and abslight of an object,
and uses that and it's world lighting value to set it's
visibility value
>=1 = Totally visible (default)
.
.
.
<=0 = Totally invisible
This value should be used in monster aiming as well.

NOTE: This only works on players since light_level info
	is taken from player's weaponmodel lighting (0-255)
=============
*/
void get_visibility (entity targ , float range_mod)
{
//NOTE: incorporate distance?
float base, divider, attack_mod;

//FIXME: .light_level gives a value of 0 if MLS_POWERMODE is on...
//Temp fix for now...
	if(((targ.classname!="player")&&(targ.classname!="bot"))||targ.drawflags&MLS_POWERMODE)
	{
		targ.visibility=1;
		return;
	}

	if(targ.effects&EF_NODRAW)
	{
		targ.visibility=0;
		return;
	}

	if(targ.drawflags&DRF_TRANSLUCENT)
	{
		if(targ.model=="models/assassin.mdl")
			divider=3+targ.level;//Bonus for hiding in shadows
		else
			divider=3;	//Makes it 3 times harder to see
	}
	else
		divider=1;

	if(targ.classname=="bot") base=128;
	else if(targ.drawflags&MLS_ABSLIGHT)
		base=targ.abslight/2.5;
	else
		base=targ.light_level/75;//75 is semi-fullbright

	if(range_mod)
		range_mod=vlen(targ.origin-self.origin)/333;
	else
		range_mod = 1;

	if(targ.last_attack>time - 3)//Remember where they were when fired
		attack_mod=time - targ.last_attack;

	targ.visibility=base/divider/range_mod + attack_mod;
}

/*
=============
float visibility_good (entity targ,float chance_mod)
MG
Does a random check to see if self can see the target based
it's visibility (calls get_visibility for that targ first)
The higher the chance_mod, the lower the chance of
good visibility.
=============
*/
float visibility_good (entity targ,float chance_mod)
{
	if(!targ)
		return FALSE;
	get_visibility(targ,TRUE);
	if(random(chance_mod)<targ.visibility)
		return TRUE;

	return FALSE;
}

/*
=============
float FindMonsterTarget ()
MG

Called by FindTarget, checks for anything alive and
visible within range and sets it as enemy (as long as
it's not the monster's controller, or has the same
controller).  Returns TRUE if it finds something,
FALSE if not.
=============
*/
float FindMonsterTarget ()
{
entity found;
float okay;
	if(self.controller.enemy!=world&&self.controller.enemy.flags2&FL_ALIVE&&visible(self.controller.enemy))
	{
		self.enemy=self.controller.enemy;
		return TRUE;
	}
	
	okay=FALSE;
	found=findradius(self.origin,1000);
	while(found!=world)
	{
		if(found!=self)
			if(found.flags2&FL_ALIVE)
				if(visible(found))
					if(found!=self.controller)
						if(found.controller!=self.controller)
						{
							if(coop)
							{
								if((found.classname!="player")&&(found.classname!="bot"))
									okay = TRUE;
							}
							else if(teamplay)
							{
								if(found.team!=self.controller.team)
									okay = TRUE;
							}
							else
								okay = TRUE;
							if(okay)
							{
								self.enemy=found;
								return TRUE;
							}
						}
		found=found.chain;
	}
	if(self.classname=="monster_imp_lord")
		self.enemy=self.controller;
	return FALSE;
}

/*
==================================================================
float CheckJump()
MG
Checks to see if the enemy is not at the same level as monster
or something is blocking the path of the monster.  If there is 
a clear jump arc to the enemy and the monster will not land in
water or lava, the monster will attampt to jump the distance.
==================================================================
*/
float CheckJump ()
{
local vector spot1, spot2, jumpdir;
float jump_height, jumpup, ignore_height;

		makevectors(self.angles);
		jumpdir=normalize(self.goalentity.origin-self.origin);
		jumpdir_z=0;
		jump_height=jumpdir*v_forward;
		if(jump_height<0.3)
			return FALSE;

        spot1=self.origin;
        spot2=self.enemy.origin;

        spot1_z=0;
        spot2_z=0;
		jump_height=16;

		if(pointcontents(spot1+v_forward*24-'0 0 10')!=CONTENT_SOLID)
			ignore_height=TRUE;

		if(self.classname!="monster_mezzoman"&&!self.spiderType)
			if(vlen(spot1-spot2)>256)
				ignore_height=FALSE;


//also check to make sure you can't walkmove forward
                if(self.jump_flag>time)            //Don't jump too many times in a row
		{
                        return FALSE;
		}
                else if(pointcontents(self.goalentity.origin)!=CONTENT_EMPTY)
		{
                        return FALSE;
		}
                else if(!visible(self.goalentity))
		{
                        return FALSE;
		}
                else if(!ignore_height&&self.goalentity.absmin_z+36>=self.absmin_z&&self.think!=SpiderJumpBegin&&self.classname!="monster_mezzoman")
		{
	                    return FALSE;
		}
                else if(!self.flags&FL_ONGROUND)
		{
                        return FALSE;
		}
                else if(!self.goalentity.flags&FL_ONGROUND&&self.goalentity.classname!="waypoint")
		{
                        return FALSE;
		}
                else if(!infront(self.goalentity))
		{
                        return FALSE;
		}
                else if(vlen(spot1-spot2)>777&&!ignore_height)
		{
                        return FALSE;
		}
                else if(vlen(spot1-spot2)<=100)//&&self.think!=SpiderMeleeBegin)
		{
                        return FALSE;
		}

		if(self.think==SpiderJumpBegin)
			jump_height=vlen((self.goalentity.absmax+self.goalentity.absmin)*0.5-self.origin)/13;
		else if(self.classname=="monster_mezzoman")
			if(self.goalentity.absmin_z>=self.absmin_z+36)
			{
				jump_height=vlen((self.goalentity.absmax+self.goalentity.absmin)*0.5-self.origin)/13;
				jumpup=TRUE;
			}
			else if(self.goalentity.absmin_z>self.absmin_z - 36)
			{
				if(ignore_height)
					jump_height=vlen((self.goalentity.absmax+self.goalentity.absmin)*0.5-self.origin)/13;
				else
					return FALSE;
			}

        spot1=self.origin;
        spot1_z=self.absmax_z;
		spot2=spot1;
		spot2_z+=36;

        traceline(spot1, spot2,FALSE,self);

        if(trace_fraction<1)
		{
			return FALSE;
		}

		if(!jumpup)
		{
		        spot1+=jumpdir*((self.maxs_x+self.maxs_y)*0.5);

		    	traceline(self.origin, spot1 + '0 0 36',FALSE,self);

	        	if(trace_fraction<1)
			{
				return FALSE;
			}

	        	traceline(spot1,spot1+jumpdir*64 - '0 0 500',FALSE,self);

		    	if(pointcontents(trace_endpos)==CONTENT_WATER||pointcontents(trace_endpos)==CONTENT_SLIME||pointcontents(trace_endpos)==CONTENT_LAVA)
			{
				return FALSE;
			}
		}

		ai_face();
		if(self.think!=SpiderJumpBegin)
	    {
		    self.jump_flag=time + 7;        //Only try to jump once every 7 seconds
			SightSound();
			if(!jumpup)
			{
			    self.velocity=jumpdir*jump_height*17*self.scale;
			    self.velocity_z = jump_height*12*self.scale;
			}
			else
			{
			    self.velocity=jumpdir*jump_height*10*self.scale;
			    self.velocity_z = jump_height*14*self.scale;
			}
			self.flags(-)FL_ONGROUND;
			if(self.th_jump)
				self.th_jump();
			else
				thinktime self : 0.3;
		}
		else
		{
			self.level=jump_height;
			return TRUE;
		}
}

/*
====================================================================
void MonsterCheckContents ()
MG
Monsters check to see if they're in lava or under water and
do damage do themselves if appropriate.
void do_contents_dam ()
Just spawns a temporary ent to damage self, using T_Damage on
self does weird stuff- won't kill self, just become invincible
====================================================================
*/
void do_contents_dam ()
{
	T_Damage(self.enemy,world,world,self.dmg,"contents");
	if(self.dmg==5)
	{
		self.classname="contents damager";
		setorigin(self,self.enemy.origin+self.enemy.view_ofs);
		DeathBubbles(1);
	}
	remove(self);
}

void MonsterCheckContents ()
{
	if(random()>0.3)
		return;

	if(pointcontents(self.origin)==CONTENT_LAVA)
	{
		if(self.flags&FL_FIREHEAL)
		{
			if(self.health<self.max_health)
				self.health+=1;
		}
		else
		{
			newmis=spawn();
			newmis.think=do_contents_dam;
			newmis.enemy=self;
			newmis.dmg=30;
			thinktime newmis : 0;
		}
	}
	if(self.movetype==MOVETYPE_SWIM||self.model=="models/skullwiz.mdl"||self.netname=="golem")
		return;
	if(pointcontents(self.origin+self.view_ofs)==CONTENT_WATER)
	{
		if(self.air_finished<time)	//Start drowning
		{
			newmis=spawn();
			newmis.think=do_contents_dam;
			newmis.enemy=self;
			newmis.dmg=5;
			thinktime newmis : 0;
		}
	}
	else
		self.air_finished=time+12;
}

/*
====================================================================
void pitch_roll_for_slope (vector slope)
(My personal favorite!)
MG
This will adjust the pitch and roll of a monster to match
a given slope - if a non-'0 0 0' slope is passed, it will
use that value, otherwise it will use the ground underneath
the monster.  If it doesn't find a surface, it does nothinh\g
and returns.
====================================================================
*/
void pitch_roll_for_slope (vector slope)
{
vector new_angles,new_angles2,old_forward,old_right;
float dot,mod;
	makevectors(self.angles);
	old_forward=v_forward;
	old_right=v_right;

	if(slope=='0 0 0')
	{
		traceline(self.origin,self.origin-'0 0 300',TRUE,self);
		if(trace_fraction>0.05&&self.movetype==MOVETYPE_STEP)
			self.flags(-)FL_ONGROUND;
		if(trace_fraction==1)
			return;
		slope=trace_plane_normal;
	}
	new_angles=vectoangles(slope);
	new_angles_x=(90-new_angles_x)*-1;//Gets actual slope
	new_angles2='0 0 0';
	new_angles2_y=new_angles_y;

	makevectors(new_angles2);

	mod=v_forward*old_right;
	if(mod<0)
		mod=1;
	else
		mod=-1;

	dot=v_forward*old_forward;
	self.angles_x=dot*new_angles_x;
	self.angles_z=(1-fabs(dot))*new_angles_x*mod;
}
/*
==============================================

IMP

==============================================
*/

/*
================================================================
checkenemy()
Checks to see if enemy is of the same monstertype and old enemy
is alive and visible.  If so, changes back to it's last enemy.
================================================================
*/
void checkenemy (void)
{
entity oldtarget;
	if(((self.enemy.classname=="player")||(self.enemy.classname=="bot"))&&self.enemy.flags2&FL_ALIVE&&self.enemy!=self.controller)
		return;

	if (!self.enemy.flags2&FL_ALIVE||self.enemy==self.controller)
	{
		if((self.controller.classname=="player")||(self.controller.classname=="bot"))
		{
			self.enemy = self.controller;
			self.goalentity=self.enemy;
		}
		else
			self.enemy = world;

		if (self.oldenemy.flags2&FL_ALIVE)
		{
			self.enemy = self.oldenemy;
			self.goalentity = self.enemy;
			self.think = self.th_run;
		}
		else if(LocateTarget())
		{
			self.goalentity = self.enemy;
			self.think = self.th_run;
		}
		else
		{
			if((self.controller.classname=="player")||(self.controller.classname=="bot"))
				self.goalentity=self.enemy=self.controller;
			else
				self.goalentity=self.enemy=world;

			if (self.pathentity)
				self.think=self.th_walk;
			else
				self.think=self.th_stand;
		}
		thinktime self : 0;
		return;
	}

	if(self.classname=="monster_imp_lord")
		return;

	if(((self.oldenemy.classname=="player")||(self.oldenemy.classname=="bot"))&&(self.oldenemy.flags2&FL_ALIVE)&&visible(self.oldenemy))
	{
		if((self.model=="models/spider.mdl"||self.model=="models/scorpion.mdl")&&self.enemy.model==self.model)
			self.enemy=self.oldenemy;
		else
		{
			oldtarget=self.enemy;
			self.enemy=self.oldenemy;
			self.oldenemy=oldtarget;
		}
		self.goalentity=self.enemy;
	}
}

/*
================================================================
fov()

Field-Of-View

Returns TRUE if vector from entity "from" to entity "targ" is
within "scope" degrees of entity "from"'s forward angle.
================================================================
*/
float fov(entity targ,entity from,float scope)
{
vector spot1,spot2;
float dot;
	spot1=from.origin+from.proj_ofs;

	spot2=(targ.absmin+targ.absmax)*0.5;

	if((from.classname=="player")||(from.classname=="bot"))
		makevectors(from.v_angle);
	else
		makevectors(from.angles);

	dot=normalize(spot2-spot1)*v_forward;
	dot=180 - (dot*180);
	if(dot<=scope)
		return TRUE;

	return FALSE;
}

/*
================================================================
check_pos_enemy()
MG
Checks to see if enemy is visible, if so, remember the spot for
waypoints, else set your waypoint at the last spot you saw him.
Also resets search_time timer if you see him.
================================================================
*/
void check_pos_enemy ()
{
	if(!self.mintel)
		return;

	if(!visible(self.enemy))
	{
		self.attack_state = AS_STRAIGHT;
		SetNextWaypoint();
		if(self.model=="models/imp.mdl")	//Imps keep looking in general area for a while
			if(self.search_time<time&&self.goalentity==self.enemy&&self.trigger_field.classname=="waypoint")
				self.goalentity=self.trigger_field;
	}
	else
	{
		if(self.model=="models/imp.mdl")
			self.search_time=time+5;	//If lose sight, keep searching for 5 secs
		self.goalentity=self.enemy;
	    self.wallspot=(self.enemy.absmin+self.enemy.absmax)*0.5;
	}
}

/*
================================================================
float clear_path (entity targ,float whole_body)
MG
returns TRUE if there is a clear shot or path between self
and "targ".  "whole_body" TRUE will check for a path.
================================================================
*/
float clear_path (entity targ,float whole_body)
{
vector destiny,org;
	destiny=targ.origin+targ.proj_ofs;

	if(self.attack_state!=AS_FERRY)
		self.attack_state = AS_STRAIGHT;

	if(whole_body)
	{
		org=(self.absmin+self.absmax)*0.5;
		tracearea (org, destiny, self.mins,self.maxs,FALSE,self);
	}
	else
	{
		org=self.origin+self.proj_ofs;
		traceline (org, destiny,FALSE,self);
	}

	if(!whole_body&&trace_ent.thingtype>=THINGTYPE_WEBS)
		traceline (trace_endpos, destiny, FALSE, trace_ent);

	if (trace_ent == targ)
		return TRUE;

	if(whole_body)
	{
		if(self.attack_state!=AS_FERRY)
			self.attack_state = AS_SLIDING;
		return FALSE;
	}

	if(trace_ent.health>25||!trace_ent.takedamage||(trace_ent.flags&FL_MONSTER&&trace_ent.classname!="player_sheep"))
	{//Don't have a clear shot, and don't want to shoot obstruction
		self.attack_state = AS_SLIDING;
		return FALSE;
	}

	return TRUE;
}

/*
================================================================
check_view(entity targ,vector org,vector dir,float dist,float interval)
MG
Will see if it can see the targ entity along the dir
given to it- used to determine which direction a monster
should move in to get a clear line of sight to the targ.
Returns the distance it took to see targ, FALSE if couldn't.
================================================================
*/
float check_view(entity targ,vector org,vector dir,float dist,float interval)
{
float dist_counter;
	newmis=spawn();
	dir=normalize(dir);
	while(dist_counter<dist)
	{
		dist_counter+=interval;
		setorigin(newmis,org+dir*dist_counter);
		if(visible2ent(targ,newmis))
		{
			traceline (newmis.origin,(targ.absmin+targ.absmax)*0.5,FALSE,self);
			if (trace_ent == targ)
			{
				remove(newmis);
				return dist_counter;
			}
		}
	}
	remove(newmis);
	return FALSE;
}

/*
================================================================
vector check_axis_move (vector checkdir,float minspeed,float maxspeed)
MG
Calls check_view for enemy along given vector, and if it
fails, checks again along opposite direction on that vector.
If check_view is successful, returns the vector*the distance
along that vector at which it found the goalentity, this
"speed" maxes out at "maxpeed" and is at least "minspeed".
Not used for waypoint or path navigation, only for going
after enemies.
================================================================
*/
vector check_axis_move (vector checkdir,float minspeed,float maxspeed)
{
float go_dist;
	checkdir=normalize(checkdir);
	if(random()<0.5)
		checkdir=checkdir*-1;
	go_dist=check_view(self.enemy,self.origin+self.view_ofs,checkdir,500,30);
	if(!go_dist&&random()<0.5)
	{
		checkdir*=-1;
		go_dist=check_view(self.enemy,self.origin+self.view_ofs,checkdir,500,30);
	}
	if(go_dist)
	{
		if(go_dist>maxspeed)
			go_dist=maxspeed;
		else if(go_dist<minspeed)
			go_dist=minspeed;
		checkdir=checkdir*go_dist;
		return checkdir;
	}
	return '0 0 0';
}

/*
================================================================
float check_z_move(float maxdist)
MG
Intended for flying monsters, will try to move up if enemy
or goalentity is above monster, down if it is below.  Uses
movestep, not velocity.
Will move a maximum step of "maxdist".
Returns FALSE if blocked, TRUE if movement completed.
================================================================
*/
float check_z_move(float maxdist)
{
float goaldist;
entity targ;
	if(self.enemy!=world&&visible(self.enemy))
		targ=self.enemy;
	else if(self.goalentity!=world)
		targ=self.goalentity;
	else
		return FALSE;

	if(targ.origin_z!=self.absmin_z)
	{
		goaldist=(targ.absmin_z+targ.absmax_z)*0.5-(self.absmax_z+self.absmin_z)*0.5;
		maxdist=fabs(maxdist);
		if(fabs(goaldist)>maxdist)
		if(goaldist>0)
			goaldist=maxdist;
		else
			goaldist=maxdist*-1;
		if(!movestep(0,0,goaldist, FALSE))
			return FALSE;
	}
	return TRUE;
}

/*
====================================================================

MEDUSA

====================================================================
*/

/*
============================================================
float lineofsight(entity targ, entity from)
MG
Traces a line along "from"'s view_ofs along v_forward and returns
VRAI if it hits targ, FAUX if not, mon ami.
============================================================
*/
float lineofsight(entity targ, entity from)
{
//FIXME: account for monster's lack of pitch if z diff
vector org,dir;
	if((from.classname=="player")||(from.classname=="bot"))
		makevectors(from.v_angle);
	else
		makevectors(from.angles);

	org=from.origin+from.view_ofs;

	dir=normalize(v_forward);

	traceline(org, org+dir*1000,FALSE,from);

	if(trace_ent!=targ)
		return FALSE;
	else
	{
		return TRUE;
	}
}

/*
====================================================================

EIDOLON

====================================================================
*/

/*
=====================================================
entity riderpath_findbest(entity subject_path)
MG
Returns closest rider path corner that "subject_path"
leads to.  Used for Rider Bosses.
=====================================================
*/
entity riderpath_findbest(entity subject_path)
{
entity search,found,best_path;
float next,num_points,position,bestdist,lastdist;

	num_points = 0;
	if (subject_path.next_path_1) 
		num_points += 1;
	if (subject_path.next_path_2) 
		num_points += 1;
	if (subject_path.next_path_3) 
		num_points += 1;
	if (subject_path.next_path_4) 
		num_points += 1;
	if (subject_path.next_path_5) 
		num_points += 1;
	if (subject_path.next_path_6) 
		num_points += 1;

	if (!num_points)
	{
		dprintf("rider path %s has no next points\n",subject_path.path_id);
		remove(self);
		return world;
	}

	bestdist=vlen(self.goalentity.origin-self.origin);
	lastdist=bestdist;
	position=0;
	best_path=world;
	while(position<num_points)
	{
		position+=1;
		if (position==1) 
			next = subject_path.next_path_1;
		else if (position==2) 
			next = subject_path.next_path_2;
		else if (position==3) 
			next = subject_path.next_path_3;
		else if (position==4) 
			next = subject_path.next_path_4;
		else if (position==5) 
			next = subject_path.next_path_5;
		else if (position==6) 
			next = subject_path.next_path_6;
		found = world;
		search = find(world, classname, "rider_path");
		while(search != world && found == world)
		{
			if (search.path_id == next) 
				found = search;
			else
				search = find(search, classname, "rider_path");
		}

		if (!found)
		{
			dprintf("Could not find rider path %s\n",next);
			remove(self);
			return world;
		}
		else
		{
			lastdist=vlen(self.goalentity.origin-found.origin);
			if(lastdist<bestdist)
			{
				best_path=found;
				bestdist=lastdist;
			}
		}
	}
	if (!best_path)
		return world;
	return best_path;			
}

/*
=====================================================================

MEZZOMAN
(Personal Favorite!)

=====================================================================
*/

/*
=========================================================
entity look_projectiles ()
MG
WARNING!  Expensive, should not be called often at all!
Searches a radius 1000 around self to find projectiles
(anything with movetype bounce, flymissile, bouncemissile)
then checks to see if it's heading at self (see "heading"
function in PROJBHVR.HC).  It finds the closest of these
and returns that entity.  If it finds nothing, returns
"world".
=========================================================
*/
entity look_projectiles ()
{
entity found, enemy_proj;
float dist, bestdist;

	found=findradius(self.origin,1000);
	bestdist=1001;
	while(found)
	{
		if(found.movetype==MOVETYPE_FLYMISSILE||found.movetype==MOVETYPE_BOUNCE||found.movetype==MOVETYPE_BOUNCEMISSILE)
		if(visible(found))
		{
			dist=vlen(found.origin-self.origin);
			if(dist<bestdist)
			{
				if(heading(self,found,0.9))//Try this small range for heading, but problem is, they won't split up & surround you as much...
				{
					bestdist=dist;
					enemy_proj=found;
				}
			}
		}
		found=found.chain;
	}
	if(enemy_proj)
	{
		self.level=bestdist/vlen(enemy_proj.velocity);
		return enemy_proj;
	}
	else
		return world;
}


/*
======================================================
float solid_under(vector startpos , vector endpos)
MG
Will check in increments of 5 pixels along a given
path from startpos to endpos if there is solid gound
at least 18 pixels below (18 is the monster step-height).
If so, returns TRUE, if there is a gap anywhere along
that, returns FALSE.
Used mainly by Mezzomen to see if they should jump
across gaps at enemy.
======================================================
*/
float solid_under(vector startpos , vector endpos)
{
float diff_count;
vector dir;
	dir=normalize(endpos-startpos);
	diff_count=vlen(endpos-startpos)/5;
	while(diff_count>0)
	{
		traceline(startpos,startpos-'0 0 18',TRUE,self);
		if(trace_fraction==1)
			return FALSE;
		startpos+=dir*5;
		diff_count-=1;
	}
	return TRUE;
}

/*
======================================================
float check_heading_left_or_right (entity object)
MG
Will check to see if the given object will be to
the left or the right of self once it gets to
self.  Uses it's current position and extrapolates
based on it's heading (velocity).
Will return:
1  = left
-1 = right
0  = neither.
Special case: If called by a monster that's not
	awake, will return opposite of these assuming
	that the monster wants to cut off player-
	only used by the Rolling Ambush Mezzomen.
======================================================
*/
float check_heading_left_or_right (entity object)
{
vector	spot1, spot2, vec;
float dot, rng, reverse;
	
	makevectors (self.angles);
	spot1 = self.origin + self.view_ofs;
	spot2 = object.origin;
//To get the eventual location of the projectile when it gets to him...
	rng=vlen(spot1-spot2);
	spot2+=normalize(object.velocity)*(rng+15);//Add a bit for good measure

	vec = normalize (spot2 - spot1);
//FIXME? What about behind me?

	if(((object.classname=="player")||(object.classname=="bot"))&&!self.monster_awake)
	{
		self.monster_awake=TRUE;
		sound(self,CHAN_VOICE,"mezzo/attack.wav",1,ATTN_NORM);
		reverse=-1;
	}
	else
		reverse=1;

	dot = vec * v_right;
	if ( dot > 0)
		return -1*reverse;

	dot = vec * (v_right*-1);
	if ( dot > 0)
		return 1*reverse;

	return 0;
}

/*
======================================================
float navigate (float walkspeed)
MG
Checks to see which side of the entity is blocked
and will move in the opposite direction using
walkmove (for left-right) or movestep (for up-down)
if it can.  Will move the specified distance.
If it can't move that way or it doesn't find a blocked
side, it returns false.

Meant for use with flying and swimming monsters
because movetogoal doesn't make them navigate!
======================================================
*/
float navigate (float walkspeed)
{
vector checkdir,org,new_angle;
float vert_size,horz_size;
	makevectors(self.angles);
	checkdir=v_right;
	org=self.origin+checkdir*self.size_x;
	vert_size=self.size_z/2;
	horz_size=(self.size_x+self.size_y)/4;
	traceline(org,org+v_forward*horz_size,FALSE,self);
	if(trace_fraction==1&&!trace_allsolid)
	{
		checkdir=v_right*-1;
		org=self.origin+checkdir*horz_size;
		traceline(org,org+v_forward*horz_size,FALSE,self);
	}
	if(self.flags&FL_FLY||self.flags&FL_SWIM)
	{
		if(trace_fraction==1&&!trace_allsolid)
		{
			checkdir=v_up;
			org=self.origin+checkdir*vert_size;
			traceline(org,org+v_forward*horz_size,FALSE,self);
		}
		if(trace_fraction==1&&!trace_allsolid)
		{
			checkdir=v_up*-1;
			org=self.origin+checkdir*vert_size;
			traceline(org,org+v_forward*horz_size,FALSE,self);
		}
	}
	if(trace_fraction<1||trace_allsolid)
	{
		if(checkdir==v_right||checkdir==v_right*-1)
		{
			new_angle=vectoangles(checkdir*-1);
			if(!walkmove(new_angle_y,walkspeed,FALSE))
			{
				return FALSE;
			}
			return TRUE;
		}
		if(checkdir==v_up)
			walkspeed*=-1;
		if(!movestep(0,0,walkspeed,FALSE))
		{
			return FALSE;
		}
		return TRUE;
	}
	return FALSE;//FOUND NO BLOCKING!!!
}

/*
=============================================================
vector extrapolate_pos_for_speed (vector p1,float pspeed,entity targ,float accept) 
MG
Estimates where the "targ" will be by the time a projectile
travelling at "pspeed" leaving "org" arrives at "targ"'s origin.
It then calculates a new spot to shoot at so that the
projectile will arrive at such spot at the same time as
"targ".  Will return '0 0 0' (FALSE) if there is not a clear
line of fire to the spot or if the new vector is out of the
acceptable range (based on dot product of original vec and
the new vec).
=============================================================
*/
vector extrapolate_pos_for_speed (vector p1,float pspeed,entity targ,float accept) 
{
float dist1,dist2,tspeed,dot,eta1,eta2,eta_delta,failed;
vector p2,p3,targ_dir,vec1,vec2;

	p2=targ.origin+targ.view_ofs;		//current target viewport
	vec1=p2 - p1;						//vector to p2
	dist1=vlen(vec1);					//distance to p2
	vec1=normalize(vec1);				//direction to p2
	targ_dir=targ.velocity;				//target velocity
	tspeed=vlen(targ_dir);				//target speed
	targ_dir=normalize(targ_dir);		//target direction
	
	eta1=dist1/pspeed;					//Estimated time of arrival of projectile to p2

	p3=p2 + targ_dir * tspeed * eta1;	//Extrapolated postion of targ at time + eta1
	dist2=vlen(p3-p1);					//new distance to p3

	eta2=dist2/pspeed;					//ETA of projectile to p3
	eta_delta=eta2-eta1;				//change in ETA's

	p3+=targ_dir*tspeed*eta_delta*random();//Add any diff in ETA to p3's location,random a little in case they slow down
	
	traceline(p1,p3,FALSE,self);
	if(trace_fraction<1)
	{
		if(trace_ent.thingtype>=THINGTYPE_WEBS)
			traceline (trace_endpos, p3, FALSE, trace_ent);
		if(trace_fraction<1)	
			if(trace_ent.health>25||!trace_ent.takedamage||(trace_ent.flags&FL_MONSTER&&trace_ent.classname!="player_sheep"))
			{//Don't have a clear shot, and don't want to shoot obstruction
				self.attack_state = AS_SLIDING;
				failed=TRUE;
			}
	}

	vec2=normalize(p3-p1);				//New vector to p3
	dot=vec1*vec2;
	if(dot<accept)						//Change in dir too great
	{
		failed=TRUE;
	}

	if(failed)
		p3='0 0 0';
			
	return p3;
}

/*
=============================================================
vector aim_adjust (entity targ)
MG
Will return a nprmalized offset vector based on the targ's 
light level, used for monster aiming at shadow-hiding players.
=============================================================
*/
vector aim_adjust (entity targ)
{
float ofs;
vector vofs;
	if(!targ)
		return '0 0 0';

	makevectors(self.angles);
	get_visibility(targ,TRUE);
	ofs=(1 - targ.visibility - skill/10)*0.1;
	if(skill<3&&ofs>0)
	{
		vofs=v_up*0.5*random(0-ofs,ofs)+v_right*1.5*random(0-ofs,ofs);
		return vofs;
	}
	return '0 0 0';
}

