/* Worminator Game Engine */
/* This program copyright 1998 by David Layne */
/* Last modified: 1:03 AM 8/29/98 */

/**************\
|** INCLUDES **|
\**************/
#include <fcntl.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <stdio.h>
#include "allegro.h"
#include "proto.h"  // All function prototypes
#include "wormdata.h"  // Allegro data file header
#include "globals.h"  // Most global variables and typedefs needed to run worminator
#include "infodefs.h"  // Declarations describing info tiles
#include "grphdefs.h"  // Declarations describing graphical property tiles
// My files
#include "text.h"  // Text routines and a global table
#include "sound.h"  // A couple special sound routines
#include "sprites.h"  // All sprite functions and declarations
#include "player.h"  // Functions dealing with the worminator
#include "anim.h"  // Functions to handle tile animation
#include "scripts.h"  // Introduction movies and special level events, as well as campaign game stuff
#include "menus.h"  // All the menu code for worminator
#include "beams.h"  // Functions to deal with beam objects

#define WORMINATOR_VERSION "Final 1.0:  Please report bugs!"

/***************************\
|** GAME TIMEING FUNCTION **|
\***************************/
void increment_speed_counter()
{
if (game_is_running == TRUE) {
	speed_counter++;
	if (speed_counter > 256) speed_counter = 256;
	}
else speed_counter = 0;
}
END_OF_FUNCTION(increment_speed_counter);

/*******************\
|** MAIN FUNCTION **|
\*******************/
int main()
{
initialize();

// Introduction videos for the full version users
if (version_size == FULL_VERSION) {
	play_memory_fli(worminator_data_file[COMPANY_LOGO_MOVIE].dat, screen, FALSE, NULL);
	rest(1000);
	fade_out(2);
	// BlackOut the screen
	rectfill(screen, 0, 0, 320, 200, 255);
	play_memory_fli(worminator_data_file[WORMINATOR_LOGO_MOVIE].dat, screen, FALSE, NULL);
	rest(500);
	fade_out(2);
	// BlackOut the screen
	rectfill(screen, 0, 0, 320, 200, 255);
	}

// Main menu
set_pallete(worminator_data_file[DEFAULT_WORMINATOR_PALLETE].dat);
blit(worminator_data_file[MAIN_TITLE_SCREEN].dat, screen, 0, 0, 0, 0, 320, 200);
current_level = -69;
wormy_menu();

do {  // Start core game loop
	game_is_running = TRUE;
	while (speed_counter > 0 && time_to_quit == FALSE) {
		game_is_running = TRUE;
		handle_input();
		if (quake_timer > 0) quake_timer--;
		if (player.timer > 0) player.timer--;
		speed_counter--;
		}
	render_map();
} while(time_to_quit == FALSE);  // End core game loop

return 0;
}

/******************\
|** HANDLE INPUT **|
\******************/
void handle_input()
{
// Go to the main menu if nessicary and allow for hotkeys
if (key[KEY_ESC]) {
	wormy_menu();
	return;
	}
if (key[KEY_F1]) {
	display_snapshot(0, FALSE);
	return;
	}
if (key[KEY_F2]) {
	if (wormy_dummy_box(3, FALSE) == TRUE) save_game("saved.dat");
	return;
	}
if (key[KEY_F3] && exists("saved.dat")) {
	if (wormy_dummy_box(1, FALSE) == TRUE) load_saved_game("saved.dat");
	return;
	}
if (key[KEY_F4]) {
	if (wormy_dummy_box(2, FALSE) == TRUE) {
		current_level = 1;
		clear_player();
		player.score = 0;
		load_map(NULL);
		}
	return;
	}
if (key[KEY_F10]) {
	if (wormy_dummy_box(0, FALSE) == TRUE) time_to_quit = TRUE;
	return;
	}
if (key[KEY_F12]) {
	start_new_custom_game();
	return;
	}

// Update key states //////////////////////////////////////////////////////////////////////
if ((key[KEY_LCONTROL] || key[KEY_RCONTROL])) {
	if (jump_key_state == 0) jump_key_state = 1;
	else if (jump_key_state == 1) jump_key_state = 2;
	}
else jump_key_state = 0;

if (key[KEY_UP]) {
	if (up_key_state == 0) up_key_state = 1;
	else if (up_key_state == 1) up_key_state = 2;
	}
else up_key_state = 0;

// Update looking up/down info.
if (key[KEY_UP] && (key[KEY_LSHIFT] || key[KEY_RSHIFT]) && player.in_water == FALSE && player.on_ground == TRUE && player.x_acceleration == 0) {
	if (look_timer > 0) look_timer = -31;
	if (look_shift < 0 && look_timer > -31) look_timer = -31;
	look_timer--;
	if (look_timer < -32) {
		look_shift--;
		if (look_shift < -48) look_shift = -48;
		look_timer = -31;
		}
	}
else if (key[KEY_DOWN] && (key[KEY_LSHIFT] || key[KEY_RSHIFT]) && player.in_water == FALSE && player.on_ground == TRUE && player.x_acceleration == 0) {
	if (look_timer < 0) look_timer = 31;
	if (look_shift > 0 && look_timer < 31) look_timer = 31;
	look_timer++;
	if (look_timer > 32) {
		look_shift++;
		if (look_shift > 48) look_shift = 48;
		look_timer = 31;
		}
	}
else if ((!key[KEY_DOWN] && !key[KEY_UP]) && !(key[KEY_LSHIFT] || key[KEY_RSHIFT]) || player.in_water == TRUE || player.on_ground == FALSE || player.x_acceleration != 0) {
	if (look_timer > 1) look_timer = 0;
	if (look_timer < -1) look_timer = 0;
	if (look_shift != 0) {
		look_timer++;
		if (look_timer > 1) {
			if (look_shift > 0) look_shift--;
			else look_shift++;
			look_timer = 0;
			}		
		}
	else look_timer = 0;
	}

if (key[KEY_SPACE] || key[KEY_ALT] || key[KEY_ALTGR]) fire_key_state++;
else if (fire_key_state > 1) fire_key_state++;
else fire_key_state = 0;

// Update player water state and spawn splashes / bubbles
if (worminator_map.info_tile_grid[(int)((player.x_position_in_pixels + 16) / 16)][(int)((player.y_position_in_pixels + 8) / 16)] == water_info_tile) {
	if (player.in_water == FALSE) spawn_sprite(WATER_SPLASH_SPRITE, player.x_position_in_pixels, player.y_position_in_pixels - 20, 0, 0, 0);
	if (rand() % 64 == 32) spawn_sprite(WATER_BUBBLE_SPRITE, player.x_position_in_pixels + 16, player.y_position_in_pixels + 16, 0, 0, 0);
	player.in_water = TRUE, player.on_ground = TRUE;
	}
else player.in_water = FALSE;

// Update fire key state
if (player.current_weapon == 1 && fire_key_state > 10) fire_key_state = 0;
else if (player.current_weapon == 2 && fire_key_state > 128) fire_key_state = 0;
else if (player.current_weapon == 3 && fire_key_state > 64) fire_key_state = 0;
else if (player.current_weapon == 4 && fire_key_state > 128) fire_key_state = 0;
else if (player.current_weapon == 5 && fire_key_state > 32) fire_key_state = 0;
else if (player.current_weapon == 6 && fire_key_state > 46) fire_key_state = 0;
else if (player.current_weapon == 7 && fire_key_state > 96) fire_key_state = 0;
else if (player.current_weapon == 8 && fire_key_state > 384) fire_key_state = 0;
else if (player.current_weapon == 9 && fire_key_state > 256) fire_key_state = 0;
else if (player.current_weapon == 0) fire_key_state = 0;

// Select a weapon if one of the number keys is pressed
if (key[KEY_1] && player.current_weapon != 0) desired_weapon = 0;
if (key[KEY_2] && player.current_weapon != 1) desired_weapon = 1;
if (key[KEY_3] && player.current_weapon != 2) desired_weapon = 2;
if (key[KEY_4] && player.current_weapon != 3) desired_weapon = 3;
if (key[KEY_5] && player.current_weapon != 4) desired_weapon = 4;
if (key[KEY_6] && player.current_weapon != 5) desired_weapon = 5;
if (key[KEY_7] && player.current_weapon != 6) desired_weapon = 6;
if (key[KEY_8] && player.current_weapon != 7) desired_weapon = 7;
if (key[KEY_9] && player.current_weapon != 8) desired_weapon = 8;
if (key[KEY_0] && player.current_weapon != 9) desired_weapon = 9;

// Switch weapons if possible
if (fire_key_state == 0 && player.current_weapon != desired_weapon) player.current_weapon = desired_weapon, update_player_stats();

// Start fire state running if the player is pressing the fire key
if (fire_key_state == 0 && (key[KEY_SPACE] || key[KEY_ALT])) fire_key_state = 1;

// Fire off weapons that fire at the start of the animation
if (fire_key_state == 1) {
	if (player.current_weapon == 1) {
		if (player.bullets_left > 0) {
			if (player.direction == 0) spawn_sprite(WORMINATOR_BULLET_SPRITE, player.x_position_in_pixels + 27, player.y_position_in_pixels + 14 + (rand() % 5), 3, NULL, 0);
			else spawn_sprite(WORMINATOR_BULLET_SPRITE, player.x_position_in_pixels, player.y_position_in_pixels + 14 + (rand() % 5), -3, NULL, 1);
			play_sample(worminator_data_file[MINIGUN_FIRE_SOUND].dat, 255, 128, 1000, FALSE);
			player.bullets_left--;
			update_player_stats();
			}
		}

	else if (player.current_weapon == 2) {
		if (player.shells_left > 0) {
			fire_shotgun();
			play_sample(worminator_data_file[SHOTGUN_FIRE_SOUND].dat, 255, 128, 1000, FALSE);
			player.shells_left--;
			update_player_stats();
			}
		}

	else if (player.current_weapon == 3 && player.in_water == FALSE) {
		if (player.rockets_left > 0) {
			if (player.direction == 0) spawn_sprite(WORMINATOR_ROCKET_SPRITE, player.x_position_in_pixels + 24, player.y_position_in_pixels + 11, .25, 0, 0);
			else spawn_sprite(WORMINATOR_ROCKET_SPRITE, player.x_position_in_pixels - 9, player.y_position_in_pixels + 11, -.25, 0, 1);
			play_sample(worminator_data_file[ROCKET_FIRE_SOUND].dat, 255, 128, 1000, FALSE);
			player.rockets_left--;
			update_player_stats();
			}
		}

	else if (player.current_weapon == 4 && player.in_water == FALSE) {
		if (player.rockets_left > 1) {
			if (player.direction == 0) {
				spawn_sprite(WORMINATOR_ROCKET_SPRITE, player.x_position_in_pixels + 24, player.y_position_in_pixels + 11, .25, -.15, 0);
				spawn_sprite(WORMINATOR_ROCKET_SPRITE, player.x_position_in_pixels + 24, player.y_position_in_pixels + 11, .25, .15, 0);
				}
			else {
				spawn_sprite(WORMINATOR_ROCKET_SPRITE, player.x_position_in_pixels - 9, player.y_position_in_pixels + 11, -.25, -.15, 1);
				spawn_sprite(WORMINATOR_ROCKET_SPRITE, player.x_position_in_pixels - 9, player.y_position_in_pixels + 11, -.25, .15, 1);
				}
			play_sample(worminator_data_file[ROCKET_FIRE_SOUND].dat, 255, 128, 1000, FALSE);
			player.rockets_left -= 2;
			update_player_stats();
			}
		}

	else if (player.current_weapon == 5 && player.in_water == FALSE) {
		if (player.cells_left > 0) {
			fire_laser_gun();
			play_sample(worminator_data_file[LASER_GUN_FIRE_SOUND].dat, 255, 128, 1000, FALSE);
			player.cells_left--;
			update_player_stats();
			}
		}

	else if (player.current_weapon == 6 && player.in_water == FALSE) {
		if (player.gasoline_left > 0) {
			player.gasoline_left--;
			update_player_stats();
			}
		}

	else if (player.current_weapon == 7 && player.in_water == FALSE) {
		if (player.mortars_left > 0) {
			if (player.direction == 0) spawn_sprite(WORMINATOR_MORTAR_SPRITE, player.x_position_in_pixels + 10, player.y_position_in_pixels + 10, 1, -1.5, 0);
			else spawn_sprite(WORMINATOR_MORTAR_SPRITE, player.x_position_in_pixels + 10, player.y_position_in_pixels + 10, -1, -1.5, 1);
			play_sample(worminator_data_file[MORTAR_FIRE_SOUND].dat, 255, 128, 1000, FALSE);
			player.mortars_left--;
			update_player_stats();
			}
		}
	else if (player.current_weapon == 8 && player.in_water == FALSE) {
		if (player.plasma_left > 0) play_sample(worminator_data_file[BFG_FIRE_SOUND].dat, 255, 128, 1000, FALSE);
		}
	}

// Fire off the bfg plasma ball if nessecary
if (fire_key_state == 180 && player.current_weapon == 8 && player.plasma_left > 0 && player.in_water == FALSE) {
	if (player.direction == 0) spawn_sprite(WORMINATOR_BFG_PLASMA_BALL_SPRITE, player.x_position_in_pixels + 31, player.y_position_in_pixels + 7, .25, 0, 0);
	else spawn_sprite(WORMINATOR_BFG_PLASMA_BALL_SPRITE, player.x_position_in_pixels - 31, player.y_position_in_pixels + 7, -.25, 0, 1);
	player.plasma_left--;
	update_player_stats();
	quake_timer += 48;
	}

// Lay a mine if nessecary
if (fire_key_state == 60 && player.current_weapon == 9 && player.mines_left > 0 && player.in_water == FALSE) {
	if (player.direction == 0) spawn_sprite(WORMINATOR_MINE_SPRITE, player.x_position_in_pixels + 21, player.y_position_in_pixels + 27, 0, 0, 0);
	else spawn_sprite(WORMINATOR_MINE_SPRITE, player.x_position_in_pixels + 1, player.y_position_in_pixels + 27, 0, 0, 1);
	player.mines_left--;
	update_player_stats();
	}

// Water keyscan movement code
if (player.in_water == TRUE) {
	if (key[KEY_UP]) player.y_acceleration -= .02;
	else if (key[KEY_DOWN]) player.y_acceleration += .02;
	else {
		if (player.y_acceleration > 0) {
			player.y_acceleration -= .02;
			if (player.y_acceleration < 0) player.y_acceleration = 0;
			}
		else {
			player.y_acceleration += .02;
			if (player.y_acceleration > 0) player.y_acceleration = 0;
			}
		}
	if (key[KEY_LEFT]) player.x_acceleration -= .02, player.direction = 1;
	else if (key[KEY_RIGHT]) player.x_acceleration += .02, player.direction = 0;
	else {
		if (player.x_acceleration > 0) {
			player.x_acceleration -= .02;
			if (player.x_acceleration < 0) player.x_acceleration = 0;
			}
		else {
			player.x_acceleration += .02;
			if (player.x_acceleration > 0) player.x_acceleration = 0;
			}
		}
	}

// Normal keyscan movement code
else {
	if (up_key_state == 1) {
		if (player.on_ground == TRUE) {
			if (check_player_teleport() == TRUE) return;
			}
		}

	if ((key[KEY_LEFT]) && !(key[KEY_RIGHT])) {
		player.direction = 1;
		if (player.on_ground == TRUE) {
			player.x_acceleration -= .05;
			}
		else {
			player.x_acceleration -= .02;
			}
		}

	if ((key[KEY_RIGHT]) && !(key[KEY_LEFT])) {
		player.direction = 0;
		if (player.on_ground == TRUE) {
			player.x_acceleration += .05;
			}
		else {
			player.x_acceleration += .02;
			}
		}

	if ((key[KEY_LCONTROL] || key[KEY_RCONTROL])) {
		if (key[KEY_DOWN]) push_player(512, -512);
		else if (player.on_ground == TRUE) player.y_acceleration = -1.7;
		}
	}

// Debug Mode Enabler/Disabler
if (key[KEY_D] && key[KEY_M] && key[KEY_L]) debug_mode_on = TRUE;
else if (key[KEY_K] && key[KEY_R] && key[KEY_P]) debug_mode_on = FALSE;

if (debug_mode_on == TRUE) {
	if (key[KEY_G]) hurt_player(10);
	if (key[KEY_W]) player.bullets_left = 999, player.shells_left = 50, player.rockets_left = 75, player.cells_left = 75, player.gasoline_left = 25, player.mortars_left = 25, player.plasma_left = 10, player.mines_left = 15;
	}

// GRAVITY /////////////////////////////////////////////////////////////////////////////////
if (player.in_water == FALSE) {
	if (player.y_acceleration <= -.5 && !(key[KEY_LCONTROL] || key[KEY_RCONTROL]) && turbo_launch == FALSE) player.y_acceleration = -.5;
	player.on_ground = FALSE;
	if (player.y_acceleration != 0 && player.on_ground == FALSE) {
		if (player.y_acceleration > 0) {
			turbo_launch = FALSE;
			if (player.y_acceleration <= 1) push_player(0, player.y_acceleration);
			else {
				push_player(0, 1);
				push_player(0, player.y_acceleration - 1);
				}
			}
		else if (player.y_acceleration < 0) {
			if (player.y_acceleration >= -1) push_player(0, player.y_acceleration);
			else {
				push_player(0, -1);
				push_player(0, player.y_acceleration + 1);
				}
			}
		}
	player.y_acceleration += 0.02;
	if (player.on_ground == TRUE && player.y_acceleration > 1) player.y_acceleration = 1;
	else if (player.y_acceleration > 2) player.y_acceleration = 2;
	}

// LEFT AND RIGHT MOVEMENT CODE /////////////////////////////////////////////////////////////
if (player.in_water == FALSE && !key[KEY_LEFT] && !key[KEY_RIGHT]) {
	if (worminator_map.info_tile_grid[(int)((player.x_position_in_pixels) / 16)][(int)((player.y_position_in_pixels + 32) / 16)] == half_slippery_info_tile || worminator_map.info_tile_grid[(int)((player.x_position_in_pixels + 16) / 16)][(int)((player.y_position_in_pixels + 32) / 16)] == half_slippery_info_tile || worminator_map.info_tile_grid[(int)((player.x_position_in_pixels + 32) / 16)][(int)((player.y_position_in_pixels + 32) / 16)] == half_slippery_info_tile) {
		if (player.x_acceleration > 0) {
			player.x_acceleration -= .015;
			if (player.x_acceleration < 0) player.x_acceleration = 0;
			}
		else {
			player.x_acceleration += .015;
			if (player.x_acceleration > 0) player.x_acceleration = 0;
			}
		}
	else if (worminator_map.info_tile_grid[(int)((player.x_position_in_pixels) / 16)][(int)((player.y_position_in_pixels + 32) / 16)] == mostly_slippery_info_tile || worminator_map.info_tile_grid[(int)((player.x_position_in_pixels + 16) / 16)][(int)((player.y_position_in_pixels + 32) / 16)] == mostly_slippery_info_tile || worminator_map.info_tile_grid[(int)((player.x_position_in_pixels + 32) / 16)][(int)((player.y_position_in_pixels + 32) / 16)] == mostly_slippery_info_tile) {
		if (player.x_acceleration > 0) {
			player.x_acceleration -= .0025;
			if (player.x_acceleration < 0) player.x_acceleration = 0;
			}
		else {
			player.x_acceleration += .0025;
			if (player.x_acceleration > 0) player.x_acceleration = 0;
			}
		}
	else player.x_acceleration = 0;
	}

if (player.x_acceleration < -1) player.x_acceleration = -1;
if (player.x_acceleration > 1) player.x_acceleration = 1;

push_player(player.x_acceleration, 0);

// UP AND DOWN MOVEMENT CODE (FOR WATER ONLY) ///////////////////////////////////////////////
if (player.in_water == TRUE) {
	if (player.y_acceleration < -1) player.y_acceleration = -1;
	if (player.y_acceleration > 1) player.y_acceleration = 1;

	push_player(0, player.y_acceleration);
	}

// Set screen position
screen_x_position = (int)(player.x_position_in_pixels - 112);
screen_y_position = (int)(player.y_position_in_pixels - 80 + look_shift);

if (screen_x_position > worminator_map.map_width * 16 - 256) screen_x_position = worminator_map.map_width * 16 - 256;
if (screen_y_position > worminator_map.map_height * 16 - 192) screen_y_position = worminator_map.map_height * 16 - 192;

if (screen_x_position < 0) screen_x_position = 0;
if (screen_y_position < 0) screen_y_position = 0;

update_sprites();
update_beams();
//if (key[KEY_C]) save_bitmap("dump.bmp", double_buffer, worminator_data_file[DEFAULT_WORMINATOR_PALLETE].dat);

pick_player_frame();

run_scripts();

if (player.frame != NULL && player.health_bars_left < 1) kill_player();

update_text_messages();
}

/****************\
|** RENDER MAP **|
\****************/
void render_map()
{
unsigned char x_tile_loop, y_tile_loop;
char weather_loop;
unsigned int start_x_tile = 0, start_y_tile = 0;
int sx = screen_x_position, sy = screen_y_position;
int plax_x, plax_y;
int sprite_loop;


if (parallax_background->w == 256 || worminator_map.map_width == 16) plax_x = 0;
else plax_x = (float)screen_x_position * (float)((float)(parallax_background->w - 256) / (worminator_map.map_width * 16 - 256));

if (parallax_background->h == 192 || worminator_map.map_height == 12) plax_y = 0;
else plax_y = (float)screen_y_position * (float)((float)(parallax_background->h - 192) / (worminator_map.map_height * 16 - 192));

// Draw the parallaxing background
blit(parallax_background, double_buffer, plax_x, plax_y, 0, 0, 256, 192);

// Snow
if (worminator_map.weather == 1) for (weather_loop = 0; weather_loop < 64; weather_loop++) putpixel(double_buffer, weather[weather_loop][0], weather[weather_loop][1], 255);

// Rain
else if (worminator_map.weather == 2) {
	for (weather_loop = 0; weather_loop < 64; weather_loop++) {
		putpixel(double_buffer, weather[weather_loop][0], weather[weather_loop][1], 20);
		putpixel(double_buffer, weather[weather_loop][0], weather[weather_loop][1] - 1, 22);
		putpixel(double_buffer, weather[weather_loop][0] + 1, weather[weather_loop][1] - 2, 24);
		putpixel(double_buffer, weather[weather_loop][0] + 1, weather[weather_loop][1] - 3, 26);
		}
	}

while (sx >= 16) {
	start_x_tile++;
	sx -= 16;
	};

while (sy >= 16) {
	start_y_tile++;
	sy -= 16;
	};

for(x_tile_loop = 0; x_tile_loop < 17; x_tile_loop++) {
	for(y_tile_loop = 0; y_tile_loop < 13; y_tile_loop++) {

		// If the background tile is masked, draw it allowing for transparency
		if (worminator_map.graphical_properties_tile_grid[start_x_tile + x_tile_loop][start_y_tile + y_tile_loop] == masked_background_grph_tile || worminator_map.graphical_properties_tile_grid[start_x_tile + x_tile_loop][start_y_tile + y_tile_loop] == masked_everything_grph_tile) masked_blit(worminator_tiles, double_buffer, worminator_map.background_tile_grid[start_x_tile + x_tile_loop][start_y_tile + y_tile_loop] * 16, 0, x_tile_loop * 16 - sx, y_tile_loop * 16 - sy, 16, 16);

		// If the tile is a normal background tile,  draw it.
		else if (worminator_map.background_tile_grid[start_x_tile + x_tile_loop][start_y_tile + y_tile_loop] != 0) blit(worminator_tiles, double_buffer, worminator_map.background_tile_grid[start_x_tile + x_tile_loop][start_y_tile + y_tile_loop] * 16, 0, x_tile_loop * 16 - sx, y_tile_loop * 16 - sy, 16, 16);

		// Draw the foreground tile if it is a masked tile and has a tile
		if (worminator_map.foreground_tile_grid[start_x_tile + x_tile_loop][start_y_tile + y_tile_loop] != NULL) {
			if (worminator_map.graphical_properties_tile_grid[start_x_tile + x_tile_loop][start_y_tile + y_tile_loop] == masked_foreground_grph_tile || worminator_map.graphical_properties_tile_grid[start_x_tile + x_tile_loop][start_y_tile + y_tile_loop] == masked_everything_grph_tile) masked_blit(worminator_tiles, double_buffer, worminator_map.foreground_tile_grid[start_x_tile + x_tile_loop][start_y_tile + y_tile_loop] * 16, 0, x_tile_loop * 16 - sx, y_tile_loop * 16 - sy, 16, 16);
			}
		}
	}

for (sprite_loop = 0; sprite_loop < number_of_sprites; sprite_loop++) {
	if (sprites[sprite_loop].frame != NULL) {
		if (sprites[sprite_loop].rotate_factor == 0) {
			if (sprites[sprite_loop].direction == 0) draw_sprite(double_buffer, worminator_data_file[sprites[sprite_loop].frame].dat, (int)(sprites[sprite_loop].x_position_in_pixels - screen_x_position), (int)(sprites[sprite_loop].y_position_in_pixels - screen_y_position));
			else draw_sprite_h_flip(double_buffer, worminator_data_file[sprites[sprite_loop].frame].dat, (int)(sprites[sprite_loop].x_position_in_pixels - screen_x_position), (int)(sprites[sprite_loop].y_position_in_pixels - screen_y_position));
			}
		else rotate_sprite(double_buffer, worminator_data_file[sprites[sprite_loop].frame].dat, (int)(sprites[sprite_loop].x_position_in_pixels - screen_x_position), (int)(sprites[sprite_loop].y_position_in_pixels - screen_y_position), itofix(sprites[sprite_loop].rotate_factor));
		}
	}

// Render all beams and draw the player
render_beams();
draw_player();

// Render all foreground tiles and a couple graphical properties tiles
for(x_tile_loop = 0; x_tile_loop < 17; x_tile_loop++) {
	for(y_tile_loop = 0; y_tile_loop < 13; y_tile_loop++) {
		if (worminator_map.graphical_properties_tile_grid[start_x_tile + x_tile_loop][start_y_tile + y_tile_loop] == parallaxing_foreground_grph_tile) blit(parallax_background, double_buffer, x_tile_loop * 16 - sx + plax_x, y_tile_loop * 16 - sy + plax_y, x_tile_loop * 16 - sx, y_tile_loop * 16 - sy, 16, 16);
		if (worminator_map.foreground_tile_grid[start_x_tile + x_tile_loop][start_y_tile + y_tile_loop] != 0) {
			if (worminator_map.graphical_properties_tile_grid[start_x_tile + x_tile_loop][start_y_tile + y_tile_loop] == translucent_foreground_grph_tile) {
				blit(worminator_tiles, spawned_sprite, worminator_map.foreground_tile_grid[start_x_tile + x_tile_loop][start_y_tile + y_tile_loop] * 16, 0, 0, 0, 16, 16);
				draw_trans_sprite(double_buffer, spawned_sprite, x_tile_loop * 16 - sx, y_tile_loop * 16 - sy);
				}
			else if (worminator_map.graphical_properties_tile_grid[start_x_tile + x_tile_loop][start_y_tile + y_tile_loop] != masked_foreground_grph_tile && worminator_map.graphical_properties_tile_grid[start_x_tile + x_tile_loop][start_y_tile + y_tile_loop] != masked_everything_grph_tile) masked_blit(worminator_tiles, double_buffer, worminator_map.foreground_tile_grid[start_x_tile + x_tile_loop][start_y_tile + y_tile_loop] * 16, 0, x_tile_loop * 16 - sx, y_tile_loop * 16 - sy, 16, 16);
			}
		}
	}

if (debug_mode_on == TRUE) textout(double_buffer, worminator_data_file[DEFAULT_WORMINATOR_FONT].dat, "Debug mode ENABLED!", 0, 0, -1);

// Draw text messages
if (text_messages[0].timer != -1) textout(double_buffer, worminator_data_file[DEFAULT_WORMINATOR_FONT].dat, text_messages[0].message, 0, 0, -1);
if (text_messages[1].timer != -1) textout(double_buffer, worminator_data_file[DEFAULT_WORMINATOR_FONT].dat, text_messages[1].message, 0, 7, -1);
if (text_messages[2].timer != -1) textout(double_buffer, worminator_data_file[DEFAULT_WORMINATOR_FONT].dat, text_messages[2].message, 0, 14, -1);
if (text_messages[3].timer != -1) textout(double_buffer, worminator_data_file[DEFAULT_WORMINATOR_FONT].dat, text_messages[3].message, 0, 21, -1);

// Handle screen shakes and do actual rendering
if (quake_timer > 0) {
	set_clip(screen, 4, 4, 259, 195);
	if (quake_timer > 512) quake_timer = 512;
	if (quake_timer > 256) blit(double_buffer, screen, 0, 0, 4 + rand() % 16 - 8, 4 + rand() % 16 - 8, 256, 192);
	else if (quake_timer > 128) blit(double_buffer, screen, 0, 0, 4 + rand() % 8 - 4, 4 + rand() % 8 - 4, 256, 192);
	else if (quake_timer > 64) blit(double_buffer, screen, 0, 0, 4 + rand() % 4 - 2, 4 + rand() % 4 - 2, 256, 192);
	else blit(double_buffer, screen, 0, 0, 4 + rand() % 2 - 1, 4 + rand() % 2 - 1, 256, 192);
	set_clip(screen, 0, 0, 320, 200);
	}
else blit(double_buffer, screen, 0, 0, 4, 4, 256, 192);
}

/********************\
|** UPDATE WEATHER **|
\********************/
void update_weather()
{
char weather_loop;

// Snow
if (worminator_map.weather == 1) {
	for(weather_loop = 0; weather_loop < 64; weather_loop++) {
		weather[weather_loop][0] += (rand() % 20 - 10) / 10 * 2;
		weather[weather_loop][0] += player.x_acceleration * -1.5;

		weather[weather_loop][1]++;

		if (weather[weather_loop][1] > 200) {
			weather[weather_loop][0] = rand() % 320;
			weather[weather_loop][1] -= 200;
			}

		if (weather[weather_loop][0] < 0) weather[weather_loop][0] += 320;
		else if (weather[weather_loop][0] >= 320) weather[weather_loop][0] -= 320;
		}

}

// Rain
else if (worminator_map.weather == 2) {
	for(weather_loop = 0; weather_loop < 64; weather_loop++) {
		weather[weather_loop][0] += -1.5;
		weather[weather_loop][0] += player.x_acceleration * -.75;

		weather[weather_loop][1] += 2;

		if (weather[weather_loop][1] > 200) {
			weather[weather_loop][0] = rand() % 320;
			weather[weather_loop][1] -= 200;
			}

		if (weather[weather_loop][0] < 0) weather[weather_loop][0] += 320;
		else if (weather[weather_loop][0] >= 320) weather[weather_loop][0] -= 320;
		}
	}
}

/******************\
|** FIRE SHOTGUN **|
\******************/
void fire_shotgun()
{
char done_looping = FALSE;
char sw, sh;
unsigned char spawn_loop;
unsigned char test_tile;
int x1, x2, y;
int sx, sy;
int hit_x, hit_sprite = -1;
int x_tile, y_tile;
int collision_detect_loop;

y = player.y_position_in_pixels + 15 + (rand() % 2);

if (player.direction == 0) {
	x1 = x2 = player.x_position_in_pixels + 27;
	hit_x = 4096;
	do {
		if (x2 > worminator_map.map_width * 16) done_looping = TRUE;
		test_tile = worminator_map.info_tile_grid[(int)((x2 + 16) / 16)][(int)(y / 16)];

		if (test_tile == solid_info_tile) done_looping = TRUE;
		else if (test_tile == solid_from_left_no_ammo_info_tile) done_looping = TRUE;
		else if (test_tile == painful_from_top_info_tile) done_looping = TRUE;
		else if (test_tile == painful_from_bottom_info_tile) done_looping = TRUE;
		else if (test_tile >= slope_up_left_to_right_45_info_tile && test_tile <= slope_filler_info_tile && test_tile != slope_up_left_to_right_225_left_info_tile && test_tile != slope_up_right_to_left_225_right_info_tile) done_looping = TRUE;
		else if (test_tile == breakable_wall_info_tile) done_looping = TRUE;

		if (done_looping == FALSE) x2 += 16;
		} while (done_looping == FALSE);
	for (collision_detect_loop = 0; collision_detect_loop < number_of_sprites; collision_detect_loop++) {
		if (sprites[collision_detect_loop].hit_points_left > 0) {
			sx = sprites[collision_detect_loop].x_position_in_pixels;
			sy = sprites[collision_detect_loop].y_position_in_pixels;
			sw = sprites[collision_detect_loop].width, sh = sprites[collision_detect_loop].height;
			if (y >= sy && y <= sy + sh && ((sx >= x1 && sx <= x2) || (sx + sw >= x1 && sx + sw <= x2))) {
				if (sx < hit_x) hit_sprite = collision_detect_loop, hit_x = sx;
				}
			}
		}
	if (hit_sprite != -1) sprites[hit_sprite].hit_points_left -= (.75 + (float)(rand() % 10) / 10.00), x2 = hit_x;
	for (spawn_loop = 0; spawn_loop < 4; spawn_loop++) spawn_sprite(SMALL_WHITE_SPARK_SPRITE, (int)(x2 * 16) / 16, y, -1, (float)((rand() % 20) - 10.00) / 10.00, NULL);
	}

else {
	x1 = x2 = player.x_position_in_pixels + 4;
	hit_x = 0;
	do {
		if (x2 < 0) done_looping = TRUE;
		test_tile = worminator_map.info_tile_grid[(int)((x2 - 16) / 16)][(int)(y / 16)];

		if (test_tile == solid_info_tile) done_looping = TRUE;
		else if (test_tile == solid_from_right_no_ammo_info_tile) done_looping = TRUE;
		else if (test_tile == painful_from_top_info_tile) done_looping = TRUE;
		else if (test_tile == painful_from_bottom_info_tile) done_looping = TRUE;
		else if (test_tile >= slope_up_left_to_right_45_info_tile && test_tile <= slope_filler_info_tile && test_tile != slope_up_left_to_right_225_left_info_tile && test_tile != slope_up_right_to_left_225_right_info_tile) done_looping = TRUE;
		else if (test_tile == breakable_wall_info_tile) done_looping = TRUE;

		if (done_looping == FALSE) x2 -= 16;
		} while (done_looping == FALSE);
	for (collision_detect_loop = 0; collision_detect_loop < number_of_sprites; collision_detect_loop++) {
		if (sprites[collision_detect_loop].hit_points_left > 0) {
			sx = sprites[collision_detect_loop].x_position_in_pixels;
			sy = sprites[collision_detect_loop].y_position_in_pixels;
			sw = sprites[collision_detect_loop].width, sh = sprites[collision_detect_loop].height;
			if (y >= sy && y <= sy + sh && ((sx >= x2 && sx <= x1) || (sx + sw >= x2 && sx + sw <= x1))) {
				if (sx > hit_x) hit_sprite = collision_detect_loop, hit_x = sx;
				}
			}
		}
	if (hit_sprite != -1) sprites[hit_sprite].hit_points_left -= (.75 + (float)(rand() % 10) / 10.00), x2 = hit_x + sprites[hit_sprite].width;
	for (spawn_loop = 0; spawn_loop < 4; spawn_loop++) spawn_sprite(SMALL_WHITE_SPARK_SPRITE, (int)(x2 * 16) / 16, y, 1, (float)((rand() % 20) - 10.00) / 10.00, NULL);
	}

if (test_tile == breakable_wall_info_tile && hit_sprite == -1) {
	if (player.direction == 0) x_tile = (int)((x2 + 16) / 16);
	else x_tile = (int)((x2 - 16) / 16);
	y_tile = (int)(y / 16);
	for (spawn_loop = 0; spawn_loop < 16; spawn_loop++) spawn_sprite(SMALL_WHITE_SPARK_SPRITE, x_tile * 16 + 8, y_tile * 16 + 8, (float)((rand() % 20) - 10.00) / 10.00, (float)((rand() % 20) - 10.00) / 10.00, NULL);
	worminator_map.foreground_tile_grid[x_tile][y_tile] = 0;
	worminator_map.info_tile_grid[x_tile][y_tile] = blank_info_tile;
	}
}

/********************\
|** FIRE LASER GUN **|
\********************/
void fire_laser_gun()
{
char done_looping = FALSE;
char sw, sh;
unsigned char spawn_loop;
unsigned char test_tile;
int x1, x2, y;
int sx, sy;
int x_tile, y_tile;
int collision_detect_loop;

y = player.y_position_in_pixels + 13;

if (player.direction == 0) {
	x1 = x2 = player.x_position_in_pixels + 36;
	do {
		if (x2 > worminator_map.map_width * 16) done_looping = TRUE;
		test_tile = worminator_map.info_tile_grid[(int)(x2 / 16)][(int)(y / 16)];

		if (test_tile == solid_info_tile) done_looping = TRUE;
		else if (test_tile == solid_from_left_no_ammo_info_tile) done_looping = TRUE;
		else if (test_tile == painful_from_top_info_tile) done_looping = TRUE;
		else if (test_tile == painful_from_bottom_info_tile) done_looping = TRUE;
		else if (test_tile >= slope_up_left_to_right_45_info_tile && test_tile <= slope_filler_info_tile && test_tile != slope_up_left_to_right_225_left_info_tile && test_tile != slope_up_right_to_left_225_right_info_tile) done_looping = TRUE;
		else if (test_tile == breakable_wall_info_tile) done_looping = TRUE;

		if (done_looping == FALSE) x2 += 16;
		} while (done_looping == FALSE);
	for (collision_detect_loop = 0; collision_detect_loop < number_of_sprites; collision_detect_loop++) {
		if (sprites[collision_detect_loop].hit_points_left > 0) {
			sx = sprites[collision_detect_loop].x_position_in_pixels;
			sy = sprites[collision_detect_loop].y_position_in_pixels;
			sw = sprites[collision_detect_loop].width, sh = sprites[collision_detect_loop].height;
			if (y >= sy && y <= sy + sh && ((sx >= x1 && sx <= x2) || (sx + sw >= x1 && sx + sw <= x2))) {
				sprites[collision_detect_loop].hit_points_left -= .5;
				for (spawn_loop = 0; spawn_loop < 4; spawn_loop++) spawn_sprite(SMALL_RED_SPARK_SPRITE, sx + (sw / 2), sy + (sh / 2), (float)((rand() % 40) - 20.00) / 20.00, (float)((rand() % 40) - 20.00) / 20.00, NULL);
				}
			}
		}
	x2 = (int)(x2 / 16) * 16;
	spawn_beam(player.x_position_in_pixels + 37, player.y_position_in_pixels + 13, x2, NULL, 217, SOLID_COLOR_BEAM);
	}

else {
	x1 = x2 = player.x_position_in_pixels - 4;
	do {
		if (x2 < 0) done_looping = TRUE;
		test_tile = worminator_map.info_tile_grid[(int)(x2 / 16)][(int)(y / 16)];

		if (test_tile == solid_info_tile) done_looping = TRUE;
		else if (test_tile == solid_from_right_no_ammo_info_tile) done_looping = TRUE;
		else if (test_tile == painful_from_top_info_tile) done_looping = TRUE;
		else if (test_tile == painful_from_bottom_info_tile) done_looping = TRUE;
		else if (test_tile >= slope_up_left_to_right_45_info_tile && test_tile <= slope_filler_info_tile && test_tile != slope_up_left_to_right_225_left_info_tile && test_tile != slope_up_right_to_left_225_right_info_tile) done_looping = TRUE;
		else if (test_tile == breakable_wall_info_tile) done_looping = TRUE;

		if (done_looping == FALSE) x2 -= 16;
		} while (done_looping == FALSE);
	for (collision_detect_loop = 0; collision_detect_loop < number_of_sprites; collision_detect_loop++) {
		if (sprites[collision_detect_loop].hit_points_left > 0) {
			sx = sprites[collision_detect_loop].x_position_in_pixels;
			sy = sprites[collision_detect_loop].y_position_in_pixels;
			sw = sprites[collision_detect_loop].width, sh = sprites[collision_detect_loop].height;
			if (y >= sy && y <= sy + sh && ((sx >= x2 && sx <= x1) || (sx + sw >= x2 && sx + sw <= x1))) {
				sprites[collision_detect_loop].hit_points_left -= .5;
				for (spawn_loop = 0; spawn_loop < 4; spawn_loop++) spawn_sprite(SMALL_RED_SPARK_SPRITE, sx + (sw / 2), sy + (sh / 2), (float)((rand() % 40) - 20.00) / 20.00, (float)((rand() % 40) - 20.00) / 20.00, NULL);
				}
			}
		}
	x2 = ((int)(x2 / 16) * 16) + 16;
	spawn_beam(x2, player.y_position_in_pixels + 13, player.x_position_in_pixels - 6, NULL, 217, SOLID_COLOR_BEAM);
	}
}

/**************\
|** LOAD MAP **|
\**************/
void load_map(char *file_name)
{
FILE *map_file;
unsigned char sprite_spawn_loop, sprite_type;

// Disable the game speed timer and remove earth quakes
game_is_running = FALSE;
quake_timer = 0;

// Fade out the display to black and set the pallete to black to cover up drawings
fade_out(32);
set_pallete(black_pallete);

// BlackOut the screen
rectfill(screen, 0, 0, 320, 200, 255);

// Reset all the sprite states
clear_sprites();

// Clear the letter order player variable and make the player face right
player.letter_order = NULL;
player.direction = 0;

// Open the map file
if (current_level == -1) map_file = fopen(file_name, "rb");
if (current_level == 1) map_file = fopen("map01.map", "rb");
if (current_level == 2) map_file = fopen("map02.map", "rb");
if (current_level == 3) map_file = fopen("map03.map", "rb");
if (current_level == 4) map_file = fopen("map04.map", "rb");
if (current_level == 5) map_file = fopen("map05.map", "rb");
if (current_level == 6) map_file = fopen("map06.map", "rb");
if (current_level == 7) map_file = fopen("map07.map", "rb");
if (current_level == 8) map_file = fopen("map08.map", "rb");
if (current_level == 9) map_file = fopen("map09.map", "rb");
if (current_level == 10) map_file = fopen("map10.map", "rb");
if (current_level == 11) map_file = fopen("map11.map", "rb");
if (current_level == 12) map_file = fopen("map12.map", "rb");
if (current_level == 13) map_file = fopen("map13.map", "rb");
if (current_level == 14) map_file = fopen("map14.map", "rb");
if (current_level == 15) map_file = fopen("map15.map", "rb");
if (current_level == 16) map_file = fopen("map16.map", "rb");

// Display a level briefing if appropriate
if (player_just_died == FALSE && current_level > 0 && current_level < 7) {
	set_pallete(worminator_data_file[DEFAULT_WORMINATOR_PALLETE].dat);
	display_snapshot(current_level + 5, FALSE);
	show_mouse(NULL);
	set_pallete(black_pallete);
	}

// Read the entire map and close the file.
fread(&worminator_map, sizeof(worminator_map), 1, map_file);
fclose(map_file);

// This re-calculates the animated tiles data array
build_animation_table();

// Position the player in the map
player.x_position_in_pixels = worminator_map.starting_sprites[0].x_position_in_pixels;
player.y_position_in_pixels = worminator_map.starting_sprites[0].y_position_in_pixels;

// Spawn every sprite in the level
for (sprite_spawn_loop = 1; sprite_spawn_loop < 255; sprite_spawn_loop++) {
	sprite_type = worminator_map.starting_sprites[sprite_spawn_loop].sprite_type;

//////// NCPs /////////////////////////////////////////////////////////////////////////////////
	if (sprite_type == 1) spawn_sprite(WORM_GIRL_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 2) spawn_sprite(WORM_BOY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 3) spawn_sprite(WORM_COP_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 4) spawn_sprite(WORM_GENTLEMEN_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 5) spawn_sprite(WORM_STRIPPER_ONE_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 6) spawn_sprite(WORM_BLONDE_NUDE_ONE_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);

//////// Getable Items ////////////////////////////////////////////////////////////////////////
	if (sprite_type == 16) spawn_sprite(HEALTH_1_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 17) spawn_sprite(HEALTH_2_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 18) spawn_sprite(HEALTH_5_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 19) spawn_sprite(HEALTH_10_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 20) spawn_sprite(MINIGUN_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 21) spawn_sprite(BULLETS_SMALL_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 22) spawn_sprite(BULLETS_LARGE_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 23) spawn_sprite(SHOTGUN_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 24) spawn_sprite(SHELLS_SMALL_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 25) spawn_sprite(SHELLS_LARGE_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 26) spawn_sprite(ROCKET_LAUNCHER_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 27) spawn_sprite(ROCKETS_SMALL_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 28) spawn_sprite(ROCKETS_LARGE_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 29) spawn_sprite(LASER_GUN_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 30) spawn_sprite(CELLS_SMALL_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 31) spawn_sprite(CELLS_LARGE_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 32) spawn_sprite(FLAMETHROWER_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 33) spawn_sprite(GASOLINE_SMALL_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 34) spawn_sprite(GASOLINE_LARGE_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 35) spawn_sprite(MORTAR_CANNON_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 36) spawn_sprite(MORTARS_SMALL_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 37) spawn_sprite(MORTARS_LARGE_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 38) spawn_sprite(BFG_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 39) spawn_sprite(PLASMA_SMALL_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 40) spawn_sprite(PLASMA_LARGE_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 41) spawn_sprite(MINELAYER_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 42) spawn_sprite(MINES_SMALL_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 43) spawn_sprite(MINES_LARGE_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);

//////// Enemies //////////////////////////////////////////////////////////////////////////////
	if (sprite_type == 48) spawn_sprite(SMALL_GUN_TURRET_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 0);
	if (sprite_type == 49) spawn_sprite(SMALL_GUN_TURRET_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 1);
	if (sprite_type == 50) spawn_sprite(MORTAR_GUN_TURRET_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 0);
	if (sprite_type == 51) spawn_sprite(MORTAR_GUN_TURRET_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 1);
	if (sprite_type == 52) spawn_sprite(BIG_GUN_TURRET_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 0);
	if (sprite_type == 53) spawn_sprite(BIG_GUN_TURRET_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 1);
	if (sprite_type == 54) spawn_sprite(FLYING_PROBE_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 0);
	if (sprite_type == 55) spawn_sprite(FLYING_PROBE_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 1);
	if (sprite_type == 56) spawn_sprite(ROBOTIC_DOG_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 0);
	if (sprite_type == 57) spawn_sprite(ROBOTIC_DOG_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 1);
	if (sprite_type == 58) spawn_sprite(DECIMATER_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 0);
	if (sprite_type == 59) spawn_sprite(DECIMATER_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 1);
	if (sprite_type == 60) spawn_sprite(GIANT_ROBOT_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 0);
	if (sprite_type == 61) spawn_sprite(GIANT_ROBOT_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 1);
	if (sprite_type == 62) spawn_sprite(SMALL_RUNNER_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 0);
	if (sprite_type == 63) spawn_sprite(SMALL_RUNNER_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 1);
	if (sprite_type == 64) spawn_sprite(MECHANICAL_CRUSHER_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 0);
	if (sprite_type == 65) spawn_sprite(MECHANICAL_CRUSHER_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 1);
	if (sprite_type == 66) spawn_sprite(NUCLEAR_WORM_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 0);
	if (sprite_type == 67) spawn_sprite(NUCLEAR_WORM_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 1);
	if (sprite_type == 68) spawn_sprite(LASER_REBEL_WORM_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 0);
	if (sprite_type == 69) spawn_sprite(LASER_REBEL_WORM_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 1);
	if (sprite_type == 70) spawn_sprite(ROCKET_REBEL_WORM_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 0);
	if (sprite_type == 71) spawn_sprite(ROCKET_REBEL_WORM_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 1);
	if (sprite_type == 72) spawn_sprite(RIFLE_REBEL_WORM_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 0);
	if (sprite_type == 73) spawn_sprite(RIFLE_REBEL_WORM_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 1);
	if (sprite_type == 74) spawn_sprite(BIG_GREEN_SLIME_BLOB_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 0);
	if (sprite_type == 75) spawn_sprite(BIG_GREEN_SLIME_BLOB_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 1);
	if (sprite_type == 76) spawn_sprite(GUMDROP_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 0);
	if (sprite_type == 77) spawn_sprite(GUMDROP_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 1);
	if (sprite_type == 78) spawn_sprite(BIG_NUCLEAR_BOMB_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 0);
	if (sprite_type == 79) spawn_sprite(SMALL_NUCLEAR_BOMB_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 0);
	if (sprite_type == 80) spawn_sprite(FIREWORM_SEGMENT_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 0);
	if (sprite_type == 81) spawn_sprite(FIREWORM_SEGMENT_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 1);
	if (sprite_type == 82) spawn_sprite(FLAME_JET_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 0);
	if (sprite_type == 83) spawn_sprite(FLAME_JET_ENEMY_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, 1);

//////// Platforms ////////////////////////////////////////////////////////////////////////////
	if (sprite_type >= 96 && sprite_type < 112) {
		spawn_sprite(FLOATING_PLATFORM_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
		sprites[number_of_sprites - 1].frame = FLOATING_PLATFORM_01 + sprite_type - 96;
		}

//////// Explosives and Misc //////////////////////////////////////////////////////////////////
	if (sprite_type == 112) spawn_sprite(SMALL_EXPLOSIVE_BARREL_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 113) spawn_sprite(SMALL_EXPLOSIVE_NUCLEAR_CRATE_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 114) spawn_sprite(SMALL_SPINNING_FAN_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 115) spawn_sprite(MEDIUM_SPINNING_FAN_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 116) spawn_sprite(LARGE_SPINNING_FAN_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 117) spawn_sprite(WORMINATOR_MINE_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels + 3, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels + 11, NULL, NULL, NULL);
	if (sprite_type == 118) spawn_sprite(PUSHABLE_CRATE_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);

//////// Doors ////////////////////////////////////////////////////////////////////////////////
	if (sprite_type == 128) spawn_sprite(GREY_DOOR_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 129) {
		spawn_sprite(GREY_DOOR_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
		sprites[number_of_sprites - 1].frame = GREY_DOOR_02;
		}
	if (sprite_type == 130) {
		spawn_sprite(GREY_DOOR_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
		sprites[number_of_sprites - 1].frame = GREY_DOOR_03;
		}
	if (sprite_type == 131) spawn_sprite(RED_KEYCARD_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 132) spawn_sprite(RED_DOOR_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 133) {
		spawn_sprite(RED_DOOR_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
		sprites[number_of_sprites - 1].frame = RED_DOOR_02;
		}
	if (sprite_type == 134) spawn_sprite(RED_DOOR_TOP_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 135) spawn_sprite(RED_DOOR_BOTTOM_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 136) spawn_sprite(GREEN_KEYCARD_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 137) spawn_sprite(GREEN_DOOR_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 138) {
		spawn_sprite(GREEN_DOOR_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
		sprites[number_of_sprites - 1].frame = GREEN_DOOR_02;
		}
	if (sprite_type == 139) spawn_sprite(GREEN_DOOR_TOP_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 140) spawn_sprite(GREEN_DOOR_BOTTOM_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 141) spawn_sprite(BLUE_KEYCARD_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 142) spawn_sprite(BLUE_DOOR_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 143) {
		spawn_sprite(BLUE_DOOR_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
		sprites[number_of_sprites - 1].frame = BLUE_DOOR_02;
		}

//////// Items Only Worth Points //////////////////////////////////////////////////////////////
	if (sprite_type == 144) spawn_sprite(LETTER_W_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 145) spawn_sprite(LETTER_O_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 146) spawn_sprite(LETTER_R_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 147) spawn_sprite(LETTER_M_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 148) spawn_sprite(SODA_SIXPACK_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 149) spawn_sprite(DOUGHNUTS_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 150) spawn_sprite(BOX_OF_DOUGHNUTS_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 151) spawn_sprite(MUD_BEER_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);

//////// Parallaxing Backgrounds //////////////////////////////////////////////////////////////
	if (sprite_type >= 160 && sprite_type < 170) {
		parallax_background = worminator_data_file[PARALLAXING_BACKGROUND_01 + sprite_type - 160].dat;
		}

//////// Spawners /////////////////////////////////////////////////////////////////////////////
	if (sprite_type == 170) spawn_sprite(BLUE_SPARK_SPAWNER_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 171) spawn_sprite(RED_SPARK_SPAWNER_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 172) spawn_sprite(WHITE_SPARK_SPAWNER_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 173) spawn_sprite(GOLD_SPARK_SPAWNER_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 174) spawn_sprite(WATER_DROP_SPAWNER_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);
	if (sprite_type == 175) spawn_sprite(NUCLEAR_BOMB_SPAWNER_SPRITE, worminator_map.starting_sprites[sprite_spawn_loop].x_position_in_pixels, worminator_map.starting_sprites[sprite_spawn_loop].y_position_in_pixels, NULL, NULL, NULL);

//////// Music ////////////////////////////////////////////////////////////////////////////////
	if (player_just_died == FALSE) {
		if (sprite_type == 176) play_midi(worminator_data_file[WORMINATOR_MUSIC_01].dat, TRUE);
		if (sprite_type == 177) play_midi(worminator_data_file[WORMINATOR_MUSIC_02].dat, TRUE);
		if (sprite_type == 178) play_midi(worminator_data_file[WORMINATOR_MUSIC_03].dat, TRUE);
		if (sprite_type == 179) play_midi(worminator_data_file[WORMINATOR_MUSIC_04].dat, TRUE);
		if (sprite_type == 180) play_midi(worminator_data_file[WORMINATOR_MUSIC_05].dat, TRUE);
		if (sprite_type == 181) play_midi(worminator_data_file[WORMINATOR_MUSIC_06].dat, TRUE);
		}
	}

screen_x_position = (player.x_position_in_pixels - 112);
screen_y_position = (player.y_position_in_pixels - 80);

if (screen_x_position > worminator_map.map_width * 16 - 256) screen_x_position = worminator_map.map_width * 16 - 256;
if (screen_y_position > worminator_map.map_height * 16 - 192) screen_y_position = worminator_map.map_height * 16 - 192;

if (screen_x_position < 0) screen_x_position = 0;
if (screen_y_position < 0) screen_y_position = 0;

// Reset the display
set_pallete(black_pallete);
blit(worminator_data_file[WORMINATOR_HUD].dat, screen, 0, 0, 0, 0, 320, 200);
update_player_stats();
hurt_player(0);
render_map();
if (worminator_map.weather != 3 && worminator_map.weather != 4) fade_in(worminator_data_file[DEFAULT_WORMINATOR_PALLETE].dat, 32);
else if (worminator_map.weather == 3) fade_in(worminator_data_file[DARK_WORMINATOR_PALLETE].dat, 32);
else if (worminator_map.weather == 4) fade_in(worminator_data_file[FOG_WORMINATOR_PALLETE].dat, 32);

// Reset the just_died variable
player_just_died = FALSE;

// This prevents choppiness from building up
speed_counter = 0;
}

/*********************\
|** LOAD SAVED GAME **|
\*********************/
void load_saved_game(char *saved_game_file_name)
{
// This is the saved game file that needs to be loaded
FILE *saved_game_file;
unsigned char sprite_spawn_loop, sprite_type;

// Disable the game speed timer and remove earth quakes
game_is_running = FALSE;
quake_timer = 0;

// Fade out the display
fade_out(32);

// Set the pallete to black so the player can't see if we draw anything
set_pallete(black_pallete);

// Open the files
saved_game_file = fopen(saved_game_file_name, "rb");

// Load the data from the saved game file
fread(&worminator_map, sizeof(worminator_map), 1, saved_game_file);
fread(&number_of_sprites, sizeof(number_of_sprites), 1, saved_game_file);
fread(&sprites, sizeof(sprites), 1, saved_game_file);
fread(&player, sizeof(player), 1, saved_game_file);
fread(&current_level, sizeof(current_level), 1, saved_game_file);

// We dont need to read anything else, so we can close the files
fclose(saved_game_file);

build_animation_table();

// Restart the music and select the correct parallaxing background
for (sprite_spawn_loop = 1; sprite_spawn_loop < 255; sprite_spawn_loop++) {
	sprite_type = worminator_map.starting_sprites[sprite_spawn_loop].sprite_type;

//////// Parallaxing Backgrounds //////////////////////////////////////////////////////////////
	if (sprite_type >= 160 && sprite_type < 170) {
		parallax_background = worminator_data_file[PARALLAXING_BACKGROUND_01 + sprite_type - 160].dat;
		}

//////// Music ////////////////////////////////////////////////////////////////////////////////
	if (sprite_type == 176) play_midi(worminator_data_file[WORMINATOR_MUSIC_01].dat, TRUE);
	if (sprite_type == 177) play_midi(worminator_data_file[WORMINATOR_MUSIC_02].dat, TRUE);
	if (sprite_type == 178) play_midi(worminator_data_file[WORMINATOR_MUSIC_03].dat, TRUE);
	if (sprite_type == 179) play_midi(worminator_data_file[WORMINATOR_MUSIC_04].dat, TRUE);
	if (sprite_type == 180) play_midi(worminator_data_file[WORMINATOR_MUSIC_05].dat, TRUE);
	if (sprite_type == 181) play_midi(worminator_data_file[WORMINATOR_MUSIC_06].dat, TRUE);
	}

screen_x_position = (player.x_position_in_pixels - 112);
screen_y_position = (player.y_position_in_pixels - 80);

if (screen_x_position > worminator_map.map_width * 16 - 256) screen_x_position = worminator_map.map_width * 16 - 256;
if (screen_y_position > worminator_map.map_height * 16 - 192) screen_y_position = worminator_map.map_height * 16 - 192;

if (screen_x_position < 0) screen_x_position = 0;
if (screen_y_position < 0) screen_y_position = 0;

// Leave whatever weapon the player had when saved alone
desired_weapon = player.current_weapon;

// Reset the display
blit(worminator_data_file[WORMINATOR_HUD].dat, screen, 0, 0, 0, 0, 320, 200);
update_player_stats();
hurt_player(0);
render_map();
if (worminator_map.weather != 3 && worminator_map.weather != 4) fade_in(worminator_data_file[DEFAULT_WORMINATOR_PALLETE].dat, 32);
else if (worminator_map.weather == 3) fade_in(worminator_data_file[DARK_WORMINATOR_PALLETE].dat, 32);
else if (worminator_map.weather == 4) fade_in(worminator_data_file[FOG_WORMINATOR_PALLETE].dat, 32);
}

/***************\
|** SAVE GAME **|
\***************/
void save_game(char* saved_game_file_name)
{
FILE *saved_game_file;

// Disable the game speed timer
game_is_running = FALSE;

saved_game_file = fopen(saved_game_file_name, "wb");

fwrite(&worminator_map, sizeof(worminator_map), 1, saved_game_file);
fwrite(&number_of_sprites, sizeof(number_of_sprites), 1, saved_game_file);
fwrite(&sprites, sizeof(sprites), 1, saved_game_file);
fwrite(&player, sizeof(player), 1, saved_game_file);
fwrite(&current_level, sizeof(current_level), 1, saved_game_file);

fclose(saved_game_file);

speed_counter = 0;
}

/****************\
|** LOAD TILES **|
\****************/
void load_tiles()
{
BITMAP *tiles;
int build_tileset_loop;

// Disable the game speed timer
game_is_running = FALSE;

tiles = worminator_data_file[TILES].dat;

worminator_tiles = create_bitmap((tiles->h / 16) * 256, 16);

// Copy the background and foreground tiles row by row
for (build_tileset_loop = 0; build_tileset_loop < (tiles->h / 16); build_tileset_loop++) {
        blit(tiles, worminator_tiles, 0, build_tileset_loop * 16, build_tileset_loop * 256, 0, 256, 16);
        }

}

/*************************\
|** INITIALIZE FUNCTION **|
\*************************/
void initialize()
{
unsigned char weather_loop;
int key_pressed;

// Disable the game speed timer
game_is_running = FALSE;

clrscr();

printf("Worminator Version: ");
printf(WORMINATOR_VERSION);
printf("\nThis program is Copyright 2000 by David Layne");
printf("\n\nBy using this program, you signify that you have read, understand, and agree to\nall terms of use stated in the included ReadMe.txt file.");

printf("\n\nInitilizing Allegro v3.9.31 WIP and Loading Configuration Data . . .");
printf("done");
allegro_init();
set_config_file("worm.cfg");
install_keyboard();
install_mouse();
install_timer();

printf("\n\nInstalling Sound . . .");
printf("done");
install_sound(DIGI_AUTODETECT, MIDI_AUTODETECT, NULL);

printf("\n\nLoading Worminator DataFile . . .");
packfile_password("security lockout");
if (exists("full.dat")) {
	version_size = FULL_VERSION;
	worminator_data_file = load_datafile("full.dat");
	printf("done\nFull Version DataFile Loaded . . .");
	}
else if (exists("stripped.dat")) {
	version_size = STRIPPED_VERSION;
	worminator_data_file = load_datafile("stripped.dat");
	printf("done\nStripped Version DataFile Loaded . . .");
	}
else {
	printf("\n\nError . . \nUnable to locate an appropriate Data File.\nPlease re-install Worminator.\n");
	exit(NULL);
	}

printf("\n\nCreating cache bitmaps and translucency tables . . .");
printf("done");
double_buffer = create_bitmap(256, 192);
spawned_sprite = create_bitmap(16, 16);
create_trans_table(&trans_table, worminator_data_file[DEFAULT_WORMINATOR_PALLETE].dat, 128, 128, 128, NULL);
color_map = &trans_table;

// Load the tiles into a cache bitmap
load_tiles();

// Cache the parallaxing background for this map
parallax_background = worminator_data_file[PARALLAXING_BACKGROUND_01].dat;

// Set up the animation timer to update 100 times a second
LOCK_FUNCTION(update_animations);
install_int(update_animations, 10);

// Set up the game speed timer
LOCK_VARIABLE(speed_counter);
LOCK_FUNCTION(increment_speed_counter);
install_int_ex(increment_speed_counter, BPS_TO_TIMER(160));

// Set up the enemy range checker to update once a second
LOCK_FUNCTION(check_enemy_ranges);
install_int(check_enemy_ranges, 1000);

printf("\n\n\n\nDone initializing.\n\n");
printf("Press:\n[1]  To run the game in standard VGA mode\n[2]  To run the game in VESA 2.0\n\nOr any other key to autodetect\n");

key_pressed = readkey();
if (key_pressed >> 8 == KEY_1) {
	clrscr();
	printf("Running in standard VGA 320 x 200 x 256\n\nPress any key to begin . . .\n");
	readkey();
	set_gfx_mode(GFX_VGA, 320, 200, 0, 0);
	}
else if (key_pressed >> 8 == KEY_2) {
	clrscr();
	printf("Running in VESA 2.0 Linear 320 x 200 x 256\n\nPress any key to begin . . .\n");
	readkey();
	set_gfx_mode(GFX_VESA2L, 320, 200, 0, 0);
	}
else set_gfx_mode(GFX_AUTODETECT, 320, 200, 0, 0);

// Make text drawn in transparent mode
text_mode(-1);

// Set up the mouse cursor
set_mouse_sprite(worminator_data_file[MOUSE_CURSOR].dat);

// Start the weather system with some random values
for (weather_loop = 0; weather_loop < 64; weather_loop++) {
	weather[weather_loop][0] = rand() % 320;
	weather[weather_loop][1] = rand() % 200;
	}

// We use allegro's dialog functions, so this is required because the default colors suck with our pallete.
gui_fg_color = 122;
gui_bg_color = 133;

// Clear out the message attributes
text_messages[0].timer = -1;
text_messages[1].timer = -1;
text_messages[2].timer = -1;
text_messages[3].timer = -1;
}

