/***************************************************************************

    M.A.M.E.32  -  Multiple Arcade Machine Emulator for Win32
    Win32 Portions Copyright (C) 1997-98 Michael Soderstrom and Chris Kirmse
    
    This file is part of MAME32, and may only be used, modified and
    distributed under the terms of the MAME license, in "readme.txt".
    By continuing to use, modify or distribute this file you indicate
    that you have read the license and understand and accept it fully.

 ***************************************************************************/

/***************************************************************************

  Keyboard.c

 ***************************************************************************/

#include "driver.h"

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <windowsx.h>
#include <assert.h>
#include "MAME32.h"
#include "M32Util.h"
#include "Keyboard.h"
#include "status.h"

#define NUMKEYSTATES    256
#define NUMVIRTUALKEYS  256

/***************************************************************************
    function prototypes
 ***************************************************************************/

static int              Keyboard_init(options_type *options);
static void             Keyboard_exit(void);
static int              Keyboard_key_pressed(int keycode);
static int              Keyboard_key_pressed_memory(int keycode);
static int              Keyboard_key_pressed_memory_repeat(int keycode, int speed);
static int              Keyboard_read_key_immediate(void);
static int              Keyboard_read_keyrepeat(void);
static const char*      Keyboard_key_name(int keycode);

static BOOL             Keyboard_OnMessage(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam, LRESULT* pResult);
static void             Keyboard_PressKey(int keycode);
static void             OnKey(HWND hWnd, UINT vk, BOOL fDown, int cRepeat, UINT flags);

/***************************************************************************
    External variables
 ***************************************************************************/

struct OSDKeyboard  Keyboard = 
{
    { Keyboard_init },                      /* init                      */
    { Keyboard_exit },                      /* exit                      */
    { Keyboard_key_pressed },               /* key_pressed               */
    { Keyboard_key_pressed_memory },        /* key_pressed_memory        */
    { Keyboard_key_pressed_memory_repeat }, /* key_pressed_memory_repeat */
    { Keyboard_read_key_immediate },        /* read_key_immediate        */   
    { Keyboard_read_keyrepeat },            /* read_keyrepeat            */
    { Keyboard_key_name },                  /* key_name                  */
    { Keyboard_pause },                     /* Pause key hit             */

    { Keyboard_PressKey },                  /* PressKey                  */
    { Keyboard_OnMessage }                  /* OnMessage                 */
};

/***************************************************************************
    Internal structures
 ***************************************************************************/

struct tKeyboard_private
{
    int     m_nInternalKeyPress;
    BYTE    m_Memory[NUMKEYSTATES];
    UINT    m_nVirtualKeyMap[NUMVIRTUALKEYS];
    int     m_nCounter;
    int     m_nKeyDelay;
    BOOL    m_bPauseKeyPressed;
};

/***************************************************************************
    Internal variables
 ***************************************************************************/

static struct tKeyboard_private This;

/***************************************************************************
    External OSD functions  
 ***************************************************************************/

/*
    put here anything you need to do when the program is started. Return 0 if 
    initialization was successful, nonzero otherwise.
*/
static int Keyboard_init(options_type *options)
{
    int     i;
    BYTE    KeyState[NUMKEYSTATES];

    memset(&This, 0, sizeof(struct tKeyboard_private));
    This.m_nInternalKeyPress = -1;

    /* Clear keyboard state. */
    GetKeyboardState(KeyState);

    for (i = 0; i < NUMKEYSTATES; i++)
        KeyState[i] &= 0x01;

    SetKeyboardState(KeyState);

    /* Pre-map OSD_KEYs to virtual keys. */
    for (i = 0; i < NUMVIRTUALKEYS; i++)
    {
        UINT nVirtualKey = MapVirtualKey(i, 1);
        if (nVirtualKey == 0)
            This.m_nVirtualKeyMap[i] = i; /* If there is no translation. */
        else
            This.m_nVirtualKeyMap[i] = nVirtualKey;
    }

    /*
       Pre-map Left/Right OSD_KEYs to virtual keys if possible. 
       Also take care of some OSD keys that are not scan codes. 
    */
    if (OnNT() == TRUE)
    {
        This.m_nVirtualKeyMap[OSD_KEY_RSHIFT]   = VK_RSHIFT;
        This.m_nVirtualKeyMap[OSD_KEY_LSHIFT]   = VK_LSHIFT;

        This.m_nVirtualKeyMap[OSD_KEY_LCONTROL] = VK_LCONTROL;
        This.m_nVirtualKeyMap[OSD_KEY_RCONTROL] = VK_RCONTROL;

        This.m_nVirtualKeyMap[OSD_KEY_ALTGR]    = VK_RMENU;
        This.m_nVirtualKeyMap[OSD_KEY_ALT]      = VK_LMENU;
    }
    else
    {
        This.m_nVirtualKeyMap[OSD_KEY_RSHIFT]   = VK_SHIFT;
        This.m_nVirtualKeyMap[OSD_KEY_LSHIFT]   = VK_SHIFT;

        This.m_nVirtualKeyMap[OSD_KEY_RCONTROL] = VK_CONTROL;
        This.m_nVirtualKeyMap[OSD_KEY_LCONTROL] = VK_CONTROL;

        This.m_nVirtualKeyMap[OSD_KEY_ALTGR]    = VK_MENU;
        This.m_nVirtualKeyMap[OSD_KEY_ALT]      = VK_MENU;
    }

    /* OSD_KEY_PAUSE is not a scan code. */
    This.m_nVirtualKeyMap[OSD_KEY_PAUSE] = VK_PAUSE;

    return 0;
}

/*
    put here cleanup routines to be executed when the program is terminated.
*/
static void Keyboard_exit(void)
{
}

/*
    Check if a key is pressed. The keycode is the standard PC keyboard code, as
    defined in osdepend.h. Return 0 if the key is not pressed, nonzero otherwise.
*/
static int Keyboard_key_pressed(int keycode)
{
    SHORT   state;

    if (MAME32App.Done() && (keycode == OSD_KEY_ANY || keycode == OSD_KEY_FAST_EXIT))
        return 1;

    if (keycode == OSD_KEY_ANY)
        return Keyboard_read_key_immediate();

    keycode = Keyboard_PseudoToKeyCode(keycode);

    if (This.m_nInternalKeyPress != -1)
    {
        if (This.m_nInternalKeyPress == keycode)
        {
            This.m_nInternalKeyPress = -1;
            return 1;
        }
    }

    if (keycode > OSD_MAX_KEY)
        return 0;

	if (keycode == OSD_KEY_PAUSE)
	{
        if (This.m_bPauseKeyPressed == TRUE)
        {
            This.m_bPauseKeyPressed = FALSE;
            return 1;
        }
    }

    MAME32App.ProcessMessages();
    state = GetAsyncKeyState(This.m_nVirtualKeyMap[keycode]);

    /* If the high-order bit is 1, the key is down; otherwise, it is up */
    if (state & 0x8000)
        return 1;

    return 0;
}

/* Report a key as pressed only when the user hits it, not while it is */
/* being kept pressed. */
static int Keyboard_key_pressed_memory(int keycode)
{
    int key = keycode;
    int res = 0;

	keycode = Keyboard_PseudoToKeyCode(keycode);

    if (Keyboard_key_pressed(keycode))
    {
        if (keycode == OSD_KEY_ANY)
            return 1;

        if (This.m_Memory[keycode] == 0)
            res = 1;
        This.m_Memory[keycode] = 1;
    }
    else
        This.m_Memory[keycode] = 0;

    return res;
}

/* report key as pulsing while it is pressed */
static int Keyboard_key_pressed_memory_repeat(int keycode, int speed)
{
    int res = 0;

    keycode = Keyboard_PseudoToKeyCode(keycode);

    if (Keyboard_key_pressed(keycode))
    {
        if (This.m_Memory[keycode] == 0)
        {
            This.m_nKeyDelay = 3;
            This.m_nCounter  = 0;
            res = 1;
        }
        else
        if (++This.m_nCounter > This.m_nKeyDelay * speed * Machine->drv->frames_per_second / 60)
        {
            This.m_nKeyDelay = 1;
            This.m_nCounter  = 0;
            res = 1;
        }
        This.m_Memory[keycode] = 1;
    }
    else
        This.m_Memory[keycode] = 0;

    return res;
}

/* If the user presses a key return it, otherwise return OSD_KEY_NONE. */
/* DO NOT wait for the user to press a key */
static int Keyboard_read_key_immediate(void)
{
    int res = 0;

    /* first of all, record keys which are NOT pressed */
    for (res = OSD_MAX_KEY; OSD_KEY_NONE < res; res--)
    {
        if (!Keyboard_key_pressed(res))
        {
            This.m_Memory[res] = 0;
        }
    }

    for (res = OSD_MAX_KEY; OSD_KEY_NONE < res; res--)
    {
        if (Keyboard_key_pressed(res))
        {
            if (This.m_Memory[res] == 0)
            {
                This.m_Memory[res] = 1;
            }
            else
                res = OSD_KEY_NONE;
            break;
        }
    }

    return res;
}

/*
    Wait for a key press and return keycode.  Support repeat
*/
static int Keyboard_read_keyrepeat()
{
    MSG Msg;
    int nKeyCode = OSD_KEY_ESC;
    
    StatusSetString("Waiting for keypress");

    while (GetMessage(&Msg, NULL, 0, 0))
    {
        TranslateMessage(&Msg);
        DispatchMessage(&Msg);

        if (Msg.message == WM_KEYDOWN
        ||  Msg.message == WM_SYSKEYDOWN)
        {
            int     nVirtualKey = Msg.wParam;   /* virtual-key code */

            nKeyCode = MapVirtualKey(nVirtualKey, 0);

            StatusSetString("");
            return nKeyCode;
        }
    }

    if (Msg.message == WM_QUIT)
    {
        MAME32App.Quit();
        nKeyCode = OSD_KEY_SPACE;
    }

    StatusSetString("");
    return nKeyCode;
}

/*
    return the name of a key
*/
static const char* Keyboard_key_name(int keycode)
{
    static char *nonedefined = "None";
    static char *keynames[] =
    {
        "ESC", "1", "2", "3", "4", "5", "6", "7", "8", "9", "0", "MINUS", "EQUAL", "BKSPACE",
        "TAB", "Q", "W", "E", "R", "T", "Y", "U", "I", "O", "P", "OPBRACE", "CLBRACE", "ENTER",
        "LCTRL", "A", "S", "D", "F", "G", "H", "J", "K", "L", "COLON", "QUOTE", "TILDE",
        "LSHIFT", "Error", "Z", "X", "C", "V", "B", "N", "M", "COMMA", ".", "SLASH", "RSHIFT",
        "*", "ALT", "SPACE", "CAPSLOCK", "F1", "F2", "F3", "F4", "F5", "F6", "F7", "F8", "F9", "F10",
        "NUMLOCK", "SCRLOCK", "HOME", "UP", "PGUP", "MINUS PAD",
        "LEFT", "5 PAD", "RIGHT", "PLUS PAD", "END", "DOWN",
        "PGDN", "INS", "DEL", "PRTSCR", "Error", "Error",
        "F11", "F12", "Error", "Error",
        "LWIN", "RWIN", "MENU", "RCTRL", "ALTGR", "PAUSE",
        "Error", "Error", "Error", "Error",
        "1 PAD", "2 PAD", "3 PAD", "4 PAD", "Error",
        "6 PAD", "7 PAD", "8 PAD", "9 PAD", "0 PAD",
        ". PAD", "= PAD", "/ PAD", "* PAD", "ENTER PAD",
    };

    if (1 <= keycode && keycode <= OSD_MAX_KEY)
        return (char*)keynames[keycode - 1];
    else
        return (char*)nonedefined;
}


/***************************************************************************
    Message handlers
 ***************************************************************************/

static BOOL Keyboard_OnMessage(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam, LRESULT* pResult)
{
    switch (Msg)
    {
        HANDLE_MESSAGE(hWnd, WM_KEYDOWN, OnKey);
    }
    return FALSE;
}

static void OnKey(HWND hWnd, UINT vk, BOOL fDown, int cRepeat, UINT flags)
{
    /*
        Detect the 'Pause' key.
        This is necessary since the Pause key is not
        easily detected when polling with GetAsyncKeyState().
        The WM_KEYDOWN and WM_KEYUP messages are to close together.
    */
    if (vk == VK_PAUSE)
        This.m_bPauseKeyPressed = TRUE;
}

/***************************************************************************
    Internal functions
 ***************************************************************************/

/* We've been paused */
void Keyboard_pause(int paused)
{
    if (paused)
    {
        StatusSetString("Paused");
        MAME32App.m_bPaused = TRUE;
    }
    else
    {
        StatusSetString("");
        MAME32App.m_bPaused = FALSE;
    }
}

/* Simulate a keypress. */
static void Keyboard_PressKey(int keycode)
{
    This.m_nInternalKeyPress = Keyboard_PseudoToKeyCode(keycode);
    PostMessage(MAME32App.m_hWnd, WM_KEYDOWN, MapVirtualKey(keycode, 1), 0);
}

/* translate a pseudo key code to to a key code */
int Keyboard_PseudoToKeyCode(int keycode)
{
    if (keycode < OSD_MAX_KEY)
        return keycode;

    switch (keycode)
    {
        case OSD_KEY_CANCEL:
            return OSD_KEY_ESC;

        case OSD_KEY_RESET_MACHINE:
            return OSD_KEY_F3;

        case OSD_KEY_SHOW_GFX:
            return OSD_KEY_F4;

        case OSD_KEY_CHEAT_TOGGLE:
            return OSD_KEY_F5;

        case OSD_KEY_FRAMESKIP_INC:
            return OSD_KEY_F9;

        case OSD_KEY_FRAMESKIP_DEC:
            return OSD_KEY_F8;

        case OSD_KEY_THROTTLE:
            return OSD_KEY_F10;

        case OSD_KEY_SHOW_FPS:
            if (Keyboard_key_pressed(OSD_KEY_LSHIFT)
          /*||  Keyboard_key_pressed(OSD_KEY_RSHIFT)*/)
                return 0;
            keycode = OSD_KEY_F11;
            break;

        case OSD_KEY_SHOW_PROFILE:
            if (!(Keyboard_key_pressed(OSD_KEY_LSHIFT)
          /*||    Keyboard_key_pressed(OSD_KEY_RSHIFT)*/))
                return 0;
            keycode = OSD_KEY_F11;
            break;

        case OSD_KEY_CONFIGURE:
            return OSD_KEY_TAB;

        case OSD_KEY_ON_SCREEN_DISPLAY:
        {
            extern int mame_debug;
            if (mame_debug)
                return 0;

            keycode = OSD_KEY_TILDE;
            break;
        }

        case OSD_KEY_DEBUGGER:
        {
            extern int mame_debug;
            if (!mame_debug)
                return 0;

            keycode = OSD_KEY_TILDE;
            break;
        }

        case OSD_KEY_SNAPSHOT:
            return OSD_KEY_F12;

        case OSD_KEY_UI_SELECT:
            return OSD_KEY_ENTER;
            break;

        case OSD_KEY_UI_LEFT:
            return OSD_KEY_LEFT;
            break;

        case OSD_KEY_UI_RIGHT:
            return OSD_KEY_RIGHT;
            break;

        case OSD_KEY_UI_UP:
            return OSD_KEY_UP;
            break;

        case OSD_KEY_UI_DOWN:
            return OSD_KEY_DOWN;
            break;
    }

    return keycode;
}

