/***************************************************************************

    M.A.M.E.32  -  Multiple Arcade Machine Emulator for Win32
    Win32 Portions Copyright (C) 1997-98 Michael Soderstrom and Chris Kirmse
    
    This file is part of MAME32, and may only be used, modified and
    distributed under the terms of the MAME license, in "readme.txt".
    By continuing to use, modify or distribute this file you indicate
    that you have read the license and understand and accept it fully.

 ***************************************************************************/

/***************************************************************************

  Joystick.c

 ***************************************************************************/

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <stdlib.h>
#include <mmsystem.h>
#include <mmreg.h>
#include <assert.h>
#include "mame32.h"
#include "Joystick.h"
#include "M32Util.h"
#include "trak.h"

#define OSD_ANALOGMAX   (127.0)
#define OSD_ANALOGMIN   (-128.0)

#ifndef JOYSTICKID3
#define JOYSTICKID3 3
#endif
#ifndef JOYSTICKID4
#define JOYSTICKID4 4
#endif

#define NUM_JOYSTICKS 4

/***************************************************************************
    function prototypes
 ***************************************************************************/

static int              Joystick_init(options_type *options);
static void             Joystick_exit(void);
static void             Joystick_poll_joysticks(void);
static int              Joystick_joy_pressed(int joycode);
static const char*      Joystick_joy_name(int joycode);
static void             Joystick_analogjoy_read(int player, int *analog_x, int *analog_y);
static int              Joystick_standard_analog_read(int player, int axis);
static BOOL             Joystick_Available(int nJoyStick);
static BOOL             Joystick_OnMessage(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam, LRESULT* pResult);

/***************************************************************************
    External variables
 ***************************************************************************/

struct OSDJoystick  Joystick = 
{
    { Joystick_init },              /* init              */
    { Joystick_exit },              /* exit              */
    { Joystick_poll_joysticks },    /* poll_joysticks     */
    { Joystick_joy_pressed },       /* joy_pressed       */
    { Joystick_joy_name },          /* joy_name          */
    { Joystick_analogjoy_read },    /* analogjoy_read    */
    { Joystick_standard_analog_read },/* standard_analog_read    */
    { Joystick_Available },         /* Available         */
    { Joystick_OnMessage },         /* OnMessage         */
};

/***************************************************************************
    Internal structures
 ***************************************************************************/

struct tJoystickInfo
{
    BOOL        m_bUseJoystick;
    UINT        m_uJoyID;

    JOYINFOEX   m_JoyInfo;
    JOYCAPS     m_JoyCaps;

    UINT        m_nDeadZone;
    DWORD       m_dwXDZmin;
    DWORD       m_dwXDZmax;
    DWORD       m_dwYDZmin;
    DWORD       m_dwYDZmax;

    double      m_dXScale;
    double      m_dYScale;
};

struct tJoystick_private
{
    struct tJoystickInfo   m_Joy[NUM_JOYSTICKS];
};

/***************************************************************************
    Internal variables
 ***************************************************************************/

static struct tJoystick_private This;

/***************************************************************************
    External OSD functions  
 ***************************************************************************/

/*
    put here anything you need to do when the program is started. Return 0 if 
    initialization was successful, nonzero otherwise.
*/
static int Joystick_init(options_type *options)
{
    int         i;
    MMRESULT    mmResult;

    for (i = 0; i < NUM_JOYSTICKS; i++)
    {
        memset(&This.m_Joy[i].m_JoyInfo, 0, sizeof(JOYINFOEX));
        memset(&This.m_Joy[i].m_JoyCaps, 0, sizeof(JOYCAPS));
        This.m_Joy[i].m_bUseJoystick = TRUE;
        This.m_Joy[i].m_nDeadZone    = 75;
        This.m_Joy[i].m_dwXDZmin     = 0;
        This.m_Joy[i].m_dwXDZmax     = 0;
        This.m_Joy[i].m_dwYDZmin     = 0;
        This.m_Joy[i].m_dwYDZmax     = 0;
        This.m_Joy[1].m_dXScale      = 0.0;
        This.m_Joy[1].m_dYScale      = 0.0;
    }
    This.m_Joy[0].m_uJoyID = JOYSTICKID1;
    This.m_Joy[1].m_uJoyID = JOYSTICKID2;
    This.m_Joy[2].m_uJoyID = JOYSTICKID3;
    This.m_Joy[3].m_uJoyID = JOYSTICKID4;

    if (!options->use_joystick)
    {
       This.m_Joy[0].m_bUseJoystick = FALSE;
       This.m_Joy[1].m_bUseJoystick = FALSE;
       This.m_Joy[2].m_bUseJoystick = FALSE;
       This.m_Joy[3].m_bUseJoystick = FALSE;
    }

    /* CMK no way to set dead zone right now */
    /*
      This.m_Joy[0].m_nDeadZone = min(max(arg, 5), 95);
      This.m_Joy[1].m_nDeadZone = min(max(arg, 5), 95);
      This.m_Joy[2].m_nDeadZone = min(max(arg, 5), 95);
      This.m_Joy[3].m_nDeadZone = min(max(arg, 5), 95);
    */

    for (i = 0; i < NUM_JOYSTICKS; i++)
    {
        if (This.m_Joy[i].m_bUseJoystick == FALSE)
            continue;
        
        if (joyGetNumDevs() == 0)
        {
            /* No driver is present. */
            This.m_Joy[i].m_bUseJoystick = FALSE;
        }
        else
        {
            /* Determine if JOYSTICKID[1-4] is plugged in. */
            This.m_Joy[i].m_JoyInfo.dwSize  = sizeof(JOYINFOEX);
            This.m_Joy[i].m_JoyInfo.dwFlags = JOY_RETURNBUTTONS | JOY_RETURNX | JOY_RETURNY;
            
            mmResult = joyGetPosEx(This.m_Joy[i].m_uJoyID, &This.m_Joy[i].m_JoyInfo);
            
            if (mmResult == JOYERR_NOERROR)
            {
                joyGetDevCaps(This.m_Joy[i].m_uJoyID, &This.m_Joy[i].m_JoyCaps, sizeof(JOYCAPS));
                
                This.m_Joy[i].m_dwXDZmin = (DWORD)((This.m_Joy[i].m_JoyCaps.wXmax - This.m_Joy[i].m_JoyCaps.wXmin) *
                                                  ((100.0 - This.m_Joy[i].m_nDeadZone) / 200.0));
                This.m_Joy[i].m_dwXDZmax = (DWORD)((This.m_Joy[i].m_JoyCaps.wXmax - This.m_Joy[i].m_JoyCaps.wXmin) *
                                                  ((100.0 + This.m_Joy[i].m_nDeadZone) / 200.0));
                This.m_Joy[i].m_dwYDZmin = (DWORD)((This.m_Joy[i].m_JoyCaps.wYmax - This.m_Joy[i].m_JoyCaps.wYmin) *
                                                  ((100.0 - This.m_Joy[i].m_nDeadZone) / 200.0));
                This.m_Joy[i].m_dwYDZmax = (DWORD)((This.m_Joy[i].m_JoyCaps.wYmax - This.m_Joy[i].m_JoyCaps.wYmin) *
                                                  ((100.0 + This.m_Joy[i].m_nDeadZone) / 200.0));

                This.m_Joy[i].m_dXScale = ((OSD_ANALOGMAX - OSD_ANALOGMIN) / (This.m_Joy[i].m_JoyCaps.wXmax - This.m_Joy[i].m_JoyCaps.wXmin));
                This.m_Joy[i].m_dYScale = ((OSD_ANALOGMAX - OSD_ANALOGMIN) / (This.m_Joy[i].m_JoyCaps.wYmax - This.m_Joy[i].m_JoyCaps.wYmin));

                if (This.m_Joy[i].m_JoyCaps.wCaps & JOYCAPS_POV4DIR)
                {
                    This.m_Joy[i].m_JoyInfo.dwFlags |= JOY_RETURNPOV;
                }
#if 0
                /* Check for U and V axis for joystick 1 */
                if (This.m_Joy[0].m_JoyCaps.wCaps & JOYCAPS_HASU)
                {
                    This.m_Joy[0].m_JoyInfo.dwFlags |= JOY_RETURNU;
                }
                if (This.m_Joy[0].m_JoyCaps.wCaps & JOYCAPS_HASV)
                {
                    This.m_Joy[0].m_JoyInfo.dwFlags |= JOY_RETURNV;
                }
#endif
            }
            else
            if (mmResult == JOYERR_UNPLUGGED)
            {
                This.m_Joy[i].m_bUseJoystick = FALSE;
            }
            else
            {
                /* Some other error with the joystick. Don't use it. */
                This.m_Joy[i].m_bUseJoystick = FALSE;
            }
        }
    }

    return 0;
}

/*
    put here cleanup routines to be executed when the program is terminated.
*/
static void Joystick_exit(void)
{
}

static void Joystick_poll_joysticks(void)
{
    MMRESULT    mmResult;

    if (This.m_Joy[0].m_bUseJoystick == TRUE)
    {        
        mmResult = joyGetPosEx(This.m_Joy[0].m_uJoyID, &This.m_Joy[0].m_JoyInfo);
        if (mmResult != JOYERR_NOERROR)
            ErrorMsg("joyGetPosEx() failed error=%x", mmResult);
    }

    if (This.m_Joy[1].m_bUseJoystick == TRUE)
    {        
        mmResult = joyGetPosEx(This.m_Joy[1].m_uJoyID, &This.m_Joy[1].m_JoyInfo);
        if (mmResult != JOYERR_NOERROR)
            ErrorMsg("joyGetPosEx() failed error=%x", mmResult);
    }

    if (This.m_Joy[2].m_bUseJoystick == TRUE)
    {        
        mmResult = joyGetPosEx(This.m_Joy[2].m_uJoyID, &This.m_Joy[2].m_JoyInfo);
        if (mmResult != JOYERR_NOERROR)
            ErrorMsg("joyGetPosEx() failed error=%x", mmResult);
    }

    if (This.m_Joy[3].m_bUseJoystick == TRUE)
    {        
        mmResult = joyGetPosEx(This.m_Joy[3].m_uJoyID, &This.m_Joy[3].m_JoyInfo);
        if (mmResult != JOYERR_NOERROR)
            ErrorMsg("joyGetPosEx() failed error=%x", mmResult);
    }

}

/*
    check if the joystick is moved in the specified direction, defined in
    osdepend.h. Return 0 if it is not pressed, nonzero otherwise.
*/
static int Joystick_joy_pressed(int joycode)
{
    if (This.m_Joy[0].m_bUseJoystick == TRUE)
    {
        switch (joycode)
        {
            case OSD_JOY_LEFT:   return This.m_Joy[0].m_JoyInfo.dwXpos <= This.m_Joy[0].m_dwXDZmin;  break;
            case OSD_JOY_RIGHT:  return This.m_Joy[0].m_dwXDZmax <= This.m_Joy[0].m_JoyInfo.dwXpos;  break;
            case OSD_JOY_UP:     return This.m_Joy[0].m_JoyInfo.dwYpos <= This.m_Joy[0].m_dwYDZmin;  break;
            case OSD_JOY_DOWN:   return This.m_Joy[0].m_dwYDZmax <= This.m_Joy[0].m_JoyInfo.dwYpos;  break;

            case OSD_JOY_FIRE1:  return This.m_Joy[0].m_JoyInfo.dwButtons & JOY_BUTTON1;    break;
            case OSD_JOY_FIRE2:  return This.m_Joy[0].m_JoyInfo.dwButtons & JOY_BUTTON2;    break;
            case OSD_JOY_FIRE3:  return This.m_Joy[0].m_JoyInfo.dwButtons & JOY_BUTTON3;    break;
            case OSD_JOY_FIRE4:  return This.m_Joy[0].m_JoyInfo.dwButtons & JOY_BUTTON4;    break;
            case OSD_JOY_FIRE5:  return This.m_Joy[0].m_JoyInfo.dwButtons & JOY_BUTTON5;    break;
            case OSD_JOY_FIRE6:  return This.m_Joy[0].m_JoyInfo.dwButtons & JOY_BUTTON6;    break;
            case OSD_JOY_FIRE7:  return This.m_Joy[0].m_JoyInfo.dwButtons & JOY_BUTTON7;    break;
            case OSD_JOY_FIRE8:  return This.m_Joy[0].m_JoyInfo.dwButtons & JOY_BUTTON8;    break;
            case OSD_JOY_FIRE9:  return This.m_Joy[0].m_JoyInfo.dwButtons & JOY_BUTTON9;    break;
            case OSD_JOY_FIRE10: return This.m_Joy[0].m_JoyInfo.dwButtons & JOY_BUTTON10;   break;
            case OSD_JOY_FIRE:   return This.m_Joy[0].m_JoyInfo.dwButtons & (JOY_BUTTON1 | JOY_BUTTON2 |
                                                                             JOY_BUTTON3 | JOY_BUTTON4 |
                                                                             JOY_BUTTON5 | JOY_BUTTON6 |
                                                                             JOY_BUTTON7 | JOY_BUTTON8 |
                                                                             JOY_BUTTON9 | JOY_BUTTON10);
                                                                             break;
        }

    }

    if (This.m_Joy[1].m_bUseJoystick == TRUE)
    {
        switch (joycode)
        {
            case OSD_JOY2_LEFT:   return This.m_Joy[1].m_JoyInfo.dwXpos <= This.m_Joy[1].m_dwXDZmin;  break;
            case OSD_JOY2_RIGHT:  return This.m_Joy[1].m_dwXDZmax <= This.m_Joy[1].m_JoyInfo.dwXpos;  break;
            case OSD_JOY2_UP:     return This.m_Joy[1].m_JoyInfo.dwYpos <= This.m_Joy[1].m_dwYDZmin;  break;
            case OSD_JOY2_DOWN:   return This.m_Joy[1].m_dwYDZmax <= This.m_Joy[1].m_JoyInfo.dwYpos;  break;

            case OSD_JOY2_FIRE1:  return This.m_Joy[1].m_JoyInfo.dwButtons & JOY_BUTTON1;    break;
            case OSD_JOY2_FIRE2:  return This.m_Joy[1].m_JoyInfo.dwButtons & JOY_BUTTON2;    break;
            case OSD_JOY2_FIRE3:  return This.m_Joy[1].m_JoyInfo.dwButtons & JOY_BUTTON3;    break;
            case OSD_JOY2_FIRE4:  return This.m_Joy[1].m_JoyInfo.dwButtons & JOY_BUTTON4;    break;
            case OSD_JOY2_FIRE5:  return This.m_Joy[1].m_JoyInfo.dwButtons & JOY_BUTTON5;    break;
            case OSD_JOY2_FIRE6:  return This.m_Joy[1].m_JoyInfo.dwButtons & JOY_BUTTON6;    break;
            case OSD_JOY2_FIRE7:  return This.m_Joy[1].m_JoyInfo.dwButtons & JOY_BUTTON7;    break;
            case OSD_JOY2_FIRE8:  return This.m_Joy[1].m_JoyInfo.dwButtons & JOY_BUTTON8;    break;
            case OSD_JOY2_FIRE9:  return This.m_Joy[1].m_JoyInfo.dwButtons & JOY_BUTTON9;    break;
            case OSD_JOY2_FIRE10: return This.m_Joy[1].m_JoyInfo.dwButtons & JOY_BUTTON10;   break;
            case OSD_JOY2_FIRE:   return This.m_Joy[1].m_JoyInfo.dwButtons & (JOY_BUTTON1 | JOY_BUTTON2 |
                                                                              JOY_BUTTON3 | JOY_BUTTON4 |
                                                                              JOY_BUTTON5 | JOY_BUTTON6 |
                                                                              JOY_BUTTON7 | JOY_BUTTON8 |
                                                                              JOY_BUTTON9 | JOY_BUTTON10);
                                                                             break;
        
        }
    }

    if (This.m_Joy[2].m_bUseJoystick == TRUE)
    {
        switch (joycode)
        {
            case OSD_JOY3_LEFT:   return This.m_Joy[2].m_JoyInfo.dwXpos <= This.m_Joy[2].m_dwXDZmin;  break;
            case OSD_JOY3_RIGHT:  return This.m_Joy[2].m_dwXDZmax <= This.m_Joy[2].m_JoyInfo.dwXpos;  break;
            case OSD_JOY3_UP:     return This.m_Joy[2].m_JoyInfo.dwYpos <= This.m_Joy[2].m_dwYDZmin;  break;
            case OSD_JOY3_DOWN:   return This.m_Joy[2].m_dwYDZmax <= This.m_Joy[2].m_JoyInfo.dwYpos;  break;

            case OSD_JOY3_FIRE1:  return This.m_Joy[2].m_JoyInfo.dwButtons & JOY_BUTTON1;    break;
            case OSD_JOY3_FIRE2:  return This.m_Joy[2].m_JoyInfo.dwButtons & JOY_BUTTON2;    break;
            case OSD_JOY3_FIRE3:  return This.m_Joy[2].m_JoyInfo.dwButtons & JOY_BUTTON3;    break;
            case OSD_JOY3_FIRE4:  return This.m_Joy[2].m_JoyInfo.dwButtons & JOY_BUTTON4;    break;
            case OSD_JOY3_FIRE5:  return This.m_Joy[2].m_JoyInfo.dwButtons & JOY_BUTTON5;    break;
            case OSD_JOY3_FIRE6:  return This.m_Joy[2].m_JoyInfo.dwButtons & JOY_BUTTON6;    break;
            case OSD_JOY3_FIRE7:  return This.m_Joy[2].m_JoyInfo.dwButtons & JOY_BUTTON7;    break;
            case OSD_JOY3_FIRE8:  return This.m_Joy[2].m_JoyInfo.dwButtons & JOY_BUTTON8;    break;
            case OSD_JOY3_FIRE9:  return This.m_Joy[2].m_JoyInfo.dwButtons & JOY_BUTTON9;    break;
            case OSD_JOY3_FIRE10: return This.m_Joy[2].m_JoyInfo.dwButtons & JOY_BUTTON10;   break;
            case OSD_JOY3_FIRE:   return This.m_Joy[2].m_JoyInfo.dwButtons & (JOY_BUTTON1 | JOY_BUTTON2 |
                                                                              JOY_BUTTON3 | JOY_BUTTON4 |
                                                                              JOY_BUTTON5 | JOY_BUTTON6 |
                                                                              JOY_BUTTON7 | JOY_BUTTON8 |
                                                                              JOY_BUTTON9 | JOY_BUTTON10);
                                                                             break;
        
        }
    }

    if (This.m_Joy[3].m_bUseJoystick == TRUE)
    {
        switch (joycode)
        {
            case OSD_JOY4_LEFT:   return This.m_Joy[3].m_JoyInfo.dwXpos <= This.m_Joy[3].m_dwXDZmin;  break;
            case OSD_JOY4_RIGHT:  return This.m_Joy[3].m_dwXDZmax <= This.m_Joy[3].m_JoyInfo.dwXpos;  break;
            case OSD_JOY4_UP:     return This.m_Joy[3].m_JoyInfo.dwYpos <= This.m_Joy[3].m_dwYDZmin;  break;
            case OSD_JOY4_DOWN:   return This.m_Joy[3].m_dwYDZmax <= This.m_Joy[3].m_JoyInfo.dwYpos;  break;

            case OSD_JOY4_FIRE1:  return This.m_Joy[3].m_JoyInfo.dwButtons & JOY_BUTTON1;    break;
            case OSD_JOY4_FIRE2:  return This.m_Joy[3].m_JoyInfo.dwButtons & JOY_BUTTON2;    break;
            case OSD_JOY4_FIRE3:  return This.m_Joy[3].m_JoyInfo.dwButtons & JOY_BUTTON3;    break;
            case OSD_JOY4_FIRE4:  return This.m_Joy[3].m_JoyInfo.dwButtons & JOY_BUTTON4;    break;
            case OSD_JOY4_FIRE5:  return This.m_Joy[3].m_JoyInfo.dwButtons & JOY_BUTTON5;    break;
            case OSD_JOY4_FIRE6:  return This.m_Joy[3].m_JoyInfo.dwButtons & JOY_BUTTON6;    break;
            case OSD_JOY4_FIRE7:  return This.m_Joy[3].m_JoyInfo.dwButtons & JOY_BUTTON7;    break;
            case OSD_JOY4_FIRE8:  return This.m_Joy[3].m_JoyInfo.dwButtons & JOY_BUTTON8;    break;
            case OSD_JOY4_FIRE9:  return This.m_Joy[3].m_JoyInfo.dwButtons & JOY_BUTTON9;    break;
            case OSD_JOY4_FIRE10: return This.m_Joy[3].m_JoyInfo.dwButtons & JOY_BUTTON10;   break;
            case OSD_JOY4_FIRE:   return This.m_Joy[3].m_JoyInfo.dwButtons & (JOY_BUTTON1 | JOY_BUTTON2 |
                                                                              JOY_BUTTON3 | JOY_BUTTON4 |
                                                                              JOY_BUTTON5 | JOY_BUTTON6 |
                                                                              JOY_BUTTON7 | JOY_BUTTON8 |
                                                                              JOY_BUTTON9 | JOY_BUTTON10);
                                                                             break;
        
        }
    }

    return 0;
}

/*
    Return the name of a joystick button.
*/
static const char* Joystick_joy_name(int joycode)
{
    static char *nonedefined = "None";
    static char *joynames[] =
    {
        "Left", "Right", "Up", "Down", "Button A",
        "Button B", "Button C", "Button D", "Button E", "Button F",
        "Button G", "Button H", "Button I", "Button J", "Any Button",
        "J2 Left", "J2 Right", "J2 Up", "J2 Down", "J2 Button A",
        "J2 Button B", "J2 Button C", "J2 Button D", "J2 Button E", "J2 Button F",
        "J2 Button G", "J2 Button H", "J2 Button I", "J2 Button J", "J2 Any Button",
        "J3 Left", "J3 Right", "J3 Up", "J3 Down", "J3 Button A",
        "J3 Button B", "J3 Button C", "J3 Button D", "J3 Button E", "J3 Button F",
        "J3 Button G", "J3 Button H", "J3 Button I", "J3 Button J", "J3 Any Button",
        "J4 Left", "J4 Right", "J4 Up", "J4 Down", "J4 Button A",
        "J4 Button B", "J4 Button C", "J4 Button D", "J4 Button E", "J4 Button F",
        "J4 Button G", "J4 Button H", "J4 Button I", "J4 Button J", "J4 Any Button",
    };

    if (0 < joycode && joycode <= OSD_MAX_JOY)
        return (char*)joynames[joycode - 1];
    else
        return (char*)nonedefined;
}

/* osd_analog_joyread() returns values from -128 to 128 */
static void Joystick_analogjoy_read(int player, int *analog_x, int *analog_y)
{
    *analog_x = *analog_y = 0;

    if (NUM_JOYSTICKS <= player)
        return;

    if (This.m_Joy[player].m_bUseJoystick == FALSE)
        return;

    *analog_x = (int)((This.m_Joy[player].m_JoyInfo.dwXpos - This.m_Joy[player].m_JoyCaps.wXmin) * (This.m_Joy[player].m_dXScale) - 128.0);
    *analog_y = (int)((This.m_Joy[player].m_JoyInfo.dwYpos - This.m_Joy[player].m_JoyCaps.wYmin) * (This.m_Joy[player].m_dYScale) - 128.0);
}

static int Joystick_standard_analog_read(int player, int axis)
{
#if 0
    if (This.m_Joy[0].m_bUseJoystick == FALSE)
        return 0;

    switch (axis)
    {
    case X_AXIS:
        if (This.m_Joy[0].m_JoyCaps.wCaps & JOYCAPS_HASU)
            return (This.m_Joy[0].m_JoyInfo.dwUpos) /
                   ((This.m_Joy[0].m_JoyCaps.wUmax - This.m_Joy[0].m_JoyCaps.wUmin) / TRAK_MAXX_RES);
        else
            return 0;
        break;

    case Y_AXIS:
        if (This.m_Joy[0].m_JoyCaps.wCaps & JOYCAPS_HASV)
            return (This.m_Joy[0].m_JoyInfo.dwVpos) /
                   ((This.m_Joy[0].m_JoyCaps.wVmax - This.m_Joy[0].m_JoyCaps.wVmin) / TRAK_MAXY_RES);
        else
            return 0;
        break;
    }
    return 0;
#else
    return 0;
#endif
}

static BOOL Joystick_Available(int nJoyStick)
{
    MMRESULT    mmResult;
    JOYINFOEX   JoyInfoEx;

    JoyInfoEx.dwSize  = sizeof(JoyInfoEx);
    JoyInfoEx.dwFlags = JOY_RETURNBUTTONS | JOY_RETURNX | JOY_RETURNY;

    if (nJoyStick == 1)
        mmResult = joyGetPosEx(JOYSTICKID1, &JoyInfoEx);
    else
    if (nJoyStick == 2)
        mmResult = joyGetPosEx(JOYSTICKID2, &JoyInfoEx);
    else
    if (nJoyStick == 3)
        mmResult = joyGetPosEx(JOYSTICKID3, &JoyInfoEx);
    else
    if (nJoyStick == 4)
        mmResult = joyGetPosEx(JOYSTICKID4, &JoyInfoEx);
    else
        return FALSE;

    if (mmResult == JOYERR_NOERROR)
        return TRUE;
    else
        return FALSE;
}

static BOOL Joystick_OnMessage(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam, LRESULT* pResult)
{
   return FALSE;
}

/***************************************************************************
    Internal functions
 ***************************************************************************/


