/***************************************************************************

    M.A.M.E.32  -  Multiple Arcade Machine Emulator for Win32
    Win32 Portions Copyright (C) 1997-98 Michael Soderstrom and Chris Kirmse
    
    This file is part of MAME32, and may only be used, modified and
    distributed under the terms of the MAME license, in "readme.txt".
    By continuing to use, modify or distribute this file you indicate
    that you have read the license and understand and accept it fully.

 ***************************************************************************/

/***************************************************************************

  DIJoystick.c

 ***************************************************************************/

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <windowsx.h>
#include <stdlib.h>
#include <dinput.h>
#include <assert.h>
#include "mame32.h"
#include "DirectInput.h"
#include "DIJoystick.h"
#include "M32Util.h"
#include "trak.h"
#include <stdio.h>
#include <math.h>

/***************************************************************************
    function prototypes
 ***************************************************************************/

static int              DIJoystick_init(options_type *options);
static void             DIJoystick_exit(void);
static void             DIJoystick_poll_joysticks(void);
static int              DIJoystick_joy_pressed(int joycode);
static const char*      DIJoystick_joy_name(int joycode);
static void             DIJoystick_analogjoy_read(int player, int *analog_x, int *analog_y);
static int              DIJoystick_standard_analog_read(int player, int axis);
static BOOL             DIJoystick_Available(int nJoyStick);
static BOOL             DIJoystick_OnMessage(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam, LRESULT* pResult);

static BOOL CALLBACK DIJoystick_EnumDeviceProc(LPDIDEVICEINSTANCE pdidi, LPVOID pv);

/***************************************************************************
    External variables
 ***************************************************************************/

struct OSDJoystick  DIJoystick = 
{
    { DIJoystick_init },                /* init              */
    { DIJoystick_exit },                /* exit              */
    { DIJoystick_poll_joysticks },      /* poll_joysticks    */
    { DIJoystick_joy_pressed },         /* joy_pressed       */
    { DIJoystick_joy_name },            /* joy_name          */
    { DIJoystick_analogjoy_read },      /* analogjoy_read    */
    { DIJoystick_standard_analog_read },/* standard_analog_read    */
    { DIJoystick_Available },           /* Available         */
    { DIJoystick_OnMessage },           /* OnMessage         */
};

/***************************************************************************
    Internal structures
 ***************************************************************************/

#define MAX_PHYSICAL_JOYSTICKS 20
#define MAX_AXES 20

typedef struct
{
   GUID guid;
   char *name;

   BOOL is_pov; /* true if this is a pov, false if it's an axis */
   BOOL pov_x; /* if is_pov == true, pov_x == true means x part of pov, false == y part of pov */
} axis_type;

typedef struct
{
   GUID guidDevice;
   char *name;

   LPDIRECTINPUTDEVICE2 did;
   
   DWORD num_axes;
   axis_type axes[MAX_AXES];

   DIJOYSTATE  dijs;

} joystick_type;

typedef struct
{
   BOOL valid;
   DWORD num_joystick;
   DWORD x_axis;
   DWORD y_axis;
} mame_joy_info_type;

#define MAX_JOYSTICKS 4

struct tDIJoystick_private
{
    int         use_count; /* the gui and game can both init/exit us, so keep
                              track */
    BOOL        m_bCoinSlot;

    DWORD num_joysticks;
    joystick_type joysticks[MAX_PHYSICAL_JOYSTICKS]; /* actual joystick data! */
   
    mame_joy_info_type mame_joy_info[MAX_JOYSTICKS];

    BOOL m_bMouseLButton; /* mouse buttons, used to override some keys, happ only! */
    BOOL m_bMouseRButton;
};

/* internal functions needing our declarations */
static BOOL CALLBACK DIJoystick_EnumAxisObjectsProc(LPCDIDEVICEOBJECTINSTANCE lpddoi,
                                                    joystick_type *joystick);
static BOOL CALLBACK DIJoystick_EnumPOVObjectsProc(LPCDIDEVICEOBJECTINSTANCE lpddoi,
                                                   joystick_type *joystick);
void ClearJoyState(DIJOYSTATE *pdijs);

static void InitJoystick(joystick_type *joystick);
static void ExitJoystick(joystick_type *joystick);


/***************************************************************************
    Internal variables
 ***************************************************************************/

static struct tDIJoystick_private   This;

GUID guidNULL = {0, 0, 0, {0, 0, 0, 0, 0, 0, 0, 0}};

/***************************************************************************
    External OSD functions  
 ***************************************************************************/
/*
    put here anything you need to do when the program is started. Return 0 if 
    initialization was successful, nonzero otherwise.
*/
static int DIJoystick_init(options_type *options)
{
    DWORD             i;
    HRESULT         hr;

    This.m_bMouseLButton = FALSE;
    This.m_bMouseRButton = FALSE;

    This.use_count++;

    if (This.num_joysticks == 0)
    {
       for (i=0;i<4;i++)
          This.mame_joy_info[i].valid = FALSE;
    }

    This.num_joysticks = 0;

    if (!options->use_joystick)
       return 0;

    if (di == NULL)
    {
        ErrorMsg("DirectInput not initialized");
        return 0;
    }

    /* enumerate for joystick devices */
    hr = IDirectInput_EnumDevices(di, DIDEVTYPE_JOYSTICK,
                 (LPDIENUMDEVICESCALLBACK)DIJoystick_EnumDeviceProc,
                 NULL,
                 DIEDFL_ALLDEVICES);
    if (FAILED(hr))
    {
        ErrorMsg("DirectInput EnumDevices() failed - %08Xh", hr);
        return 0;
    }

    /* create each joystick device, enumerate each joystick for axes, etc */
    for (i=0;i<This.num_joysticks;i++)
    {
       InitJoystick(&This.joysticks[i]);
    }

    /* Are there any joysticks attached? */
    if (This.num_joysticks < 1)
    {
        //ErrorMsg("DirectInput EnumDevices didn't find any joysticks");
        return 0;
    }
    
    for (i=0;i<4;i++)
    {
       This.mame_joy_info[i].valid = TRUE;
       This.mame_joy_info[i].num_joystick = options->physical_joysticks[i];
       if (This.mame_joy_info[i].num_joystick >= This.num_joysticks)
       {
          This.mame_joy_info[i].valid = FALSE;
          continue;
       }

       This.mame_joy_info[i].x_axis = options->x_axes[i];
       if (This.mame_joy_info[i].x_axis >=
           This.joysticks[This.mame_joy_info[i].num_joystick].num_axes)
       {
          This.mame_joy_info[i].valid = FALSE;
          continue;
       }


       This.mame_joy_info[i].y_axis = options->y_axes[i];
       if (This.mame_joy_info[i].y_axis >=
           This.joysticks[This.mame_joy_info[i].num_joystick].num_axes)
       {
          This.mame_joy_info[i].valid = FALSE;
          continue;
       }

    }

    return 0;
}

/*
    put here cleanup routines to be executed when the program is terminated.
*/
static void DIJoystick_exit(void)
{
    DWORD i;

    This.use_count--;

    if (This.use_count > 0)
        return;

    for (i=0;i<This.num_joysticks;i++)
       ExitJoystick(&This.joysticks[i]);
    
    This.num_joysticks = 0;
}

static void DIJoystick_poll_joysticks(void)
{
    HRESULT     hr;
    DWORD i;
    
    This.m_bCoinSlot = 0;

    This.m_bMouseLButton = (GetKeyState(VK_LBUTTON) & 0x8000);
    This.m_bMouseRButton = (GetKeyState(VK_RBUTTON) & 0x8000);

    for (i=0;i<This.num_joysticks;i++)
    {
       int j;
       BOOL found = FALSE;

       /* start by clearing the structure, then fill it in if possible */

       ClearJoyState(&This.joysticks[i].dijs);

       if (This.joysticks[i].did == NULL)
           continue;

       /* only poll the ones we're actually using */
       for (j=0;j<MAX_JOYSTICKS;j++)
          if (This.mame_joy_info[j].valid == TRUE &&
              This.mame_joy_info[j].num_joystick == i)
             found = TRUE;
       if (!found)
          continue;

       hr = IDirectInputDevice2_Poll(This.joysticks[i].did);

       hr = IDirectInputDevice2_GetDeviceState(This.joysticks[i].did,sizeof(DIJOYSTATE),
                                               &This.joysticks[i].dijs);
       if (FAILED(hr))
       {
          if (hr == DIERR_INPUTLOST || hr == DIERR_NOTACQUIRED)
          {
             hr = IDirectInputDevice2_Acquire(This.joysticks[i].did);
          }
          continue;
       }
    }
}

/*
    check if the DIJoystick is moved in the specified direction, defined in
    osdepend.h. Return 0 if it is not pressed, nonzero otherwise.
*/

static int DIJoystick_joy_pressed(int joycode)
{
   DIJOYSTATE *pdijs;
   int i;
   int dz = 50;
   int mame_joystick = 0;

   while (joycode > OSD_JOY_FIRE)
   {
      joycode -= OSD_JOY_FIRE;
      mame_joystick++;
   }

   if (This.mame_joy_info[mame_joystick].valid == FALSE)
      return 0;

   pdijs = &This.joysticks[This.mame_joy_info[mame_joystick].num_joystick].dijs;
       
   switch (joycode)
   {
   case OSD_JOY_LEFT:
   case OSD_JOY_RIGHT:
   case OSD_JOY_UP:
   case OSD_JOY_DOWN:
   {
      int axis_value = 128;
      axis_type *axis;
      if (joycode == OSD_JOY_LEFT || joycode == OSD_JOY_RIGHT)
      {
         axis = &This.joysticks[This.mame_joy_info[mame_joystick].num_joystick].axes
            [This.mame_joy_info[mame_joystick].x_axis];
      }
      else
      {
         axis = &This.joysticks[This.mame_joy_info[mame_joystick].num_joystick].axes
            [This.mame_joy_info[mame_joystick].y_axis];
      }

      if (axis->is_pov)
      {
         int pov_value = pdijs->rgdwPOV[0];
         if (LOWORD(pov_value) != 0xffff)
         {
            /* not centered! */

            int angle = (pov_value + 27000) % 36000;
            angle = (36000 - angle) % 36000;
            angle /= 100;
            
            /* angle is now in degrees counterclockwise from x axis*/
            if (axis->pov_x)
               axis_value = 128 + (int)(127*cos(2*PI*angle/360.0));
            else
               axis_value = 128 - (int)(127*sin(2*PI*angle/360.0));
         }
      }
      else
      {
         if (IsEqualGUID(&axis->guid,&GUID_XAxis))
            axis_value = pdijs->lX;
         if (IsEqualGUID(&axis->guid,&GUID_YAxis))
            axis_value = pdijs->lY;
         if (IsEqualGUID(&axis->guid,&GUID_ZAxis))
            axis_value = pdijs->lZ;
         if (IsEqualGUID(&axis->guid,&GUID_RxAxis))
            axis_value = pdijs->lRx;
         if (IsEqualGUID(&axis->guid,&GUID_RyAxis))
            axis_value = pdijs->lRy;
         if (IsEqualGUID(&axis->guid,&GUID_RzAxis))
            axis_value = pdijs->lRz;
      }

      //printf("input %i axis value %i\n",joycode,axis_value);
      switch (joycode)
      {
      case OSD_JOY_LEFT:
      case OSD_JOY_UP:
         return axis_value < 128 - 128 * dz / 100;
         
      case OSD_JOY_RIGHT:
      case OSD_JOY_DOWN:
         return axis_value > 128 + 128 * dz / 100;
      }
      /* can't get here */
   }      
   case OSD_JOY_FIRE1:
   case OSD_JOY_FIRE2:
   case OSD_JOY_FIRE3:
   case OSD_JOY_FIRE4:
   case OSD_JOY_FIRE5:
   case OSD_JOY_FIRE6:
   case OSD_JOY_FIRE7:
   case OSD_JOY_FIRE8:
   case OSD_JOY_FIRE9:
   case OSD_JOY_FIRE10:
      return pdijs->rgbButtons[joycode-OSD_JOY_FIRE1] & 0x80;

   case OSD_JOY_FIRE:
      for (i=0;i<10;i++)
         if (pdijs->rgbButtons[i] & 0x80)
            return TRUE;
      return FALSE;
   default:
      return 0;
   }
   /* can't get here */
   return 0;
}

/*
    Return the name of a Joystick button.
*/
static const char * DIJoystick_joy_name(int joycode)
{
    return Joystick.joy_name(joycode);
}

/* osd_analog_joyread() returns values from -128 to 128 */
static void DIJoystick_analogjoy_read(int player, int *analog_x, int *analog_y)
{
    assert(player >= 0 && player < 4);

    *analog_x = *analog_y = 0;

    if (This.mame_joy_info[player].valid == FALSE)
        return;

    *analog_x = This.joysticks[This.mame_joy_info[player].num_joystick].dijs.lX - 128;
    *analog_y = This.joysticks[This.mame_joy_info[player].num_joystick].dijs.lY - 128;
}

static int DIJoystick_standard_analog_read(int player, int axis)
{
    int retval;
   
    assert(player >= 0 && player < 4);

    if (This.mame_joy_info[player].valid == FALSE)
       return 0;

    switch (axis)
    {
    case X_AXIS:
        retval = (This.joysticks[This.mame_joy_info[player].num_joystick].dijs.lRz - 128) /
            (128/(TRAK_MAXX_RES/2));
        if (retval > TRAK_MAXX_RES)
            retval = TRAK_MAXX_RES;
        if (retval < -TRAK_MAXX_RES)
            retval = -TRAK_MAXX_RES;
        return retval;
    case Y_AXIS:
        retval = 0;
        return retval;
    }
    return 0;
}


static BOOL DIJoystick_Available(int nJoyStick)
{
    static BOOL bBeenHere = FALSE;
    static BOOL bAvailable = FALSE;
    HRESULT     hr;
    GUID        guidDevice = guidNULL;
    LPDIRECTINPUTDEVICE didTemp;
    LPDIRECTINPUTDEVICE didJoystick;

    if (di == NULL)
    {
        return FALSE;
    }

    if (bBeenHere == FALSE)
        bBeenHere = TRUE;
    else
        return bAvailable;

    /* enumerate for joystick devices */
    hr = IDirectInput_EnumDevices(di, DIDEVTYPE_JOYSTICK,
                 (LPDIENUMDEVICESCALLBACK)inputEnumDeviceProc,
                 &guidDevice,
                 DIEDFL_ALLDEVICES);
    if (FAILED(hr))
    {
       return FALSE;
    }

    /* Are there any joysticks attached? */
    if (IsEqualGUID(&guidDevice, &guidNULL))
    {
        return FALSE;
    }

    hr = IDirectInput_CreateDevice(di, &guidDevice, &didTemp, NULL);
    if (FAILED(hr))
    {
        return FALSE;
    }

    /* Determine if DX5 is available by a QI for a DX5 interface. */
    hr = IDirectInputDevice_QueryInterface(didTemp,
                                           &IID_IDirectInputDevice2,
                                           &didJoystick);
    if (FAILED(hr))
    {
        bAvailable = FALSE;
    }
    else
    {
        bAvailable = TRUE;
        IDirectInputDevice_Release(didJoystick);
    }

    /* dispose of the temp interface */
    IDirectInputDevice_Release(didTemp);

    return bAvailable;
}

static BOOL DIJoystick_OnMessage(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam, LRESULT* pResult)
{
    return FALSE;
}

BOOL DIJoystick_IsHappInterface()
{
    return FALSE;
}

BOOL DIJoystick_KeyPressed(int osd_key)
{
/*
    if (!This.mame_joy_info[0].m_bHappInterface)
        return FALSE;

    switch (osd_key)
    {
    case OSD_KEY_1 :
        return This.m_bMouseLButton;
    case OSD_KEY_2 :
        return This.m_bMouseRButton;
    case OSD_KEY_3 :
        return This.m_bCoinSlot;
    case OSD_KEY_ESC :
        return This.m_bMouseLButton && This.m_bMouseRButton && 
           DIJoystick_joy_pressed(OSD_JOY_DOWN);
    }
*/
    return FALSE;
}

int DIJoystick_GetNumPhysicalJoysticks()
{
   return This.num_joysticks;
}

char * DIJoystick_GetPhysicalJoystickName(int num_joystick)
{
   return This.joysticks[num_joystick].name;
}

int DIJoystick_GetNumPhysicalJoystickAxes(int num_joystick)
{
   return This.joysticks[num_joystick].num_axes;
}

char * DIJoystick_GetPhysicalJoystickAxisName(int num_joystick,int num_axis)
{
   return This.joysticks[num_joystick].axes[num_axis].name;
}


/***************************************************************************
    Internal functions
 ***************************************************************************/
BOOL CALLBACK DIJoystick_EnumDeviceProc(LPDIDEVICEINSTANCE pdidi, LPVOID pv)
{
   char buffer[5000];

   This.joysticks[This.num_joysticks].guidDevice = pdidi->guidInstance;

   sprintf(buffer,"%s (%s)",pdidi->tszInstanceName,pdidi->tszProductName);
   This.joysticks[This.num_joysticks].name = (char *)malloc(strlen(buffer)+1);
   strcpy(This.joysticks[This.num_joysticks].name,buffer);

   This.num_joysticks++;
   
   //ErrorMsg("got joystick, %s",This.joysticks[This.num_joysticks-1].name);

   return DIENUM_CONTINUE;
}

BOOL CALLBACK DIJoystick_EnumAxisObjectsProc(LPCDIDEVICEOBJECTINSTANCE lpddoi,
                                             joystick_type *joystick)
{
   DIPROPRANGE diprg;
   HRESULT hr;

   joystick->axes[joystick->num_axes].guid = lpddoi->guidType;

   joystick->axes[joystick->num_axes].name = (char *)malloc(strlen(lpddoi->tszName)+1);
   strcpy(joystick->axes[joystick->num_axes].name,lpddoi->tszName);

   joystick->axes[joystick->num_axes].is_pov = FALSE;

   //ErrorMsg("got axis %s, offset %i",lpddoi->tszName, joystick->num_axes);
   
   diprg.diph.dwSize       = sizeof(diprg);
   diprg.diph.dwHeaderSize = sizeof(diprg.diph);
   diprg.diph.dwObj        = lpddoi->dwOfs;
   diprg.diph.dwHow        = DIPH_BYOFFSET;
   diprg.lMin              = 0;
   diprg.lMax              = 255;
   
   hr = IDirectInputDevice2_SetProperty(joystick->did,DIPROP_RANGE,&diprg.diph);
   if (FAILED(hr)) /* if this fails, don't use this axis */
   {
       free(joystick->axes[joystick->num_axes].name);
       joystick->axes[joystick->num_axes].name = NULL;
       return DIENUM_CONTINUE;
   }
#ifdef JOY_DEBUG
   if (FAILED(hr))
   {
      ErrorMsg("DirectInput SetProperty() joystick axis %s failed - %08Xh\n",
               joystick->axes[joystick->num_axes].name,hr);
   }
#endif
    
   /* Set axis dead zone to 0; we need accurate #'s for analog joystick reading. */
   
   hr = SetDIDwordProperty(joystick->did,DIPROP_DEADZONE,lpddoi->dwOfs,DIPH_BYOFFSET,0);
#ifdef JOY_DEBUG
   if (FAILED(hr))
   {
      ErrorMsg("DirectInput SetProperty() joystick axis %s dead zone failed - %08Xh\n",
               joystick->axes[joystick->num_axes].name,hr);

   }
#endif
   joystick->num_axes++;
   
   return DIENUM_CONTINUE;
}

BOOL CALLBACK DIJoystick_EnumPOVObjectsProc(LPCDIDEVICEOBJECTINSTANCE lpddoi,
                                            joystick_type *joystick)
{
   /* add 2 axes, one x and one y */
   
   joystick->axes[joystick->num_axes].guid = lpddoi->guidType;
   joystick->axes[joystick->num_axes].name = (char *)malloc(strlen(lpddoi->tszName)+1
                                                            + strlen(" virtual X axis"));
   strcpy(joystick->axes[joystick->num_axes].name,lpddoi->tszName);
   strcat(joystick->axes[joystick->num_axes].name," virtual X axis");
   joystick->axes[joystick->num_axes].is_pov = TRUE;
   joystick->axes[joystick->num_axes].pov_x = TRUE;
   joystick->num_axes++;
   
   joystick->axes[joystick->num_axes].guid = lpddoi->guidType;
   joystick->axes[joystick->num_axes].name = (char *)malloc(strlen(lpddoi->tszName)+1
                                                            + strlen(" virtual Y axis"));
   strcpy(joystick->axes[joystick->num_axes].name,lpddoi->tszName);
   strcat(joystick->axes[joystick->num_axes].name," virtual Y axis");
   joystick->axes[joystick->num_axes].is_pov = TRUE;
   joystick->axes[joystick->num_axes].pov_x = FALSE;
   joystick->num_axes++;
   
   return DIENUM_CONTINUE;
}

void ClearJoyState(DIJOYSTATE *pdijs)
{
    memset(pdijs,0,sizeof(DIJOYSTATE));
    pdijs->lX = 128;
    pdijs->lY = 128;
    pdijs->lZ = 128;
    pdijs->lRx = 128;
    pdijs->lRy = 128;
    pdijs->lRz = 128;
    pdijs->rglSlider[0] = 128;
    pdijs->rglSlider[1] = 128;
    pdijs->rgdwPOV[0] = -1;
    pdijs->rgdwPOV[1] = -1;
    pdijs->rgdwPOV[2] = -1;
    pdijs->rgdwPOV[3] = -1;
}

                                        
void InitJoystick(joystick_type *joystick)
{
    LPDIRECTINPUTDEVICE didTemp;
    HRESULT hr;

    joystick->did = NULL;
    joystick->num_axes = 0;

    /* get a did1 interface first... */
    hr = IDirectInput_CreateDevice(di,&joystick->guidDevice,&didTemp,NULL);
    if (FAILED(hr))
    {
       ErrorMsg("DirectInput CreateDevice() joystick failed - %08Xh\n",hr);
       return;
    }
    
    /* get a did2 interface to work with polling (most) joysticks */
    hr = IDirectInputDevice_QueryInterface(didTemp,
                                           &IID_IDirectInputDevice2,
                                           &joystick->did);

    /* dispose of the temp interface */
    IDirectInputDevice_Release(didTemp);

    /* check result of getting the did2 */
    if (FAILED(hr))
    {
       /* no error message because this happens in dx3 */
       /* ErrorMsg("DirectInput QueryInterface joystick failed\n"); */
       joystick->did = NULL;
       return;
    }

    
    hr = IDirectInputDevice2_SetDataFormat(joystick->did,&c_dfDIJoystick);
    if (FAILED(hr))
    {
       ErrorMsg("DirectInput SetDataFormat() joystick failed - %08Xh\n", hr);
       return;
    }

    hr = IDirectInputDevice2_SetCooperativeLevel(joystick->did,MAME32App.m_hWnd,
                                                 DISCL_NONEXCLUSIVE | DISCL_BACKGROUND);
    if (FAILED(hr))
    {
       ErrorMsg("DirectInput SetCooperativeLevel() joystick failed - %08Xh\n", hr);
       return;
    }

    /* enumerate our axes */

    hr = IDirectInputDevice_EnumObjects(joystick->did,(LPDIENUMDEVICEOBJECTSCALLBACK)
                                        DIJoystick_EnumAxisObjectsProc,joystick,DIDFT_AXIS);
    if (FAILED(hr))
    {
        ErrorMsg("DirectInput EnumObjects() Axes failed - %08Xh",hr);
        return;
    }

    /* enumerate our POV hats */

    hr = IDirectInputDevice_EnumObjects(joystick->did,(LPDIENUMDEVICEOBJECTSCALLBACK)
                                        DIJoystick_EnumPOVObjectsProc,joystick,DIDFT_POV);
    if (FAILED(hr))
    {
        ErrorMsg("DirectInput EnumObjects() POVs failed - %08Xh",hr);
        return;
    }

    hr = IDirectInputDevice2_Acquire(joystick->did);
    if (FAILED(hr)) 
    {
       ErrorMsg("DirectInputDevice Acquire joystick failed!\n");
       return;
    }

    /* start by clearing the structures */

    ClearJoyState(&joystick->dijs);
}

void ExitJoystick(joystick_type *joystick)
{
    DWORD i;
    
    if (joystick->did != NULL)
    {
        IDirectInputDevice_Unacquire(joystick->did);
        IDirectInputDevice_Release(joystick->did);
        joystick->did = NULL;
    }
    
    for (i=0;i<joystick->num_axes;i++)
    {
        if (joystick->axes[i].name)
            free(joystick->axes[i].name);
        joystick->axes[i].name = NULL;
    }
    
    if (joystick->name != NULL)
    {
        free(joystick->name);
        joystick->name = NULL;
    }
}




