/***************************************************************************

    M.A.M.E.32  -  Multiple Arcade Machine Emulator for Win32
    Win32 Portions Copyright (C) 1997-98 Michael Soderstrom and Chris Kirmse
    
    This file is part of MAME32, and may only be used, modified and
    distributed under the terms of the MAME license, in "readme.txt".
    By continuing to use, modify or distribute this file you indicate
    that you have read the license and understand and accept it fully.

 ***************************************************************************/
 
 /***************************************************************************

  win32ui.c

  Win32 GUI code.

  Created 8/12/97 by Christopher Kirmse (ckirmse@ricochet.net)
  Additional code November 1997 by Jeff Miller (miller@aa.net)
  Much more July 1998 by Mike Haaland (mhaaland@hypertech.com)
  Added Spitters/Property Sheets/Removed Tabs/Added Tree Control in
  Nov/Dec 1998 - Mike Haaland

***************************************************************************/

#include "driver.h"
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <shellapi.h>
#include <windowsx.h>
#include <stdio.h>
#include <io.h>
#include <fcntl.h>
#include <commctrl.h>
#include <commdlg.h>
#include <strings.h>
#include <sys/stat.h>
#include <zmouse.h>         /* Intellimouse */
#include <wingdi.h>
#include <tchar.h>

#include "resource.h"
#include "mame32.h"
#include "M32Util.h"
#include "DirectDraw.h"
#include "DirectInput.h"
#include "DDrawDisplay.h"   /* For display modes. */

#ifndef NOSEAL
#include "SealSound.h"      /* For sound devices. */
#endif

#include "Joystick.h"       /* For joystick avalibility. */
#include "DIJoystick.h"     /* For DIJoystick avalibility. */
#include "DIKeyboard.h"     /* For DIKeyboard avalibility. */
#include "osdepend.h"
#include "mzip.h"
#include "file.h"
#include "audit32.h"
#include "Directories.h"
#include "Screenshot.h"
#include "Properties.h"
#include "ColumnEdit.h"
#include "TreeView.h"
#include "Splitters.h"
#include "JPeg.h"

// Uncomment this to use internal background bitmaps
// #define INTERNAL_BKBITMAP


#define MM_PLAY_GAME (WM_APP + 15000)

/* Max size of a sub-menu */
#define DBU_MIN_WIDTH  292
#define DBU_MIN_HEIGHT 190

int MIN_WIDTH  = DBU_MIN_WIDTH;
int MIN_HEIGHT = DBU_MIN_HEIGHT;

typedef BOOL (WINAPI *common_file_dialog_proc)(LPOPENFILENAME lpofn);

/***************************************************************************
    function prototypes
 ***************************************************************************/

/* Exported prototypes */
void             UpdateDisplayModeUI(HWND hwnd, DWORD dwDepth);
BOOL             GameUsesTrackball(int game);
void             DepthSelectionChange(HWND hWnd, HWND hWndCtrl);
void             OptOnHScroll(HWND hwnd, HWND hwndCtl, UINT code, int pos);
void             SetYM3812Enabled(HWND hWnd, int index);
void             SetStereoEnabled(HWND hWnd, int index);
void             InitializeMisc(HWND hDlg);
void             InitializeOptions(HWND hDlg);

static BOOL             Win32UI_init(HINSTANCE hInstance, LPSTR lpCmdLine);
static void             Win32UI_exit();

static BOOL             PumpMessage();
static BOOL             PumpAndReturnMessage(MSG *pmsg);
static void             OnIdle();
static void             OnSize(HWND hwnd, UINT state, int width, int height);
static long WINAPI      MameWindowProc(HWND hwnd,UINT message,UINT wParam,LONG lParam);

static void             SetView(int menu_id,int listview_style);
static void             ResetListView();
static void             UpdateGameList();
static void             ReloadIcons(HWND hWnd);
static void             PollGUIJoystick();
static void             FlickerSelectionChange(HWND hwnd);
static void             GammaSelectionChange(HWND hwnd);
static void             BrightnessSelectionChange(HWND hwnd);
static void             BeamSelectionChange(HWND hwnd);
static BOOL             MameCommand(HWND hwnd,int id, HWND hwndCtl, UINT codeNotify);

static void             SetStandardTitleBar();
static BOOL             MamePickerNotify(NMHDR *nm);
static BOOL             TreeViewNotify(NMHDR *nm);
static char*            ConvertAmpersandString(const char *s);

static void             LoadListBitmap(void);
static void             PaintControl(HWND hWnd, BOOL useBitmap);

static void             InitPicker();
static int CALLBACK     ListCompareFunc(LPARAM index1, LPARAM index2, int sort_subitem);
static void             InitializeDisplayModeUI(HWND hwnd);
static void             InitializeSoundUI(HWND hwnd);
static void             InitializeDepthUI(HWND hwnd);
static void             InitializeDisplayTypeUI(HWND hwnd);
static void             InitializeSkippingUI(HWND hwnd);
static void             InitializeRotateUI(HWND hwnd);
static void             EnableSubItems(HWND hwnd);

static int              GetSelectedPick();
static int              GetSelectedPickItem();
static void             SetSelectedPick(int new_index);
static void             SetSelectedPickItem(int val);

static BOOL             ParseCommandLine(char *command_line);
static LRESULT CALLBACK AboutDialogProc(HWND hDlg, UINT Msg, WPARAM wParam, LPARAM lParam);
static LRESULT CALLBACK DirectXDialogProc(HWND hDlg, UINT Msg, WPARAM wParam, LPARAM lParam);
static BOOL CALLBACK    MameHelpDialogProc(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam);
static LRESULT CALLBACK OptionsSubDialogProc(HWND hDlg, UINT Msg, WPARAM wParam, LPARAM lParam);

static void             MamePlayRecordGame();
static void             MamePlayBackGame();
static BOOL             CommonFileDialog(common_file_dialog_proc cfd,char *filename);
static void             MamePlayGame();
static void             MamePlayGameWithOptions(struct GameOptions mame_options);
static void             ColumnSort(int column, BOOL bColumn);

static void             ToggleScreenShot(void);
static void             AdjustMetrics(void);
static void             SetLargeScreenShotCheck(void);
static void             EnablePlayOptions(int nIndex, options_type *o);

/* for header */
static LRESULT CALLBACK HeaderWndProc(HWND, UINT, WPARAM, LPARAM);
static void             Header_SetSortInfo(HWND hwndList, int nCol, BOOL bAsc);
static void             Header_Initialize(HWND hwndList);

/* Icon routines */
static BOOL             CreateIcons(HWND hWnd);
static int              WhichIcon(int nItem);
static void             AddIcon(int index);

/* Context Menu handlers */
static void             UpdateMenu(HWND hwndList, HMENU hMenu);
static BOOL             HandleContextMenu( HWND hWnd, WPARAM wParam, LPARAM lParam);
static BOOL             HeaderOnContextMenu(HWND hWnd, WPARAM wParam, LPARAM lParam);

/* Re/initialize the ListView header columns */
static void             ResetColumnDisplay(BOOL firstime);

/* Custom Draw item */
static void             DrawItem(LPDRAWITEMSTRUCT lpDrawItemStruct);
static int              GetNumColumns(HWND hWnd);

/* Column Order as Displayed */
static BOOL             oldControl = FALSE;
static int              realColumn[COLUMN_MAX];

/* Background Image handles also accessed from TreeView.c */
HPALETTE         hPALbg   = 0;
HBITMAP          hBitmap  = 0;
MYBITMAPINFO     bmDesc;

static BOOL			    screenShotAvailable = FALSE;

/* Icon variables */
static HIMAGELIST       hLarge;
static HIMAGELIST       hSmall;
static int              *icon_index = NULL;

/* First Item in the current list */
static int              firstListItem = 0;

/***************************************************************************
    External variables
 ***************************************************************************/

int win32_debug;

/***************************************************************************
    Internal structures
 ***************************************************************************/

/* Which edges of a control are anchored to the corresponding side of the parent window */
#define RA_LEFT     0x01
#define RA_RIGHT    0x02
#define RA_TOP      0x04
#define RA_BOTTOM   0x08
#define RA_ALL      0x0F

#define RA_END  0
#define RA_ID   1
#define RA_HWND 2

typedef struct 
{
    int         type;       /* Either RA_ID or RA_HWND, to indicate which member of u is used; or RA_END
                               to signify last entry */
    union {                 /* Can identify a child window by control id or by handle */
        int     id;         /* Window control id */
        HWND    hwnd;       /* Window handle */    
    } u;
    int         action;     /* What to do when control is resized */
    void        *subwindow; /* Points to a Resize structure for this subwindow; NULL if none */
} ResizeItem;

typedef struct {
    RECT rect;          /* Client rect of window; must be initialized before first resize */
    ResizeItem *items;  /* Array of subitems to be resized */
} Resize;

static void             ResizeWindow(HWND hParent, Resize *r);

/* List view Icon defines */
#define ICONMAP_WIDTH       16
#define ICONMAP_HEIGHT      16
#define LG_ICONMAP_WIDTH    32
#define LG_ICONMAP_HEIGHT   32
#define EXLG_ICONMAP_WIDTH  48
#define EXLG_ICONMAP_HEIGHT 48

#define NUM_ICONS           4

/* List view Column text */
char *column_names[COLUMN_MAX] = {
    "Game",
    "ROMs",
    "Samples",
    "Directory",
    "Type",
    "Trackball",
    "Played",
    "Manufacturer",
    "Year",
    "Clone Of"
};

/* Icon names we will load and use */
static char *icon_names[] = {
    "noroms",
    "roms",
    "unknown",
    "warning"
};


/***************************************************************************
    Internal variables
 ***************************************************************************/
static HWND hMain;
static HACCEL hAccel;

static HWND hPicker;
static HWND hwndList;
static HWND hTreeView;

static HWND hOptsDlog;
static HWND hMiscDlog;
static HWND hSSDlog;
static HWND hDlog = 0;

static HINSTANCE hInst;

static HFONT hFont;     /* Font for list view */

static int              page_index;
static game_data_type*  game_data; 
static int              game_count;

static int  last_sort = 0;

/* global data--know where to send messages */
static BOOL in_emulation;

/* quit after game */
static BOOL quit;

/* idle work at startup */
static BOOL idle_work;
static int  game_index;

/* current menu check for listview */
static int current_view_id;

//static BOOL show_all; /* whether currently showing all games in the list view */

/* Tree control variables */
static BOOL bShowTree = 1;
static BOOL bShowToolBar = 1;
static BOOL bShowStatusBar = 1;

/* use a joystick subsystem in the gui? */
static struct OSDJoystick *joygui;

/* Intellimouse available? (0 = no) */
static UINT uiMsh_MsgMouseWheel;

/* sort columns in reverse order */
static BOOL reverse_sort = FALSE;
static UINT lastColumnClick = 0;
static UINT m_uHeaderSortCol = 0;
static BOOL m_fHeaderSortAsc = TRUE;
static WNDPROC g_lpHeaderWndProc = NULL;

/* Tool and Status bar variables */
HWND hStatusBar = 0;
HWND hToolBar = 0;
#define NUM_TOOLBUTTONS 11

TBBUTTON tbb[NUM_TOOLBUTTONS] = {
    {0, 0,                  TBSTATE_ENABLED, TBSTYLE_SEP,        0, 0},
    {0, ID_VIEW_FOLDERS,    TBSTATE_ENABLED, TBSTYLE_CHECK,      0, 0},
    {1, ID_VIEW_SCREEN_SHOT,TBSTATE_ENABLED, TBSTYLE_CHECK,      0, 1},
    {0, 0,                  TBSTATE_ENABLED, TBSTYLE_SEP,        0, 0},
    {2, ID_VIEW_ICON,       TBSTATE_ENABLED, TBSTYLE_CHECKGROUP, 0, 2},
    {3, ID_VIEW_SMALL_ICON, TBSTATE_ENABLED, TBSTYLE_CHECKGROUP, 0, 3},
    {4, ID_VIEW_LIST_MENU,  TBSTATE_ENABLED, TBSTYLE_CHECKGROUP, 0, 4},
    {5, ID_VIEW_DETAIL,     TBSTATE_ENABLED, TBSTYLE_CHECKGROUP, 0, 5},
    {0, 0,                  TBSTATE_ENABLED, TBSTYLE_SEP,        0, 0},
    {6, ID_HELP_ABOUT,      TBSTATE_ENABLED, TBSTYLE_BUTTON,     0, 6},
    {7, ID_HELP_CONTENTS,   TBSTATE_ENABLED, TBSTYLE_BUTTON,     0, 7}
};

#define NUM_TOOLTIPS 7

char szTbStrings[NUM_TOOLTIPS + 1][30] = {
    "Toggle Folder List",
    "Toggle Screen Shot",
    "Large Icons",
    "Small Icons",
    "List",
    "Details",
    "About",
    "Help"
};

int CommandToString[] = {
    ID_VIEW_FOLDERS,
    ID_VIEW_SCREEN_SHOT,
    ID_VIEW_ICON,
    ID_VIEW_SMALL_ICON,
    ID_VIEW_LIST_MENU,
    ID_VIEW_DETAIL,
    ID_HELP_ABOUT,
    ID_HELP_CONTENTS,
    -1
};

/* How to resize main window */
static ResizeItem main_resize_items[] = {
    { RA_HWND, 0,            RA_LEFT  | RA_RIGHT | RA_TOP,      NULL },
    { RA_HWND, 0,            RA_LEFT  | RA_RIGHT | RA_BOTTOM,   NULL },
    { RA_ID,   IDC_DIVIDER,  RA_LEFT  | RA_RIGHT | RA_TOP,      NULL },
    { RA_ID,   IDC_TREE,     RA_LEFT  | RA_BOTTOM | RA_TOP,     NULL },
    { RA_ID,   IDC_LIST,     RA_ALL,                            NULL },
    { RA_ID,   IDC_SPLITTER, RA_LEFT  | RA_BOTTOM | RA_TOP,     NULL },
    { RA_ID,   IDC_SPLITTER2,RA_RIGHT | RA_BOTTOM | RA_TOP,     NULL },
    { RA_ID,   IDC_SSFRAME,  RA_RIGHT | RA_BOTTOM | RA_TOP,     NULL },
    { RA_ID,   IDC_SSPICTURE,RA_RIGHT | RA_BOTTOM | RA_TOP,     NULL },
    { RA_ID,   IDC_SSNONE,   RA_RIGHT | RA_BOTTOM,              NULL },
    { RA_ID,   IDC_SSDEFPIC, RA_RIGHT | RA_TOP,                 NULL },
    { RA_END,  0,            0,                                 NULL }
};

static Resize main_resize = { {0, 0, 0, 0}, main_resize_items };

/* our dialog/configured options */
static options_type playing_game_options;

/* last directory for common file dialogs */
static char last_directory[MAX_PATH];

/***************************************************************************
    External functions  
 ***************************************************************************/

void CopyToolTipText (LPTOOLTIPTEXT lpttt)
{
    int   i;
    int   iButton = lpttt->hdr.idFrom;
    LPSTR pString;
    LPSTR pDest = lpttt->lpszText;
    
    // Map command ID to string index
    for (i = 0 ; CommandToString[i] != -1; i++)
    {
        if (CommandToString[i] == iButton)
        {
            iButton = i;
            break;
        }
    }

    // Check for valid parameter
    pString = (iButton > NUM_TOOLTIPS) ? "Invalid Button Index" : szTbStrings[iButton];
    
    lstrcpy (pDest, pString) ;
}

int WINAPI WinMain(HINSTANCE    hInstance,
                   HINSTANCE    hPrevInstance,
                   LPSTR        lpCmdLine,
                   int          nCmdShow)
{
    MSG     msg;

    if (!Win32UI_init(hInstance, lpCmdLine))
        return 1;

    /*
        Simplified MFC Run() alg. See mfc/src/thrdcore.cpp.
    */
    for (;;)
    {
        /* phase1: check to see if we can do idle work */
        while (idle_work && !PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE))
        {
            /* call OnIdle while idle_work */
            OnIdle();
        }

        /* phase2: pump messages while available */
        do
        {
            /* pump message, but quit on WM_QUIT */
            if (!PumpMessage())
            {
                Win32UI_exit();
                return msg.wParam;
            }

        } while (PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE));
    }

    return 0;
}

/***************************************************************************
    Internal functions
 ***************************************************************************/

#ifndef ToolBar_CheckButton
#define ToolBar_CheckButton(hWnd, idButton, fCheck) \
    SendMessage(hWnd, TB_CHECKBUTTON, (WPARAM)idButton, (LPARAM)MAKELONG(fCheck, 0));
#endif

typedef struct tagPOPUPSTRING
{
    HMENU hMenu;
    UINT uiString;
} POPUPSTRING;

#define MAX_MENUS 3

POPUPSTRING popstr[MAX_MENUS];

HWND InitToolbar(HWND hParent)
{
    return CreateToolbarEx(hParent,
        WS_CHILD | WS_VISIBLE | WS_CLIPSIBLINGS |
        CCS_TOP | TBSTYLE_FLAT | TBSTYLE_TOOLTIPS,
        1,
        8,
        hInst,
        IDB_TOOLBAR,
        tbb,
        NUM_TOOLBUTTONS,
        16,
        16,
        0,
        0,
        sizeof(TBBUTTON));
}

HWND InitStatusBar(HWND hParent)
{
    HWND hWndSB;

    HMENU hMenu = GetMenu(hParent);
    
    popstr[0].hMenu = 0;
    popstr[0].uiString = 0;
    popstr[1].hMenu = hMenu;
    popstr[1].uiString = IDS_UI_FILE;
    popstr[2].hMenu = GetSubMenu(hMenu,1);
    popstr[2].uiString = IDS_UI_TOOLBAR;
    popstr[3].hMenu = 0;
    popstr[3].uiString = 0;
 
    hWndSB =     CreateStatusWindow(
        WS_CHILD | WS_VISIBLE | WS_CLIPSIBLINGS |
        CCS_BOTTOM | SBARS_SIZEGRIP,
        "Ready",
        hParent,
        2);
    return hWndSB;
}

LRESULT Statusbar_MenuSelect (HWND hwnd, WPARAM wParam, LPARAM lParam)
{
    UINT fuFlags = (UINT) HIWORD (wParam) ;
    HMENU hMainMenu = NULL ;
    int iMenu = 0 ;
    
    // Handle non-system popup menu descriptions.
    if ((fuFlags & MF_POPUP) &&
        (!(fuFlags & MF_SYSMENU)))
    {
        for (iMenu = 1 ; iMenu < MAX_MENUS ; iMenu++)
        {
            if ((HMENU) lParam == popstr[iMenu].hMenu)
            {
                hMainMenu = (HMENU) lParam ;
                break ;
            }
        }
    }
    
    // Display helpful text in status bar
    MenuHelp (WM_MENUSELECT, wParam, lParam, hMainMenu, hInst, 
        hStatusBar, (UINT *) &popstr[iMenu]) ;
    
    return 0 ;
}

static BOOL Win32UI_init(HINSTANCE hInstance, LPSTR lpCmdLine)
{
    WNDCLASS    wndclass;
    RECT        rect;
    int         i;
    struct GameDriver * drvr;
    int j, found;

    joygui = NULL;

    game_count = 0;
    while (drivers[game_count] != 0)
        game_count++;

    game_data = (game_data_type *)malloc(game_count * sizeof(game_data_type));

    wndclass.style           = CS_HREDRAW | CS_VREDRAW;
    wndclass.lpfnWndProc     = MameWindowProc;
    wndclass.cbClsExtra      = 0;
    wndclass.cbWndExtra      = DLGWINDOWEXTRA;
    wndclass.hInstance       = hInstance;
    wndclass.hIcon           = LoadIcon(hInstance, 
        MAKEINTRESOURCE(IDI_MAME32_ICON));
    wndclass.hCursor         = NULL;
    wndclass.hbrBackground   = (HBRUSH)(COLOR_3DFACE + 1);
    wndclass.lpszMenuName    = MAKEINTRESOURCE(IDR_UI_MENU);
    wndclass.lpszClassName   = "MAME32";

    RegisterClass(&wndclass);

    InitCommonControls();

    // Are we using an Old comctl32.dll?
    if (oldControl = (GetDllVersion() == FALSE))
    {
        char        buf[300];
        
        strcpy(buf, "Mame32 has detected and old version of comctl32.dll\n\n");
        strcat(buf, "Game Properties, many configuration options and\n");
        strcat(buf, "features are not available without an updated DLL\n\n");
        strcat(buf, "Please install the common control update found at:\n\n");
        strcat(buf, "http://msdn.microsoft.com/visualc/downloads/updates.asp\n\n");
        strcat(buf, "Would you like to continue without using the new features?\n");
        if (IDNO == MessageBox(0, buf, "MAME32 Outdated comctl32.dll Warning", MB_YESNO | MB_ICONWARNING))
            return FALSE;
    }

    OptionsInit(game_count);

    for (i = 0; i < game_count; i++)
    {
        found = FALSE;
        if ((drvr = (struct GameDriver *)drivers[i]->clone_of) != (struct GameDriver *)0)
        {
            for (j = 0; j < game_count; j++)
            {
                if (drvr == drivers[j])
                {
                    found = TRUE;
                    break;
                }
            }
        }
        if (!found && drivers[i]->clone_of == 0)
            found = TRUE;

        game_data[i].in_list = found;
    }

    // init files after OptionsInit to init paths
    File.init();
    strcpy(last_directory, GetInpDir());

    hMain = MAME32App.m_hwndUI = CreateDialog(hInstance,
        MAKEINTRESOURCE(IDD_MAIN), 0, NULL);

    hPicker = hMain;

    /* Stash hInstance for later use */
    hInst = hInstance;

    hToolBar = InitToolbar(hMain);
 
    hStatusBar = InitStatusBar(hMain);

    main_resize_items[0].u.hwnd = hToolBar;
    main_resize_items[1].u.hwnd = hStatusBar;

    /* In order to handle 'Large Fonts' as the Windows
     * default setting, we need to make the dialogs small
     * enough to fit in our smallest window size with
     * large fonts, then resize the picker, tab and button
     * controls to fill the window, no matter which font
     * is currently set.  This will still look like bad
     * if the user uses a bigger default font than 125%
     * (Large Fonts) on the Windows display setting tab.
     *
     * NOTE: This has to do with Windows default font size
     * settings, NOT our picker font size.
     */

    GetClientRect(hMain, &rect);
    
    hTreeView       = GetDlgItem(hPicker, IDC_TREE);
    hwndList        = GetDlgItem(hPicker, IDC_LIST);

    InitSplitters();


    AddSplitter(GetDlgItem(hPicker, IDC_SPLITTER), hTreeView, hwndList,
                AdjustSplitter1Rect);
    AddSplitter(GetDlgItem(hPicker, IDC_SPLITTER2), hwndList,
                GetDlgItem(hPicker,IDC_SSFRAME),AdjustSplitter2Rect);

    /* Initial adjustment of controls on the Picker window */
    ResizePickerControls(hPicker);

    /* Reset the font */
    {
        LOGFONT     logfont;

        GetListFont(&logfont);
        hFont = CreateFontIndirect(&logfont);
        if (hFont != NULL)
        {
            if (hwndList != NULL)
                SetWindowFont(hwndList, hFont, FALSE);
            if (hTreeView != NULL)
                SetWindowFont(hTreeView, hFont, FALSE);
        }
    }

    /* Update the screen shot option */
    //UpdateScreenShot();

    /* Set up the option dialog boxes for rapid display */
    hOptsDlog = CreateDialog(hInst, 
        MAKEINTRESOURCE(IDD_DISPLAY), hMain, OptionsSubDialogProc);

    /* Initialize the Display options tab */
    InitializeOptions(hOptsDlog);
    InitializeMisc(hOptsDlog);
    //UpdateDialogItems(hOptsDlog);
    NewOptionsToDialog(hOptsDlog, TAB_DISPLAY, GetDefaultOptions());

    /* Sound/Input tab */
    hMiscDlog = CreateDialog(hInst,
        MAKEINTRESOURCE(IDD_MISC), hMain, OptionsSubDialogProc);
 
    /* Initialize the Sound/Input tab */
    InitializeOptions(hMiscDlog);
    InitializeMisc(hMiscDlog);
    //UpdateDialogItems(hMiscDlog);
    NewOptionsToDialog(hMiscDlog, TAB_MISC, GetDefaultOptions());

    idle_work = TRUE;
    game_index = 0;

    switch(GetViewMode())
    {
    case VIEW_LARGE_ICONS:
        SetView(ID_VIEW_ICON,LVS_ICON);
        break;
    case VIEW_SMALL_ICONS:
        SetView(ID_VIEW_SMALL_ICON,LVS_SMALLICON);
        break;
    case VIEW_INLIST:
        SetView(ID_VIEW_LIST_MENU,LVS_LIST);
        break;
    case VIEW_REPORT:
    default:
        SetView(ID_VIEW_REPORT,LVS_REPORT);
        break;
    }

    SetLargeScreenShotCheck();
    bShowTree = GetShowFolderList();
    bShowToolBar = GetShowToolBar();
    bShowStatusBar = GetShowStatusBar();

    CheckMenuItem(GetMenu(hMain), ID_VIEW_FOLDERS, (bShowTree) ? MF_CHECKED : MF_UNCHECKED);
    ToolBar_CheckButton(hToolBar, ID_VIEW_FOLDERS, (bShowTree) ? MF_CHECKED : MF_UNCHECKED);
    CheckMenuItem(GetMenu(hMain), ID_VIEW_TOOLBARS, (bShowToolBar) ? MF_CHECKED : MF_UNCHECKED);
    ShowWindow(hToolBar, (bShowToolBar) ? SW_SHOW : SW_HIDE);
    CheckMenuItem(GetMenu(hMain), ID_VIEW_STATUS, (bShowStatusBar) ? MF_CHECKED : MF_UNCHECKED);
    ShowWindow(hStatusBar, (bShowStatusBar) ? SW_SHOW : SW_HIDE);

    if (oldControl)
    {
        EnableMenuItem(GetMenu(hMain), ID_CUSTOMIZE_FIELDS, MF_GRAYED);
        EnableMenuItem(GetMenu(hMain), ID_GAME_PROPERTIES,  MF_GRAYED);
    }

    // Init DirectDraw
    if (!DirectDrawInitialize())
    {
       DialogBox(GetModuleHandle(NULL),MAKEINTRESOURCE(IDD_DIRECTX), NULL, DirectXDialogProc);
       return FALSE;
    }

    LoadListBitmap();

    InitTree(hTreeView, game_count);

    /* Subclass the tree control for painting */
    Tree_Initialize(hTreeView);

    SetCurrentFolder(GetFolderByID(GetSavedFolderID()));

    // Initialize listview columns
    InitPicker();

    SetFocus(hwndList);

    // Init Intellimouse
    {
        UINT uiMsh_Msg3DSupport;
        UINT uiMsh_MsgScrollLines;
        BOOL f3DSupport;
        INT  iScrollLines;
        HWND hwndMsWheel;

        hwndMsWheel = HwndMSWheel(
            &uiMsh_MsgMouseWheel, &uiMsh_Msg3DSupport,
            &uiMsh_MsgScrollLines, &f3DSupport, &iScrollLines);
    }

    // Init DirectInput
    if (!DirectInputInitialize())
    {
       DialogBox(GetModuleHandle(NULL),MAKEINTRESOURCE(IDD_DIRECTX), NULL, DirectXDialogProc);
       return FALSE;
    }
    
    if (ParseCommandLine(lpCmdLine))
    {
        SetCurrentFolder(GetFolderByID(0));
        ResetListView();
        PostMessage(hMain, MM_PLAY_GAME, 0, 0);
    }
    else
    {
        AdjustMetrics();
    }
    
#ifdef MAME_DEBUG
    {
        win32_debug = TRUE;
    }
#endif

    if (win32_debug)
    {
        int     cFile;
        FILE*   pFILE;
        
        AllocConsole();

        cFile = _open_osfhandle((long)GetStdHandle(STD_OUTPUT_HANDLE), _O_TEXT);
        pFILE = _fdopen(cFile, "w");
        *stdout = *pFILE;

        cFile = _open_osfhandle((long)GetStdHandle(STD_ERROR_HANDLE), _O_TEXT);
        pFILE = _fdopen(cFile, "w");
        *stderr = *pFILE;

#if 0
        cFile = _open_osfhandle((long)GetStdHandle(STD_INPUT_HANDLE), _O_TEXT);
        pFILE = _fdopen(cFile, "r");
        *stdin = *pFILE;
#endif
        setvbuf(stdout, NULL, _IONBF, 0);
        setvbuf(stderr, NULL, _IONBF, 0);
    }

    hAccel = LoadAccelerators(hInstance, MAKEINTRESOURCE(IDA_TAB_KEYS));

    InitZip(hMain);

    if (joygui != NULL)
    {
        /* hack to get joystick to start */
        options_type o;
        o.use_joystick = TRUE;
        if (joygui->init(&o) != 0)
            joygui = NULL;
        else
            SetTimer(hMain,0,25,NULL);
    }

    return TRUE;
}

/* Moved here because it is called in several places */
void InitializeMisc(HWND hDlg)
{
    Button_Enable(GetDlgItem(hDlg, IDC_JOYSTICK), Joystick.Available(1));
    Button_Enable(GetDlgItem(hDlg, IDC_DIJOYSTICK), DIJoystick.Available(1));
    Button_Enable(GetDlgItem(hDlg, IDC_DIKEYBOARD), DIKeyboard_Available());

    Edit_LimitText(GetDlgItem(hDlg, IDC_SKIP_LINES), 4);
    SendDlgItemMessage(hDlg, IDC_SKIP_LINES_SPIN, UDM_SETRANGE, 0,
                        (LPARAM)MAKELONG(9999, 0));
    SendDlgItemMessage(hDlg, IDC_SKIP_LINES_SPIN, UDM_SETPOS, 0,
                        (LPARAM)MAKELONG(0, 0));

    Edit_LimitText(GetDlgItem(hDlg, IDC_SKIP_COLUMNS), 4);
    SendDlgItemMessage(hDlg, IDC_SKIP_COLUMNS_SPIN, UDM_SETRANGE, 0,
                        (LPARAM)MAKELONG(9999, 0));
    SendDlgItemMessage(hDlg, IDC_SKIP_COLUMNS_SPIN, UDM_SETPOS, 0,
                        (LPARAM)MAKELONG(0, 0));

    SendMessage(GetDlgItem(hDlg, IDC_GAMMA), TBM_SETRANGE,
                (WPARAM)FALSE,
                (LPARAM)MAKELONG(0, 30)); /* 0.50 - 2.00 in .05 increments */

    SendMessage(GetDlgItem(hDlg, IDC_BRIGHTNESS), TBM_SETRANGE,
                (WPARAM)FALSE,
                (LPARAM)MAKELONG(0, 100));

    SendMessage(GetDlgItem(hDlg, IDC_FLICKER), TBM_SETRANGE,
                (WPARAM)FALSE,
                (LPARAM)MAKELONG(0, 100));

    SendMessage(GetDlgItem(hDlg, IDC_BEAM), TBM_SETRANGE,
                (WPARAM)FALSE,
                (LPARAM)MAKELONG(0, 300)); /* 1.00 - 16.00 in .05 increments */

}

static void Win32UI_exit()
{
    ExitZip();

    if (joygui != NULL)
        joygui->exit();

    DestroyAcceleratorTable(hAccel);

    if (game_data != NULL)
        free(game_data);
    if (icon_index != NULL)
        free(icon_index);

    DirectInputClose();
    DirectDrawClose();

    SetSavedFolderID(GetCurrentFolderID());

    FreeFolders();

    SetViewMode(current_view_id - ID_VIEW_ICON);

    if (hBitmap)
        DeleteObject(hBitmap);

    if (hPALbg)
        DeleteObject(hPALbg);

    FreeScreenShot();

    OptionsExit();

    File.exit();

    WinHelp(hMain, "mame32.hlp", HELP_QUIT, 0);
}

static long WINAPI MameWindowProc(HWND hWnd,UINT message,UINT wParam,LONG lParam)
{
    static int  nTab;
    MINMAXINFO  *mminfo;
    int         i;

    switch (message)
    {
    case WM_INITDIALOG:
        /* Initialize info for resizing subitems */
        GetClientRect(hWnd, &main_resize.rect);
        return TRUE;

    case WM_SETTINGCHANGE:
        AdjustMetrics();
        return 0;

    case WM_SIZE:
        OnSize(hWnd, wParam, LOWORD(lParam), HIWORD(wParam));
        UpdateScreenShot();
        return TRUE;

    case WM_MENUSELECT:
        return Statusbar_MenuSelect(hWnd, wParam, lParam);

    case MM_PLAY_GAME:
        MamePlayGame();
        return TRUE;

    case WM_INITMENUPOPUP:
        UpdateMenu(GetDlgItem(hWnd, IDC_LIST), GetMenu(hWnd));
        break;
        
    case WM_CONTEXTMENU:
        if(HandleContextMenu(hWnd, wParam, lParam))
            return FALSE;
        break;

    case WM_COMMAND:
        return MameCommand(hWnd,(int)(LOWORD(wParam)),(HWND)(lParam),(UINT)HIWORD(wParam));

    case WM_GETMINMAXINFO:
        /* Don't let the window get too small; it can break resizing */
        mminfo = (MINMAXINFO *) lParam;
        mminfo->ptMinTrackSize.x = MIN_WIDTH;
        mminfo->ptMinTrackSize.y = MIN_HEIGHT;
        return 0;
      
    case WM_TIMER:
        PollGUIJoystick();
        return TRUE;

    case WM_CLOSE:
        {
            /* save current item */
            RECT rect;
            AREA area;
            int widths[COLUMN_MAX];
            int order[COLUMN_MAX];
            int shown[COLUMN_MAX];
            int tmpOrder[COLUMN_MAX];
            int nItem;
            int nColumnMax = 0;

            /* Restore the window before we attempt to save parameters,
             * This fixed the lost window on startup problem.
             */
            ShowWindow(hWnd, SW_RESTORE);

            GetColumnOrder(order);
            GetColumnShown(shown);
            GetColumnWidths(widths);

            nColumnMax = GetNumColumns(hwndList);

            if (oldControl)            
            {
                for (i = 0; i < nColumnMax; i++)
                {
                    widths[realColumn[i]] = ListView_GetColumnWidth(hwndList, i);
                }
            }
            else
            {
                /* Get the Column Order and save it */
                ListView_GetColumnOrderArray(hwndList, nColumnMax, tmpOrder);

                for (i = 0; i < nColumnMax; i++)
                {
                    widths[realColumn[i]] = ListView_GetColumnWidth(hwndList, i);
                    order[i] = realColumn[tmpOrder[i]];
                }
            }
            
            SetColumnWidths(widths);
            SetColumnOrder(order);

            SetSplitterPos(SPLITTER_LEFT, nSplitterOffset[0]);
            SetSplitterPos(SPLITTER_RIGHT, nSplitterOffset[1]);

            GetWindowRect(hWnd, &rect);
            area.x = rect.left;
            area.y = rect.top;
            area.width  = rect.right - rect.left;
            area.height = rect.bottom - rect.top;
            SetWindowArea(&area);

            /* Save the users current game options and default game */
            nItem = GetSelectedPickItem();
            SetDefaultGame(drivers[nItem]->description);

            PostQuitMessage(0);
            break;
        }

    case WM_LBUTTONDOWN:
        OnLButtonDown(hWnd, (UINT)wParam, MAKEPOINTS(lParam));
        break;

    case WM_MOUSEMOVE:
        OnMouseMove(hWnd, (UINT)wParam, MAKEPOINTS(lParam));
        break;

    case WM_LBUTTONUP:
        OnLButtonUp(hWnd, (UINT)wParam, MAKEPOINTS(lParam));
        break;

    case WM_NOTIFY:
        // Where is this message intended to go
        {
            LPNMHDR lpNmHdr = (LPNMHDR)lParam;
            
            // Fetch tooltip text
            if (lpNmHdr->code == TTN_NEEDTEXT)
            {
                LPTOOLTIPTEXT lpttt = (LPTOOLTIPTEXT)lParam;
                CopyToolTipText( lpttt );
            }
            if (lpNmHdr->hwndFrom == hwndList)
                return MamePickerNotify( lpNmHdr );
            if (lpNmHdr->hwndFrom == hTreeView)
                return TreeViewNotify( lpNmHdr );
        }
        break;

    case WM_DRAWITEM:
        {
            LPDRAWITEMSTRUCT lpDis = (LPDRAWITEMSTRUCT)lParam;
            switch(lpDis->CtlID)
            {
            case IDC_LIST:
                DrawItem((LPDRAWITEMSTRUCT)lParam);
                break;
            }
        }
        break;

    case WM_PAINT:
        if (GetShowScreenShot())
        { 
            PAINTSTRUCT ps;
            RECT        rect;
            RECT        fRect;
            POINT       p = {0, 0};

            BeginPaint(hWnd, &ps);

            ClientToScreen(hWnd, &p);
            GetWindowRect(GetDlgItem(hWnd, IDC_SSFRAME), &fRect);
            OffsetRect(&fRect, -p.x, -p.y);
            
            if (GetScreenShotRect(GetDlgItem(hWnd, IDC_SSFRAME), &rect))
            {
                DWORD dwStyle;
                DWORD dwStyleEx;

                dwStyle   = GetWindowLong(GetDlgItem(hWnd, IDC_SSPICTURE), GWL_STYLE);
                dwStyleEx = GetWindowLong(GetDlgItem(hWnd, IDC_SSPICTURE), GWL_EXSTYLE);

                AdjustWindowRectEx(&rect, dwStyle, FALSE, dwStyleEx);
                MoveWindow(GetDlgItem(hWnd, IDC_SSPICTURE),
                    fRect.left  + rect.left,
                    fRect.top   + rect.top,
                    rect.right  - rect.left,
                    rect.bottom - rect.top,
                    TRUE);
            }

            ShowWindow(GetDlgItem(hWnd, IDC_SSFRAME),   SW_SHOW);
            ShowWindow(GetDlgItem(hWnd, IDC_SSDEFPIC),  SW_HIDE);
            ShowWindow(GetDlgItem(hWnd, IDC_SSNONE),    SW_HIDE);
            ShowWindow(GetDlgItem(hWnd, IDC_SSPICTURE), SW_SHOW);

			/* Paint Screenshot Frame rect with the background image */
			PaintControl(GetDlgItem(hWnd, IDC_SSFRAME), screenShotAvailable);

            /* If the screen shot is not available, show a message */
            if ((DrawScreenShot(GetDlgItem(hWnd, IDC_SSPICTURE))) == FALSE)
            {
                ShowWindow(GetDlgItem(hWnd, IDC_SSPICTURE), SW_HIDE);
				if (hBitmap == NULL ||  screenShotAvailable == FALSE)
					ShowWindow(GetDlgItem(hWnd, IDC_SSFRAME),  SW_SHOW);
                ShowWindow(GetDlgItem(hWnd, IDC_SSDEFPIC),  SW_SHOW);
                ShowWindow(GetDlgItem(hWnd, IDC_SSNONE),    SW_SHOW);
            }
            
            EndPaint(hWnd, &ps);
        }
        break;

    case WM_DESTROY:
        if (hFont != NULL)
        {
            DeleteObject(hFont);
            hFont = NULL;
        }
        PostQuitMessage(0);
        return 0;

    case WM_MOUSEWHEEL:
        {
            // WM_MOUSEWHEEL will be in Win98
            short zDelta = (short)HIWORD(wParam);

            if (zDelta < 0)
                SetSelectedPickItem(GetSelectedPick() + 1);
            else
                SetSelectedPickItem(GetSelectedPick() - 1);
            return 0;
        }
        break;

    default:
        if (message == uiMsh_MsgMouseWheel)
        {
            int zDelta = (int)wParam; /* wheel rotation */
            // WORD xPos = LOWORD(lParam); /* horizontal position of pointer */
            // WORD yPos = HIWORD(lParam); /* vertical position of pointer */ 
            
            if (zDelta < 0)
                SetSelectedPick(GetSelectedPick() + 1);
            else
                SetSelectedPick(GetSelectedPick() - 1);
            return 0;
        }
        break;
    }

    return DefWindowProc(hWnd, message, wParam, lParam);
}

static BOOL PumpMessage()
{
    MSG msg;

    return PumpAndReturnMessage(&msg);
}

static BOOL PumpAndReturnMessage(MSG *pmsg)
{
    if (!(GetMessage(pmsg, NULL, 0, 0)))
        return FALSE;
 
    if (!TranslateAccelerator(hMain, hAccel, pmsg))
    {
        if (!IsDialogMessage(hMain,pmsg))
        {
            TranslateMessage(pmsg);
            DispatchMessage(pmsg);
        }
    }
    return TRUE;
}

static void EmptyQueue()
{
    MSG msg;

    while (PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE))
    {
        if (!PumpMessage())
            ExitProcess(0);
    }
}

static void ColumnSort(int column, BOOL bColumn)
{
    LV_FINDINFO lvfi;
    static int  first_time = TRUE;
    int         use_column = column;

    if (bColumn) // Was a column enum used??
    {
        int i;

        for (i = 0; i < COLUMN_MAX; i++)
        {
            if(realColumn[i] == column)
            {
                use_column = i;
                break;
            }
        }
    }

    /* Sort 'em */
    if (first_time == TRUE)
    {
        ListView_SortItems(hwndList, ListCompareFunc, 0);
        reverse_sort = FALSE;
        if (column != 0 || reverse_sort)
            ListView_SortItems(hwndList, ListCompareFunc, realColumn[use_column]);
        first_time = FALSE;
    }
    else
    {
        reverse_sort = (column == last_sort && !reverse_sort) ? TRUE : FALSE;
        ListView_SortItems(hwndList, ListCompareFunc, realColumn[use_column]);
        last_sort = use_column;
        SetSortColumn(realColumn[use_column] * (reverse_sort ? -1 : 1));
    }

    lvfi.flags = LVFI_PARAM;
    lvfi.lParam = GetSelectedPickItem();
    ListView_EnsureVisible(hwndList, ListView_FindItem(hwndList, -1, &lvfi), FALSE);
    Header_SetSortInfo(hwndList, use_column, !reverse_sort);
}

static void OnIdle()
{
    int                     i;
    LV_FINDINFO             lvfi;
    BOOL                    success;

//    if (!show_all)
//        SetWindowRedraw(hwndList, FALSE);

    if (game_count <= game_index)
    {
        idle_work = FALSE;

        SetWindowRedraw(hwndList, TRUE);

        lvfi.flags = LVFI_STRING;
        lvfi.psz   = GetDefaultGame();
        i = ListView_FindItem(hwndList, -1, &lvfi);
            
        SetSelectedPick((i != -1) ? i : 0);
        InitGames(game_count);
        ResetListView();
        
        return;
    }

    AddIcon(game_index);
        
    if (GetHasRoms(game_index) == UNKNOWN)
    {
        /* Only check for the ZIP file once */
        success = File_ExistZip(drivers[game_index]->name,OSD_FILETYPE_ROM);

        if (!success)
            success = FindRomSet(game_index);

        //game_data[game_index].has_roms = success;
        SetHasRoms(game_index, success);
    }
 
    /* Check for SAMPLES */
    if (GetHasSamples(game_index) == UNKNOWN)
    {
        SetHasSamples(game_index, FindSampleSet(game_index));
    }

    ListView_RedrawItems(hwndList, game_index, game_index);

    game_index++;
}

static void OnSize(HWND hWnd, UINT nState, int nWidth, int nHeight)
{
    static BOOL firstTime = TRUE;

    if (nState != SIZE_MAXIMIZED && nState != SIZE_RESTORED)
        return;

    ResizeWindow(hWnd, &main_resize);
    if (!firstTime)
        OnSizeSplitter(hMain);
    else
        firstTime = FALSE;
    /* Update the splitters structures as appropriate */
    RecalcSplitters();
}

static void ResizeWindow(HWND hParent, Resize *r)
{
    int index = 0, dx, dy;
    HWND hControl;
    RECT parent_rect, rect;
    ResizeItem *ri;
    POINT p = {0, 0};

    if (hParent == NULL)
        return;

    /* Calculate change in width and height of parent window */
    GetClientRect(hParent, &parent_rect);
    dx = parent_rect.right - r->rect.right;
    dy = parent_rect.bottom - r->rect.bottom;
    ClientToScreen(hParent, &p);
   
    while (r->items[index].type != RA_END)
    {
        ri = &r->items[index];
        if (ri->type == RA_ID)
            hControl = GetDlgItem(hParent, ri->u.id);
        else 
            hControl = ri->u.hwnd;
      
        if (hControl == NULL)
        {
            index++;
            continue;
        }

        /* Get control's rectangle relative to parent */
        GetWindowRect(hControl, &rect);
        OffsetRect(&rect, -p.x, -p.y);

        if (!(ri->action & RA_LEFT))
            rect.left += dx;

        if (!(ri->action & RA_TOP))
            rect.top += dy;

        if (ri->action & RA_RIGHT)
            rect.right += dx;

        if (ri->action & RA_BOTTOM)
            rect.bottom += dy;
      
        MoveWindow(hControl, rect.left, rect.top, 
            (rect.right - rect.left), (rect.bottom - rect.top), TRUE);

        /* Take care of subcontrols, if appropriate */
        if (ri->subwindow != NULL)
            ResizeWindow(hControl, ri->subwindow);

        index++;
    }

    /* Record parent window's new location */
    memcpy(&r->rect, &parent_rect, sizeof(RECT));
}

void PaintControl(HWND hWnd, BOOL useBitmap)
{
	RECT        rcClient;
	HRGN        rgnBitmap;
	HPALETTE    hPAL;
	HDC			hDC = GetDC(hWnd);

    /* So we don't paint over the controls border */
    GetClientRect(hWnd, &rcClient);
    InflateRect(&rcClient, -2, -2);

	if (hBitmap && useBitmap)
	{
		int         i, j;
		HDC         htempDC;

		htempDC = CreateCompatibleDC(hDC);	
		SelectObject(htempDC, hBitmap);

		rgnBitmap = CreateRectRgnIndirect(&rcClient);
		SelectClipRgn(hDC, rgnBitmap);
		DeleteObject(rgnBitmap);
				
		hPAL = (! hPALbg) ? CreateHalftonePalette(hDC) : hPALbg;
				
		if(GetDeviceCaps(htempDC, RASTERCAPS) & RC_PALETTE && hPAL != NULL )
		{
			SelectPalette(htempDC, hPAL, FALSE );
			RealizePalette(htempDC);
		}
				
		for( i = rcClient.left; i < rcClient.right; i += bmDesc.bmWidth )
			for( j = rcClient.top; j < rcClient.bottom; j += bmDesc.bmHeight )
				BitBlt(hDC, i, j, bmDesc.bmWidth, bmDesc.bmHeight, htempDC, 0, 0, SRCCOPY );
			
		DeleteDC(htempDC);
		
		if (! bmDesc.bmColors)
		{
			DeleteObject(hPAL);
			hPAL = 0;
		}
	}
	else	/* Use standard control face color */
	{
		HBRUSH	hBrush = CreateSolidBrush(GetSysColor(COLOR_BTNFACE));

		SelectObject(hDC, hBrush);
		FillRect(hDC, &rcClient, hBrush);
		DeleteObject(hBrush);
	}
	ReleaseDC(hWnd, hDC);
}

char* TriStateToText(int nState)
{
    if (nState == TRUE)
        return "Yes";

    if (nState == FALSE)
        return "No";

    return "?";
}

LPCSTR GetCloneParent(int nItem)
{
    if (drivers[nItem]->clone_of != NULL && game_data[nItem].in_list)
        return drivers[nItem]->clone_of->description;
    return "";
}

static BOOL MamePickerNotify(NMHDR *nm)
{
    NM_LISTVIEW *pnmv;
    static int nLastState = -1;

    switch (nm->code)
    {
    case NM_DBLCLK :
        MamePlayGame();
        return TRUE;

    case LVN_GETDISPINFO:
        {
            LV_DISPINFO* pnmv = (LV_DISPINFO*)nm;
            int nItem = pnmv->item.lParam;

            if (pnmv->item.mask & LVIF_IMAGE)
            {
               pnmv->item.iImage = WhichIcon(nItem);
            }

            if (pnmv->item.mask & LVIF_STATE)
                pnmv->item.state = 0;

            if (pnmv->item.mask & LVIF_TEXT)
            {
                switch (realColumn[pnmv->item.iSubItem])
                {
                case COLUMN_GAMES:
                    // Driver description
                    pnmv->item.pszText = (char *)drivers[nItem]->description;
                    break;

                case COLUMN_ROMS:
                    // Has Roms
                    pnmv->item.pszText = 
                        TriStateToText(GetHasRoms(nItem));
                    break;

                case COLUMN_SAMPLES:
                    // Samples
                    if (drivers[nItem]->samplenames == 0 ||
                        drivers[nItem]->samplenames[0] == '\0')
                    {
                        pnmv->item.pszText = "";
                    }
                    else
                    {
                        pnmv->item.pszText = 
                            TriStateToText(GetHasSamples(nItem));
                    }
                    break;

                case COLUMN_DIRECTORY:
                    // Driver name (directory)
                    pnmv->item.pszText = (char*)drivers[nItem]->name;
                    break;

                case COLUMN_TYPE:
                    // Vector/Raster
                    if (drivers[nItem]->drv->video_attributes & VIDEO_TYPE_VECTOR)
                        pnmv->item.pszText = "Vector";
                    else
                        pnmv->item.pszText = "Raster";
                    break;

                case COLUMN_TRACKBALL:
                    // Trackball
                    if (GameUsesTrackball(nItem))
                        pnmv->item.pszText = "Yes";
                    else
                        pnmv->item.pszText = "No";
                    break;

                case COLUMN_PLAYED:
                    // times played
                    {
                        static char buf[100];
                        sprintf(buf,"%i",GetPlayCount(nItem));
                        pnmv->item.pszText = buf;
                    }
                    break;

                case COLUMN_MANUFACTURER:
                    // Manufacturer
                    pnmv->item.pszText = (char *)drivers[nItem]->manufacturer;
                    break;

                case COLUMN_YEAR:
                    // Year
                    pnmv->item.pszText = (char *)drivers[nItem]->year;
                    break;

                case COLUMN_CLONE:
                    pnmv->item.pszText = (char *)GetCloneParent(nItem);
                    break;

                }
            }
        }
        return TRUE;

    case LVN_ITEMCHANGED :
        pnmv = (NM_LISTVIEW *)nm;

        if ((pnmv->uOldState & LVIS_SELECTED) 
            && !(pnmv->uNewState & LVIS_SELECTED))
        {
            /* leaving item */
            /* printf("leaving %s\n",drivers[pnmv->lParam]->name); */
        }

        if (!(pnmv->uOldState & LVIS_SELECTED) 
            && (pnmv->uNewState & LVIS_SELECTED))
        {
            char buf[200];
            int  nState;
            MENUITEMINFO mmi;

            /* We load the screen shot DIB here, instead
             * of loading everytime we want to display it.
             */
            nState = LoadScreenShot(pnmv->lParam);
			screenShotAvailable = nState;
            if (nState == TRUE || nState != nLastState)
            {
                HWND hWnd;

                if (GetShowScreenShot() &&
                    (hWnd = GetDlgItem(hPicker, IDC_SSFRAME)))
                {
                    RECT    rect;
                    HWND    hParent;
                    POINT   p = {0, 0};

                    hParent = GetParent(hWnd);

                    GetWindowRect(hWnd,&rect);
                    ClientToScreen(hParent, &p);
                    OffsetRect(&rect, -p.x, -p.y);
                    InvalidateRect(hParent, &rect, FALSE);
                    UpdateWindow(hParent);
                    nLastState = nState;
                }
            }

            /* printf("entering %s\n",drivers[pnmv->lParam]->name); */

            sprintf(buf,"&Play %s",
                ConvertAmpersandString(drivers[pnmv->lParam]->description));
            SendMessage(hStatusBar, SB_SETTEXT, 0, (LPARAM)drivers[pnmv->lParam]->description);

            mmi.cbSize = sizeof(mmi);
            mmi.fMask = MIIM_TYPE;
            mmi.fType = MFT_STRING;
            mmi.dwTypeData = buf;
            mmi.cch = strlen(mmi.dwTypeData);
            SetMenuItemInfo(GetMenu(hMain),ID_FILE_PLAY,FALSE,&mmi);
        }
        return TRUE;

    case LVN_COLUMNCLICK :
        {
            pnmv = (NM_LISTVIEW *)nm;
            ColumnSort(pnmv->iSubItem, FALSE);
        }
        return TRUE;
    }
    return FALSE;
}

static BOOL TreeViewNotify(LPNMHDR nm)
{
    LPNMTREEVIEW lpNmtv;

    switch (nm->code)
    {
    case TVN_SELCHANGED :
        lpNmtv = (LPNMTREEVIEW)nm;
        {
            HTREEITEM hti = TreeView_GetSelection(hTreeView);
            TVITEM  tvi;

            tvi.mask = TVIF_PARAM | TVIF_HANDLE;
            tvi.hItem = hti;

            if (TreeView_GetItem( hTreeView, &tvi ))
            {
                SetCurrentFolder((LPTREEFOLDER)tvi.lParam);
                ResetListView();
            }
            return TRUE;
        }
        break;
    }

    return FALSE;
}

static BOOL HeaderOnContextMenu(HWND hWnd, WPARAM wParam, LPARAM lParam) 
{
    HMENU hMenuLoad;
    HMENU hMenu;
    // Right button was clicked on header
    POINT   pt;
    RECT    rcCol;
    int     index;
    int     i;
    BOOL    found = FALSE;
    HWND    hwndHeader;

    if ((HWND)wParam != GetDlgItem(hWnd, IDC_LIST))
        return FALSE;

    hwndHeader = GetDlgItem(hwndList, 0);

    pt.x = LOWORD(lParam);
    pt.y = HIWORD(lParam);

    ScreenToClient(hwndHeader, &pt);

    // Determine the column index
    for( i = 0; Header_GetItemRect(hwndHeader, i, &rcCol); i++ )
    {
        if( PtInRect(&rcCol, pt ) )
        {
            index = i;
            found = TRUE;
            break;
        }
    }

    // Do the context menu
    if (found)
    {
        hMenuLoad = LoadMenu(hInst, MAKEINTRESOURCE(IDR_CONTEXT_HEADER));
        hMenu = GetSubMenu(hMenuLoad, 0);
        lastColumnClick = i;
        TrackPopupMenu(hMenu,
            TPM_LEFTALIGN | TPM_RIGHTBUTTON,
            LOWORD(lParam),
            HIWORD(lParam),
            0,
            hWnd,
            NULL);
    
        DestroyMenu(hMenuLoad);
    
        return TRUE;
    }
    return FALSE;
}


static char * ConvertAmpersandString(const char *s)
{
    /* takes a string and changes any ampersands to double ampersands,
       for setting text of window controls that don't allow us to disable
       the ampersand underlining.
     */
    /* returns a static buffer--use before calling again */

    static char buf[200];
    char *ptr;

    ptr = buf;
    while (*s)
    {
        if (*s == '&')
            *ptr++ = *s;
            *ptr++ = *s++;
    }
    *ptr = 0;

    return buf;
}

static void PollGUIJoystick()
{
    if (in_emulation)
        return;

    joygui->poll_joystick();

    if (joygui->joy_pressed(OSD_JOY_LEFT) || joygui->joy_pressed(OSD_JOY_UP))
    {
        SetSelectedPick(GetSelectedPick() - 1);

        SetFocus(hwndList);
    }
    if (joygui->joy_pressed(OSD_JOY_RIGHT) || joygui->joy_pressed(OSD_JOY_DOWN))
    {
        SetSelectedPick(GetSelectedPick() + 1);
       
        SetFocus(hwndList);
    }
    if (joygui->joy_pressed(OSD_JOY_FIRE))
    {
        SetFocus(hwndList);
        MamePlayGame();
    }
}

static void SetView(int menu_id,int listview_style)
{
    /* first uncheck previous menu item */
    CheckMenuRadioItem(GetMenu(hMain), ID_VIEW_ICON, ID_VIEW_REPORT,
        menu_id, MF_CHECKED);
    ToolBar_CheckButton(hToolBar, menu_id, MF_CHECKED);
    SetWindowLong(hwndList,GWL_STYLE,(GetWindowLong(hwndList,GWL_STYLE) & ~LVS_TYPEMASK)
                  | listview_style);
    
    current_view_id = menu_id;
    SetViewMode(menu_id - ID_VIEW_ICON);

    switch(menu_id)
    {
    case ID_VIEW_ICON:
        ListView_SetIconSpacing(hwndList, GetSystemMetrics(SM_CXICONSPACING),
            GetSystemMetrics(SM_CYICONSPACING));
        ResetListView();
        break;
    }
}

static void SetLargeScreenShotCheck(void)
{
    CheckMenuItem(GetMenu(hMain), ID_LARGE_SCREEN_SHOT,
        GetLargeScreenShot() ? MF_CHECKED : MF_UNCHECKED);
}

static void ResetListView()
{
    int     i;
    int     current_game;
    LV_ITEM lvi;

    LPTREEFOLDER lpFolder = GetCurrentFolder();
    if (! lpFolder)
        return;

    //SetWindowRedraw(hwndList,FALSE);

    current_game = GetSelectedPickItem();

    ListView_DeleteAllItems(hwndList);

    ListView_SetItemCount(hwndList, game_count);

    lvi.mask = LVIF_TEXT | LVIF_IMAGE | LVIF_PARAM; 
    lvi.stateMask = 0;
  
    i = -1;

    do
    {
        int firstTime = TRUE;
        /* Add the games that are in this folder */
        if ((i = FindGame(lpFolder, i + 1)) != -1)
        {
            if (firstTime)
            {
                firstListItem = i;
                firstTime = FALSE;
            }
            lvi.iItem    = i;
            lvi.iSubItem = 0; 
            lvi.lParam   = i;
            // Do not set listview to LVS_SORTASCENDING or LVS_SORTDESCENDING
            lvi.pszText  = LPSTR_TEXTCALLBACK;
            lvi.iImage   = I_IMAGECALLBACK;
            ListView_InsertItem(hwndList,&lvi);
        }
    } while (i != -1);


    SetStandardTitleBar();

    if (!idle_work)
    {
        int sorted;
        int column;

        sorted = (GetSortColumn() < 0);

        do 
        {
            column = abs(GetSortColumn());
            ColumnSort(column, TRUE);
        } while (sorted != reverse_sort);
   
        SetWindowRedraw(hwndList,TRUE);
    }
    SetSelectedPickItem(current_game);
}


static void UpdateGameList()
{
    int i;

    for (i = 0; i < game_count; i++)
    {
        SetHasRoms(i, UNKNOWN);
        SetHasSamples(i, UNKNOWN); 
    }

    idle_work  = TRUE;
    game_index = 0;

    /* Let REFRESH also load new background if found */
    ReloadIcons(hwndList);
    LoadListBitmap();
    ResetListView();

    SetDefaultGame(drivers[GetSelectedPickItem()]->description);
    SetSelectedPick(0); /* To avoid flickering. */
}

static void SetStandardTitleBar()
{
    int             games_shown = 0;
    char            window_title[80];
    int             i;
    LPTREEFOLDER    lpFolder = GetCurrentFolder();

    if (! lpFolder)
        return;

    i = -1;
    do
    {
        if ((i = FindGame(lpFolder, i + 1)) != -1)
        {
            games_shown++;
        }
    } while (i != -1);


    /* Show number of games in the current 'View' on the title bar */
    sprintf(window_title,"MAME32 (%d games)", games_shown);
    if (MAME32App.m_bMMXDetected) 
       strcat(window_title," MMX");

    SetWindowText(hMain, window_title);
}


BOOL GameUsesTrackball(int game)
{
    BOOL trackball = FALSE;
    int port;

    /* new trackball support */
    if (drivers[game]->input_ports != 0)
    {
        port = 0;
        while (drivers[game]->input_ports[port].type != IPT_END)
        {
            int type = drivers[game]->input_ports[port].type & ~IPF_MASK;
            if (type == IPT_DIAL
            ||  type == IPT_PADDLE
            ||  type == IPT_TRACKBALL_X
            ||  type == IPT_TRACKBALL_Y
            ||  type == IPT_AD_STICK_X
            ||  type == IPT_AD_STICK_Y)
            {
                return TRUE;
            }
            port++;
        }
        return FALSE;
    }
    return FALSE;
}

static void PickFont(void)
{
    LOGFONT font;
    CHOOSEFONT cf;

    GetListFont(&font);
    cf.lStructSize = sizeof(CHOOSEFONT);
    cf.hwndOwner = hMain;
    cf.lpLogFont = &font;
    cf.rgbColors = GetListFontColor();
    cf.Flags = CF_SCREENFONTS | CF_INITTOLOGFONTSTRUCT | CF_EFFECTS;
    if (!ChooseFont(&cf))
      return;

    SetListFont(&font);
    if (hFont != NULL)
        DeleteObject(hFont);
    hFont = CreateFontIndirect(&font);
    if (hFont != NULL)
    {
        SetWindowFont(hwndList, hFont, TRUE);
        SetWindowFont(hTreeView, hFont, TRUE);
        ListView_SetTextColor(hwndList, cf.rgbColors);
        TreeView_SetTextColor(hTreeView,cf.rgbColors);
        SetListFontColor(cf.rgbColors);
        ResetListView();
    }
}


static HWND hApply = 0;
static HWND hOptsParent = 0;

static LRESULT CALLBACK OptionsSubDialogProc(HWND hDlg, UINT Msg, WPARAM wParam, LPARAM lParam)
{
    switch (Msg)
    {
    case WM_INITDIALOG:
        /* hide the "use default options" checkbox */
        ShowWindow(GetDlgItem(hDlg,IDC_USE_DEFAULT), SW_HIDE);
        return 0;

    case WM_HELP:
        /* User clicked the ? from the upper right on a control */
        WinHelp(((LPHELPINFO) lParam)->hItemHandle, "mame32.hlp", HELP_WM_HELP, GetHelpIDs());
        break;

    case WM_CONTEXTMENU: 
        WinHelp((HWND) wParam, "mame32.hlp", HELP_CONTEXTMENU, GetHelpIDs());
        break; 

    case WM_HSCROLL:
        /* slider changed */
        HANDLE_WM_HSCROLL(hDlg, wParam, lParam, OptOnHScroll);
        EnableWindow(hApply, TRUE);
        break;

    case WM_COMMAND:
        {
            /* Below, 'changed' is used to signify the 'Apply'
             * button should be enabled.
             */
            WORD wID         = GET_WM_COMMAND_ID(wParam, lParam);
            HWND hWndCtrl    = GET_WM_COMMAND_HWND(wParam, lParam);
            WORD wNotifyCode = GET_WM_COMMAND_CMD(wParam, lParam);
            BOOL changed     = FALSE;

            switch (wID)
            {
            case IDC_DEPTH:
                if (wNotifyCode == LBN_SELCHANGE)
                {
                    DepthSelectionChange(hDlg, hWndCtrl);
                    changed = TRUE;
                }
                break;
            case IDC_DISPLAYTYPE:
            case IDC_SIZES:
            case IDC_FRAMESKIP:
            case IDC_ROTATE:
            case IDC_SOUNDTYPE:
            case IDC_AUDIODEV:
            case IDC_SAMPLERATE:
            case IDC_SAMPLEBITS:
                if (wNotifyCode == CBN_SELCHANGE)
                    changed = TRUE;
                break;

            case IDC_SKIP_LINES:
            case IDC_SKIP_COLUMNS:
                if (wNotifyCode == EN_UPDATE)
                    changed = TRUE;
                break;

            default:
                if (wNotifyCode == BN_CLICKED)
                {
                    switch (wID)
                    {
                    case IDC_SCANLINES:
                        if (Button_GetCheck(GetDlgItem(hDlg, IDC_SCANLINES)))
                        {
                            Button_SetCheck(GetDlgItem(hDlg, IDC_VSCANLINES), FALSE);
                            Button_SetCheck(GetDlgItem(hDlg, IDC_USEBLIT),    FALSE);
                        }
                        break;
                    case IDC_VSCANLINES:
                        if (Button_GetCheck(GetDlgItem(hDlg, IDC_VSCANLINES)))
                        {
                            Button_SetCheck(GetDlgItem(hDlg, IDC_SCANLINES), FALSE);
                            Button_SetCheck(GetDlgItem(hDlg, IDC_USEBLIT),   FALSE);
                        }
                        break;
                    case IDC_USEBLIT:
                        if (Button_GetCheck(GetDlgItem(hDlg, IDC_USEBLIT)))
                        {
                            Button_SetCheck(GetDlgItem(hDlg, IDC_VSCANLINES),FALSE);
                            Button_SetCheck(GetDlgItem(hDlg, IDC_SCANLINES), FALSE);
                        }
                        break;
                    }
                    changed = TRUE;
                }
            }
           
            /* Enable the apply button */
            if (changed == TRUE)
            {
                EnableWindow(hApply, TRUE);
            }
        }
        break;
    }
    return 0;
}

/* This centers a dialog in the TabControl */
static BOOL CenterSubDialog(HWND hParent, HWND hWndCenter)
{
    RECT rect, wRect;
    int wExtra, lExtra;

    GetClientRect(hParent, &rect);
    GetClientRect(hWndCenter, &wRect);

    wExtra = (rect.right - wRect.right);
    lExtra = (rect.bottom - wRect.bottom) - 8;

    wExtra = (wExtra > 0) ? wExtra / 2 : 0;
    lExtra = (lExtra > 0) ? lExtra / 2 : 0;

    // map parent coordinates to child coordinates
    return SetWindowPos(hWndCenter, NULL, wExtra, lExtra, -1, -1,
        SWP_NOSIZE | SWP_NOZORDER | SWP_NOACTIVATE);
}

static LRESULT CALLBACK OptionsDialogProc(HWND hDlg, UINT Msg, WPARAM wParam, LPARAM lParam)
{
    TC_ITEM     tci; 
    char        s[200];
    NMHDR       *nmptr;
    /* These are here so no one else uses them */
    static int  nChanged = FALSE;
    static int  nTab = TAB_DISPLAY;
    static HWND hOpts, hMisc;
    static HWND hSub = 0;
    static HWND hSubTab = 0;
    
    switch (Msg)
    {
    case WM_INITDIALOG:
        /* Store handle to the Apply button */
        hApply = GetDlgItem(hDlg, IDAPPLY);
        hSubTab = GetDlgItem(hDlg, IDC_TAB);

        /* Populate the tab control */
        tci.mask = TCIF_TEXT | TCIF_IMAGE; 
        tci.iImage  = -1; 
        tci.pszText = s;

        tci.pszText = "Display";
        TabCtrl_InsertItem( hSubTab, TAB_DISPLAY, &tci);
    
        tci.pszText = "Sound/Input";
        TabCtrl_InsertItem( hSubTab, TAB_MISC, &tci);
        
        TabCtrl_SetCurSel( hSubTab, nTab);

        /* Create the tab 'pages' */
        /* FIXME This is temporary until ParseCommandLine()
         * No longer needs to access controls directly
         */

        hOpts = hOptsDlog;
        hMisc = hMiscDlog;
        hOptsParent = GetParent(hOpts);
        SetParent(hOpts, hDlg);
        SetParent(hMisc, hDlg);

        hSub = (nTab == TAB_DISPLAY) ? hOpts : hMisc;
        /* Initialize all the tabs */

        NewOptionsToDialog(hOpts, TAB_DISPLAY, GetDefaultOptions());
        NewOptionsToDialog(hMisc, TAB_MISC, GetDefaultOptions());
        TabCtrl_SetCurSel( hSubTab, nTab - TAB_DISPLAY);
        CenterSubDialog(hDlg, hSub);
        Button_Enable(GetDlgItem(hMisc, IDC_DIJOYSTICK), DIJoystick.Available(1));
        Button_Enable(GetDlgItem(hMisc, IDC_DIKEYBOARD), DIKeyboard_Available());
        ShowWindow(hSub, SW_SHOW);
        EnableWindow(hDlg, TRUE);
        EnableWindow(hApply, FALSE);
        nChanged = FALSE;
        return 1;
  
    case WM_KILLFOCUS:
        if (hDlg == GetParent((HWND)wParam))
            SetFocus(hDlg);
        return 0;

    case WM_COMMAND:
        {
            WORD wID         = GET_WM_COMMAND_ID(wParam, lParam);
            HWND hWndCtrl    = GET_WM_COMMAND_HWND(wParam, lParam);
            WORD wNotifyCode = GET_WM_COMMAND_CMD(wParam, lParam);

            switch (wID)
            {
            case IDOK:
                /* Store options and destroy temp dialog windows */
                NewDialogToOptions(hOpts, TAB_DISPLAY, GetDefaultOptions());
                NewDialogToOptions(hMisc, TAB_MISC, GetDefaultOptions());

                SetParent(hOpts, hOptsParent);
                SetParent(hMisc, hOptsParent);
                ShowWindow(hOpts, SW_HIDE);
                ShowWindow(hMisc, SW_HIDE);

                hApply = 0;
                /* Let main know to reset game options dialog */
                EndDialog(hDlg, 1);
                return 1;

            case IDCANCEL:

                SetParent(hOpts, hOptsParent);
                SetParent(hMisc, hOptsParent);
                ShowWindow(hOpts, SW_HIDE);
                ShowWindow(hMisc, SW_HIDE);

                hApply = 0;
                /* Let main window know not to reset game options */
                EndDialog(hDlg, nChanged);
                return 1;

            case IDAPPLY:
                /* Store options and disable 'Apply' button */
                NewDialogToOptions(hOpts, TAB_DISPLAY, GetDefaultOptions());
                NewDialogToOptions(hMisc, TAB_MISC, GetDefaultOptions());
                EnableWindow(hApply, FALSE);
                nChanged = TRUE;
                return 1;
            }
        }
        return FALSE;


    case WM_NOTIFY:
        nmptr = (LPNMHDR) lParam;
        switch (nmptr->code)
        {
        case TCN_SELCHANGE:
            switch (nTab)
            {
            case TAB_DISPLAY:
            case TAB_MISC:
                ShowWindow(hSub, SW_HIDE);
                break;
            }
            nTab = TabCtrl_GetCurSel(hSubTab) + TAB_DISPLAY;
            switch (nTab)
            {
            case TAB_DISPLAY:
            case TAB_MISC:
                hSub = (nTab == TAB_DISPLAY) ? hOpts : hMisc;
            }
            CenterSubDialog(hDlg, hSub);
            ShowWindow(hSub, SW_SHOW);
            TabCtrl_SetCurSel( hSubTab, nTab - TAB_DISPLAY);
            return TRUE;
        }
        break;
    }
    return 0;
}

/* Moved here cause it's called in a few places */
void InitializeOptions(HWND hDlg)
{
    InitializeDepthUI(hDlg);
    InitializeDisplayModeUI(hDlg);
    InitializeSoundUI(hDlg);
    InitializeDisplayTypeUI(hDlg);
    InitializeSkippingUI(hDlg);
    InitializeRotateUI(hDlg);
}

void OptOnHScroll(HWND hwnd, HWND hwndCtl, UINT code, int pos)
{
    if (hwndCtl == GetDlgItem(hwnd, IDC_FLICKER))
    {
        FlickerSelectionChange(hwnd);
    }
    else
    if (hwndCtl == GetDlgItem(hwnd, IDC_GAMMA))
    {
        GammaSelectionChange(hwnd);
    }
    else
    if (hwndCtl == GetDlgItem(hwnd, IDC_BRIGHTNESS))
    {
        BrightnessSelectionChange(hwnd);
    }
    else
    if (hwndCtl == GetDlgItem(hwnd, IDC_BEAM))
    {
        BeamSelectionChange(hwnd);
    }
}

/* Handle changes to the Flicker slider */
static void FlickerSelectionChange(HWND hwnd)
{
    char buf[100];
    int  nValue;

    /* Get the current value of the control */
    nValue = SendMessage(GetDlgItem(hwnd, IDC_FLICKER), TBM_GETPOS, 0, 0);
    
    /* Now set the static flicker display the new value to match */
    sprintf(buf, "%3d", nValue);
    Static_SetText(GetDlgItem(hwnd, IDC_FLICKERDISP), buf);
}

/* Handle changes to the Beam slider */
static void BeamSelectionChange(HWND hwnd)
{
    char buf[100];
    UINT nValue;
    double dBeam;

    /* Get the current value of the control */
    nValue = SendMessage(GetDlgItem(hwnd, IDC_BEAM), TBM_GETPOS, 0, 0);
    
    dBeam = nValue / 20.0 + 1.0;

    /* Set the static display to the new value */
    sprintf(buf, "%03.02f", dBeam);
    Static_SetText(GetDlgItem(hwnd, IDC_BEAMDISP), buf);
}


/* Handle changes to the Gamma slider */
static void GammaSelectionChange(HWND hwnd)
{
    char buf[100];
    UINT nValue;
    double dGamma;

    /* Get the current value of the control */
    nValue = SendMessage(GetDlgItem(hwnd, IDC_GAMMA), TBM_GETPOS, 0, 0);
    
    dGamma = nValue / 20.0 + 0.5;

    /* Set the static display to the new value */
    sprintf(buf, "%03.02f", dGamma);
    Static_SetText(GetDlgItem(hwnd, IDC_GAMMADISP), buf);
}

/* Handle changes to the Brightness slider */
static void BrightnessSelectionChange(HWND hwnd)
{
    char buf[100];
    int  nValue;

    /* Get the current value of the control */
    nValue = SendMessage(GetDlgItem(hwnd, IDC_BRIGHTNESS), TBM_GETPOS, 0, 0);
    
    /* Set the static display to the new value */
    sprintf(buf, "%3d%%", nValue);
    Static_SetText(GetDlgItem(hwnd, IDC_BRIGHTNESSDISP), buf);
}

/* Handle changes to the Color Depth drop down */
void DepthSelectionChange(HWND hWnd, HWND hWndCtrl)
{
    int     nCurSelection;

    nCurSelection = ComboBox_GetCurSel(hWndCtrl);
    if (nCurSelection != CB_ERR)
    {
        int     nDepth;

        nDepth = ComboBox_GetCurSel(hWndCtrl) + 1;
        nDepth *= 8;
        UpdateDisplayModeUI(hWnd, nDepth);
    }
}

static BOOL MameCommand(HWND hwnd,int id, HWND hwndCtl, UINT codeNotify)
{
    switch (id)
    {
    case ID_FILE_PLAY :
        MamePlayGame();
        return TRUE;
    case ID_FILE_PLAY_RECORD :
        MamePlayRecordGame();
        return TRUE;
    case ID_FILE_PLAY_BACK :
        MamePlayBackGame();
        return TRUE;
    case ID_FILE_AUDIT :
        AuditDialog(hMain);
        InitGames(game_count);
        ResetListView();
        SetFocus(hwndList);
        return TRUE;
    case ID_FILE_EXIT :
        PostMessage(hMain,WM_CLOSE,0,0);
        return TRUE;
    case ID_VIEW_ICON :
        SetView(ID_VIEW_ICON,LVS_ICON);
        return TRUE;
    case ID_VIEW_REPORT :
        SetView(ID_VIEW_REPORT,LVS_REPORT);
        return TRUE;
    case ID_VIEW_SMALL_ICON :
        SetView(ID_VIEW_SMALL_ICON,LVS_SMALLICON);
        return TRUE;
    case ID_VIEW_LIST_MENU :
        SetView(ID_VIEW_LIST_MENU,LVS_LIST);
        return TRUE;
    // Arrange Icons submenu
    case ID_VIEW_BYGAME:
        ColumnSort(COLUMN_GAMES, TRUE);
        break;
    case ID_VIEW_BYDIRECTORY:
        ColumnSort(COLUMN_DIRECTORY, TRUE);
        break;
    case ID_VIEW_BYMANUFACTURER:
        ColumnSort(COLUMN_MANUFACTURER, TRUE);
        break;
    case ID_VIEW_BYTIMESPLAYED:
        ColumnSort(COLUMN_PLAYED, TRUE);
        break;
    case ID_VIEW_BYTYPE:
        ColumnSort(COLUMN_TYPE, TRUE);
        break;
    case ID_VIEW_BYYEAR:
        ColumnSort(COLUMN_YEAR, TRUE);
        break;
    case ID_VIEW_FOLDERS:
        bShowTree = !bShowTree;
        SetShowFolderList(bShowTree);
        CheckMenuItem(GetMenu(hMain), ID_VIEW_FOLDERS, (bShowTree) ? MF_CHECKED : MF_UNCHECKED);
        ToolBar_CheckButton(hToolBar, ID_VIEW_FOLDERS, (bShowTree) ? MF_CHECKED : MF_UNCHECKED);
        UpdateScreenShot();
        break;
    case ID_VIEW_TOOLBARS:
        bShowToolBar = !bShowToolBar;
        SetShowToolBar(bShowToolBar);
        CheckMenuItem(GetMenu(hMain), ID_VIEW_TOOLBARS, (bShowToolBar) ? MF_CHECKED : MF_UNCHECKED);
        ToolBar_CheckButton(hToolBar, ID_VIEW_TOOLBARS, (bShowToolBar) ? MF_CHECKED : MF_UNCHECKED);
        ShowWindow(hToolBar, (bShowToolBar) ? SW_SHOW : SW_HIDE);
        ResizePickerControls(hPicker);
        UpdateScreenShot();
        break;

    case ID_VIEW_STATUS:
        bShowStatusBar = !bShowStatusBar;
        SetShowStatusBar(bShowStatusBar);
        CheckMenuItem(GetMenu(hMain), ID_VIEW_STATUS, (bShowStatusBar) ? MF_CHECKED : MF_UNCHECKED);
        ToolBar_CheckButton(hToolBar, ID_VIEW_STATUS, (bShowStatusBar) ? MF_CHECKED : MF_UNCHECKED);
        ShowWindow(hStatusBar, (bShowStatusBar) ? SW_SHOW : SW_HIDE);
        ResizePickerControls(hPicker);
        UpdateScreenShot();
        break;

    // Header Context Menu
    case ID_SORT_ASCENDING:
        last_sort = -1;
        ColumnSort(lastColumnClick, FALSE);
        break;
    case ID_SORT_DESCENDING:
        last_sort = lastColumnClick;
        ColumnSort(lastColumnClick, FALSE);
        break;
    case ID_CUSTOMIZE_FIELDS:
        if (DialogBox(GetModuleHandle(NULL),
            MAKEINTRESOURCE(IDD_COLUMNS), hMain, ColumnDialogProc) == TRUE)
            ResetColumnDisplay(FALSE);
        SetFocus(hwndList);
        return TRUE;

    // View Menu
    case ID_VIEW_LINEUPICONS:
        ResetListView();
        break;
    case ID_GAME_PROPERTIES:
        InitPropertyPage(hInst, hwnd, GetSelectedPickItem());
        break;
    case IDC_SHOW_SCREEN_SHOT:
    case ID_VIEW_SCREEN_SHOT:
        ToggleScreenShot();
        if (current_view_id == ID_VIEW_ICON)
            ResetListView();
        break;
    case ID_UPDATE_GAMELIST:
        UpdateGameList();
        break;
    case ID_LARGE_SCREEN_SHOT:
        SetLargeScreenShot((GetLargeScreenShot()) ? FALSE : TRUE);
        SetLargeScreenShotCheck();
#if 0 /* Do I need to do this? */
        {
            HWND hWnd;

            if (GetShowScreenShot() &&
                (hWnd = GetDlgItem(hPicker, IDC_SSFRAME)))
            {
                RECT    rect;
                HWND    hParent;
                POINT   p = {0, 0};

                hParent = GetParent(hWnd);

                GetWindowRect(hWnd,&rect);
                ClientToScreen(hParent, &p);
                OffsetRect(&rect, -p.x, -p.y);
                InvalidateRect(hParent, &rect, FALSE);
                UpdateWindow(hParent);
            }
        }

#else
        if (GetShowScreenShot())
            UpdateWindow(hwnd);
#endif
        break;
    case ID_OPTIONS_FONT:
        PickFont();
        SetFocus(hwndList);
        return TRUE;

    case ID_OPTIONS_DEFAULTS :
        /* Check the return value to see if changes were applied */
        DialogBox(GetModuleHandle(NULL),
            MAKEINTRESOURCE(IDD_OPTIONS_DIALOG),hMain,OptionsDialogProc);
        SetFocus(hwndList);
        return TRUE;

    case ID_OPTIONS_DIR:
        {
            int  nResult;
            BOOL bUpdateRoms;
            BOOL bUpdateSamples;

            nResult = DialogBox(GetModuleHandle(NULL),
                                MAKEINTRESOURCE(IDD_DIRECTORIES),
                                hMain,
                                DirectoriesDialogProc);
            bUpdateRoms    = ((nResult & DIRDLG_ROMS)    == DIRDLG_ROMS)    ? TRUE : FALSE;
            bUpdateSamples = ((nResult & DIRDLG_SAMPLES) == DIRDLG_SAMPLES) ? TRUE : FALSE;

            /* update file code */
            if (bUpdateRoms == TRUE)
                File_UpdateRomPath(GetRomDirs());

            if (bUpdateSamples == TRUE)
                File_UpdateSamplePath(GetSampleDirs());

            /* update game list */
            if (bUpdateRoms    == TRUE ||  bUpdateSamples == TRUE)
                UpdateGameList();
            SetFocus(hwndList);
        }
        return TRUE;

    case ID_HELP_CONTENTS:
        WinHelp(hMain, "mame32.hlp", HELP_FINDER, 0);
        break;
    case ID_HELP_WHATS_NEW32:
        WinHelp(hMain, "mame32.hlp", HELP_CONTEXT, 100);
        break;
    case ID_HELP_QUICKSTART:
        WinHelp(hMain, "mame32.hlp", HELP_CONTEXT, 101);        
        break;

    case ID_HELP_RELEASE:
        DisplayTextFile(hMain, "Readme.txt"); 
        break;
    case ID_HELP_WHATS_NEW:
        DisplayTextFile(hMain, "Whatsnew.txt");
        break;
    case ID_HELP_CHEATS:
        DisplayTextFile(hMain, "Cheat.doc"); 
        break;
    case ID_HELP_GAMELIST:
        DisplayTextFile(hMain, "Gamelist.txt"); 
        break;

    case ID_HELP_ABOUT :
        DialogBox(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_ABOUT),
            hMain, AboutDialogProc);
        SetFocus(hwndList);
        return TRUE;

    case IDC_PLAY_GAME:
        MamePlayGame();
        break;
    }
#if 0
    switch (codeNotify)
    {
    case BN_CLICKED:
        switch (id)
        {
        case IDC_PLAY:
            MamePlayGame();
            return TRUE;
        }
        break;

    case LBN_SELCHANGE:
        switch (id)
        {
        case IDC_DEPTH:
            DepthSelectionChange(hwnd, hwndCtl);
            break;
        }
        break;
    }
#endif
    SetFocus(hwndList);
    return FALSE;
}

void SetStereoEnabled(HWND hWnd, int index)
{
    BOOL enabled = FALSE;
    HWND hCtrl;
   
    if (hCtrl = GetDlgItem(hWnd, IDC_STEREO))
    {
        if (drivers[index]->drv->sound_attributes & SOUND_SUPPORTS_STEREO)
            enabled = TRUE;

        EnableWindow(hCtrl, enabled);
        if (!enabled)
            Button_SetCheck(hCtrl, FALSE);
    }
}

void SetYM3812Enabled(HWND hWnd, int index)
{
    int i;
    BOOL enabled;
    HWND hCtrl;
   
    if (hCtrl = GetDlgItem(hWnd, IDC_USE_FM_YM3812))
    {
        enabled = FALSE;
        for (i = 0; i < MAX_SOUND; i++)
        {
            if (drivers[index]->drv->sound[i].sound_type == SOUND_YM3812
            ||  drivers[index]->drv->sound[i].sound_type == SOUND_YM3526
            ||  drivers[index]->drv->sound[i].sound_type == SOUND_YM2413)
                enabled = TRUE;
        }
        
        EnableWindow(hCtrl, enabled);
        if (!enabled)
            Button_SetCheck(hCtrl, FALSE);
    }
}

static void LoadListBitmap()
{
    HGLOBAL     hDIBbg;
    char*       pFileName = 0;
    UINT        nResource = 0;

    if (hBitmap)
    {
        DeleteObject(hBitmap);
        hBitmap = 0;
    }

    if (hPALbg)
    {
        DeleteObject(hPALbg);
        hPALbg = 0;
    }

    /* Pick images based on number of colors avaliable. */
    if (GetDepth(hwndList) <= 8)
    {
        pFileName = "bkgnd16";
        //nResource = IDB_BKGROUND16;
    }
    else
    {
        pFileName = "bkground";
        //nResource = IDB_BKGROUND;
    }

    if (LoadDIB(pFileName, &hDIBbg, &hPALbg))
    {        
        HDC hDC = GetDC(hwndList);
        hBitmap = DIBToDDB(hDC, hDIBbg, &bmDesc);
        GlobalFree(hDIBbg);
        ReleaseDC(hwndList, hDC);
    }
#ifdef INTERNAL_BKBITMAP
    else
        hBitmap = LoadBitmapAndPalette( hInst, MAKEINTRESOURCE(nResource), &hPALbg, &bmDesc);
#endif
}

/* Initialize the Picker and List controls */
static void InitPicker()
{
    int order[COLUMN_MAX];
    int shown[COLUMN_MAX];
    int i;

    Header_Initialize(hwndList);

    // Disabled column customize with old Control
    if (oldControl)
    {
        for (i = 0; i < COLUMN_MAX ; i++)
        {
            order[i] = i;
            shown[i] = TRUE;
        }
        SetColumnOrder(order);
        SetColumnShown(shown);
    }

    /* Create IconsList for ListView Control */
    CreateIcons(hwndList);
    GetColumnOrder(realColumn);

    ResetColumnDisplay(TRUE);
}

/* Re/initialize the ListControl Columns */
static void ResetColumnDisplay(BOOL firstime)
{
    LV_FINDINFO lvfi;
    LV_COLUMN   lvc;
    int         i;
    int         nColumn;
    int         widths[COLUMN_MAX];
    int         order[COLUMN_MAX];
    int         shown[COLUMN_MAX];
    int         nGame = GetSelectedPickItem();

    GetColumnWidths(widths);
    GetColumnOrder(order);
    GetColumnShown(shown);

    if (!firstime)
    {
        nColumn = GetNumColumns(hwndList);

        // The first time thru this won't happen, on purpose
        for (i = 0; i < nColumn; i++)
        {
            widths[realColumn[i]] = ListView_GetColumnWidth(hwndList, i);
        }

        SetColumnWidths(widths);

        for (i = nColumn; i > 0; i--)
        {
            ListView_DeleteColumn(hwndList, i - 1);
        }
    }

    ListView_SetExtendedListViewStyle(hwndList,
        LVS_EX_FULLROWSELECT | LVS_EX_HEADERDRAGDROP);

    lvc.mask = LVCF_FMT | LVCF_WIDTH | LVCF_TEXT | LVCF_SUBITEM;

    lvc.fmt = LVCFMT_LEFT; 

    nColumn = 0;
    for (i = 0; i < COLUMN_MAX; i++)
    {
        if (shown[order[i]])
        {
            lvc.mask = LVCF_FMT | LVCF_WIDTH | LVCF_SUBITEM;

            lvc.mask |= LVCF_TEXT;
            lvc.pszText = column_names[order[i]];
            lvc.iSubItem = nColumn;
            lvc.cx = widths[order[i]]; 
            ListView_InsertColumn(hwndList, nColumn, &lvc);
            realColumn[nColumn] = order[i];
            nColumn++;
        }
    }

    // Fill this in so we can still sort on columns NOT shown
    for (i = 0; i < COLUMN_MAX && nColumn < COLUMN_MAX; i++)
    {
        if (!shown[order[i]])
        {
            realColumn[nColumn] = order[i];
            nColumn++;
        }
    }

    ListView_SetTextColor(hwndList, GetListFontColor());

    SetSelectedPickItem(nGame);

    ResetListView();

    //if (show_all)
    {
        lvfi.flags = LVFI_STRING;
        lvfi.psz = GetDefaultGame();
        i = ListView_FindItem(hwndList, -1, &lvfi);
        SetSelectedPick(i);
    }
}

HICON LoadIconFromFile(char *iconname)
{
    HICON       hIcon = 0;
    struct stat file_stat;
    char        tmpStr[MAX_PATH];
    const char* sDirName = GetImgDir();

    sprintf(tmpStr,"icons/%s.ico",iconname);
    if (stat(tmpStr, &file_stat) != 0 ||
        (hIcon = ExtractIcon(hInst, tmpStr, 0)) == 0)
    {
        sprintf(tmpStr,"%s/%s.ico", sDirName, iconname);
        if (stat(tmpStr, &file_stat) == 0)
            hIcon = ExtractIcon(hInst, tmpStr, 0);
    }
    return hIcon;
}

void AddIcon(int index)
{
    HICON hIcon = 0;

    if (icon_index[index] != 1)
        return;

    if ((hIcon = LoadIconFromFile((char *)drivers[index]->name)) == 0)
    {
        if (drivers[index]->clone_of != 0)
            hIcon = LoadIconFromFile((char *)drivers[index]->clone_of->name);
    }

    if (hIcon)
    {
        int nIconPos = ImageList_AddIcon(hSmall, hIcon);
        ImageList_AddIcon(hLarge, hIcon);
        if (nIconPos != -1)
            icon_index[index] = nIconPos;
    }
}

static void ReloadIcons(HWND hWnd)
{
    if ((hSmall = ListView_GetImageList(hWnd, LVSIL_SMALL)) != NULL)
        ImageList_Destroy(hSmall);
    
    if ((hLarge = ListView_GetImageList(hWnd, LVSIL_NORMAL)) != NULL)
        ImageList_Destroy(hLarge);

    hSmall = 0;
    hLarge = 0;

    CreateIcons(hWnd);
}

/* create iconlist and Listview control */
static BOOL CreateIcons(HWND hWnd)
{
    HICON   hIcon;
    INT     i;

    if (!icon_index)
        icon_index = malloc(sizeof(int) * game_count);

    if (icon_index != NULL)
    {
        for (i = 0; i < game_count; i++)
            icon_index[i] = 1;
    }

    hSmall = ImageList_Create (ICONMAP_WIDTH, ICONMAP_HEIGHT,
        ILC_COLORDDB | ILC_MASK, NUM_ICONS, NUM_ICONS + game_count);

    hLarge = ImageList_Create (LG_ICONMAP_WIDTH, LG_ICONMAP_HEIGHT,
        ILC_COLORDDB | ILC_MASK, NUM_ICONS, NUM_ICONS + game_count);

    for (i = IDI_WIN_NOROMS; i <= IDI_WIN_REDX; i++)
    {
        if ((hIcon = LoadIconFromFile(icon_names[i - IDI_WIN_NOROMS])) == 0)
            hIcon = LoadIcon (hInst, MAKEINTRESOURCE(i));

        if ((ImageList_AddIcon (hSmall, hIcon) == -1)
            || (ImageList_AddIcon (hLarge, hIcon) == -1))
            return FALSE;
    }

    // Be sure that all the small icons were added.
    if (ImageList_GetImageCount (hSmall) < NUM_ICONS)
        return FALSE;

    // Be sure that all the large icons were added.
    if (ImageList_GetImageCount (hLarge) < NUM_ICONS)
        return FALSE;
 
    // Associate the image lists with the list view control.
    ListView_SetImageList (hWnd, hSmall, LVSIL_SMALL);
   
    ListView_SetImageList (hWnd, hLarge, LVSIL_NORMAL);
  
    return TRUE;
}


/* Sort subroutine to sort the list control */
static int CALLBACK ListCompareFunc(LPARAM index1, LPARAM index2, int sort_subitem)
{
    int value;
    const char *name1 = NULL;
    const char *name2 = NULL;
    int   nTemp1, nTemp2;

    switch (sort_subitem)
    {
    case COLUMN_GAMES :
        value = stricmp(drivers[index1]->description,drivers[index2]->description);
        break;

    case COLUMN_ROMS :
        nTemp1 = GetHasRoms(index1);
        nTemp2 = GetHasRoms(index2);
        if (nTemp1 == nTemp2)
            return ListCompareFunc(index1, index2, COLUMN_GAMES);

        if (nTemp1 == TRUE || nTemp2 == FALSE)
            value = -1;
        else
            value = 1;
        break;

    case COLUMN_SAMPLES :
        if ((nTemp1 = GetHasSamples(index1)) == TRUE)
            nTemp1++;

        if ((nTemp2 = GetHasSamples(index2)) == TRUE)
            nTemp2++;

        if (drivers[index1]->samplenames == 0 ||
            drivers[index1]->samplenames[0] == '\0')
            nTemp1 -= 1;

        if (drivers[index2]->samplenames == 0 ||
            drivers[index2]->samplenames[0] == '\0')
            nTemp2 -= 1;

        if (nTemp1 == nTemp2)
            return ListCompareFunc(index1, index2, COLUMN_GAMES);

        value = nTemp2 - nTemp1;
        break;

    case COLUMN_DIRECTORY :
        value = stricmp(drivers[index1]->name, drivers[index2]->name);
        break;

    case COLUMN_TYPE :
        if ((drivers[index1]->drv->video_attributes & VIDEO_TYPE_VECTOR) ==
            (drivers[index2]->drv->video_attributes & VIDEO_TYPE_VECTOR))
            return ListCompareFunc(index1, index2, COLUMN_GAMES);
  
        value = (drivers[index1]->drv->video_attributes & VIDEO_TYPE_VECTOR) -
               (drivers[index2]->drv->video_attributes & VIDEO_TYPE_VECTOR);
        break;

    case COLUMN_TRACKBALL :
        if (GameUsesTrackball(index1) == GameUsesTrackball(index2))
            return ListCompareFunc(index1,index2,COLUMN_GAMES);
  
        value = GameUsesTrackball(index1) - GameUsesTrackball(index2);
        break;

    case COLUMN_PLAYED :
       value = GetPlayCount(index1) - GetPlayCount(index2);
       if (value == 0)
          return ListCompareFunc(index1,index2,COLUMN_GAMES);

       break;

    case COLUMN_MANUFACTURER :
        if (stricmp(drivers[index1]->manufacturer,drivers[index2]->manufacturer) == 0)
            return ListCompareFunc(index1,index2,COLUMN_GAMES);

        value = stricmp(drivers[index1]->manufacturer,drivers[index2]->manufacturer);
        break;

    case COLUMN_YEAR :
        if (stricmp(drivers[index1]->year,drivers[index2]->year) == 0)
            return ListCompareFunc(index1,index2,COLUMN_GAMES);

        value = stricmp(drivers[index1]->year,drivers[index2]->year);
        break;

    case COLUMN_CLONE :
        name1 = (drivers[index1]->clone_of != 0 && game_data[index1].in_list) ? drivers[index1]->clone_of->description : NULL;
        name2 = (drivers[index2]->clone_of != 0 && game_data[index2].in_list) ? drivers[index2]->clone_of->description : NULL;
        if (name1 == name2)
            return ListCompareFunc(index1, index2, COLUMN_GAMES);

        if (name2 == NULL)
            value = -1;
        else if (name1 == NULL)
            value = 1;
        else
            value = stricmp(name1, name2);
        break;

    default :
        return ListCompareFunc(index1,index2,COLUMN_GAMES);
    }

    if (reverse_sort && value != 0)
        value = -(value);
    
    return value;
}

/* Adjust possible choices in the Screen Size drop down */
void UpdateDisplayModeUI(HWND hwnd, DWORD dwDepth)
{
    int                     i;
    char                    buf[100];
    struct tDisplayModes*   pDisplayModes;
    int                     nPick;
    int                     nCount = 0;
    int                     nSelection = 0;
    DWORD                   w = 0, h = 0;
    HWND                    hCtrl = GetDlgItem(hwnd, IDC_SIZES);

    if (!hCtrl)
        return;

    /* Find out what is currently selected if anything. */
    nPick = ComboBox_GetCurSel(hCtrl);
    if (nPick != 0 && nPick != CB_ERR)
    {
        ComboBox_GetText(GetDlgItem(hwnd, IDC_SIZES), buf, 100);
        if (sscanf(buf, "%d x %d", &w, &h) != 2)
        {
            w = 0;
            h = 0;
        }
    }

    /* Remove all items in the list. */
    ComboBox_ResetContent(hCtrl);

    ComboBox_AddString(hCtrl, "Auto size");

    pDisplayModes = DDraw_GetDisplayModes(hMain);

    for (i = 0; i < pDisplayModes->m_nNumModes; i++)
    {
        if (pDisplayModes->m_Modes[i].m_dwBPP == dwDepth)
        {
            nCount++;
            sprintf(buf, "%i x %i", pDisplayModes->m_Modes[i].m_dwWidth,
                                    pDisplayModes->m_Modes[i].m_dwHeight);
            ComboBox_AddString(hCtrl, buf);

            if (w == pDisplayModes->m_Modes[i].m_dwWidth
            &&  h == pDisplayModes->m_Modes[i].m_dwHeight)
                nSelection = nCount;
        }
    }

    ComboBox_SetCurSel(hCtrl, nSelection);
}

/* Initialize the Display options to 8 bit (256 color) mode */
static void InitializeDisplayModeUI(HWND hwnd)
{
    UpdateDisplayModeUI(hwnd, 8);
}

/* Initialize the sound options */
static void InitializeSoundUI(HWND hwnd)
{
    int                         i, sound;
    HWND                        hCtrl;
#ifndef NOSEAL
    struct tSealSoundDevices*   pSealSoundDevices;
    pSealSoundDevices = SealSound_GetSoundDevices();
#endif

    sound = SOUND_NONE;

    if (hCtrl = GetDlgItem(hwnd, IDC_AUDIODEV))
    {
        ComboBox_SetExtendedUI(hCtrl, TRUE);
        
#ifndef NOSEAL
        if (pSealSoundDevices->m_nNumDevices == 0)
        {
#endif
            Button_Enable(hCtrl, FALSE);
            sound = SOUND_NONE;
            
#ifndef NOSEAL
        }
        else
        {
            Button_Enable(hCtrl, TRUE);
            sound = SOUND_SEAL;
            
            for (i = 0; i < pSealSoundDevices->m_nNumDevices; i++)
            {
                ComboBox_AddString(hCtrl, pSealSoundDevices->m_Devices[i].m_pName);
            }
            
            ComboBox_SetCurSel(hCtrl, 0);
            
            /* Set DirectSound as default if it exists */
            for (i = 0; i < pSealSoundDevices->m_nNumDevices; i++)
            {
                if (pSealSoundDevices->m_Devices[i].m_wProductID == SEALSOUND_PRODUCT_DSOUND)
                    ComboBox_SetCurSel(hCtrl, i);
            }
        }
#endif
    }

    if (hCtrl = GetDlgItem(hwnd, IDC_SOUNDTYPE))
    {
        ComboBox_AddString(hCtrl, "No Sound");
#ifndef NOSEAL
        ComboBox_AddString(hCtrl, "Seal");
#endif
        
#ifndef NOMIDAS
        ComboBox_AddString(hCtrl, "MIDAS");
#endif
        ComboBox_AddString(hCtrl, "DirectSound");
        
        ComboBox_SetCurSel(GetDlgItem(hwnd, IDC_SOUNDTYPE), sound);
    }

    if (hCtrl = GetDlgItem(hwnd, IDC_SAMPLERATE))
    {
        ComboBox_AddString(hCtrl, "11025");
        ComboBox_AddString(hCtrl, "22050");
        ComboBox_AddString(hCtrl, "44100");
        ComboBox_SetCurSel(hCtrl, 1);
    }
    if (hCtrl = GetDlgItem(hwnd, IDC_SAMPLEBITS))
    {
        ComboBox_AddString(hCtrl, "8");
        ComboBox_AddString(hCtrl, "16");
        ComboBox_SetCurSel(hCtrl, 0);
    }
}

/* Populate the Display type drop down */
static void InitializeDisplayTypeUI(HWND hwnd)
{
    HWND hCtrl = GetDlgItem(hwnd, IDC_DISPLAYTYPE);

    if (hCtrl)
    {
        ComboBox_AddString(hCtrl, "Full Screen");
        ComboBox_AddString(hCtrl, "Window");
    }
}

/* Populate the Frame Skipping drop down */
static void InitializeSkippingUI(HWND hwnd)
{
    HWND hCtrl = GetDlgItem(hwnd, IDC_FRAMESKIP);

    if (hCtrl)
    {
        ComboBox_AddString(hCtrl, "Draw every frame");
        ComboBox_AddString(hCtrl, "Draw every other frame");
        ComboBox_AddString(hCtrl, "Draw every third frame");
        ComboBox_AddString(hCtrl, "Draw every fourth frame");
    }
}

/* Populate the Rotate drop down */
static void InitializeRotateUI(HWND hwnd)
{
    HWND hCtrl = GetDlgItem(hwnd, IDC_ROTATE);

    if (hCtrl)
    {
        ComboBox_AddString(hCtrl, "None");   /* 0 */
        ComboBox_AddString(hCtrl, "Right");  /* 1 */
        ComboBox_AddString(hCtrl, "Left");   /* 2 */
    }
}

/* Populate the Color depth drop down */
static void InitializeDepthUI(HWND hwnd)
{
    HWND hCtrl = GetDlgItem(hwnd, IDC_DEPTH);

    if (hCtrl)
    {
        ComboBox_AddString(hCtrl, "256 Colors");
        ComboBox_AddString(hCtrl, "High Color (16 bit)");
    }
}

static int GetSelectedPick()
{
    // returns index of listview selected item

    // This will return -1 if not found
    return ListView_GetNextItem(hwndList, -1, LVIS_SELECTED | LVIS_FOCUSED);
}

static int GetSelectedPickItem()
{
    // returns lParam (game index) of listview selected item
    LV_ITEM lvi;

    lvi.iItem = GetSelectedPick();
    if (lvi.iItem == -1)
        return 0;

    lvi.iSubItem = 0;
    lvi.mask = LVIF_PARAM;
    ListView_GetItem(hwndList, &lvi);

    return lvi.lParam;
}

static void SetSelectedPick(int new_index)
{
    if (new_index < 0)
        new_index = 0;

    ListView_SetItemState(hwndList, new_index, LVIS_FOCUSED | LVIS_SELECTED,
                          LVIS_FOCUSED | LVIS_SELECTED);
    ListView_EnsureVisible(hwndList, new_index, FALSE);
}

static void SetSelectedPickItem(int val)
{
    int i;
    LV_FINDINFO lvfi;

    if (val < 0)
        return;

    lvfi.flags = LVFI_PARAM;
    lvfi.lParam = val;
    i = ListView_FindItem(hwndList, -1, &lvfi);
    if (i == -1)
    {
        POINT p = {0,};
        lvfi.flags = LVFI_NEARESTXY;
        lvfi.pt = p;
        i = ListView_FindItem(hwndList, -1, &lvfi);
    }
    SetSelectedPick((i == -1) ? 0 : i);
}

/* Return the number of games currently displayed */
int GetNumGames()
{
   return game_count;
}

game_data_type * GetGameData()
{
    return game_data;
}

/* returns if we should immediately start the game */
static BOOL ParseCommandLine(char *command_line)
{
    BOOL help;
    BOOL cpu_detect = TRUE;
    BOOL ignore_badoptions = FALSE;
    int h = 0, w = 0;
    int i,j;
    int argc;
    char *argv[200];
    char buf[500];

    argc = 1;
    argv[argc] = strtok(command_line, " \t\n");

    if (argv[1] != NULL)
    {
        argc++;
        while ((argv[argc] = strtok(NULL, " \t\n")) != NULL)
            argc++;
    }

    help = FALSE;

    for (i = 1; i < argc; i++)
    {
        if (argv[i][0] == '-' && argv[i][1] == 'i')
            if (strcmp(argv[i],"-ignorebadoptions") == 0)
                ignore_badoptions = TRUE;
    }

    for (i = 1; i < argc; i++)
    {
        if (sscanf(argv[i], "-%dx%d", &w, &h) == 2
        ||  stricmp(argv[i], "-resolution") == 0)
        {
            struct tDisplayModes*   pDisplayModes;
            BOOL                    bFound = FALSE;

            if (stricmp(argv[i], "-resolution") == 0)
            {
                if (argc <= ++i)
                    break;

                if (sscanf(argv[i], "%dx%d", &w, &h) != 2)
                    break;
            }

            pDisplayModes = DDraw_GetDisplayModes(hMain);

            /* Verify that such a mode exists, 8 or 16 bit. */
            for (j = 0; j < pDisplayModes->m_nNumModes; j++)
            {
                if (w == (int)pDisplayModes->m_Modes[j].m_dwWidth 
                &&  h == (int)pDisplayModes->m_Modes[j].m_dwHeight)
                {
                    bFound = TRUE;
                    /* Set to FULL_SCREEN */
                    ComboBox_SetCurSel(GetDlgItem(hOptsDlog, IDC_DISPLAYTYPE), 0);
                    break;
                }
            }

            if (bFound == TRUE)
            {
                /* Select the mode in the list. */
                int nSelection = 0;
                int nCount;

                nCount = ComboBox_GetCount(GetDlgItem(hOptsDlog, IDC_SIZES));        
                while (0 < nCount--)
                {
                    char    buf[100];
                    int     nWidth, nHeight;

                    ComboBox_GetLBText(GetDlgItem(hOptsDlog, IDC_SIZES), nCount, buf);

                    if (sscanf(buf, "%d x %d", &nWidth, &nHeight) == 2)
                    {
                        if (nWidth  == w
                        &&  nHeight == h)
                        {
                            nSelection = nCount;
                            break;
                        }
                    }
                }
                ComboBox_SetCurSel(GetDlgItem(hOptsDlog, IDC_SIZES), nSelection);
            }
            else
            {
                sprintf(buf, "Error parsing screen size: %dx%dx not supported", w, h);
                MessageBox(NULL, buf, "MAME32", MB_OK | MB_ICONERROR);
            }
            continue;
        }

        if (stricmp(argv[i], "-gamma") == 0)
        {
            UINT nValue;
            double dGamma;
            char cBuf[40];

            if (argc <= ++i)
                break;

            dGamma = atof(argv[i]);
            if (0.20 < dGamma && dGamma < 2.00)
                continue;

            nValue = (UINT)((dGamma - 0.5) * 20.0);
            SendMessage(GetDlgItem(hOptsDlog, IDC_GAMMA), TBM_SETPOS,
                        (WPARAM)TRUE,
                        (LPARAM)nValue);
            sprintf(cBuf, "%03.02f", dGamma);
            Static_SetText(GetDlgItem(hOptsDlog, IDC_GAMMADISP), cBuf);
            continue;
        }
        if (stricmp(argv[i], "-brightness") == 0)
        {
            int nValue;
            char cBuf[40];

            if (argc <= ++i)
                break;

            nValue = atoi(argv[i]);

            SendMessage(GetDlgItem(hOptsDlog, IDC_BRIGHTNESS), TBM_SETPOS,
                        (WPARAM)TRUE,
                        (LPARAM)nValue);
            sprintf(cBuf, "%3d%%", nValue);
            Static_SetText(GetDlgItem(hOptsDlog, IDC_BRIGHTNESSDISP), cBuf);
            continue;
        }

        if (stricmp(argv[i], "-double") == 0)
        {
            Button_SetCheck(GetDlgItem(hOptsDlog, IDC_DOUBLE), TRUE);
            continue;
        }
        if (stricmp(argv[i], "-nodouble") == 0)
        {
            Button_SetCheck(GetDlgItem(hOptsDlog, IDC_DOUBLE), FALSE);
            continue;
        }

        if (stricmp(argv[i], "-noddraw") == 0)
        {
            Button_SetCheck(GetDlgItem(hOptsDlog, IDC_WINDOW_DDRAW), FALSE);
            continue;
        }

        if (stricmp(argv[i], "-skiplines") == 0)
        {
            int pos;

            if (argc <= ++i)
                break;

            pos = max(0, atoi(argv[i]));
            SendDlgItemMessage(hOptsDlog, IDC_SKIP_LINES_SPIN, UDM_SETPOS, 0,
                                (LPARAM)MAKELONG(pos, 0));
            continue;
        }

        if (stricmp(argv[i], "-skipcolumns") == 0)
        {
            int pos;

            if (argc <= ++i)
                break;

            pos = max(0, atoi(argv[i]));
            SendDlgItemMessage(hOptsDlog, IDC_SKIP_COLUMNS_SPIN, UDM_SETPOS, 0,
                                (LPARAM)MAKELONG(pos, 0));
            continue;
        }

        if (stricmp(argv[i], "-scanlines") == 0)
        {
            Button_SetCheck(GetDlgItem(hOptsDlog, IDC_SCANLINES),  TRUE);
            Button_SetCheck(GetDlgItem(hOptsDlog, IDC_USEBLIT),    FALSE);
            Button_SetCheck(GetDlgItem(hOptsDlog, IDC_VSCANLINES), FALSE);
            continue;
        }
        if (stricmp(argv[i], "-noscanlines") == 0)
        {
            Button_SetCheck(GetDlgItem(hOptsDlog, IDC_SCANLINES), FALSE);
            continue;
        }

        if (stricmp(argv[i], "-vscanlines") == 0)
        {
            Button_SetCheck(GetDlgItem(hOptsDlog, IDC_VSCANLINES), TRUE);
            Button_SetCheck(GetDlgItem(hOptsDlog, IDC_USEBLIT),    FALSE);
            Button_SetCheck(GetDlgItem(hOptsDlog, IDC_SCANLINES),  FALSE);
            continue;
        }

        if (stricmp(argv[i], "-novscanlines") == 0)
        {
            Button_SetCheck(GetDlgItem(hOptsDlog, IDC_VSCANLINES), FALSE);
            continue;
        }

        if (stricmp(argv[i], "-ror") == 0)
        {
            ComboBox_SetCurSel(GetDlgItem(hOptsDlog, IDC_ROTATE), ROTATE_RIGHT);
            continue;
        }
        if (stricmp(argv[i], "-rol") == 0)
        {
            ComboBox_SetCurSel(GetDlgItem(hOptsDlog, IDC_ROTATE), ROTATE_LEFT);
            continue;
        }
        if (stricmp(argv[i], "-flipx") == 0)
        {
            Button_SetCheck(GetDlgItem(hOptsDlog, IDC_FLIPX), TRUE);
            continue;
        }
        if (stricmp(argv[i], "-flipy") == 0)
        {
            Button_SetCheck(GetDlgItem(hOptsDlog, IDC_FLIPY), TRUE);
            continue;
        }
        if (stricmp(argv[i], "-depth") == 0)
        {
            if (argc <= ++i)
                break;

            if (stricmp(argv[i], "8") == 0)
            {
                ComboBox_SetCurSel(GetDlgItem(hOptsDlog, IDC_DEPTH), 0);
                UpdateDisplayModeUI(hOptsDlog, 8);
            }
            else
            if (stricmp(argv[i], "16") == 0)
            {
                ComboBox_SetCurSel(GetDlgItem(hOptsDlog, IDC_DEPTH), 1);
                UpdateDisplayModeUI(hOptsDlog, 16);
            }
            continue;
        }
        if (stricmp(argv[i], "-screen") == 0)
        {
            ComboBox_SetCurSel(GetDlgItem(hOptsDlog, IDC_DISPLAYTYPE), 0);
            continue;
        }
        if (stricmp(argv[i], "-window") == 0)
        {
            ComboBox_SetCurSel(GetDlgItem(hOptsDlog, IDC_DISPLAYTYPE), 1);
            continue;
        }
        if (stricmp(argv[i], "-useblit") == 0)
        {
            Button_SetCheck(GetDlgItem(hOptsDlog, IDC_USEBLIT),    TRUE);
            Button_SetCheck(GetDlgItem(hOptsDlog, IDC_SCANLINES),  FALSE);
            Button_SetCheck(GetDlgItem(hOptsDlog, IDC_VSCANLINES), FALSE);
            continue;
        }

        if (stricmp(argv[i], "-nommx") == 0)
        {
            Button_SetCheck(GetDlgItem(hMiscDlog, IDC_DISABLE_MMX), TRUE);
            continue;
        }

        /* vector */
        if (stricmp(argv[i], "-antialias") == 0)
        {
            Button_SetCheck(GetDlgItem(hOptsDlog, IDC_ANTIALIAS), TRUE);
            continue;
        }
        if (stricmp(argv[i], "-noantialias") == 0)
        {
            Button_SetCheck(GetDlgItem(hOptsDlog, IDC_ANTIALIAS), FALSE);
            continue;
        }
        if (stricmp(argv[i], "-translucency") == 0)
        {
            Button_SetCheck(GetDlgItem(hOptsDlog, IDC_TRANSLUCENCY), TRUE);
            continue;
        }
        if (stricmp(argv[i], "-notranslucency") == 0)
        {
            Button_SetCheck(GetDlgItem(hOptsDlog, IDC_TRANSLUCENCY), FALSE);
            continue;
        }
        if (stricmp(argv[i], "-beam") == 0)
        {
            UINT nValue;
            double dBeam;
            char cBuf[40];

            if (argc <= ++i)
                break;

            dBeam = atof(argv[i]);
            if (1.00 < dBeam && dBeam < 16.00)
                continue;

            nValue = (UINT)((dBeam - 1.0) * 20.0);
            SendMessage(GetDlgItem(hOptsDlog, IDC_BEAM), TBM_SETPOS,
                        (WPARAM)TRUE,
                        (LPARAM)nValue);
            sprintf(cBuf, "%03.02f", dBeam);
            Static_SetText(GetDlgItem(hOptsDlog, IDC_BEAMDISP), cBuf);
            continue;
        }
        if (stricmp(argv[i], "-flicker") == 0)
        {
            int nFlicker;
            char cBuf[40];

            if (argc <= ++i)
                break;

            nFlicker = atoi(argv[i]);

            SendMessage(GetDlgItem(hOptsDlog, IDC_FLICKER), TBM_SETPOS,
                        (WPARAM)TRUE,
                        (LPARAM)nFlicker);
            sprintf(cBuf, "%3d", nFlicker);
            Static_SetText(GetDlgItem(hOptsDlog, IDC_FLICKERDISP), cBuf);
            continue;
        }

        if (stricmp(argv[i], "-frameskip") == 0)
        {
            i++;
            if (i < argc && argv[i][0] != '-')
            {
                int skip = atoi(argv[i]);

                /* the Main mame code allows from 0 to 3 for this value */
                if (skip < 0) skip = 0;
                if (skip > 3) skip = 3;
                ComboBox_SetCurSel(GetDlgItem(hOptsDlog, IDC_FRAMESKIP), skip);
                continue;
            }
            else
            {
                i--;
            }
            if (!ignore_badoptions)
                help = 1;
            continue;
        }

        if (stricmp(argv[i], "-vg") == 0)
        {
            Button_SetCheck(GetDlgItem(hOptsDlog, IDC_VECTOR_DOUBLE), TRUE);
            continue;
        }

        /* Sound */
        if (stricmp(argv[i], "-nosound") == 0)
        {
            ComboBox_SetCurSel(GetDlgItem(hMiscDlog, IDC_SOUNDTYPE), 0);
            continue;
        }
        if (stricmp(argv[i], "-midas") == 0)
        {
            ComboBox_SetCurSel(GetDlgItem(hMiscDlog, IDC_SOUNDTYPE), 2);
            continue;
        }
        if (stricmp(argv[i], "-directsound") == 0)
        {
            ComboBox_SetCurSel(GetDlgItem(hMiscDlog, IDC_SOUNDTYPE), 3);
            continue;
        }

#ifndef NOSEAL
        if (stricmp(argv[i], "-soundcard") == 0)
        {
            struct tSealSoundDevices*   pSealSoundDevices;
            BOOL                        bFound = FALSE;
            int                         soundcard;

            if (argc <= ++i)
                break;
            soundcard = atoi(argv[i]);

            pSealSoundDevices = SealSound_GetSoundDevices();

            for (j = 0; j < pSealSoundDevices->m_nNumDevices; j++)
            {
                if (soundcard == j)
                {
                    bFound = TRUE;
                    ComboBox_SetCurSel(GetDlgItem(hMiscDlog, IDC_SOUNDTYPE), 1);
                    ComboBox_SetCurSel(GetDlgItem(hMiscDlog, IDC_AUDIODEV), j);
                    break;
                }
            }

            if (bFound == FALSE)
            {
                sprintf(buf, "Error parsing -soundcard: %d not supported", soundcard);
                MessageBox(NULL, buf, "MAME32", MB_OK | MB_ICONERROR);
            }
            continue;
        }
#endif

        if (stricmp(argv[i], "-sr") == 0)
        {
            if (argc <= ++i)
                break;

            if (stricmp(argv[i], "11025") == 0)
                ComboBox_SetCurSel(GetDlgItem(hMiscDlog, IDC_SAMPLERATE), 0);
            else
            if (stricmp(argv[i], "22050") == 0)
                ComboBox_SetCurSel(GetDlgItem(hMiscDlog, IDC_SAMPLERATE), 1);
            else
            if (stricmp(argv[i], "44100") == 0)
                ComboBox_SetCurSel(GetDlgItem(hMiscDlog, IDC_SAMPLERATE), 2);

            continue;
        }
        if (stricmp(argv[i], "-sb") == 0)
        {
            if (argc <= ++i)
                break;

            if (stricmp(argv[i], "8") == 0)
                ComboBox_SetCurSel(GetDlgItem(hMiscDlog, IDC_SAMPLEBITS), 0);
            else
            if (stricmp(argv[i], "16") == 0)
                ComboBox_SetCurSel(GetDlgItem(hMiscDlog, IDC_SAMPLEBITS), 1);

            continue;
        }
        if (stricmp(argv[i], "-stereo") == 0)
        {
            Button_SetCheck(GetDlgItem(hMiscDlog, IDC_STEREO), TRUE);
            continue;
        }
        if (stricmp(argv[i], "-nostereo") == 0)
        {
            Button_SetCheck(GetDlgItem(hMiscDlog, IDC_STEREO), FALSE);
            continue;
        }
        if (stricmp(argv[i], "-ym3812opl") == 0)
        {
            Button_SetCheck(GetDlgItem(hMiscDlog, IDC_USE_FM_YM3812), TRUE);
            continue;
        }
        if (stricmp(argv[i], "-noym3812opl") == 0)
        {
            Button_SetCheck(GetDlgItem(hMiscDlog, IDC_USE_FM_YM3812), FALSE);
            continue;
        }
        if (stricmp(argv[i], "-ym2203opl") == 0)
        {
            continue;
        }
        if (stricmp(argv[i], "-noym2203opl") == 0)
        {
            continue;
        }

        /* Input */
        if (stricmp(argv[i], "-mouse") == 0)
        {
            Button_SetCheck(GetDlgItem(hMiscDlog, IDC_AI_MOUSE),    TRUE);
            Button_SetCheck(GetDlgItem(hMiscDlog, IDC_AI_JOYSTICK), FALSE);
            continue;
        }
        if (stricmp(argv[i], "-nomouse") == 0)
        {
            Button_SetCheck(GetDlgItem(hMiscDlog, IDC_AI_MOUSE),    FALSE);
            Button_SetCheck(GetDlgItem(hMiscDlog, IDC_AI_JOYSTICK), TRUE);
            continue;
        }
        if (stricmp(argv[i], "-joy") == 0)
        {
            Button_SetCheck(GetDlgItem(hMiscDlog, IDC_JOYSTICK), TRUE);
            continue;
        }
        if (stricmp(argv[i], "-nojoy") == 0)
        {
            Button_SetCheck(GetDlgItem(hMiscDlog, IDC_JOYSTICK), FALSE);
            continue;
        }
        if (stricmp(argv[i], "-dikbd") == 0)
        {
            if (DIKeyboard_Available())
                Button_SetCheck(GetDlgItem(hMiscDlog, IDC_DIKEYBOARD), TRUE);
            continue;
        }
        if (stricmp(argv[i], "-dijoy") == 0)
        {
            if (DIJoystick.Available(1))
                Button_SetCheck(GetDlgItem(hMiscDlog, IDC_DIJOYSTICK), TRUE);
            continue;
        }

        /* Misc */
        if (stricmp(argv[i], "-cheat") == 0)
        {
            Button_SetCheck(GetDlgItem(hMiscDlog, IDC_CHEAT), TRUE);
            continue;
        }

        if (stricmp(argv[i], "-log") == 0)
        {
            Button_SetCheck(GetDlgItem(hMiscDlog, IDC_LOG), TRUE);
            continue;
        }

        if (stricmp(argv[i], "-profiler") == 0)
        {
            Button_SetCheck(GetDlgItem(hMiscDlog, IDC_PROFILER), TRUE);
            continue;
        }

        if (stricmp(argv[i], "-noautopause") == 0)
        {
            Button_SetCheck(GetDlgItem(hMiscDlog, IDC_AUTO_PAUSE), FALSE);
            continue;
        }

        if (stricmp(argv[i], "-quit") == 0)
        {
            quit = TRUE;
            continue;
        }

        if (stricmp(argv[i], "-joygui") == 0)
        {
            joygui = &Joystick;
            continue;
        }
 
        if (stricmp(argv[i], "-dijoygui") == 0)
        {
            joygui = &DIJoystick;
             continue;
        }

        if (stricmp(argv[i], "-debug") == 0)
        {
            win32_debug = 1;
            continue;
        }

        if (stricmp(argv[i], "-debug2") == 0)
        {
            win32_debug = 2;
            continue;
        }

        if (stricmp(argv[i], "-nocpudetect") == 0)
        {
            cpu_detect = FALSE;
            continue;
        }

        /* Are they asking for help? */
        if (stricmp(argv[i], "-help") == 0 || stricmp(argv[i], "-?") == 0)
        {
            help = 1;
            break;
        }

        /* Let the user decide if they want to contunue
         * if we encounter an Unknown command line option.
         *
         * This allows people who want to use a different
         * front end, to continue to do so.
         *
         * In addition, there is a -ignorebadoptios command
         * line flag.
         */
        if (argv[i][0] == '-' && ignore_badoptions == FALSE)
        {
            char cBuf[200];

            sprintf(cBuf, "Unknown option '%s', continue anyway?", argv[i]);
            if (IDCANCEL == MessageBox(0, cBuf, "Mame32 - Unknown Option",
                MB_YESNO | MB_ICONQUESTION))
            {
                ExitProcess(0);
            }
            else
            {
                ignore_badoptions = TRUE;
            }
            continue;
        }
    }

    if (cpu_detect)
    {
        MAME32App.m_bMMXDetected = MAME32App.DetectMMX();
    }
    if (help)
    {
        DialogBox(GetModuleHandle(NULL), MAKEINTRESOURCE(IDD_HELP), NULL, MameHelpDialogProc);
        ExitProcess(0);
    }

    if (argc >= 2 && argv[1][0] != '-')
    {
        i = 0;
        while (drivers[i])
        {
            if (!stricmp(argv[1], drivers[i]->name))
            {
                LV_FINDINFO lvfi;
                int index;

                lvfi.flags  = LVFI_PARAM;
                lvfi.lParam = i;
                index = ListView_FindItem(hwndList, -1, &lvfi);
                SetSelectedPick(index);
                // Does this fix it?
                return TRUE;
            }
            i++;
        }
        sprintf(buf, "MAME does not support the game '%s'", argv[1]);
        MessageBox(NULL, buf, "MAME32", MB_OK | MB_ICONERROR);
    }

    return FALSE;
}

static HWND hHelp;
static void HelpPrintln(char *s)
{
    HWND hEdit;

    hEdit = GetDlgItem(hHelp, IDC_HELP_TEXT);

    Edit_SetSel(hEdit, Edit_GetTextLength(hEdit), Edit_GetTextLength(hEdit));
    Edit_ReplaceSel(hEdit, s);
    Edit_SetSel(hEdit, Edit_GetTextLength(hEdit), Edit_GetTextLength(hEdit));
    Edit_ReplaceSel(hEdit, "\r\n");
}

static LRESULT CALLBACK AboutDialogProc(HWND hDlg, UINT Msg, WPARAM wParam, LPARAM lParam)
{
    switch (Msg)
    {
    case WM_INITDIALOG:

        {
            char tmp[80];
#if defined(RELEASE_CANDIDATE)
            sprintf(tmp,"Version 0.%d RC%d", VERSION, RELEASE_CANDIDATE);
#elif defined(BETA_VERSION)
            sprintf(tmp,"Version 0.%d BETA %d", VERSION, BETA_VERSION);
#else
            sprintf(tmp,"Version 0.%d", VERSION);
#endif
            Static_SetText(GetDlgItem(hDlg,IDC_VERSION), tmp);
        }
        
        // ShowWindow(GetDlgItem(hDlg,IDC_USE_DEFAULT), FALSE);
        return 1;
  
    case WM_COMMAND :
        EndDialog(hDlg, 0);
        return 1;
    }
    return 0;
}


static LRESULT CALLBACK DirectXDialogProc(HWND hDlg, UINT Msg, WPARAM wParam, LPARAM lParam)
{
   HWND hEdit;

   char *directx_help =
       "MAME32 requires DirectX version 3 or later, which is a set of operating\r\n"
       "system extensions by Microsoft for Windows 95 and Windows NT.\r\n\r\n"
       "Visit Microsoft's DirectX web page at http://www.microsoft.com/directx\r\n"
       "download DirectX, install it, and then run MAME32 again.\r\n";

    switch (Msg)
    {
    case WM_INITDIALOG:
        hEdit = GetDlgItem(hDlg,IDC_DIRECTX_HELP);
        Edit_SetSel(hEdit, Edit_GetTextLength(hEdit), Edit_GetTextLength(hEdit));
        Edit_ReplaceSel(hEdit,directx_help);
        return 1;
  
    case WM_COMMAND :
       if (LOWORD(wParam) == IDB_WEB_PAGE)
          ShellExecute(hMain,NULL,"http://www.microsoft.com/directx",NULL,NULL,SW_SHOWNORMAL);   

       if (LOWORD(wParam) == IDCANCEL || LOWORD(wParam) == IDB_WEB_PAGE)
          EndDialog(hDlg, 0);
       return 1;
    }
    return 0;
}

static BOOL CALLBACK MameHelpDialogProc(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    switch (uMsg)
    {
        case WM_INITDIALOG :
            hHelp = hwndDlg;
            HelpPrintln("M.A.M.E. - Multiple Arcade Machine Emulator");
            HelpPrintln("Copyright (C) 1997-98  by Nicola Salmoria and the MAME team.");
            HelpPrintln("");
            HelpPrintln("MAME32 version created and maintained by Michael Soderstrom and Christopher Kirmse.");
            HelpPrintln("Please read the readme32.txt or mame32.hlp for more information.  Report ONLY Windows "
              "specific bugs to ckirmse@ricochet.net, AFTER reading the readme32.txt.");
            HelpPrintln("");
            HelpPrintln("Usage:");
            HelpPrintln("");
            HelpPrintln("MAME32 [game] [options]");
            HelpPrintln("");
            HelpPrintln("See the readme32.txt or mame32.hlp file for options");

            return TRUE;

    case WM_COMMAND :
        switch (LOWORD(wParam))
        {
            case IDOK :
                EndDialog(hwndDlg, 0);
                return TRUE;
            case IDCANCEL :
                EndDialog(hwndDlg, 1);
                return TRUE;
        }
        break;
    }
    return FALSE;
}

options_type * GetPlayingGameOptions()
{
    return &playing_game_options;
}

static void MamePlayRecordGame()
{
    char filename[MAX_PATH];
    *filename = 0;
    
    if (CommonFileDialog(GetSaveFileName,filename))
    {
        struct GameOptions mame_options;
        
        mame_options.record = osd_fopen(filename,0,OSD_FILETYPE_INPUTLOG,1);
        mame_options.playback = NULL;
        
        MamePlayGameWithOptions(mame_options);
        
        osd_fclose(mame_options.record);
    }
}

static void MamePlayBackGame()
{
    char filename[MAX_PATH];
    *filename = 0;
    
    if (CommonFileDialog(GetOpenFileName,filename))
    {
        struct GameOptions mame_options;
        
        mame_options.record = NULL;
        mame_options.playback = osd_fopen(filename,0,OSD_FILETYPE_INPUTLOG,0);
        
        MamePlayGameWithOptions(mame_options);
        
        osd_fclose(mame_options.playback);
    }
}

static BOOL CommonFileDialog(common_file_dialog_proc cfd,char *filename)
{
    BOOL success;
    
    OPENFILENAME of;
    
    of.lStructSize = sizeof(of);
    of.hwndOwner = hMain;
    of.hInstance = NULL;
    of.lpstrFilter = "MAME input files (*.inp)\0*.inp\0All files (*.*)\0*.*\0";
    of.lpstrCustomFilter = NULL;
    of.nMaxCustFilter = 0;
    of.nFilterIndex = 1;
    of.lpstrFile = filename;
    of.nMaxFile = MAX_PATH;
    of.lpstrFileTitle = NULL;
    of.nMaxFileTitle = 0;
    of.lpstrInitialDir = last_directory;
    of.lpstrTitle = NULL; 
    of.Flags = OFN_EXPLORER | OFN_NOCHANGEDIR | OFN_PATHMUSTEXIST | OFN_HIDEREADONLY;
    of.nFileOffset = 0;
    of.nFileExtension = 0;
    of.lpstrDefExt = "inp"; 
    of.lCustData = 0;
    of.lpfnHook = NULL;
    of.lpTemplateName = NULL;
    
    success = cfd(&of);
    if (success)
    {
        //GetDirectory(filename,last_directory,sizeof(last_directory));
    }
    
    return success;
}

static void MamePlayGame()
{
    /* mame's internal option structure */
    struct GameOptions mame_options;

    mame_options.record     = NULL;
    mame_options.playback   = NULL;

    MamePlayGameWithOptions(mame_options);
}

static void MamePlayGameWithOptions(struct GameOptions mame_options)
{
    int index;
    char buf[5000];
    RECT rect;

    index = GetSelectedPickItem();

    //NewDialogToOptions(hOptsDlog, TAB_DISPLAY, GetGameOptions(index));
    //NewDialogToOptions(hMiscDlog, TAB_MISC, GetGameOptions(index));
    memcpy(&playing_game_options, GetGameOptions(index), sizeof(options_type));
    
    /* Deal with options that can be disabled. */   
    EnablePlayOptions(index, &playing_game_options);

    /* based on our raw options, convert the ones that we can into
       regular MAME config options */

    if (playing_game_options.error_log)
        mame_options.errorlog = fopen("error.log", "wa");
    else
        mame_options.errorlog = NULL;

#ifdef MAME_DEBUG
    mame_options.mame_debug = 1;
#else
    mame_options.mame_debug = 0;
#endif
    mame_options.frameskip  = playing_game_options.frame_skip;
    mame_options.cheat      = playing_game_options.cheat;
    mame_options.samplerate = playing_game_options.sample_rate;
    mame_options.samplebits = playing_game_options.sample_bits;
    mame_options.norotate   = 0;

    {
    extern int use_emulated_ym3812;
    use_emulated_ym3812 = (playing_game_options.fm_ym3812 == 0);
    }

    /*
        Need sample rate initialised _before_ rom loading
        for optional rom regions.
    */
    if (playing_game_options.sound == SOUND_NONE)
    {
        /* update the Machine structure to show that sound is disabled */
		Machine->sample_rate = 0;
		mame_options.samplerate = 0;
    }

    switch (playing_game_options.rotate)
    {
    case ROTATE_RIGHT:
        mame_options.ror = 1;
        mame_options.rol = 0;
        break;

    case ROTATE_LEFT:
        mame_options.ror = 0;
        mame_options.rol = 1;
        break;

    default:
        mame_options.ror = 0;
        mame_options.rol = 0;
        break;
    }
    mame_options.flipx = playing_game_options.flipx;
    mame_options.flipy = playing_game_options.flipy;

    if (joygui != NULL)
        KillTimer(hMain,0);

    sprintf(buf, "%s - MAME32", drivers[index]->description);
    SetWindowText(hMain, buf);

    GetWindowRect(hMain, &rect);

    in_emulation = TRUE;

    ShowWindow(hMain, SW_HIDE);

    if (run_game(index,&mame_options) == 0)
    {
       IncrementPlayCount(index);
       ListView_RedrawItems(hwndList,GetSelectedPickItem(),GetSelectedPickItem());
    }
    else
    {
        const struct RomModule *romp = drivers[index]->rom;

        ShowWindow(hMain, SW_SHOW);

        /* This needs to be changed to use a window with a scrollbar
         * because the list can be too large to fit on the screen.
         */
        sprintf(buf, "Unable to initialize machine emulation.\r\n\r\n"
                     "Most likely you have a corrupt ROM file.\r\n"
                     "The following files must exist in the specified\r\n"
                     "directory or .zip file for %s to run.\r\n\r\n",
                     drivers[index]->description);

        sprintf(buf + strlen(buf), "Directory:\r\n");
        GetCurrentDirectory(sizeof(buf) - strlen(buf), buf + strlen(buf));
        sprintf(buf + strlen(buf), "\\%s\r\n\r\n", drivers[index]->name);
        sprintf(buf + strlen(buf), "or .zip file:\r\n");
        GetCurrentDirectory(sizeof(buf) - strlen(buf), buf + strlen(buf));
        sprintf(buf + strlen(buf), "\\roms\\%s.zip\r\n\r\n", drivers[index]->name);

        sprintf(buf + strlen(buf), "Files:\r\n");
        while (romp->name || romp->offset || romp->length)
        {
            romp++; /* skip memory region definition */

            while (romp->length)
            {
                char name[100];
                int length;

                sprintf(name, romp->name, drivers[index]->name);

                length = 0;

                do
                {
                    /* ROM_RELOAD */
                    if (romp->name == (char *)-1)
                        length = 0; /* restart */

                    length += romp->length & ~0x80000000;

                    romp++;
                } while (romp->length && (romp->name == 0 || romp->name == (char *)-1));

                sprintf(buf + strlen(buf), "%-12s\t%u bytes\n", name, length);
            }
        }

        MessageBox(hMain, buf, "MAME32 Error", MB_OK | MB_ICONERROR);
    }

    in_emulation = FALSE;
    
    if (quit)
    {
       PostMessage(hMain, WM_CLOSE, 0, 0);
       return;
    }

    // resort if sorting on # of times played
    if (last_sort == 6)
       ListView_SortItems(hwndList, ListCompareFunc, 6);

    SetStandardTitleBar();

    ShowWindow(hMain, SW_SHOW);
    SetFocus(hwndList);
    
    SetWindowPos(hMain, 0, 0, 0, rect.right - rect.left,
        rect.bottom - rect.top,
        SWP_NOMOVE | SWP_NOZORDER | SWP_NOCOPYBITS | SWP_SHOWWINDOW);

    if (joygui != NULL)
        SetTimer(hMain,0,25,NULL);

    if (playing_game_options.error_log && mame_options.errorlog)
        fclose(mame_options.errorlog);
}

/* Toggle ScreenShot ON/OFF */
static void ToggleScreenShot(void)
{
    BOOL showScreenShot = GetShowScreenShot();

    SetShowScreenShot((showScreenShot) ? FALSE : TRUE);
    UpdateScreenShot();

    /* Invalidate the List control */
    if( hBitmap != NULL && showScreenShot)
    {
        RECT rcClient;
        GetClientRect( hwndList, &rcClient );
        InvalidateRect( hwndList, &rcClient, TRUE );
    }
}

int bottomMargin;
int topMargin;

/* Adjust the list view and screenshot button based on GetShowScreenShot() */
void UpdateScreenShot(void)
{
    HWND hList = GetDlgItem(hPicker, IDC_LIST);
    HWND hParent = GetParent(hList);
    RECT rect;
    int  nWidth, show;
    int  nTreeWidth = 0;

    /* Size the List Control in the Picker */
    GetClientRect(hParent, &rect);

    if (bShowStatusBar)
        rect.bottom -= bottomMargin;
    if (bShowToolBar)
        rect.top += topMargin;

    if (GetShowScreenShot())
    {
        nWidth = nSplitterOffset[1];
        show   = SW_SHOW;
        CheckMenuItem(GetMenu(hMain),ID_VIEW_SCREEN_SHOT, MF_CHECKED);
        ToolBar_CheckButton(hToolBar, ID_VIEW_SCREEN_SHOT, MF_CHECKED);
    }
    else
    {
        nWidth = rect.right;
        show   = SW_HIDE;
        CheckMenuItem(GetMenu(hMain),ID_VIEW_SCREEN_SHOT, MF_UNCHECKED);
        ToolBar_CheckButton(hToolBar, ID_VIEW_SCREEN_SHOT, MF_UNCHECKED);
    }
    ShowWindow(GetDlgItem(hParent, IDC_SSPICTURE), SW_HIDE);
    ShowWindow(GetDlgItem(hParent, IDC_SSFRAME),   SW_HIDE);
    ShowWindow(GetDlgItem(hParent, IDC_SSDEFPIC),  SW_HIDE);
    ShowWindow(GetDlgItem(hParent, IDC_SSNONE),    SW_HIDE);

    /* Tree control */
    if (bShowTree)
    {
        nTreeWidth = nSplitterOffset[0];
        if (nTreeWidth >= nWidth)
        {
            nTreeWidth = nWidth - 10;
        }

        MoveWindow(GetDlgItem(hParent, IDC_TREE), 0, rect.top + 2, nTreeWidth - 2,
            (rect.bottom - rect.top) - 4 , TRUE);
        
        MoveWindow(GetDlgItem(hParent, IDC_SPLITTER), nTreeWidth, rect.top + 2,
            4, (rect.bottom - rect.top) - 4, TRUE);

        ShowWindow(GetDlgItem(hParent, IDC_TREE), SW_SHOW);
    }
    else
    {
        ShowWindow(GetDlgItem(hParent, IDC_TREE), SW_HIDE);
    }

    /* Splitter window #2 */
    MoveWindow(GetDlgItem(hParent, IDC_SPLITTER2), nSplitterOffset[1],
        rect.top + 2, 4, (rect.bottom - rect.top) - 2, FALSE);

    /* List control */
    MoveWindow(hList, 2 + nTreeWidth, rect.top + 2,
        (nWidth - nTreeWidth) - 2, (rect.bottom - rect.top) - 4 , TRUE);

    ShowWindow(GetDlgItem(hParent, IDC_SSPICTURE), show);
    ShowWindow(GetDlgItem(hParent, IDC_SSFRAME),   show);
    ShowWindow(GetDlgItem(hParent, IDC_SSDEFPIC),  show);
    ShowWindow(GetDlgItem(hParent, IDC_SSNONE),    show);
}

void ResizePickerControls(HWND hWnd)
{
    RECT picRect;
    RECT frameRect;
    RECT rect, sRect;
    int  nListWidth, nScreenShotWidth, picX, picY, nTreeWidth;
    static BOOL firstTime = TRUE;
    int  doSSControls = TRUE;

    /* Size the List Control in the Picker */
    GetClientRect(hWnd, &rect);

    /* Calc 1/2 of the display for the screenshot and 1/4 for the treecontrol */
    if (firstTime)
    {
        RECT rWindow;

        nListWidth = rect.right / 2;
        nSplitterOffset[1] = nListWidth;
        nSplitterOffset[0] = nListWidth / 2;
        GetWindowRect(hStatusBar, &rWindow);
        bottomMargin = rWindow.bottom - rWindow.top;
        GetWindowRect(hToolBar, &rWindow);
        topMargin = rWindow.bottom - rWindow.top;
        //buttonMargin = (sRect.bottom + 4);

        firstTime = FALSE;
    }
    else
    {
        doSSControls = GetShowScreenShot();
        nListWidth = nSplitterOffset[1];
    }

    if (bShowStatusBar)
        rect.bottom -= bottomMargin;

    if (bShowToolBar)
        rect.top += topMargin;

    MoveWindow(GetDlgItem(hWnd, IDC_DIVIDER), rect.left, rect.top - 4, rect.right, 2, TRUE);

    nScreenShotWidth = (rect.right - nListWidth) - 4;

    /* Tree Control */
    nTreeWidth = nSplitterOffset[0];
    MoveWindow(GetDlgItem(hWnd, IDC_TREE), 4, rect.top + 2,
        nTreeWidth - 4, (rect.bottom - rect.top) - 4, TRUE);

    /* Splitter window #1 */
    MoveWindow(GetDlgItem(hWnd, IDC_SPLITTER), nTreeWidth, rect.top + 2,
        4, (rect.bottom - rect.top) - 4, TRUE);

    /* List control */
    MoveWindow(GetDlgItem(hWnd, IDC_LIST), 4 + nTreeWidth, rect.top + 2,
        (nListWidth - nTreeWidth) - 4, (rect.bottom - rect.top) - 4, TRUE);

    /* Splitter window #2 */
    MoveWindow(GetDlgItem(hWnd, IDC_SPLITTER2), nListWidth, rect.top + 2,
        4, (rect.bottom - rect.top) - 2, doSSControls);

    /* resize the Screen shot frame */
    MoveWindow(GetDlgItem(hWnd, IDC_SSFRAME), nListWidth + 4, rect.top + 2,
        nScreenShotWidth - 2, (rect.bottom - rect.top) - 4, doSSControls);

    /* The screen shot controls */
    GetClientRect(GetDlgItem(hWnd, IDC_SSFRAME), &frameRect);
    GetClientRect(GetDlgItem(hWnd, IDC_SSDEFPIC), &picRect);

    picX = (frameRect.right - (picRect.right - picRect.left)) / 2;
    picY = (frameRect.bottom - (picRect.bottom - picRect.top)) / 2;

    /* Mame logo */
    MoveWindow(GetDlgItem(hWnd, IDC_SSDEFPIC), nListWidth + picX + 4,
        rect.top + 48, picRect.right + 2, picRect.bottom + 2, doSSControls);

    /* Screen shot sunken frame */
    MoveWindow(GetDlgItem(hWnd, IDC_SSPICTURE), nListWidth + picX + 4,
        rect.top + picY, picRect.right, picRect.bottom, doSSControls);

    GetClientRect(GetDlgItem(hWnd, IDC_SSNONE), &sRect);
    picX = (frameRect.right - sRect.right) / 2;

    /* Text control - message if no screen shot exists */
    MoveWindow(GetDlgItem(hWnd, IDC_SSNONE), nListWidth + picX + 4,
        rect.bottom - (sRect.bottom + 14),
        sRect.right, sRect.bottom, doSSControls);
}

static void AdjustMetrics(void)
{
    HDC hDC;
    TEXTMETRIC tm;
    int xtraX, xtraY;
    AREA area;
    int  offX, offY;
    int  maxX, maxY;
  
    // WM_SETTINGCHANGE also
    xtraX =  GetSystemMetrics(SM_CXFIXEDFRAME); // Dialog frame width
    xtraY =  GetSystemMetrics(SM_CYFIXEDFRAME); // Dialog frame height
    xtraY += GetSystemMetrics(SM_CYMENUSIZE);   // Menu height
    xtraY += GetSystemMetrics(SM_CYCAPTION);    // Caption Height
    maxX  =  GetSystemMetrics(SM_CXSCREEN);     // Screen Width
    maxY  =  GetSystemMetrics(SM_CYSCREEN);     // Screen Height

    hDC = GetDC(hMain);
    GetTextMetrics (hDC, &tm);
    
    // Convert MIN Width/Height from Dialog Box Units to pixels.
    MIN_WIDTH  = (int)((tm.tmAveCharWidth / 4.0) * DBU_MIN_WIDTH) + xtraX;
    MIN_HEIGHT = (int)((tm.tmHeight / 8.0) * DBU_MIN_HEIGHT) + xtraY;
    ReleaseDC(hMain, hDC);

    //ListView_SetTextBkColor(hwndList, CLR_NONE);
    ListView_SetBkColor(hwndList, GetSysColor(COLOR_WINDOW));
    ListView_SetTextColor(hwndList, GetListFontColor());
    TreeView_SetBkColor(hTreeView, GetSysColor(COLOR_WINDOW));
    TreeView_SetTextColor(hTreeView, GetListFontColor());
    GetWindowArea(&area);

    offX = area.x + area.width;
    offY = area.y + area.height;
    
    if (offX > maxX)
    {
        offX = maxX;
        area.x = (offX - area.width > 0) ? (offX - area.width) : 0;
    }
    if (offY > maxY)
    {
        offY = maxY;
        area.y = (offY - area.height > 0) ? (offY - area.height) : 0;
    }

    SetWindowArea(&area);
    SetWindowPos(hMain, 0, area.x, area.y, area.width, area.height, SWP_NOZORDER | SWP_SHOWWINDOW);
}


/* Adjust options - tune them to the currently selected game */
static void EnablePlayOptions(int nIndex, options_type *o)
{
    if (!MAME32App.m_bMMXDetected)
        o->disable_mmx = TRUE;

    if (!Joystick.Available(1))
        o->use_joystick = FALSE;

    if (!DIKeyboard_Available())
        o->di_keyboard = FALSE;

    if (!DIJoystick.Available(1))
        o->di_joystick = FALSE;

     /* Trackball / Mouse options */
    if (!GameUsesTrackball(nIndex))
        o->use_ai_mouse = FALSE;
    
    /* Double size */
    if (o->scale == 1)
    {
        o->hscan_lines = FALSE;
        o->vscan_lines = FALSE;
    }

    /* 16 bit color */
    if (! (drivers[nIndex]->drv->video_attributes & VIDEO_SUPPORTS_16BIT))
    {
        o->depth = 8;
    }
}

static int WhichIcon(int nItem)
{
    int iconRoms;
    
    iconRoms = GetHasRoms(nItem);
    
    /* Show Red-X if the ROMs are present and flaged as NOT WORKING */
    if (iconRoms == 1 && drivers[nItem]->flags & GAME_NOT_WORKING)
        iconRoms = 3;
    
    /* Use custom icon if found */
    if (iconRoms == 1 && icon_index != NULL)
        iconRoms = icon_index[nItem];

    return iconRoms;
}

BOOL HandleContextMenu( HWND hWnd, WPARAM wParam, LPARAM lParam)
{
    HMENU hMenuLoad;
    HMENU hMenu;
    
    if((HWND)wParam != GetDlgItem(hWnd, IDC_LIST))
        return FALSE;

    if (current_view_id == ID_VIEW_REPORT &&
        HeaderOnContextMenu(hWnd, wParam, lParam) == TRUE)
        return TRUE;

    hMenuLoad = LoadMenu(hInst, MAKEINTRESOURCE(IDR_CONTEXT_MENU));
    hMenu = GetSubMenu(hMenuLoad, 0);

    UpdateMenu(GetDlgItem(hWnd, IDC_LIST), hMenu);
    
    TrackPopupMenu( hMenu,
        TPM_LEFTALIGN | TPM_RIGHTBUTTON,
        LOWORD(lParam),
        HIWORD(lParam),
        0,
        hWnd,
        NULL);
    
    DestroyMenu(hMenuLoad);
    
    return TRUE;
}

void UpdateMenu(HWND hwndList, HMENU hMenu)
{
    char            buf[200];
    MENUITEMINFO    mItem;
    int             nGame = GetSelectedPickItem();

    sprintf(buf,"&Play %s", ConvertAmpersandString(drivers[nGame]->description));

    mItem.cbSize     = sizeof(mItem);
    mItem.fMask      = MIIM_TYPE;
    mItem.fType      = MFT_STRING;
    mItem.dwTypeData = buf;
    mItem.cch        = strlen(mItem.dwTypeData);

    SetMenuItemInfo(hMenu, ID_FILE_PLAY, FALSE, &mItem);

    if (oldControl)
    {
        EnableMenuItem(hMenu, ID_CUSTOMIZE_FIELDS, MF_GRAYED);
        EnableMenuItem(hMenu, ID_GAME_PROPERTIES,  MF_GRAYED);
    }
}

/* Add ... to Items in ListView if needed */
LPCTSTR MakeShortString(HDC hDC, LPCTSTR lpszLong, int nColumnLen, int nOffset)
{
    static const CHAR szThreeDots[]="...";
    static CHAR szShort[MAX_PATH];
    int nStringLen=lstrlen(lpszLong);
    int nAddLen;
    SIZE size;
    int i;

    GetTextExtentPoint32(hDC,lpszLong, nStringLen, &size);
    if(nStringLen==0 || size.cx + nOffset<=nColumnLen)
        return(lpszLong);

    lstrcpy(szShort,lpszLong);
    GetTextExtentPoint32(hDC,szThreeDots,sizeof(szThreeDots),&size);
    nAddLen=size.cx;

    for( i = nStringLen-1; i > 0; i--)
    {
        szShort[i]=0;
        GetTextExtentPoint32(hDC, szShort,i, &size);
        if(size.cx + nOffset + nAddLen <= nColumnLen)
            break;
    }

    lstrcat(szShort,szThreeDots);

    return(szShort);
}

/* DrawItem for ListView */
static void DrawItem(LPDRAWITEMSTRUCT lpDrawItemStruct)
{
    HDC         hDC = lpDrawItemStruct->hDC;
    RECT        rcItem = lpDrawItemStruct->rcItem;
    UINT        uiFlags = ILD_TRANSPARENT;
    HIMAGELIST  hImageList;
    int         nItem = lpDrawItemStruct->itemID;
    COLORREF    clrTextSave, clrBkSave;
    COLORREF    clrImage = GetSysColor(COLOR_WINDOW);
    static CHAR szBuff[MAX_PATH];
    BOOL        bFocus = (GetFocus() == hwndList);
    LPCTSTR     pszText;
    UINT        nStateImageMask;
    BOOL        bSelected;
    LV_COLUMN   lvc;
    LV_ITEM     lvi;
    RECT        rcAllLabels;
    RECT        rcLabel;
    RECT        rcIcon;
    int         offset;
    SIZE        size;
    int         i;
    int         nColumn;
    int         nColumnMax = 0;
    int         nResults = 0;
    int         order[COLUMN_MAX];


    nColumnMax = GetNumColumns(hwndList);

    if (oldControl)
    {
        GetColumnOrder(order);
    }
    else
    {
        /* Get the Column Order and save it */
        ListView_GetColumnOrderArray(hwndList, nColumnMax, order);

        /* Disallow moving column 0 */
        if (order[0] != 0)
        {
            for (i = 0; i < nColumnMax; i++)
            {
                if (order[i] == 0)
                {
                    order[i] = order[0];
                    order[0] = 0;
                }
            }
            ListView_SetColumnOrderArray(hwndList, nColumnMax, order);
        }
    }

    // Labels are offset by a certain amount  
    // This offset is related to the width of a space character
    GetTextExtentPoint32(hDC, " ", 1 , &size);
    offset = size.cx * 2;

    lvi.mask = LVIF_TEXT | LVIF_IMAGE | LVIF_STATE;
    lvi.iItem = nItem;
    lvi.iSubItem = order[0];
    lvi.pszText = szBuff;
    lvi.cchTextMax = sizeof(szBuff);
    lvi.stateMask = 0xFFFF;       // get all state flags
    ListView_GetItem(hwndList, &lvi);

    // This makes NO sense, but doesn't work without it?
    strcpy(szBuff, lvi.pszText);

    bSelected =((lvi.state & LVIS_DROPHILITED) || ( (lvi.state & LVIS_SELECTED)
        && ((bFocus) || (GetWindowLong(hwndList, GWL_STYLE) & LVS_SHOWSELALWAYS))));

    ListView_GetItemRect(hwndList, nItem, &rcAllLabels, LVIR_BOUNDS);

    ListView_GetItemRect(hwndList, nItem, &rcLabel, LVIR_LABEL);
    rcAllLabels.left = rcLabel.left;

    if( hBitmap != NULL )
    {
        RECT        rcClient;
        HRGN        rgnBitmap;
        RECT        rcTmpBmp = rcItem;
        RECT        rcFirstItem;
        int         i, j;
        HPALETTE    hPAL;
        HDC         htempDC;

        htempDC = CreateCompatibleDC(hDC);

        SelectObject(htempDC, hBitmap);

        GetClientRect(hwndList, &rcClient); 
        rcTmpBmp.right = rcClient.right;
        // We also need to check whether it is the last item
        // The update region has to be extended to the bottom if it is
        if( nItem == ListView_GetItemCount(hwndList) - 1 )
            rcTmpBmp.bottom = rcClient.bottom;
        
        rgnBitmap = CreateRectRgnIndirect(&rcTmpBmp);
        SelectClipRgn(hDC, rgnBitmap);
        DeleteObject(rgnBitmap);

        hPAL = (! hPALbg) ? CreateHalftonePalette(hDC) : hPALbg;

        if(GetDeviceCaps(htempDC, RASTERCAPS) & RC_PALETTE && hPAL != NULL )
        {
            SelectPalette(htempDC, hPAL, FALSE );
            RealizePalette(htempDC);
        }
        
        ListView_GetItemRect(hwndList, 0, &rcFirstItem, LVIR_BOUNDS);
        
        for( i = rcFirstItem.left; i < rcClient.right; i += bmDesc.bmWidth )
            for( j = rcFirstItem.top; j < rcClient.bottom; j += bmDesc.bmHeight )
                BitBlt(hDC, i, j, bmDesc.bmWidth, bmDesc.bmHeight, htempDC, 0, 0, SRCCOPY );
        
        DeleteDC(htempDC);

        if (! bmDesc.bmColors)
        {
            DeleteObject(hPAL);
            hPAL = 0;
        }
    }

    SetTextColor(hDC, GetListFontColor());

    if(bSelected)
    {
        HBRUSH hBrush;
        if (bFocus)
        {
            clrTextSave = SetTextColor(hDC, GetSysColor(COLOR_HIGHLIGHTTEXT));
            clrBkSave = SetBkColor(hDC, GetSysColor(COLOR_HIGHLIGHT));
            hBrush = CreateSolidBrush(GetSysColor(COLOR_HIGHLIGHT));
        }
        else
        {
            clrTextSave = SetTextColor(hDC, GetSysColor(COLOR_BTNTEXT));
            clrBkSave = SetBkColor(hDC, GetSysColor(COLOR_BTNFACE));
            hBrush = CreateSolidBrush(GetSysColor(COLOR_BTNFACE));
        }
        SelectObject(hDC, hBrush);
        FillRect(hDC, &rcAllLabels, hBrush);
        DeleteObject(hBrush);
    }
    else if( hBitmap == NULL )
    {
        HBRUSH hBrush;

        hBrush = CreateSolidBrush( GetSysColor(COLOR_WINDOW));
        FillRect(hDC, &rcAllLabels, hBrush);
        DeleteObject(hBrush);
    }

    if(lvi.state & LVIS_CUT)
    {
        clrImage = GetSysColor(COLOR_WINDOW);
        uiFlags |= ILD_BLEND50;
    }
    else if(bSelected)
    {
        if (bFocus)
            clrImage = GetSysColor(COLOR_HIGHLIGHT);
        else
            clrImage = GetSysColor(COLOR_BTNFACE);

        uiFlags |= ILD_BLEND50;
    }

    nStateImageMask = lvi.state & LVIS_STATEIMAGEMASK;

    if(nStateImageMask)
    {
        int nImage = (nStateImageMask >> 12) - 1;
        hImageList = ListView_GetImageList(hwndList, LVSIL_STATE);
        if(hImageList)
            ImageList_Draw(hImageList, nImage, hDC, rcItem.left, rcItem.top, ILD_TRANSPARENT);
    }

    ListView_GetItemRect(hwndList, nItem, &rcIcon, LVIR_ICON);

    hImageList = ListView_GetImageList(hwndList, LVSIL_SMALL);
    if(hImageList)
    {
        UINT nOvlImageMask = lvi.state & LVIS_OVERLAYMASK;
        if(rcItem.left < rcItem.right-1)
            ImageList_DrawEx(hImageList, lvi.iImage, hDC, rcIcon.left, rcIcon.top,
                16, 16, GetSysColor(COLOR_WINDOW), clrImage, uiFlags | nOvlImageMask);
    }

    ListView_GetItemRect(hwndList, nItem, &rcItem,LVIR_LABEL);

    pszText = MakeShortString(hDC, szBuff,rcItem.right - rcItem.left,  2 * offset);

    rcLabel = rcItem;
    rcLabel.left += offset;
    rcLabel.right -= offset;

    DrawText(hDC, pszText,-1, &rcLabel,
        DT_LEFT | DT_SINGLELINE | DT_NOPREFIX | DT_NOCLIP | DT_VCENTER);

    for(nColumn = 1; nColumn < nColumnMax ; nColumn++)
    {
        int nRetLen;
        UINT nJustify;
        LV_ITEM lvi;
        
        lvc.mask = LVCF_FMT | LVCF_WIDTH;
        ListView_GetColumn(hwndList, order[nColumn] , &lvc);

        lvi.mask = LVIF_TEXT;
        lvi.iItem = nItem; 
        lvi.iSubItem = order[nColumn];
        lvi.pszText = szBuff;
        lvi.cchTextMax = sizeof(szBuff);

        if (ListView_GetItem(hwndList, &lvi) == FALSE)
            continue;

        /* This shouldn't oughtta be, but it's needed!!! */
        strcpy(szBuff, lvi.pszText);

        rcItem.left = rcItem.right;
        rcItem.right += lvc.cx;

        nRetLen = strlen(szBuff);
        if(nRetLen == 0)
            continue;

        pszText = MakeShortString(hDC, szBuff, rcItem.right - rcItem.left, 2 * offset);

        nJustify = DT_LEFT;

        if(pszText == szBuff)
        {
            switch(lvc.fmt & LVCFMT_JUSTIFYMASK)
            {
            case LVCFMT_RIGHT:
                nJustify = DT_RIGHT;
                break;
            case LVCFMT_CENTER:
                nJustify = DT_CENTER;
                break;
            default:
                break;
            }
        }

        rcLabel = rcItem;
        rcLabel.left += offset;
        rcLabel.right -= offset;
        DrawText(hDC, pszText, -1, &rcLabel,
            nJustify | DT_SINGLELINE | DT_NOPREFIX | DT_NOCLIP | DT_VCENTER);
    }

    if(lvi.state & LVIS_FOCUSED && bFocus)
        DrawFocusRect(hDC, &rcAllLabels);

    if(bSelected)
    {
        SetTextColor(hDC, clrTextSave);
        SetBkColor(hDC, clrBkSave);
    }
}

/* Header code - Directional Arrows */
LRESULT CALLBACK HeaderWndProc( HWND hwnd,  UINT uMsg,  WPARAM wParam,  LPARAM lParam)
{
    switch(uMsg)
    {
    case WM_ERASEBKGND:
        //if (current_view_id == ID_VIEW_REPORT && hBitmap)
        //    return TRUE;
        break;

    case WM_DRAWITEM:
        {
            /* UINT idCtl = (UINT)wParam; */
            LPDRAWITEMSTRUCT lpdis = (LPDRAWITEMSTRUCT)lParam;
            RECT rcLabel;
            int nSavedDC;
            HRGN hrgn;
            int nOffset;
            SIZE size;
            TCHAR tszText[256];
            HD_ITEM hdi;
            UINT uFormat;
            HDC hdc;
            
            hdc = lpdis->hDC;
            
            CopyRect(&rcLabel, &(lpdis->rcItem));
            
            /* save the DC */
            nSavedDC = SaveDC(hdc);
            
            /* set clip region to column */
            hrgn = CreateRectRgnIndirect(&rcLabel);
            SelectObject(hdc, hrgn);
            DeleteObject(hrgn);
            
            /* draw the background */
            FillRect(hdc, &rcLabel, GetSysColorBrush(COLOR_3DFACE));
            
            /* offset the label */
            GetTextExtentPoint32(hdc, _T(" "), 1, &size);
            nOffset = size.cx * 2;
            
            /* get the column text and format */
            hdi.mask = HDI_TEXT | HDI_FORMAT;
            hdi.pszText = tszText;
            hdi.cchTextMax = 255;
            Header_GetItem(GetDlgItem(hwnd, 0), lpdis->itemID, &hdi);
            
            /* determine format for drawing label */
            uFormat = DT_SINGLELINE | DT_NOPREFIX | DT_NOCLIP |
                DT_VCENTER | DT_END_ELLIPSIS;
            
            /* determine justification */
            if (hdi.fmt & HDF_CENTER)
                uFormat |= DT_CENTER;
            else if (hdi.fmt & HDF_RIGHT)
                uFormat |= DT_RIGHT;
            else
                uFormat |= DT_LEFT;
            
            /* adjust the rect if selected */
            if (lpdis->itemState & ODS_SELECTED)
            {
                rcLabel.left++;
                rcLabel.top += 2;
                rcLabel.right++;
            }
            
            /* adjust rect for sort arrow */
            if (lpdis->itemID == m_uHeaderSortCol)
            {
                rcLabel.right -= (3 * nOffset);
            }
            
            rcLabel.left += nOffset;
            rcLabel.right -= nOffset;
            
            /* draw label */
            if (rcLabel.left < rcLabel.right)
                DrawText(hdc, tszText, -1, &rcLabel, uFormat);
            
            /* draw the arrow */
            if (lpdis->itemID == m_uHeaderSortCol)
            {
                RECT rcIcon;
                HPEN hpenLight, hpenShadow, hpenOld;
                
                CopyRect(&rcIcon, &(lpdis->rcItem));
                
                hpenLight = CreatePen(PS_SOLID, 1, GetSysColor(COLOR_3DHILIGHT));
                hpenShadow = CreatePen(PS_SOLID, 1, GetSysColor(COLOR_3DSHADOW));
                hpenOld = SelectObject(hdc, hpenLight);
                
                if (m_fHeaderSortAsc)
                {
                    /* draw triangle pointing up */
                    MoveToEx(hdc, rcIcon.right - 2 * nOffset, nOffset - 1, NULL);
                    LineTo(hdc, rcIcon.right - 3 * nOffset / 2,
                        rcIcon.bottom - nOffset);
                    LineTo(hdc, rcIcon.right - 5 * nOffset / 2 - 2,
                        rcIcon.bottom - nOffset);
                    MoveToEx(hdc, rcIcon.right - 5 * nOffset / 2 - 1,
                        rcIcon.bottom - nOffset, NULL);
                    
                    SelectObject(hdc, hpenShadow);
                    LineTo(hdc, rcIcon.right - 2 * nOffset, nOffset - 2);
                }
                else
                {
                    /* draw triangle pointing down */
                    MoveToEx(hdc, rcIcon.right - 3 * nOffset / 2, nOffset - 1, 
                        NULL);
                    LineTo(hdc, rcIcon.right - 2 * nOffset - 1,
                        rcIcon.bottom - nOffset);
                    LineTo(hdc, rcIcon.right - 2 * nOffset - 1,
                        rcIcon.bottom - nOffset);
                    MoveToEx(hdc, rcIcon.right - 2 * nOffset - 1,
                        rcIcon.bottom - nOffset, NULL);
                    
                    SelectObject(hdc, hpenShadow);
                    LineTo(hdc, rcIcon.right - 5 * nOffset / 2 - 1,
                        nOffset - 1);
                    LineTo(hdc, rcIcon.right - 3 * nOffset / 2,
                        nOffset - 1);
                }
                SelectObject(hdc, hpenOld);
                DeletePen(hpenLight);
                DeletePen(hpenShadow);
            }
            RestoreDC(hdc, nSavedDC);
            return 1;
        }
          
    case WM_NOTIFY:
        {
            HD_NOTIFY   *pHDN = (HD_NOTIFY*)lParam;
            
            // This code is for using bitmap in the background
            // Invalidate the right side of the control when a column is resized
            if (pHDN->hdr.code == HDN_ITEMCHANGINGW ||
                pHDN->hdr.code == HDN_ITEMCHANGINGA)
            {
                if( hBitmap != NULL )
                {
                    RECT rcClient;
                    DWORD dwPos;
                    POINT pt;
                    
                    dwPos = GetMessagePos();
                    pt.x = LOWORD(dwPos);
                    pt.y = HIWORD(dwPos);
                    
                    GetClientRect(hwnd, &rcClient );
                    ScreenToClient( hwnd, &pt );
                    rcClient.left = pt.x;
                    InvalidateRect( hwnd, &rcClient, FALSE );
                }
            }
        }
        break;
    }

    /* message not handled */
    return CallWindowProc(g_lpHeaderWndProc, hwnd, uMsg, wParam, lParam);
}

/* Subclass the Listview Header */
static void Header_Initialize(HWND hwndList)
{
    /* this will subclass the listview (where WM_DRAWITEM gets sent for
       the header control) */
    g_lpHeaderWndProc = (WNDPROC)GetWindowLong(hwndList, GWL_WNDPROC);
    SetWindowLong(hwndList, GWL_WNDPROC, (LONG)HeaderWndProc);
}


/* Set Sort information needed by Header */
static void Header_SetSortInfo(HWND hwndList, int nCol, BOOL bAsc)
{
    HWND hwndHeader;
    HD_ITEM hdi;

    m_uHeaderSortCol = nCol;
    m_fHeaderSortAsc = bAsc;

    hwndHeader = GetDlgItem(hwndList, 0);

    /* change this item to owner draw */
    hdi.mask = HDI_FORMAT;
    Header_GetItem(hwndHeader, nCol, &hdi);
    hdi.fmt |= HDF_OWNERDRAW;
    Header_SetItem(hwndHeader, nCol, &hdi);

    /* repaint the header */
    InvalidateRect(hwndHeader, NULL, TRUE);
}


int GetNumColumns(HWND hWnd)
{
    int  nColumn = 0;
    int  i;
    HWND hwndHeader;
    int  shown[COLUMN_MAX];

    GetColumnShown(shown);
    hwndHeader = GetDlgItem(hWnd, 0);

    if (oldControl || (nColumn = Header_GetItemCount(hwndHeader)) < 1)
    {
        nColumn = 0;
        for (i = 0; i < COLUMN_MAX ; i++ )
        {
            if (shown[i])
                nColumn++;
        }
    }
    return nColumn;
}

void GetRealColumnOrder(int order[])
{
    int tmpOrder[COLUMN_MAX];
    int nColumnMax;
    int i;

    nColumnMax = GetNumColumns(hwndList);

    /* Get the Column Order and save it */
    if (!oldControl)
    {
        ListView_GetColumnOrderArray(hwndList, nColumnMax, tmpOrder);
        
        for (i = 0; i < nColumnMax; i++)
        {
            order[i] = realColumn[tmpOrder[i]];
        }
    }
}

void UpdateCurrentOptions(int iGame)
{
    UpdateScreenShot();
}

/* End of source file */

