/***************************************************************************

    M.A.M.E.32  -  Multiple Arcade Machine Emulator for Win32
    Win32 Portions Copyright (C) 1997-98 Michael Soderstrom and Chris Kirmse
    
    This file is part of MAME32, and may only be used, modified and
    distributed under the terms of the MAME license, in "readme.txt".
    By continuing to use, modify or distribute this file you indicate
    that you have read the license and understand and accept it fully.

 ***************************************************************************/
 
 /***************************************************************************

  options.c

  Stores global options and per-game options;

***************************************************************************/

#include "driver.h"

#include <windowsx.h>
#include <windows.h>
#include <winreg.h>
#include <commctrl.h>
#include "mame32.h"
#include "m32util.h"
#include <resource.h>
#include <stdio.h>
#include <malloc.h>
#include <math.h>
#include <sys/stat.h>
#include "ddrawdisplay.h"

/***************************************************************************
    Internal function prototypes
 ***************************************************************************/

extern void UpdateDisplayModeUI(HWND hwnd, DWORD dwDepth);

void  SaveOptions(void);
void  LoadOptions(void);

void  LoadRegGameOptions(HKEY hKey, options_type *o);
DWORD GetRegOption(HKEY hKey, char *name);
void  GetRegBoolOption(HKEY hKey, char *name, BOOL *value);
char  *GetRegStringOption(HKEY hKey, char *name);

BOOL  SaveRegGameOptions(HKEY hKey, options_type *o);
void  PutRegOption(HKEY hKey, char *name, DWORD value);
void  PutRegBoolOption(HKEY hKey, char *name, BOOL value);
void  PutRegStringOption(HKEY hKey, char *name, char *option);

void  DisplayDialogToOptions(HWND hwnd, options_type *o);
void  MiscDialogToOptions(HWND hwnd, options_type *o);

void  OptionsToDisplayDialog(HWND hwnd, options_type *o);
void  OptionsToMiscDialog(HWND hwnd, options_type *o);

static int  ViewStringToVal(char *ptr);
static void ColumnDecodeString(int values[], char *ptr);
static void ColumnEncodeString(int values[], char *ptr);

static void SavePlayCount(int game_index);

/***************************************************************************
    Internal defines
 ***************************************************************************/

/* This describes how the options are packaged when stored to the registry
 
   Scale            DWORD   
   Rotate           DWORD   0 - 2
   Width            DWORD   width
   Height           DWORD   height
   Sound            DWORD   sound
   SealSound        DWORD   seal_index
   SampleRate       DWORD   sample_rate
   SampleBits       DWORD   sample_bits
   FramesSkip       DWORD   0 to 100
   Gamma            SZ_REG  float (String)
   IsWindow         SZ_REG  BOOL  (String) "0" = FALSE, "1" = TRUE
   BestFit          SZ_REG  BOOL  (String) "0" = FALSE, "1" = TRUE
   DoubleVector     SZ_REG  BOOL  (String) "0" = FALSE, "1" = TRUE
   WindowDDraw      SZ_REG  BOOL  (String) "0" = FALSE, "1" = TRUE
   HScanLines       SZ_REG  BOOL  (String) "0" = FALSE, "1" = TRUE
   VScanLines       SZ_REG  BOOL  (String) "0" = FALSE, "1" = TRUE
   UseJoystick      SZ_REG  BOOL  (String) "0" = FALSE, "1" = TRUE
   UseTrackball     SZ_REG  BOOL  (String) "0" = FALSE, "1" = TRUE
   DiKeyboard       SZ_REG  BOOL  (String) "0" = FALSE, "1" = TRUE
   DiJoystick       SZ_REG  BOOL  (String) "0" = FALSE, "1" = TRUE
   Cheat            SZ_REG  BOOL  (String) "0" = FALSE, "1" = TRUE
   AutoPause        SZ_REG  BOOL  (String) "0" = FALSE, "1" = TRUE
   ErrorLog         SZ_REG  BOOL  (String) "0" = FALSE, "1" = TRUE
 */

/* Used to create/retrieve Registry database */
// #define KEY_BASE "Software\\Freeware\\TestMame32"
#define KEY_BASE "Software\\Freeware\\Mame32"
#define KEY_FMT (KEY_BASE "\\%s")

/***************************************************************************
    Internal structures
 ***************************************************************************/

/***************************************************************************
    Internal variables
 ***************************************************************************/

static options_type global;
static options_type *game;
static int num_games;

// Default sizes based on 8pt font w/sort arrow in that column
static int default_column_widths[] = { 186, 68, 84, 84, 64, 88, 74,108, 60,144 };
static int default_column_shown[]  = {   1,  0,  1,  1,  1,  1,  1,  1,  1,  1 };
// Hidden columns need to go at the end of the order array
static int default_column_order[]  = {   0,  2,  3,  4,  5,  6,  7,  8,  9,  1 };

static settings_type settings;

static char *view_modes[VIEW_MAX] = {
    "Large Icons",
    "Small Icons",
    "List",
    "Details"
};


/***************************************************************************
    External functions  
 ***************************************************************************/

void OptionsInit(int total_games)
{
    int i;

    num_games = total_games;

    settings.show_all_games = TRUE;
    settings.show_unavailable_games = FALSE;
    settings.folder_id = 0;
    settings.view = VIEW_REPORT;
    settings.show_folderlist = TRUE;
    settings.show_toolbar = TRUE;
    settings.show_statusbar = TRUE;
    settings.show_screenshot = TRUE;
    settings.large_screenshot = TRUE;
    settings.show_no_clones = FALSE;
    strcpy(settings.default_game,"pacman");

    for (i = 0; i < COLUMN_MAX; i++)
    {
        settings.column_widths[i] = default_column_widths[i];
        settings.column_order[i]  = default_column_order[i];
        settings.column_shown[i]  = default_column_shown[i];
    }

    settings.sort_column = 0;
    settings.area.x      = 0;
    settings.area.y      = 0;
    settings.area.width  = 640;
    settings.area.height = 400;
    settings.splitter[0] = 150;
    settings.splitter[1] = 300;
    settings.romdirs     = strdup(".;roms");
    settings.sampledirs  = strdup(".;samples");
    settings.cfgdir      = strdup("cfg");
    settings.hidir       = strdup("hi");
    settings.inpdir      = strdup("inp");
    settings.imgdir      = strdup("images");
    settings.statedir    = strdup("sta");
    settings.artdir      = strdup("artwork");

    settings.list_font.lfHeight = -8;
    settings.list_font.lfWidth = 0;
    settings.list_font.lfEscapement = 0;
    settings.list_font.lfOrientation = 0;
    settings.list_font.lfWeight = FW_NORMAL;
    settings.list_font.lfItalic = FALSE;
    settings.list_font.lfUnderline = FALSE;
    settings.list_font.lfStrikeOut = FALSE;
    settings.list_font.lfCharSet = ANSI_CHARSET;
    settings.list_font.lfOutPrecision = OUT_DEFAULT_PRECIS;
    settings.list_font.lfClipPrecision = CLIP_DEFAULT_PRECIS;
    settings.list_font.lfQuality = DEFAULT_QUALITY;
    settings.list_font.lfPitchAndFamily = DEFAULT_PITCH | FF_DONTCARE;

    strcpy(settings.list_font.lfFaceName, "MS Sans Serif");

    settings.list_font_color = (COLORREF)-1;

    global.use_default = FALSE;

    global.is_window        = FALSE;
    global.display_best_fit = TRUE;
    global.width            = 640;
    global.height           = 480;
    global.skip_lines       = 0;
    global.skip_columns     = 0;
    global.double_vector    = FALSE;
    global.window_ddraw     = TRUE;
    global.frame_skip       = 0;
    global.use_dirty        = TRUE;
    global.hscan_lines      = TRUE;
    global.vscan_lines      = FALSE;
    global.use_blit         = FALSE;
    global.disable_mmx      = TRUE;
    global.scale            = 2;
    global.rotate           = ROTATE_NONE;
    global.flipx            = FALSE;
    global.flipy            = FALSE;
    global.gamma            = 1.0;
    global.brightness       = 100;
    global.depth            = 8;
    global.antialias        = TRUE;
    global.translucency     = TRUE;
    global.beam             = 1.0;
    global.flicker          = 0;

    global.use_joystick = FALSE;
    global.use_ai_mouse = TRUE;
    global.di_keyboard  = TRUE;
    global.di_joystick  = TRUE;

#ifndef NOSEAL
    global.sound        = SOUND_DIRECT;
    global.seal_index   = 0;
#else
    global.sound        = SOUND_SEAL;
    global.seal_index   = 3;
#endif
    global.sample_rate  = 22050;
    global.sample_bits  = 8;
    global.stereo       = TRUE;

    global.fm_ym3812    = FALSE;

    global.cheat        = FALSE;
    global.auto_pause   = TRUE;
    global.error_log    = FALSE;
    global.profile      = FALSE;

    global.play_count   = 0;
    global.has_roms     = UNKNOWN;
    global.has_samples  = UNKNOWN;

    game = (options_type *)malloc(num_games * sizeof(options_type));
    for (i = 0; i < num_games; i++)
    {
        game[i] = global;
        game[i].use_default = TRUE;
    }

    LoadOptions();
}

void OptionsExit()
{
    SaveOptions();
    free(game);
    free(settings.romdirs);
    free(settings.sampledirs);
    free(settings.cfgdir);
    free(settings.hidir);
    free(settings.inpdir);
    free(settings.imgdir);
    free(settings.statedir);
    free(settings.artdir);
}

options_type * GetDefaultOptions()
{
    return &global;
}

options_type * GetGameOptions(int num_game)
{
    int play_count  = game[num_game].play_count;
    int has_roms    = game[num_game].has_roms;
    int has_samples = game[num_game].has_samples;

    if (game[num_game].use_default)
    {
        game[num_game]              = global;
        game[num_game].use_default  = TRUE;
        game[num_game].play_count   = play_count;
        game[num_game].has_roms     = has_roms;
        game[num_game].has_samples  = has_samples;

    }
    return &game[num_game];
}

void SetViewMode(int val)
{
    settings.view = val;
}

int GetViewMode(void)
{
    return settings.view;
}

void SetShowAllGames(BOOL val)
{
    settings.show_all_games = val;
}

BOOL GetShowAllGames(void)
{
    return settings.show_all_games;
}

void SetShowUnavailableGames(BOOL val)
{
    settings.show_unavailable_games = val;
}

BOOL GetShowUnavailableGames(void)
{
    return settings.show_unavailable_games;
}

void SetSavedFolderID(UINT val)
{
    settings.folder_id = val;
}

UINT GetSavedFolderID(void)
{
    return settings.folder_id;
}

void SetShowScreenShot(BOOL val)
{
    settings.show_screenshot = val;
}

BOOL GetShowScreenShot(void)
{
    return settings.show_screenshot;
}

void SetShowFolderList(BOOL val)
{
    settings.show_folderlist = val;
}

BOOL GetShowFolderList(void)
{
    return settings.show_folderlist;
}

void SetShowStatusBar(BOOL val)
{
    settings.show_statusbar = val;
}

BOOL GetShowStatusBar(void)
{
    return settings.show_statusbar;
}

void SetShowToolBar(BOOL val)
{
    settings.show_toolbar = val;
}

BOOL GetShowToolBar(void)
{
    return settings.show_toolbar;
}

void SetLargeScreenShot(BOOL val)
{
    settings.large_screenshot = val;
}

BOOL GetLargeScreenShot(void)
{
    return settings.large_screenshot;
}

void SetShowNoClones(BOOL val)
{
    settings.show_no_clones = val;
}

BOOL GetShowNoClones(void)
{
    return settings.show_no_clones;
}

void SetDefaultGame(const char *name)
{
    strcpy(settings.default_game,name);
}

const char *GetDefaultGame(void)
{
    return settings.default_game;
}

void SetWindowArea(AREA *area)
{
    memcpy(&settings.area, area, sizeof(AREA));
}

void GetWindowArea(AREA *area)
{
    memcpy(area, &settings.area, sizeof(AREA));
}

void SetListFont(LOGFONT *font)
{
    memcpy(&settings.list_font, font, sizeof(LOGFONT));
}

void GetListFont(LOGFONT *font)
{
    memcpy(font, &settings.list_font, sizeof(LOGFONT));
}

void SetListFontColor(COLORREF uColor)
{
    if (settings.list_font_color == GetSysColor(COLOR_WINDOWTEXT))
        settings.list_font_color = (COLORREF)-1;
    else
        settings.list_font_color = uColor;
}

COLORREF GetListFontColor()
{
    if (settings.list_font_color == (COLORREF)-1)
        return (GetSysColor(COLOR_WINDOWTEXT));

    return settings.list_font_color;
}

void SetColumnWidths(int widths[])
{
    int i;

    for (i=0; i < COLUMN_MAX; i++)
        settings.column_widths[i] = widths[i];
}

void GetColumnWidths(int widths[])
{
    int i;

    for (i=0; i < COLUMN_MAX; i++)
        widths[i] = settings.column_widths[i];
}

void SetSplitterPos(int splitterId, int pos)
{
    if (splitterId < SPLITTER_MAX)
        settings.splitter[splitterId] = pos;
}

int  GetSplitterPos(int splitterId)
{
    if (splitterId < SPLITTER_MAX)
        return settings.splitter[splitterId];

    return -1; /* Error */
};

static void SplitterEncodeString(int value[], char *ptr)
{
    int  i;
    char tmpStr[100];

    sprintf(tmpStr,"%d", value[0]);
    
    strcpy(ptr,tmpStr);

    for (i = 1; i < SPLITTER_MAX; i++)
    {
        sprintf(tmpStr, ",%d", value[i]);
        strcat(ptr,tmpStr);
    }
}

static void SplitterDecodeString(int value[], char *ptr)
{
    int  i;
    char *s, *p;
    char tmpStr[100];

    if (ptr == NULL)
        return;

    strcpy (tmpStr, ptr);
    p = tmpStr;
    
    for (i = 0; p && i < SPLITTER_MAX; i++)
    {
        s = p;
        
        if ((p = strchr(s,',')) != NULL && *p == ',')
        {
            *p = '\0';
            p++;
        }
        value[i] = atoi(s);
    }
}

static void ColumnEncodeString(int value[], char *ptr)
{
    int  i;
    char tmpStr[100];

    sprintf(tmpStr,"%d", value[0]);
    
    strcpy(ptr,tmpStr);

    for (i = 1; i < COLUMN_MAX; i++)
    {
        sprintf(tmpStr, ",%d", value[i]);
        strcat(ptr,tmpStr);
    }
}

static void ColumnDecodeString(int value[], char *ptr)
{
    int  i;
    char *s, *p;
    char tmpStr[100];

    if (ptr == NULL)
        return;

    strcpy (tmpStr, ptr);
    p = tmpStr;
    
    for (i = 0; p && i < COLUMN_MAX; i++)
    {
        s = p;
        
        if ((p = strchr(s,',')) != NULL && *p == ',')
        {
            *p = '\0';
            p++;
        }
        value[i] = atoi(s);
    }
}

void SetColumnOrder(int order[])
{
    int i;

    for (i = 0; i < COLUMN_MAX; i++)
        settings.column_order[i] = order[i];
}

void GetColumnOrder(int order[])
{
    int i;

    for (i = 0; i < COLUMN_MAX; i++)
        order[i] = settings.column_order[i];
}

void SetColumnShown(int shown[])
{
    int i;

    for (i = 0; i < COLUMN_MAX; i++)
        settings.column_shown[i] = shown[i];
}

void GetColumnShown(int shown[])
{
    int i;

    for (i = 0; i < COLUMN_MAX; i++)
        shown[i] = settings.column_shown[i];
}

void SetSortColumn(int column)
{
    settings.sort_column = column;
}

int GetSortColumn(void)
{
    return settings.sort_column;
}

const char* GetRomDirs(void)
{
    return settings.romdirs;
}

void SetRomDirs(const char* paths)
{
    if (settings.romdirs != NULL)
    {
        free(settings.romdirs);
        settings.romdirs = NULL;
    }

    if (paths != NULL)
        settings.romdirs = strdup(paths);
}

const char* GetSampleDirs(void)
{
    return settings.sampledirs;
}

void SetSampleDirs(const char* paths)
{
    if (settings.sampledirs != NULL)
    {
        free(settings.sampledirs);
        settings.sampledirs = NULL;
    }

    if (paths != NULL)
        settings.sampledirs = strdup(paths);
}

const char* GetCfgDir(void)
{
    return settings.cfgdir;
}

void SetCfgDir(const char* path)
{
    if (settings.cfgdir != NULL)
    {
        free(settings.cfgdir);
        settings.cfgdir = NULL;
    }

    if (path != NULL)
        settings.cfgdir = strdup(path);
}

const char* GetHiDir(void)
{
    return settings.hidir;
}

void SetHiDir(const char* path)
{
    if (settings.hidir != NULL)
    {
        free(settings.hidir);
        settings.hidir = NULL;
    }

    if (path != NULL)
        settings.hidir = strdup(path);
}

const char* GetInpDir(void)
{
    return settings.inpdir;
}

void SetInpDir(const char* path)
{
    if (settings.inpdir != NULL)
    {
        free(settings.inpdir);
        settings.inpdir = NULL;
    }

    if (path != NULL)
        settings.inpdir = strdup(path);
}

const char* GetImgDir(void)
{
    return settings.imgdir;
}

void SetImgDir(const char* path)
{
    if (settings.imgdir != NULL)
    {
        free(settings.imgdir);
        settings.imgdir = NULL;
    }

    if (path != NULL)
        settings.imgdir = strdup(path);
}

const char* GetStateDir(void)
{
    return settings.statedir;
}

void SetStateDir(const char* path)
{
    if (settings.statedir != NULL)
    {
        free(settings.statedir);
        settings.statedir = NULL;
    }

    if (path != NULL)
        settings.statedir = strdup(path);
}

const char* GetArtDir(void)
{
    return settings.artdir;
}

void SetArtDir(const char* path)
{
    if (settings.artdir != NULL)
    {
        free(settings.artdir);
        settings.artdir = NULL;
    }

    if (path != NULL)
        settings.artdir = strdup(path);
}

void ResetGameOptions(int num_game)
{
    int play_count  = game[num_game].play_count;
    int has_roms    = game[num_game].has_roms;
    int has_samples = game[num_game].has_samples;

    game[num_game]              = global;
    game[num_game].play_count   = play_count;
    game[num_game].has_roms     = has_roms;
    game[num_game].has_samples  = has_samples;
}

int  GetHasRoms(int num_game)
{
    return game[num_game].has_roms;
}

void SetHasRoms(int num_game, int has_roms)
{
    game[num_game].has_roms = has_roms;
}

int  GetHasSamples(int num_game)
{
    return game[num_game].has_samples;
}

void SetHasSamples(int num_game, int has_samples)
{
    game[num_game].has_samples = has_samples;
}

void IncrementPlayCount(int num_game)
{
   game[num_game].play_count++;
   // Save play_count to the registry
   SavePlayCount(num_game);
}

int GetPlayCount(int num_game)
{
   return game[num_game].play_count;
}

void NewDialogToOptions(HWND hwnd, int iTab, options_type *o)
{
    o->use_default = Button_GetCheck(GetDlgItem(hwnd, IDC_USE_DEFAULT));

    switch (iTab)
    {
    case TAB_DISPLAY:
        DisplayDialogToOptions(hwnd, o);
        break;
    case TAB_MISC:
        MiscDialogToOptions(hwnd, o);
        break;
    }
}

void NewOptionsToDialog(HWND hwnd, int iTab, options_type *o)
{
    /* Use DEFAULTS Checkbox */
    Button_SetCheck(GetDlgItem(hwnd, IDC_USE_DEFAULT), o->use_default);

    if (o->use_default)
        o = GetDefaultOptions();

    switch (iTab)
    {
    case TAB_DISPLAY:
        OptionsToDisplayDialog(hwnd, o);
        break;

    case TAB_MISC:
        OptionsToMiscDialog(hwnd, o);
        break;

    default:
        /* Do 'em both??? */
        OptionsToDisplayDialog(hwnd, o);
        OptionsToMiscDialog(hwnd, o);
    }
}


/***************************************************************************
    Internal functions
 ***************************************************************************/

static int ViewStringToVal(char *ptr)
{
    int i;

    for (i = VIEW_LARGE_ICONS; i < VIEW_MAX; i++)
        if (strcmp(ptr,view_modes[i]) == 0)
            return i;
    return VIEW_REPORT;
}

void DisplayDialogToOptions(HWND hwnd, options_type *o)
{
    char buf[100];
    int index;

    /* Game Display Options box */

    /* Display mode control */
    o->is_window = ComboBox_GetCurSel(GetDlgItem(hwnd, IDC_DISPLAYTYPE));

    /* Screen size control */
    index = ComboBox_GetCurSel(GetDlgItem(hwnd, IDC_SIZES));
    o->display_best_fit = (index > 0) ? FALSE : TRUE;

    if (! o->display_best_fit)
    {
        char    buf[100];
        DWORD   w, h;

        ComboBox_GetText(GetDlgItem(hwnd, IDC_SIZES), buf, 100);
        if (sscanf(buf, "%d x %d", &w, &h) == 2)
        {
            o->width  = w;
            o->height = h;
        }
        else
        {
            o->display_best_fit = TRUE;
        }
    }

    /* Color Depth control, values can be 8 or 16 (BitsPerPixel) */
    o->depth = (ComboBox_GetCurSel(GetDlgItem(hwnd, IDC_DEPTH))) ? 16 : 8;

    /* Skip lines control */
    Edit_GetText(GetDlgItem(hwnd, IDC_SKIP_LINES), buf, 100);
    if (sscanf(buf, "%d", &o->skip_lines) != 1)
        o->skip_lines = 0;

    /* Skip columns control */
    Edit_GetText(GetDlgItem(hwnd, IDC_SKIP_COLUMNS), buf, 100);
    if (sscanf(buf, "%d", &o->skip_columns) != 1)
        o->skip_columns = 0;

    /* Direct Draw, Dirty Rect (draw only changes) controls */
    o->window_ddraw  = Button_GetCheck(GetDlgItem(hwnd, IDC_WINDOW_DDRAW));
    o->use_dirty   = Button_GetCheck(GetDlgItem(hwnd, IDC_DIRTY));

    /* Double display size, values can be 1 = no or 2 = yes */
    o->scale = (Button_GetCheck(GetDlgItem(hwnd, IDC_DOUBLE))) ? 2 : 1;

    /* Blit and scanline controls */
    o->use_blit    = Button_GetCheck(GetDlgItem(hwnd, IDC_USEBLIT));
    o->hscan_lines = Button_GetCheck(GetDlgItem(hwnd, IDC_SCANLINES));
    o->vscan_lines = Button_GetCheck(GetDlgItem(hwnd, IDC_VSCANLINES));

    /* Frame Skipping Options box */
    
    /* Frame skip control */
    o->frame_skip = ComboBox_GetCurSel(GetDlgItem(hwnd, IDC_FRAMESKIP));

    /* Vector Options box */

    /* Double vector game control */
    o->double_vector = Button_GetCheck(GetDlgItem(hwnd, IDC_VECTOR_DOUBLE));

    /* Gamma control */
    {
        UINT nValue;
        nValue = SendMessage(GetDlgItem(hwnd, IDC_GAMMA), TBM_GETPOS, 0, 0);
        o->gamma = nValue / 20.0 + 0.5;
        /* Set the static display to the new value */
        sprintf(buf, "%03.02f", o->gamma);
        Static_SetText(GetDlgItem(hwnd, IDC_GAMMADISP), buf);
    }

    /* Brightness control */
    o->brightness = SendMessage(GetDlgItem(hwnd, IDC_BRIGHTNESS), TBM_GETPOS, 0, 0);

    /* Set the static display to the new value to match */
    sprintf(buf, "%3d%%", o->brightness);
    Static_SetText(GetDlgItem(hwnd, IDC_BRIGHTNESSDISP), buf);


    /* Anti-Alias control */
    o->antialias = Button_GetCheck(GetDlgItem(hwnd, IDC_ANTIALIAS));

    /* Translucency control */
    o->translucency = Button_GetCheck(GetDlgItem(hwnd, IDC_TRANSLUCENCY));

    /* Beam control */
    {
        UINT nValue;
        nValue = SendMessage(GetDlgItem(hwnd, IDC_BEAM), TBM_GETPOS, 0, 0);
        o->beam = nValue / 20.0 + 1.0;
        /* Set the static display to the new value */
        sprintf(buf, "%03.02f", o->beam);
        Static_SetText(GetDlgItem(hwnd, IDC_BEAMDISP), buf);
    }

    /* Flicker slider control */
    o->flicker = SendMessage(GetDlgItem(hwnd, IDC_FLICKER), TBM_GETPOS, 0, 0);
    /* Now set the static flicker display the new value to match */
    sprintf(buf, "%3d", o->flicker);
    Static_SetText(GetDlgItem(hwnd, IDC_FLICKERDISP), buf);

    /* Rotate Options box */

    /* Rotate control */
    o->rotate  = ComboBox_GetCurSel(GetDlgItem(hwnd, IDC_ROTATE));

    /* Flip x/y controls */
    o->flipx   = Button_GetCheck(GetDlgItem(hwnd, IDC_FLIPX));
    o->flipy   = Button_GetCheck(GetDlgItem(hwnd, IDC_FLIPY));
}

void MiscDialogToOptions(HWND hwnd, options_type *o)
{
    char buf[100];

    /* Sound Options box */

    /* Sound System control */
    o->sound        = ComboBox_GetCurSel(GetDlgItem(hwnd, IDC_SOUNDTYPE));

    /* Seal Audio Device control */
    o->seal_index   = ComboBox_GetCurSel(GetDlgItem(hwnd, IDC_AUDIODEV));

    /* Sample rate control */
    ComboBox_GetText(GetDlgItem(hwnd, IDC_SAMPLERATE), buf, 8);
    if (sscanf(buf, "%d", &o->sample_rate) != 1)
        o->sample_rate = 22050; /* default 22050 */

    /* Sample bits control */
    ComboBox_GetText(GetDlgItem(hwnd, IDC_SAMPLEBITS), buf, 8);
    if (sscanf(buf, "%d", &o->sample_bits) != 1)
        o->sample_bits = 8; /* default 8 bits */

    /* Stereo control */
    o->stereo       = Button_GetCheck(GetDlgItem(hwnd, IDC_STEREO));

    /* YM-3812 FM Synthesis control */
    o->fm_ym3812    = Button_GetCheck(GetDlgItem(hwnd, IDC_USE_FM_YM3812));

    /* Input Devices Options box */
    
    /* Use joystick, Trackball / spin inputs controls */
    o->use_joystick = Button_GetCheck(GetDlgItem(hwnd, IDC_JOYSTICK));
    o->use_ai_mouse = Button_GetCheck(GetDlgItem(hwnd, IDC_AI_MOUSE));

    /* Direct Input Options box */

    /* Direct keyboard and Joystick controls */
    o->di_keyboard  = Button_GetCheck(GetDlgItem(hwnd, IDC_DIKEYBOARD));
    o->di_joystick  = Button_GetCheck(GetDlgItem(hwnd, IDC_DIJOYSTICK));

    /* Miscellaneous Options box */

    /* Enable cheats, Auto-Pause, Errlog and MMX check boxes */
    o->cheat        = Button_GetCheck(GetDlgItem(hwnd, IDC_CHEAT));
    o->auto_pause   = Button_GetCheck(GetDlgItem(hwnd, IDC_AUTO_PAUSE));
    o->disable_mmx  = Button_GetCheck(GetDlgItem(hwnd, IDC_DISABLE_MMX));
    o->error_log    = Button_GetCheck(GetDlgItem(hwnd, IDC_LOG));
    o->profile      = Button_GetCheck(GetDlgItem(hwnd, IDC_PROFILER));
}


/* Set the controls in the Display Options Dialog to match 'o' */
void OptionsToDisplayDialog(HWND hwnd, options_type *o)
{
    char buf[100];

    /* Game Display Options box */

    ComboBox_SetCurSel(GetDlgItem(hwnd, IDC_DISPLAYTYPE),o->is_window & 1);

    /* Adjust Available options in the Display Size control */
    /* This must be done before setting depth? */
    UpdateDisplayModeUI(hwnd, o->depth); 

    /* The value can be 8 or 16, so shifting right by 4 yields 0 or 1 */
    ComboBox_SetCurSel(GetDlgItem(hwnd, IDC_DEPTH), (o->depth >> 4) & 1);
    UpdateDisplayModeUI(hwnd, o->depth);

    /* Screen size drop down list */
    if (o->display_best_fit)
    {
        ComboBox_SetCurSel(GetDlgItem(hwnd, IDC_SIZES), 0);
    }
    else
    {
        /* Select the mode in the list. */
        int nSelection = 0;
        int nCount = 0;

        /* Get the number of items in the control */
        nCount = ComboBox_GetCount(GetDlgItem(hwnd, IDC_SIZES));

        while (0 < nCount--)
        {
            char    buf[100];
            int     nWidth, nHeight;

            /* Get the screen size */
            ComboBox_GetLBText(GetDlgItem(hwnd, IDC_SIZES), nCount, buf);

            if (sscanf(buf, "%d x %d", &nWidth, &nHeight) == 2)
            {
                /* If we match, set nSelection to the right value */
                if (o->width  == nWidth &&  o->height == nHeight)
                {
                    nSelection = nCount;
                    break;
                }
            }
        }
        ComboBox_SetCurSel(GetDlgItem(hwnd, IDC_SIZES), nSelection);
    }

    /* Send message to skip lines control */
    SendDlgItemMessage(hwnd, IDC_SKIP_LINES_SPIN, UDM_SETPOS, 0,
                        (LPARAM)MAKELONG(o->skip_lines, 0));

    /* Send message to skip columns control */
    SendDlgItemMessage(hwnd, IDC_SKIP_COLUMNS_SPIN, UDM_SETPOS, 0,
                        (LPARAM)MAKELONG(o->skip_columns, 0));

    /* Gamma */
    {
        UINT nValue;
        nValue = (UINT)((o->gamma - 0.5) * 20.0);
        /* Notify the Gamma control */
        SendMessage(GetDlgItem(hwnd, IDC_GAMMA), TBM_SETPOS,
                    (WPARAM)TRUE, 
                    (LPARAM)nValue);

        /* Set the static display to the new value */
        sprintf(buf, "%03.02f", o->gamma);
        Static_SetText(GetDlgItem(hwnd, IDC_GAMMADISP), buf);
    }

    /* Brightness */
    SendMessage(GetDlgItem(hwnd, IDC_BRIGHTNESS), TBM_SETPOS,
                (WPARAM)TRUE, 
                (LPARAM)o->brightness);
    /* Set the static display to the new value */
    sprintf(buf, "%3d%%", o->brightness);
    Static_SetText(GetDlgItem(hwnd, IDC_BRIGHTNESSDISP), buf);


    /* Direct Draw, Dirty Rect (draw only changes) & double size controls */
    Button_SetCheck(GetDlgItem(hwnd, IDC_WINDOW_DDRAW), o->window_ddraw);
    Button_SetCheck(GetDlgItem(hwnd, IDC_DIRTY),        o->use_dirty);
    Button_SetCheck(GetDlgItem(hwnd, IDC_DOUBLE),       o->scale == 2);
    
    /* Blit and scanline controls */
    Button_SetCheck(GetDlgItem(hwnd, IDC_USEBLIT),      o->use_blit);
    Button_SetCheck(GetDlgItem(hwnd, IDC_SCANLINES),    o->hscan_lines);
    Button_SetCheck(GetDlgItem(hwnd, IDC_VSCANLINES),   o->vscan_lines);

    /* Frame Skip control */
    ComboBox_SetCurSel(GetDlgItem(hwnd, IDC_FRAMESKIP), o->frame_skip);

    /* Vector Options box */

    /* Double vector game size */
    Button_SetCheck(GetDlgItem(hwnd, IDC_VECTOR_DOUBLE),o->double_vector);

    /* Anti-aliasing, and Translucency controls */
    Button_SetCheck(GetDlgItem(hwnd, IDC_ANTIALIAS),    o->antialias);
    Button_SetCheck(GetDlgItem(hwnd, IDC_TRANSLUCENCY), o->translucency);

    /* Notify the Flicker control */
    SendMessage(GetDlgItem(hwnd, IDC_FLICKER), TBM_SETPOS,
                (WPARAM)TRUE, 
                (LPARAM)o->flicker);
    /* Now set the static flicker display the new value to match */
    sprintf(buf, "%3d", o->flicker);
    Static_SetText(GetDlgItem(hwnd, IDC_FLICKERDISP), buf);

    /* Beam */
    {
        UINT nValue;
        nValue = (UINT)((o->beam - 1.0) * 20.0);
        /* Notify the Beam control */
        SendMessage(GetDlgItem(hwnd, IDC_BEAM), TBM_SETPOS,
                    (WPARAM)TRUE, 
                    (LPARAM)nValue);

        /* Set the static display to the new value */
        sprintf(buf, "%03.02f", o->beam);
        Static_SetText(GetDlgItem(hwnd, IDC_BEAMDISP), buf);
    }

    /* Rotation Options box */

    /* Rotate and flip controls */
    ComboBox_SetCurSel(GetDlgItem(hwnd, IDC_ROTATE),    o->rotate);
    Button_SetCheck(GetDlgItem(hwnd, IDC_FLIPX),        o->flipx);
    Button_SetCheck(GetDlgItem(hwnd, IDC_FLIPY),        o->flipy);
}

/* Set the controls in the Sound/Input Dialog to match 'o' */
void OptionsToMiscDialog(HWND hwnd, options_type *o)
{
    /* Sound Options box */

    /* Sound System and Audio device controls */
    ComboBox_SetCurSel(GetDlgItem(hwnd, IDC_SOUNDTYPE), o->sound);
    ComboBox_SetCurSel(GetDlgItem(hwnd, IDC_AUDIODEV),  o->seal_index);

    /* Sample rate control
       (Sample rates = Control values)
       11025 = 0, 22050 = 1, 44100 = 2
     */
    ComboBox_SetCurSel(GetDlgItem(hwnd, IDC_SAMPLERATE),
        (o->sample_rate / 22050) % 3);

    /* The value can be 8 or 16, so shifting right by 4 yields 0 or 1 */
    ComboBox_SetCurSel(GetDlgItem(hwnd, IDC_SAMPLEBITS),
        (o->sample_bits >> 4) & 1);

    /* Stereo control */
    Button_SetCheck(GetDlgItem(hwnd, IDC_STEREO), o->stereo);

    /* Use YM-3812 FM Synthesis control */
    Button_SetCheck(GetDlgItem(hwnd, IDC_USE_FM_YM3812), o->fm_ym3812);

    /* Input Devices Options box */

    /* Use joystick, Trackball / spin inputs controls */
    Button_SetCheck(GetDlgItem(hwnd, IDC_JOYSTICK),     o->use_joystick);
    Button_SetCheck(GetDlgItem(hwnd, IDC_AI_MOUSE),     o->use_ai_mouse);
    Button_SetCheck(GetDlgItem(hwnd, IDC_AI_JOYSTICK), !o->use_ai_mouse);

    /* Direct Input Options box */

    /* Direct Joystick and Keyboard controls */
    Button_SetCheck(GetDlgItem(hwnd, IDC_DIKEYBOARD),   o->di_keyboard);
    Button_SetCheck(GetDlgItem(hwnd, IDC_DIJOYSTICK),   o->di_joystick);

    /* Miscellaneous Options box */

    /* Enable cheats, Auto-Pause, Errlog and MMX check boxes */
    Button_SetCheck(GetDlgItem(hwnd, IDC_CHEAT),        o->cheat);
    Button_SetCheck(GetDlgItem(hwnd, IDC_AUTO_PAUSE),   o->auto_pause);
    Button_SetCheck(GetDlgItem(hwnd, IDC_DISABLE_MMX),  o->disable_mmx);
    Button_SetCheck(GetDlgItem(hwnd, IDC_LOG),          o->error_log);
    Button_SetCheck(GetDlgItem(hwnd, IDC_PROFILER),     o->profile);
}

void LoadOptions(void)
{
    HKEY  hKey;
    DWORD value;
    LONG  result;
    char  keyString[80];
    int   i;
    char  *ptr;

    /* Get to HKEY_CURRENT_USER\Software\Freeware\Mame32 */
    result = RegOpenKeyEx(HKEY_CURRENT_USER,KEY_BASE,
                          0, KEY_QUERY_VALUE, &hKey);

    if (result == ERROR_SUCCESS)
    {
        if ((ptr = GetRegStringOption(hKey, "DefaultGame")) != NULL)
            strcpy(settings.default_game,ptr);
//        GetRegBoolOption(hKey, "ShowAllGames", &settings.show_all_games);
//        GetRegBoolOption(hKey, "ShowUnavailableGames", &settings.show_unavailable_games);

        if ((value = GetRegOption(hKey, "FolderID")) != -1)
            settings.folder_id = ((int)value < 0) ? 0 : value;

        if ((ptr = GetRegStringOption(hKey, "ListMode")) != NULL)
            settings.view = ViewStringToVal(ptr);

        else if ((value = GetRegOption(hKey, "ListDetails")) != -1)
        {   // Cleanup old style reg key
            settings.view = value + 2;
            RegDeleteValue(hKey,"ListDetails");
        }

        GetRegBoolOption(hKey, "ShowScreenShot",  &settings.show_screenshot);
        GetRegBoolOption(hKey, "LargeScreenShot",  &settings.large_screenshot);
//        GetRegBoolOption(hKey, "ShowNoClones",  &settings.show_no_clones);
        GetRegBoolOption(hKey, "ShowFolderList",  &settings.show_folderlist);
        GetRegBoolOption(hKey, "ShowStatusBar",  &settings.show_toolbar);
        GetRegBoolOption(hKey, "ShowToolBar",  &settings.show_statusbar);
        if ((value = GetRegOption(hKey, "X")) != -1)
            settings.area.x = ((int)value < 0) ? 0 : value;
        if ((value = GetRegOption(hKey, "Y")) != -1)
            settings.area.y = ((int)value < 0) ? 0 : value;
        if ((value = GetRegOption(hKey, "Width")) != -1)
            settings.area.width = ((int)value < 0) ? 0 : value;
        if ((value = GetRegOption(hKey, "Height")) != -1)
            settings.area.height = ((int)value < 0) ? 0 : value;
        if ((value = GetRegOption(hKey, "SortColumn")) != -1)
            settings.sort_column = ((int)value < 0) ? 0 : value;
        if ((value = GetRegOption(hKey, "FontColor")) != -1)
        {
            if (value == GetSysColor(COLOR_WINDOWTEXT))
                settings.list_font_color = (COLORREF)-1;
            else
                settings.list_font_color = value;
        }
        
        /* Only get the column width if detailed is true */
        if (settings.view == VIEW_REPORT)
        {
            if ((ptr = GetRegStringOption(hKey,"ColumnWidths")) != NULL)
                ColumnDecodeString(settings.column_widths, ptr);
            else
            {
                char temp[20];
                int  i;
                for (i = 0; i < COLUMN_MAX; i++)
                {
                    sprintf(temp, "Column%i", i);
                    if ((value = GetRegOption(hKey, temp)) != -1)
                    {   // Cleanup old style reg key, don't use 'em
                        // settings.column_widths[i] = value;
                        RegDeleteValue(hKey, temp);
                    }
                }
            }
        }
        if ((ptr = GetRegStringOption(hKey,"Splitters")) != NULL)
                SplitterDecodeString(settings.splitter, ptr);
        if ((ptr = GetRegStringOption(hKey, "ListFont")) != NULL)
            FontDecodeString(ptr, &settings.list_font);
        if ((ptr = GetRegStringOption(hKey, "ColumnOrder")) != NULL)
            ColumnDecodeString(settings.column_order, ptr);
        if ((ptr = GetRegStringOption(hKey, "ColumnShown")) != NULL)
            ColumnDecodeString(settings.column_shown, ptr);
        
        // TODO: should have different GetRegStringOption to get unlimited size dir paths
        if ((ptr = GetRegStringOption(hKey, "RomDirs")) != NULL)
            SetRomDirs(ptr);
        if ((ptr = GetRegStringOption(hKey, "SampleDirs")) != NULL)
            SetSampleDirs(ptr);
        if ((ptr = GetRegStringOption(hKey, "CfgDir")) != NULL)
            SetCfgDir(ptr);
        if ((ptr = GetRegStringOption(hKey, "HiDir")) != NULL)
            SetHiDir(ptr);
        if ((ptr = GetRegStringOption(hKey, "InpDir")) != NULL)
            SetInpDir(ptr);
        if ((ptr = GetRegStringOption(hKey, "ImgDir")) != NULL)
            SetImgDir(ptr);
        if ((ptr = GetRegStringOption(hKey, "StateDir")) != NULL)
            SetStateDir(ptr);
        if ((ptr = GetRegStringOption(hKey, "ArtDir")) != NULL)
            SetArtDir(ptr);

        RegCloseKey(hKey);
    }

    sprintf(keyString,KEY_FMT,"Defaults");
    result = RegOpenKeyEx(HKEY_CURRENT_USER, keyString, 0, KEY_QUERY_VALUE, &hKey);
    if (result == ERROR_SUCCESS)
    {
        LoadRegGameOptions(hKey, &global);
        RegCloseKey(hKey);
    }

    for (i = 0 ; i < num_games; i++)
    {
        sprintf(keyString,KEY_FMT,drivers[i]->name);
        result = RegOpenKeyEx(HKEY_CURRENT_USER, keyString, 0, KEY_QUERY_VALUE, &hKey);
        if (result == ERROR_SUCCESS)
        {
            LoadRegGameOptions(hKey, &game[i]);
            RegCloseKey(hKey);
        }
    }
}

void LoadRegGameOptions(HKEY hKey, options_type *o)
{
    char  *ptr;
    DWORD value;
    DWORD size;

    if ((value = GetRegOption(hKey, "PlayCount")) != -1)
        o->play_count = value;
    if ((value = GetRegOption(hKey, "ROMS")) != -1)
        o->has_roms = value;
    if ((value = GetRegOption(hKey, "Samples")) != -1)
        o->has_samples = value;

    // look for IsWindow.  If it's not there, then use default options for this game
    if (RegQueryValueEx(hKey, "IsWindow", 0, &value, NULL, &size) != ERROR_SUCCESS)
       return;

    o->use_default = FALSE;

    GetRegBoolOption(hKey, "IsWindow", &o->is_window);
    GetRegBoolOption(hKey, "BestFit",  &o->display_best_fit);

    GetRegBoolOption(hKey, "DoubleVector", &o->double_vector);
    GetRegBoolOption(hKey, "WindowDDraw",  &o->window_ddraw);
    GetRegBoolOption(hKey, "Dirty",        &o->use_dirty);
    GetRegBoolOption(hKey, "HScanLines",   &o->hscan_lines);
    GetRegBoolOption(hKey, "VScanLines",   &o->vscan_lines);
    GetRegBoolOption(hKey, "UseBlit",      &o->use_blit);
    GetRegBoolOption(hKey, "FlipX",        &o->flipx);
    GetRegBoolOption(hKey, "FlipY",        &o->flipy);
    GetRegBoolOption(hKey, "AntiAlias",    &o->antialias);
    GetRegBoolOption(hKey, "Translucency", &o->translucency);    

    GetRegBoolOption(hKey, "UseJoystick",         &o->use_joystick);
    GetRegBoolOption(hKey, "UseAnalogInputMouse", &o->use_ai_mouse);
    GetRegBoolOption(hKey, "DiKeyboard",          &o->di_keyboard);
    GetRegBoolOption(hKey, "DiJoystick",          &o->di_joystick);

    GetRegBoolOption(hKey, "Cheat",      &o->cheat);
    GetRegBoolOption(hKey, "AutoPause",  &o->auto_pause);
    GetRegBoolOption(hKey, "DisableMMX", &o->disable_mmx);
    GetRegBoolOption(hKey, "ErrorLog",   &o->error_log);
    GetRegBoolOption(hKey, "Profile",    &o->profile);

    if ((value = GetRegOption(hKey, "Scale")) != -1)
        o->scale = value;
    if ((value = GetRegOption(hKey, "Rotate")) != -1)
        o->rotate = value;
    if ((value = GetRegOption(hKey, "Width")) != -1)
        o->width = value;
    if ((value = GetRegOption(hKey, "Height")) != -1)
        o->height = value;
    if ((value = GetRegOption(hKey, "Sound")) != -1)
        o->sound = value;
    if ((value = GetRegOption(hKey, "SealSound")) != -1)
        o->seal_index = value;
    if ((value = GetRegOption(hKey, "SampleRate")) != -1)
        o->sample_rate = value;
    if ((value = GetRegOption(hKey, "SampleBits")) != -1)
        o->sample_bits = value;
    if ((value = GetRegOption(hKey, "Stereo")) != -1)
        o->stereo = value;
    if ((value = GetRegOption(hKey, "FMYM3812")) != -1)
        o->fm_ym3812 = value;
    if ((value = GetRegOption(hKey, "FrameSkip")) != -1)
        o->frame_skip = value;
    if ((value = GetRegOption(hKey, "Depth")) != -1)
        o->depth = value;
    if ((value = GetRegOption(hKey, "Flicker")) != -1)
        o->flicker = value;
    if ((value = GetRegOption(hKey, "SkipLines")) != -1)
        o->skip_lines = value;
    if ((value = GetRegOption(hKey, "SkipColumns")) != -1)
        o->skip_columns = value;
    if ((value = GetRegOption(hKey, "Brightness")) != -1)
        o->brightness = value;

    if ((ptr = GetRegStringOption(hKey, "Gamma")) != NULL)
        sscanf(ptr, "%lf", &o->gamma);

    if ((ptr = GetRegStringOption(hKey, "Beam")) != NULL)
        sscanf(ptr, "%lf", &o->beam);
}


DWORD GetRegOption(HKEY hKey, char *name)
{
    DWORD dwType;
    DWORD dwSize;
    DWORD value = -1;

    if (RegQueryValueEx(hKey, name, 0, &dwType, NULL, &dwSize) == ERROR_SUCCESS)
    {
        if (dwType == REG_DWORD)
        {
            dwSize = 4;
            RegQueryValueEx(hKey, name, 0, &dwType, (unsigned char *)&value, &dwSize);
        }
    }
    return value;
}

void GetRegBoolOption(HKEY hKey, char *name, BOOL *value)
{
    char *ptr;

    if ((ptr = GetRegStringOption(hKey, name)) != NULL)
    {
        *value = (*ptr == '0') ? FALSE : TRUE;
    }
}

char *GetRegStringOption(HKEY hKey, char *name)
{
    DWORD dwType;
    DWORD dwSize;
    static char str[300];

    memset(str,'\0',300);

    if (RegQueryValueEx(hKey, name, 0, &dwType, NULL, &dwSize) == ERROR_SUCCESS)
    {
        if (dwType == REG_SZ)
        {
            dwSize = 299;
            RegQueryValueEx(hKey, name, 0, &dwType, (unsigned char *)str, &dwSize);
        }
    }
    else
    {
        return NULL;
    }

    return str;
}

static void SavePlayCount(int game_index)
{
    HKEY  hKey, hSubkey;
    DWORD dwDisposition = 0;
    LONG  result;
    char  keyString[300];

    /* Get to HKEY_CURRENT_USER\Software\Freeware\Mame32 */
    result = RegCreateKeyEx(HKEY_CURRENT_USER,KEY_BASE,
                            0, "", REG_OPTION_NON_VOLATILE,
                            KEY_ALL_ACCESS, NULL, &hKey, &dwDisposition);
    if (result == ERROR_SUCCESS)
    {
        sprintf(keyString,KEY_FMT,drivers[game_index]->name);
        result = RegCreateKeyEx(HKEY_CURRENT_USER, keyString,
                                0, "", REG_OPTION_NON_VOLATILE,
                                KEY_ALL_ACCESS, NULL, &hSubkey, &dwDisposition);

        if (result == ERROR_SUCCESS)
        {
            PutRegOption(hKey, "PlayCount", game[game_index].play_count);
            RegCloseKey(hSubkey);
        }
        RegCloseKey(hKey);
    }
}

void SaveOptions(void)
{
    HKEY  hKey, hSubkey;
    DWORD dwDisposition = 0;
    LONG  result;
    char  keyString[300];
    int   i;
    BOOL  saved;

    /* Get to HKEY_CURRENT_USER\Software\Freeware\Mame32 */
    result = RegCreateKeyEx(HKEY_CURRENT_USER,KEY_BASE,
                            0, "", REG_OPTION_NON_VOLATILE,
                            KEY_ALL_ACCESS, NULL, &hKey, &dwDisposition);

    if (result == ERROR_SUCCESS)
    {
        PutRegStringOption(hKey, "SaveVersion", "0.34");
        PutRegStringOption(hKey, "DefaultGame", settings.default_game);
        //PutRegBoolOption(hKey, "ShowAllGames", settings.show_all_games);
        //PutRegBoolOption(hKey, "ShowUnavailableGames", settings.show_unavailable_games);
        PutRegOption(hKey,"FolderID", settings.folder_id);
        PutRegStringOption(hKey, "ListMode", view_modes[settings.view]);
        PutRegBoolOption(hKey, "ShowScreenShot", settings.show_screenshot);
        PutRegBoolOption(hKey, "LargeScreenShot", settings.large_screenshot);
        //PutRegBoolOption(hKey, "ShowNoClones", settings.show_no_clones);
        PutRegBoolOption(hKey, "ShowToolBar", settings.show_toolbar);
        PutRegBoolOption(hKey, "ShowStatusBar", settings.show_statusbar);
        PutRegBoolOption(hKey, "ShowFolderList", settings.show_folderlist);
        PutRegOption(hKey,"SortColumn", settings.sort_column);
        PutRegOption(hKey,"X", settings.area.x);
        PutRegOption(hKey,"Y", settings.area.y);
        PutRegOption(hKey,"Width", settings.area.width);
        PutRegOption(hKey,"Height", settings.area.height);
        ColumnEncodeString(settings.column_widths, keyString);
        PutRegStringOption(hKey, "ColumnWidths", keyString);
        SplitterEncodeString(settings.splitter, keyString);
        PutRegStringOption(hKey, "Splitters", keyString);

        if (settings.list_font_color != (COLORREF)-1 )
            PutRegOption(hKey, "FontColor", settings.list_font_color);
        else
            RegDeleteValue(hKey, "FontColor");

        FontEncodeString(&settings.list_font, keyString);
        PutRegStringOption(hKey, "ListFont", keyString);
        ColumnEncodeString(settings.column_order, keyString);
        PutRegStringOption(hKey, "ColumnOrder", keyString);
        ColumnEncodeString(settings.column_shown, keyString);
        PutRegStringOption(hKey, "ColumnShown", keyString);
        PutRegStringOption(hKey, "RomDirs",    settings.romdirs);
        PutRegStringOption(hKey, "SampleDirs", settings.sampledirs);
        PutRegStringOption(hKey, "CfgDir",     settings.cfgdir);
        PutRegStringOption(hKey, "HiDir",      settings.hidir);
        PutRegStringOption(hKey, "InpDir",     settings.inpdir);
        PutRegStringOption(hKey, "ImgDir",     settings.imgdir);
        PutRegStringOption(hKey, "StateDir",   settings.statedir);
        PutRegStringOption(hKey, "ArtDir",     settings.artdir);
    }

    sprintf(keyString,KEY_FMT,"Defaults");
    result = RegCreateKeyEx(HKEY_CURRENT_USER, keyString,
                            0, "", REG_OPTION_NON_VOLATILE,
                            KEY_ALL_ACCESS, NULL, &hSubkey, &dwDisposition);
    if (result == ERROR_SUCCESS)
    {
        global.use_default = FALSE;
        SaveRegGameOptions(hSubkey, &global);
        RegCloseKey(hSubkey);
    }

    for (i = 0 ; i < num_games; i++)
    {
        sprintf(keyString,KEY_FMT,drivers[i]->name);
        result = RegCreateKeyEx(HKEY_CURRENT_USER, keyString,
                                0, "", REG_OPTION_NON_VOLATILE,
                                KEY_ALL_ACCESS, NULL, &hSubkey, &dwDisposition);

        if (result == ERROR_SUCCESS)
        {
            saved = SaveRegGameOptions(hSubkey, &game[i]);
            RegCloseKey(hSubkey);
            if (saved == FALSE)
                RegDeleteKey(hKey,drivers[i]->name);
        }
    }
    RegCloseKey(hKey);
}

BOOL SaveRegGameOptions(HKEY hKey, options_type *o)
{
    char  str[30];

    PutRegOption(hKey, "PlayCount", o->play_count);
    
    PutRegOption(hKey, "ROMS", o->has_roms);
    PutRegOption(hKey, "Samples", o->has_samples);

    if (o->use_default == TRUE)
    {
        RegDeleteValue(hKey, "Scale");
        RegDeleteValue(hKey, "IsWindow");
        RegDeleteValue(hKey, "BestFit");
        RegDeleteValue(hKey, "DoubleVector");
        RegDeleteValue(hKey, "WindowDDraw");
        RegDeleteValue(hKey, "Dirty");
        RegDeleteValue(hKey, "HScanLines");
        RegDeleteValue(hKey, "VScanLines");
        RegDeleteValue(hKey, "UseBlit");
        RegDeleteValue(hKey, "UseJoystick");
        RegDeleteValue(hKey, "UseAnalogInputMouse");
        RegDeleteValue(hKey, "UseTrackball");
        RegDeleteValue(hKey, "DiKeyboard");
        RegDeleteValue(hKey, "DiJoystick");
        RegDeleteValue(hKey, "Cheat");
        RegDeleteValue(hKey, "AutoPause");
        RegDeleteValue(hKey, "DisableMMX");
        RegDeleteValue(hKey, "ErrorLog");
        RegDeleteValue(hKey, "Profile");
        RegDeleteValue(hKey, "Scale");
        RegDeleteValue(hKey, "Rotate");
        RegDeleteValue(hKey, "Width");
        RegDeleteValue(hKey, "Height");
        RegDeleteValue(hKey, "Sound");
        RegDeleteValue(hKey, "SealSound");
        RegDeleteValue(hKey, "SampleRate");
        RegDeleteValue(hKey, "SampleBits");
        RegDeleteValue(hKey, "Stereo");
        RegDeleteValue(hKey, "FMYM3812");
        RegDeleteValue(hKey, "FrameSkip");
        RegDeleteValue(hKey, "Gamma");
        RegDeleteValue(hKey, "Brightness");
        RegDeleteValue(hKey, "FlipX");
        RegDeleteValue(hKey, "FlipY");
        RegDeleteValue(hKey, "Depth");
        RegDeleteValue(hKey, "AntiAlias");
        RegDeleteValue(hKey, "Translucency");      
        RegDeleteValue(hKey, "Beam");
        RegDeleteValue(hKey, "Flicker");
        RegDeleteValue(hKey, "SkipLines");
        RegDeleteValue(hKey, "SkipColumns");

        return (o->has_roms != UNKNOWN);
    }

    PutRegBoolOption(hKey, "IsWindow", o->is_window);
    PutRegBoolOption(hKey, "BestFit",  o->display_best_fit);

    PutRegBoolOption(hKey, "DoubleVector", o->double_vector);
    PutRegBoolOption(hKey, "WindowDDraw",  o->window_ddraw);
    PutRegBoolOption(hKey, "Dirty",        o->use_dirty);
    PutRegBoolOption(hKey, "HScanLines",   o->hscan_lines);
    PutRegBoolOption(hKey, "VScanLines",   o->vscan_lines);
    PutRegBoolOption(hKey, "UseBlit",      o->use_blit);
    PutRegBoolOption(hKey, "FlipX",        o->flipx);
    PutRegBoolOption(hKey, "FlipY",        o->flipy);
    PutRegBoolOption(hKey, "AntiAlias",    o->antialias);
    PutRegBoolOption(hKey, "Translucency", o->translucency);   

    PutRegBoolOption(hKey, "UseJoystick",         o->use_joystick);
    PutRegBoolOption(hKey, "UseAnalogInputMouse", o->use_ai_mouse);
    PutRegBoolOption(hKey, "DiKeyboard",          o->di_keyboard);
    PutRegBoolOption(hKey, "DiJoystick",          o->di_joystick);

    PutRegBoolOption(hKey, "Cheat",      o->cheat);
    PutRegBoolOption(hKey, "AutoPause",  o->auto_pause);
    PutRegBoolOption(hKey, "DisableMMX", o->disable_mmx);
    PutRegBoolOption(hKey, "ErrorLog",   o->error_log);
    PutRegBoolOption(hKey, "Profile",    o->profile);

    PutRegOption(hKey, "Scale",       o->scale);
    PutRegOption(hKey, "Rotate",      o->rotate);
    PutRegOption(hKey, "Width",       o->width);
    PutRegOption(hKey, "Height",      o->height);
    PutRegOption(hKey, "Sound",       o->sound);
    PutRegOption(hKey, "SealSound",   o->seal_index);
    PutRegOption(hKey, "SampleRate",  o->sample_rate);
    PutRegOption(hKey, "SampleBits",  o->sample_bits);
    PutRegOption(hKey, "Stereo",      o->stereo);
    PutRegOption(hKey, "FMYM3812",    o->fm_ym3812);
    PutRegOption(hKey, "FrameSkip",   o->frame_skip);
    PutRegOption(hKey, "Depth",       o->depth);
    PutRegOption(hKey, "Flicker",     o->flicker);
    PutRegOption(hKey, "SkipLines",   o->skip_lines);
    PutRegOption(hKey, "SkipColumns", o->skip_columns);
    PutRegOption(hKey, "Brightness",  o->brightness);
    sprintf(str, "%03.02f", o->gamma);
    PutRegStringOption(hKey, "Gamma", str);
    sprintf(str, "%03.02f", o->beam);
    PutRegStringOption(hKey, "Beam", str);

    return TRUE;
}

void PutRegOption(HKEY hKey, char *name, DWORD value)
{
    RegSetValueEx(hKey, name, 0, REG_DWORD, (void *)&value, sizeof(DWORD));
}

void PutRegBoolOption(HKEY hKey, char *name, BOOL value)
{
    char str[2];

    str[0] = (value) ? '1' : '0';
    str[1] = '\0';

    RegSetValueEx(hKey, name, 0, REG_SZ, (void *)str, 2);
}

void PutRegStringOption(HKEY hKey, char *name, char *option)
{
    RegSetValueEx(hKey, name, 0, REG_SZ, (void *)option, strlen(option) + 1);
}

/* End of options.c */
