/* TreeView.c - TreeView support routines */

#include "TreeView.h"
#include "driver.h"
#include "resource.h"
#include "Screenshot.h"

#include <stdio.h>  /* for sprintf */
#include <stdlib.h> /* For malloc and free */
#include <commctrl.h>

char *folderName[8] = {
    "All Games",
    "Available",
    "Unavailable",
    "Working",
    "Non-Working",
    "Manufacturer",
    "Year"
};

enum {
    ICON_FOLDER_OPEN = 0,
    ICON_FOLDER,
    ICON_FOLDER_AVAILABLE,
    ICON_FOLDER_CUSTOM,
    ICON_FOLDER_MANUFACTURER,
    ICON_FOLDER_UNAVAILABLE,
    ICON_FOLDER_YEAR,
    ICON_MANUFACTURER,
    ICON_WORKING,
    ICON_NONWORKING,
    ICON_YEAR,
    ICON_STEREO,
    ICON_NEOGEO
};


typedef struct {
    LPSTR       m_lpTitle;
    FOLDERTYPE  m_nFolderType;
    UINT        m_nFolderId;
    UINT        m_nParent;
    DWORD       m_dwFlags;
    UINT        m_nIconId;
} FOLDERDATA, *LPFOLDERDATA;

#define NUM_FOLDERS 16

FOLDERDATA folderData[NUM_FOLDERS] = {
    {"All Games",   IS_ROOT,  FOLDER_ALLGAMES,    FOLDER_NONE,  0,        ICON_FOLDER},
    {"Available",   IS_ROOT,  FOLDER_AVAILABLE,   FOLDER_NONE,  0,        ICON_FOLDER_AVAILABLE},
    {"Unavaliable", IS_ROOT,  FOLDER_UNAVAILABLE, FOLDER_NONE,  0,        ICON_FOLDER_UNAVAILABLE},
    {"Neo-Geo",     IS_ROOT,  FOLDER_NEOGEO,      FOLDER_NONE,  0,        ICON_NEOGEO},
    {"Manufacturer",IS_ROOT,  FOLDER_MANUFACTURER,FOLDER_NONE,  0,        ICON_FOLDER_MANUFACTURER},
    {"Year",        IS_ROOT,  FOLDER_YEAR,        FOLDER_NONE,  0,        ICON_FOLDER_YEAR},
    {"Working",     IS_ROOT,  FOLDER_WORKING,     FOLDER_NONE,  0,        ICON_WORKING},
    {"Non-Working", IS_ROOT,  FOLDER_NONWORKING,  FOLDER_NONE,  0,        ICON_NONWORKING},
    {"Custom",      IS_ROOT,  FOLDER_CUSTOM,      FOLDER_NONE,  F_CUSTOM, ICON_FOLDER_CUSTOM},   
    {"Favorites",   IS_FOLDER,FOLDER_FAVORITE,    FOLDER_CUSTOM,F_CUSTOM, ICON_FOLDER_CUSTOM},
    {"Originals",   IS_ROOT,  FOLDER_ORIGINAL,    FOLDER_NONE,  0,        ICON_FOLDER},
    {"Clones",      IS_ROOT,  FOLDER_CLONES,      FOLDER_NONE,  0,        ICON_FOLDER},
    {"Raster",      IS_ROOT,  FOLDER_RASTER,      FOLDER_NONE,  0,        ICON_FOLDER},
    {"Vector",      IS_ROOT,  FOLDER_VECTOR,      FOLDER_NONE,  0,        ICON_FOLDER},
    {"Trackball",   IS_ROOT,  FOLDER_TRACKBALL,   FOLDER_NONE,  0,        ICON_FOLDER},
    {"Stereo",      IS_ROOT,  FOLDER_STEREO,      FOLDER_NONE,  0,        ICON_STEREO}
};

TREEFOLDER **   treeFolders = 0;
UINT            numFolders  = 0;

static UINT         folderArrayLength = 0;
static LPTREEFOLDER lpCurrentFolder = 0;
static UINT         nCurrentFolder = 0;

static BOOL     CreateTreeIcons(HWND hWnd);
static char *   FixString(char *s);

/* for subclassing the TreeView */
static WNDPROC g_lpTreeWndProc = 0;

/* De-allocate all folder memory */
void FreeFolders(void)
{
    int             i = 0;
    LPFOLDERDATA    fData = 0;

    if (treeFolders != 0)
    {
        for (i = numFolders - 1; i >= 0; i--)
        {
            DeleteFolder(treeFolders[i]);
            treeFolders[i] = 0;
            numFolders--;
        }
        free(treeFolders);
        treeFolders = 0;
    }
    numFolders = 0;
}

/* Can be called to re-initialize the array of treeFolders */
BOOL InitFolders(UINT nGames)
{
    int             i = 0;
    LPFOLDERDATA    fData = 0;

    if (treeFolders != 0)
    {
        for (i = numFolders - 1; i >= 0; i--)
        {
            DeleteFolder(treeFolders[i]);
            treeFolders[i] = 0;
            numFolders--;
        }
    }
    numFolders = 0;
    if (folderArrayLength == 0)
    {
        folderArrayLength = 200;
        treeFolders = (TREEFOLDER **)malloc(sizeof(TREEFOLDER **) * folderArrayLength);
        if (!treeFolders)
        {
            folderArrayLength = 0;
            return 0;
        }
        else
        {
            memset(treeFolders,'\0', sizeof(TREEFOLDER **) * folderArrayLength);
        }
    }
    for (i = 0; i < NUM_FOLDERS; i++)
    {
        fData = &folderData[i];

        treeFolders[numFolders] = NewFolder(fData->m_lpTitle, fData->m_nFolderType,
                                            fData->m_nFolderId, fData->m_nParent,
                                            fData->m_nIconId, fData->m_dwFlags, nGames);
        if (treeFolders[numFolders])
            numFolders++;
    }

    SetCurrentFolder(treeFolders[0]);

    InitGames(nGames);

    return TRUE;
}

/* Add a folder to the list.  Does not allocate */
BOOL AddFolder(LPTREEFOLDER lpFolder)
{
    UINT i = 0;

    if (!lpFolder)
        return FALSE;

    if (numFolders + 1 >= folderArrayLength)
    {
        TREEFOLDER ** tmpFolders = treeFolders;

        i = folderArrayLength;

        folderArrayLength += 10;

        treeFolders = (TREEFOLDER **)malloc(sizeof(TREEFOLDER **) * folderArrayLength);
        if (!treeFolders)
        {
            folderArrayLength = i;
            treeFolders = tmpFolders;
            return FALSE;
        }
        for (; i < numFolders; i++)
        {
            treeFolders[i] = tmpFolders[i];
        }
        free(tmpFolders);
        for (i = numFolders; i < folderArrayLength; i++)
        {
            treeFolders[i] = 0;
        }
    }

    treeFolders[numFolders] = lpFolder;
    numFolders++;

    return TRUE;
}

/* Remove a folder from the list, but do NOT delete it */
BOOL RemoveFolder(LPTREEFOLDER lpFolder)
{
    int     found = -1;
    UINT    i = 0;

    /* Find the folder */
    for (i = 0; i < numFolders && found == -1; i++)
    {
        if (lpFolder == treeFolders[i])
        {
            found = i;
        }
    }
    if (found != -1)
    {
        numFolders--;
        treeFolders[i] = 0; /* In case we removed the last folder */

        /* Move folders up in the array if needed */
        for (i = (UINT)found; (UINT)found < numFolders; i++)
            treeFolders[i] = treeFolders[i+1];
    }
    return (found != -1) ? TRUE : FALSE;
}

/* Allocate and initialize a NEW TREEFOLDER */
LPTREEFOLDER NewFolder(LPSTR lpTitle, FOLDERTYPE nFolderType,
                       UINT nFolderId, int nParent, UINT nIconId, DWORD dwFlags, UINT nBits)
{
    LPTREEFOLDER lpFolder = (LPTREEFOLDER)malloc(sizeof(TREEFOLDER));
    if (lpFolder)
    {
        memset(lpFolder, '\0', sizeof (TREEFOLDER));
        if (lpFolder->m_lpTitle = (LPSTR)malloc(strlen(lpTitle) + 1))
        {
            strcpy((char *)lpFolder->m_lpTitle,lpTitle);
            lpFolder->m_lpGameBits = 0;
        }
        else
        {
            free(lpFolder);
            return (LPTREEFOLDER)0;
        }
        if (nBits < 8 || (lpFolder->m_lpGameBits = NewBits(nBits)) == NULL)
        {
            free(lpFolder->m_lpTitle);
            free(lpFolder);
            return (LPTREEFOLDER)0;
        }
        lpFolder->m_nFolderId = nFolderId;
        lpFolder->m_nFolderType = nFolderType;
        lpFolder->m_nParent = nParent;
        lpFolder->m_nIconId = nIconId;
        lpFolder->m_dwFlags = dwFlags;
    }
    return lpFolder;
}

/* Deallocate the passed in LPTREEFOLDER */
void DeleteFolder(LPTREEFOLDER lpFolder)
{
    if (lpFolder)
    {
        if (lpFolder->m_lpGameBits)
        {
            DeleteBits(lpFolder->m_lpGameBits);
            lpFolder->m_lpGameBits = 0;
        }
        free(lpFolder->m_lpTitle);
        lpFolder->m_lpTitle = 0;
        free(lpFolder);
        lpFolder = 0;
    }
}

void InitTree(HWND hWnd, UINT nGames)
{
    UINT            i;
    TVINSERTSTRUCT	tvs;
	TVITEM			tvi;

	tvs.hInsertAfter = TVI_SORT;

    if (numFolders == 0)
        InitFolders(nGames);

    CreateTreeIcons( hWnd );

    for (i = 0; i < numFolders; i++)
    {
        LPTREEFOLDER lpFolder = treeFolders[i];

	    if (lpFolder->m_nFolderType == IS_ROOT)
		{
            HTREEITEM   hti;
            UINT        jj;

			tvi.mask    = TVIF_TEXT | TVIF_PARAM | TVIF_IMAGE | TVIF_SELECTEDIMAGE;
			tvs.hParent = TVI_ROOT;
			tvi.pszText = lpFolder->m_lpTitle;
			tvi.lParam  = (LPARAM)lpFolder;
			tvi.iImage	= lpFolder->m_nIconId;
			tvi.iSelectedImage = 0;

			tvs.item = tvi;

			// Add root branch
			hti =	TreeView_InsertItem( hWnd, &tvs );

            if (i == 0 || lpFolder->m_nFolderId == GetSavedFolderID())
                TreeView_SelectItem( hWnd, hti );

			for (jj = 0; jj < numFolders; jj++)
			{
				LPTREEFOLDER tmp = treeFolders[jj];

				if (tmp->m_nParent != FOLDER_NONE &&
                    tmp->m_nParent == lpFolder->m_nFolderId)
				{
                    HTREEITEM shti;

					tvi.mask    = TVIF_TEXT | TVIF_PARAM | TVIF_IMAGE | TVIF_SELECTEDIMAGE;
					tvs.hParent = hti;
					tvi.iImage	= tmp->m_nIconId;
					tvi.iSelectedImage = 0;
					tvi.pszText = tmp->m_lpTitle;
					tvi.lParam	= (LPARAM)tmp;
	
					tvs.item    = tvi;
					// Add it to this tree branch
					shti = TreeView_InsertItem( hWnd, &tvs );
                    if (tmp->m_nFolderId == GetSavedFolderID())
                        TreeView_SelectItem( hWnd, shti );
				}
			}
		}
    }
}

void SetCurrentFolder(LPTREEFOLDER lpFolder)
{
    lpCurrentFolder = (lpFolder == 0) ? treeFolders[0] : lpFolder;
    nCurrentFolder = (lpCurrentFolder) ? lpCurrentFolder->m_nFolderId : 0;
}

LPTREEFOLDER GetCurrentFolder()
{
    return lpCurrentFolder;
}

UINT GetCurrentFolderID()
{
    return nCurrentFolder;
}

LPTREEFOLDER GetFolder(UINT nFolder)
{
    return (nFolder < numFolders) ? treeFolders[nFolder] : (LPTREEFOLDER)0;
}

LPTREEFOLDER GetFolderByID(UINT nID)
{
    UINT i;

    for (i = 0; i < numFolders; i++)
    {
        if (treeFolders[i]->m_nFolderId == nID)
        {
            return treeFolders[i];
        }
    }

    return (LPTREEFOLDER)0;
}

void AddGame(LPTREEFOLDER lpFolder, UINT nGame)
{
    SetBit(lpFolder->m_lpGameBits, nGame);
}

void RemoveGame(LPTREEFOLDER lpFolder, UINT nGame)
{
    ClearBit(lpFolder->m_lpGameBits, nGame);
}

int FindGame(LPTREEFOLDER lpFolder, int nGame)
{
    return FindBit(lpFolder->m_lpGameBits, nGame, TRUE);
}

void InitGames(UINT nGames)
{
    static BOOL firstTime = TRUE;
    UINT    i, jj;
    UINT    tmpNumFolders;
    BOOL *  done = NULL;
    game_data_type * gameData = GetGameData();
    UINT    nFolderId = FOLDER_END;

    if (! numFolders)
        return;

    tmpNumFolders = numFolders;

    for (i = 0; i < tmpNumFolders; i++)
    {
        LPTREEFOLDER lpFolder = treeFolders[i];
        switch(lpFolder->m_nFolderId)
        {
        case FOLDER_ALLGAMES:
            // Clear the bitmask
            SetAllBits( lpFolder->m_lpGameBits, FALSE);
            for (jj = 0; jj < nGames; jj++)
            {
                AddGame(lpFolder, jj);
            }
            break;

        case FOLDER_AVAILABLE:
            SetAllBits( lpFolder->m_lpGameBits, FALSE);
            for (jj = 0; jj < nGames; jj++)
            {
                if (GetHasRoms(jj) == TRUE)
                    AddGame(lpFolder, jj);
            }
            break;

        case FOLDER_UNAVAILABLE:
            SetAllBits( lpFolder->m_lpGameBits, FALSE);
            for (jj = 0; jj < nGames; jj++)
            {
                if (GetHasRoms(jj) == FALSE)
                    AddGame(lpFolder, jj);
            }
            break;
        case FOLDER_NEOGEO:
            SetAllBits( lpFolder->m_lpGameBits, FALSE);
            for (jj = 0; jj < nGames; jj++)
            {
                if (!gameData[jj].in_list)
                    AddGame(lpFolder, jj);
            }
            break;

        case FOLDER_YEAR:
            if (!firstTime)
                break;

            SetAllBits( lpFolder->m_lpGameBits, FALSE);
            done = (LPBOOL)malloc(sizeof(BOOL) * nGames);
            memset(done, '\0', sizeof(BOOL) * nGames);
            for (jj = 0; jj < nGames; jj++)
            {
                LPTREEFOLDER lpTemp = 0;
                UINT j;
                char cTmp[40];

		        if (done[jj])
			        continue;

                strcpy(cTmp,FixString((char *)drivers[jj]->year));
                if (cTmp[4] == '?')
                    cTmp[4] = '\0';

                lpTemp = NewFolder(cTmp, IS_FOLDER, nFolderId++,
                                   FOLDER_YEAR, ICON_YEAR, 0, nGames);
                AddFolder(lpTemp);
                done[jj] = TRUE;
                
                for (j = 0; j < nGames; j++)
                {
                    if (strncmp(cTmp, FixString((char *)drivers[j]->year), 4) == 0)
			        {
				        done[j] = TRUE;
				        AddGame(lpTemp, j);
			        }
                }
            }
            free(done);
            break;

        case FOLDER_MANUFACTURER:
            if (!firstTime)
                break;

            SetAllBits( lpFolder->m_lpGameBits, FALSE);
            done = (LPBOOL)malloc(sizeof(BOOL) * nGames);
            memset(done, '\0', sizeof(BOOL) * nGames);
            for (jj = 0; jj < nGames; jj++)
            {
                LPTREEFOLDER lpTemp = 0;
                UINT j;
                char cTmp[40];

		        if (done[jj])
			        continue;

                strcpy(cTmp,FixString((char *)drivers[jj]->manufacturer));
                lpTemp = NewFolder(cTmp, IS_FOLDER, nFolderId++,
                                   FOLDER_MANUFACTURER, ICON_MANUFACTURER, 0, nGames);
                AddFolder(lpTemp);
                done[jj] = TRUE;
                
                for (j = 0; j < nGames; j++)
                {
                    if (strncmp(cTmp, FixString((char *)drivers[j]->manufacturer), 7) == 0)
			        {
				        done[j] = TRUE;
				        AddGame(lpTemp, j);
			        }
                }
            }
            free(done);
            break;
        case FOLDER_WORKING:
            SetAllBits( lpFolder->m_lpGameBits, FALSE);
            for (jj = 0; jj < nGames; jj++)
            {
                if (!drivers[jj]->flags)
                    AddGame(lpFolder, jj);
            }
            break;
        case FOLDER_NONWORKING:
            SetAllBits( lpFolder->m_lpGameBits, FALSE);
            for (jj = 0; jj < nGames; jj++)
            {
                if (drivers[jj]->flags)
                    AddGame(lpFolder, jj);
            }
            break;
        case FOLDER_ORIGINAL:
            SetAllBits( lpFolder->m_lpGameBits, FALSE);
            for (jj = 0; jj < nGames; jj++)
            {
                if (!drivers[jj]->clone_of || !gameData[jj].in_list)
                    AddGame(lpFolder, jj);
            }
            break;
        case FOLDER_CLONES:
            SetAllBits( lpFolder->m_lpGameBits, FALSE);
            for (jj = 0; jj < nGames; jj++)
            {
                if (drivers[jj]->clone_of && gameData[jj].in_list)
                    AddGame(lpFolder, jj);
            }
            break;
        case FOLDER_RASTER:
            SetAllBits( lpFolder->m_lpGameBits, FALSE);
            for (jj = 0; jj < nGames; jj++)
            {
                if ((drivers[jj]->drv->video_attributes  & VIDEO_TYPE_VECTOR) == 0)
                    AddGame(lpFolder, jj);
            }
            break;
        case FOLDER_VECTOR:
            SetAllBits( lpFolder->m_lpGameBits, FALSE);
            for (jj = 0; jj < nGames; jj++)
            {
                if (drivers[jj]->drv->video_attributes  & VIDEO_TYPE_VECTOR)
                    AddGame(lpFolder, jj);
            }
            break;
        case FOLDER_TRACKBALL:
            SetAllBits( lpFolder->m_lpGameBits, FALSE);
            for (jj = 0; jj < nGames; jj++)
            {
                if (GameUsesTrackball(jj))
                    AddGame(lpFolder, jj);
            }
            break;
        case FOLDER_FAVORITE:
            SetAllBits( lpFolder->m_lpGameBits, FALSE);
            for (jj = 0; jj < nGames; jj++)
            {
                if (GetPlayCount(jj))
                    AddGame(lpFolder, jj);
            }
            break;
        case FOLDER_STEREO:
            SetAllBits( lpFolder->m_lpGameBits, FALSE);
            for (jj = 0; jj < nGames; jj++)
            {
                if (drivers[jj]->drv->sound_attributes & SOUND_SUPPORTS_STEREO)
                    AddGame(lpFolder, jj);
            }
            break;
        }
    }
    firstTime = FALSE;
}

char * FixString(char *s)
{
    static char tmp[40];
    char *      ptr;

    if (*s == '?' || *s == '<')
        strcpy(tmp,"<unknown>");
    else
    {
        int i = 0;

        ptr = s;
        while(*ptr)
        {
            if (*ptr == ' ' && ptr[1] == '(')
                break;

            if (*ptr == '[' || *ptr == ']')
                ptr++;
            else
                tmp[i++] = *ptr++;
            
        }
        tmp[i] = '\0';
    }
    return tmp;
}

HIMAGELIST hTreeSmall;

static char treeIconNames[13][15] = {
    "foldopen",
    "folder",
    "foldavail",
    "custom",
    "foldmanu",
    "foldunav",
    "foldyear",
    "manufact",
    "working",
    "nonwork",
    "year",
    "sound",
    "neo-geo"    
};

/* create iconlist and Treeview control */
static BOOL CreateTreeIcons(HWND hWnd)
{
    HICON   hIcon;
    INT     i;
    HINSTANCE hInst = GetModuleHandle(0);

    hTreeSmall = ImageList_Create (16, 16, ILC_COLORDDB | ILC_MASK, 11, 11);

    for (i = IDI_FOLDER_OPEN; i <= IDI_NEOGEO; i++)
    {
        if ((hIcon = LoadIconFromFile(treeIconNames[i - IDI_FOLDER_OPEN])) == 0)
            hIcon = LoadIcon (hInst, MAKEINTRESOURCE(i));

        if (ImageList_AddIcon (hTreeSmall, hIcon) == -1)
            return FALSE;
    }

    // Be sure that all the small icons were added.
    if (ImageList_GetImageCount (hTreeSmall) < 11)
        return FALSE;

    // Associate the image lists with the list view control.
    TreeView_SetImageList (hWnd, hTreeSmall, LVSIL_NORMAL);
  
    return TRUE;
}

static BOOL TreeCtrlOnPaint(HWND hWnd, UINT uMsg,  WPARAM wParam,  LPARAM lParam)
{

	PAINTSTRUCT ps;
    HDC         hDC = GetDC(hWnd);
    RECT        rcClip, rcClient;
    HDC         memDC;
    HBITMAP     bitmap;

    if (hBitmap == 0)
        return 1;

    BeginPaint(hWnd, &ps);

    GetClipBox(hDC, &rcClip );
	GetClientRect(hWnd, &rcClient);
    
    // Create a compatible memory DC
	memDC = CreateCompatibleDC( hDC );

	// Select a compatible bitmap into the memory DC
    bitmap = CreateCompatibleBitmap( hDC, rcClient.right - rcClient.left, rcClient.bottom - rcClient.top );
	SelectObject( memDC, bitmap );
    
	// First let the control do its default drawing.
    CallWindowProc(g_lpTreeWndProc, hWnd, uMsg, (WPARAM)memDC, 0);
	// DefWindowProc( hWnd, WM_PAINT, (WPARAM)memDC, 0 );

	// Draw bitmap in the background if one has been set
	if( hBitmap != NULL )
    {
        HPALETTE    hPAL;        
        HDC maskDC;
        HBITMAP maskBitmap;
		HDC tempDC;
        HDC imageDC;
        HBITMAP bmpImage;
        int i, j;
        RECT rcRoot;

        // Now create a mask
		maskDC = CreateCompatibleDC(hDC);	
        
		// Create monochrome bitmap for the mask
		maskBitmap = CreateBitmap( rcClient.right - rcClient.left, rcClient.bottom - rcClient.top, 
						1, 1, NULL );
        SelectObject( maskDC, maskBitmap );
		SetBkColor( memDC, GetSysColor( COLOR_WINDOW ) );

		// Create the mask from the memory DC
		BitBlt( maskDC, 0, 0, rcClient.right - rcClient.left, rcClient.bottom - rcClient.top, memDC, 
					rcClient.left, rcClient.top, SRCCOPY );


		tempDC = CreateCompatibleDC(hDC);
        SelectObject(tempDC, hBitmap );

        imageDC = CreateCompatibleDC( hDC );
		bmpImage = CreateCompatibleBitmap( hDC, rcClient.right - rcClient.left, 
						rcClient.bottom - rcClient.top );
        SelectObject(imageDC, bmpImage );

        hPAL = (! hPALbg) ? CreateHalftonePalette(hDC) : hPALbg;

		if( GetDeviceCaps(hDC, RASTERCAPS) & RC_PALETTE && hPAL != NULL )
        {
		    SelectPalette(hDC, hPAL, FALSE );
            RealizePalette(hDC);
			SelectPalette(imageDC, hPAL, FALSE );
        }
        
        // Get x and y offset
        TreeView_GetItemRect( hWnd, TreeView_GetRoot(hWnd), &rcRoot, FALSE );
		rcRoot.left = -GetScrollPos( hWnd, SB_HORZ );

		// Draw bitmap in tiled manner to imageDC
		for( i = rcRoot.left; i < rcClient.right; i += bmDesc.bmWidth )
			for( j = rcRoot.top; j < rcClient.bottom; j += bmDesc.bmHeight )
				BitBlt(imageDC,  i, j, bmDesc.bmWidth, bmDesc.bmHeight, tempDC, 
							0, 0, SRCCOPY );

		// Set the background in memDC to black. Using SRCPAINT with black and any other
		// color results in the other color, thus making black the transparent color
		SetBkColor(memDC, RGB(0,0,0));
        SetTextColor(memDC, RGB(255,255,255));
		BitBlt(memDC, rcClip.left, rcClip.top, rcClip.right - rcClip.left,
                rcClip.bottom - rcClip.top,
                maskDC, rcClip.left, rcClip.top, SRCAND);

		// Set the foreground to black. See comment above.
		SetBkColor(imageDC, RGB(255,255,255));
        SetTextColor(imageDC, RGB(0,0,0));
		BitBlt(imageDC, rcClip.left, rcClip.top, rcClip.right - rcClip.left, 
                rcClip.bottom - rcClip.top, maskDC, 
				rcClip.left, rcClip.top, SRCAND);

		// Combine the foreground with the background
		BitBlt(imageDC, rcClip.left, rcClip.top, rcClip.right - rcClip.left,
                rcClip.bottom - rcClip.top, 
				memDC, rcClip.left, rcClip.top, SRCPAINT);
		// Draw the final image to the screen
        
		BitBlt(hDC, rcClip.left, rcClip.top, rcClip.right - rcClip.left,
                rcClip.bottom - rcClip.top, 
				imageDC, rcClip.left, rcClip.top, SRCCOPY );

        DeleteDC(maskDC);
        DeleteDC(imageDC);
        DeleteDC(tempDC);
        DeleteObject(bmpImage);
        DeleteObject(maskBitmap);
    }
    else
    {
		BitBlt(hDC,  rcClip.left, rcClip.top, rcClip.right - rcClip.left, 
				rcClip.bottom - rcClip.top,
                memDC, rcClip.left, rcClip.top, SRCCOPY );
	}
    DeleteObject(bitmap);
    DeleteDC(memDC);
    EndPaint(hWnd, &ps);
    ReleaseDC(hWnd, hDC);

    return 0;
}

/* Header code - Directional Arrows */
static LRESULT CALLBACK TreeWndProc( HWND hWnd,  UINT uMsg,  WPARAM wParam,  LPARAM lParam)
{
    switch(uMsg)
    {
    case WM_ERASEBKGND:
        if (hBitmap)
            return TRUE;
        break;

    case WM_PAINT:
        if (TreeCtrlOnPaint( hWnd, uMsg, wParam, lParam) == 0)
            return 0;
        break;
    }

    /* message not handled */
    return CallWindowProc(g_lpTreeWndProc, hWnd, uMsg, wParam, lParam);
}


/* Subclass the Treeview Header */
void Tree_Initialize(HWND hWnd)
{
    /* this will subclass the listview (where WM_DRAWITEM gets sent for
       the header control) */
    g_lpTreeWndProc = (WNDPROC)GetWindowLong(hWnd, GWL_WNDPROC);
    SetWindowLong(hWnd, GWL_WNDPROC, (LONG)TreeWndProc);
}


/* End of source file */