/***************************************************************************

    M.A.M.E.32  -  Multiple Arcade Machine Emulator for Win32
    Win32 Portions Copyright (C) 1997-98 Michael Soderstrom and Chris Kirmse
    
    This file is part of MAME32, and may only be used, modified and
    distributed under the terms of the MAME license, in "readme.txt".
    By continuing to use, modify or distribute this file you indicate
    that you have read the license and understand and accept it fully.

 ***************************************************************************/

/***************************************************************************

  Keyboard.c

 ***************************************************************************/

#include "driver.h"

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <windowsx.h>
#include <assert.h>
#include "MAME32.h"
#include "M32Util.h"
#include "Keyboard.h"
#include "status.h"

#define NUMKEYSTATES    256

/***************************************************************************
    function prototypes
 ***************************************************************************/

static int              Keyboard_init(options_type *options);
static void             Keyboard_exit(void);
static int              Keyboard_key_pressed(int keycode);
static int              Keyboard_key_pressed_memory(int keycode);
static int              Keyboard_key_pressed_memory_repeat(int keycode, int speed);
static int              Keyboard_read_key_immediate(void);
static int              Keyboard_read_key(int translate);
static int              Keyboard_read_keyrepeat(int translate);
static const char*      Keyboard_key_name(int keycode);
static BOOL             Keyboard_OnMessage(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam, LRESULT* pResult);
static void             Keyboard_PressKey(int keycode);

/***************************************************************************
    External variables
 ***************************************************************************/

struct OSDKeyboard  Keyboard = 
{
    { Keyboard_init },                      /* init                      */
    { Keyboard_exit },                      /* exit                      */
    { Keyboard_key_pressed },               /* key_pressed               */
    { Keyboard_key_pressed_memory },        /* key_pressed_memory        */
    { Keyboard_key_pressed_memory_repeat }, /* key_pressed_memory_repeat */
    { Keyboard_read_key_immediate },        /* read_key_immediate        */   
    { Keyboard_read_key },                  /* read_key                  */
    { Keyboard_read_keyrepeat },            /* read_keyrepeat            */
    { Keyboard_key_name },                  /* key_name                  */

    { Keyboard_PressKey },                  /* PressKey                  */
    { Keyboard_OnMessage },                 /* OnMessage                 */
};

/***************************************************************************
    Internal structures
 ***************************************************************************/

struct tKeyboard_private
{
    int     m_nInternalKeyPress;
    BYTE    m_Memory[NUMKEYSTATES];
    int     m_nCounter;
};

/***************************************************************************
    Internal variables
 ***************************************************************************/

static struct tKeyboard_private This;

/***************************************************************************
    External OSD functions  
 ***************************************************************************/

/*
    put here anything you need to do when the program is started. Return 0 if 
    initialization was successful, nonzero otherwise.
*/
static int Keyboard_init(options_type *options)
{
    int     i;
    BYTE    KeyState[NUMKEYSTATES];

    memset(&This, 0, sizeof(struct tKeyboard_private));
    This.m_nInternalKeyPress = -1;

    GetKeyboardState(KeyState);

    for (i = 0; i < NUMKEYSTATES; i++)
        KeyState[i] &= 0x01;

    SetKeyboardState(KeyState);

    return 0;
}

/*
    put here cleanup routines to be executed when the program is terminated.
*/
static void Keyboard_exit(void)
{
}

/*
    Check if a key is pressed. The keycode is the standard PC keyboard code, as
    defined in osdepend.h. Return 0 if the key is not pressed, nonzero otherwise.
*/
static int Keyboard_key_pressed(int keycode)
{
    SHORT   state;
    int     nVirtualKey;

    if (This.m_nInternalKeyPress != -1)
    {
        if (This.m_nInternalKeyPress == keycode)
        {
            This.m_nInternalKeyPress = -1;
            return 1;
        }
    }

    if (OSD_MAX_KEY < keycode)
    {
        switch (keycode)
        {
            case OSD_KEY_CANCEL:
                keycode = OSD_KEY_ESC;
                break;

            case OSD_KEY_SHOW_GFX:
                keycode = OSD_KEY_F4;
                break;

            case OSD_KEY_RESET_MACHINE:
                keycode = OSD_KEY_F3;
                break;

            case OSD_KEY_CHEAT_TOGGLE:
				keycode = OSD_KEY_F5;
                break;

            case OSD_KEY_JOY_CALIBRATE:
                keycode = OSD_KEY_F7;
                break;

            case OSD_KEY_FRAMESKIP:
                keycode = OSD_KEY_F8;
                break;

            case OSD_KEY_THROTTLE:
                keycode = OSD_KEY_F10;
                break;

            case OSD_KEY_SHOW_FPS:
                if (Keyboard_key_pressed(OSD_KEY_LSHIFT)
              /*||  Keyboard_key_pressed(OSD_KEY_RSHIFT)*/)
                    return 0;
                keycode = OSD_KEY_F11;
                break;

            case OSD_KEY_SHOW_PROFILE:
                if (!(Keyboard_key_pressed(OSD_KEY_LSHIFT)
              /*||    Keyboard_key_pressed(OSD_KEY_RSHIFT)*/))
                    return 0;
                keycode = OSD_KEY_F11;
                break;

            case OSD_KEY_CONFIGURE:
                keycode = OSD_KEY_TAB;
                break;

            case OSD_KEY_ON_SCREEN_DISPLAY:
            {
                extern int mame_debug;
                if (mame_debug)
                    return 0;

                keycode = OSD_KEY_TILDE;
                break;
            }

            case OSD_KEY_DEBUGGER:
            {
                extern int mame_debug;
                if (!mame_debug)
                    return 0;

                keycode = OSD_KEY_TILDE;
                break;
            }

            case OSD_KEY_PAUSE:
                keycode = OSD_KEY_P;
                break;

            case OSD_KEY_UNPAUSE:
                keycode = OSD_KEY_P;
                break;

            case OSD_KEY_SNAPSHOT:
                keycode = OSD_KEY_F12;
                break;

            default:
                return 0;
        }
    }

    /* Map OSD_KEY to VK. */
    nVirtualKey = MapVirtualKey(keycode, 1);

    MAME32App.ProcessMessages();
    state = GetAsyncKeyState(nVirtualKey);

    /* If the high-order bit is 1, the key is down; otherwise, it is up */
    if (state & 0x8000)
        return 1;

    return 0;
}

/* Report a key as pressed only when the user hits it, not while it is */
/* being kept pressed. */
static int Keyboard_key_pressed_memory(int keycode)
{
    int res = 0;

    if (keycode == OSD_KEY_UNPAUSE) keycode = OSD_KEY_PAUSE;    /* we use the same key */

    if (osd_key_pressed(keycode))
    {
        if (This.m_Memory[keycode] == 0)
            res = 1;
        This.m_Memory[keycode] = 1;
    }
    else
        This.m_Memory[keycode] = 0;

    return res;
}

/* report key as pulsing while it is pressed */
static int Keyboard_key_pressed_memory_repeat(int keycode, int speed)
{
    int res = 0;

    if (osd_key_pressed(keycode))
    {
        if (This.m_Memory[keycode] == 0 || ++This.m_nCounter > speed * Machine->drv->frames_per_second / 60)
        {
            This.m_nCounter = 0;
            res = 1;
        }
        This.m_Memory[keycode] = 1;
    }
    else
        This.m_Memory[keycode] = 0;

    return res;
}

/* If the user presses a key return it, otherwise return OSD_KEY_NONE. */
/* DO NOT wait for the user to press a key */
static int Keyboard_read_key_immediate(void)
{
    int res = 0;

    /* first of all, record keys which are NOT pressed */
    for (res = OSD_MAX_KEY; OSD_KEY_NONE < res; res--)
    {
        if (!osd_key_pressed(res))
        {
            This.m_Memory[res] = 0;
            This.m_Memory[key_to_pseudo_code(res)] = 0;
        }
    }

    for (res = OSD_MAX_KEY; OSD_KEY_NONE < res; res--)
    {
        if (osd_key_pressed(res))
        {
            if (This.m_Memory[res] == 0)
            {
                This.m_Memory[res] = 1;
                This.m_Memory[key_to_pseudo_code(res)] = 1;
            }
            else
                res = OSD_KEY_NONE;
            break;
        }
    }

    return res;
}

/*
    Wait for a key press and return the keycode.
*/
static int Keyboard_read_key(int translate)
{
    MSG Msg;
    
    StatusSetString("Waiting for keypress");

    while (GetMessage(&Msg, NULL, 0, 0))
    {
        TranslateMessage(&Msg);
        DispatchMessage(&Msg);

        if (Msg.message == WM_KEYDOWN
        ||  Msg.message == WM_SYSKEYDOWN)
        {
            int     nVirtualKey = Msg.wParam;   /* virtual-key code */
            int     nKeyCode;

            nKeyCode = MapVirtualKey(nVirtualKey, 0);

            if (translate)
                return Keyboard_KeyToPseudoCode(nKeyCode);
            else
                return nKeyCode;
        }
    }

    if (Msg.message == WM_QUIT)
    {
        MAME32App.Quit();
        return OSD_KEY_SPACE;
    }

    return OSD_KEY_ESC;
}

/*
    Wait for a key press and return keycode.  Support repeat
*/
static int Keyboard_read_keyrepeat(int translate)
{
    return Keyboard_read_key(translate);
}

/*
    return the name of a key
*/
static const char* Keyboard_key_name(int keycode)
{
    static char *nonedefined = "None";
    static char *keynames[] =
    {
        "ESC", "1", "2", "3", "4", "5", "6", "7", "8", "9", "0", "MINUS", "EQUAL", "BKSPACE",
        "TAB", "Q", "W", "E", "R", "T", "Y", "U", "I", "O", "P", "OPBRACE", "CLBRACE", "ENTER",
        "LCTRL", "A", "S", "D", "F", "G", "H", "J", "K", "L", "COLON", "QUOTE", "TILDE",
        "LSHIFT", "Error", "Z", "X", "C", "V", "B", "N", "M", "COMMA", ".", "SLASH", "RSHIFT",
        "*", "ALT", "SPACE", "CAPSLOCK", "F1", "F2", "F3", "F4", "F5", "F6", "F7", "F8", "F9", "F10",
        "NUMLOCK", "SCRLOCK", "HOME", "UP", "PGUP", "MINUS PAD",
        "LEFT", "5 PAD", "RIGHT", "PLUS PAD", "END", "DOWN",
        "PGDN", "INS", "DEL", "RCTRL", "ALTGR", "Error",
        "F11", "F12", "Error", "Error",
        "Error", "Error", "Error", "Error", "Error",
        "Error", "Error", "Error", "Error", "Error",
        "1 PAD", "2 PAD", "3 PAD", "4 PAD", "Error",
        "6 PAD", "7 PAD", "8 PAD", "9 PAD", "0 PAD",
        ". PAD", "= PAD", "/ PAD", "* PAD", "ENTER PAD",
        "Error", "Error", "Error", "Error", "Error",
        "Error", "Error", "PAUSE",
    };

    if (1 <= keycode && keycode <= OSD_MAX_KEY)
        return (char*)keynames[keycode - 1];
    else
        return (char*)nonedefined;
}

/* translate a scancode to a pseudo key code */
int Keyboard_KeyToPseudoCode(int keycode)
{
    switch (keycode)
    {
        case OSD_KEY_ESC:
            return OSD_KEY_CANCEL;

        case OSD_KEY_F3:
            return OSD_KEY_RESET_MACHINE;

        case OSD_KEY_F4:
            return OSD_KEY_SHOW_GFX;

        case OSD_KEY_F5:
            return OSD_KEY_CHEAT_TOGGLE;

        case OSD_KEY_F7:
            return OSD_KEY_JOY_CALIBRATE;

        case OSD_KEY_F8:
            return OSD_KEY_FRAMESKIP;

        case OSD_KEY_F10:
            return OSD_KEY_THROTTLE;

        case OSD_KEY_F11:
            if (osd_key_pressed(OSD_KEY_LSHIFT)
            ||  osd_key_pressed(OSD_KEY_RSHIFT))
                return OSD_KEY_SHOW_PROFILE;
            return OSD_KEY_SHOW_FPS;

        case OSD_KEY_F12:
            return OSD_KEY_SNAPSHOT;

        case OSD_KEY_TILDE:
        {
            extern int mame_debug;

            if (mame_debug)
                return OSD_KEY_DEBUGGER;
            else
                return OSD_KEY_ON_SCREEN_DISPLAY;
        }

        case OSD_KEY_P:
            return OSD_KEY_PAUSE;/* OSD_KEY_UNPAUSE? */

        case OSD_KEY_TAB:
            return OSD_KEY_CONFIGURE;
    }
    return keycode;
}

/***************************************************************************
    Message handlers
 ***************************************************************************/

static BOOL Keyboard_OnMessage(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam, LRESULT* pResult)
{
    return FALSE;
}

/***************************************************************************
    Internal functions
 ***************************************************************************/

/* Simulate a keypress. */
static void Keyboard_PressKey(int keycode)
{
    This.m_nInternalKeyPress = keycode;
    PostMessage(MAME32App.m_hWnd, WM_KEYDOWN, MapVirtualKey(keycode, 1), 0);
}
