/***************************************************************************

    M.A.M.E.32  -  Multiple Arcade Machine Emulator for Win32
    Win32 Portions Copyright (C) 1997-98 Michael Soderstrom and Chris Kirmse
    
    This file is part of MAME32, and may only be used, modified and
    distributed under the terms of the MAME license, in "readme.txt".
    By continuing to use, modify or distribute this file you indicate
    that you have read the license and understand and accept it fully.

 ***************************************************************************/

/***************************************************************************

  Display.c

 ***************************************************************************/

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <assert.h>
#include <math.h>
#include "MAME32.h"
#include "Display.h"
#include "M32Util.h"
#include "driver.h"
#include "uclock.h"
#include "file.h"
#include "profiler.h"

#define MEMORY 10
#define MAXDISPLAYTEXT  256
extern int frameskip;

/***************************************************************************
    function prototypes
 ***************************************************************************/

static int                Display_init(options_type *options);
static void               Display_exit(void);
static struct osd_bitmap* Display_new_bitmap(int width,int height,int depth);
static void               Display_clearbitmap(struct osd_bitmap* bitmap);
static void               Display_free_bitmap(struct osd_bitmap* bitmap);
static void               Display_update_display(void);
static void               Display_set_gamma(float gamma);
static void               Display_set_brightness(int brightness);
static void               Display_UpdateFPS(BOOL bShow, int nSpeed, int nFPS, int nMachineFPS);

static void               Throttle(void);
static void               AdjustColorMap(void);

/***************************************************************************
    External variables
 ***************************************************************************/

struct OSDDisplay   OSDDisplay = 
{
    { Display_init },               /* init              */
    { Display_exit },               /* exit              */
    { Display_new_bitmap },         /* new_bitmap        */
    { Display_clearbitmap },        /* clearbitmap       */
    { Display_free_bitmap },        /* free_bitmap       */
    { 0 },                          /* create_display    */
    { 0 },                          /* set_display       */
    { 0 },                          /* close_display     */
    { 0 },                          /* allocate_colors   */
    { 0 },                          /* modify_pen        */
    { 0 },                          /* get_pen           */
    { 0 },                          /* mark_dirty        */
    { Display_update_display },     /* update_display    */
    { 0 },                          /* led_w             */
    { Display_set_gamma },          /* set_gamma         */
    { Display_get_gamma },          /* get_gamma         */
    { Display_set_brightness },     /* set_brightness    */
    { Display_get_brightness },     /* get_brightness    */
    { 0 },                          /* save_snapshot     */

    { 0 },                          /* OnMessage         */
    { 0 },                          /* Refresh           */
    { 0 },                          /* GetBlackPen       */
    { Display_UpdateFPS },          /* UpdateFPS         */
};

/***************************************************************************
    Internal structures
 ***************************************************************************/

struct tDisplay_private
{
    /* Gamma/Brightness */
    unsigned char   m_pColorMap[OSD_NUMPENS];
    double          m_dGamma;
    int             m_nBrightness;

    /* Profile info */
    BOOL            m_bShowProfile;
    BOOL            m_bUseProfile;

    /* Frames per second (FPS) info */
    BOOL            m_bShowFPS;
    int             m_nShowFPSTemp;
    int             m_nFramesRendered;

    /* Speed throttling */
    BOOL            m_bThrottled;
    int             m_nSpeed;
    int             m_nMemory;
    int             m_nSpareTime;
    uclock_t        m_Prev[MEMORY];

    /* average frame rate data */
    int             frames_displayed;
    uclock_t        start_time;
    uclock_t        end_time;  /* to calculate fps average on exit */
};

#define FRAMES_TO_SKIP 20   /* skip the first few frames from the FPS calculation */
                            /* to avoid counting the copyright and info screens */

/***************************************************************************
    Internal variables
 ***************************************************************************/

static struct tDisplay_private      This;
static int                          snapno = 0;

/***************************************************************************
    External OSD functions  
 ***************************************************************************/

/*
    put here anything you need to do when the program is started. Return 0 if 
    initialization was successful, nonzero otherwise.
*/
static int Display_init(options_type *options)
{
    int i;

    /* Reset the Snapshot number */
    snapno = 0;
    
    This.m_dGamma            = max(0, options->gamma);
    This.m_nBrightness       = max(0, options->brightness);
    This.m_nFramesRendered   = 0;
    This.m_bThrottled        = TRUE;
    This.m_bShowProfile      = FALSE;
    This.m_bUseProfile       = options->profile;    
    This.m_bShowFPS          = FALSE;
    This.m_nShowFPSTemp      = 0;
    This.m_nSpeed            = 0;
    This.m_nMemory           = 0;
    This.m_nSpareTime        = 0;
    for (i = 0; i < MEMORY; i++)
        This.m_Prev[i] = uclock();

    /* Initialize map. */
    AdjustColorMap();
    
    This.frames_displayed = 0;
    This.start_time       = 0;
    This.end_time         = 0;

    return 0;
}

/*
    put here cleanup routines to be executed when the program is terminated.
*/
static void Display_exit(void)
{
   if (This.frames_displayed > FRAMES_TO_SKIP)
      printf("Average FPS: %f\n",
             (double)UCLOCKS_PER_SEC / (This.end_time - This.start_time) *
             (This.frames_displayed - FRAMES_TO_SKIP));
}

/*
    Set the bitmap to black.
*/
static void Display_clearbitmap(struct osd_bitmap* bitmap)
{
    int i;

    for (i = 0; i < bitmap->height; i++)
        memset(bitmap->line[i], MAME32App.m_pDisplay->GetBlackPen(),
               bitmap->width * (bitmap->depth / 8));

    if (bitmap == Machine->scrbitmap)
    {
        extern int bitmap_dirty;
        bitmap_dirty = 1;

        osd_mark_dirty(0, 0, bitmap->width - 1, bitmap->height - 1, 1);
    }
}

/*
    Create a bitmap. Also call osd_clearbitmap() to appropriately initialize it to 
    the background color.
*/
static struct osd_bitmap* Display_new_bitmap(int width, int height, int depth)
{
    struct osd_bitmap*  pBitmap;

    if (Machine->orientation & ORIENTATION_SWAP_XY)
    {
        int     temp;

        temp   = width;
        width  = height;
        height = temp;
    }

    pBitmap = (struct osd_bitmap*)malloc(sizeof(struct osd_bitmap));

    if (pBitmap != NULL)
    {
        unsigned char*  bm;
        int     i;
        int     safety;
        int     rdwidth;

        pBitmap->width  = width;
        pBitmap->height = height;
        pBitmap->depth  = depth;

        if (pBitmap->width > 32)
           safety = 8;
        else
           safety = 0;

        rdwidth = (width + 7) & ~7;     /* round width to a quadword */

        bm = (unsigned char*)malloc((rdwidth + 2 * safety) * (height + 2 * safety) * depth / 8);
        if (bm == NULL)
        {
            free(pBitmap);
            return NULL;
        }

        if ((pBitmap->line = malloc(height * sizeof(unsigned char *))) == 0)
        {
            free(bm);
            free(pBitmap);
            return 0;
        }

        for (i = 0; i < height; i++)
            pBitmap->line[i] = &bm[(i + safety) * (rdwidth + 2 * safety) * depth / 8 + safety];

        pBitmap->_private = bm;

        osd_clearbitmap(pBitmap);
    }

    return pBitmap;
}

/*
    Free memory allocated in create_bitmap.
*/
static void Display_free_bitmap(struct osd_bitmap* pBitmap)
{
    if (pBitmap != NULL)
    {
        free(pBitmap->line);
        free(pBitmap->_private);
        free(pBitmap);
    }
}

/*
    Update the display.
*/
static void Display_update_display(void)
{
    extern int frameskip;

    This.m_nFramesRendered++;

    /*
        Adjust frameskip.
    */
    if (osd_key_pressed_memory(OSD_KEY_FRAMESKIP))
    {
        frameskip = (frameskip + 1) % 4;
        if (This.m_bShowFPS == FALSE)
            This.m_nShowFPSTemp = 2 * Machine->drv->frames_per_second;

        /* reset the frame counter every time the frameskip key is pressed, so */
        /* we'll measure the average FPS on a consistent status. */
        This.frames_displayed = 0;
    }

    /*
        Toggle throttle.
    */
    if (osd_key_pressed_memory(OSD_KEY_THROTTLE))
    {
        This.m_bThrottled = (This.m_bThrottled == TRUE) ? FALSE : TRUE;

        /* reset the frame counter every time the throttle key is pressed, so */
        /* we'll measure the average FPS on a consistent status. */
        This.frames_displayed = 0;
    }

  
    /*
        Wait until it's time to update the screen.
    */
    Throttle();

}

static void Display_set_gamma(float gamma)
{
    if (Machine->scrbitmap->depth == 8)
    {
        if (This.m_dGamma < 0.2) This.m_dGamma = 0.2;
        if (This.m_dGamma > 3.0) This.m_dGamma = 3.0;

        This.m_dGamma = gamma;
        AdjustColorMap();
    }
}

float Display_get_gamma()
{
    return (float)This.m_dGamma;
}

static void Display_set_brightness(int brightness)
{
    if (Machine->scrbitmap->depth == 8)
    {
        This.m_nBrightness = brightness;
        AdjustColorMap();
    }
}

int Display_get_brightness()
{
    return This.m_nBrightness;
}

void Display_MapColor(unsigned char* pRed, unsigned char* pGreen, unsigned char* pBlue)
{
    *pRed   = This.m_pColorMap[(unsigned char)*pRed];
    *pGreen = This.m_pColorMap[(unsigned char)*pGreen];
    *pBlue  = This.m_pColorMap[(unsigned char)*pBlue];
}


void Display_WriteBitmap(struct osd_bitmap* tBitmap, PALETTEENTRY* pPalEntries)
{
    BITMAPFILEHEADER    bmfHeader;
    BITMAPINFOHEADER    bmiHeader;
    RGBQUAD             bmiColors[OSD_NUMPENS];
    char                sFileName[MAX_PATH];
    int                 nWritten;
    int                 i;
    int                 nResult;
    void*               pFile;

    /*
        Create a unique BMP file.
    */

    do
    {
        if (snapno == 0)    /* first of all try with the "gamename.bmp" */
            sprintf(sFileName, "%.8s", Machine->gamedrv->name);
        else                /* otherwise use "nameNNNN.bmp" */
            sprintf(sFileName, "%.4s%04d", Machine->gamedrv->name, snapno);

        /* Avoid overwriting of existing files */
        nResult = osd_faccess(sFileName, OSD_FILETYPE_SCREENSHOT);
               
        if (nResult != 0)
            snapno++;

    } while (nResult != 0);


    if ((pFile = osd_fopen(sFileName, "", OSD_FILETYPE_SCREENSHOT, TRUE)) == NULL)
    {
        ErrorMsg("osd_fopen failed opening %s", sFileName);
        return;
    }
 
    /*
        Init Colormap.
    */
    if (tBitmap->depth == 8)
    {
        for (i = 0; i < OSD_NUMPENS; i++)
        {
            bmiColors[i].rgbRed         = pPalEntries->peRed;
            bmiColors[i].rgbGreen       = pPalEntries->peGreen;
            bmiColors[i].rgbBlue        = pPalEntries->peBlue;
            bmiColors[i].rgbReserved    = 0;
            pPalEntries++;
        }
    }

    /*
        Init BITMAPINFOHEADER.

        For biBitCount == 16:
        "The bitmap has a maximum of 2^16 colors. If the biCompression member of
        the BITMAPINFOHEADER is BI_RGB, the bmiColors member is NULL. Each WORD
        in the bitmap array represents a single pixel. The relative intensities
        of red, green, and blue are represented with 5 bits for each color component.
        The value for blue is in the least significant 5 bits, followed by 5 bits
        each for green and red, respectively. The most significant bit is not used."
    */

    bmiHeader.biSize            = sizeof(BITMAPINFOHEADER);
    bmiHeader.biWidth           = tBitmap->width;
    bmiHeader.biHeight          = tBitmap->height;
    bmiHeader.biPlanes          = 1;
    bmiHeader.biBitCount        = (tBitmap->depth == 8) ? 8 : 16;
    bmiHeader.biCompression     = BI_RGB;
    bmiHeader.biSizeImage       = 0; /* "This may be set to 0 for BI_RGB bitmaps." */
    bmiHeader.biXPelsPerMeter   = 0;
    bmiHeader.biYPelsPerMeter   = 0;
    bmiHeader.biClrUsed         = (tBitmap->depth == 8) ? OSD_NUMPENS : 0;
    bmiHeader.biClrImportant    = 0;

    /*
        Init BITMAPFILEHEADER.
    */

    bmfHeader.bfType        = 0x4d42;
    bmfHeader.bfReserved1   = 0;
    bmfHeader.bfReserved2   = 0;
    bmfHeader.bfSize        = sizeof(BITMAPFILEHEADER) +
                              sizeof(BITMAPINFOHEADER) +
                              bmiHeader.biClrUsed * sizeof(RGBQUAD) +
                              bmiHeader.biSizeImage;
    bmfHeader.bfOffBits     = sizeof(BITMAPFILEHEADER) +
                              sizeof(BITMAPINFOHEADER) +
                              bmiHeader.biClrUsed * sizeof(RGBQUAD);

    /* BITMAPFILEHEADER */
    
    nWritten = osd_fwrite(pFile, (void*)&bmfHeader, sizeof(BITMAPFILEHEADER));
    if (nWritten != sizeof(BITMAPFILEHEADER))
        ErrorMsg("osd_fwrite failed writing BITMAPFILEHEADER");
    
    /* BITMAPINFOHEADER */
    
    nWritten = osd_fwrite(pFile, (void*)&bmiHeader, sizeof(BITMAPINFOHEADER));
    if (nWritten != sizeof(BITMAPINFOHEADER))
        ErrorMsg("osd_fwrite failed writing BITMAPINFOHEADER");
    
    /* RGBQUAD */
    
    if (tBitmap->depth == 8)
    {
        nWritten = osd_fwrite(pFile, (void*)&bmiColors, bmiHeader.biClrUsed * sizeof(RGBQUAD));
        if (nWritten != (int)(bmiHeader.biClrUsed * sizeof(RGBQUAD)))
            ErrorMsg("osd_fwrite failed writing RGBQUADs");
    }
    
    /*
        Write image data line by line, bottom up.
    */

    for (i = tBitmap->height - 1; 0 <= i; i--)
    {
        if (tBitmap->depth == 8)
        {
            nWritten = osd_fwrite(pFile, (void*)tBitmap->line[i], tBitmap->width * tBitmap->depth / 8);
            if (nWritten != (tBitmap->width * tBitmap->depth / 8))
                ErrorMsg("osd_fwrite failed writing image data");
        }
        else
        {
            unsigned short  pLine[1024];
            unsigned short* ptr = (unsigned short*)tBitmap->line[i];
            unsigned char   r, g, b;
            int x;

            /* convert bitmap data to 555 */
            for (x = 0; x < tBitmap->width; x++)
            {
                osd_get_pen(*ptr++, &r, &g, &b);
                pLine[x] = ((r & 0x1F) << 10) |
                           ((g & 0x1F) <<  5) |
                            (b & 0x1F);
            }

            nWritten = osd_fwrite(pFile, (void*)pLine, tBitmap->width * tBitmap->depth / 8);
            if (nWritten != (tBitmap->width * tBitmap->depth / 8))
                ErrorMsg("osd_fwrite failed writing image data");
        }
    }

    osd_fclose(pFile);

    snapno++;
}

static void Display_UpdateFPS(BOOL bShow, int nSpeed, int nFPS, int nMachineFPS)
{
    int  trueorientation;
    int  i, l;
    char buf[30];

    if (bShow == TRUE)
    {
        /*
            hack: force the display into standard orientation to avoid
            rotating the text
        */
        trueorientation = Machine->orientation;
        Machine->orientation = ORIENTATION_DEFAULT;

        sprintf(buf, " %3d%%(%3d/%d fps)", nSpeed, nFPS, nMachineFPS);
        l = strlen(buf);
        for (i = 0; i < l; i++)
            drawgfx(Machine->scrbitmap, Machine->uifont, buf[i], DT_COLOR_WHITE, 0, 0,
                    Machine->uiwidth + Machine->uixmin - (l - i) * Machine->uifont->width,
                    Machine->uiymin,
                    0, TRANSPARENCY_NONE, 0);

        Machine->orientation = trueorientation;
    }
    else
    {
        /* NOTE: should only clear FPS display area, not the entire display. */
        osd_clearbitmap(Machine->scrbitmap);
        osd_mark_dirty(0, 0, Machine->scrbitmap->width - 1, Machine->scrbitmap->height - 1, 1);
    }
}

int Display_GetFramesRendered()
{
    return This.m_nFramesRendered;
}

void Display_ResetFramesRendered()
{
    This.m_nFramesRendered = 0;
}

BOOL Display_Throttled()
{
    return This.m_bThrottled;
}

/* Write messages in the screen. */
void Display_Textout(char* buf, int x, int y)
{
    int trueorientation, l, i;

    /* hack: force the display into standard orientation to avoid */
    /* rotating the text */
    trueorientation = Machine->orientation;
    Machine->orientation = ORIENTATION_DEFAULT;

    l = strlen(buf);
    for (i = 0; i < l; i++)
        drawgfx(Machine->scrbitmap, 
                Machine->uifont,
                buf[i],
                DT_COLOR_WHITE,
                0, 0,
                x + i * Machine->uifont->width + Machine->uixmin,
                y + Machine->uiymin,
                0, TRANSPARENCY_NONE, 0);

    Machine->orientation = trueorientation;

    /* mark dirty? */
}

BOOL Display_UpdateOnScreenDisplay()
{
    BOOL    bClearScreen = FALSE;

    /*
        Frames per second.
    */
    if (osd_key_pressed_memory(OSD_KEY_SHOW_FPS))
    {
        if (This.m_nShowFPSTemp)
        {
            This.m_nShowFPSTemp = 0;
            MAME32App.m_pDisplay->UpdateFPS(FALSE, 0, 0, 0);
        }
        else
        {
            This.m_bShowFPS = (This.m_bShowFPS == TRUE) ? FALSE : TRUE;
            if (This.m_bShowFPS == FALSE)
                MAME32App.m_pDisplay->UpdateFPS(FALSE, 0, 0, 0);
        }
    }

    if (This.m_nShowFPSTemp)
    {
        This.m_nShowFPSTemp--;
        if ((This.m_bShowFPS == FALSE)
        &&  (This.m_nShowFPSTemp == 0))
            MAME32App.m_pDisplay->UpdateFPS(FALSE, 0, 0, 0);
    }       

    /*
        Profile.
    */
    if (This.m_bUseProfile && osd_key_pressed_memory(OSD_KEY_SHOW_PROFILE))
    {
        This.m_bShowProfile = (This.m_bShowProfile == TRUE) ? FALSE : TRUE;
        if (This.m_bShowProfile == FALSE)
            bClearScreen = TRUE;
    }

    if (This.m_bShowProfile)
        Profiler_display();

    return bClearScreen;
}

/***************************************************************************
    Internal functions
 ***************************************************************************/

static void Throttle(void)
{
    uclock_t curr;

#if 0

    if (This.m_bThrottled == TRUE)
    {
        osd_profiler(OSD_PROFILE_IDLE);

        do
        {
            curr = uclock();
        } while ((curr - This.m_Prev[This.m_nMemory]) < (frameskip + 1) * UCLOCKS_PER_SEC / Machine->drv->frames_per_second);
        
        osd_profiler(OSD_PROFILE_END);
    }
    else curr = uclock();

#else

    curr = uclock();

    /* for the FPS average calculation */
    if (++This.frames_displayed == FRAMES_TO_SKIP)
        This.start_time = curr;
    else
        This.end_time = curr;

    if (This.m_bThrottled == TRUE)
    {
        int frametime = (frameskip + 1) * UCLOCKS_PER_SEC / Machine->drv->frames_per_second;
        int thisspare;
        int start = curr;
#define MAX_FRAMES  2

        osd_profiler(OSD_PROFILE_IDLE);

        thisspare = frametime - (curr - This.m_Prev[This.m_nMemory]);
        if ((This.m_nSpareTime + thisspare) > 0)
        {
            do
            {   
                curr = uclock();
                thisspare = frametime - (curr - This.m_Prev[This.m_nMemory]);
            } while ((This.m_nSpareTime + thisspare) > 0);
            This.m_nSpareTime += thisspare;
        }
        else
        {
            This.m_nSpareTime += thisspare;
            This.m_nSpareTime = max(This.m_nSpareTime, -(frametime >> MAX_FRAMES));
        }

        osd_profiler(OSD_PROFILE_END);
    }

#endif

    This.m_nMemory = (This.m_nMemory + 1) % MEMORY;

    /*
        Show frames per second.
    */
    if (This.m_bShowFPS == TRUE || This.m_nShowFPSTemp != 0)
    {
        int fps;

        if (curr - This.m_Prev[This.m_nMemory])
        {
            int divdr = Machine->drv->frames_per_second * (curr - This.m_Prev[This.m_nMemory]) / (100 * MEMORY);

            This.m_nSpeed = (UCLOCKS_PER_SEC * (frameskip + 1) + divdr / 2) / divdr;
        }

        fps = (Machine->drv->frames_per_second / (frameskip + 1) * This.m_nSpeed + 50) / 100;

        MAME32App.m_pDisplay->UpdateFPS(TRUE, This.m_nSpeed, fps, Machine->drv->frames_per_second);
    }

    This.m_Prev[This.m_nMemory] = curr;

}

static void AdjustColorMap(void)
{
    int i;
    
    for (i = 0; i < OSD_NUMPENS; i++)
    {
        This.m_pColorMap[i] = (unsigned char)((OSD_NUMPENS - 1.0)
                            * (This.m_nBrightness / 100.0)
                            * pow(((double)i / (OSD_NUMPENS - 1.0)), 1.0 / This.m_dGamma));
    }
}


