/***************************************************************************

    M.A.M.E.32  -  Multiple Arcade Machine Emulator for Win32
    Win32 Portions Copyright (C) 1997-98 Michael Soderstrom and Chris Kirmse
    
    This file is part of MAME32, and may only be used, modified and
    distributed under the terms of the MAME license, in "readme.txt".
    By continuing to use, modify or distribute this file you indicate
    that you have read the license and understand and accept it fully.

 ***************************************************************************/

/***************************************************************************

  DIKeyboard.c

 ***************************************************************************/

#include "driver.h"

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <windowsx.h>
#include <dinput.h>
#include "MAME32.h"
#include "M32Util.h"
#include "DirectInput.h"
#include "Keyboard.h"
#include "DIKeyboard.h"
#include "DIJoystick.h"
#include "status.h"

/***************************************************************************
    function prototypes
 ***************************************************************************/

static void         PollKeyboard(void);

static int          DIKeyboard_init(options_type *options);
static void         DIKeyboard_exit(void);
static int          DIKeyboard_key_pressed(int keycode);
static int          DIKeyboard_key_pressed_memory(int keycode);
static int          DIKeyboard_key_pressed_memory_repeat(int keycode,int speed);
static int          DIKeyboard_read_key_immediate(void);
static int          DIKeyboard_read_key(int translate);
static int          DIKeyboard_read_keyrepeat(int translate);
static const char*  DIKeyboard_key_name(int keycode);
static void         DIKeyboard_PressKey(int keycode);
static BOOL         DIKeyboard_OnMessage(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam, LRESULT* pResult);
static void         OnActivateApp(HWND hWnd, BOOL fActivate, DWORD dwThreadId);
static void         OnKey(HWND hwnd, UINT vk, BOOL fDown, int cRepeat, UINT flags);

/***************************************************************************
    External variables
 ***************************************************************************/

struct OSDKeyboard  DIKeyboard = 
{
    { DIKeyboard_init },                      /* init              */
    { DIKeyboard_exit },                      /* exit              */
    { DIKeyboard_key_pressed },               /* key_pressed       */
    { DIKeyboard_key_pressed_memory },        /* key_pressed_memory */
    { DIKeyboard_key_pressed_memory_repeat }, /* key_pressed_memory_repeat */
    { DIKeyboard_read_key_immediate },        /* read_key_immediate */
    { DIKeyboard_read_key },                  /* read_key          */
    { DIKeyboard_read_keyrepeat },            /* read_keyrepeat    */
    { DIKeyboard_key_name },                  /* key_name          */

    { DIKeyboard_PressKey },                  /* PressKey          */
    { DIKeyboard_OnMessage }                  /* OnMessage         */
};

/***************************************************************************
    Internal structures
 ***************************************************************************/

struct tKeyboard_private
{
    byte                 m_key[OSD_MAX_PSEUDO + 1];
    byte                 m_memory[256];
    int                  m_counter; /* for key_pressed_memory_repeat */
    LPDIRECTINPUTDEVICE  m_didKeyboard;
    byte                 m_internal_key[OSD_MAX_PSEUDO + 1];
};

/***************************************************************************
    Internal variables
 ***************************************************************************/

static struct tKeyboard_private This;

/***************************************************************************
    External OSD functions  
 ***************************************************************************/

/*
    put here anything you need to do when the program is started. Return 0 if 
    initialization was successful, nonzero otherwise.
*/
static int DIKeyboard_init(options_type *options)
{
    HRESULT hr;
    int i;

    This.m_didKeyboard       = NULL;

    for (i = 0; i <= OSD_MAX_PSEUDO; i++)
    {
        This.m_key[i] = 0;
        This.m_internal_key[i] = 0;
    }

    for (i=0;i<256;i++)
        This.m_memory[i] = 0;

    if (di == NULL)
    {
        ErrorMsg("DirectInput not initialized");
        return 1;
    }

    /* setup the keyboard */
    hr = IDirectInput_CreateDevice(di, &GUID_SysKeyboard, &This.m_didKeyboard, NULL);

    if (FAILED(hr)) 
    {
        ErrorMsg("DirectInputCreateDevice failed!\n");
        return 1;
    }
   
    hr = IDirectInputDevice_SetDataFormat(This.m_didKeyboard, &c_dfDIKeyboard);

    if (FAILED(hr)) 
    {
        ErrorMsg("DirectInputDevice SetDataFormat failed\n");
        return 1;
    }
   
    /* should be using DISCL_FOREGROUND, but this call fails sometimes when we do that.
       It didn't in my previous mame code, so something's fishy--perhaps the new 
       window isn't active yet or something? CMK */
    hr = IDirectInputDevice_SetCooperativeLevel(This.m_didKeyboard, MAME32App.m_hWnd,
                            DISCL_NONEXCLUSIVE | DISCL_FOREGROUND);
   
    if (FAILED(hr)) 
    {
       ErrorMsg("DirectInputDevice SetCooperativeLevel failed!\n");
       return 1;
    }
   
    hr = IDirectInputDevice_Acquire(This.m_didKeyboard);

    return 0;
}

/*
  put here cleanup routines to be executed when the program is terminated.
  */
static void DIKeyboard_exit(void)
{
    if (!This.m_didKeyboard)
        return;

    /*
     Cleanliness is next to godliness.  Unacquire the device
     one last time just in case we got really confused and tried
     to exit while the device is still acquired.
     */
    IDirectInputDevice_Unacquire(This.m_didKeyboard);
    IDirectInputDevice_Release(This.m_didKeyboard);

    This.m_didKeyboard = NULL;

}

/*
  Check if a key is pressed. The keycode is the standard PC keyboard code, as
  defined in osdepend.h. Return 0 if the key is not pressed, nonzero otherwise.
  */
static int DIKeyboard_key_pressed(int keycode)
{
    MAME32App.ProcessMessages();
    PollKeyboard();

    return This.m_key[keycode];
}

/* Report a key as pressed only when the user hits it, not while it is */
/* being kept pressed. */
static int DIKeyboard_key_pressed_memory(int keycode)
{
    int retval = 0;

    if (keycode == OSD_KEY_UNPAUSE) keycode = OSD_KEY_PAUSE;    /* we use the same key */

    if (osd_key_pressed(keycode))
    {
        if (This.m_memory[keycode] == 0)
            retval = 1;
        This.m_memory[keycode] = 1;
    }
    else
        This.m_memory[keycode] = 0;

    return retval;
}

/* report key as pulsing while it is pressed */
static int DIKeyboard_key_pressed_memory_repeat(int keycode,int speed)
{
    int res = 0;

    if (osd_key_pressed(keycode))
    {
        if (This.m_memory[keycode] == 0 || 
            ++This.m_counter > speed * Machine->drv->frames_per_second / 60)
        {
            This.m_counter = 0;
            res = 1;
        }
        This.m_memory[keycode] = 1;
    }
    else
        This.m_memory[keycode] = 0;

    return res;

}

/* If the user presses a key return it, otherwise return OSD_KEY_NONE. */
/* DO NOT wait for the user to press a key */
static int DIKeyboard_read_key_immediate(void)
{
    int retval;


    /* first of all, record keys which are NOT pressed */
    for (retval = OSD_MAX_KEY;retval > OSD_KEY_NONE;retval--)
    {
        if (!osd_key_pressed(retval))
        {
            This.m_memory[retval] = 0;
            This.m_memory[key_to_pseudo_code(retval)] = 0;
        }
    }

    for (retval = OSD_MAX_KEY;retval > OSD_KEY_NONE;retval--)
    {
        if (osd_key_pressed(retval))
        {
            if (This.m_memory[retval] == 0)
            {
                This.m_memory[retval] = 1;
                This.m_memory[key_to_pseudo_code(retval)] = 1;
            }
            else 
                retval = OSD_KEY_NONE;
            break;
        }
    }

    return retval;
}

/*
  Wait for a key press and return the keycode.
  */
static int DIKeyboard_read_key(int translate)
{
    MSG msg;
    int osd_key;
    int i;

    StatusSetString("Waiting for keypress");

    while (MAME32App.PumpAndReturnMessage(&msg))
    {
        if (msg.message == WM_KEYDOWN || msg.message == WM_SYSKEYDOWN ||
            msg.message == WM_KEYUP   || msg.message == WM_SYSKEYUP)
        {
            int max_keys = translate ? OSD_MAX_PSEUDO : OSD_MAX_KEY;

            PollKeyboard();
            osd_key = 0;
            for (i = 1; i <= max_keys; i++)
            {
                if (This.m_key[i])
                {
                    if (translate)
                        osd_key = Keyboard_KeyToPseudoCode(i);
                    else
                        osd_key = i;
                    break;
                }
            }
            if (osd_key == 0)
                continue;
          
            return osd_key;
        }
    }

    /* can't get here */
    return 1;
}

/*
  Wait for a key press and return keycode.  Support repeat
  */
static int DIKeyboard_read_keyrepeat(int translate)
{
    int i;
   
    for (i = 0; i <= OSD_MAX_PSEUDO; i++) 
        This.m_key[i] = FALSE;
   
    return osd_read_key(translate);
}

/*
  return the name of a key
  */
static const char* DIKeyboard_key_name(int keycode)
{
    /* call keyboard implementation. */
   
    return Keyboard.key_name(keycode);
}

BOOL DIKeyboard_Available(void)
{
    HRESULT     hr;
    GUID        guidNULL = {0, 0, 0, {0, 0, 0, 0, 0, 0, 0, 0}};
    GUID        guidDevice = guidNULL;

    if (di == NULL)
    {
        return FALSE;
    }

    /* enumerate for keyboard devices */
    hr = IDirectInput_EnumDevices(di, DIDEVTYPE_KEYBOARD,
                 (LPDIENUMDEVICESCALLBACK)inputEnumDeviceProc,
                 &guidDevice,
                 DIEDFL_ATTACHEDONLY);
    if (FAILED(hr))
    {
        return FALSE;
    }

    if (!IsEqualGUID(&guidDevice, &guidNULL))
    {
        return TRUE;
    }

    return FALSE;
}

/***************************************************************************
    Message handlers
 ***************************************************************************/

static BOOL DIKeyboard_OnMessage(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam, LRESULT* pResult)
{
    switch (Msg)
    {
        PEEK_MESSAGE(hWnd, WM_ACTIVATEAPP,  OnActivateApp);
        HANDLE_MESSAGE(hWnd, WM_KEYDOWN,    OnKey);
        HANDLE_MESSAGE(hWnd, WM_KEYUP,      OnKey);
    }
    return FALSE;
}

static void OnActivateApp(HWND hWnd, BOOL fActivate, DWORD dwThreadId)
{
    if (!This.m_didKeyboard)
        return;
   
    if (MAME32App.m_bIsActive == TRUE)
        IDirectInputDevice_Acquire(This.m_didKeyboard);
    else
        IDirectInputDevice_Unacquire(This.m_didKeyboard);

    PollKeyboard();
}

static void OnKey(HWND hwnd, UINT vk, BOOL fDown, int cRepeat, UINT flags)
{
}

/***************************************************************************
  Internal functions
 ***************************************************************************/

/* Simulate a keypress. */
static void DIKeyboard_PressKey(int keycode)
{
    This.m_internal_key[keycode] = 1;
}

static void PollKeyboard()
{
    BYTE    diks[256];             /* DirectInput keyboard state buffer */
    HRESULT hr;
    int i;
    extern int mame_debug;

again:
   
    hr = IDirectInputDevice_GetDeviceState(This.m_didKeyboard, sizeof(diks), &diks);
    if (hr == DIERR_INPUTLOST || hr == DIERR_NOTACQUIRED) 
    {
        /*
            DirectInput is telling us that the input stream has
            been interrupted.  We aren't tracking any state
            between polls, so we don't have any special reset
            that needs to be done.  We just re-acquire and
            try again.
        */
        hr = IDirectInputDevice_Acquire(This.m_didKeyboard);
        if (SUCCEEDED(hr)) 
            goto again;
    }

    if (SUCCEEDED(hr))
    {
       for (i = 1; i <= OSD_MAX_KEY; i++) 
          This.m_key[i] = (diks[i] & 0x80);
       
       /* map in the gray arrow keys too */
       for (i = DIK_HOME; i <= DIK_DELETE; i++)
          if (diks[i] & 0x80)
             This.m_key[i - 0x80] = 1;
       
       /* and the right alt & control keys */
       if (diks[DIK_RMENU] & 0x80)
          This.m_key[OSD_KEY_ALTGR] = 1;
       
       if (diks[DIK_RCONTROL] & 0x80)
          This.m_key[OSD_KEY_RCONTROL] = 1;
       
       /* the Enter on numeric keypad */
       if (diks[DIK_NUMPADENTER] & 0x80)
          This.m_key[OSD_KEY_ENTER_PAD] = 1;
       
       if (DIJoystick_IsHappInterface())
       {
          /* let it control player 1 start '1', player 2 start '2', and
             escape keys */
          
          if (DIJoystick_KeyPressed(OSD_KEY_1))
             This.m_key[OSD_KEY_1] = 1;
          if (DIJoystick_KeyPressed(OSD_KEY_2))
             This.m_key[OSD_KEY_2] = 1;
          if (DIJoystick_KeyPressed(OSD_KEY_3))
             This.m_key[OSD_KEY_3] = 1;
          if (DIJoystick_KeyPressed(OSD_KEY_ESC))
             This.m_key[OSD_KEY_ESC] = 1;
       }
       
       /* map in the fake OS dependant keys */
       This.m_key[OSD_KEY_CANCEL]            = This.m_key[OSD_KEY_ESC];
       This.m_key[OSD_KEY_RESET_MACHINE]     = This.m_key[OSD_KEY_F3];
       This.m_key[OSD_KEY_PAUSE]             = This.m_key[OSD_KEY_P];
       This.m_key[OSD_KEY_UNPAUSE]           = This.m_key[OSD_KEY_P];
       This.m_key[OSD_KEY_CONFIGURE]         = This.m_key[OSD_KEY_TAB];
       This.m_key[OSD_KEY_ON_SCREEN_DISPLAY] = This.m_key[OSD_KEY_TILDE] && !mame_debug;
       This.m_key[OSD_KEY_SHOW_GFX]          = This.m_key[OSD_KEY_F4];
       This.m_key[OSD_KEY_JOY_CALIBRATE]     = This.m_key[OSD_KEY_F7];
       This.m_key[OSD_KEY_FRAMESKIP]         = This.m_key[OSD_KEY_F8];
       This.m_key[OSD_KEY_THROTTLE]          = This.m_key[OSD_KEY_F10];
       This.m_key[OSD_KEY_SHOW_FPS]          = This.m_key[OSD_KEY_F11] && !(This.m_key[OSD_KEY_LSHIFT] || This.m_key[OSD_KEY_RSHIFT]);
       This.m_key[OSD_KEY_SHOW_PROFILE]      = This.m_key[OSD_KEY_F11] &&  (This.m_key[OSD_KEY_LSHIFT] || This.m_key[OSD_KEY_RSHIFT]);
       This.m_key[OSD_KEY_SNAPSHOT]          = This.m_key[OSD_KEY_F12];
       This.m_key[OSD_KEY_CHEAT_TOGGLE]      = This.m_key[OSD_KEY_F5];
       This.m_key[OSD_KEY_DEBUGGER]          = This.m_key[OSD_KEY_TILDE] && mame_debug;
    }

    /* map in internally pressed keys */
    for (i=0;i<=OSD_MAX_PSEUDO;i++)
    {
        This.m_key[i] = This.m_key[i] || This.m_internal_key[i];
        This.m_internal_key[i] = 0;
    }

}



