/***************************************************************************

    M.A.M.E.32  -  Multiple Arcade Machine Emulator for Win32
    Win32 Portions Copyright (C) 1997-98 Michael Soderstrom and Chris Kirmse
    
    This file is part of MAME32, and may only be used, modified and
    distributed under the terms of the MAME license, in "readme.txt".
    By continuing to use, modify or distribute this file you indicate
    that you have read the license and understand and accept it fully.

 ***************************************************************************/

/***************************************************************************

  DIJoystick.c

 ***************************************************************************/

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <windowsx.h>
#include <stdlib.h>
#include <dinput.h>
#include <assert.h>
#include "mame32.h"
#include "DirectInput.h"
#include "DIJoystick.h"
#include "M32Util.h"
#include "trak.h"

/***************************************************************************
    function prototypes
 ***************************************************************************/

static int              DIJoystick_init(options_type *options);
static void             DIJoystick_exit(void);
static void             DIJoystick_poll_joystick(void);
static int              DIJoystick_joy_pressed(int joycode);
static const char*      DIJoystick_joy_name(int joycode);
static void             DIJoystick_analogjoy_read(int *analog_x, int *analog_y);
static int              DIJoystick_standard_analog_read(int axis);
static BOOL             DIJoystick_Available(int nJoyStick);
static BOOL             DIJoystick_OnMessage(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam, LRESULT* pResult);

static BOOL CALLBACK DIJoystick_EnumDeviceProc(LPDIDEVICEINSTANCE pdidi, LPVOID pv);

/***************************************************************************
    External variables
 ***************************************************************************/

struct OSDJoystick  DIJoystick = 
{
    { DIJoystick_init },                /* init              */
    { DIJoystick_exit },                /* exit              */
    { DIJoystick_poll_joystick },       /* poll_joystick     */
    { DIJoystick_joy_pressed },         /* joy_pressed       */
    { DIJoystick_joy_name },            /* joy_name          */
    { DIJoystick_analogjoy_read },      /* analogjoy_read    */
    { DIJoystick_standard_analog_read },/* standard_analog_read    */
    { DIJoystick_Available },           /* Available         */
    { DIJoystick_OnMessage },           /* OnMessage         */
};

/***************************************************************************
    Internal structures
 ***************************************************************************/

typedef struct
{
    BOOL        m_bUseJoystick;
    GUID        guidDevice;
    LPDIRECTINPUTDEVICE2 didJoystick;

    DIJOYSTATE  dijs;

    UINT        m_nDeadZone;

    BOOL m_bHasPOV; /* if the joystick has a pov hat, use it as joystick 2,
                       for the 2 joystick games (vanguard, robotron, etc.) */

    BOOL        m_bHappInterface;   /* used for Chris Kirmse's arcade joystick
                                    setup, using controls & interface board
                                    from www.happcontrols.com */

} joy_info_type;

#define MAX_JOYSTICKS 4

struct tDIJoystick_private
{
    int         use_count; /* the gui and game can both init/exit us, so keep
                              track */
    BOOL        m_bCoinSlot;        /* ditto */

    int num_joysticks;
    joy_info_type joy_info[MAX_JOYSTICKS]; /* actual joystick data! */

    BOOL m_bMouseLButton; /* mouse buttons, used to override some keys, happ only! */
    BOOL m_bMouseRButton;
};

/***************************************************************************
    Internal variables
 ***************************************************************************/

static struct tDIJoystick_private   This;

GUID guidNULL = {0, 0, 0, {0, 0, 0, 0, 0, 0, 0, 0}};

/***************************************************************************
    External OSD functions  
 ***************************************************************************/
#include <stdio.h>
/*
    put here anything you need to do when the program is started. Return 0 if 
    initialization was successful, nonzero otherwise.
*/
static int DIJoystick_init(options_type *options)
{
    int             i;
    HRESULT         hr;
    DIPROPRANGE     diprg;
    DIDEVCAPS       diJoystickCaps; 
    LPDIRECTINPUTDEVICE didTemp;

    This.m_bMouseLButton = FALSE;
    This.m_bMouseRButton = FALSE;

    This.use_count++;

    if (This.joy_info[0].didJoystick == NULL)
    {
        /* if not already setup... */
       
        for (i=0;i<MAX_JOYSTICKS;i++)
        {
            This.joy_info[i].m_bUseJoystick = TRUE;
            This.joy_info[i].m_nDeadZone = 20;
        }
    }
    
    if (!options->use_joystick)
    {
       for (i=0;i<MAX_JOYSTICKS;i++)
          This.joy_info[i].m_bUseJoystick = FALSE;
    }

    /* CMK no way to set dead zone right now */
    /*
      for (i=0;i<MAX_JOYSTICKS;i++)
          This.joy_info[i].m_nDeadZone = min(max(arg, 5), 95);

    if (This.joy_info[0].didJoystick != NULL)
        return 0; /* already initialized, so we're ok */

    This.num_joysticks = 0;

    if (di == NULL)
    {
        ErrorMsg("DirectInput not initialized");
        for (i=0;i<MAX_JOYSTICKS;i++)
           This.joy_info[i].m_bUseJoystick = FALSE;
        return 0;
    }

    if (This.joy_info[0].m_bUseJoystick == FALSE)
        return 0;

    /* enumerate for joystick devices */
    hr = IDirectInput_EnumDevices(di, DIDEVTYPE_JOYSTICK,
                 (LPDIENUMDEVICESCALLBACK)DIJoystick_EnumDeviceProc,
                 NULL,
                 DIEDFL_ALLDEVICES);
    if (FAILED(hr))
    {
        ErrorMsg("DirectInput EnumDevices() failed - %08Xh", hr);
        This.joy_info[0].m_bUseJoystick = FALSE;
        return 0;
    }

    for (i=This.num_joysticks;i<MAX_JOYSTICKS;i++)
       This.joy_info[i].m_bUseJoystick = 0;

    /* printf("found joysticks %i\n",This.num_joysticks); */

    /* Are there any joysticks attached? */
    if (This.num_joysticks < 1)
    {
        ErrorMsg("DirectInput EnumDevices didn't find any joysticks");
        This.joy_info[0].m_bUseJoystick = FALSE;

        return 0;
    }
    
    for (i=0;i<This.num_joysticks;i++)
    {
        This.joy_info[0].m_bHappInterface   = FALSE;
       
        /* get a did1 interface first... */
        hr = IDirectInput_CreateDevice(di, &This.joy_info[i].guidDevice, &didTemp, NULL);
        if (FAILED(hr))
        {
            ErrorMsg("DirectInput CreateDevice() joystick failed - %08Xh\n", hr);
            This.joy_info[i].m_bUseJoystick = FALSE;
            continue;
        }

        /* get a did2 interface to work with polling (most) joysticks */
        hr = IDirectInputDevice_QueryInterface(didTemp,
                                               &IID_IDirectInputDevice2,
                                               &This.joy_info[i].didJoystick);
        if (FAILED(hr))
        {
            ErrorMsg("DirectInput QueryInterface joystick failed\n");
            This.joy_info[i].m_bUseJoystick = FALSE;
            continue;
        }

        /* dispose of the temp interface */
        IDirectInputDevice_Release(didTemp);

        hr = IDirectInputDevice2_SetDataFormat(This.joy_info[i].didJoystick, &c_dfDIJoystick);
        if (FAILED(hr))
        {
            ErrorMsg("DirectInput SetDataFormat() joystick failed - %08Xh\n", hr);
            This.joy_info[i].m_bUseJoystick = FALSE;
            continue;
        }

        hr = IDirectInputDevice2_SetCooperativeLevel(This.joy_info[i].didJoystick,MAME32App.m_hWnd,
                                                     DISCL_NONEXCLUSIVE | DISCL_BACKGROUND);
        if (FAILED(hr))
        {
            ErrorMsg("DirectInput SetCooperativeLevel() joystick failed - %08Xh\n", hr);
            This.joy_info[i].m_bUseJoystick = FALSE;
            continue;
        }

        /*  Set X-axis range to (0...255) */

        diprg.diph.dwSize       = sizeof(diprg);
        diprg.diph.dwHeaderSize = sizeof(diprg.diph);
        diprg.diph.dwObj        = DIJOFS_X;
        diprg.diph.dwHow        = DIPH_BYOFFSET;
        diprg.lMin              = 0;
        diprg.lMax              = 255;
        
        hr = IDirectInputDevice2_SetProperty(This.joy_info[i].didJoystick, DIPROP_RANGE, 
                                             &diprg.diph);
        if (hr != DI_OK)
        {
            ErrorMsg("DirectInput SetProperty() joystick x axis failed - %08Xh\n", hr);
            This.joy_info[i].m_bUseJoystick = FALSE;
            continue;
        }

        /* And again for Y-axis range. */

        diprg.diph.dwObj        = DIJOFS_Y;

        hr = IDirectInputDevice2_SetProperty(This.joy_info[i].didJoystick, DIPROP_RANGE, 
                                             &diprg.diph);
        if (hr != DI_OK)
        {
            ErrorMsg("DirectInput SetProperty() joystick y axis failed - %08Xh\n", hr);
            This.joy_info[i].m_bUseJoystick = FALSE;
            continue;
        }

        /* Set X axis dead zone to 0; we need accurate #'s for analog joystick reading. */

        hr = SetDIDwordProperty(This.joy_info[i].didJoystick, DIPROP_DEADZONE, DIJOFS_X, 
                                DIPH_BYOFFSET,0);
        if (hr != DI_OK)
        {
            ErrorMsg("DirectInput SetDIDwordProperty() joystick x axis failed - %08Xh\n", hr);
            continue;
        }

        /* Set Y axis dead zone to 0; we need accurate #'s for analog joystick reading. */

        hr = SetDIDwordProperty(This.joy_info[i].didJoystick, DIPROP_DEADZONE, DIJOFS_Y, 
                                DIPH_BYOFFSET,0);
        if (hr != DI_OK)
        {
            ErrorMsg("DirectInput SetDIDwordProperty() joystick y axis failed - %08Xh\n", hr);
            continue;
        }

        diJoystickCaps.dwSize = sizeof(DIDEVCAPS); 
        hr = IDirectInputDevice_GetCapabilities(This.joy_info[i].didJoystick, &diJoystickCaps); 

        This.joy_info[i].m_bHappInterface = (diJoystickCaps.dwButtons == 24);
        This.joy_info[i].m_bHasPOV = (diJoystickCaps.dwPOVs > 0);

        if (!(diJoystickCaps.dwFlags & DIDC_ATTACHED))
            This.joy_info[i].m_bUseJoystick = FALSE;

        hr = IDirectInputDevice2_Acquire(This.joy_info[i].didJoystick);
        if (FAILED(hr)) 
        {
            ErrorMsg("DirectInputDevice Acquire joystick failed!\n");
            continue;
        }
    }

    /* for (i=0;i<MAX_JOYSTICKS;i++)
       printf("%i\n",This.joy_info[i].m_bUseJoystick); */

    return 0;
}

/*
    put here cleanup routines to be executed when the program is terminated.
*/
static void DIJoystick_exit(void)
{
    int i;

    This.use_count--;

    if (This.use_count > 0)
        return;

    for (i=0;i<MAX_JOYSTICKS;i++)
       if (This.joy_info[i].didJoystick) 
       {
          IDirectInputDevice_Unacquire(This.joy_info[i].didJoystick);
          IDirectInputDevice_Release(This.joy_info[i].didJoystick);
          This.joy_info[i].didJoystick = NULL;
       }
    
}

static void DIJoystick_poll_joystick(void)
{
    HRESULT     hr;
    int i;
    
    This.m_bCoinSlot = 0;

    This.m_bMouseLButton = (GetKeyState(VK_LBUTTON) & 0x8000);
    This.m_bMouseRButton = (GetKeyState(VK_RBUTTON) & 0x8000);

    for (i=0;i<MAX_JOYSTICKS;i++)
    {
       /* start by clearing the structure, then fill it in if possible */

       memset(&This.joy_info[i].dijs,0,sizeof(DIJOYSTATE));
       This.joy_info[i].dijs.lX = 128;
       This.joy_info[i].dijs.lY = 128;
       This.joy_info[i].dijs.lRz = 32768;
       This.joy_info[i].dijs.rgdwPOV[0] = -1;

       if (!This.joy_info[i].didJoystick || !This.joy_info[i].m_bUseJoystick)
           continue;

       hr = IDirectInputDevice2_Poll(This.joy_info[i].didJoystick);

       hr = IDirectInputDevice2_GetDeviceState(This.joy_info[i].didJoystick,sizeof(DIJOYSTATE),
                                               &This.joy_info[i].dijs);
       if (FAILED(hr))
       {
           if (hr == DIERR_INPUTLOST || hr == DIERR_NOTACQUIRED)
           {
              hr = IDirectInputDevice2_Acquire(This.joy_info[i].didJoystick);
           }
           continue;
       }

       if (This.joy_info[i].m_bHappInterface)
          if (This.joy_info[i].dijs.rgbButtons[12] & 0x80)
             This.m_bCoinSlot = 1;

    }
}

/*
    check if the DIJoystick is moved in the specified direction, defined in
    osdepend.h. Return 0 if it is not pressed, nonzero otherwise.
*/

static int DIJoystick_joy_pressed(int joycode)
{
    int i;

    switch (joycode)
    {
    case OSD_JOY_LEFT:
        return This.joy_info[0].dijs.lX < 128 - 128*This.joy_info[0].m_nDeadZone/100;

    case OSD_JOY_RIGHT:
        return This.joy_info[0].dijs.lX > 128 + 128*This.joy_info[0].m_nDeadZone/100;

    case OSD_JOY_UP:
        return This.joy_info[0].dijs.lY < 128 - 128*This.joy_info[0].m_nDeadZone/100;
           
    case OSD_JOY_DOWN:
        return This.joy_info[0].dijs.lY > 128 + 128*This.joy_info[0].m_nDeadZone/100;

    case OSD_JOY_FIRE1:
    case OSD_JOY_FIRE2:
    case OSD_JOY_FIRE3:
    case OSD_JOY_FIRE4:
    case OSD_JOY_FIRE5:
    case OSD_JOY_FIRE6:
    case OSD_JOY_FIRE7:
    case OSD_JOY_FIRE8:
    case OSD_JOY_FIRE9:
    case OSD_JOY_FIRE10:
        return This.joy_info[0].dijs.rgbButtons[joycode-OSD_JOY_FIRE1] & 0x80;

    case OSD_JOY_FIRE:
        for (i=0;i<10;i++)
            if (This.joy_info[0].dijs.rgbButtons[i] & 0x80)
                return TRUE;
        return FALSE;

    case OSD_JOY2_LEFT:
        return This.joy_info[1].dijs.lX < 128 - 128*This.joy_info[1].m_nDeadZone/100;

    case OSD_JOY2_RIGHT:
        return This.joy_info[1].dijs.lX > 128 + 128*This.joy_info[1].m_nDeadZone/100;

    case OSD_JOY2_UP:
        return This.joy_info[1].dijs.lY < 128 - 128*This.joy_info[1].m_nDeadZone/100;
           
    case OSD_JOY2_DOWN:
        return This.joy_info[1].dijs.lY > 128 + 128*This.joy_info[1].m_nDeadZone/100;

    case OSD_JOY2_FIRE1:
    case OSD_JOY2_FIRE2:
    case OSD_JOY2_FIRE3:
    case OSD_JOY2_FIRE4:
    case OSD_JOY2_FIRE5:
    case OSD_JOY2_FIRE6:
    case OSD_JOY2_FIRE7:
    case OSD_JOY2_FIRE8:
    case OSD_JOY2_FIRE9:
    case OSD_JOY2_FIRE10:
        if (This.joy_info[1].m_bUseJoystick == FALSE)
            return 0;
        return This.joy_info[1].dijs.rgbButtons[joycode-OSD_JOY2_FIRE1] & 0x80;

    case OSD_JOY2_FIRE:
        if (This.joy_info[1].m_bUseJoystick == FALSE)
            return 0;
        for (i=0;i<10;i++)
            if (This.joy_info[1].dijs.rgbButtons[i] & 0x80)
                return TRUE;
        return FALSE;


    case OSD_JOY3_LEFT :
        if (This.joy_info[2].m_bUseJoystick == TRUE)
            if (This.joy_info[2].dijs.lX < 128 - 128*This.joy_info[2].m_nDeadZone/100)
                return 1;
       if (LOWORD(This.joy_info[0].dijs.rgdwPOV[0]) == 0xffff)
           return 0;
       return (This.joy_info[0].m_bHasPOV || This.joy_info[0].m_bHappInterface) && 
           This.joy_info[0].dijs.rgdwPOV[0] >= 22500 && This.joy_info[0].dijs.rgdwPOV[0] <= 31500;
        
    case OSD_JOY3_RIGHT :
        if (This.joy_info[2].m_bUseJoystick == TRUE)
            if (This.joy_info[2].dijs.lX > 128 + 128*This.joy_info[2].m_nDeadZone/100)
                return 1;

        if (LOWORD(This.joy_info[0].dijs.rgdwPOV[0]) == 0xffff)
            return 0;
        return (This.joy_info[0].m_bHasPOV || This.joy_info[0].m_bHappInterface) && 
            This.joy_info[0].dijs.rgdwPOV[0] >= 4500 && This.joy_info[0].dijs.rgdwPOV[0] <= 13500;

    case OSD_JOY3_UP :
        if (This.joy_info[2].m_bUseJoystick == TRUE)
            if (This.joy_info[2].dijs.lY < 128 - 128*This.joy_info[2].m_nDeadZone/100)
                return 1;
        if (LOWORD(This.joy_info[0].dijs.rgdwPOV[0]) == 0xffff)
            return 0;
        return (This.joy_info[0].m_bHasPOV || This.joy_info[0].m_bHappInterface) && 
           (This.joy_info[0].dijs.rgdwPOV[0] <= 4500 || This.joy_info[0].dijs.rgdwPOV[0] >= 31500);
       
    case OSD_JOY3_DOWN :
        if (This.joy_info[2].m_bUseJoystick == TRUE)
            if (This.joy_info[2].dijs.lY > 128 + 128*This.joy_info[2].m_nDeadZone/100)
                return 1;
       if (LOWORD(This.joy_info[0].dijs.rgdwPOV[0]) == 0xffff)
           return 0;
        return (This.joy_info[0].m_bHasPOV || This.joy_info[0].m_bHappInterface) && 
            This.joy_info[0].dijs.rgdwPOV[0] >= 13500 && This.joy_info[0].dijs.rgdwPOV[0] <= 22500;

    case OSD_JOY3_FIRE1:
    case OSD_JOY3_FIRE2:
    case OSD_JOY3_FIRE3:
    case OSD_JOY3_FIRE4:
    case OSD_JOY3_FIRE5:
    case OSD_JOY3_FIRE6:
    case OSD_JOY3_FIRE7:
    case OSD_JOY3_FIRE8:
    case OSD_JOY3_FIRE9:
    case OSD_JOY3_FIRE10:
       if (This.joy_info[2].m_bUseJoystick == FALSE)
           return 0;
       return This.joy_info[2].dijs.rgbButtons[joycode-OSD_JOY3_FIRE1] & 0x80;

    case OSD_JOY3_FIRE:
        if (This.joy_info[2].m_bUseJoystick == FALSE)
            return 0;
        for (i=0;i<10;i++)
            if (This.joy_info[2].dijs.rgbButtons[2] & 0x80)
                return TRUE;
        return FALSE;

    case OSD_JOY4_LEFT:
        return This.joy_info[3].dijs.lX < 128 - 128*This.joy_info[3].m_nDeadZone/100;

    case OSD_JOY4_RIGHT:
        return This.joy_info[3].dijs.lX > 128 + 128*This.joy_info[3].m_nDeadZone/100;

    case OSD_JOY4_UP:
        return This.joy_info[3].dijs.lY < 128 - 128*This.joy_info[3].m_nDeadZone/100;
           
    case OSD_JOY4_DOWN:
        return This.joy_info[3].dijs.lY > 128 + 128*This.joy_info[3].m_nDeadZone/100;

    case OSD_JOY4_FIRE1:
    case OSD_JOY4_FIRE2:
    case OSD_JOY4_FIRE3:
    case OSD_JOY4_FIRE4:
    case OSD_JOY4_FIRE5:
    case OSD_JOY4_FIRE6:
    case OSD_JOY4_FIRE7:
    case OSD_JOY4_FIRE8:
    case OSD_JOY4_FIRE9:
    case OSD_JOY4_FIRE10:
        if (This.joy_info[3].m_bUseJoystick == FALSE)
            return 0;
        return This.joy_info[3].dijs.rgbButtons[joycode-OSD_JOY4_FIRE1] & 0x80;

    case OSD_JOY4_FIRE:
        if (This.joy_info[3].m_bUseJoystick == FALSE)
            return 0;
        for (i=0;i<10;i++)
            if (This.joy_info[3].dijs.rgbButtons[i] & 0x80)
                return TRUE;
        return FALSE;

    default:
        return 0;
    }
    return 0;
}

/*
    Return the name of a Joystick button.
*/
static const char* DIJoystick_joy_name(int joycode)
{
    return Joystick.joy_name(joycode);
}

/* osd_analog_joyread() returns values from -128 to 128 */
static void DIJoystick_analogjoy_read(int *analog_x, int *analog_y)
{
    *analog_x = *analog_y = 0;

    if (This.joy_info[0].m_bUseJoystick == FALSE)
        return;

    *analog_x = This.joy_info[0].dijs.lX - 128;
    *analog_y = This.joy_info[0].dijs.lY - 128;
}

static int DIJoystick_standard_analog_read(int axis)
{
    int retval;

    switch (axis)
    {
        case X_AXIS:
            retval = (This.joy_info[0].dijs.lRz - 32768) / (32768/(TRAK_MAXX_RES/2));
            if (retval > TRAK_MAXX_RES)
               retval = TRAK_MAXX_RES;
            if (retval < -TRAK_MAXX_RES)
               retval = -TRAK_MAXX_RES;
            return retval;
        case Y_AXIS:
            retval = 0;
            return retval;
    }
    return 0;
}


static BOOL DIJoystick_Available(int nJoyStick)
{
    static BOOL bBeenHere = FALSE;
    static BOOL bAvailable = FALSE;
    HRESULT     hr;
    GUID        guidDevice = guidNULL;
    LPDIRECTINPUTDEVICE didTemp;
    LPDIRECTINPUTDEVICE didJoystick;

    if (di == NULL)
    {
        return FALSE;
    }

    if (bBeenHere == FALSE)
        bBeenHere = TRUE;
    else
        return bAvailable;

    /* enumerate for joystick devices */
    hr = IDirectInput_EnumDevices(di, DIDEVTYPE_JOYSTICK,
                 (LPDIENUMDEVICESCALLBACK)inputEnumDeviceProc,
                 &guidDevice,
                 DIEDFL_ALLDEVICES);
    if (FAILED(hr))
    {
       return FALSE;
    }

    /* Are there any joysticks attached? */
    if (IsEqualGUID(&guidDevice, &guidNULL))
    {
        return FALSE;
    }

    hr = IDirectInput_CreateDevice(di, &guidDevice, &didTemp, NULL);
    if (FAILED(hr))
    {
        return FALSE;
    }

    /* Determine if DX5 is available by a QI for a DX5 interface. */
    hr = IDirectInputDevice_QueryInterface(didTemp,
                                           &IID_IDirectInputDevice2,
                                           &didJoystick);
    if (FAILED(hr))
    {
        bAvailable = FALSE;
    }
    else
    {
        bAvailable = TRUE;
        IDirectInputDevice_Release(didJoystick);
    }

    /* dispose of the temp interface */
    IDirectInputDevice_Release(didTemp);

    return bAvailable;
}

static BOOL DIJoystick_OnMessage(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam, LRESULT* pResult)
{
    return FALSE;
}

BOOL DIJoystick_IsHappInterface()
{
    return This.num_joysticks > 0 && This.joy_info[0].m_bHappInterface;
}

BOOL DIJoystick_KeyPressed(int osd_key)
{
    if (!This.joy_info[0].m_bHappInterface)
        return FALSE;

    switch (osd_key)
    {
    case OSD_KEY_1 :
        return This.m_bMouseLButton;
    case OSD_KEY_2 :
        return This.m_bMouseRButton;
    case OSD_KEY_3 :
        return This.m_bCoinSlot;
    case OSD_KEY_ESC :
        return This.m_bMouseLButton && This.m_bMouseRButton && 
           DIJoystick_joy_pressed(OSD_JOY_DOWN);
    }
    return FALSE;
}


/***************************************************************************
    Internal functions
 ***************************************************************************/
BOOL CALLBACK DIJoystick_EnumDeviceProc(LPDIDEVICEINSTANCE pdidi, LPVOID pv)
{
    This.joy_info[This.num_joysticks].guidDevice = pdidi->guidInstance;

    This.num_joysticks++;

    if (This.num_joysticks < 4)
       return DIENUM_CONTINUE;

    return DIENUM_STOP;

}
