/*
 * Snes9x - Portable Super Nintendo Entertainment System (TM) emulator.
 *
 * (c) Copyright 1996, 1997, 1998 Gary Henderson (gary@daniver.demon.co.uk) and
 *                                Jerremy Koot (jkoot@euronet.nl)
 *
 * Super FX C emulator code 
 * (c) Copyright 1997, 1998 Lestat (lstat@hotmail.com) and
 *                          Gary Henderson.
 * Super FX assembler emulator code (c) Copyright 1998 zsKnight and _Demo_.
 *
 * DSP1 emulator code (c) Copyright 1998 Lestat and Gary Henderson.
 * DOS port code contains the works of other authors. See headers in
 * individual files.
 *
 * Permission to use, copy, modify and distribute Snes9x in both binary and
 * source form, for non-commercial purposes, is hereby granted without fee,
 * providing that this license information and copyright notice appear with
 * all copies and any derived work.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event shall the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Snes9x is freeware for PERSONAL USE only. Commercial users should
 * seek permission of the copyright holders first. Commercial use includes
 * charging money for Snes9x or software derived from Snes9x.
 *
 * The copyright holders request that bug fixes and improvements to the code
 * should be forwarded to them so everyone can benefit from the modifications
 * in future versions.
 *
 * Super NES and Super Nintendo Entertainment System are trademarks of
 * Nintendo Co., Limited and its subsidiary companies.
 */

#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <errno.h>

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <signal.h>
#include <crt0.h>
#include <dir.h>

#include <allegro.h>
#undef   TRUE
#undef   FALSE

#include "snes9x.h"
#include "memmap.h"
#include "debug.h"
#include "cpuexec.h"
#include "ppu.h"
#include "snapshot.h"
#include "apu.h"
#include "display.h"
#include "gfx.h"
#include "soundux.h"

#include "inifuncs.h"
#include "cheats.h"
#include "gui.h"

uint32 joypads [5] = {0};
int NumControllers = 2;

#ifdef JOYSTICK_SUPPORT
void InitJoysticks ();
void ReadJoysticks ();
#endif

bool8 joystick_has_four_buttons = FALSE;
bool8 joystick_has_six_buttons = FALSE;
bool8 joystick_has_eight_buttons = FALSE;

#ifdef SIDEWINDER_SUPPORT
void InitSidewinders ();
int ReadSidewinders ();
bool8 alternate_sidewinder_mapping = FALSE;
#endif

void InitTimer ();
void S9xActivateUI ();

char *rom_filename = NULL;
char *snapshot_filename = NULL;

extern char FileFormat[], ScreenShotPath[];
extern bool8 nothingloaded, Windows_Friendly;
char FileFormat[80];
char ScreenShotPath[200], SnapshotPath[200], RomPath[200];
bool8 nothingloaded=TRUE, Windows_Friendly=TRUE;
bool8 UseGUI = TRUE;

void OutOfMemory ()
{
    fprintf (stderr, "\
Snes9X: Memory allocation failure - not enough RAM/virtual memory available.\n\
        S9xExiting...\n");
    Memory.Deinit ();
    S9xDeinitAPU ();
    
    exit (1);
}

void S9xParseArg (char **argv, int &i, int argc)
{
    if (strcmp (argv [i], "-4") == 0 ||
	     strcasecmp (argv [i], "-four") == 0)
    {
	joystick_has_four_buttons = TRUE;
	Settings.JoystickEnabled = 1;
    }
    else if (strcmp (argv [i], "-6") == 0 ||
	     strcasecmp (argv [i], "-six") == 0)
    {
	joystick_has_six_buttons = TRUE;
	Settings.JoystickEnabled = 1;
    }
    else if (strcmp (argv [i], "-8") == 0 ||
	     strcasecmp (argv [i], "-eight") == 0)
    {
	joystick_has_eight_buttons = TRUE;
	Settings.JoystickEnabled = 1;
    }
    else if (strcmp (argv [i], "-j") == 0 ||
	     strcasecmp (argv [i], "-nojoy") == 0)
	Settings.JoystickEnabled = 0;
    else if (strcasecmp (argv [i], "-b") == 0 ||
	     strcasecmp (argv [i], "-bs") == 0 ||
	     strcasecmp (argv [i], "-buffersize") == 0)
    {
	if (i + 1 < argc)
	    Settings.SoundBufferSize = atoi (argv [++i]);
	else
	    S9xUsage ();
    }
    else if (strcmp (argv [i], "-l") == 0 ||
	     strcasecmp (argv [i], "-loadsnapshot") == 0)
    {
	if (i + 1 < argc)
	    snapshot_filename = argv [++i];
	else
	    S9xUsage ();
    }
#ifdef SIDEWINDER_SUPPORT
    else if (strcmp (argv [i], "-a") == 0)
    {
	alternate_sidewinder_mapping = TRUE;
    }
#endif
    else
	S9xParseDisplayArg (argv, i, argc);
}

int main (int argc, char **argv)
{
    bool8 AutoGUI = FALSE;
    _crt0_startup_flags = _CRT0_FLAG_FILL_SBRK_MEMORY;

    if (argc < S9xMinCommandLineArgs ())
    {
        AutoGUI = TRUE;
        //S9xUsage ();
    }

    ZeroMemory (&Settings, sizeof (Settings));

#ifdef JOYSTICK_SUPPORT
    Settings.JoystickEnabled = TRUE;
#else
    Settings.JoystickEnabled = FALSE;
#endif

    Settings.SoundPlaybackRate = 4;
    Settings.Stereo = TRUE;
    Settings.SoundBufferSize = 0;
    Settings.SPCTo65c816Ratio = 2;
    Settings.DisableSoundEcho = FALSE;
    Settings.APUEnabled = TRUE;
    Settings.H_Max = (100 * SNES_CYCLES_PER_SCANLINE) / 100;
    Settings.SkipFrames = AUTO_FRAMERATE;
    Settings.Shutdown = TRUE;
    Settings.FrameTimePAL = 20000;
    Settings.FrameTimeNTSC = 16667;
    Settings.FrameTime = Settings.FrameTimeNTSC;
    Settings.DisableSampleCaching = FALSE;
    Settings.DisableMasterVolume = FALSE;
    Settings.Mouse = TRUE;
    Settings.SuperScope = TRUE;
    Settings.MultiPlayer5 = TRUE;
    Settings.ControllerOption = SNES_MULTIPLAYER5;
    Settings.Transparency = FALSE;
    Settings.SixteenBit = FALSE;
    Settings.SupportHiRes = FALSE;

    rom_filename = S9xParseArgs (argv, argc);

    Settings.Transparency = Settings.ForceTransparency;
    if (Settings.ForceNoTransparency)
	Settings.Transparency = FALSE;

    if (Settings.Transparency)
	Settings.SixteenBit = TRUE;

    Settings.HBlankStart = (256 * Settings.H_Max) / SNES_HCOUNTER_MAX;

    if (!Memory.Init () || !S9xInitAPU())
	OutOfMemory ();

    uint32 saved_flags = CPU.Flags;
    if (!snapshot_filename)
    {
	if (rom_filename)
	{
	    if (!Memory.LoadROM (rom_filename))
	    {
		char dir [_MAX_DIR];
		char drive [_MAX_DRIVE];
		char name [_MAX_FNAME];
		char ext [_MAX_EXT];
		char fname [_MAX_PATH];

		_splitpath (rom_filename, drive, dir, name, ext);
		_makepath (fname, drive, dir, name, ext);

		strcpy (fname, S9xGetROMDirectory ());
		strcat (fname, SLASH_STR);
		strcat (fname, name);
		if (ext [0])
		{
		    strcat (fname, ".");
		    strcat (fname, ext);
		}
		_splitpath (fname, drive, dir, name, ext);
		_makepath (fname, drive, dir, name, ext);
		if (!Memory.LoadROM (fname))
		{
		    printf ("Error opening: %s\n", rom_filename);
		    exit (1);
		}
                nothingloaded = FALSE;
	    }
	    Memory.LoadSRAM (S9xGetSRAMFilename ());
	}
	else
	{
	    S9xReset ();
	    Settings.Paused |= 2;
	}
    }
    CPU.Flags = saved_flags;

    S9xInitInputDevices ();

    S9xInitDisplay (argc, argv);
    if (!S9xGraphicsInit ())
	OutOfMemory ();
	
    S9xTextMode ();
    (void) S9xInitSound (Settings.SoundPlaybackRate, Settings.Stereo,
		      Settings.SoundBufferSize);
    if (!Settings.APUEnabled)
	S9xSetSoundMute (TRUE);

    if (snapshot_filename)
    {
	int Flags = CPU.Flags & (DEBUG_MODE_FLAG | TRACE_FLAG);
	if (!S9xLoadSnapshot (snapshot_filename))
	    exit (1);
	CPU.Flags |= Flags;
    }

    S9xGraphicsMode ();

    sprintf (String, "\"%s\" %s: %s", Memory.ROMName, TITLE, VERSION);
    S9xSetTitle (String);
    
#ifdef JOYSTICK_SUPPORT
    uint32 JoypadSkip = 0;
#endif

    InitTimer ();
    if (!Settings.APUEnabled)
	S9xSetSoundMute (FALSE);

    if (rom_filename == NULL)
       AutoGUI = TRUE;

    if (AutoGUI)
        S9xActivateUI();

    while (1)
    {
	if (!Settings.Paused
#ifdef DEBUGGER
	    || (CPU.Flags & (DEBUG_MODE_FLAG | SINGLE_STEP_FLAG))
#endif
           )
	    S9xMainLoop ();

	if (Settings.Paused
#ifdef DEBUGGER
	    || (CPU.Flags & DEBUG_MODE_FLAG)
#endif
           )
	{
	    S9xSetSoundMute (TRUE);
	}

#ifdef DEBUGGER
	if (CPU.Flags & DEBUG_MODE_FLAG)
	{
	    S9xDoDebug ();
	}
	else
#endif
	if (Settings.Paused)
	    S9xProcessEvents (TRUE);

#ifdef JOYSTICK_SUPPORT
	if (Settings.JoystickEnabled && (JoypadSkip++ & 1) == 0)
	    ReadJoysticks ();
#endif
	S9xProcessEvents (FALSE);
	
	if (!Settings.Paused
#ifdef DEBUGGER
	    && !(CPU.Flags & DEBUG_MODE_FLAG)
#endif	    
           )
	{
	    S9xSetSoundMute (FALSE);
	}
    }
    return (0);
}

void S9xExit ()
{
    S9xDeinitDisplay ();
    Memory.SaveSRAM (S9xGetSRAMFilename ());
    Memory.Deinit ();
    S9xDeinitAPU ();
    
    exit (0);
}

void S9xActivateUI ()
{
    S9xTextMode();
    S9xSetSoundMute (TRUE);
    if (UseGUI)
        if (mainmnu()==-1)
            S9xExit();
        else {
            S9xGraphicsMode();
            if (Settings.APUEnabled)
                S9xSetSoundMute (FALSE);
            extern volatile uint32 FrameTimer;
            FrameTimer = 0;
            return;
        }
    S9xExit();
}

#ifdef SIDEWINDER_SUPPORT
int SidewinderToSNES [] =
{
    SNES_UP_MASK, SNES_DOWN_MASK, SNES_RIGHT_MASK, SNES_LEFT_MASK,
    SNES_B_MASK, SNES_A_MASK, SNES_A_MASK, 
    SNES_Y_MASK, SNES_X_MASK, SNES_B_MASK, 
    SNES_TL_MASK, SNES_TR_MASK,
    SNES_START_MASK, SNES_SELECT_MASK
};

int SidewinderToSNES2 [] =
{
    SNES_UP_MASK, SNES_DOWN_MASK, SNES_RIGHT_MASK, SNES_LEFT_MASK,
    SNES_A_MASK, SNES_B_MASK, SNES_A_MASK, 
    SNES_X_MASK, SNES_Y_MASK, SNES_B_MASK, 
    SNES_TL_MASK, SNES_TR_MASK,
    SNES_START_MASK, SNES_SELECT_MASK
};
#endif

const char *GetHomeDirectory ()
{
#ifdef __MSDOS
    return (".");
#else
    return (getenv ("HOME"));
#endif
}

const char *S9xGetSnapshotDirectory ()
{
    static char filename [PATH_MAX];
    const char *snapshot;
    
    if (!(snapshot = getenv ("SNES96_SNAPSHOT_DIR")))
    {
        if (strcmp(SnapshotPath, "") != 0)
        {
            strcpy (filename, SnapshotPath);
            mkdir (filename, 0777);
        }
        else
        {
            strcpy (filename, "");
        }
    }
    else
	return (snapshot);

    return (filename);
}

const char *S9xGetSRAMFilename ()
{
    static char filename [PATH_MAX];
    char drive [_MAX_DRIVE];
    char dir [_MAX_DIR];
    char fname [_MAX_FNAME];
    char ext [_MAX_EXT];

    _splitpath (Memory.ROMFilename, drive, dir, fname, ext);
    strcpy (filename, S9xGetSnapshotDirectory ());
    if (strcmp(filename, "") == 0)
    {
        fnmerge( filename, drive, dir, fname, ".srm");
    }
    else
    {
        strcat (filename, SLASH_STR);
        strcat (filename, fname);
        strcat (filename, ".srm");
    }

    return (filename);
}

const char *S9xGetROMDirectory ()
{
    const char *roms;
    
    if (strcmp(RomPath, "") != 0)
        return RomPath;
    else
        if (!(roms = getenv ("SNES96_ROM_DIR")))
            return ("");
        else
            return (roms);
}

const char *basename (const char *f)
{
    const char *p;
    if ((p = strrchr (f, '/')) != NULL || (p = strrchr (f, '\\')) != NULL)
	return (p + 1);

    if (p = strrchr (f, SLASH_CHAR))
	return (p + 1);

    return (f);
}

const char *S9xChooseFilename (bool8 read_only)
{
    char def [PATH_MAX];
    char title [PATH_MAX];
    char drive [_MAX_DRIVE];
    char dir [_MAX_DIR];
    char ext [_MAX_EXT];

    fnsplit (Memory.ROMFilename, drive, dir, def, ext);
    //strcat (def, ".s96");
    sprintf (title, "%s snapshot filename",
	    read_only ? "Select load" : "Choose save");
    const char *filename;

    S9xSetSoundMute (TRUE);
    filename = S9xSelectFilename (def, S9xGetSnapshotDirectory (), "s96", title);
    S9xSetSoundMute (FALSE);
    return (filename);
}

bool8 S9xOpenSnapshotFile (const char *fname, bool8 read_only, STREAM *file)
{
    char filename [PATH_MAX];
    char drive [_MAX_DRIVE];
    char dir [_MAX_DIR];
    char ext [_MAX_EXT];

    char filename2 [PATH_MAX];
    char ext2 [_MAX_EXT];

    fnsplit (fname, drive, dir, filename, ext);

    if (*drive || *dir == '/' ||
	*dir == '\\' ||
	(*dir == '.' && (*(dir + 1) == '/'
			 || *(dir + 1) == '\\'
        )))
    {
	strcpy (filename, fname);
        if (ext[1] == 0 || ext[0] == 0)
	    strcat (filename, ".s96");
    }
    else
    {
	strcpy (filename, S9xGetSnapshotDirectory ());
        if (strcmp(filename, "") == 0)
        {
            fnsplit (Memory.ROMFilename, drive, dir, filename2, ext2);
            fnmerge (filename, drive, dir, fname, "");
        }
        else
        {
            strcat (filename, SLASH_STR);
            strcat (filename, fname);
            if (ext[1] == 0 || ext[0] == 0)
                strcat (filename, ".s96");
        }
    }
    
#ifdef ZLIB
    if (read_only)
    {
	if ((*file = OPEN_STREAM (filename, "rb")))
	    return (TRUE);
    }
    else
    {
	if ((*file = OPEN_STREAM (filename, "wb")))
	    return (TRUE);
    }
#else
    char command [PATH_MAX];
    
    if (read_only)
    {
	sprintf (command, "gzip -d <\"%s\"", filename);
	if (*file = popen (command, "r"))
	    return (TRUE);
    }
    else
    {
	sprintf (command, "gzip --best >\"%s\"", filename);
	if (*file = popen (command, "wb"))
	    return (TRUE);
    }
#endif
    return (FALSE);
}

void S9xCloseSnapshotFile (STREAM file)
{
#ifdef ZLIB
    CLOSE_STREAM (file);
#else
    pclose (file);
#endif
}

bool8 S9xInitUpdate ()
{
    return (TRUE);
}

bool8 S9xDeinitUpdate (int Width, int Height, bool8 sixteen_bit)
{
    S9xPutImage (Width, Height);
#if 0
    extern int plot2_count, plot4_count, plot8_count, plotobj_count;

    printf ("\r%5d, %5d, %5d, %5d", plot2_count, plot4_count, plot8_count, plotobj_count);
    plot2_count = 0;
    plot4_count = 0;
    plot8_count = 0;
    plotobj_count = 0;
    fflush (stdout);
#endif
    return (TRUE);
}

static unsigned long now ()
{
    static unsigned long seconds_base = 0;
    struct timeval tp;
    gettimeofday (&tp, NULL);
    if (!seconds_base)
	seconds_base = tp.tv_sec;

    return ((tp.tv_sec - seconds_base) * 1000 + tp.tv_usec / 1000);
}

void OutputFrameRate ()
{
    static int FrameCount = 0;
    static unsigned long then = now ();

    if (++FrameCount % 60 == 0)
    {
	unsigned long here = now ();
//	printf ("\rFrame rate: %.2lfms", (double) (here - then) / 60);
//	fflush (stdout);
	then = here;
    }
}

void S9xToggleSoundChannel (int c)
{
    static int channel_enable = 255;
    if (c == 8)
	channel_enable = 255;
    else
	channel_enable ^= 1 << c;
    S9xSetSoundControl (channel_enable);
}

static long log2 (long num)
{
    long n = 0;

    while (num >>= 1)
	n++;

    return (n);
}

static long power (int num, int pow)
{
    long val = num;
    int i;
    
    if (pow == 0)
	return (1);

    for (i = 1; i < pow; i++)
	val *= num;

    return (val);
}

uint32 S9xReadJoypad (int which1)
{
    if (which1 < NumControllers)
	return (0x80000000 | joypads [which1]);
    return (0);
}

extern char S9X_IniFilename[];
char S9X_IniFilename[256];

void ReadIniDefaults(char * exefilename)
{
   char sdrv[3], sdir[200], sfile[200], sext[200], inifn[256];
   char S[200] = {0}, S1[200] = {0};
   int i, q;

   fnsplit (exefilename, sdrv, sdir, sfile, sext);
   fnmerge (inifn, sdrv, sdir, sfile, ".ini");
   strcpy (S9X_IniFilename, inifn);
   itoa(Settings.SkipFrames - 1, S1, 10);
   GetPrivateProfileString ("SNES9X", "FrameRate", S1, S, 200, inifn);
   if (strnicmp(S, "Auto", 4) == 0)
       Settings.SkipFrames = AUTO_FRAMERATE;
   else
       Settings.SkipFrames = atoi(S) + 1;
   Settings.BGLayering = GetPrivateProfileInt ("SNES9X", "Layering", Settings.BGLayering, inifn);
   //Settings.OldStyleJoypadsEnable = GetPrivateProfileInt("SNES9X", "OldStyleJoypadsEnable",
   //                                 Settings.OldStyleJoypadsEnable, inifn);
   Settings.Shutdown = GetPrivateProfileInt("SNES9X", "IdleDetect", Settings.Shutdown, inifn);
   Settings.APUEnabled = GetPrivateProfileInt("SNES9X", "SPCEmulation", Settings.APUEnabled, inifn);
   Settings.SoundSkipMethod = GetPrivateProfileInt("SNES9X", "SoundSkipper", Settings.SoundSkipMethod, inifn);

   GetPrivateProfileString("DOS Version", "JoystickSettings", "0", S, 200, inifn);
   i = atoi(S);
   //if (strnicmp("SnesPro", S, 7) == 0)
   //   SnesPro = TRUE;
   //else
   if (i==0)
   {
      Settings.JoystickEnabled = FALSE;
   }
   else if (i==2)
   {
      Settings.JoystickEnabled = TRUE;
      joystick_has_four_buttons = FALSE;
      joystick_has_six_buttons = FALSE;
   }
   else if (i==4)
   {
      Settings.JoystickEnabled = TRUE;
      joystick_has_four_buttons = TRUE;
      joystick_has_six_buttons = FALSE;
   }
   else if (i==6)
   {
      Settings.JoystickEnabled = TRUE;
      joystick_has_four_buttons = FALSE;
      joystick_has_six_buttons = TRUE;
   }
   Settings.SoundPlaybackRate = GetPrivateProfileInt("DOS Version", "SoundPlaybackRate", Settings.SoundPlaybackRate, inifn);
   Settings.Stereo = GetPrivateProfileInt("DOS Version", "Stereo", Settings.Stereo, inifn);
   //Settings.FastRedraw = GetPrivateProfileInt("SNES9X", "TileBasedRedraw", Settings.FastRedraw, inifn);
   extern int vid_mode, vid_xres, vid_yres, vid_adaptor;
   itoa(vid_mode, S1, 10);
   GetPrivateProfileString("DOS Version", "VideoMode", S1, S, 200, inifn);
   if (strnicmp(S, "custom", 6)!=0)
       vid_mode=atoi(S);
   else
   {
       vid_mode = 255;
       vid_adaptor = GetPrivateProfileInt("DOS Version", "VideoAdaptor", 2, inifn);
       vid_xres = GetPrivateProfileInt("DOS Version", "VideoXRes", 320, inifn);
       vid_yres = GetPrivateProfileInt("DOS Version", "VideoYRes", 240, inifn);
   }
   Windows_Friendly = GetPrivateProfileInt("DOS Version", "WindowsFriendly", Windows_Friendly, inifn);
   i = GetPrivateProfileInt("DOS Version", "AllowGUI", 1, inifn);

   if (i==0)
      UseGUI=FALSE;
   else if (i==1)
   {
      UseGUI = TRUE;
      //SaveResolution = TRUE;
      UseSFX = TRUE;
   }
   else if (i==2)
   {
      UseGUI = TRUE;
      //SaveResolution = TRUE;
      UseSFX = FALSE;
   }
   else if (i==3)
   {
      UseGUI = TRUE;
      //SaveResolution = FALSE;
      UseSFX = TRUE;
   }
   else if (i==4)
   {
      UseGUI = TRUE;
      //SaveResolution = FALSE;
      UseSFX = FALSE;
   }
   else
   {
      UseGUI = TRUE;
      //SaveResolution = FALSE;
      UseSFX = FALSE;
   }
/*
   GetPrivateProfileString("DOS Version", "FPUStrings", "Auto", S, 200, inifn);
   if (strnicmp(S, "yes", 3)==0)
   {
       ForceFPU = TRUE;
       ForceFast = TRUE;
   }
   else if (strnicmp(S, "no", 2)==0)
   {
       ForceFPU = TRUE;
       ForceFast = FALSE;
   }
   else
       ForceFPU = FALSE;
*/
   GetPrivateProfileString("DOS Version", "SnapFormat", "PCX", S, 200, inifn);
   if (strnicmp(S, "PCX", 3)==0)
      strcpy(FileFormat, ".PCX");
   else if (strnicmp(S, "TGA", 3)==0)
      strcpy(FileFormat, ".TGA");
   else
      strcpy(FileFormat, ".PCX");
   GetPrivateProfileString("DOS Paths", "ScreenShotPath", "", ScreenShotPath, 200, inifn);
   GetPrivateProfileString("DOS Paths", "SnapshotPath", ".\\snesnaps", SnapshotPath, 200, inifn);
   GetPrivateProfileString("DOS Paths", "RomPath", ".\\roms", RomPath, 200, inifn);

   GradientClr1 = GetPrivateProfileInt("DOS Version", "GUIGradient1", GradientClr1, inifn);
   GradientClr2 = GetPrivateProfileInt("DOS Version", "GUIGradient2", GradientClr2, inifn);
}

void ReadGameDefaults();
void ReadGameDefaults()
{
   int i;

   //if (!(noiniinter & 1))
   //   Settings.Interleaved = GetPrivateProfileInt(Memory.ROMName, "De-Interleave", Settings.Interleaved, "SNESROMS.INI");
   //Settings.FastRedraw = GetPrivateProfileInt(Memory.ROMName, "TileBasedRedraw", Settings.FastRedraw, "SNESROMS.INI");
   Settings.SoundSkipMethod = GetPrivateProfileInt(Memory.ROMName, "SoundSkipper", Settings.SoundSkipMethod, "SNESROMS.INI");
   Settings.APUEnabled = GetPrivateProfileInt(Memory.ROMName, "SPCEmulation", Settings.APUEnabled, "SNESROMS.INI");
   Settings.Shutdown = GetPrivateProfileInt(Memory.ROMName, "IdleDetect", Settings.Shutdown, "SNESROMS.INI");
   i = GetPrivateProfileInt(Memory.ROMName, "H-Cycles", -1, "SNESROMS.INI");
   if (i != -1)
      Settings.H_Max = (SNES_CYCLES_PER_SCANLINE * i) / 100;
   Settings.DisableGraphicWindows = GetPrivateProfileInt(Memory.ROMName, "NoWindowing", Settings.DisableGraphicWindows, "SNESROMS.INI");
   //Settings.OldStyleJoypadsEnable = GetPrivateProfileInt(Memory.ROMName, "OldStyleJoypadsEnable", Settings.OldStyleJoypadsEnable, "SNESROMS.INI");
   Settings.ForcePAL = GetPrivateProfileInt(Memory.ROMName, "ForcePAL", Settings.ForcePAL, "SNESROMS.INI");
   Settings.ForceNTSC = GetPrivateProfileInt(Memory.ROMName, "ForceNTSC", Settings.ForceNTSC, "SNESROMS.INI");
   Settings.BGLayering = GetPrivateProfileInt(Memory.ROMName, "BGLayering", Settings.BGLayering, "SNESROMS.INI");
/*
   if (!(noiniinter & 2))
   {
       Memory.HiROM = GetPrivateProfileInt(Memory.ROMName, "ForceHiROM", Memory.HiROM, "SNESROMS.INI");
       Memory.LoROM = GetPrivateProfileInt(Memory.ROMName, "ForceLoROM", Memory.LoROM, "SNESROMS.INI");
   }

   noiniinter = 0;*/
}
