/*
 * Snes9x - Portable Super Nintendo Entertainment System (TM) emulator.
 *
 * This file (c) Copyright 1997, 1998 Chad Kitching
 *
 * Permission to use, copy, modify and distribute Snes9x in both binary and
 * source form, for non-commercial purposes, is hereby granted without fee,
 * providing that this license information and copyright notice appear with
 * all copies and any derived work.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event shall the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Snes9x is freeware for PERSONAL USE only. Commercial users should
 * seek permission of the copyright holders first. Commercial use includes
 * charging money for Snes9x or software derived from Snes9x.
 *
 * The copyright holders request that bug fixes and improvements to the code
 * should be forwarded to them so everyone can benefit from the modifications
 * in future versions.
 *
 * Super NES and Super Nintendo Entertainment System are trademarks of
 * Nintendo Co., Limited and its subsidiary companies.
 */

#include <bios.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <dir.h>
#include <limits.h>
#include <ctype.h>
#include <fcntl.h>
#include <keys.h>
#include <go32.h>
#include <conio.h>
#include <allegro.h>

#undef TRUE
#undef FALSE

#include "snes9x.h"
#include "s9xgrph.h"
#include "memmap.h"
#include "cpuexec.h"
#include "ppu.h"
#include "apu.h"
#include "dma.h"
#include "cpuid.h"
#include "cheats.h"
#include "cheat.h"
#include "display.h"
#include "snapshot.h"
#include "soundux.h"
#include "gui.h"
#include "inifuncs.h"
#include "msdos.h"

int DoOtherMenu();

bool8 neverrun = FALSE;
bool8 InGUI = FALSE;
bool8 canchangeapu = FALSE;
int GradientClr1=0x000080, GradientClr2=0x1088D0, olddepth;
bool8 SaveResolution = FALSE, UseSFX = TRUE;

static int WantQuit = 0;

int mainmnu();

PALETTE s9gui_backuppal = s9gui_logopal;

void DoJoyconf(int number);
void _stub_bank_switch();
void _stub_bank_switch_end();
void ChangeStatus(int i);

union RGB2
{
   char R;
   char G;
   char B;
   char filler;
};

BITMAP *our_mouse = NULL;

#define FOCSEL D_WANTFOCUS|D_SELECTED
#define FOCUS  D_WANTFOCUS
#define EXFOC  D_WANTFOCUS|D_EXIT

char apuskipstr[30]="0\0";
char frskipstr[30]="5\0";

int GetIntFromString(char * Str)
{
    int i = 0, j = 0, k = 0;
    bool8 hexmode = FALSE, octmode = FALSE, decmode = TRUE;

    while (Str[j] != 0)
    {
        if (Str[j] == '$')
        {
            hexmode = TRUE;
            octmode = FALSE;
            decmode = FALSE;
        }
        else if (Str[j] == 'o')
        {
            hexmode = FALSE;
            octmode = TRUE;
            decmode = FALSE;
        }
        else if (Str[j] == '#')
        {
            hexmode = FALSE;
            octmode = FALSE;
            decmode = TRUE;
        }
        if (hexmode)
        {
            if ((Str[j]>='0' && Str[j]<='9') || (Str[j]>='a' && Str[j]<='f') ||
                (Str[j]>='A' && Str[j]<='F'))
            {
                i<<=4;
                i&=0xFFFFFFF0;
                if (Str[j]>='a')
                   k = Str[j] - 'a' + 10;
                else if (Str[j]>='A')
                   k = Str[j] - 'A' + 10;
                else if (Str[j]>='0')
                   k = Str[j] - '0';
                i|=k;
            }
        }
        if (octmode)
        {
            if (Str[j]>='0' && Str[j]<='7')
            {
                i<<=3;
                k = Str[j] - '0';
                i|=k;
            }
        }
        if (decmode)
        {
            if (Str[j]>='0' && Str[j]<='9')
            {
                i*=10;
                k = Str[j] - '0';
                i+=k;
            }
        }
        j++;
    }
    return i;
}

void unrotate_screen(BITMAP * lastscreen)
{
   static BITMAP * tempbuffer, * offscreen;
   int startx, deltax, j, starty, deltay;
   int q, i;
   register int x, k, y;

   offscreen = create_bitmap(gfx_driver->w, gfx_driver->h);
   tempbuffer = create_bitmap(gfx_driver->w, gfx_driver->h);
   show_mouse(NULL);
   blit(screen, offscreen, 0, 0, 0, 0, gfx_driver->w, gfx_driver->h);
   for (i=0; i<100; i+=(SaveResolution ? 4:8))
   {
        clear(tempbuffer);
        starty = ((gfx_driver->h>>1)*i)/100;
        deltay = gfx_driver->h - (starty<<1);
        for (y=q=0; q < deltay; q++)
        {
             y = (q * gfx_driver->h) / deltay;
             j = 100 - ((gfx_driver->h/2 - y) * i) / ((gfx_driver->h<<2)/5);

             deltax = (gfx_driver->w * 100 / j);
             startx = (gfx_driver->w - deltax) >> 1;

             if (y > gfx_driver->h/2)
             {
                for (x=gfx_driver->w; x>0; x--)
                {
                    k = startx + ((deltax * x)/gfx_driver->w);
                    tempbuffer->line[starty + q][x]=offscreen->line[y][k];
                }
            }
            else
            {
                j = (gfx_driver->w*(0-startx))/deltax;
                for (x=(gfx_driver->w*(gfx_driver->w-startx))/deltax; x>j; x--)
                {
                    k = startx + ((deltax * x)/gfx_driver->w);
                    tempbuffer->line[starty + q][x]=offscreen->line[y][k];
                }
            }
        }
        blit(tempbuffer, screen, 0, 0, 0, 0, gfx_driver->w, gfx_driver->h);
        rest(5);
   }
   for (i=100; i>0; i-=(SaveResolution ? 4:8))
   {
        clear(tempbuffer);
        starty = ((gfx_driver->h>>1)*i)/100;
        deltay = gfx_driver->h - (starty<<1);
        for (y=q=0; q < deltay; q++)
        {
             y = (q * gfx_driver->h) / deltay;
             j = 100 - ((y - gfx_driver->h/2) * i) / ((gfx_driver->h<<2)/5);

             deltax = (gfx_driver->w * 100 / j);
             startx = (gfx_driver->w - deltax) >> 1;

             if (y < gfx_driver->h/2)
             {
                for (x=gfx_driver->w; x>0; x--)
                {
                    k = startx + ((deltax * x)/gfx_driver->w);
                    tempbuffer->line[starty + q][x]=lastscreen->line[y][k];
                }
            }
            else
            {
                j = (gfx_driver->w*(0-startx))/deltax;
                for (x=(gfx_driver->w*(gfx_driver->w-startx))/deltax; x>j; x--)
                {
                    k = startx + ((deltax * x)/gfx_driver->w);
                    tempbuffer->line[starty + q][x]=lastscreen->line[y][k];
                }
            }
        }
        blit(tempbuffer, screen, 0, 0, 0, 0, gfx_driver->w, gfx_driver->h);
        rest(5);
   }
   blit(lastscreen, screen, 0, 0, 0, 0, gfx_driver->w, gfx_driver->h);
   destroy_bitmap(tempbuffer);
   destroy_bitmap(offscreen);
}

void rotate_screen(BITMAP * offscreen, DIALOG * dialog_disp)
{
   BITMAP * tempbuffer;
   BITMAP * backscreen;
   BITMAP * backupscreen;
   int x, y, startx, deltax, j, k, starty, deltay;
   int q, i;

   backscreen = create_bitmap(gfx_driver->w, gfx_driver->h);
   tempbuffer = create_bitmap(gfx_driver->w, gfx_driver->h);

   show_mouse(NULL);
   blit(screen, offscreen, 0, 0, 0, 0, gfx_driver->w, gfx_driver->h);
   for (i=0; i<100; i+=(SaveResolution ? 4:8))
   {
        clear(tempbuffer);
        starty = ((gfx_driver->h>>1)*i)/100;
        deltay = gfx_driver->h - (starty<<1);
        for (y=q=0; q < deltay; q++)
        {
             y = (q * gfx_driver->h) / deltay;
             j = 100 - ((gfx_driver->h/2 - y) * i) / ((gfx_driver->h<<2)/5);

             deltax = (gfx_driver->w * 100 / j);
             startx = (gfx_driver->w - deltax) >> 1;

             if (y > gfx_driver->h/2)
             {
                for (x=gfx_driver->w; x>0; x--)
                {
                    k = startx + ((deltax * x)/gfx_driver->w);
                    tempbuffer->line[starty + q][x]=offscreen->line[y][k];
                }
            }
            else
            {
                j = (gfx_driver->w*(0-startx))/deltax;
                for (x=(gfx_driver->w*(gfx_driver->w-startx))/deltax; x>j; x--)
                {
                    k = startx + ((deltax * x)/gfx_driver->w);
                    tempbuffer->line[starty + q][x]=offscreen->line[y][k];
                }
            }
        }
        blit(tempbuffer, screen, 0, 0, 0, 0, gfx_driver->w, gfx_driver->h);
        rest(5);
   }
   backupscreen = screen;
   screen = backscreen;
   DIALOG * tempdlg = dialog_disp;
   set_dialog_color(tempdlg, 160, 168);
   centre_dialog(tempdlg);
   stretch_blit(&s9gui_logo, screen, 0, 0, s9gui_logo.w, s9gui_logo.h, 0, 0, gfx_driver->w, gfx_driver->h);
   while (tempdlg->proc)
   {
      SEND_MESSAGE(tempdlg, MSG_START, 0);
      SEND_MESSAGE(tempdlg, MSG_DRAW, 0);
      SEND_MESSAGE(tempdlg, MSG_END, 0);
      tempdlg++;
   }
   screen = backupscreen;
   for (i=100; i>0; i-=(SaveResolution ? 4:8))
   {
        clear(tempbuffer);
        starty = ((gfx_driver->h>>1)*i)/100;
        deltay = gfx_driver->h - (starty<<1);
        for (y=q=0; q < deltay; q++)
        {
             y = (q * gfx_driver->h) / deltay;
             j = 100 - ((y - gfx_driver->h/2) * i) / ((gfx_driver->h<<2)/5);

             deltax = (gfx_driver->w * 100 / j);
             startx = (gfx_driver->w - deltax) >> 1;

             if (y < gfx_driver->h/2)
             {
                for (x=gfx_driver->w; x>0; x--)
                {
                    k = startx + ((deltax * x)/gfx_driver->w);
                    tempbuffer->line[starty + q][x]=backscreen->line[y][k];
                }
            }
            else
            {
                j = (gfx_driver->w*(0-startx))/deltax;
                for (x=(gfx_driver->w*(gfx_driver->w-startx))/deltax; x>j; x--)
                {
                    k = startx + ((deltax * x)/gfx_driver->w);
                    tempbuffer->line[starty + q][x]=backscreen->line[y][k];
                }
            }
        }
        blit(tempbuffer, screen, 0, 0, 0, 0, gfx_driver->w, gfx_driver->h);
        rest(5);
   }
   blit(backscreen, screen, 0, 0, 0, 0, gfx_driver->w, gfx_driver->h);
   destroy_bitmap(backscreen);
   destroy_bitmap(tempbuffer);
}

static char Freqlist[8][32] = {"Off", "8192 Hz", "11025 Hz", "16500 Hz", "22050 Hz", "29300 Hz", "36600 Hz", "44000 Hz"};

char * FreqLister(int index, int *list_size)
{
    *list_size = 8;
    if (index<0) return NULL;
    if (index>8) return NULL;
    return (char*)&(Freqlist[index]);
}

static DIALOG options_dialog[] =
{
   /* (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)  (d1)  (d2)  (dp) */
   { d_3d_box_proc,     0,    0,    242,  198,  0,    0,    0,    D_EXIT,  0,    0,    "Options" },
   { d_group_proc,      8,    23,   122,  61,   0,    0,    0,    0,       0,    0,    "Advanced" },
   { d_3dcheck_proc,    14,   29,   91,   9,    0,    0,    'D',  FOCSEL,  0,    0,    "H-&DMA Enabled" },
   { d_3dcheck_proc,    14,   41,   91,   9,    0,    0,    'I',  D_HIDDEN,0,    0,    "&IRQ Enabled" },
   { d_text_proc,       14,   42,   91,   9,    0,    0,    'O',  FOCUS,   0,    0,    "" },
   { d_3dradio_proc,    14,   55,   91,   9,    0,    0,    'E',  FOCSEL,  1,    0,    "APU &Emulation" },
   { d_3dradio_proc,    14,   68,   91,   9,    0,    0,    'K',  FOCUS,   1,    0,    "APU S&kipping:" },
   { d_3dedit_proc,     98,   66,   24,   13,   0,    0,    0,    FOCUS,   1,    0,     apuskipstr },

   { d_group_proc,      8,    91,   122,  76,   0,    0,    0,    0,       0,    0,    "Video Settings" },
   { d_3dradio_proc,    14,   97,   91,   9,    0,    0,    'S',  FOCUS,   2,    0,    "Frame &Skip:" },
   { d_3dedit_proc,     98,   95,   24,   13,   0,    0,    0,    FOCUS,   3,    0,     frskipstr },
   { d_3dradio_proc,    14,   110,  91,   9,    0,    0,    'A',  FOCUS,   2,    0,    "&Auto Adjust" },
   { d_speeddial,       14,   123,  108,  12,   0,    0,    0,    FOCUS, 100, 0xC80019, NULL },
   { d_text_proc,       14,   139,  91,   9,    0,    0,    'T',  FOCUS,   0,    0,    "" },
   { d_3dcheck_proc,    14,   152,  91,   9,    0,    0,    'C',  FOCSEL,  0,    0,    "&Clip Windows" },

   { d_group_proc,      134,  23,   100,  60,   0,    0,    0,    0,       0,    0,    "Emulation options" },
   { d_text_proc,       140,  30,   1,    1,    0,    0,    'H',  0,       0,    0,    "&H-cycles:" },
   { d_speeddial,       140,  42,   86,   12,   0,    0,    0,    FOCUS,  90, 0xC80032, NULL },
   { d_3dcheck_proc,    140,  56,   91,   9,    0,    0,    'P',  FOCSEL,  0,    0,    "A&PU Speed-ups" },
   { d_3dcheck_proc,    140,  68,   91,   9,    0,    0,    'E',  D_DISABLED, 0, 0,    "CPU Sp&eed-ups" },

   { d_3dbutton_proc,   140,  85,   80,   16,   0,    0,    '1',  D_DISABLED, 0, 0,    "Config Joy&1" },
   { d_3dbutton_proc,   140,  107,  80,   16,   0,    0,    '2',  D_DISABLED, 0, 0,    "Config Joy&2" },
   { d_3dbutton_proc,   140,  129,  80,   16,   0,    0,    'M',  EXFOC,   0,    0,    "Video &Mode" },
   { d_3dbutton_proc,   140,  150,  80,   16,   0,    0,    'U',  D_DISABLED, 0, 0,    "So&und Options" },
   { d_3dcheck_proc,    180,  170,  50,   9,    0,    0,    'V',  FOCSEL,  0,    0,    "Sa&ve" },
   { d_3dbutton_proc,   8,    175,  80,   16,   0,    0,    13,   EXFOC,   0,    0,    "Ok" },
   { d_3dbutton_proc,   94,   175,  80,   16,   0,    0,    27,   EXFOC,   0,    0,    "Cancel" },
   { NULL }
};

#define opt_clip_win       14
#define opt_h_cycdial      17
#define opt_speedhack      18
#define opt_Joy1_button    20
#define opt_Joy2_button    21
#define opt_VideoModes     22
#define opt_SndOpt         23
#define opt_save_check     24
//#define opt_local_check    25
#define opt_Ok_button      25
#define opt_Cancel_button  26

typedef struct
{
    int width;
    int height;
    int mode;
} Mode;

static Mode modes [] = {
    {320, 240, GFX_MODEX},
    {320, 200, GFX_VGA},
    {256, 256, GFX_VGA},
    {256, 240, GFX_VGA},
    {640, 480, GFX_AUTODETECT},
    {640, 480, GFX_VESA1},
    {640, 480, GFX_VESA2L},
    {640, 400, GFX_XTENDED},
    {800, 600, GFX_AUTODETECT}
};

int DoOptions()
{
   static int SaveAll=-1;
   char str[50];

   centre_dialog(options_dialog);
   set_dialog_color(options_dialog, 160, 168);
   options_dialog[2].flags = Settings.DisableHDMA ? FOCUS : FOCSEL;
   options_dialog[5].flags = Settings.APUEnabled ? FOCSEL : FOCUS;
   options_dialog[6].flags = Settings.APUEnabled ? FOCUS : FOCSEL;
   options_dialog[9].flags = Settings.SkipFrames == AUTO_FRAMERATE ? FOCUS : FOCSEL;
   if (Settings.SkipFrames != AUTO_FRAMERATE)
      sprintf( frskipstr, "%U", Settings.SkipFrames - 1);

   options_dialog[11].flags = Settings.SkipFrames == AUTO_FRAMERATE ? FOCSEL : FOCUS;
   options_dialog[opt_clip_win].flags = Settings.DisableGraphicWindows ? FOCUS : FOCSEL;
   options_dialog[opt_speedhack].flags = Settings.Shutdown ? FOCSEL : FOCUS;
   if (Settings.PAL)
       options_dialog[12].d1 = ((Settings.FrameTime * 100) + 5) / Settings.FrameTimePAL;
   else
       options_dialog[12].d1 = ((Settings.FrameTime * 100) + 5) / Settings.FrameTimeNTSC;
   options_dialog[opt_h_cycdial].d1 = (((Settings.H_Max * 1000) / SNES_CYCLES_PER_SCANLINE) + 5) / 10;
   if (!Settings.APUEnabled && !neverrun && !canchangeapu)
       options_dialog[5].flags |= D_DISABLED;
   else
       canchangeapu = TRUE;
   sprintf( apuskipstr, "%i", Settings.SoundSkipMethod);
   SaveAll = GetPrivateProfileInt("DOS Version", "SaveSettings", SaveAll, S9X_IniFilename);

   if (SaveAll == 1)
      options_dialog[opt_save_check].flags = D_SELECTED;
   else if (SaveAll == 0)
      options_dialog[opt_save_check].flags = 0;
   int ret=-1;
   while ((ret != opt_Ok_button) && (ret != opt_Cancel_button) && (ret != 0))
   {
      show_mouse(NULL); 
      ret = popup_dialog(options_dialog, 1);
      show_mouse(screen);
      if (ret == opt_Joy1_button || ret == opt_Joy2_button)
      {
         DoJoyconf(ret - opt_Joy1_button + 1);
      }
      if (ret == opt_VideoModes)
      {
         if (vid_mode != 255)
         {
             vid_adaptor = modes[vid_mode].mode;
             vid_xres = modes[vid_mode].width;
             vid_yres = modes[vid_mode].height;
         }
         vid_mode = 255;

         gfx_mode_3dselect(&vid_adaptor, &vid_xres, &vid_yres, &highresmode, &highcolourmode);

         S9xGraphicsDeinit();
         S9xDeinitDisplay();
         S9xInitDisplay(0, NULL);
         S9xGraphicsInit();
         IPPU.RenderThisFrame = FALSE;
         olddepth = _color_depth;
         set_color_depth(8);

         if (vid_xres == 256 && vid_yres == 256 && (vid_adaptor == GFX_VGA || vid_adaptor == 0))
             vid_mode = 2;
         //if (vid_xres == 256 && vid_yres == 240 && (vid_adaptor == GFX_VGA || vid_adaptor == 0))
         //    vid_mode = 3;
      }
      if (ret == opt_Ok_button)
      {
          if (options_dialog[2].flags & D_SELECTED)
             Settings.DisableHDMA = FALSE;
          else
             Settings.DisableHDMA = TRUE;
          if (options_dialog[5].flags & D_SELECTED)
             Settings.APUEnabled = TRUE;
          else
             Settings.APUEnabled = FALSE;
          Settings.SoundSkipMethod = atoi(apuskipstr);
          if (options_dialog[11].flags & D_SELECTED)
             Settings.SkipFrames = AUTO_FRAMERATE;
          else
             Settings.SkipFrames = atoi(frskipstr) + 1;
          if (Settings.PAL)
             Settings.FrameTime = (options_dialog[12].d1 * Settings.FrameTimePAL) / 100;
          else
             Settings.FrameTime = (options_dialog[12].d1 * Settings.FrameTimeNTSC) / 100;
          if (install_int_ex (TimerInterrupt, (long)(((double)Settings.FrameTime * 1193181)/1000000)) < 0)
              Settings.SkipFrames = 4;
          Settings.DisableGraphicWindows = !(options_dialog[opt_clip_win].flags & D_SELECTED);
          Settings.H_Max = (SNES_CYCLES_PER_SCANLINE * options_dialog[opt_h_cycdial].d1) / 100;
          Settings.Shutdown = (options_dialog[opt_speedhack].flags & D_SELECTED) ? TRUE : FALSE;
          SaveAll = (options_dialog[opt_save_check].flags & D_SELECTED) ? 1:0;
          /*
          sprintf(str, "%i", SaveAll);
          WritePrivateProfileString("DOS Version", "SaveSettings", str, S9X_IniFilename);
          if (options_dialog[opt_save_check].flags & D_SELECTED)
          {
              sprintf(str, "%i", Settings.SoundSkipMethod);
              WritePrivateProfileString("SNES9X", "SoundSkipper", str, S9X_IniFilename);
              sprintf(str, "%i", Settings.APUEnabled);
              WritePrivateProfileString("SNES9X", "SPCEmulation", str, S9X_IniFilename);
              sprintf(str, "%i", Settings.Shutdown);
              WritePrivateProfileString("SNES9X", "IdleDetect", str, S9X_IniFilename);
              sprintf(str, "%U", Settings.SkipFrames - 1);
              if (Settings.SkipFrames == AUTO_FRAMERATE)
                 WritePrivateProfileString("SNES9X", "FrameRate", "Auto", S9X_IniFilename);
              else
                 WritePrivateProfileString("SNES9X", "FrameRate", str, S9X_IniFilename);
              sprintf(str, "%i", vid_mode);
              WritePrivateProfileString("DOS Version", "VideoMode", vid_mode != 255 ? str : "Custom", S9X_IniFilename);
              sprintf(str, "%i", Settings.SupportHiRes);
              WritePrivateProfileString("DOS VERSION", "HighResSupport", str, S9X_IniFilename);
              sprintf(str, "%i", Settings.SixteenBit);
              WritePrivateProfileString("DOS VERSION", "HighColour", str, S9X_IniFilename);
              sprintf(str, "%i", vid_adaptor);
              WritePrivateProfileString("DOS Version", "VideoAdaptor", str, S9X_IniFilename);
              sprintf(str, "%i", vid_xres);
              WritePrivateProfileString("DOS Version", "VideoXRes", str, S9X_IniFilename);
              sprintf(str, "%i", vid_yres);
              WritePrivateProfileString("DOS Version", "VideoYRes", str, S9X_IniFilename);
              sprintf(str, "%i", options_dialog[opt_h_cycdial].d1);
              WritePrivateProfileString("SNES9X", "H-Cycles", str, S9X_IniFilename);
              sprintf(str, "%i", Settings.DisableGraphicWindows);
              WritePrivateProfileString("SNES9X", "NoWindowing", str, S9X_IniFilename);
              sprintf(str, "%i", Settings.BGLayering);
              WritePrivateProfileString("SNES9X", "Layering", str, S9X_IniFilename);
          }
          */
      }
   }
   return 0;
}

int jc_dlist_proc(int msg, DIALOG * d, int c);
int jc_config_proc(int msg, DIALOG * d, int c);
int jc_calibr_proc(int msg, DIALOG * d, int c);
int jc_initia_proc(int msg, DIALOG * d, int c);
#define JoyDev 10
static char Driverlist[JoyDev][64] = {
      "Keyboard",
      "Joystick (2 button)",
      "Joystick (4 button)",
      "Joystick (6 button)",
      "Gravis GRiP",
      "Sidewinder",
      "SnesPRO! (LPT1)",
      "SnesPRO! (LPT2)",
      "SnesPRO! (LPT3)",
      "Off"};

char * DrvLister(int index, int *list_size)
{
    if (index < 0)
    {
       if (list_size)
          *list_size = JoyDev;
       return NULL;
    }

    if (index>JoyDev) return NULL;
    return Driverlist[index];
}


char ConfigTitleStr[120] = "Config";
char statustxt1[120] = "Ready.";

static DIALOG Controller_dialog[] =
{
   /* (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)  (d1)  (d2)  (dp) */
   { d_3d_box_proc,     0,    0,    320,  198,  0,    0,    0,    D_EXIT,  0,    0,    ConfigTitleStr },
   { d_bitmap_proc,     8,    30,   127,  62,   0,    0,    0,    0,       0,    0,    &s9gui_snespad1 },
   { jc_dlist_proc,     8,    100,  127,  45,   0,    0,    0,    FOCUS,   0,    0,    DrvLister},
   { d_3dbutton_proc,   10,   155,   80,  16,   0,    0,    13,   EXFOC,   0,    0,    "Ok" },
   { d_3dbutton_proc,   100,  155,   80,  16,   0,    0,    27,   EXFOC,   0,    0,    "Cancel" },
   { jc_config_proc,    140,  30,    80,  16,   0,    0,    'C',  EXFOC,   0,    0,    "&Configure" },
   { jc_calibr_proc,    140,  50,    80,  16,   0,    0,    'A',  EXFOC,   0,    0,    "C&alibrate" },
   { jc_initia_proc,    140,  70,    80,  16,   0,    0,    'I',  EXFOC,   0,    0,    "Re&initialise" },
   { d_bevel_proc,      4,    177,  311,  15,   0,    0,    0,    0,       5,    0,    NULL},
   { d_text_proc,       8,    181,    1,   1,   0,    0,    0,    0,       0,    0,    statustxt1},
   { NULL }
};

#define jc_conf 5
#define jc_cali 6
#define jc_init 7
#define jc_status 9

#define ConfigButton(btn, dispfnc)                                 \
        sprintf(statustxt1, "Press the new `"btn"' key.      ");   \
        SEND_MESSAGE(Controller_dialog + jc_status, MSG_DRAW, 0);  \
        while (!keypressed()) {                                    \
            poll_joystick();                                       \
            blinker=!blinker;                                      \
            if (blinker)                                           \
                dispfnc;                                           \
            else                                                   \
                blit(TEST2, TEST, 0, 0, 0, 0, TEST2->w, TEST2->h); \
            SEND_MESSAGE(Controller_dialog + 1, MSG_DRAW, 0);      \
            rest(150);                                             \
        }

int jc_config_proc(int msg, DIALOG * d, int c)
{
    int blinker = 0;
        BITMAP * TEST, * TEST2;
    int ret = d_3dbutton_proc(msg, d, c);

    if (ret == D_CLOSE)
    {
        ret = D_O_K;

        TEST2 = (BITMAP*)Controller_dialog[1].dp;
        TEST = create_bitmap(TEST2->w, TEST2->h);
        blit(TEST2, TEST, 0, 0, 0, 0, TEST2->w, TEST2->h);
        Controller_dialog[1].dp = TEST;

        ConfigButton("UP", triangle(TEST, 22, 25, 19, 28, 25, 28, 170));
        readkey();

        ConfigButton("LEFT", triangle(TEST, 13, 34, 11, 36, 13, 39, 170));
        readkey();

        ConfigButton("DOWN", triangle(TEST, 20, 45, 25, 45, 22, 48, 170));
        readkey();

        ConfigButton("RIGHT", triangle(TEST, 32, 34, 34, 36, 32, 39, 170));
        readkey();

        ConfigButton("X", circlefill(TEST, 103, 28, 3, 170));
        readkey();

        ConfigButton("Y", circlefill(TEST, 93, 35, 3, 170));
        readkey();

        ConfigButton("A", circlefill(TEST, 111, 37, 3, 170));
        readkey();

        ConfigButton("B", circlefill(TEST, 102, 44, 3, 170));
        readkey();

        ConfigButton("L", line(TEST, 22, 11, 44, 11, 170));
        readkey();

        ConfigButton("R", line(TEST, 81, 11, 102, 11, 170));
        readkey();

        ConfigButton("SELECT", line(TEST, 55, 34, 49, 40, 170));
        readkey();

        ConfigButton("START", line(TEST, 65, 34, 60, 40, 170));
        readkey();
/*
        sprintf(statustxt1, "Press the new `LEFT' key.      ");
        SEND_MESSAGE(Controller_dialog + jc_status, MSG_DRAW, 0);
        while (!keypressed()) {
            poll_joystick();
            blinker=!blinker;
            if (blinker)
                triangle(TEST, 13, 34, 11, 36, 13, 39, 170);
            else
                blit(TEST2, TEST, 0, 0, 0, 0, TEST2->w, TEST2->h);
            SEND_MESSAGE(Controller_dialog + 1, MSG_DRAW, 0);
            rest(150);
        }
        readkey();

        sprintf(statustxt1, "Press the new `DOWN' key.      ");
        SEND_MESSAGE(Controller_dialog + jc_status, MSG_DRAW, 0);
        while (!keypressed()) {
            poll_joystick();
            blinker=!blinker;
            if (blinker)
                triangle(TEST, 20, 45, 25, 45, 22, 48, 170);
            else
                blit(TEST2, TEST, 0, 0, 0, 0, TEST2->w, TEST2->h);
            SEND_MESSAGE(Controller_dialog + 1, MSG_DRAW, 0);
            rest(150);
        }
        readkey();

        sprintf(statustxt1, "Press the new `RIGHT' key.      ");
        SEND_MESSAGE(Controller_dialog + jc_status, MSG_DRAW, 0);
        while (!keypressed()) {
            poll_joystick();
            blinker=!blinker;
            if (blinker)
                triangle(TEST, 32, 34, 34, 36, 32, 39, 170);
            else
                blit(TEST2, TEST, 0, 0, 0, 0, TEST2->w, TEST2->h);
            SEND_MESSAGE(Controller_dialog + 1, MSG_DRAW, 0);
            rest(150);
        }
        readkey();

        sprintf(statustxt1, "Press the new `X' key.      ");
        SEND_MESSAGE(Controller_dialog + jc_status, MSG_DRAW, 0);
        while (!keypressed()) {
            poll_joystick();
            blinker=!blinker;
            if (blinker)
                circlefill(TEST, 103, 28, 3, 170);
            else
                blit(TEST2, TEST, 0, 0, 0, 0, TEST2->w, TEST2->h);
            SEND_MESSAGE(Controller_dialog + 1, MSG_DRAW, 0);
            rest(150);
        }
        readkey();*/
        show_mouse(NULL);
        destroy_bitmap(TEST);
        Controller_dialog[1].dp = TEST2;
        SEND_MESSAGE(Controller_dialog + 1, MSG_DRAW, 0);
        show_mouse(screen);
    }
    return ret;
}

int jc_calibr_proc(int msg, DIALOG * d, int c)
{
    int ret = d_3dbutton_proc(msg, d, c);
    if (ret == D_CLOSE)
    {
        BITMAP * TEST, * TEST2;
        bool8 blinker = FALSE;

        show_mouse(NULL);
        ret = d_3dbutton_proc(MSG_DRAW, d, c);
        sprintf(statustxt1, "Move the joystick to the upper left, then press button 1");
        SEND_MESSAGE(Controller_dialog + jc_status, MSG_DRAW, 0);
        if (initialise_joystick())
        {
           show_mouse(screen);
           alert_3d("Error", NULL, "Couldn't initialize joystick!", "Ok", NULL, 13, 0);
           return ret;
        }
        TEST2 = (BITMAP*)Controller_dialog[1].dp;
        TEST = create_bitmap(TEST2->w, TEST2->h);
        blit(TEST2, TEST, 0, 0, 0, 0, TEST2->w, TEST2->h);
        Controller_dialog[1].dp = TEST;
        while (joy_b1)
           poll_joystick();
        while (!joy_b1)
        {
            poll_joystick();
            blinker=!blinker;
            if (blinker)
            {
               triangle(TEST, 22, 25, 19, 28, 25, 28, 170);
               triangle(TEST, 13, 34, 11, 36, 13, 39, 170);
            }
            else
               blit(TEST2, TEST, 0, 0, 0, 0, TEST2->w, TEST2->h);
            SEND_MESSAGE(Controller_dialog + 1, MSG_DRAW, 0);
            rest(150);
        }
        calibrate_joystick_tl();
        while (joy_b1)
           poll_joystick();
        sprintf(statustxt1, "Move the joystick to the lower right, then press button 1");
        SEND_MESSAGE(Controller_dialog + jc_status, MSG_DRAW, 0);
        while (!joy_b1)
        {
            poll_joystick();
            blinker=!blinker;
            if (blinker)
            {
               triangle(TEST, 20, 45, 25, 45, 22, 48, 170);
               triangle(TEST, 32, 34, 34, 36, 32, 39, 170);
            }
            else
               blit(TEST2, TEST, 0, 0, 0, 0, TEST2->w, TEST2->h);
            SEND_MESSAGE(Controller_dialog + 1, MSG_DRAW, 0);
            rest(150);
        }
        calibrate_joystick_br();
        show_mouse(screen);
        if (alert_3d("Save calibration", NULL, "Save calibration data to snes9x.joy?", "Yes", "No", 'Y', 'N') == 1)
           save_joystick_data("snes9x.joy");
        show_mouse(NULL);
        destroy_bitmap(TEST);
        Controller_dialog[1].dp = TEST2;
        SEND_MESSAGE(Controller_dialog + 1, MSG_DRAW, 0);
        show_mouse(screen);
    }
    return ret;
}

int jc_initia_proc(int msg, DIALOG * d, int c)
{
    int ret = d_3dbutton_proc(msg, d, c);
    if (ret == D_CLOSE)
    {
       joystick_has_four_buttons = FALSE;
       joystick_has_six_buttons = FALSE;
       Settings.JoystickEnabled = FALSE;
       //SnesPro = FALSE;
       Keyboard_P2 = FALSE;
       switch (Controller_dialog[2].d1)
       {
          case 0: break;  // Keyboard doesn't reinit.  Only test.
          case 1:
                  joystick_has_four_buttons = FALSE;
                  joystick_has_six_buttons = FALSE;
          case 2:
                  if (Controller_dialog[2].d1 == 2)
                  {
                      joystick_has_four_buttons = TRUE;
                      joystick_has_six_buttons = FALSE;
                  }
          case 3:
                  if (Controller_dialog[2].d1 == 3)
                  {
                      joystick_has_four_buttons = FALSE;
                      joystick_has_six_buttons = TRUE;
                  }
                  if (initialise_joystick())
                  {
                      show_mouse(screen);
                      alert_3d("Error", NULL, "Couldn't initialize joystick!", "Ok", NULL, 13, 0);
                      return ret;
                  }
                  break;
          case 4:
                  alert_3d("Error", NULL, "GRiP not supported yet!", "Ok", NULL, 13, 0);
                  break;
          case 5:
                  //par_ioport = 0x378;
                  //SnesPro = TRUE;
                  break;
          case 6: 
                  //par_ioport = 0x278;
                  //SnesPro = TRUE;
                  break;
          case 7:
                  //par_ioport = 0x3BC;
                  //SnesPro = TRUE;
                  break;
          case 8: break;
        }
        d_3dbutton_proc(MSG_DRAW, d, 0);
        ret = D_O_K;
    }
    return ret;
}


int jc_dlist_proc(int msg, DIALOG * d, int c)
{
    int ret;
    static int lastchosen, recurseflag = 0;
    font = &s9gui_smallfont;
    recurseflag++;
    ret = d_3dlist_proc(msg, d, c);
    if ((msg == MSG_CLICK) || (ret == D_USED_CHAR) || ret == D_CLOSE)
    {
        lastchosen = d->d1;
        if (ret == D_CLOSE)
            ret = D_O_K;
        if (d->d1 == 0 || d->d1 >= 5)
        {
           font = &s9gui_xm6x10;
           Controller_dialog[jc_cali].flags = D_DISABLED;
           SEND_MESSAGE(Controller_dialog + jc_cali, MSG_DRAW, 0);
           font = &s9gui_smallfont;
        }
        else
        {
           font = &s9gui_xm6x10;
           Controller_dialog[jc_cali].flags = D_EXIT;
           SEND_MESSAGE(Controller_dialog + jc_cali, MSG_DRAW, 0);
           font = &s9gui_smallfont;
        }
        if (d->d1 >= 5)
        {
           font = &s9gui_xm6x10;
           Controller_dialog[jc_conf].flags = D_DISABLED;
           Controller_dialog[jc_init].flags = D_DISABLED;
           SEND_MESSAGE(Controller_dialog + jc_conf, MSG_DRAW, 0);
           SEND_MESSAGE(Controller_dialog + jc_init, MSG_DRAW, 0);
           font = &s9gui_smallfont;
        }
        else
        {
           font = &s9gui_xm6x10;
           Controller_dialog[jc_conf].flags = D_EXIT;
           Controller_dialog[jc_init].flags = D_EXIT;
           SEND_MESSAGE(Controller_dialog + jc_conf, MSG_DRAW, 0);
           SEND_MESSAGE(Controller_dialog + jc_init, MSG_DRAW, 0);
           font = &s9gui_smallfont;
        }
    }
    recurseflag--;
    if (!recurseflag)
       font = &s9gui_xm6x10;
    return ret;
}

void DoJoyconf(int number)
{
    BITMAP * TEST = create_bitmap(gfx_driver->w, gfx_driver->h);
    static BITMAP * tempbuffer, * offscreen;
    int ret;

    sprintf(ConfigTitleStr, "Configure Joypad #%i", number);
    Controller_dialog[1].dp = number == 1 ? (void*)&s9gui_snespad1 : (void*)&s9gui_snespad2;
    set_dialog_color(Controller_dialog, 160, 168);
    centre_dialog(Controller_dialog);
    if (UseSFX)
        rotate_screen(TEST, (DIALOG*)Controller_dialog);
    else
    {
        show_mouse(NULL);
        blit(screen, TEST, 0, 0, 0, 0, gfx_driver->w, gfx_driver->h);
    }
    ret = do_dialog(Controller_dialog, 1);
    if (UseSFX)
        unrotate_screen(TEST);
    else
    {
        show_mouse(NULL);
        blit(TEST, screen, 0, 0, 0, 0, gfx_driver->w, gfx_driver->h);
    }
    destroy_bitmap(TEST);
    ret = -1;
    install_keyboard();
}

static RGB gclr1, gclr2;
int toptxtpos=0;

int toptxt(char * txtString)
{
    int i=0, q=0, j, l;
    char tststr[4]="";

    tststr[1] = 0;
    q = l = (gfx_driver->w - text_length(&s9gui_helv17b, txtString))>>1;
    for (i=0; i < strlen(txtString); i++)
    {
        tststr[0]=txtString[i];
        gclr1.r = gclr1.g = gclr1.b = 0;
        _set_color((i & 3) + 249, &gclr1);
        textout(screen, &s9gui_helv17b, tststr, q, 0, (i & 3) + 249);
        textout(screen, &s9gui_helv17b, "_", q + text_length(&s9gui_helv17b, tststr), 0, 253);
        for (j = 63; j>0; j-=4)
        {
            gclr1.r = gclr1.b = j;
            gclr1.g = 63 - ((63-j)>>2);
            _set_color((i & 3) + 249, &gclr1);
            rest(5);
            if (keypressed()) return 1;
        }
        _set_color(253, &gclr1);
        textout(screen, &s9gui_helv17b, tststr, q, 0, 253);
        q += text_length(&s9gui_helv17b, tststr);
        textout(screen, &s9gui_helv17b, "_", q, 0, 8);
    }
    rest(30);
    toptxtpos = q;
    return 0;
}

int fadetop()
{
    int i;

    get_color(253, &gclr1);
    for (i=63;i>0;i--)
    {
        if (gclr1.r) gclr1.r--;
        if (gclr1.g) gclr1.g--;
        if (gclr1.b) gclr1.b--;
        _set_color(253, &gclr1);
        rest(10);
        if (keypressed()) return 1;
    }
    stretch_blit(&s9gui_logo, screen, 0, 0, s9gui_logo.w, 
                 (text_height(&s9gui_helv17b)+1)*s9gui_logo.h/gfx_driver->h, 0, 0,
                 gfx_driver->w, text_height(&s9gui_helv17b)+1);
    //rectfill(screen, 0, 0, gfx_driver->w, text_height(&s9gui_helv17b), 0)
    return 0;
}

int bottxt(char * txtString)
{
    int i=0, q=0, j, l;
    char tststr[4]="";

    tststr[1] = 0;
    q = l = (gfx_driver->w - text_length(&s9gui_tms17b, txtString))>>1;
    for (i=0; i < strlen(txtString); i++)
    {
         tststr[0]=txtString[i];
         gclr1.r = gclr1.g = gclr1.b = 0;
         _set_color((i & 3) + 249, &gclr1);
         textout(screen, &s9gui_tms17b, tststr, q, 20, (i & 3) + 249);
         textout(screen, &s9gui_tms17b, "_", q + text_length(&s9gui_tms17b, tststr), 20, 253);
         for (j = 63; j>0; j-=4)
         {
              gclr1.r = gclr1.b = j;
              gclr1.g = 63 - ((63-j)>>2);
              _set_color((i & 3) + 249, &gclr1);
              rest(5);
              if (keypressed()) return 1;
         }
         _set_color(254, &gclr1);
         textout(screen, &s9gui_tms17b, tststr, q, 20, 254);
         q += text_length(&s9gui_tms17b, tststr);
         textout(screen, &s9gui_tms17b, "_", q, 20, 8);
    }
    for (i=0;i<10;i++)
    {
         rest(150);
         textout(screen, &s9gui_tms17b, "_", q, 20, i&1 ? 8 : 254);
         if (keypressed()) return 1;
    }
    for (i=63;i>0;i--)
    {
         if (gclr1.r) gclr1.r--;
         if (gclr1.g) gclr1.g--;
         if (gclr1.b) gclr1.b--;
         _set_color(254, &gclr1);
         rest(10);
         if (keypressed()) return 1;
    }
    stretch_blit(&s9gui_logo, screen, 0, 20*s9gui_logo.h/gfx_driver->h, s9gui_logo.w,
                 (text_height(&s9gui_helv17b)*s9gui_logo.h/gfx_driver->h)+1, 0, 20,
                 gfx_driver->w, text_height(&s9gui_helv17b)+1);
    //rectfill(screen, 0, 20, gfx_driver->w, 20+text_height(&s9gui_tms17b), 0)
    return 0;
}

int docredits()
{
   int i=0;
   char tststr[4]="";
   static RGB gclr1;
   //Clear screen.  Just snes9x logo.
   show_mouse(NULL);
   tststr[0] = tststr[1] = tststr[2] = tststr[3] = 0;
   stretch_blit(&s9gui_logo, screen, 0, 0, s9gui_logo.w, s9gui_logo.h, 0, 0, gfx_driver->w, gfx_driver->h);
   //if (keypressed()) return;
   gclr1.r = gclr1.g = gclr1.b = 0;
   _set_color(253, &gclr1);
   _set_color(254, &gclr1);
   text_mode(-1);

   while (!keypressed())
   {
       if (toptxt(TITLE ":DOS " VERSION )) break;
       if (bottxt("Beyond Limits...")) break;
       if (fadetop()) break;
       if (toptxt("Compiled on")) break;
       if (bottxt(__DATE__ " " __TIME__)) break;
       if (fadetop()) break;
       if (toptxt("Special Thanks")) break;
       if (bottxt("DJ Delorie (DJGPP)")) break;
       if (bottxt("Shawn Hargreaves (Allegro)")) break;
       if (bottxt("Jean-loup Gailly & Mark Adler (ZLib)")) break;
       if (bottxt("Jared Hoag (Snes9X background graphic)")) break;
       if (bottxt("All the beta testers who've helped")) break;
       if (bottxt("And all the users who've supported us.")) break;
       if (fadetop()) break;
       if (toptxt("Press ESC to return")) break;
       for (i=0;i<10;i++)
       {
            rest(150);
            textout(screen, &s9gui_helv17b, "_", toptxtpos, 0, i&1 ? 0 : 253);\
            if (keypressed()) break;
       }
       if (i<10) break;
       fadetop();
   }
   while (keypressed())
      readkey();
   stretch_blit(&s9gui_logo, screen, 0, 0, s9gui_logo.w, s9gui_logo.h, 0, 0, gfx_driver->w, gfx_driver->h);
}

static char CheatList[200][80];
int CheatLast = 0;
int CheatFirst = 0;

void OutputCheat(char* Str)
{
    if (CheatLast < 200)
    {
       strcpy(CheatList[CheatLast], Str);
       CheatLast ++;
    }
}

char * CheatLister(int index, int *list_size)
{
    if (index < 0) {
       if (list_size)
          *list_size = CheatLast;
       return NULL;
    }
    if (index<0) return NULL;
    if (index>CheatLast) return NULL;
    return (CheatList[index]);
}

static char listitem[255];
int cheatlist_mode;
char * CheatLister2(int index, int *list_size)
{
    if (index < 0) {
       if (list_size)
          if (cheatlist_mode != 1)
              *list_size = MAX_CHEATS;
          else
              *list_size = 10;
       return NULL;
    }
    if (index<0) return NULL;
    if ((index>MAX_CHEATS && cheatlist_mode != 1) || (index>10 && cheatlist_mode == 1))
        return NULL;
    if (cheatlist_mode == 0)
    {
        if (Constants[index].Enabled == FALSE)
            sprintf(listitem, "%2i. Empty slot", index + 1);
        else
            sprintf(listitem, "%2i. Address: $%06X Value: %3U ($%02X)",
                    index + 1, Constants[index].RAddress,
                    Constants[index].Value, Constants[index].Value);
        return listitem;
    }
    else
    {
        if (index >= num_cheats)
            sprintf(listitem, "%2i. Empty slot", index + 1);
        else
            switch (Cheats[index].num_bytes) {
                case 1: sprintf (listitem, "%2i. $%06X <- %02X",
                                 index + 1, Cheats[index].address,
                                 Cheats[index].bytes[0]);
                        break;
                case 2: sprintf (listitem, "%2i. $%06X <- %02X, %02X",
                                 index + 1, Cheats[index].address,
                                 Cheats[index].bytes[0], Cheats[index].bytes[1]);
                        break;
                case 3: sprintf (listitem, "%2i. $%06X <- %02X, %02X, %02X",
                                 index + 1, Cheats[index].address,
                                 Cheats[index].bytes[0], Cheats[index].bytes[1],
                                 Cheats[index].bytes[2]);
                        break;
                default: sprintf (listitem, "%2i.  Error.", index + 1);
            }

/*            sprintf(listitem, "%2i. $%06X <- %02X, %02X, %02X",
                    index + 1, Cheats[index].address,
                    Cheats[index].bytes[0], Cheats[index].bytes[1],
                    Cheats[index].bytes[2]);*/
        return listitem;
    }
    sprintf(listitem, "Empty slot (#%i)", index + 1);
    return listitem;
}

char searchtext[80]="0";
char statustext[80]="Ready.";

int search_button(int msg, DIALOG *d, int c);
int clear_button(int msg, DIALOG *d, int c);
int reset_button(int msg, DIALOG *d, int c);
int list_button(int msg, DIALOG *d, int c);
int cf_list_proc(int msg, DIALOG *d, int c);

static DIALOG cheatfind_dialog[] =
{
   /* (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)  (d1)  (d2)  (dp) */
   { d_3d_box_proc,     0,    0,    280,  195,  0,    0,    0,    D_EXIT,  0,    0,    "Find Cheats" },
   { d_text_proc,       8,    23,     1,    1,  0,    0,    0,    0,       0,    0,    "&Find:" },
   { d_3dedit_proc,     40,   20,    80,   14,  0,    0,    'F',  0,       9,    0,    searchtext },
   { search_button,     210,  20,    60,   16,  0,    0,    'S',  D_EXIT,  5,    0,    "&Search" },
   { d_3dbutton_proc,   210,  40,    60,   16,  0,    0,    27,   D_EXIT,  5,    0,    "Close" },
   { clear_button,      142,  20,    60,   16,  0,    0,    'C',  D_EXIT,  5,    0,    "&Clear" },
   { reset_button,      142,  40,    60,   16,  0,    0,    'R',  D_EXIT,  5,    0,    "&Reset" },
   { list_button,       210,  60,    60,   16,  0,    0,    'L',  D_EXIT,  5,    0,    "&List" },
   { d_3dradio_proc,    40,   38,    91,    9,  0,    0,    'E',  FOCSEL,  1,    0,    "&Exact" },
   { d_3dradio_proc,    40,   50,    91,    9,  0,    0,    'N',  FOCUS,   1,    0,    "&Near" },
   { d_3dradio_proc,    40,   62,    91,    9,  0,    0,    'H',  FOCUS,   1,    0,    "C&hanged" },
   { cf_list_proc,      8,    84,   262,   86,  0,    0,    0,    0,       -1,   0,    CheatLister},
   { d_bevel_proc,      4,   175,   270,   15,  0,    0,    0,    0,       5,    0,    NULL},
   { d_text_proc,       8,   179,     1,    1,  0,    0,    0,    0,       0,    0,    statustext},
   { NULL }
};

int cf_list_proc(int msg, DIALOG *d, int c)
{
    int ret;
    static int recurseflag = 0;
    font = &s9gui_smallfont;
    recurseflag++;
    ret = d_3dlist_proc(msg, d, c);
    recurseflag--;
    if (!recurseflag)
       font = &s9gui_xm6x10;
    return ret;
}

char ch_address[255] = "$$", ch_value[255] = "", ch_type[255] = "Add&ress";
int cheatslist_proc(int msg, DIALOG *d, int c);

int add_button(int msg, DIALOG *d, int c);
int remove_button(int msg, DIALOG *d, int c);

#define id_claddr  2
#define id_clvalu  4
#define id_ggcode  5
#define id_parcode 6
#define id_gfcode  7
#define id_cladd   8
#define id_clrem   9
#define id_clexit  10
#define id_cllist  11
#define id_clstat  13

static DIALOG cheatlist_dialog[] =
{
   /* (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)  (d1)  (d2)  (dp) */
   { d_3d_box_proc,     0,    0,    280,  195,  0,    0,    0,    D_EXIT,  0,    0,    "List Cheats" },
   { d_text_proc,       8,    23,     1,    1,  0,    0,    0,    0,       0,    0,    ch_type },
   { d_3dedit_proc,     57,   20,    80,   14,  0,    0,    'R',  0,       8,    0,    ch_address + 1 },
   { d_text_proc,       8,    43,     1,    1,  0,    0,    0,    0,       0,    0,    "&Value:" },
   { d_3dedit_proc,     57,   40,    80,   14,  0,    0,    'V',  0,       9,    0,    ch_value },
   { d_3dradio_proc,    20,   38,    91,    9,  0,    0,    'G',  FOCSEL,  1,    0,    "&Game Genie" },
   { d_3dradio_proc,    20,   50,    91,    9,  0,    0,    'O',  FOCUS,   1,    0,    "Pr&o Action" },
   { d_3dradio_proc,    20,   62,    91,    9,  0,    0,    'L',  FOCUS,   1,    0,    "Go&ldfinger" },
   { add_button,        152,  20,    55,   16,  0,    0,    13,   D_EXIT,  5,    0,    "Add" },
   { remove_button,     152,  40,    55,   16,  0,    0,    'M',  D_EXIT,  5,    0,    "Re&move" },
   { d_3dbutton_proc,   215,  20,    55,   16,  0,    0,    27,   D_EXIT,  5,    0,    "Close" },
   { cheatslist_proc,   8,    73,   262,   95,  0,    0,    0,    D_EXIT,  0,    0,    CheatLister2},
   { d_bevel_proc,      4,   175,   270,   15,  0,    0,    0,    0,       5,    0,    NULL},
   { d_text_proc,       8,   179,     1,    1,  0,    0,    0,    0,       0,    0,    statustext},
   { NULL }
};

int cheatslist_proc(int msg, DIALOG *d, int c)
{
    int ret;
    static int lastchosen, recurseflag = 0;
    font = &s9gui_smallfont;
    recurseflag++;
    ret = d_3dlist_proc(msg, d, c);
    if ((msg == MSG_CLICK && lastchosen != d->d1) || ret == D_CLOSE)
    {
        show_mouse(NULL);
        if (cheatlist_mode == 0)
        {
           if (Constants[d->d1].Enabled == FALSE)
           {
               strcpy(ch_address + 1, "");
               strcpy(ch_value, "");
           }
           sprintf(ch_address + 1, "%X", Constants[d->d1].RAddress);
           sprintf(ch_value, "%U", Constants[d->d1].Value);
        }
        else
        {

            sprintf(ch_address, "%i", d->d1);
            sprintf(ch_value, "%i", d->d1);
        }
        font = &s9gui_xm6x10;
        SEND_MESSAGE(&cheatlist_dialog[id_claddr], MSG_DRAW, 0);
        if (!(cheatlist_dialog[id_clvalu].flags & D_HIDDEN))
            SEND_MESSAGE(&cheatlist_dialog[id_clvalu], MSG_DRAW, 0);
        font = &s9gui_smallfont;
        show_mouse(screen);
        lastchosen = d->d1;
        if (ret == D_CLOSE)
            ret = D_O_K;
    }
    recurseflag--;
    if (!recurseflag)
       font = &s9gui_xm6x10;
    return ret;
}

int search_button(int msg, DIALOG *d, int c)
{
   int ret;
   ret = d_3dbutton_proc(msg, d, c);
   if (ret == D_CLOSE)
   {
      char teststring[255];
      sprintf(teststring, "Read %s as %i", searchtext, GetIntFromString(searchtext));
      //alert_3d("TEST", NULL, teststring, "OK", NULL, 13, 0);
      if (cheatfind_dialog[8].flags & D_SELECTED)
          FindExact(GetIntFromString(searchtext));
      else if (cheatfind_dialog[9].flags & D_SELECTED)
          FindNear(GetIntFromString(searchtext));
      else if (cheatfind_dialog[10].flags & D_SELECTED)
          FindChanged(GetIntFromString(searchtext));
      ret = D_OK;
      show_mouse(NULL);
      SEND_MESSAGE(d, MSG_DRAW, 0);
      show_mouse(NULL);
      set_mouse_sprite(&s9gui_cursor);
      show_mouse(NULL);
      //set_mouse_sprite_d(&s9gui_cursor_d);
      show_mouse(NULL);
      set_mouse_sprite_focus(14,4);
      show_mouse(NULL);
      show_mouse(screen);
   }
   return ret;
}

int clear_button(int msg, DIALOG *d, int c)
{
   int ret;

   ret = d_3dbutton_proc(msg, d, c);
   if (ret == D_CLOSE)
   {
      CheatLast = 0;
      CheatFirst = 0;

      show_mouse(NULL);
      SEND_MESSAGE(&cheatfind_dialog[11], MSG_START, 0);
      SEND_MESSAGE(&cheatfind_dialog[11], MSG_DRAW, 0);
      SEND_MESSAGE(d, MSG_DRAW, 0);
      show_mouse(screen);
   }
   return ret;
}

int reset_button(int msg, DIALOG *d, int c)
{
   int ret;

   ret = d_3dbutton_proc(msg, d, c);
   if (ret == D_CLOSE)
   {
      //CheatLast = 1;
      //CheatFirst = 0;

      ClearCheatResults();
      SetStatusInfo("Search reset.", STATUS_DONE);

      show_mouse(NULL);
      //SEND_MESSAGE(&cheatfind_dialog[11], MSG_START, 0);
      SEND_MESSAGE(&cheatfind_dialog[11], MSG_DRAW, 0);
      SEND_MESSAGE(d, MSG_DRAW, 0);
      show_mouse(screen);
   }
}

int list_button(int msg, DIALOG *d, int c)
{
   int ret;

   ret = d_3dbutton_proc(msg, d, c);
   if (ret == D_CLOSE)
   {
      CheatLast = 0;
      CheatFirst = 0;
      SetStatusInfo("Listing....", STATUS_WAIT);
      ShowFound();
      SetStatusInfo("Ready.", STATUS_DONE);

      show_mouse(NULL);
      //SEND_MESSAGE(&cheatfind_dialog[11], MSG_START, 0);
      SEND_MESSAGE(&cheatfind_dialog[11], MSG_DRAW, 0);
      SEND_MESSAGE(d, MSG_DRAW, 0);
      show_mouse(screen);
   }
}

int add_button(int msg, DIALOG *d, int c)
{
   int ret;

   ret = d_3dbutton_proc(msg, d, c);
   if (ret == D_CLOSE)
   {
      show_mouse(NULL);
      char teststring[255];
      if (cheatlist_mode == 0)
      {
         SetStatusInfo("Cheat Added.", STATUS_DONE);
         KeepConstant(GetIntFromString(ch_address), GetIntFromString(ch_value));
      }
      else
      {
         //sprintf(teststring, "Add cheat for code %s", ch_address);
         //alert_3d("Verify", NULL, teststring, "OK", NULL, 0, 0);
         if (cheatlist_dialog[id_ggcode].flags & D_SELECTED)
         {
              uint32 address;
              uint8 byte;
              const char *error;
              if ((error = S9xGameGenieToRaw (ch_address+1, address, byte)) == NULL)
                  S9xAddCheat (address, TRUE, FALSE, 1, byte);
              else
                  alert_3d ("Snes9x: WARNING", NULL, error, "OK", NULL, 0, 0);
         }
         else if (cheatlist_dialog[id_gfcode].flags & D_SELECTED)
         {
              uint32 address;
              uint8 bytes [3];
              bool8 sram;
              uint8 num_bytes;
              const char *error;
              if ((error = S9xGoldFingerToRaw (ch_address+1, address, sram,
                                            num_bytes, bytes)) == NULL)
                  S9xAddCheat (address, FALSE, sram, num_bytes, bytes [0],
                            bytes [1], bytes [2]);
              else
                  alert_3d ("Snes9x: WARNING", NULL, error, "OK", NULL, 0, 0);
         }
         else if (cheatlist_dialog[id_parcode].flags & D_SELECTED)
         {
              uint32 address;
              uint8 byte;
              const char *error;
              if ((error = S9xProActionReplayToRaw (ch_address+1, address, byte)) == NULL)
                  S9xAddCheat (address, TRUE, FALSE, 1, byte);
              else
                  alert_3d ("Snes9x: WARNING", NULL, error, "OK", NULL, 0, 0);
         }
      }
          
      show_mouse(NULL);
      SEND_MESSAGE(&cheatlist_dialog[id_cllist], MSG_DRAW, 0);
      SEND_MESSAGE(d, MSG_DRAW, 0);
      show_mouse(screen);
   }
}

int remove_button(int msg, DIALOG *d, int c)
{
   int ret;

   ret = d_3dbutton_proc(msg, d, c);
   if (ret == D_CLOSE)
   {
      char teststring[255];
      if (cheatlist_mode == 0)
         RemoveConstant(cheatlist_dialog[id_cllist].d1);
      else
      {
         sprintf(teststring, "Remove code cheat at index %i", cheatlist_dialog[id_cllist].d1);
         alert_3d("Verify", NULL, teststring, "OK", NULL, 0, 0);
      }
          
      show_mouse(NULL);
      //SEND_MESSAGE(&cheatfind_dialog[11], MSG_START, 0);
      SEND_MESSAGE(&cheatlist_dialog[id_cllist], MSG_DRAW, 0);
      SEND_MESSAGE(d, MSG_DRAW, 0);
      show_mouse(screen);
   }
}

void SetStatusInfo( char * Str, int status_type)
{
    
    strcpy(statustext, Str);
    if (active_dialog == cheatfind_dialog)
    {
       if (status_type == STATUS_WAIT)
          cheatfind_dialog[13].fg = 169;
       else
          cheatfind_dialog[13].fg = 160;
       show_mouse(NULL);
       SEND_MESSAGE(&cheatfind_dialog[12], MSG_DRAW, 0);
       SEND_MESSAGE(&cheatfind_dialog[13], MSG_DRAW, 0);
    }
    else
    {
       if (status_type == STATUS_WAIT)
          cheatlist_dialog[id_clstat].fg = 169;
       else
          cheatlist_dialog[id_clstat].fg = 160;
       show_mouse(NULL);
       SEND_MESSAGE(&cheatlist_dialog[id_clstat-1], MSG_DRAW, 0);
       SEND_MESSAGE(&cheatlist_dialog[id_clstat], MSG_DRAW, 0);
    }

    if (status_type == STATUS_WAIT)
    {
       show_mouse(NULL);
       set_mouse_sprite(&s9gui_hglass);
       show_mouse(NULL);
       set_mouse_sprite_focus(8,16);
       show_mouse(NULL);
    }
    else
    {
       show_mouse(NULL);
       set_mouse_sprite(&s9gui_cursor);
       show_mouse(NULL);
       set_mouse_sprite_focus(14,4);
       show_mouse(NULL);
    }
    show_mouse(screen);
}

int CheatCodeDialog()
{
    text_mode(0);
    centre_dialog(cheatlist_dialog);
    cheatlist_mode = 1;
    cheatlist_dialog[id_claddr].d1 = 12;
    cheatlist_dialog[id_clvalu-1].flags |= D_HIDDEN;
    cheatlist_dialog[id_clvalu].flags |= D_HIDDEN;
    cheatlist_dialog[id_ggcode].flags = D_SELECTED;
    cheatlist_dialog[id_parcode].flags = 0;
    cheatlist_dialog[id_gfcode].flags = 0;
    sprintf(ch_type, "C&ode:");
    cheatlist_dialog[id_claddr].key = 'O';
    set_dialog_color(cheatlist_dialog, 160, 168);
    popup_dialog(cheatlist_dialog, 1);
    text_mode(-1);
    return 0;
}

int CheatAddDialog()
{
    text_mode(0);
    centre_dialog(cheatlist_dialog);
    cheatlist_mode = 0;
    cheatlist_dialog[id_clvalu-1].flags = 0;
    cheatlist_dialog[id_clvalu].flags = 0;
    cheatlist_dialog[id_claddr].d1 = 8;
    cheatlist_dialog[id_ggcode].flags = D_HIDDEN;
    cheatlist_dialog[id_parcode].flags = D_HIDDEN;
    cheatlist_dialog[id_gfcode].flags = D_HIDDEN;
    sprintf(ch_type, "Add&ress:");
    cheatlist_dialog[id_claddr].key = 'R';
    set_dialog_color(cheatlist_dialog, 160, 168);
    popup_dialog(cheatlist_dialog, 1);
    text_mode(-1);
    return 0;
}

int CheatFindDialog()
{
    text_mode(0);
    centre_dialog(cheatfind_dialog);
    set_dialog_color(cheatfind_dialog, 160, 168);
    popup_dialog(cheatfind_dialog, 1);
    text_mode(-1);
    return 0;
}

int ROMOpenDialog()
{
    static char pathstr[256] = "";
    if (strcmp(pathstr, "") == 0)
        strcpy(pathstr, S9xGetROMDirectory ());
    else {
        char drv[5], dir[254], fname[254], ext[254];
        fnsplit(pathstr, drv, dir, fname, ext);
        fnmerge(pathstr, drv, dir, "", "");
    }
    if (FileSelect("Open File...", pathstr, "SMC;FIG;078;SFC;GZ;SWC;1;ZIP", TRUE, TRUE, &Settings.ForceInterleaved))
    {
        //Load rom into memory.
        int BakFPU;
        show_mouse(NULL);
        stretch_blit(&s9gui_logo, screen, 0, 0, s9gui_logo.w, s9gui_logo.h, 0, 0, gfx_driver->w, gfx_driver->h);
        text_mode(-1);
        Memory.SaveSRAM (S9xGetSRAMFilename ());
        Memory.LoROM = Memory.HiROM = FALSE;

        extern int noiniinter;
        //noiniinter = 1;   // Override the De-interleave in the ini.
        if (!Memory.LoadROM(pathstr))
        {
           alert_3d("Error", String, NULL, "Cancel", NULL, 27, 0);
        }
        Memory.LoadSRAM (S9xGetSRAMFilename ());
        S9xReset ();
        //ResetPPU ();
        //ResetDMA ();
        //ResetAPU ();
        neverrun = TRUE;
        nothingloaded = FALSE;
        ChangeStatus(nothingloaded);

        if (install_int_ex (TimerInterrupt, (long)(((double)Settings.FrameTime * 1193181)/1000000)) < 0)
            Settings.SkipFrames = 3;

        show_mouse(screen);
    }
    ChangeStatus(nothingloaded);
    return 0;
}

int QuitEmulator()
{
    text_mode(0);
    if (alert_3d("Warning!", NULL, "Are you sure you want to quit?", "Ok", "Cancel", 0, 27)==1)
    {
       if (UseSFX)
       {
           BITMAP * TEST, * TEST2;
           PALETTE TEST3;
           show_mouse(NULL);

           int sx1, sy1;
           sx1=gfx_driver->w;
           sy1=gfx_driver->h;
           TEST=create_bitmap(gfx_driver->w, gfx_driver->h);
           TEST2=create_bitmap(gfx_driver->w, gfx_driver->h);
           blit(screen, TEST, 0, 0, 0, 0, gfx_driver->w, gfx_driver->h);
           for (int q=256; q>0; q-=8)
           //for (int q=0; q<=256; q+=8)
           {
               clear(TEST2);
               fade_interpolate(black_pallete, s9gui_logopal, TEST3, q>>2, 0, 255);
               rotate_sprite(TEST2, TEST, 0, 0, (q&255)<<16);
               stretch_blit(TEST2, screen, (sx1-((sx1*q)/256))>>1, (sy1-((sy1*q)/256))>>1, (sx1*q)/256, (sy1*q)/256, 0, 0, sx1, sy1);
               set_palette(TEST3);

               rest (10);
           }
           destroy_bitmap(TEST2);
           destroy_bitmap(TEST);
           show_mouse(screen);
       }
       WantQuit = 1;
       return 0;
    }
    text_mode(-1);
}

int ResetQuery ()
{
    if (alert_3d("Warning!", NULL, "You will lose all progress.  Continue?", "Ok", "Mommy, I'm scared!", 0, 27)==1)
    {
        S9xReset ();
        neverrun = TRUE;
    }
}

int OtherLoadSnap()
{
    BITMAP * TEST;
    TEST=create_bitmap(gfx_driver->w, gfx_driver->h);
    show_mouse(NULL);
    blit(screen, TEST, 0, 0, 0, 0, gfx_driver->w, gfx_driver->h);
    if (!S9xLoadSnapshot (S9xChooseFilename (TRUE)))
    {
        blit(TEST, screen, 0, 0, 0, 0, gfx_driver->w, gfx_driver->h);
        show_mouse(screen);
        destroy_bitmap(TEST);
        nothingloaded = FALSE;
        return 0;
     }
     S9xSetSoundMute(TRUE);
     if (stricmp(String, ""))
         alert_3d("Warning", NULL, String, "Ok", NULL, 0, 0);
     blit(TEST, screen, 0, 0, 0, 0, gfx_driver->w, gfx_driver->h);
     show_mouse(screen);
     destroy_bitmap(TEST);
     while (joy_b1 || joy_b2)
         poll_joystick();
     ChangeStatus(nothingloaded);
}

int OtherSaveSnap ()
{
     Snapshot (S9xChooseFilename (TRUE));
     while (joy_b1 || joy_b2)
        poll_joystick();
}


char cpuinf[160], rominf[160], rominf0[160], rominf1[160], miscinf[160], miscinf1[160];
static DIALOG info_dialog[] =
{
   /* (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)  (d1)  (d2)  (dp) */
   { d_3d_box_proc,     0,    0,    300,  120,  0,    0,    0,    0,       0,    0,    "Information" },
   { d_text_proc,       8,    20,     1,    1,  0,    0,    0,    0,       0,    0,    rominf },
   { d_text_proc,       8,    32,     1,    1,  0,    0,    0,    0,       0,    0,    rominf0 },
   { d_text_proc,       8,    44,     1,    1,  0,    0,    0,    0,       0,    0,    rominf1 },
   { d_text_proc,       8,    58,     1,    1,  0,    0,    0,    0,       0,    0,    cpuinf },
   { d_text_proc,       8,    70,     1,    1,  0,    0,    0,    0,       0,    0,    miscinf },
   { d_text_proc,       8,    82,     1,    1,  0,    0,    0,    0,       0,    0,    miscinf1 },
   { d_3dbutton_proc,   130,  100,   40,   16,  0,    0,    13,   D_EXIT,  0,    0,    "OK" },
   { NULL }
};


static MENU CheatSub[] = {
    { "Search..."               ,CheatFindDialog ,NULL },
    { "Apply Patch..."          ,CheatAddDialog  ,NULL },
    { "Game Codes..."           ,CheatCodeDialog ,NULL, D_DISABLED },
    { NULL, NULL, NULL       }
};

static MENU OtherSub[] = {
    { "Information..."          ,NULL            ,NULL, D_DISABLED },
    { "Reset..."                ,ResetQuery      ,NULL },
    { "Reload SRAM..."          ,NULL            ,NULL, D_DISABLED },
    { "Save Snapshot..."        ,OtherSaveSnap   ,NULL },
    { "Load Snapshot..."        ,OtherLoadSnap   ,NULL },
    {  NULL                     ,NULL            ,NULL },
};

static MENU menu_main[] =
{
    { "Open ROM..."             ,ROMOpenDialog   ,NULL },
    { "Options..."              ,DoOptions       ,NULL },
    { "Cheat"                   ,NULL            ,CheatSub },
    { "Other"                   ,NULL            ,OtherSub },
    { "About..."                ,docredits       ,NULL },
    { "Resume"                  ,NULL            ,NULL },
    { "Quit"                    ,QuitEmulator    ,NULL },
    {  NULL                     ,NULL            ,NULL },
};

void ChangeStatus(int i)
{
    if (i) {
        menu_main[2].flags = menu_main[3].flags = menu_main[5].flags = D_DISABLED;
    }
    else {
        menu_main[2].flags = menu_main[3].flags = menu_main[5].flags = 0;
    }
}

int mainmnu()
{
   PALETTE mypal;
   int i, result = 0;
   olddepth = _color_depth;

   printf("Starting GUI....\n");
   fflush(stdout);
   SaveResolution = TRUE;
   InGUI = TRUE;
   FONT* oldfont;
   oldfont=font;
   gui_mouse_focus = 0;
   if (nothingloaded)
      neverrun = TRUE;

   font = &s9gui_xm6x10;

   //Initialise joystick positional settings.
   //joy_b1 = joy_b2 = joy_b3 = joy_b4 = joy_up = joy_down = joy_left = joy_right = 0;

   printf("Config palette\n");
   fflush(stdout);
   for (i=0; i<32; i++)
   {
       s9gui_logopal[i+176].r=(((GradientClr1>>18)&0x3F)*(32-i))/32 +
                      (((GradientClr2>>18)&0x3F)*i)/32;
       if (s9gui_logopal[i+176].r > 63) s9gui_logopal[i+176].r = 63;

       s9gui_logopal[i+176].g=(((GradientClr1>>10)&0x3F)*(32-i))/32 +
                      (((GradientClr2>>10)&0x3F)*i)/32;
       if (s9gui_logopal[i+176].g > 63) s9gui_logopal[i+176].g = 63;

       s9gui_logopal[i+176].b=(((GradientClr1>>2)&0x3F)*(32-i))/32 +
                      (((GradientClr2>>2)&0x3F)*i)/32;
       if (s9gui_logopal[i+176].b > 63) s9gui_logopal[i+176].b = 63;
   }


   memcpy(mypal, s9gui_logopal, sizeof(mypal));
   set_color_depth(8);

   //printf("Set mode\n");
   //fflush(stdout);
   set_gfx_mode(1, 320, 200, 0, 0);

   //printf("Install mouse\n");
   //fflush(stdout);

   install_mouse();

   set_palette(mypal);

   set_mouse_sprite(&s9gui_cursor);
   //set_mouse_sprite_d(&s9gui_cursor_d);
   set_mouse_sprite_focus(14,4);
   gui_fg_color=160;
   gui_bg_color=168;

   //if (SnesPro)
   //    alert_3d("Error", NULL, "SNESPRO ON!", "Ok", NULL, 13, 0);

   //printf("Set-up keyboard\n");
   //fflush(stdout);

   install_keyboard();
   i = -1;
   stretch_blit(&s9gui_logo, screen, 0, 0, s9gui_logo.w, s9gui_logo.h, 0, 0, gfx_driver->w, gfx_driver->h);
   ChangeStatus(nothingloaded);

   //printf("textout()\n");
   //fflush(stdout);

   //textout(screen, font, "This is a test!", 50, 50, 5);

   //printf("gui_textout()\n");
   //fflush(stdout);

   //gui_textout(screen, "This is a test~!", 50, 50, 4, FALSE);

   //printf("Show menu\n");
   //fflush(stdout);

   while (i != 5 && !WantQuit)
       i = do_menu3d(menu_main, 1, 1);
   remove_keyboard();
   //result = MainMnu();
   show_mouse(NULL);
   remove_mouse();
   set_gfx_mode (GFX_TEXT, 0, 0, 0, 0);
   font=oldfont;
   InGUI = FALSE;
   set_color_depth(olddepth);
   if (WantQuit)
       return -1;
   return 0;
}
