/*
 * Snes9x - Portable Super Nintendo Entertainment System (TM) emulator.
 *

 * Permission to use, copy, modify and distribute Snes9x in both binary and
 * source form, for non-commercial purposes, is hereby granted without fee,
 * providing that this license information and copyright notice appear with
 * all copies and any derived work.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event shall the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Snes9x is freeware for PERSONAL USE only. Commercial users should
 * seek permission of the copyright holders first. Commercial use includes
 * charging money for Snes9x or software derived from Snes9x.
 *
 * The copyright holders request that bug fixes and improvements to the code
 * should be forwarded to them so everyone can benefit from the modifications
 * in future versions.
 *
 * Super NES and Super Nintendo Entertainment System are trademarks of
 * Nintendo Co., Limited and its subsidiary companies.
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <dir.h>
#include <unistd.h>
#include <fcntl.h>
#include <bios.h>
#include <go32.h>
#include <allegro.h>
#include <ctype.h>

#include "snes9x.h"
#include "port.h"
#include "cheats.h"
#include "gui.h"
#include "inifuncs.h"

static char fs_thingie1[200] = "Name:  N/A\0";
static char fs_thingie2[200] = "Type:  N/A\0";
static char fs_thingie3[200] = "       N/A\0";

static int fs_edit_proc(int, DIALOG *, int );
static int fs_flist_proc(int, DIALOG *, int );
static int fs_dlist_proc(int, DIALOG *, int );
static int fs_okbut_proc(int, DIALOG *, int );
static char *fs_flist_getter(int, int *);
static char *fs_dlist_getter(int, int *);

void getstats(const char* S);

#define FLIST_SIZE      2048

typedef struct FLIST
{
   char dir[256];
   int size;
   char *name[FLIST_SIZE];
} FLIST;

static FLIST *flist = NULL;

static char *fext = NULL;

static DIALOG alert_dialog[] =
{
   /* (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)  (d1)  (d2)  (dp) */
   { d_3d_box_proc,     0,    0,    0,    0,    0,    0,    0,    0,       0,    0,    NULL },
   { d_ctext_proc,      0,    0,    0,    0,    0,    0,    0,    D_HIDDEN,0,    0,    "" },
   { d_ctext_proc,      0,    0,    0,    0,    0,    0,    0,    0,       0,    0,    NULL },
   { d_ctext_proc,      0,    0,    0,    0,    0,    0,    0,    0,       0,    0,    NULL },
   { d_3dbutton_proc,   0,    0,    0,    0,    0,    0,    0,    D_EXIT,  0,    0,    NULL },
   { d_3dbutton_proc,   0,    0,    0,    0,    0,    0,    0,    D_EXIT,  0,    0,    NULL },
   { d_3dbutton_proc,   0,    0,    0,    0,    0,    0,    0,    D_EXIT,  0,    0,    NULL },
   { NULL }
};
                                   
#define A_S1  1
#define A_S2  2
#define A_S3  3
#define A_B1  4
#define A_B2  5
#define A_B3  6


/* alert3_3d:
 *  Displays a simple alert box, containing three lines of text (s1-s3),
 *  and with either one, two, or three buttons. The text for these buttons 
 *  is passed in b1, b2, and b3 (NULL for buttons which are not used), and
 *  the keyboard shortcuts in c1 and c2. Returns 1, 2, or 3 depending on 
 *  which button was selected.
 */
int alert3_3d(char *s1, char *s2, char *s3, char *b1, char *b2, char *b3, int c1, int c2, int c3)
{
   int maxlen = 0;
   int len1, len2, len3;
   int avg_w = text_length(font, "o");
   int avg_h = text_height(font);
   int buttons = 0, lines = 0;
   int b[3];
   int c;

   #define SORT_OUT_BUTTON(x) {                                            \
      if (b##x) {                                                          \
	 alert_dialog[A_B##x].flags &= ~D_HIDDEN;                          \
	 alert_dialog[A_B##x].key = c##x;                                  \
	 alert_dialog[A_B##x].dp = b##x;                                   \
         len##x = gui_strlen((unsigned char*)b##x);                        \
	 b[buttons++] = A_B##x;                                            \
      }                                                                    \
      else {                                                               \
	 alert_dialog[A_B##x].flags |= D_HIDDEN;                           \
	 len##x = 0;                                                       \
      }                                                                    \
   }

   alert_dialog[A_S1].dp = alert_dialog[A_S2].dp = alert_dialog[A_S3].dp = 
			   alert_dialog[A_B1].dp = alert_dialog[A_B2].dp = "";

   if (s1) {
      alert_dialog[0].dp = s1;
      maxlen = text_length(font, s1);
   }
   else
   {
      alert_dialog[0].dp = "Alert!";
      maxlen = text_length(font, "Alert!");
   }

   if (s2) {
      alert_dialog[A_S2].dp = s2;
      len1 = text_length(font, s2);
      alert_dialog[A_S2].flags &= ~D_HIDDEN;
      lines++;
      if (len1 > maxlen)
	 maxlen = len1;
   }
   else
      alert_dialog[A_S2].flags |= D_HIDDEN;

   if (s3) {
      alert_dialog[A_S3].dp = s3;
      len1 = text_length(font, s3);
      alert_dialog[A_S3].flags &= ~D_HIDDEN;
      lines++;
      if (len1 > maxlen)
	 maxlen = len1;
   }
   else
      alert_dialog[A_S3].flags |= D_HIDDEN;

   SORT_OUT_BUTTON(1);
   SORT_OUT_BUTTON(2);
   SORT_OUT_BUTTON(3);

   len1 = MAX(len1, MAX(len2, len3)) + avg_w*3;
   if (len1*buttons > maxlen)
      maxlen = len1*buttons;

   maxlen += avg_w*4;
   alert_dialog[0].w = maxlen + 6;
   alert_dialog[A_S1].x = alert_dialog[A_S2].x = alert_dialog[A_S3].x = 
						alert_dialog[0].x + maxlen/2;

   alert_dialog[A_B1].w = alert_dialog[A_B2].w = alert_dialog[A_B3].w = len1;

   alert_dialog[A_B1].x = alert_dialog[A_B2].x = alert_dialog[A_B3].x = 
				       alert_dialog[0].x + maxlen/2 - len1/2;

   if (buttons == 3) {
      alert_dialog[b[0]].x = alert_dialog[0].x + maxlen/2 - len1*3/2 - avg_w;
      alert_dialog[b[2]].x = alert_dialog[0].x + maxlen/2 + len1/2 + avg_w;
   }
   else if (buttons == 2) {
      alert_dialog[b[0]].x = alert_dialog[0].x + maxlen/2 - len1 - avg_w;
      alert_dialog[b[1]].x = alert_dialog[0].x + maxlen/2 + avg_w;
   }

   alert_dialog[0].h = (int)(avg_h*(3.75+lines) + 15);
   //alert_dialog[A_S1].y = alert_dialog[0].y + avg_h;
   alert_dialog[A_S2].y = alert_dialog[0].y + (int)(avg_h * 1.25) + 9;
   if (s2)
      alert_dialog[A_S3].y = alert_dialog[0].y + (int)(avg_h * 2.25) + 9;
   else
      alert_dialog[A_S3].y = alert_dialog[0].y + (int)(avg_h * 1.25) + 9;

   alert_dialog[A_S1].h = alert_dialog[A_S2].h = alert_dialog[A_S2].h = avg_h;

   if (lines<2)
      alert_dialog[A_B1].y = alert_dialog[A_B2].y = alert_dialog[A_B3].y =
                                                   alert_dialog[0].y + avg_h*3 + 9;
   else
      alert_dialog[A_B1].y = alert_dialog[A_B2].y = alert_dialog[A_B3].y =
                                                   alert_dialog[0].y + avg_h*4 + 9;

   alert_dialog[A_B1].h = alert_dialog[A_B2].h = alert_dialog[A_B3].h = 16;

   centre_dialog(alert_dialog);
   set_dialog_color(alert_dialog, gui_fg_color, gui_bg_color);

   clear_keybuf();

   do {
   } while (mouse_b);

   show_mouse(NULL);  //
   c = popup_dialog(alert_dialog, A_B1);
   show_mouse(screen);

   if (c == A_B1)
      return 1;
   else if (c == A_B2)
      return 2;
   else
      return 3;
}



/* alert:
 *  Displays a simple alert box, containing three lines of text (s1-s3),
 *  and with either one or two buttons. The text for these buttons is passed
 *  in b1 and b2 (b2 may be null), and the keyboard shortcuts in c1 and c2.
 *  Returns 1 or 2 depending on which button was selected.
 */
int alert_3d(char *s1, char *s2, char *s3, char *b1, char *b2, int c1, int c2)
{
   int ret;

   ret = alert3_3d(s1, s2, s3, b1, b2, NULL, c1, c2, 0);

   if (ret > 2)
      ret = 2;

   return ret;
}


static DIALOG file_selector[] =
{
   /* (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)  (d1)  (d2)  (dp) */
   { d_3d_box_proc,     0,    0,    304,  214,  0,    0,    0,    0,       0,    0,    "File Select" },
   { d_ctext_proc,      152,  18,   1,    1,    0,    0,    0,    0,       0,    0,    "" },
   { d_3dbutton_proc,   208,  111,  80,   16,   0,    0,    0,    D_EXIT,  0,    0,    "OK" },
   { d_3dbutton_proc,   208,  135,  80,   16,   0,    0,    27,   D_EXIT,  0,    0,    "Cancel" },
   { fs_edit_proc,      16,   24,   272,  16,   0,    0,    0,    0,       79,   0,    NULL },
   { fs_flist_proc,     16,   46,   176,  105,  0,    0,    'F',  D_EXIT,  0,    0,    fs_flist_getter },
   { fs_dlist_proc,     208,  46,   80,   55,   0,    0,    'D',  D_EXIT,  0,    0,    fs_dlist_getter },
   { d_fptext_proc,     16,   158,  272,  1,    0,    0,    0,    0,       0,    0,    fs_thingie1 },
   { d_fptext_proc,     16,   170,  272,  1,    0,    0,    0,    0,       0,    0,    fs_thingie2 },
   { d_fptext_proc,     16,   182,  272,  1,    0,    0,    0,    0,       0,    0,    fs_thingie3 },
   { d_3dcheck_proc,    16,   194,  180,  9,    0,    0,    'I',  0,       0,    0,    "De-&interleave" },
   { NULL }
};

static DIALOG file_selector1[] =
{
   /* (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)  (d1)  (d2)  (dp) */
   { d_3d_box_proc,     0,    0,    304,  214,  0,    0,    0,    0,       0,    0,    "File Select" },
   { d_ctext_proc,      152,  18,   1,    1,    0,    0,    0,    0,       0,    0,    "" },
   { fs_okbut_proc,     208,  111,  80,   16,   0,    0,    0,    D_EXIT,  0,    0,    "OK" },
   { d_3dbutton_proc,   208,  135,  80,   16,   0,    0,    27,   D_EXIT,  0,    0,    "Cancel" },
   { fs_edit_proc,      16,   24,   272,  16,   0,    0,    0,    0,       79,   0,    NULL },
   { fs_flist_proc,     16,   46,   176,  105,  0,    0,    'F',  D_EXIT,  0,    0,    fs_flist_getter },
   { fs_dlist_proc,     208,  46,   80,   55,   0,    0,    'D',  D_EXIT,  0,    0,    fs_dlist_getter },
   { d_fptext_proc,     16,   158,  270,  1,    0,    0,    0,    0,       0,    0,    fs_thingie1 },
   { d_fptext_proc,     16,   170,  270,  1,    0,    0,    0,    0,       0,    0,    fs_thingie2 },
   { d_fptext_proc,     16,   182,  270,  1,    0,    0,    0,    0,       0,    0,    fs_thingie3 },
   { d_3dcheck_proc,    16,   194,  180,  9,    0,    0,    'I',  0,       0,    0,    "De-&interleave" },
   { NULL }
};

#define FS_MESSAGE   1
#define FS_OK        2
#define FS_CANCEL    3
#define FS_EDIT      4
#define FS_FILES     5
#define FS_DISKS     6
#define FS_RMNAME    7
#define FS_RMSTAT    8
#define FS_DEINT     10
                     
char drvlist[30] = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
int  numdrives = 26;
bool issnesrom = FALSE, must_exist = FALSE;

void fd_initdrvlst()
{
   union __dpmi_regs r;
   int j, i;

   numdrives=0;
   i=0;
   for (; i<26; i++)
   {
       r.x.ax=0x4408;
       r.x.bx=i+1;
       __dpmi_int(0x21, &r);
       if ((r.x.flags&1)==0)
       {
          drvlist[numdrives]='A'+i;
          if (i!=1 || (_bios_equiplist() & 192) > 64)
              numdrives++;         // This is NOT the right way to do this,
                                   // but I have no other way to determine
                                   // single drive systems
       }
       else
       {
           r.x.ax=0x3200;
           r.x.dx=i+1;
           __dpmi_int(0x21, &r);
           if ((r.x.ax&0xFF)==0)
           {
              drvlist[numdrives]='A'+i;
              numdrives++;
           }
           else
           {
               r.x.ax=0x150B;
               r.x.cx=i;
               __dpmi_int(0x2F, &r);
               if ((r.x.bx==0xadad)&&(r.x.ax!=0)) // CDROM drive
               {
                  drvlist[numdrives]='A'+i;
                  numdrives++;
               }
           }
       }
   }
   //if (j>26) j=26;
   //numdrives=j;
   if (numdrives<26)
       for (i=numdrives; i<=numdrives; i++)
          drvlist[i]=0;
}

/* fs_dlist_getter:
 *  Listbox data getter routine for the file selector disk list.
 */
static char *fs_dlist_getter(int index, int *list_size)
{
   static char d[200] = "A: ";
   union __dpmi_regs r;

   if (index < 0) {
      if (list_size)
         *list_size = numdrives;
      return NULL;
   }

   r.x.ax=0x4408;
   r.x.bx=drvlist[index]-'A'+1;
   __dpmi_int(0x21, &r);
   if ((r.x.flags&1)==0 && r.x.ax==0)
       strcpy(d, " A:");   //Diskette drive [removable]
   else
   {
       r.x.ax=0x4409;
       r.x.bx=drvlist[index]-'A'+1;
       __dpmi_int(0x21, &r);
       if (r.x.ax & 0x1000)
       {  // Remote [network or cdrom]
          r.x.ax=0x150B;
          r.x.cx=drvlist[index]-'A';
          __dpmi_int(0x2F, &r);
          if ((r.x.bx==0xadad)&&(r.x.ax!=0)) // CDROM drive
             strcpy(d, " A:");
          else
             strcpy(d, " A:");     // Is remote (network)
       }
       else
       {
          r.x.ax=0x150B;
          r.x.cx=drvlist[index]-'A';
          __dpmi_int(0x2F, &r);
          if ((r.x.bx==0xadad)&&(r.x.ax!=0)) // CDROM drive
             strcpy(d, " A:");
          else
             strcpy(d, " A:");    // Must be a hard disk.
       }
   }

   d[2] = drvlist[index]; // 'A'+index;
   return d;
}


/* fs_okbut_proc:
 *  Dialog procedure for the file selector Ok button.
 */
static int fs_okbut_proc(int msg, DIALOG *d, int c)
{
   int ret;
   char *s = (char*)file_selector[FS_EDIT].dp;

   ret = d_3dbutton_proc(msg, d, c);
   if (ret == D_CLOSE)
   {
       if (must_exist)
       {
           if (!exists(s))
           {
              alert_3d("Warning", NULL, "The specified file could not be found.", "Ok", NULL, 13, 0);
              ret = D_O_K;
           }
       }
   }
   return ret;
}

/* fs_dlist_proc:
 *  Dialog procedure for the file selector disk list.
 */
static int fs_dlist_proc(int msg, DIALOG *d, int c)
{
   int ret, i;
   char *s = (char*)file_selector[FS_EDIT].dp;

   if (d->d1 > numdrives) d->d1=numdrives;
   if (msg == MSG_START)
   {
      d->d1 = d->d2 = 0;
      if (s[1] == ':')
         for (i=0; i<numdrives; i++)
            if (drvlist[i]==s[0])
               d->d1 = i;
   }

   ret = d_3dlist_proc(msg, d, c);

   if (ret == D_CLOSE) {
      *(s++) = drvlist[d->d1]; // 'A'+d->d1; //
      *(s++) = ':';
      *(s++) = '\\';
      *s = 0;
      show_mouse(NULL);
      getstats(s);
      SEND_MESSAGE(file_selector+FS_FILES, MSG_START, 0);
      SEND_MESSAGE(file_selector+FS_FILES, MSG_DRAW, 0);
      SEND_MESSAGE(file_selector+FS_EDIT, MSG_START, 0);
      SEND_MESSAGE(file_selector+FS_EDIT, MSG_DRAW, 0);
      show_mouse(screen);
      return D_O_K;
   }

   return ret;
}



/* fs_edit_proc:
 *  Dialog procedure for the file selector editable string.
 */
static int fs_edit_proc(int msg, DIALOG *d, int c)
{
   char *s = (char*)d->dp;
   int ch;
   int attr;
   int x;
   char b[256];

   if (msg == MSG_START) {
      if (s[0]) {
	 _fixpath(s, b);
	 errno = 0;

	 x = s[strlen(s)-1];
	 if ((x=='/') || (x=='\\'))
	    put_backslash(b);

	 for (x=0; b[x]; x++) {
	    if (b[x] == '/')
	       s[x] = '\\';
	    else
	       s[x] = toupper(b[x]);
	 }
	 s[x] = 0;
      }
   }

   if (msg == MSG_KEY) {
      if (*s)
	 ch = s[strlen(s)-1];
      else
	 ch = 0;
      if (ch == ':')
	 put_backslash(s);
      else
      {
	 if ((ch != '/') && (ch != '\\')) {
	    if (file_exists(s, FA_RDONLY | FA_HIDDEN | FA_DIREC, &attr)) {
	       if (attr & FA_DIREC)
		  put_backslash(s);
               else if (access(s, D_OK)==0)
		  put_backslash(s);
               else {
                  if (must_exist && exists(s))
                      return D_CLOSE;
                  else if (!must_exist)
                      return D_CLOSE;
                  else 
                      alert_3d("Warning", NULL, "The specified file could not be found.", "Ok", NULL, 13, 0);
               }
	    }
            else if (strcmp(s + strlen(s) - 3, "\\..")==0) {
               put_backslash(s);
            }
            else {
               if (must_exist && exists(s))
                   return D_CLOSE;
               else if (!must_exist)
                   return D_CLOSE;
               else if (strcmp(s + strlen(s) - 3, "\\..")!=0)
                   alert_3d("Warning", NULL, "The specified file could not be found.", "Ok", NULL, 13, 0);
               else {
                   show_mouse(NULL);
                   SEND_MESSAGE(file_selector+FS_FILES, MSG_START, 0);
                   SEND_MESSAGE(file_selector+FS_FILES, MSG_DRAW, 0);
                   SEND_MESSAGE(d, MSG_START, 0);
                   SEND_MESSAGE(d, MSG_DRAW, 0);
                   show_mouse(screen);
               }
            }
	 }
      }
      show_mouse(NULL);
      SEND_MESSAGE(file_selector+FS_FILES, MSG_START, 0);
      SEND_MESSAGE(file_selector+FS_FILES, MSG_DRAW, 0);
      SEND_MESSAGE(d, MSG_START, 0);
      SEND_MESSAGE(d, MSG_DRAW, 0);
      show_mouse(screen);
      return D_O_K;
   }

   if (msg==MSG_CHAR) {
      ch = c & 0xff;
      if ((ch >= 'a') && (ch <= 'z'))
	 c = (c & 0xffffff00L) | (ch - 'a' + 'A');
      else if (ch == '/')
	 c = (c & 0xffffff00L) | '\\';
      else if (_USE_LFN) {
	 if ((ch > 127) || ((ch < 32) && (ch != 8) && (ch != 0)))
	    return D_O_K;
      }
      else {
	 if ((ch != '\\') && (ch != '_') && (ch != ':') && (ch != '.') && 
	     ((ch < 'A') || (ch > 'Z')) && ((ch < '0') || (ch > '9')) &&
	     (ch != 8) && (ch != 127) && (ch != 0))
	    return D_O_K;
      }
   }

   if (msg==MSG_DRAW)
      getstats(s);

   return d_3dedit_proc(msg, d, c); 
}



/* fs_flist_putter:
 *  Callback routine for for_each_file() to fill the file selector listbox.
 */
static void fs_flist_putter(char *str, int attrib)
{
   int c, c2;
   char *s, *ext, *tok;
   char tmp[512];
   static char ext_tokens[] = " ,;";

   s = get_filename(str);
   strupr(s);

   if ((fext) && (!(attrib & FA_DIREC))) {
      strcpy(tmp, fext);
      ext = get_extension(s);
      tok = strtok(tmp, ext_tokens);
      while (tok) {
	 if (stricmp(ext, tok) == 0)
	    break;
	 tok = strtok(NULL, ext_tokens);
      }
      if (!tok)
	 return;
   }

   if ((flist->size < FLIST_SIZE) && (strcmp(s,".") != 0)) {
      for (c=0; c<flist->size; c++) {
	 if (flist->name[c][strlen(flist->name[c])-1]=='\\') {
	    if (attrib & FA_DIREC)
               if (strcmp(s, flist->name[c]+1) < 0)
		  break;
	 }
	 else {
	    if (attrib & FA_DIREC)
	       break;
            if (strcmp(s, flist->name[c]+1) < 0)
	       break;
	 }
      }
      for (c2=flist->size; c2>c; c2--)
	 flist->name[c2] = flist->name[c2-1];

      flist->name[c] = (char*)malloc(strlen(s) + ((attrib & FA_DIREC) ? 3 : 2));
      strcpy(flist->name[c]+1, s);
      if (attrib & FA_DIREC)
      {
         flist->name[c][0] = 132;
         put_backslash(flist->name[c]);
      }
      else
         flist->name[c][0] = 133;
      flist->size++;
   }
}



/* fs_flist_getter:
 *  Listbox data getter routine for the file selector list.
 */
static char *fs_flist_getter(int index, int *list_size)
{
   if (index < 0) {
      if (list_size)
	 *list_size = flist->size;
      return NULL;
   }
   return flist->name[index];
}

static char DosExtErrors[0x69][80]={
   "No error",                        
   "Function number invalid",         
   "File not found",
   "Path not found",
   "Too many files open",             
   "Access denied",                   
   "Invalid handle",                  
   "Memory control block destroyed",  
   "Insufficient memory",             
   "Memory block address invalid",    
   "Environment invalid",             
   "Format invalid",
   "Access code invalid",
   "Data invalid",                    
   "-Reserved-",                      
   "Invalid drive",                   
   "Cannot remove current directory", 
   "Not same device",                 
   "No more files",                   
   "Disk write-protected",            
   "Unknown unit",                    
   "Drive not ready",                 
   "Unknown command",                 
   "Data error",                      
   "Bad request structure length",    
   "Seek error",                      
   "Unknown media type (non-DOS disk)",
   "Sector not found",                
   "Printer out of paper",            
   "Write fault",                     
   "Read fault",                      
   "General failure",                 
   "Sharing violation",               
   "Lock violation",                  
   "Invalid disk change",             
   "FCB unavailable",                 
   "Sharing buffer overflow",         
   "Code page mismatch",              
   "Cannot complete file operation (EOF)",
   "Insufficient disk space",         
   "-reserved-", "-reserved-", "-reserved-", "-reserved-", "-reserved-",
   "-reserved-", "-reserved-", "-reserved-", "-reserved-", "-reserved-",
   "Network request not supported",   
   "Remote computer not responding",  
   "Duplicate name on network",       
   "Network name not found",          
   "Network busy",                    
   "Network device no longer exists", 
   "NetBIOS command limit exceeded",  
   "Network adaptor hardware error",  
   "Incorrect response from network", 
   "Unexpected network error",        
   "Incompatible remote adaptor",     
   "Print queue filled",              
   "Queue not full",                  
   "Not enough space to print file",  
   "Network name was deleted",        
   "Access denied to network",        
   "Network device type incorrect",   
   "Network name not found",          
   "Network name limit exceeded",     
   "Network BIOS session limit exceeded",
   "Temporarily paused",              
   "Network request not accepted",    
   "Network print/disk redir paused", 
   "Network software not installed/Invalid version", 
   "Unexpected adaptor close/Account expired", 
   "Password expired",
   "login attempt invalid at this time", 
   "Disk limit exceeded on this network node", 
   "Not logged in",
   "reserved",
   "File already exists",
   "reserved",
   "Cannot create directory",
   "Error accessing device",
   "Too many redirections",
   "Duplicate redirection",
   "Invalid password",
   "Invalid parameter",
   "Network write fault",
   "Function not supported on network",
   "Required system component not installed",
   "MSCDEX: Unknown error",
   "MSCDEX: Not ready",
   "MSCDEX: EMS memory no longer valid",
   "MSCDEX: Invalid format",
   "MSCDEX: Door open"
};

static char DosExtAction[][80]=
{
   "",
   "Please retry",
   "Try again later",
   "Please re-enter the file or directory.",
   "Abort",
   "Abort now",
   "Ignore",
   "Fix the error, then try again."
};

static char DosExtClass[][80]=
{
   "",
   "",
   "",
   "Authorization ",
   "Internal ",
   "Hardware ",
   "System ",
   "Application ",
   "",
   "",
   "",
   "Media",
   "",
   "Unknown"
};

static char DosExtLocus[][80]=
{
   "",
   "Unknown",
   "Disk",
   "Network",
   "Serial Device",
   "Memory",
};

/* fs_flist_proc:
 *  Dialog procedure for the file selector list.
 */
static int fs_flist_proc(int msg, DIALOG *d, int c)
{
   int i, ret, reallyused = 0;
   int sel = d->d1;
   char *s = (char*)file_selector[FS_EDIT].dp;
   static int recurse_flag = 0;

   font = &s9gui_smallfont;
   if (msg == MSG_START) {
      if (!flist)
         flist = (FLIST*)malloc(sizeof(FLIST));
      else {
	 for (i=0; i<flist->size; i++)
	    if (flist->name[i])
	       free(flist->name[i]);
      }
      if (!flist) {
	 errno = ENOMEM;
         font = &s9gui_xm6x10;
	 return D_CLOSE; 
      }
      flist->size = 0;
      for (i=0; i<flist->size; i++)
	 flist->name[i] = NULL;
      strcpy(flist->dir, s);
      getstats(s);
      *get_filename(flist->dir) = 0;
      put_backslash(flist->dir);
      strcat(flist->dir,"*.*");
      for_each_file(flist->dir, FA_RDONLY | FA_DIREC | FA_ARCH, (void (*)(...))fs_flist_putter, 0);
      if (errno)
      {
         struct _DOSERROR de;
         char errstr[255], titstr[255], errstr1[255];
         
         _dosexterr(&de);
         sprintf(errstr, "%s", DosExtErrors[de.exterror]);
         sprintf(errstr1, "%s", DosExtAction[de.action]);
         sprintf(titstr, "%s%s Error", DosExtLocus[de.locus],
                 DosExtClass[de.errclass]);
         alert_3d(titstr, errstr, errstr1, "OK", NULL, 13, 0);
      }
      *get_filename(flist->dir) = 0;
      d->d1 = d->d2 = 0;
      sel = 0;
   }

   if (msg == MSG_END) {
      if (flist) {
	 for (i=0; i<flist->size; i++)
	    if (flist->name[i])
	       free(flist->name[i]);
	 free(flist);
      }
      flist = NULL;
   }

   recurse_flag++;
   ret = d_3dlist_proc(msg,d,c);     /* call the parent procedure */

   if (msg == MSG_CHAR && ret == D_O_K) {  // Jerremy's suggestion
       if (toupper(flist->name[d->d1][1]) == toupper(c&0xFF) && flist->size > d->d1)
       {
          for (i=d->d1+1; i<flist->size; i++)
             if (toupper(flist->name[i][1]) == toupper(c&0xFF))
             {
                 d->d1 = i;
                 ret = D_USED_CHAR;
                 i = flist->size;
             }
          if (ret != D_USED_CHAR)
             for (i=0; i<d->d1; i++)
                if (toupper(flist->name[i][1]) == toupper(c&0xFF))
                {
                    d->d1 = i;
                    ret = D_USED_CHAR;
                    i = flist->size;
                }
       }
       else
       {
          for (i=0; i<flist->size; i++)
             if (toupper(flist->name[i][1]) == toupper(c&0xFF))
             {
                 d->d1 = i;
                 ret = D_USED_CHAR;
                 i = flist->size;
             }
       }
       show_mouse(NULL);
       d_3dlist_proc(MSG_START, d, 0);
       SEND_MESSAGE(d, MSG_DRAW, 0);
       show_mouse(screen);
   }

   recurse_flag--;
   if (recurse_flag == 0)
       font = &s9gui_xm6x10;

   if (((sel != d->d1) || (ret == D_CLOSE)) && (recurse_flag == 0)) {
      strcpy(s, flist->dir);
      *get_filename(s) = 0;
      put_backslash(s);
      strcat(s, flist->name[d->d1] + 1);
      show_mouse(NULL);
      getstats(s);
      SEND_MESSAGE(file_selector+FS_EDIT, MSG_START, 0);
      SEND_MESSAGE(file_selector+FS_EDIT, MSG_DRAW, 0);
      show_mouse(screen);

      if (ret == D_CLOSE)
	 return SEND_MESSAGE(file_selector+FS_EDIT, MSG_KEY, 0);
   }

   return ret;
}

#include <zlib.h>
#define STREAM gzFile
#define READ_STREAM(p,l,s) gzread (s,p,l)
#define WRITE_STREAM(p,l,s) gzwrite (s,p,l)
#define OPEN_STREAM(f,m) gzopen (f,m)
#define CLOSE_STREAM(s) gzclose (s)

void getstats(const char* S)
{
    //STREAM ROMFile;
    return;
}


/* FileSelect:
 *  Displays the Allegro file selector, with the message as caption. 
 *  Allows the user to select a file, and stores the selection in path 
 *  (which should have room for at least 80 characters). The files are
 *  filtered according to the file extensions in ext. Passing NULL
 *  includes all files, "PCX;BMP" includes only files with .PCX or .BMP
 *  extensions. Returns zero if it was closed with the Cancel button, 
 *  non-zero if it was OK'd.
 */
int FileSelect(char *message, char *path, char *ext, bool8 mustexist, bool8 snesrom, bool8 * deinterleave)
{
   int ret;
   char *p;

   if (snesrom)
     issnesrom = TRUE;
   else
     issnesrom = FALSE;

   must_exist = mustexist;

   // Restore defaults
   memcpy(file_selector, file_selector1, sizeof(file_selector1));

   file_selector[0].dp = message;
   file_selector[FS_DEINT].flags = D_WANTFOCUS;
   file_selector[FS_EDIT].dp = path;
   fext = ext;

   if (SaveResolution)
   {
      file_selector[0].x = file_selector[0].y = 0;
      file_selector[0].w = 282;
      file_selector[0].h = 175;

      file_selector[2].x = 192;     //OK Button
      file_selector[2].y = 91;
      file_selector[2].h = 14;

      file_selector[3].x = 192;     //Cancel Button
      file_selector[3].y = 109;
      file_selector[3].h = 14;

      file_selector[4].x = 8;       //Edit Box
      file_selector[4].y = 22;
      file_selector[4].h = 14;
      file_selector[4].w = 264;

      file_selector[5].x = 8;       //File List
      file_selector[5].y = 42;
      file_selector[5].h = 81;

      file_selector[6].x = 192;     //Drive List
      file_selector[6].y = 42;
      file_selector[6].h = 45;

      file_selector[7].x = 8;       //Static text
      file_selector[7].y = 128;

      file_selector[8].x = 8;       //Static text
      file_selector[8].y = 138;

      file_selector[9].x = 8;       //Static text
      file_selector[9].y = 148;

      file_selector[10].x = 8;       //Check box
      file_selector[10].y = 158;
   }

   if (snesrom)
   {
      file_selector[7].proc = d_fptext_proc;
      if (SaveResolution)
          file_selector[0].h = 175;
      else
          file_selector[0].h = 214;
   }
   else
   {
      file_selector[7].proc = NULL;
      if (SaveResolution)
          file_selector[0].h = 132;
      else
          file_selector[0].h = 166;
   }

   if (path[0] == 0) {
      getcwd(path, 80);
      for (p=path; *p; p++)
	 if (*p=='/')
	    *p = '\\';
	 else
	    *p = toupper(*p);

      put_backslash(path);
   }

   clear_keybuf();

   do {
   } while (mouse_b);

   fd_initdrvlst();

   centre_dialog(file_selector);
   set_dialog_color(file_selector, 160, 168);
   show_mouse(NULL); 
   ret = popup_dialog(file_selector, FS_EDIT);
   show_mouse(screen);

   if ((ret == FS_CANCEL) || (*get_filename(path) == 0))
      return FALSE;

   p = get_extension(path);
   if ((*p == 0) && (ext) && (!strpbrk(ext, " ,;"))) {
      *p = '.';
      strcpy(p+1, ext);
   }

   *deinterleave = (file_selector[FS_DEINT].flags & D_SELECTED) == D_SELECTED;

   return TRUE;
}

//End of Fileselector.

static char *gfx_mode_getter(int index, int *list_size);
static char *gfx_card_getter(int index, int *list_size);
static int gfx_card_proc(int, DIALOG *, int );
static int gfx_mode_proc(int, DIALOG *, int );

static DIALOG gfx_mode_dialog[] =
{
   /* (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)  (d1)  (d2)  (dp) */
   { d_3d_box_proc,     0,    0,    290,  176,  0,    0,    0,    0,       0,    0,    "Graphics Mode" },
   { d_3dbutton_proc,   180,  102,  100,  16,   0,    0,    0,    D_EXIT,  0,    0,    "OK" },
   { d_3dbutton_proc,   180,  122,  100,  16,   0,    0,    27,   D_EXIT,  0,    0,    "Cancel" },
   { gfx_card_proc,     8,    22,   164,  116,  0,    0,    0,    D_EXIT,  0,    0,    gfx_card_getter },
   { gfx_mode_proc,     180,  22,   100,  75,   0,    0,    0,    D_EXIT,  3,    0,    gfx_mode_getter },
   { d_3dcheck_proc,    8,    146,  91,   9,    0,    0,    0,    0,       0,    0,    "High-colour" },
   { d_3dcheck_proc,    8,    158,  91,   9,    0,    0,    0,    0,       0,    0,    "High-resolution" },
   { NULL }
};

#define GFX_CANCEL         2
#define GFX_DRIVER_LIST    3
#define GFX_MODE_LIST      4
#define GFX_HIGH_COLOR     5
#define GFX_HIGH_RES       6


typedef struct GFX_MODE_DATA
{
   int w;
   int h;
   char *s;
   int xmode;
} GFX_MODE_DATA;


static GFX_MODE_DATA gfx_mode_data[] =
{
   { 320,   200,  "320x200",   0 },
   { 320,   240,  "320x240",   0 },
   { 512,   384,  "512x384",   0 },
   { 640,   400,  "640x400",   0 },
   { 640,   480,  "640x480",   0 },
   { 800,   600,  "800x600",   0 },
   { 1024,  768,  "1024x768",  0 },
   { 1280,  1024, "1280x1024", 0 },
   { 1600,  1200, "1600x1200", 0 },
   { 256,   200,  "256x200",   1 },
   { 256,   224,  "256x224",   1 },
   { 256,   240,  "256x240",   2 },
   { 256,   256,  "256x256",   2 },
   { 320,   400,  "320x400",   1 },
   { 320,   480,  "320x480",   1 },
   { 320,   600,  "320x600",   1 },
   { 360,   200,  "360x200",   1 },
   { 360,   240,  "360x240",   1 },
   { 360,   270,  "360x270",   1 },
   { 360,   360,  "360x360",   1 },
   { 360,   400,  "360x400",   1 },
   { 360,   480,  "360x480",   1 },
   { 360,   600,  "360x600",   1 },
   { 376,   282,  "376x282",   1 },
   { 376,   308,  "376x308",   1 },
   { 376,   564,  "376x564",   1 },
   { 400,   150,  "400x150",   1 },
   { 400,   300,  "400x300",   1 },
   { 400,   600,  "400x600",   1 }
};


static GFX_MODE_DATA gfx_svga_mode_data[] =
{
   { 320,   200,  "320x200",   0 },
   { 320,   240,  "320x240",   0 },
   { 512,   384,  "512x384",   0 },
   { 640,   400,  "640x400",   0 },
   { 640,   480,  "640x480",   0 },
   { 800,   600,  "800x600",   0 },
   { 1024,  768,  "1024x768",  0 },
   { 1280,  1024, "1280x1024", 0 },
   { 1600,  1200, "1600x1200", 0 }
};

static GFX_MODE_DATA gfx_vga_mode_data[] =
{
   { 320,   200,  "320x200",   0 },
   { 256,   240,  "256x240",   0 },
   { 256,   256,  "256x256",   0 }
};

static GFX_MODE_DATA gfx_modesx_data[] =
{
   { 640,   400,  "640x400",   0 },
};

static GFX_MODE_DATA gfx_modex_data[] =
{
   { 320,   200,  "320x200",   0 },
   { 320,   240,  "320x240",   0 },
   { 256,   200,  "256x200",   1 },
   { 256,   224,  "256x224",   1 },
   { 256,   240,  "256x240",   2 },
   { 256,   256,  "256x256",   2 },
   { 320,   400,  "320x400",   1 },
   { 320,   480,  "320x480",   1 },
   { 320,   600,  "320x600",   1 },
   { 360,   200,  "360x200",   1 },
   { 360,   240,  "360x240",   1 },
   { 360,   270,  "360x270",   1 },
   { 360,   360,  "360x360",   1 },
   { 360,   400,  "360x400",   1 },
   { 360,   480,  "360x480",   1 },
   { 360,   600,  "360x600",   1 },
   { 376,   282,  "376x282",   1 },
   { 376,   308,  "376x308",   1 },
   { 376,   564,  "376x564",   1 },
   { 400,   150,  "400x150",   1 },
   { 400,   300,  "400x300",   1 },
   { 400,   600,  "400x600",   1 }
};

static int gfx_card_proc(int msg, DIALOG * d, int c)
{
   int ret, i;
   ret = d_3dlist_proc(msg, d, c);
   if (msg == MSG_CLICK || msg == MSG_KEY || msg == MSG_CHAR)
   {
       show_mouse(NULL);
       SEND_MESSAGE(&gfx_mode_dialog[GFX_MODE_LIST], MSG_START, 0);
       SEND_MESSAGE(&gfx_mode_dialog[GFX_MODE_LIST], MSG_DRAW, 0);
       SEND_MESSAGE(&gfx_mode_dialog[GFX_MODE_LIST], MSG_KEY, 0);
       i = gfx_mode_dialog[GFX_DRIVER_LIST].d1;
       if ((i == 0) || (i == 3) || (i == 4) || (i == 5))
          gfx_mode_dialog[GFX_HIGH_COLOR].flags &= ~D_DISABLED;
       else
          gfx_mode_dialog[GFX_HIGH_COLOR].flags |= D_DISABLED;
       SEND_MESSAGE(&gfx_mode_dialog[GFX_HIGH_COLOR], MSG_DRAW, 0);
       show_mouse(screen);
   }
}

static int gfx_mode_proc(int msg, DIALOG * d, int c)
{
   int ret, i, card, w, h;
   ret = d_3dlist_proc(msg, d, c);
   if (msg == MSG_CLICK || msg == MSG_KEY || msg == MSG_CHAR)
   {
      card = gfx_mode_dialog[GFX_DRIVER_LIST].d1;

      switch (card)
      {
          case 0:
             w = gfx_mode_data[gfx_mode_dialog[GFX_MODE_LIST].d1].w;
             h = gfx_mode_data[gfx_mode_dialog[GFX_MODE_LIST].d1].h;
             break;
          case 1:
             w = gfx_vga_mode_data[gfx_mode_dialog[GFX_MODE_LIST].d1].w;
             h = gfx_vga_mode_data[gfx_mode_dialog[GFX_MODE_LIST].d1].h;
             break;
          case 2:
             w = gfx_modex_data[gfx_mode_dialog[GFX_MODE_LIST].d1].w;
             h = gfx_modex_data[gfx_mode_dialog[GFX_MODE_LIST].d1].h;
             break;
          case 7:
             w = gfx_modesx_data[gfx_mode_dialog[GFX_MODE_LIST].d1].w;
             h = gfx_modesx_data[gfx_mode_dialog[GFX_MODE_LIST].d1].h;
             break;
          default:
             w = gfx_svga_mode_data[gfx_mode_dialog[GFX_MODE_LIST].d1].w;
             h = gfx_svga_mode_data[gfx_mode_dialog[GFX_MODE_LIST].d1].h;
             break;
       }
       if (w >= 512 && h >= 478)
          gfx_mode_dialog[GFX_HIGH_RES].flags &= ~D_DISABLED;
       else
          gfx_mode_dialog[GFX_HIGH_RES].flags |= D_DISABLED;
       SEND_MESSAGE(&gfx_mode_dialog[GFX_HIGH_RES], MSG_DRAW, 0);
    }
}

/* gfx_mode_getter:
 *  Listbox data getter routine for the graphics mode list.
 */
static char *gfx_mode_getter(int index, int *list_size)
{
   switch (gfx_mode_dialog[GFX_DRIVER_LIST].d1)
   {
       case 0:
              if (index < 0) {
                  if (list_size)
                      *list_size = sizeof(gfx_mode_data) / sizeof(GFX_MODE_DATA);
                  return NULL;
              }
              return gfx_mode_data[index].s;
              break;
       case 1:
              if (index < 0) {
                  if (list_size)
                      *list_size = sizeof(gfx_vga_mode_data) / sizeof(GFX_MODE_DATA);
                  return NULL;
              }
              return gfx_vga_mode_data[index].s;
              break;
       case 2:
              if (index < 0) {
                  if (list_size)
                      *list_size = sizeof(gfx_modex_data) / sizeof(GFX_MODE_DATA);
                  return NULL;
              }
              return gfx_modex_data[index].s;
              break;
       case 7:
              if (index < 0) {
                  if (list_size)
                      *list_size = sizeof(gfx_modesx_data) / sizeof(GFX_MODE_DATA);
                  return NULL;
              }
              return gfx_modesx_data[index].s;
              break;
       default:
              if (index < 0) {
                  if (list_size)
                      *list_size = sizeof(gfx_svga_mode_data) / sizeof(GFX_MODE_DATA);
                  return NULL;
              }
              return gfx_svga_mode_data[index].s;
              break;
   }
   return gfx_mode_data[index].s;
}


/* gfx_card_getter:
 *  Listbox data getter routine for the graphics card list.
 */
static char *gfx_card_getter(int index, int *list_size)
{
   static char *card[] = {
      "Autodetect",
      "VGA mode 13h",
      "Mode-X",

   #ifdef DJGPP
	 /* djgpp drivers */
	 "VESA 1.x",
	 "VESA 2.0 (banked)",
	 "VESA 2.0 (linear)",
	 "VBE/AF",
	 "Xtended mode",
	 "ATI 18800/28800",
	 "ATI mach64",
	 "Cirrus 64xx",
	 "Cirrus 54xx",
	 "Paradise",
	 "S3",
	 "Trident",
	 "Tseng ET3000",
	 "Tseng ET4000",
	 "Video-7"

   #else
	 /* linux drivers */
	 "SVGALIB"
   #endif
   };

   if (index < 0) {
      if (list_size)
	 *list_size = sizeof(card) / sizeof(char *);
      return NULL;
   }

   return card[index];
}


/* gfx_mode_select:
 *  Displays the Allegro graphics mode selection dialog, which allows the
 *  user to select a screen mode and graphics card. Stores the selection
 *  in the three variables, and returns zero if it was closed with the 
 *  Cancel button, or non-zero if it was OK'd.
 */
int gfx_mode_3dselect(int *card, int *w, int *h, int *hires, int *bits16)
{
   int ret, i;

   clear_keybuf();

   do {
   } while (mouse_b);

   centre_dialog(gfx_mode_dialog);
   set_dialog_color(gfx_mode_dialog, gui_fg_color, gui_bg_color);
   if (*w >= 512 && *h >= 478)
      gfx_mode_dialog[GFX_HIGH_RES].flags &= ~D_DISABLED;
   else
      gfx_mode_dialog[GFX_HIGH_RES].flags |= D_DISABLED;
   if ((*card == 0) || (*card == 3) || (*card == 4) || (*card == 5))
      gfx_mode_dialog[GFX_HIGH_COLOR].flags &= ~D_DISABLED;
   else
      gfx_mode_dialog[GFX_HIGH_COLOR].flags |= D_DISABLED;

   gfx_mode_dialog[GFX_DRIVER_LIST].d1 = *card;
   switch (*card)
   {
       case 0:
          for (i=sizeof(gfx_mode_data) / sizeof(GFX_MODE_DATA); i>=0; i--)
             if (gfx_mode_data[i].w == *w && gfx_mode_data[i].h == *h)
                 gfx_mode_dialog[GFX_MODE_LIST].d1 = i;
          break;
       case 1:
          for (i=sizeof(gfx_vga_mode_data) / sizeof(GFX_MODE_DATA); i>=0; i--)
             if (gfx_vga_mode_data[i].w == *w && gfx_vga_mode_data[i].h == *h)
                 gfx_mode_dialog[GFX_MODE_LIST].d1 = i;
          break;
       case 2:
          for (i=sizeof(gfx_modex_data) / sizeof(GFX_MODE_DATA); i>=0; i--)
             if (gfx_modex_data[i].w == *w && gfx_modex_data[i].h == *h)
                 gfx_mode_dialog[GFX_MODE_LIST].d1 = i;
          break;
       case 7:
          for (i=sizeof(gfx_modesx_data) / sizeof(GFX_MODE_DATA); i>=0; i--)
             if (gfx_modesx_data[i].w == *w && gfx_modesx_data[i].h == *h)
                 gfx_mode_dialog[GFX_MODE_LIST].d1 = i;
          break;
       default:
          for (i=sizeof(gfx_svga_mode_data) / sizeof(GFX_MODE_DATA); i>=0; i--)
             if (gfx_svga_mode_data[i].w == *w && gfx_svga_mode_data[i].h == *h)
                 gfx_mode_dialog[GFX_MODE_LIST].d1 = i;
          break;
   }

   ret = popup_dialog(gfx_mode_dialog, GFX_DRIVER_LIST);

   if (ret == GFX_CANCEL)
      return FALSE;

   *card = gfx_mode_dialog[GFX_DRIVER_LIST].d1;

   switch (*card)
   {
       case 0:
          *w = gfx_mode_data[gfx_mode_dialog[GFX_MODE_LIST].d1].w;
          *h = gfx_mode_data[gfx_mode_dialog[GFX_MODE_LIST].d1].h;
          break;
       case 1:
          *w = gfx_vga_mode_data[gfx_mode_dialog[GFX_MODE_LIST].d1].w;
          *h = gfx_vga_mode_data[gfx_mode_dialog[GFX_MODE_LIST].d1].h;
          break;
       case 2:
          *w = gfx_modex_data[gfx_mode_dialog[GFX_MODE_LIST].d1].w;
          *h = gfx_modex_data[gfx_mode_dialog[GFX_MODE_LIST].d1].h;
          break;
       case 7:
          *w = gfx_modesx_data[gfx_mode_dialog[GFX_MODE_LIST].d1].w;
          *h = gfx_modesx_data[gfx_mode_dialog[GFX_MODE_LIST].d1].h;
          break;
       default:
          *w = gfx_svga_mode_data[gfx_mode_dialog[GFX_MODE_LIST].d1].w;
          *h = gfx_svga_mode_data[gfx_mode_dialog[GFX_MODE_LIST].d1].h;
          break;
   }

   if (*card != GFX_VGA)
   {
       BITMAP * testbmp;
       PALETTE  savepal;
       int      mh;

       extern BITMAP *_mouse_screen;
       mh = (_mouse_screen == screen);
       if (mh)
           show_mouse(NULL);
       testbmp = create_bitmap(gfx_driver->w, gfx_driver->h);
       get_palette(savepal);
       blit(screen, testbmp, 0, 0, 0, 0, testbmp->w, testbmp->h);
       ret = set_gfx_mode(*card, *w, *h, 0, 0);
       if (SaveResolution)
           set_gfx_mode(1, 320, 200, 0, 0);
       else
           set_gfx_mode(GFX_AUTODETECT, 640, 480, 0, 0);
       blit(testbmp, screen, 0, 0, 0, 0, testbmp->w, testbmp->h);
       set_palette(savepal);
       destroy_bitmap(testbmp);
       if (mh)
           show_mouse(screen);
       if (ret)
       {
            alert_3d("Graphics error!\n", "Resolution invalid", allegro_error, "Ok", NULL, 0, 0);
            *card = 1;     // Revert to defaults.
            *w = 256;
            *h = 256;
            *hires = *bits16 = FALSE;
       }
   }
   if (! (gfx_mode_dialog[GFX_HIGH_RES].flags & D_DISABLED))
      *hires = (gfx_mode_dialog[GFX_HIGH_RES].flags & D_SELECTED) != 0;
   else
      *hires = FALSE;
   if (! (gfx_mode_dialog[GFX_HIGH_COLOR].flags & D_DISABLED))
      *bits16 = (gfx_mode_dialog[GFX_HIGH_COLOR].flags & D_SELECTED) != 0;
   else
      *bits16 = FALSE;
   return TRUE;
}

