/*
 * Snes9x - Portable Super Nintendo Entertainment System (TM) emulator.
 *
 * This file (c) Copyright 1997, 1998 Chad Kitching
 *
 * Permission to use, copy, modify and distribute Snes9x in both binary and
 * source form, for non-commercial purposes, is hereby granted without fee,
 * providing that this license information and copyright notice appear with
 * all copies and any derived work.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event shall the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Snes9x is freeware for PERSONAL USE only. Commercial users should
 * seek permission of the copyright holders first. Commercial use includes
 * charging money for Snes9x or software derived from Snes9x.
 *
 * The copyright holders request that bug fixes and improvements to the code
 * should be forwarded to them so everyone can benefit from the modifications
 * in future versions.
 *
 * Super NES and Super Nintendo Entertainment System are trademarks of
 * Nintendo Co., Limited and its subsidiary companies.
 */
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>

#include <allegro.h>
#include "s9xgrph.h"
#include "snes9x.h"
#include "cheats.h"
#include "gui.h"

typedef char *(*getfuncptr)(int, int *);

int d_3dcheck_proc(int msg, DIALOG *d, int c);
void box3d(int x1, int y1, int x2, int y2);
void box3d2(int x1, int y1, int x2, int y2);
void box3d3(int x1, int y1, int x2, int y2);
int d_3d_box_proc(int msg, DIALOG *d, int c);

extern int GUI_Hilight, GUI_Window, GUI_Shadow, GUI_Black, GUI_White, GUI_Selectedfg, GUI_Selectedbg;

int GUI_Hilight = 175;
int GUI_Window  = 168;
int GUI_Shadow  = 167;
int GUI_Black   = 160;
int GUI_White   = 175;
int GUI_Selectedbg = 164;
int GUI_Selectedfg = 175;

// These box3dX draw various 3d styled boxes in the controls.

// Used for menus, and 3d boxes
void box3d(int x1, int y1, int x2, int y2)
{
   //Left side.
   vline(screen, x1, y1, y2-1, GUI_Window);
   vline(screen, x1+1, y1+1, y2-2, GUI_Hilight);
   vline(screen, x1+2, y1+2, y2-3, GUI_Window);

   //Bottom side.
   hline(screen, x1, y2, x2, GUI_Black);
   hline(screen, x1+1, y2-1, x2-1, GUI_Shadow);
   hline(screen, x1+2, y2-2, x2-2, GUI_Window);

   //Right side.
   vline(screen, x2, y1, y2, GUI_Black);
   vline(screen, x2-1, y1+1, y2-1, GUI_Shadow);
   vline(screen, x2-2, y1+2, y2-2, GUI_Window);

   //Top side.
   hline(screen, x1, y1, x2-1, GUI_Window);
   hline(screen, x1+1, y1+1, x2-2, GUI_Hilight);
   hline(screen, x1+2, y1+2, x2-3, GUI_Window);
}

// Used for buttons (depressed), list boxes and checkboxes
void box3d2(int x1, int y1, int x2, int y2)
{
   //Left side.
   vline(screen, x1, y1, y2-1, GUI_Shadow);
   vline(screen, x1+1, y1+1, y2-2, GUI_Black);

   //Bottom side.
   hline(screen, x1, y2, x2, GUI_Hilight);
   hline(screen, x1+1, y2-1, x2-1, GUI_Window);

   //Right side.
   vline(screen, x2, y1, y2, GUI_Hilight);
   vline(screen, x2-1, y1+1, y2-1, GUI_Window);

   //Top side.
   hline(screen, x1, y1, x2-1, GUI_Shadow);
   hline(screen, x1+1, y1+1, x2-2, GUI_Black);
}

// Used for Buttons.
void box3d3(int x1, int y1, int x2, int y2)
{
   //Left side.
   vline(screen, x1, y1, y2-1, GUI_Hilight);
   vline(screen, x1+1, y1+1, y2-2, GUI_Window);

   //Bottom side.
   hline(screen, x1, y2, x2, GUI_Black);
   hline(screen, x1+1, y2-1, x2-1, GUI_Shadow);

   //Right side.
   vline(screen, x2, y1, y2, GUI_Black);
   vline(screen, x2-1, y1+1, y2-1, GUI_Shadow);

   //Top side.
   hline(screen, x1, y1, x2-1, GUI_Hilight);
   hline(screen, x1+1, y1+1, x2-2, GUI_Window);
}

// Used for menu selection
void box3d4(int x1, int y1, int x2, int y2)
{
   vline(screen, x1,y1,y2,GUI_Shadow);
   hline(screen, x1,y1,x2,GUI_Shadow);

   vline(screen, x2,y1,y2,GUI_Hilight);
   hline(screen, x1,y2,x2,GUI_Hilight);
}

// Used on speed dials
void box3d5(int x1, int y1, int x2, int y2)
{
   vline(screen, x1,y1,y2,GUI_Shadow);
   hline(screen, x1,y1,x2,GUI_Shadow);

   vline(screen, x2,y1,y2,GUI_Hilight);
   hline(screen, x1,y2,x2,GUI_Hilight);
}

// Draws a beveled box.
void bevel3d(int x1, int y1, int x2, int y2)
{
   rect(screen, x1, y1, x2, y2, GUI_Hilight);
   rect(screen, x1+1, y1+1, x2-1, y2-1, GUI_Hilight);
   rect(screen, x1, y1, x2-1, y2-1, GUI_Shadow);
}

// Removes a 3d box from the screen
void unbox(int x1, int y1, int x2, int y2)
{
   //Left side.
   vline(screen, x1, y1, y2-1, GUI_Window);
   vline(screen, x1+1, y1+1, y2-2, GUI_Window);

   //Bottom side.
   hline(screen, x1, y2, x2, GUI_Window);
   hline(screen, x1+1, y2-1, x2-1, GUI_Window);

   //Right side.
   vline(screen, x2, y1, y2, GUI_Window);
   vline(screen, x2-1, y1+1, y2-1, GUI_Window);

   //Top side.
   hline(screen, x1, y1, x2-1, GUI_Window);
   hline(screen, x1+1, y1+1, x2-2, GUI_Window);
}


/* d_3d_box_proc:
      Control for displaying a window complete with titlebar, and
      close button.  A Windows 95/98 style dialog box.
 */
int d_3d_box_proc(int msg, DIALOG *d, int c)
{
   int i, state1, state2;
   FONT * myfont = &s9gui_titlefont;

   if (msg==MSG_DRAW) {
      int fg = (d->flags & D_DISABLED) ? gui_mg_color : d->fg;
      if (d->d1==0)
      {
          rectfill(screen, d->x+1, d->y + text_height(myfont) + 7, d->x+d->w-2, d->y + text_height(myfont) + 9, d->bg);
          box3d(d->x, d->y, d->x+d->w-1, d->y + text_height(myfont) + 9);
          for (i = d->x+3; i <= (d->x+d->w)-4; i++)
             vline(screen, i, d->y + 3, d->y + text_height(myfont) + 7, ((i-(d->x+3)) * 31 / (d->w-4)) + 177); //(((i - (d->x + 2)) * 31) / d->w-6) + 177);
          text_mode(-1);
          textout(screen, myfont, (char*)d->dp, d->x + 7, d->y + 5, 175);
          rectfill(screen, d->x+1, d->y + text_height(myfont) + 7, d->x+d->w-2, d->y+d->h-2, d->bg);
          box3d(d->x, d->y, d->x+d->w-1, d->y+d->h-1);
      }
      i = d->y + 4 + (text_height(myfont) >> 1);

      if (d->flags & D_SELECTED)
      {
         rectfill(screen, d->x + d->w - 17, i - 4, d->x + d->w - 6, i + 4, GUI_Window);
         hline(screen, d->x + d->w - 17, i - 4, d->x + d->w - 6, GUI_Black);
         vline(screen, d->x + d->w - 17, i - 4, i + 5, GUI_Black);
         hline(screen, d->x + d->w - 17, i + 5, d->x + d->w - 6, GUI_Hilight);
         vline(screen, d->x + d->w - 6, i - 4, i + 5, GUI_Hilight);

         hline(screen, d->x + d->w - 16, i - 3, d->x + d->w - 7, GUI_Shadow);
         vline(screen, d->x + d->w - 16, i - 3, i + 4, GUI_Shadow);
         hline(screen, d->x + d->w - 16, i + 4, d->x + d->w - 7, GUI_Window);
         vline(screen, d->x + d->w - 7, i - 3, i + 4, GUI_Window);

         line(screen, d->x + d->w - 13, i + 3, d->x + d->w -  9, i - 1, GUI_Black);
         line(screen, d->x + d->w - 13, i - 1, d->x + d->w -  9, i + 3, GUI_Black);
         line(screen, d->x + d->w - 14, i + 3, d->x + d->w - 10, i - 1, GUI_Black);
         line(screen, d->x + d->w - 14, i - 1, d->x + d->w - 10, i + 3, GUI_Black);
      }
      else
      {
         rectfill(screen, d->x + d->w - 17, i - 4, d->x + d->w - 6, i + 4, GUI_Window);
         hline(screen, d->x + d->w - 17, i - 4, d->x + d->w - 6, GUI_Hilight);
         vline(screen, d->x + d->w - 17, i - 4, i + 5, GUI_Hilight);
         hline(screen, d->x + d->w - 17, i + 5, d->x + d->w - 6, GUI_Black);
         vline(screen, d->x + d->w - 6, i - 4, i + 5, GUI_Black);

         hline(screen, d->x + d->w - 16, i - 3, d->x + d->w - 7, GUI_Window);
         vline(screen, d->x + d->w - 16, i - 3, i + 4, GUI_Window);
         hline(screen, d->x + d->w - 16, i + 4, d->x + d->w - 7, GUI_Shadow);
         vline(screen, d->x + d->w - 7, i - 3, i + 4, GUI_Shadow);
 
         if (d->flags & D_EXIT)
         {
            line(screen, d->x + d->w - 14, i + 2, d->x + d->w - 10, i - 2, GUI_Black);
            line(screen, d->x + d->w - 14, i - 2, d->x + d->w - 10, i + 2, GUI_Black);
            line(screen, d->x + d->w - 15, i + 2, d->x + d->w - 11, i - 2, GUI_Black);
            line(screen, d->x + d->w - 15, i - 2, d->x + d->w - 11, i + 2, GUI_Black);
         }
         else
         {
            line(screen, d->x + d->w - 13, i + 3, d->x + d->w - 9, i - 1, GUI_Hilight);
            line(screen, d->x + d->w - 13, i - 1, d->x + d->w - 9, i + 3, GUI_Hilight);
            line(screen, d->x + d->w - 14, i + 3, d->x + d->w - 10, i - 1, GUI_Hilight);
            line(screen, d->x + d->w - 14, i - 1, d->x + d->w - 10, i + 3, GUI_Hilight);
   
            line(screen, d->x + d->w - 14, i + 2, d->x + d->w - 10, i - 2, GUI_Shadow);
            line(screen, d->x + d->w - 14, i - 2, d->x + d->w - 10, i + 2, GUI_Shadow);
            line(screen, d->x + d->w - 15, i + 2, d->x + d->w - 11, i - 2, GUI_Shadow);
            line(screen, d->x + d->w - 15, i - 2, d->x + d->w - 11, i + 2, GUI_Shadow);
         }
      }

      d->d1=1;
   }
   else if ((msg==MSG_CLICK) && (d->flags & D_EXIT))
   {
      state1 = d->flags & D_SELECTED;

      /* track the mouse until it is released */
      i = d->y + 4 + (text_height(myfont) >> 1);
      while (mouse_b) {
         state2 = ((mouse_x >= d->x + d->w - 17) && (mouse_y >= i-4) &&
                  (mouse_x <= d->x + d->w - 6) && (mouse_y <= i+5));

         /* redraw? */
         if (((state1) && (!state2)) || ((state2) && (!state1))) {
            d->flags ^= D_SELECTED;
            state1 = d->flags & D_SELECTED;
            show_mouse(NULL);
            SEND_MESSAGE(d, MSG_DRAW, 0);
            show_mouse(screen);
         }
      }

      /* should we close the dialog? */
      if (d->flags & D_SELECTED) {
         d->flags ^= D_SELECTED;
         return D_CLOSE;
      }
   }
   else if (msg==MSG_END)
   {
       d->d1=0;
   }

   return D_O_K;
}

// Helper routine for displaying a dotted box around an item.
static void dotted_rect(int x1, int y1, int x2, int y2, int fg, int bg)
{
   int c;

   for (c=x1; c<=x2; c+=2) {
      putpixel(screen, c, y1, fg);
      putpixel(screen, c, y2, fg);
   }

   for (c=x1+1; c<=x2; c+=2) {
      putpixel(screen, c, y1, bg);
      putpixel(screen, c, y2, bg);
   }

   for (c=y1; c<=y2; c+=2) {
      putpixel(screen, x1, c, fg);
      putpixel(screen, x2, c, fg);
   }

   for (c=y1+1; c<=y2; c+=2) {
      putpixel(screen, x1, c, bg);
      putpixel(screen, x2, c, bg);
   }
}

/* d_3dcheck_proc:
      Displays a 2-state checkbox on the screen.  Chains to allegro's
      standard button to handle other messages not covered here.
 */
int d_3dcheck_proc(int msg, DIALOG *d, int c)
{
   int x;
   int fg;

   if (msg==MSG_DRAW) {
      fg = (d->flags & D_DISABLED) ? gui_mg_color : d->fg;
      text_mode(d->bg);
      rectfill(screen, d->x+d->h+(d->h>>1)-1, d->y+(d->h-(text_height(font)-gui_font_baseline))/2 - 1,
                  d->x+d->h+(d->h>>1)+gui_strlen((unsigned char*)d->dp)+1, d->y+text_height(font)+gui_font_baseline+1,
                  d->bg);
      if (d->flags & D_DISABLED)
      {
         text_mode(-1);
         gui_textout(screen, (unsigned char*)d->dp, d->x+d->h+(d->h>>1)+1, d->y+(d->h-(text_height(font)-gui_font_baseline))/2+1, GUI_Hilight, FALSE);
         gui_textout(screen, (unsigned char*)d->dp, d->x+d->h+(d->h>>1), d->y+(d->h-(text_height(font)-gui_font_baseline))/2, GUI_Shadow, FALSE);
         text_mode(d->bg); 
      }
      else
         gui_textout(screen, (unsigned char*)d->dp, d->x+d->h+(d->h>>1), d->y+(d->h-(text_height(font)-gui_font_baseline))/2, fg, FALSE);

      x = d->x;
      rectfill(screen, x+1, d->y+1, x+d->h-1, d->y+d->h-1, (d->flags & D_DISABLED) ? GUI_Window : GUI_White);
      box3d2(x, d->y, x+d->h, d->y+d->h);
      if (d->flags & D_SELECTED) {
         vline(screen, x+3, d->y+4, d->y+5, fg);
         vline(screen, x+4, d->y+5, d->y+6, fg);
         vline(screen, x+5, d->y+4, d->y+5, fg);
         vline(screen, x+6, d->y+3, d->y+4, fg);
      }
      if (d->flags & D_GOTFOCUS)
         dotted_rect(d->x+d->h+(d->h>>1)-1, d->y+(d->h-(text_height(font)-gui_font_baseline))/2 - 1,
                     d->x+d->h+(d->h>>1)+gui_strlen((unsigned char*)d->dp)+1, d->y+text_height(font)+gui_font_baseline+1,
                     fg, d->bg);
         //dotted_rect(x+1, d->y+1, x+d->h-1, d->y+d->h-1, fg, d->bg);
      return D_O_K;
   } 

   return d_button_proc(msg, d, 0);
}

/* d_group_proc:
      Displays a beveled box with a title to provide a visual separator
      to group sets of controls.  It can also display a title for the group.
 */
int d_group_proc(int msg, DIALOG *d, int c)
{
   int i;
   if (msg==MSG_DRAW) {
      int fg = (d->flags & D_DISABLED) ? gui_mg_color : d->fg;
      bevel3d(d->x, d->y, d->x+d->w-1, d->y+d->h-1);
      text_mode(d->bg);
      //gui_textout(screen, (unsigned char*)d->dp, d->x + text_height(font), d->y-4
      gui_textout(screen, (unsigned char*)d->dp, d->x + 5, (d->y)-((text_height(font)-gui_font_baseline))/2, d->fg, FALSE);
   }

   return D_O_K;
}

/* d_bevel_proc:
      Capable of displaying a number of different 3D boxes, or lines.
      Purely a visual control.
 */
int d_bevel_proc(int msg, DIALOG *d, int c)
{
   int i;
   if (msg==MSG_DRAW) {
      int fg = (d->flags & D_DISABLED) ? gui_mg_color : d->fg;
      rectfill(screen, d->x, d->y, d->x+d->w, d->y+d->h, d->bg);
      if (d->d1 == 0)
      {
          bevel3d(d->x, d->y, d->x+d->w, d->y+d->h);
      }
      else if (d->d1 == 1)
      {
          box3d(d->x, d->y, d->x+d->w, d->y+d->h);
      }
      else if (d->d1 == 2)
      {
          box3d2(d->x, d->y, d->x+d->w, d->y+d->h);
      }
      else if (d->d1 == 3)
      {
          box3d3(d->x, d->y, d->x+d->w, d->y+d->h);
      }
      else if (d->d1 == 4)
      {
          box3d4(d->x, d->y, d->x+d->w, d->y+d->h);
      }
      else if (d->d1 == 5)
      {
          box3d5(d->x, d->y, d->x+d->w, d->y+d->h);
      }
      else if (d->d1 == 6)
      {
          vline(screen, d->x, d->y, d->y + d->h, GUI_Shadow);
          vline(screen, d->x + d->w, d->y, d->y + d->h, GUI_Hilight);
      }
      else if (d->d1 == 7)
      {
          hline(screen, d->x, d->y, d->x + d->w, GUI_Shadow);
          hline(screen, d->x, d->y + d->h, d->x + d->w, GUI_Hilight);
      }
   }

   return D_O_K;
}

/* d_3dbutton_proc:
    A button.
 */
int d_3dbutton_proc(int msg, DIALOG *d, int c)
{
   int state1, state2;
   int swap;
   int g;

   switch (msg) {

      case MSG_DRAW:
	 if (d->flags & D_SELECTED) {
	    g = 1;
            rectfill(screen, d->x+1, d->y+1, d->x+d->w-2, d->y+d->h-2, d->bg);
            box3d2( d->x, d->y, d->x + d->w, d->y + d->h);
	 }
	 else {
	    g = 0; 
            rectfill(screen, d->x+1, d->y+1, d->x+d->w-2, d->y+d->h-2, d->bg);
            box3d3( d->x, d->y, d->x + d->w, d->y + d->h);
	 }

         if (d->flags & D_DISABLED)
         {
            text_mode(-1);
            gui_textout(screen, (unsigned char*)d->dp, d->x+d->w/2+g+1, d->y+d->h/2-text_height(font)/2+g+1, GUI_White, TRUE);
            gui_textout(screen, (unsigned char*)d->dp, d->x+d->w/2+g, d->y+d->h/2-text_height(font)/2+g, GUI_Shadow, TRUE);
         }
         else
         {
            text_mode(d->bg);
            gui_textout(screen, (unsigned char*)d->dp, d->x+d->w/2+g, d->y+d->h/2-text_height(font)/2+g, d->fg, TRUE);
         }

	 if ((d->flags & D_GOTFOCUS) && 
	     (!(d->flags & D_SELECTED) || !(d->flags & D_EXIT)))
            dotted_rect(d->x+2, d->y+2, d->x+d->w-3+g, d->y+d->h-3+g, d->fg, d->bg);
	 break;

      case MSG_WANTFOCUS:
	 return D_WANTFOCUS;

      case MSG_KEY:
	 /* close dialog? */
	 if (d->flags & D_EXIT)
	    return D_CLOSE;

	 /* or just toggle */
	 d->flags ^= D_SELECTED;
	 show_mouse(NULL);
	 SEND_MESSAGE(d, MSG_DRAW, 0);
	 show_mouse(screen);
	 break;

      case MSG_CLICK:
	 /* what state was the button originally in? */
	 state1 = d->flags & D_SELECTED;
	 if (d->flags & D_EXIT)
	    swap = FALSE;
	 else
	    swap = state1;

	 /* track the mouse until it is released */
	 while (mouse_b) {
	    state2 = ((mouse_x >= d->x) && (mouse_y >= d->y) &&
		     (mouse_x <= d->x + d->w) && (mouse_y <= d->y + d->h));
	    if (swap)
	       state2 = !state2;

	    /* redraw? */
	    if (((state1) && (!state2)) || ((state2) && (!state1))) {
	       d->flags ^= D_SELECTED;
	       state1 = d->flags & D_SELECTED;
	       show_mouse(NULL);
	       SEND_MESSAGE(d, MSG_DRAW, 0);
	       show_mouse(screen);
	    }
	 }

	 /* should we close the dialog? */
	 if ((d->flags & D_SELECTED) && (d->flags & D_EXIT)) {
	    d->flags ^= D_SELECTED;
	    return D_CLOSE;
	 }
	 break; 
   }

   return D_O_K;
}

static int curs_blink=0, last_curs_state=255;

static void cursor_blinker(...)
{
   curs_blink = !curs_blink;
}

END_OF_FUNCTION (cursor_blinker)

/* d_3dedit_proc:
      Editable text box, complete with vertical cursor.
 */
int d_3dedit_proc(int msg, DIALOG *d, int c)
{
   int f, l, p, w, x, fg;
   char *s;
   char buf[2];

   s = (char*)d->dp;
   l = strlen(s);
   if (d->d2 > l)
      d->d2 = l;

   switch (msg) {

      case MSG_START:
	 d->d2 = l;
         LOCK_FUNCTION(cursor_blinker);
         LOCK_VARIABLE(curs_blink);
         curs_blink=0;
         last_curs_state=255;
         install_int(cursor_blinker, 300);
	 break;

      case MSG_END:
         remove_int(cursor_blinker);
         return D_O_K;

      case MSG_DRAW:
         fg = (d->flags & D_DISABLED) ? gui_mg_color : d->fg;
	 buf[1] = 0;
	 x = 0;

         rectfill(screen, d->x, d->y, d->x+d->w, d->y+d->h, GUI_White); //
         text_mode(GUI_White);
	 for (p=0; p<=l; p++) {
            buf[0] = s[p] ? s[p] : ' ';
	    w = text_length(font, buf);
	    if (x+w > d->w)
	       break;

	    f = ((p == d->d2) && (d->flags & D_GOTFOCUS));
            textout(screen, font, buf, d->x+x+3, d->y+2, fg);
            if (f)
            {
                if (curs_blink)
                    vline(screen, d->x+x+3, d->y + 2, d->y + 1 + text_height(font), GUI_Shadow);
            }
	    x += w;
	 }

         box3d2(d->x, d->y, d->x + d->w, d->y + d->h);

	 break;

      case MSG_CLICK:
	 buf[1] = 0;
	 x = d->x;

	 for (p=0; p<l; p++) {
	    buf[0] = s[p];
	    x += text_length(font, buf);
	    if (x > mouse_x)
	       break;
	 }

	 d->d2 = MID(0, p, l);
	 show_mouse(NULL);
	 SEND_MESSAGE(d, MSG_DRAW, 0);
	 show_mouse(screen);
	 break;

      case MSG_WANTFOCUS:
      case MSG_LOSTFOCUS:
      case MSG_KEY:
         curs_blink=TRUE;
	 return D_WANTFOCUS;

      case MSG_IDLE:
         if (!(d->flags & D_GOTFOCUS))
             return D_O_K;
         if (curs_blink != last_curs_state)
         {
            show_mouse(NULL);
            fg = (d->flags & D_DISABLED) ? gui_mg_color : d->fg;
            buf[1] = 0;
            x = 0;
            text_mode(GUI_White);
            for (p=0; p<=l; p++)
            {
               buf[0] = s[p] ? s[p] : ' ';
               w = text_length(font, buf);
               if (x+w > d->w)
                  break;

               f = ((p == d->d2) && (d->flags & D_GOTFOCUS));
               textout(screen, font, buf, d->x+x+3, d->y+2, fg);
               if (f)
               {
                  if (curs_blink)
                      vline(screen, d->x+x+3, d->y + 2, d->y + 1 + text_height(font), GUI_Shadow);
               }
               x += w;
            }
            show_mouse(screen);
            last_curs_state = curs_blink;
         }
         break;

      case MSG_CHAR:
         curs_blink=TRUE;
         if ((c >> 8) == KEY_LEFT) {
	    if (d->d2 > 0)
	       d->d2--;
	 }
         else if ((c >> 8) == KEY_RIGHT) {
	    if (d->d2 < l)
	       d->d2++;
	 }
         else if ((c >> 8) == KEY_HOME) {
	    d->d2 = 0;
	 }
         else if ((c >> 8) == KEY_END) {
	    d->d2 = l;
	 }
         else if ((c >> 8) == KEY_DEL) {
	    if (d->d2 < l)
	       for (p=d->d2; s[p]; p++)
		  s[p] = s[p+1];
	 }
	 else if ((c >> 8) == KEY_BACKSPACE) {
	    if (d->d2 > 0) {
	       d->d2--;
	       for (p=d->d2; s[p]; p++)
		  s[p] = s[p+1];
	    } 
	 }
	 else {
	    c &= 0xff;
	    if ((c >= 32) && (c <= 126)) {
	       if (l < d->d1) {
		  while (l >= d->d2) {
		     s[l+1] = s[l];
		     l--;
		  }
		  s[d->d2] = c;
		  d->d2++;
	       }
	    }
	    else
	       return D_O_K;
	 }

	 /* if we changed something, better redraw... */ 
	 show_mouse(NULL);
	 SEND_MESSAGE(d, MSG_DRAW, 0);
	 show_mouse(screen);
	 return D_USED_CHAR;
   }

   return D_O_K;
}

/* draw_listbox:
      Helper function to draw a listbox object.
 */
static void draw_listbox(DIALOG *d)
{
   int height, listsize, i, len, bar, x, y, w;
   int fg_color, fg, bg;
   char *s;
   char store;
   
   (*(getfuncptr)d->dp)(-1, &listsize);
   height = (d->h-4) / text_height(font);
   bar = (listsize > height);
   w = (bar ? d->w-14 : d->w-3);
   fg_color = (d->flags & D_DISABLED) ? gui_mg_color : d->fg;

   /* draw frame */
   box3d2(d->x, d->y, d->x + d->w, d->y + d->h);
   if (bar) {
      vline(screen, d->x+d->w-12, d->y+2, d->y+d->h-2, fg_color);
      vline(screen, d->x+d->w-11, d->y+2, d->y+d->h-2, GUI_Window);

      if (d->flags & D_GOTFOCUS) {
         dotted_rect(d->x+2, d->y+2, d->x+d->w-13, d->y+d->h-2, fg_color, GUI_White);
      }
      else {
         rect(screen, d->x+2, d->y+2, d->x+d->w-13, d->y+d->h-2, GUI_White);
      }
   }
   else {
      if (d->flags & D_GOTFOCUS)
         dotted_rect(d->x+2, d->y+2, d->x+d->w-2, d->y+d->h-2, fg_color, GUI_White);
      else
         rect(screen, d->x+2, d->y+2, d->x+d->w-2, d->y+d->h-2, GUI_White);
   }

   /* draw box contents */
   for (i=0; i<height; i++) {
      if (d->d2+i < listsize) {
	 if (d->d2+i == d->d1) { 
            bg = GUI_Selectedbg;
            fg = GUI_Selectedfg;
	 }
	 else {
	    fg = fg_color;
            bg = GUI_White;
	 }
	 s = (*(getfuncptr)d->dp)(i+d->d2, NULL);
         x = d->x + 3;
         y = d->y + 3 + i*text_height(font);
	 text_mode(bg);
	 rectfill(screen, x, y, x+7, y+text_height(font)-1, bg); 
         x += 1;
	 len = strlen(s);
	 store = 0;
	 while (text_length(font, s) >= d->w - (bar ? 22 : 10)) {
	    s[len] = store;
	    len--;
	    store = s[len];
	    s[len] = 0;
	 }
	 textout(screen, font, s, x, y, fg); 
	 x += text_length(font, s);
	 s[len] = store;
	 if (x <= d->x+w) 
	    rectfill(screen, x, y, d->x+w, y+text_height(font)-1, bg);
      }
      else
         rectfill(screen, d->x+3,  d->y+2+i*text_height(font), 
                  d->x+w, d->y+1+(i+1)*text_height(font), GUI_White);
   }

   if (d->y+2+i*text_height(font) <= d->y+d->h-3)
      rectfill(screen, d->x+3, d->y+2+i*text_height(font), 
                                       d->x+w, d->y+d->h-3, GUI_White);

   /* draw scrollbar */
   if (bar) {
      i = ((d->h-4) * height + listsize/2) / listsize;
      x = d->x+d->w-10;
      y = d->y+2;

      if (d->d2 > 0) {
	 len = (((d->h-4) * d->d2) + listsize/2) / listsize;
	 rectfill(screen, x, y, x+8, y+len-1, d->bg);
	 y += len;
      }
      if (y+i < d->y+d->h-2) {
	 rectfill(screen, x, y, x+8, d->y+d->h-2, d->bg);
         box3d3(x, y, x+8, y+i);
	 y += i;
      }
      else {
         rectfill(screen, x, y, x+8, d->y+d->h-2, d->bg);
         box3d3(x, y, x+8, d->y+d->h-2);
      }
   }
}



/* scroll_listbox:
      Helper function to scroll through a listbox.
 */
static void scroll_listbox(DIALOG *d, int listsize)
{
   int height = (d->h-3) / text_height(font);

   if (!listsize) {
      d->d1 = d->d2 = 0;
      return;
   }

   /* check selected item */
   if (d->d1 < 0)
      d->d1 = 0;
   else
      if (d->d1 >= listsize)
	 d->d1 = listsize-1;

   /* check scroll position */
   while ((d->d2 > 0) && (d->d2 + height > listsize))
      d->d2--;

   if (d->d2 >= d->d1) {
      if (d->d1 < 0)
	 d->d2 = 0;
      else
	 d->d2 = d->d1;
   }
   else {
      while ((d->d2 + height-1) < d->d1)
	 d->d2++;
   }
}



/* _handle_listbox_click:
      Helper to process a click on a listbox, doing hit-testing and moving
      the selection.
 */
void _handle_listbox_click(DIALOG *d)
{
   int listsize, height;
   int i;

   (*(getfuncptr)d->dp)(-1,&listsize);
   if (!listsize)
      return;

   height = (d->h-3) / text_height(font);

   i = MID(0, ((mouse_y - d->y - 2) / text_height(font)), 
			   ((d->h-3) / text_height(font) - 1));
   i += d->d2;
   if (i < d->d2)
      i = d->d2;
   else {
      if (i > d->d2 + height-1)
	 i = d->d2 + height-1;
      if (i >= listsize)
	 i = listsize-1;
   }

   if (mouse_y <= d->y)
      i = MAX(i-1, 0);
   else if (mouse_y >= d->y+d->h)
      i = MIN(i+1, listsize-1);

   if (i != d->d1) {
      d->d1 = i;
      i = d->d2;
      scroll_listbox(d, listsize);
      show_mouse(NULL);
      SEND_MESSAGE(d, MSG_DRAW, 0);
      show_mouse(screen);
      if (i != d->d2)
	 rest(MID(10, text_height(font)*16-d->h, 100));
   }
}



/* handle_listbox_scroll_click:
      Helper to process a click on a listbox scrollbar.
 */
static void handle_listbox_scroll_click(DIALOG *d)
{
   int listsize, height, i, len, x, y;

   (*(getfuncptr)d->dp)(-1, &listsize);
   height = (d->h-3) / text_height(font);

   while (mouse_b) {
      i = ((d->h-4) * height + listsize/2) / listsize;
      len = ((d->h) * d->d2 + listsize/2) / listsize + 2;
      if ((mouse_y >= d->y+len) && (mouse_y <= d->y+len+i)) {
	 x = mouse_y - len + 2;
	 while (mouse_b) {
	    y = (listsize * (mouse_y - x) + d->h/2) / d->h;
	    if (y > listsize-height)
	       y = listsize-height;
	    if (y < 0)
	       y = 0;
	    if (y != d->d2) {
	       d->d2 = y;
	       show_mouse(NULL);
	       SEND_MESSAGE(d, MSG_DRAW, 0);
	       show_mouse(screen);
	    }
	 }
      }
      else {
	 if (mouse_y <= d->y+len)
	    y = d->d2 - height;
	 else
	    y = d->d2 + height;
	 if (y > listsize-height)
	    y = listsize-height;
	 if (y < 0)
	    y = 0;
	 if (y != d->d2) {
	    d->d2 = y;
	    show_mouse(NULL);
	    SEND_MESSAGE(d, MSG_DRAW, 0);
	    show_mouse(screen);
	    rest(200);
	 }
      }
   }
}



/* d_3dlist_proc:
      A listbox.  Sizing is still a bit buggy, so you may need to tweak the
      size of this control until it allows you to scroll from top to bottom
      properly, and doesn't have excess space at the bottom of the list.
 */
int d_3dlist_proc(int msg, DIALOG *d, int c)
{
   int listsize, i, bottom, height, bar;

   switch (msg) {

      case MSG_START:
	 (*(getfuncptr)d->dp)(-1, &listsize);
	 scroll_listbox(d, listsize);
	 break;

      case MSG_DRAW:
	 draw_listbox(d);
	 break;

      case MSG_CLICK:
	 (*(getfuncptr)d->dp)(-1, &listsize);
	 height = (d->h-3) / text_height(font);
	 bar = (listsize > height);
	 if ((!bar) || (mouse_x < d->x+d->w-10)) {
	    while (mouse_b)
	       _handle_listbox_click(d);
	 }
	 else {
	    handle_listbox_scroll_click(d);
	 }
	 break;

      case MSG_DCLICK:
	 (*(getfuncptr)d->dp)(-1, &listsize);
	 height = (d->h-3) / text_height(font);
	 bar = (listsize > height);
	 if ((!bar) || (mouse_x < d->x+d->w-10)) {
	    if (d->flags & D_EXIT) {
	       if (listsize) {
		  i = d->d1;
		  SEND_MESSAGE(d, MSG_CLICK, 0);
		  if (i == d->d1) 
		     return D_CLOSE;
	       }
	    }
	 }
	 break;

      case MSG_KEY:
	 (*(getfuncptr)d->dp)(-1, &listsize);
	 if ((listsize) && (d->flags & D_EXIT))
	    return D_CLOSE;
	 break;

      case MSG_WANTFOCUS:
	 return D_WANTFOCUS;

      case MSG_CHAR:
	 (*(getfuncptr)d->dp)(-1,&listsize);
	 if (listsize) {
	    c >>= 8;

	    bottom = d->d2 + (d->h-3)/text_height(font) - 1;
	    if (bottom >= listsize-1)
	       bottom = listsize-1;

            if (c == KEY_UP)
	       d->d1--;
            else if (c == KEY_DOWN)
	       d->d1++;
            else if (c == KEY_HOME)
	       d->d1 = 0;
            else if (c == KEY_END)
	       d->d1 = listsize-1;
            else if (c == KEY_PGUP) {
	       if (d->d1 > d->d2)
		  d->d1 = d->d2;
	       else
		  d->d1 -= (bottom - d->d2);
	    }
            else if (c == KEY_PGDN) {
	       if (d->d1 < bottom)
		  d->d1 = bottom;
	       else
		  d->d1 += (bottom - d->d2);
	    } 
	    else 
	       return D_O_K;

	    /* if we changed something, better redraw... */ 
	    scroll_listbox(d, listsize);
	    show_mouse(NULL);
	    SEND_MESSAGE(d, MSG_DRAW, 0);
	    show_mouse(screen); 
	    return D_USED_CHAR;
	 }
	 break;
   }

   return D_O_K;
}

// Draws 3d circles for the option boxes (poorly)
void circle3dmaker(int x, int y, bool checked, bool disabled)
{
    if (!disabled)
       rectfill(screen, x+1, y+1, x+7, y+7, GUI_White);

    hline(screen, x+2, y, x+6, GUI_Shadow);
    vline(screen, x, y+2, y+6, GUI_Shadow);
    putpixel(screen, x+1, y+1, GUI_Shadow);
    putpixel(screen, x+1, y+7, GUI_Shadow);

    hline(screen, x+2, y+8, x+6, GUI_Hilight);
    vline(screen, x+8, y+2, y+6, GUI_Hilight);
    putpixel(screen, x+7, y+7, GUI_Hilight);
    putpixel(screen, x+7, y+1, GUI_Hilight);

    hline(screen, x+2, y+1, x+6, GUI_Black);
    vline(screen, x+1, y+2, y+6, GUI_Black);
    putpixel(screen, x+2, y+2, GUI_Black);
    putpixel(screen, x+6, y+2, GUI_Black);

    hline(screen, x+2, y+7, x+6, GUI_Window);
    vline(screen, x+7, y+2, y+6, GUI_Window);
    putpixel(screen, x+6, y+6, GUI_Window);
    putpixel(screen, x+2, y+6, GUI_Window);
}

/* d_3dradio_proc:
      Draws 3d circular radio buttons.
 */
int d_3dradio_proc(int msg, DIALOG *d, int c)
{
   int x, y, center, r, ret, fg;

   switch(msg) {

      case MSG_DRAW:
         fg = (d->flags & D_DISABLED) ? gui_mg_color : d->fg;
	 text_mode(d->bg);
         rectfill(screen, d->x+d->h+(d->h>>1)-1, d->y+(d->h-(text_height(font)-gui_font_baseline))/2 - 1,
                  d->x+d->h+(d->h>>1)+gui_strlen((unsigned char*)d->dp)+1, d->y+text_height(font)+gui_font_baseline+1,
                  d->bg);
         if (d->flags & D_DISABLED)
         {
            text_mode(-1);
            gui_textout(screen, (unsigned char*)d->dp, d->x+d->h+(d->h>>1)+1, d->y+(d->h-(text_height(font)-gui_font_baseline))/2+1, GUI_Hilight, FALSE);
            gui_textout(screen, (unsigned char*)d->dp, d->x+d->h+(d->h>>1), d->y+(d->h-(text_height(font)-gui_font_baseline))/2, GUI_Shadow, FALSE);
            text_mode(d->bg); 
         }
         else
            gui_textout(screen, (unsigned char*)d->dp, d->x+d->h+(d->h>>1), d->y+(d->h-(text_height(font)-gui_font_baseline))/2, fg, FALSE);

	 x = d->x;
	 r = d->h/2;
	 center = x+r;
	 rectfill(screen, x+1, d->y+1, x+d->h-1, d->y+d->h-1, d->bg);

	 switch (d->d2) {

	    case 1:
	       rect(screen, x, d->y, x+d->h, d->y+d->h, fg);
	       if (d->flags & D_SELECTED)
		  rectfill(screen, x+r/2, d->y+r/2, x+d->h-r/2, d->y+d->h-r/2, fg);
	       break;

	    default:
               circle3dmaker(x, d->y, false, d->flags & D_DISABLED);
               if (d->flags & D_SELECTED)
                  rectfill(screen, d->x+3, d->y+3, d->x+5, d->y+5, fg);
	       break;
	 }

	 if (d->flags & D_GOTFOCUS)
            dotted_rect(d->x+d->h+(d->h>>1)-1, d->y+(d->h-(text_height(font)-gui_font_baseline))/2 - 1,
                        d->x+d->h+(d->h>>1)+gui_strlen((unsigned char*)d->dp)+1, d->y+text_height(font)+gui_font_baseline+1,
                        fg, d->bg);

	 return D_O_K;

      case MSG_KEY:
      case MSG_CLICK:
	 if (d->flags & D_SELECTED)
	    return D_O_K;
      break;

      case MSG_RADIO:
	 if ((c == d->d1) && (d->flags & D_SELECTED)) {
	    d->flags &= ~D_SELECTED;
	    show_mouse(NULL);
	    SEND_MESSAGE(d, MSG_DRAW, 0);
	    show_mouse(screen);
	 }
	 break;
   }

   ret = d_button_proc(msg, d, 0);

   if (((msg==MSG_KEY) || (msg==MSG_CLICK)) &&
       (d->flags & D_SELECTED) && (!(d->flags & D_CLOSE))) {
      d->flags &= ~D_SELECTED;
      broadcast_dialog_message(MSG_RADIO, d->d1);
      d->flags |= D_SELECTED;
   }

   return ret;
}

/* d_speeddial:
      Control for displaying an adjustable bar for regulating speeds.
 */
int d_speeddial(int msg, DIALOG *d, int c)
{
   char tmpstr[200];
   switch (msg)
   {
      case MSG_DRAW:
               if (d->d1 < (d->d2 & 0xFFFF))
                  d->d1 = (d->d2 & 0xFFFF);
               else if (d->d1 > (d->d2 >> 16))
                  d->d1 = (d->d2 >> 16);
               //rect(screen, d->x, d->y, d->x + d->w, d->y + d->h, GUI_Black);
               rectfill(screen, d->x+2, d->y+2, d->x + d->w - 2, d->y + d->h - 2, GUI_Shadow);
               if (d->d1 == 100)
                   rectfill(screen, d->x + 2, d->y + 2, d->x + (d->w>>1) - 1, d->y + d->h - 2, GUI_Selectedbg);
               else if (d->d1 < 100)
                   rectfill(screen, d->x + 2, d->y + 2, d->x + (((d->w>>1))*(d->d1 - (d->d2&0xFFFF)) / (100-(d->d2&0xFFFF))), d->y + d->h - 2, GUI_Selectedbg);
               else if (d->d1 > 100)
                   rectfill(screen, d->x + 2, d->y + 2, d->x + (d->w>>1) + (((d->w>>1) - 1)*(d->d1 - 100) / ((d->d2>>16)-100)), d->y + d->h - 2, GUI_Selectedbg);
               text_mode(-1);
               sprintf(tmpstr, "%i%%", d->d1);
               textout_centre(screen, &s9gui_smallfont, tmpstr, d->x + (d->w>>1), d->y + 3, GUI_White);
               box3d5(d->x, d->y, d->x + d->w, d->y + d->h);
               rect(screen, d->x + 1, d->y + 1, d->x + d->w - 1, d->y + d->h - 1, GUI_Window);
               putpixel(screen, d->x + (d->w>>1), d->y + d->h, GUI_Black);
               putpixel(screen, d->x + (d->w>>1), d->y, GUI_Black);
               putpixel(screen, d->x + (d->w>>1), d->y + d->h - 1, GUI_Black);
               putpixel(screen, d->x + (d->w>>1), d->y + 1, GUI_Black);
               if (d->flags & D_GOTFOCUS)
                   dotted_rect(d->x+1, d->y+1, d->x+d->w-1, d->y+d->h-1,
                               d->fg, d->bg);
               break;

      case MSG_WANTFOCUS:
      case MSG_LOSTFOCUS:
      case MSG_KEY:
               return D_WANTFOCUS;


      case MSG_CHAR:
               c >>= 8;

               if (c == KEY_UP)
                  d->d1++;
               else if (c == KEY_DOWN)
                  d->d1--;
               else if (c == KEY_HOME)
                  d->d1 = 100;
               else if (c == KEY_END)
                  d->d1 = 100;
               else if (c == KEY_PGUP)
                  d->d1 += 20;
               else if (c == KEY_PGDN)
                  d->d1 -= 20;
               else 
                  return D_O_K;

               if (d->d1 < (d->d2 & 0xFFFF))
                  d->d1 = (d->d2 & 0xFFFF);
               else if (d->d1 > (d->d2 >> 16))
                  d->d1 = (d->d2 >> 16);

               show_mouse(NULL);
               SEND_MESSAGE(d, MSG_DRAW, 0);
               show_mouse(screen); 
               return D_USED_CHAR;

      case MSG_CLICK:
               int mx, mx1;
               mx1 = !mx;
               while (mouse_b != 0)
               {
                 mx = mouse_x - d->x;
                 if ((mx != mx1) && (mx > 0) && (mx < d->w))
                 {
                     if (mx < (d->w>>1))
                     {
                         d->d1 = mx * (100 - d->d2&0xFFFF) / (d->w>>1) + (d->d2 & 0xFFFF);
                         show_mouse(NULL);
                         SEND_MESSAGE(d, MSG_DRAW, 0);
                         show_mouse(screen);
                     }
                     else if (mx > (d->w>>1))
                     {
                         d->d1 = (mx - (d->w>>1)) * ((d->d2>>16)-100) / (d->w>>1) + 100;
                         show_mouse(NULL);
                         SEND_MESSAGE(d, MSG_DRAW, 0);
                         show_mouse(screen);
                     }
                     else
                     {
                         d->d1 = 100;
                         show_mouse(NULL);
                         SEND_MESSAGE(d, MSG_DRAW, 0);
                         show_mouse(screen);
                     }
                 }
                 mx1 = mx;
               }
               break;
   }
   return D_O_K;
}

/* d_fptext_proc:
    Simple dialog procedure: draws the text string which is pointed to by dp
    in a fixed-width font.
 */
int d_fptext_proc(int msg, DIALOG *d, int c)
{
   FONT * oldfont;
   if (msg==MSG_DRAW) {
      int fg = (d->flags & D_DISABLED) ? gui_mg_color : d->fg;
      oldfont = font;
      font = &s9gui_smallfont;
      text_mode(d->bg);
      rectfill(screen, d->x, d->y, d->x+d->w, d->y+text_height(font), d->bg);
      gui_textout(screen, (unsigned char*)d->dp, d->x, d->y, fg, FALSE);
      font = oldfont;
   }

   return D_O_K;
}

/* d_keygrab_proc:
      Keygrabber button.  When clicked or activated (by pressing enter), stores
      the next key scancode in d1, and displays the key name on the button face.
      Does not accept numeric row or function keys as valid.  Esc terminates
      reprogramming.
 */
int d_keygrab_proc(int msg, DIALOG *d, int c)
{
   // Incomplete.
}

extern BITMAP *_mouse_screen;

typedef struct MENU_INFO            /* information about a popup menu */
{
   MENU *menu;                      /* the menu itself */
   struct MENU_INFO *parent;        /* the parent menu, or NULL for root */
   int bar;                         /* set if it is a top level menu bar */
   int size;                        /* number of items in the menu */
   int sel;                         /* selected item */
   int x, y, w, h;                  /* screen position of the menu */
   int (*proc)();                   /* callback function */
   BITMAP *saved;                   /* saved what was underneath it */
} MENU_INFO;

/* get_menu_pos:
      Calculates the coordinates of an object within a top level menu bar.
 */
static void get_menu_pos(MENU_INFO *m, int c, int *x, int *y, int *w)
{
   int c2;

   if (m->bar) {
      *x = m->x+1;

      for (c2=0; c2<c; c2++)
         *x += gui_strlen((unsigned char*)m->menu[c2].text) + 16;

      *y = m->y+1;
      *w = gui_strlen((unsigned char*)m->menu[c].text) + 16;
   }
   else {
      *x = m->x+2;
      *y = m->y+c*(text_height(font)+4)+4;
      *w = m->w-4;
   }
}

/* draw_menu_item:
      Draws an item from a popup menu onto the screen.
 */
static void draw_menu_item(MENU_INFO *m, int c)
{
   int fg, bg;
   int i, x, y, w;
   char buf[80], *tok;

   fg = gui_fg_color;
   bg = gui_bg_color;

   get_menu_pos(m, c, &x, &y, &w);

   rectfill(screen, x, y, x+w-1, y+text_height(font)+3, bg);
   text_mode(bg);
   if (m->menu[c].text[0]) {

      for (i=0; (m->menu[c].text[i]) && (m->menu[c].text[i] != '\t'); i++)
	 buf[i] = m->menu[c].text[i];
      buf[i] = 0;

      if (m->menu[c].flags & D_DISABLED) {
          text_mode(-1);
          gui_textout(screen, (unsigned char*)m->menu[c].text, x+8, y+2, GUI_White, FALSE);
          gui_textout(screen, (unsigned char*)m->menu[c].text, x+9, y+3, GUI_Shadow, FALSE);
      }
      else
          gui_textout(screen, (unsigned char*)m->menu[c].text, x+8, y+2, fg, FALSE);

      if (m->menu[c].text[i] == '\t') {
	 tok = m->menu[c].text+i+1;
         gui_textout(screen, (unsigned char*)tok, x+w-gui_strlen((unsigned char*)tok)-8, y+2, fg, FALSE);
      }
   }
   else
      hline(screen, x, y+text_height(font)/2+2, x+w, fg);

   if (m->menu[c].child)
       triangle(screen, x+w-(text_height(font)/2)-5, y+3,
            x+w-(text_height(font)/2)-5, y+text_height(font)+1,
            x+w-6, y+(text_height(font)/2)+2, gui_fg_color);

   if (c == m->sel)
      box3d4( x+2, y, x+w-3, y+text_height(font)+3);
}

/* draw_menu:
      Draws a popup menu onto the screen.
 */
static void draw_menu(MENU_INFO *m)
{
   int c;

   rectfill(screen, m->x, m->y, m->x+m->w-1, m->y+m->h-1, gui_bg_color);
   box3d(m->x, m->y, m->x+m->w-1, m->y+m->h-1);

   for (c=0; m->menu[c].text; c++)
      draw_menu_item(m, c);
}

/* menu_mouse_object:
      Returns the index of the object the mouse is currently on top of.
 */
static int menu_mouse_object(MENU_INFO *m)
{
   int c;
   int x, y, w;

   for (c=0; c<m->size; c++) {
      get_menu_pos(m, c, &x, &y, &w);

      if ((mouse_x >= x) && (mouse_x < x+w) &&
	  (mouse_y >= y) && (mouse_y < y+(text_height(font)+4)))
	 return (m->menu[c].text[0]) ? c : -1;
   }

   return -1;
}

/* mouse_in_parent_menu:
      Recursively checks if the mouse is inside a menu or any of its parents.
 */
static int mouse_in_parent_menu(MENU_INFO *m) 
{
   int c;

   if (!m)
      return FALSE;

   c = menu_mouse_object(m);
   if ((c >= 0) && (c != m->sel))
      return TRUE;

   return mouse_in_parent_menu(m->parent);
}

/* fill_menu_info:
      Fills a menu info structure when initialising a menu.
 */
static void fill_menu_info(MENU_INFO *m, MENU *menu, MENU_INFO *parent, int bar, int x, int y, int minw, int minh)
{
   int c, i;
   int extra = 0;
   char buf[80], *tok;

   m->menu = menu;
   m->parent = parent;
   m->bar = bar;
   m->x = x;
   m->y = y;
   m->w = 2;
   m->h = (m->bar) ? (text_height(font)+6) : 8;
   m->proc = NULL;
   m->sel = -1;

   /* calculate size of the menu */
   for (m->size=0; m->menu[m->size].text; m->size++) {
      for (i=0; (m->menu[m->size].text[i]) && (m->menu[m->size].text[i] != '\t'); i++)
	 buf[i] = m->menu[m->size].text[i];
      buf[i] = 0;

      c = gui_strlen((unsigned char*)buf)+6;

      if (m->bar) {
	 m->w += c+16;
      }
      else {
	 m->h += text_height(font)+4;
	 m->w = MAX(m->w, c+16);
      }

      if (m->menu[m->size].text[i] == '\t') {
	 tok = m->menu[m->size].text+i+1;
         c = gui_strlen((unsigned char*)tok);
	 extra = MAX(extra, c);
      }
   }
   if (extra)
      m->w += extra+16;

   m->w = MAX(m->w, minw);
   m->h = MAX(m->h, minh);
}

/* menu_key_shortcut:
      Returns true if c is indicated as a keyboard shortcut by a '&' character
      in the specified string.
 */
static int menu_key_shortcut(int c, char *s)
{
   while (*s) {
      if (*s == '&') {
	 s++;
	 if ((*s != '&') && (tolower(*s) == tolower(c & 0xff)))
	    return TRUE;
      }
      s++;
   }

   return FALSE;
}

/* menu_alt_key:
      Searches a menu for keyboard shortcuts, for the alt+letter to bring
      up a menu.
 */
int menu_alt_key(int k, MENU *m)
{
   char *s;
   int c;

   if (k & 0xff)
      return 0;

   k = key_ascii_table[k>>8];

   for (c=0; m[c].text; c++) {
      s = m[c].text;
      while (*s) {
	 if (*s == '&') {
	    s++;
	    if ((*s != '&') && (tolower(*s) == tolower(k)))
	       return k;
	 }
	 s++;
      }
   }

   return 0;
}

/* _3d_do_menu:
      The core menu control function, called by do_menu() and d_menu_proc().
 */
int _3d_do_menu(MENU *menu, MENU_INFO *parent, int bar, int x, int y, int repos, int *dret, int minw, int minh)
{
   MENU_INFO m;
   MENU_INFO *i;
   int c, c2;
   int ret = -1;
   int mouse_on = mouse_b;
   int old_sel;
   int mouse_sel;
   int _x, _y;
   int redraw = TRUE;
   int mouse_visible = (_mouse_screen == screen);

   show_mouse(NULL);

   fill_menu_info(&m, menu, parent, bar, x, y, minw, minh);

   if (repos) {
      m.x = MID(0, m.x, SCREEN_W-m.w-1);
      m.y = MID(0, m.y, SCREEN_H-m.h-1);
   }

   /* save screen under the menu */
   m.saved = create_bitmap(m.w+1, m.h+1); 

   if (m.saved)
      blit(screen, m.saved, m.x, m.y, 0, 0, m.w+1, m.h+1);
   else
      errno = ENOMEM;

   m.sel = mouse_sel = menu_mouse_object(&m);
   if ((m.sel < 0) && (!mouse_b) && (!bar))
      m.sel = 0;

   show_mouse(screen);

   do {
      old_sel = m.sel;

      c = menu_mouse_object(&m);
      if ((mouse_b) || (c != mouse_sel))
	 m.sel = mouse_sel = c;

      if (mouse_b) {                                  /* if button pressed */
	 if ((mouse_x < m.x) || (mouse_x > m.x+m.w) ||
	     (mouse_y < m.y) || (mouse_y > m.y+m.h)) {
	    if (!mouse_on)                            /* dismiss menu? */
	       break;

	    if (mouse_in_parent_menu(m.parent))       /* back to parent? */
	       break;
	 }

	 if ((m.sel >= 0) && (m.menu[m.sel].child))   /* bring up child? */
	    ret = m.sel;

	 mouse_on = TRUE;
	 clear_keybuf();
      }
      else {                                          /* button not pressed */
	 if (mouse_on)                                /* selected an item? */
	    ret = m.sel;

	 mouse_on = FALSE;

	 if (keypressed()) {                          /* keyboard input */
	    c = readkey();

	    if ((c & 0xff) == 27) {
	       ret = -1;
	       goto getout;
	    }

	    switch (c >> 8) {

	       case KEY_LEFT:
		  if (m.parent) {
		     if (m.parent->bar) {
			simulate_keypress(KEY_LEFT<<8);
			simulate_keypress(KEY_DOWN<<8);
		     }
		     ret = -1;
		     goto getout;
		  }
		  /* fall through */

	       case KEY_UP:
		  if ((((c >> 8) == KEY_LEFT) && (m.bar)) ||
		      (((c >> 8) == KEY_UP) && (!m.bar))) {
		     c = m.sel;
		     do {
			c--;
			if (c < 0)
			   c = m.size - 1;
		     } while ((!(m.menu[c].text[0])) && (c != m.sel));
		     m.sel = c;
		  }
		  break;

	       case KEY_RIGHT:
		  if (((m.sel < 0) || (!m.menu[m.sel].child)) &&
		      (m.parent) && (m.parent->bar)) {
		     simulate_keypress(KEY_RIGHT<<8);
		     simulate_keypress(KEY_DOWN<<8);
		     ret = -1;
		     goto getout;
		  }
		  /* fall through */

	       case KEY_DOWN:
		  if ((m.sel >= 0) && (m.menu[m.sel].child) &&
		      ((((c >> 8) == KEY_RIGHT) && (!m.bar)) ||
		       (((c >> 8) == KEY_DOWN) && (m.bar)))) {
		     ret = m.sel;
		  }
		  else if ((((c >> 8) == KEY_RIGHT) && (m.bar)) ||
			   (((c >> 8) == KEY_DOWN) && (!m.bar))) {
		     c = m.sel;
		     do {
			c++;
			if (c >= m.size)
			   c = 0;
		     } while ((!(m.menu[c].text[0])) && (c != m.sel));
		     m.sel = c;
		  }
		  break;

	       case KEY_SPACE:
	       case KEY_ENTER:
		  if (m.sel >= 0)
		     ret = m.sel;
		  break;

	       default:
		  if ((!m.parent) && ((c & 0xff) == 0))
		     c = menu_alt_key(c, m.menu);
		  for (c2=0; m.menu[c2].text; c2++) {
		     if (menu_key_shortcut(c, m.menu[c2].text)) {
			ret = m.sel = c2;
			break;
		     }
		  }
		  if (m.parent) {
		     i = m.parent;
		     for (c2=0; i->parent; c2++)
			i = i->parent;
		     c = menu_alt_key(c, i->menu);
		     if (c) {
			while (c2-- > 0)
			   simulate_keypress(27);
			simulate_keypress(c);
			ret = -1;
			goto getout;
		     }
		  }
		  break;
	    }
	 }
      }

      if ((redraw) || (m.sel != old_sel)) {           /* selection changed? */
	 show_mouse(NULL);

	 if (redraw) {
	    draw_menu(&m);
	    redraw = FALSE;
	 }
	 else {
	    if (old_sel >= 0)
	       draw_menu_item(&m, old_sel);

	    if (m.sel >= 0)
	       draw_menu_item(&m, m.sel);
	 }

	 show_mouse(screen);
      }

      if ((ret >= 0) && (m.menu[ret].flags & D_DISABLED))
	 ret = -1;

      if (ret >= 0) {                                 /* child menu? */
	 if (m.menu[ret].child) {
	    if (m.bar) {
	       get_menu_pos(&m, ret, &_x, &_y, &c);
	       _x += 6;
	       _y += text_height(font)+7;
	    }
	    else {
	       _x = m.x+m.w*2/3;
	       _y = m.y + (text_height(font)+4)*ret + text_height(font)/4+2;
	    }
            c = _3d_do_menu(m.menu[ret].child, &m, FALSE, _x, _y, TRUE, NULL, 0, 0);
	    if (c < 0) {
	       ret = -1;
	       mouse_on = FALSE;
	       mouse_sel = menu_mouse_object(&m);
	    }
	 }
      }

      if ((m.bar) && (!mouse_b) && (!keypressed()) &&
	  ((mouse_x < m.x) || (mouse_x > m.x+m.w) ||
	   (mouse_y < m.y) || (mouse_y > m.y+m.h)))
	 break;

   } while (ret < 0);

   getout:

   if (dret)
      *dret = 0;

   /* callback function? */
   if ((!m.proc) && (ret >= 0)) {
      active_menu = &m.menu[ret];
      m.proc = active_menu->proc;
   }

   if (ret >= 0) {
      if (parent)
	 parent->proc = m.proc;
      else  {
	 if (m.proc) {
	    c = m.proc();
	    if (dret)
	       *dret = c;
	 }
      }
   }

   /* restore screen */
   if (m.saved) {
      show_mouse(NULL);
      blit(m.saved, screen, 0, 0, m.x, m.y, m.w+1, m.h+1);
      destroy_bitmap(m.saved);
   }

   show_mouse(mouse_visible ? screen : NULL);

   return ret;
}

/* do_menu3d:
      Displays and animates a popup menu at the specified screen position,
      returning the index of the item that was selected, or -1 if it was
      dismissed. If the menu crosses the edge of the screen it will be moved.
 */
int do_menu3d(MENU *menu, int x, int y)
{
   int ret = _3d_do_menu(menu, NULL, FALSE, x, y, TRUE, NULL, 0, 0);

   do {
   } while (mouse_b);

   return ret;
}
