#include <stdio.h>
#include <ctype.h>
#include "g_local.h"

dm2_t		dm2;
cvar_t		*demoname;
cvar_t		*gamedir;
cvar_t		*timescale;
qboolean	stopdemo;

extern char *ED_NewString (char *string);

FILE *DemoOpen (dm2_t *p, char *filename)
{
	memset(p, 0, sizeof(dm2_t));
	p->file = fopen(filename, "rb");
	return p->file;
}

void DemoClose(dm2_t *p)
{
	if (!p->file)
		return;

	FreeBlock(&p->curblock);

//	for (i = 0; i < MAX_CONFIGSTRINGS; i++)
//	{
//		if (p->configstrings[i])
//		{
//			gi.TagFree(p->configstrings[i]);
//			p->configstrings[i] = NULL;
//		}
//	}

//	memset(p, 0, sizeof(dm2_t));
	p->file = NULL;
}

int FindInBlock (block_t *block, int from, int id)
{
	int a;
	
	for (a = from+1; a < block->num_messages; a++)
		if (block->message[a].id == id)
			return a;

	return -1;
}


void FreeBlock (block_t *block)
{
	int i;

	for (i = 0; i < block->num_messages; i++)
	{
		if (block->message[i].data)
			free(block->message[i].data);
//			gi.TagFree(block->message[i].data);
	}
	if (block->buffer)
		free(block->buffer);
//		gi.TagFree(block->buffer);

	memset(block, 0, sizeof(block_t));
}

void GetBaseline (dm2_t *p, int num)
{
	edict_t *ent;

	ent = EDICT(p, num);
	if (!ent || !ent->inuse)
		return;

	// assume spawnbaselines won't give 0 for a modelindex
	if (p->baselines[num].modelindex)
		ent->dm2 = p->baselines[num];
	else
		memset (&ent->dm2, 0, sizeof(entity_state_t));
	ent->baseline = false;
}

edict_t *ValidateEntity (dm2_t *p, int num)
{
	edict_t *ent;

	ent = EDICT(p, num);
	// spawn new ent if needed
	if (!ent || !ent->inuse)
	{
		EDICT(p, num) = ent = G_Spawn();
		ent->baseline = true;
	}
	return ent;
}

qboolean ProcessBlock (dm2_t *p, block_t *block)
{
	int msgnum;
	message_t *msg;
	
	WriteLog("Processing\n");

	for (msgnum = 0; msgnum < block->num_messages; msgnum++)
	{
		msg = &block->message[msgnum];
		switch (msg->id)
		{
		case svc_print:
			{
				print_t *data = (print_t *)msg->data;
				gi.bprintf (data->level, "%s", data->string); // newline is part of string
			}
			break;
		case svc_muzzleflash:
			{
				muzzleflash_t *data = (muzzleflash_t *)msg->data;
				edict_t *ent;
				
				ent = ValidateEntity (p, data->entity);
				gi.WriteByte(svc_muzzleflash);
				gi.WriteShort(ent-g_edicts);
				gi.WriteByte(data->value);
				gi.multicast(ent->s.origin, MULTICAST_PVS);
			}
			break;
		case svc_muzzleflash2:
			{
				muzzleflash_t *data = (muzzleflash_t *)msg->data;
				edict_t *ent;

				ent = ValidateEntity(p, data->entity);
				gi.WriteByte(svc_muzzleflash2);
				gi.WriteShort(ent-g_edicts);
				gi.WriteByte(data->value);
				gi.multicast(ent->s.origin, MULTICAST_PVS);
			}
			break;
		case svc_temp_entity:
			{
				temp_entity_t *data = (temp_entity_t *)msg->data;
				switch (data->entitytype)
				{
				// point entity
				case TE_EXPLOSION1:
				case TE_EXPLOSION2:
				case TE_ROCKET_EXPLOSION:
				case TE_GRENADE_EXPLOSION:
				case TE_ROCKET_EXPLOSION_WATER:
				case TE_GRENADE_EXPLOSION_WATER:
				case TE_BFG_EXPLOSION:
				case TE_BFG_BIGEXPLOSION:
				case TE_BOSSTPORT:
				case 28:
					gi.WriteByte(svc_temp_entity);
					gi.WriteByte(data->entitytype);
					gi.WritePosition(data->origin);
					gi.multicast(data->origin, MULTICAST_PVS);
					break;
				// impact entity
				case TE_GUNSHOT:
				case TE_BLOOD:
				case TE_BLASTER:
				case TE_SHOTGUN:
				case TE_SPARKS:
				case TE_SCREEN_SPARKS:
				case TE_SHIELD_SPARKS:
				case TE_BULLET_SPARKS:
				case TE_GREENBLOOD:
					gi.WriteByte(svc_temp_entity);
					gi.WriteByte(data->entitytype);
					gi.WritePosition(data->origin);
					gi.WriteDir(data->movedir);
					gi.multicast(data->origin, MULTICAST_PVS);
					break;
				// line entity
				case TE_RAILTRAIL:
				case TE_BUBBLETRAIL:
				case TE_BFG_LASER:
				case TE_PLASMATRAIL:
					gi.WriteByte(svc_temp_entity);
					gi.WriteByte(data->entitytype);
					gi.WritePosition(data->origin);
					gi.WritePosition(data->endpos);
					gi.multicast(data->origin, MULTICAST_PVS);
					break;
				// special entity
				case TE_SPLASH:
					gi.WriteByte(svc_temp_entity);
					gi.WriteByte(data->entitytype);
					gi.WriteByte(data->count);
					gi.WritePosition(data->origin);
					gi.WriteDir(data->movedir);
					gi.WriteByte(data->sound);
					gi.multicast(data->origin, MULTICAST_PVS);
					break;
				case TE_LASER_SPARKS:
				case TE_WELDING_SPARKS:
				case 29:
					gi.WriteByte(svc_temp_entity);
					gi.WriteByte(data->entitytype);
					gi.WriteByte(data->count);
					gi.WritePosition(data->origin);
					gi.WriteDir(data->movedir);
					gi.WriteByte(data->color);
					gi.multicast(data->origin, MULTICAST_PVS);
					break;
				case TE_PARASITE_ATTACK:
				case TE_MEDIC_CABLE_ATTACK:
					gi.WriteByte(svc_temp_entity);
					gi.WriteByte(data->entitytype);
					gi.WriteShort(ValidateEntity(p, data->entity)-g_edicts);
					gi.WritePosition(data->origin);
					gi.WritePosition(data->endpos);
					gi.multicast(data->origin, MULTICAST_PVS);
					break;
				case TE_GRAPPLE_CABLE:
					gi.WriteByte(svc_temp_entity);
					gi.WriteByte(data->entitytype);
					gi.WriteShort(ValidateEntity(p, data->entity)-g_edicts);
					gi.WritePosition(data->origin);
					gi.WritePosition(data->endpos);
					gi.WritePosition(data->offset);
					gi.multicast(data->origin, MULTICAST_PVS);
					break;
				default:
					break;
				}
			}
			break;
		case svc_layout:
			{
				layout_t *data = (layout_t *)msg->data;
				char string[MAX_STRING_LENGTH];
				char search[32], replace[32];
				char *pstr;
				int i, x, y, num;
				edict_t *other;

				// layouts can make references to configstrings
				// so we must intercept these and use the right
				// one instead
				// FIXME: move to dm2::layout
				strcpy(string, data->text);
				pstr = string;
				while (1)
				{
					if ((pstr = strstr(pstr, "client ")) == NULL)
						break;
					sscanf(pstr, "client %i %i %i", &x, &y, &num);

					sprintf(search, "client %i %i %i ", x, y, num);
					sprintf(replace, "client %i %i %i ", x, y, PLAYER(p, num));
					strnsub(pstr, search, replace, strlen(pstr)+2);
					pstr++;
				}

				for (i = 0; i < game.maxclients; i++)
				{
					other = g_edicts + i + 1;
					if (!other->inuse)
						continue;
					if (!(other->client->pers.dm2flags & DM2_SHOWSTATUS))
						continue;

					gi.WriteByte(svc_layout);
					gi.WriteString(string);
					gi.unicast(other, false);
				}
			}
			break;
		case svc_inventory:
			{
				inventory_t *data = (inventory_t *)msg->data;
				int i, j;
				edict_t *other;

				for (i = 0; i < game.maxclients; i++)
				{
					other = g_edicts + i + 1;
					if (!other->inuse)
						continue;
					if (!(other->client->pers.dm2flags & DM2_SHOWSTATUS))
						continue;

					gi.WriteByte(svc_inventory);
					for (j = 0; j < MAX_ITEMS; j++)
						gi.WriteShort(data->inventory[j]);
					gi.unicast(other, false);
				}
			}
			break;
		case svc_sound:
			{
				sound_t *data = (sound_t *)msg->data;
				edict_t *ent;
				long mask;
				short mix=0;
				
				mask = data->mask;
				ent = EDICT(p, data->entity);
				if (ent && mask & 0x08)
				{
					if (mask & 0x04)
						gi.positioned_sound(data->origin, ent, data->channel, SOUND(p, data->soundnum), data->vol, data->attenuation, data->timeofs);
					else
						gi.sound(ent, data->channel, SOUND(p, data->soundnum), data->vol, data->attenuation, data->timeofs);
				}
				else
				{
					if (mask & 0x08)
					{
//						gi.dprintf("Warning: Invalid entity %d %d\n", data->entity, data->soundnum);
//						mask &= ~0x08;
						ent = ValidateEntity (p, data->entity);
					}

					gi.WriteByte(svc_sound);
					gi.WriteByte(mask);
					gi.WriteByte(SOUND(p, data->soundnum));
					if (mask & 0x01)
						gi.WriteByte(data->vol*255);
					if (mask & 0x02)
							gi.WriteByte(data->attenuation*64);
					if (mask & 0x10)
						gi.WriteByte(data->timeofs*1000);
					if (mask & 0x08)
					{
						mix = ((ent - g_edicts) << 3) | (data->channel & 0x07);
						gi.WriteShort(mix);
					}
					if (mask & 0x04)
						gi.WritePosition(data->origin);

					if (mask & 0x04)
						gi.multicast (data->origin, MULTICAST_PHS);
					// some sounds have unknown origins (why?)
					else if (!VectorCompare(ent->s.origin, vec3_origin))
						gi.multicast (ent->s.origin, MULTICAST_PHS);
					else
					{
	//					gi.dprintf("\nWarning: Unknown multicast\n\n");
						gi.multicast (p->vieworigin, MULTICAST_PHS);
					}
				}
			}
			break;
		case svc_stufftext:
			{
				stufftext_t *data = (stufftext_t *)msg->data;
				int		j, ofs=0;
				float	num;
				char	string[MAX_STRING_LENGTH], dm2name[MAX_STRING_LENGTH];
				char	search[MAX_STRING_LENGTH], replace[MAX_STRING_LENGTH];
				char	*mapname, *temp;

				// ignore pre-level commands
				if (!strcmp(data->text, "precache\n"))
					break;

				// FIXME: handle compound commands
				if (!strncmp(data->text, "map ", 4))
					ofs = 4;
				else if (!strncmp(data->text, "gamemap ", 8))
					ofs = 8;
				else if (!strncmp(data->text, "demomap ", 8))
					ofs = 8;
				
				if (ofs)
				{
					strcpy(string, data->text);

					temp = string + ofs;
					if (strstr(temp, ".dm2"))
					{
						strcpy(dm2name, temp);
						if ((j = strcspn(dm2name, ";\n")) < strlen(dm2name))
							dm2name[j] = 0;
						gi.cvar_set("demo", dm2name);
						mapname = DM2Map(dm2name);
						strcpy(search, data->text);
						search[ofs+strlen(dm2name)] = 0;
						sprintf(replace, "gamemap %s", mapname);
						strnsub(string, search, replace, MAX_STRING_LENGTH);
						WriteLog("CommandString: \"%s\"\n", string);
					}
					
					DemoClose(p);
					WriteLog("Closed dm2 to change maps\n");
					gi.AddCommandString(string);
					return true;
				}
				
				if (!strncmp(data->text, "timescale ", 10))
					ofs = 10;
				else if (!strncmp(data->text, "set timescale ", 10))
					ofs = 14;
				else
					ofs = 0;

				if (ofs)
				{
					// hack hack hack!
					sscanf(data->text+ofs, "%f", &num);
//					gi.cvar_set("timescale", va("%f", num));
					timescale->value = num;
					break;
				}

				gi.WriteByte(svc_stufftext);
				gi.WriteString(data->text);
				gi.multicast(vec3_origin, MULTICAST_ALL);
			}
			break;
		case svc_serverdata:
			{
				serverdata_t *data = (serverdata_t *)msg->data;

				p->version = data->serverversion;
				p->viewent = data->player + 1;
				p->isdemo = data->isdemo;
			}
			break;
		case svc_configstring:
			{
				configstring_t *data = (configstring_t *)msg->data;
				int i, num;
				char *temp;

				strncpy(p->configstrings[data->index], data->string, MAX_QPATH);

				// this may be overwritten later
				p->configstring[data->index] = data->index;

				if (data->index == CS_MAXCLIENTS)
					p->maxclients = atoi(data->string);
				else if (data->index == CS_NAME)
					strncpy(p->mapname, data->string, sizeof(p->mapname));
				else if (data->index >= CS_MODELS && data->index < CS_SOUNDS)				
					p->configstring[data->index] = CS_MODELS+dm2_modelindex(data->string);
				else if (data->index >= CS_SOUNDS && data->index < CS_IMAGES)
					p->configstring[data->index] = CS_SOUNDS+dm2_soundindex(data->string);
				else if (data->index >= CS_IMAGES && data->index < CS_LIGHTS)
					p->configstring[data->index] = CS_IMAGES+dm2_imageindex(data->string);
				else if (data->index >= CS_STATUSBAR && data->index < CS_MAXCLIENTS)
				{
					temp = data->string;
					while(temp = strstr(temp, "pic "))
					{
						// watch out for statpic command,
						// so make sure there is whitespace before
						if (temp == data->string || *(temp-1) == ' ' || 
							*(temp-1) == '\t')
						{
							sscanf(temp, "pic %d ", &num);
							p->statusbar[num] = STAT_IMAGE;
						}
						temp++;
					}
					temp = data->string;					
					while(temp = strstr(temp, "stat_string "))
					{
						sscanf(temp, "stat_string %d ", &num);
						p->statusbar[num] = STAT_STRING;
						temp++;
					}
				}
				else if (data->index >= CS_ITEMS && data->index < CS_PLAYERSKINS)
				{
					for (i = CS_ITEMS+1; i < CS_ITEMS+MAX_ITEMS; i++)
					{
						if (!level.configstrings[i][0])
						{
							multicast_configstring(i, data->string);
							p->configstring[data->index] = i;
							break;
						}
					}
				}
				else if (data->index >= CS_PLAYERSKINS && data->index < CS_MISC)
				{
					// leave room for demo configstrings
					for (i = CS_PLAYERSKINS+game.maxclients; i < CS_PLAYERSKINS+MAX_CLIENTS; i++)
					{
						if (!level.configstrings[i][0])
						{
							multicast_configstring(i, data->string);
							p->configstring[data->index] = i;
							break;
						}
					}
				}
				else if (data->index >= CS_MISC && data->index < MAX_CONFIGSTRINGS)
				{
					for (i = CS_MISC; i < MAX_CONFIGSTRINGS; i++)
					{
						if (!level.configstrings[i][0])
						{
							multicast_configstring(i, data->string);
							p->configstring[data->index] = i;
							break;
						}
					}
				}
			}
			break;
		case svc_spawnbaseline:
			{
				spawnbaseline_t *data = (spawnbaseline_t *)msg->data;
				
				// don't worry about mask, taken care of in ParseBlock
				p->baselines[data->entity] = data->s;
			}
			break;
		case svc_centerprint:
			{
				centerprint_t *data = (centerprint_t *)msg->data;
				int		i;
				edict_t	*other;

				for (i = 0; i < game.maxclients; i++)
				{
					other = g_edicts + i + 1;
					if (!other->inuse)
						continue;
					if (!(other->client->pers.dm2flags & DM2_SHOWSTATUS))
						continue;

					gi.centerprintf(other, data->text);
				}
			}
			break;
		case svc_playerinfo:
			{
				playerinfo_t *data = (playerinfo_t *)msg->data;
				long	mask, mask2;
				int		i;
				edict_t	*other;

				mask = data->mask;
				mask2 = data->mask2;

				// check whether baselines are needed
				if (p->curblock.oldframe == -1)
				{
					memset (&p->ps, 0, sizeof(player_state_t));
					// needed?
//					p->ps.fov = 90;
				}

				//if (mask & 0x0001) p->ps.pmove.pm_type = data->pm_type;
				if (mask & 0x0002)
				{
					for (i = 0; i < 3; i++)
						p->ps.pmove.origin[i] = data->origin[i]*8;
					VectorCopy(data->origin, p->vieworigin);
					// set relevant player origins to this
					// in case sounds are played same frame
					// otherwise, won't hear sounds made in same
					// frame as when we spawned or teleport
					for (i = 0; i < game.maxclients; i++)
					{
						other = g_edicts + i + 1;
						if (!other->inuse)
							continue;
						if (!(other->client->pers.dm2flags & (DM2_LOCKPOS | DM2_LOCKVIEW)))
							continue;
						VectorCopy(data->origin, other->s.origin);
						gi.linkentity(other);
					}
				}
				if (mask & 0x0004)
				{
					for (i = 0; i < 3; i++)
						p->ps.pmove.velocity[i] = data->velocity[i]*8;
				}
				if (mask & 0x0008) p->ps.pmove.pm_time = data->pm_time;
				if (mask & 0x0010) p->ps.pmove.pm_flags = data->pm_flags;
				if (mask & 0x0020) p->ps.pmove.gravity = data->gravity;
				if (mask & 0x0040)
				{
					for (i = 0; i < 3; i++)
						p->ps.pmove.delta_angles[i] = ANGLE2SHORT(data->delta_angles[i]);
				}
				if (mask & 0x0080) VectorCopy (data->viewoffset, p->ps.viewoffset);
				if (mask & 0x0100) VectorCopy (data->viewangles, p->ps.viewangles);
				if (mask & 0x0200) VectorCopy (data->kick_angles, p->ps.kick_angles);
				if (mask & 0x1000) p->ps.gunindex = MODEL(p, data->gunindex);
				if (mask & 0x2000)
				{
					p->ps.gunframe = data->gunframe;
					VectorCopy (data->gunoffset, p->ps.gunoffset);
					VectorCopy (data->gunangles, p->ps.gunangles);
				}
				if (mask & 0x0400)
				{
					for (i = 0; i < 4; i++)
						p->ps.blend[i] = data->blend[i];
				}

				if (mask & 0x0800) p->ps.fov = data->fov;
				if (mask & 0x4000) p->ps.rdflags = data->rdflags;
				for (i = 0; i < MAX_STATS; i++)
				{
					if (mask2 & (0x00000001 << i))
					{
						switch (p->statusbar[i])
						{
						case STAT_IMAGE:
							p->ps.stats[i] = IMAGE(p, data->stats[i]);
							break;
						case STAT_STRING:
							p->ps.stats[i] = INVEN(p, data->stats[i]);
							break;
						default:
							p->ps.stats[i] = data->stats[i];
							break;
						}
					}
				}
			}
			break;
		case svc_packetentities:
			{
				packetentities_t *data = (packetentities_t *)msg->data;
				long mask, i;
				edict_t *ent, *other;

				ent = ValidateEntity(p, data->entity);

				// check whether baselines are needed
				if (ent->baseline)
					GetBaseline (p, data->entity);

				mask = data->mask;
				if (mask & 0x00000800) ent->dm2.modelindex = data->modelindex;
				if (mask & 0x00100000) ent->dm2.modelindex2 = data->modelindex2;
				if (mask & 0x00200000) ent->dm2.modelindex3 = data->modelindex3;
				if (mask & 0x00400000) ent->dm2.modelindex4 = data->modelindex4;
				// don't have to worry about bytes or shorts
				if (mask & 0x00020010) ent->dm2.frame = data->frame;
				if (mask & 0x02010000) ent->dm2.skinnum = data->skin;
				if (mask & 0x00084000) ent->dm2.effects = data->effects;
				if (mask & 0x00041000) ent->dm2.renderfx = data->renderfx;
				if (mask & 0x00000001) ent->dm2.origin[0] = data->origin[0];
				if (mask & 0x00000002) ent->dm2.origin[1] = data->origin[1];
				if (mask & 0x00000200) ent->dm2.origin[2] = data->origin[2];
				if (mask & 0x00000400) ent->dm2.angles[0] = data->angles[0];
				if (mask & 0x00000004) ent->dm2.angles[1] = data->angles[1];
				if (mask & 0x00000008) ent->dm2.angles[2] = data->angles[2];
				if (mask & 0x01000000) VectorCopy(data->old_origin, ent->dm2.old_origin);
				if (mask & 0x04000000) ent->dm2.sound = data->sound;
				ent->dm2.event = data->event;
				if (data->remove)
				{	// remove bit
					G_FreeEdict(ent);
					EDICT(p, data->entity) = ent = NULL;
					break;
				}
				else
					ent->svflags &= ~SVF_NOCLIENT;

				ent->s.modelindex = MODEL(p, ent->dm2.modelindex);
				ent->s.modelindex2 = MODEL(p, ent->dm2.modelindex2);
				ent->s.modelindex3 = MODEL(p, ent->dm2.modelindex3);
				ent->s.modelindex4 = MODEL(p, ent->dm2.modelindex4);
				ent->s.frame = ent->dm2.frame;
				if (ent->s.modelindex == 255)
				{
					ent->s.skinnum = PLAYER(p, ent->dm2.skinnum);
					ent->s.skinnum |= VWeapIndex(p, ent->dm2.skinnum) << 8;
				}
				else if (ent->dm2.renderfx & RF_CUSTOMSKIN)
					ent->s.skinnum = IMAGE(p, ent->dm2.skinnum);
				else
					ent->s.skinnum = ent->dm2.skinnum;

				ent->s.effects = ent->dm2.effects;
				ent->s.renderfx = ent->dm2.renderfx;
				VectorCopy(ent->dm2.origin, ent->s.origin);
				VectorCopy(ent->dm2.angles, ent->s.angles);
				VectorCopy(ent->dm2.old_origin, ent->s.old_origin);
				ent->s.sound = ent->dm2.sound;
				ent->s.event = ent->dm2.event;

				if (data->entity == p->viewent)
				{
					if (mask & 0x00000800)
						p->viewentmodel = ent->s.modelindex;
					// if nobody is freemoving, make viewent invisible
					// FIXME: allow server override
					for (i = 0; i < game.maxclients; i++)
					{
						other = g_edicts + i + 1;
						if (!other->inuse)
							continue;
						if (other->client->pers.dm2flags & DM2_LOCKPOS)
							continue;
						
						ent->s.modelindex = p->viewentmodel;
						goto Done;
					}
					ent->s.modelindex = 0;
				}
Done:
				// BSP models don't work unless gi.setmodel is called
				// check for modelnames with '*' followed by number
				if (inbounds(ent->s.modelindex, 1, 254))
				{
//					if ((level.configstrings[CS_MODELS+ent->s.modelindex][0] == '*') &&
//						(isdigit(level.configstrings[CS_MODELS+ent->s.modelindex][1])))
						gi.setmodel(ent, level.configstrings[CS_MODELS+ent->s.modelindex]);
				}
				gi.linkentity(ent);
			}
			break;
		case svc_frame:
			{
				frame_t *data = (frame_t *)msg->data;
				int i;
				edict_t *ent;

				block->framenum = data->seq1;
				block->oldframe = data->seq2;

				// reset already spawned ents to baselines
				// non-spawned ents that have a baseline are
				// taken care of in svc_packentities
				if (data->seq2 == -1)
				{
					for (i = 0; i < MAX_EDICTS; i++)
					{
						ent = EDICT(p, i);
						if (ent && ent->inuse)
							GetBaseline(p, i);
					}
				}
			}
			break;
		default:
//			gi.error ("Unknown type: %d\n", msg->id);
			break;
		}
	}
	return false;
}

block_t *ReadBlock (block_t *dest, dm2_t *p)
{
	WriteLog("Reading block\n");
	if (fread(&dest->size, 1, 4, p->file) < 4)
	{
		if (p->isdemo == RECORD_SERVER)
			dest->size = -1;
		else
			gi.error ("Error reading block size from dm2 file.\n");
	}

	// check if end of demo
	if (dest->size == -1)
	{
		gi.dprintf("Closing demo file.\n");
		WriteLog("ReadBlock: Closed dm2 (size == -1)\n");
		DemoClose (p);
		return dest;
	}

//	dest->buffer = gi.TagMalloc(dest->size, TAG_LEVEL);
	dest->buffer = malloc(dest->size);

	if (fread(dest->buffer, 1, dest->size, p->file) < dest->size)
		gi.error ("Error reading block data from dm2 file.\n");
	
	dest->p = dest->buffer;
	dest->end = &(dest->buffer[dest->size]);
	return dest;
}

qboolean NextFrame(dm2_t *p)
{
	int	i;

	if (p->curblock.size == -1)
		return false;
	WriteLog("Nextframe\n");

//	if (!stopdemo)
//	{
		ReadBlock(&p->curblock, p);
		if (p->curblock.size == -1 || !p->file)
		{
//			gi.AddCommandString("killserver\n");
//			return true;
			return false;
		}
		WriteLog("About to parse block\n");

		ParseBlock(p, true);
//	}
	WriteLog("About to process block\n");
	if (ProcessBlock (p, &p->curblock))
		return true;
	// serverrecord demos don't tell us if an entity is removed
	// via the record bit, so assume that an entity that hasn't
	// been updated this frame has been removed
	if (p->isdemo == RECORD_SERVER)
	{
		for (i = 0; i < MAX_EDICTS; i++)
		{
			if (EDICT(p, i) && !p->curblock.entityupdated[i])
			{
				G_FreeEdict(EDICT(p, i));
				EDICT(p, i) = NULL;
			}
		}
	}
	FreeBlock(&p->curblock);
//	gi.dprintf("Framenum: %d\n", p->curblock.framenum);
	WriteLog("End nextframe\n");
	return false;
}

void ProcessPreFrame (dm2_t *p)
{
	int framenum = 0, i = 0;
	size_t oldpos;
	
	rewind(p->file);
	while (1)
	{
		oldpos = ftell(p->file);
		ReadBlock(&p->curblock, p);
		if (p->curblock.size == -1)
			return;
		framenum = ParseBlock(p, true);
		if (framenum)
		{
			// get player state since player spawns
			// before first frame is run
			playerinfo_t *data;
			int j, mask, mask2;

			// serverrecords have no player states,
			// so set vieworigin to lowest entity number's origin
			// (hopefully a player)
			if (p->isdemo == RECORD_SERVER)
			{
				packetentities_t *data;
				j = FindInBlock(&p->curblock, 0, svc_packetentities);
				if (j == -1)
					break;
				data = (packetentities_t *)p->curblock.message[i].data;
				VectorCopy(data->origin, p->vieworigin);
				p->vieworigin[2] += 32;	// lift up a bit
				WriteLog("Using entity %d as origin at %s\n", data->entity, vtos(p->vieworigin));
				VectorCopy(data->angles, p->ps.viewangles);
				for (i = 0; i < 3; i++)
					p->ps.pmove.origin[i] = p->vieworigin[i]*8;
				break;
			}
			j = FindInBlock(&p->curblock, 0, svc_playerinfo);
			if (j == -1)
				break;
			data = (playerinfo_t *)p->curblock.message[j].data;
			mask = data->mask;
			mask2 = data->mask2;

			if (mask & 0x0002)
			{
				for (i = 0; i < 3; i++)
					p->ps.pmove.origin[i] = data->origin[i]*8;
				VectorCopy(data->origin, p->vieworigin);
			}
			if (mask & 0x0004)
			{
				for (i = 0; i < 3; i++)
					p->ps.pmove.velocity[i] = data->velocity[i]*8;
			}
			if (mask & 0x0008) p->ps.pmove.pm_time = data->pm_time;
			if (mask & 0x0010) p->ps.pmove.pm_flags = data->pm_flags;
			if (mask & 0x0020) p->ps.pmove.gravity = data->gravity;
			if (mask & 0x0040)
			{
				for (i = 0; i < 3; i++)
					p->ps.pmove.delta_angles[i] = ANGLE2SHORT(data->delta_angles[i]);
			}
			if (mask & 0x0080) VectorCopy (data->viewoffset, p->ps.viewoffset);
			if (mask & 0x0100) VectorCopy (data->viewangles, p->ps.viewangles);
			if (mask & 0x0200) VectorCopy (data->kick_angles, p->ps.kick_angles);
			if (mask & 0x1000) p->ps.gunindex = MODEL(p, data->gunindex);
			if (mask & 0x2000)
			{
				p->ps.gunframe = data->gunframe;
				VectorCopy (data->gunoffset, p->ps.gunoffset);
				VectorCopy (data->gunangles, p->ps.gunangles);
			}
			if (mask & 0x0400)
			{
				for (i = 0; i < 4; i++)
					p->ps.blend[i] = data->blend[i];
			}
			if (mask & 0x0800) p->ps.fov = data->fov;
			if (mask & 0x4000) p->ps.rdflags = data->rdflags;
			for (i = 0; i < MAX_STATS; i++)
			{
				if (mask2 & (0x00000001 << i))
				{
					switch (p->statusbar[i])
					{
					case STAT_IMAGE:
						p->ps.stats[i] = IMAGE(p, data->stats[i]);
						break;
					case STAT_STRING:
						p->ps.stats[i] = INVEN(p, data->stats[i]);
						break;
					default:
						p->ps.stats[i] = data->stats[i];
						break;
					}
				}
			}
			break;
		}
		i++;
		ProcessBlock(p, &p->curblock);
		FreeBlock(&p->curblock);
//		gi.dprintf("preframe: %d\n", i);
	}
	FreeBlock(&p->curblock);
	fseek(p->file, oldpos, SEEK_SET);
	WriteLog("Preprocessed %d blocks\n", i);
//	gi.dprintf("First frame: %d\n", framenum);
//	Com_Printf("Blocks without a frame: %d\n\n\n\n", i);
}

int ParseBlock(dm2_t *p, qboolean change)
{
	long	id, framenum = 0;
	block_t	*block = &p->curblock;
	message_t msg;
	
//	FreeBlock(block);
	if (block->size == -1)
		return -1;
	block->num_messages = 0;

	while (block->p < block->end)
	{
		id = ReadByte(block);
		msg.id	= id;
		msg.data= NULL;

		WriteLog("%d: ", block->num_messages);

		switch(id)
		{
		case svc_bad:
			{
			}
			break;
		case svc_muzzleflash:
			{
//				muzzleflash_t *data = gi.TagMalloc(sizeof(muzzleflash_t), TAG_LEVEL);
				muzzleflash_t *data = malloc(sizeof(muzzleflash_t));
			
				data->entity = ReadShort(block);
				data->value = ReadByte(block);
				WriteLog("muzzleflash: ent %d %d\n", data->entity, data->value);
				msg.data = (void *)data;
			}
			break;
		case svc_muzzleflash2:
			{
//				muzzleflash_t *data = gi.TagMalloc(sizeof(muzzleflash_t), TAG_LEVEL);
				muzzleflash_t *data = malloc(sizeof(muzzleflash_t));
				
				data->entity = ReadShort(block);
				data->value = ReadByte(block);
				WriteLog("muzzleflash2: ent %d %d\n", data->entity, data->value);
				msg.data = (void *)data;
			}
			break;
		case svc_temp_entity:
			{
//				temp_entity_t *data = gi.TagMalloc(sizeof(temp_entity_t), TAG_LEVEL);
				temp_entity_t *data = malloc(sizeof(temp_entity_t));
				
				data->entitytype = ReadByte(block);
				WriteLog("tempentity: %d\n", data->entitytype);
				switch (data->entitytype)
				{
				// point entity
				case TE_EXPLOSION1:
				case TE_EXPLOSION2:
				case TE_ROCKET_EXPLOSION:
				case TE_GRENADE_EXPLOSION:
				case TE_ROCKET_EXPLOSION_WATER:
				case TE_GRENADE_EXPLOSION_WATER:
				case TE_BFG_EXPLOSION:
				case TE_BFG_BIGEXPLOSION:
				case TE_BOSSTPORT:
				case 28:
					ReadPosition(block, data->origin);
					break;
				// impact entity
				case TE_GUNSHOT:
				case TE_BLOOD:
				case TE_BLASTER:
				case TE_SHOTGUN:
				case TE_SPARKS:
				case TE_SCREEN_SPARKS:
				case TE_SHIELD_SPARKS:
				case TE_BULLET_SPARKS:
				case TE_GREENBLOOD:
					ReadPosition(block, data->origin);
					ReadDir(block, data->movedir);
					break;
				// line entity
				case TE_RAILTRAIL:
				case TE_BUBBLETRAIL:
				case TE_BFG_LASER:
				case TE_PLASMATRAIL:
					ReadPosition(block, data->origin);
					ReadPosition(block, data->endpos);
					break;
				// special entity
				case TE_SPLASH:
					data->count = ReadByte(block);
					ReadPosition(block, data->origin);
					ReadDir(block, data->movedir);
					data->sound = ReadByte(block);
					break;
				case TE_LASER_SPARKS:
				case TE_WELDING_SPARKS:
				case 29:
					data->count = ReadByte(block);
					ReadPosition(block, data->origin);
					ReadDir(block, data->movedir);
					data->color = ReadByte(block);
					break;
				case TE_PARASITE_ATTACK:
				case TE_MEDIC_CABLE_ATTACK:
					data->entity = ReadShort(block);
					ReadPosition(block, data->origin);
					ReadPosition(block, data->endpos);
					break;
				case TE_GRAPPLE_CABLE:
					data->entity = ReadShort(block);
					ReadPosition(block, data->origin);
					ReadPosition(block, data->endpos);
					ReadPosition(block, data->offset);
					break;
				default:
					gi.error("Unknown TE type: 0x%x", data->entitytype);
					break;
				}
				msg.data = (void *)data;
			}
			break;
		case svc_layout:
			{
//				layout_t *data = gi.TagMalloc(sizeof(layout_t), TAG_LEVEL);
				layout_t *data = malloc(sizeof(layout_t));

				ReadString(block, data->text);
				WriteLog("layout: %s\n", data->text);
				msg.data = (void *)data;
			}
			break;
		case svc_inventory:
			{
				int i;
//				inventory_t *data = gi.TagMalloc(sizeof(inventory_t), TAG_LEVEL);
				inventory_t *data = malloc(sizeof(inventory_t));
				
				for (i=0 ; i<MAX_ITEMS ; i++)
				{
					data->inventory[i] = ReadShort(block);
				}
				WriteLog("inventory\n");
				msg.data = (void *)data;
			}
			break;
		case svc_nop:
			{
			}
			break;
		case svc_disconnect:
			{
			}
			break;
		case svc_reconnect:
			{
			}
			break;
		case svc_sound:
			{
				long mask, mix;
//				sound_t *data = gi.TagMalloc(sizeof(sound_t), TAG_LEVEL);
				sound_t *data = malloc(sizeof(sound_t));
				
				data->mask = mask = ReadByte(block);
				data->soundnum = ReadByte(block);
				data->vol = (mask & 0x01) ? ((float)ReadByte(block) / 255.0) : (1.0);
				data->attenuation = (mask & 0x02) ? ((float)ReadByte(block) / 64.0) : (1.0);
				data->timeofs = (mask & 0x10) ? ((float)ReadByte(block) * 0.001) : (0.0);
				if (mask & 0x08)
				{
					mix = ReadShort(block);
					data->entity = (mix >> 3) & 0x3FF;
					data->channel = mix & 0x07;
				}
				else
				{
					data->channel = 0;
					data->entity = 0;
				}
				if (mask & 0x04)
				{
					ReadPosition(block, data->origin);
				}
				WriteLog("sound: %d\n", data->soundnum);
				msg.data = (void *)data;
			}
			break;
		case svc_print:
			{
//				print_t *data = gi.TagMalloc(sizeof(print_t), TAG_LEVEL);
				print_t *data = malloc(sizeof(print_t));

				data->level = ReadByte(block);
				ReadString(block, data->string);
				WriteLog("print: %s", data->string);
				msg.data = (void *)data;
			}
			break;
		case svc_stufftext:
			{
//				stufftext_t *data = gi.TagMalloc(sizeof(stufftext_t), TAG_LEVEL);
				stufftext_t *data = malloc(sizeof(stufftext_t));
				
				ReadString(block, data->text);
				WriteLog("stufftext: %s\n", data->text);

				msg.data = (void *)data;
			}
			break;
		case svc_serverdata:
			{
//				serverdata_t *data = gi.TagMalloc(sizeof(serverdata_t), TAG_LEVEL);
				serverdata_t *data = malloc(sizeof(serverdata_t));
					
				data->serverversion = ReadLong(block);
				data->key = ReadLong(block);
				data->isdemo = ReadByte(block);
				ReadString(block, data->game);
				data->player = ReadShort(block);
				ReadString(block, data->mapname);
				WriteLog("serverdata\n");

				msg.data = (void *)data;
			}
			break;
		case svc_configstring:
			{
//				configstring_t *data = gi.TagMalloc(sizeof(configstring_t), TAG_LEVEL);
				configstring_t *data = malloc(sizeof(configstring_t));
				char	buf[MAX_STRING_LENGTH];

				memset(data, 0, sizeof(configstring_t));
				data->index = ReadShort(block);
				ReadString(block, buf);
				strncpy(data->string, buf, MAX_QPATH);
				if (strlen(buf) >= MAX_QPATH)
					WriteLog("Warning: truncated configstring from %d chars to 64 chars\n", strlen(buf));
				WriteLog("configstring: %d %s\n", data->index, data->string);

				msg.data = (void *)data;
			}
			break;
		case svc_spawnbaseline:
			{
//				spawnbaseline_t *data = gi.TagMalloc(sizeof(spawnbaseline_t), TAG_LEVEL);
				spawnbaseline_t *data = malloc(sizeof(spawnbaseline_t));
				long mask;

				memset(data, 0, sizeof(spawnbaseline_t));
				mask = ReadByte(block);
				if (mask & 0x00000080) mask |= (ReadByte(block) <<	8);
				if (mask & 0x00008000) mask |= (ReadByte(block) << 16);
				if (mask & 0x00800000) mask |= (ReadByte(block) << 24);
				data->mask = mask;
				data->entity = (mask & 0x00000100) ? ReadShort(block) : ReadByte(block);
				if (mask & 0x00000800) data->s.modelindex = ReadByte(block);
				if (mask & 0x00100000) data->s.modelindex2 = ReadByte(block);
				if (mask & 0x00200000) data->s.modelindex3 = ReadByte(block);
				if (mask & 0x00400000) data->s.modelindex4 = ReadByte(block);
				if (mask & 0x00000010) data->s.frame = ReadByte(block);
				if (mask & 0x00020000) data->s.frame = ReadShort(block);
				if (mask & 0x00010000)
				{
					if (mask & 0x02000000) data->s.skinnum = ReadLong(block);
					else data->s.skinnum = ReadByte(block);
				}
				else
				{
					if (mask & 0x02000000) data->s.skinnum = ReadShort(block);
				}
				if (mask & 0x00004000)
				{
					if (mask & 0x00080000) data->s.effects = ReadLong(block);
					else data->s.effects = ReadByte(block);
				}
				else
				{
					if (mask & 0x00080000) data->s.effects = ReadShort(block);
				}
				if (mask & 0x00001000)
				{
					if (mask & 0x00040000) data->s.renderfx = ReadLong(block);
					else data->s.renderfx = ReadByte(block);
				}
				else
				{
					if (mask & 0x00040000) data->s.renderfx = ReadShort(block);
				}
				if (mask & 0x00000001) data->s.origin[0] = ReadCoord(block);
				if (mask & 0x00000002) data->s.origin[1] = ReadCoord(block);
				if (mask & 0x00000200) data->s.origin[2] = ReadCoord(block);
				if (mask & 0x00000400) data->s.angles[0] = ReadAngle(block);
				if (mask & 0x00000004) data->s.angles[1] = ReadAngle(block);
				if (mask & 0x00000008) data->s.angles[2] = ReadAngle(block);
				if (mask & 0x01000000) ReadPosition(block, data->s.old_origin);
				if (mask & 0x04000000) data->s.sound = ReadByte(block);
				data->s.event = (mask & 0x00000020) ? ReadByte(block) : 0;
				if (mask & 0x08000000) data->s.solid = ReadShort(block);
				WriteLog("spawnbaseline: %d\n", data->entity);

				msg.data = (void *)data;
			}
			break;
		case svc_centerprint:
			{
//				centerprint_t *data = gi.TagMalloc(sizeof(centerprint_t), TAG_GAME);
				centerprint_t *data = malloc(sizeof(centerprint_t));

				ReadString(block, data->text);

				WriteLog("centerprint: %s\n", data->text);
				msg.data = (void *)data;
			}
			break;
		case svc_download:
			{
				int size;

				size = ReadShort(block);	// size
				ReadByte(block);	// percent

				WriteLog("download\n");
				if (p->version >= 32)
				{
					int i;
					for (i=0 ; i<size ; i++)
					{
						ReadByte(block);	// data[i]
					}
				}
			}
			break;
		case svc_playerinfo:
			{
//				playerinfo_t *data = gi.TagMalloc(sizeof(playerinfo_t), TAG_LEVEL);
				playerinfo_t *data = malloc(sizeof(playerinfo_t));
				long mask, mask2;
				int i;

				data->mask = mask = ReadShort(block);
				if (mask & 0x0001) data->pm_type = ReadByte(block);
				if (mask & 0x0002)
				{	
					ReadPosition(block, data->origin);
				}
				if (mask & 0x0004)
				{
					ReadPosition(block, data->velocity);
				}
				if (mask & 0x0008) data->pm_time = ReadByte(block);
				if (mask & 0x0010) data->pm_flags = ReadByte(block);
				if (mask & 0x0020) data->gravity = ReadShort(block);
				if (mask & 0x0040)
				{
					data->delta_angles[0] = ReadAngle16(block);
					data->delta_angles[1] = ReadAngle16(block);
					data->delta_angles[2] = ReadAngle16(block);
				}
				if (mask & 0x0080)
				{
					ReadOffsetVec(block, data->viewoffset);
				}
				if (mask & 0x0100)
				{
					data->viewangles[0] = ReadAngle16(block);
					data->viewangles[1] = ReadAngle16(block);
					data->viewangles[2] = ReadAngle16(block);
				}
				if (mask & 0x0200)
				{
					ReadOffsetVec(block, data->kick_angles);
				}
				if (mask & 0x1000) data->gunindex = ReadByte(block);
				if (mask & 0x2000)
				{
					data->gunframe = ReadByte(block);
					ReadOffsetVec(block, data->gunoffset);
					ReadOffsetVec(block, data->gunangles);
				}
				if (mask & 0x0400) ReadBlendVec(block, data->blend);
				if (mask & 0x0800)
					data->fov = ReadByte(block);
				if (mask & 0x4000) data->rdflags = ReadByte(block);
				data->mask2 = mask2 = ReadLong(block);
				for (i=0;i<MAX_STATS;i++)
				{
					if (mask2 & (0x00000001 << i)) data->stats[i] = ReadShort(block);
				}
				WriteLog("playerinfo\n");

				msg.data = (void *)data;
			}
			break;
		case svc_packetentities:
			{
				long mask;
				long entity;
				packetentities_t *data;
				
				while (1)
				{
					mask = ReadByte(block);
					if (mask & 0x00000080) mask |= (ReadByte(block) <<8);
					if (mask & 0x00008000) mask |= (ReadByte(block) << 16);
					if (mask & 0x00800000) mask |= (ReadByte(block) << 24);
					entity = (mask & 0x00000100) ? ReadShort(block) : ReadByte(block);
					
					if (entity == 0) break;
//					data = gi.TagMalloc(sizeof(packetentities_t), TAG_LEVEL);
					data = malloc(sizeof(packetentities_t));
					memset (data, 0, sizeof(packetentities_t));

					data->mask = mask;
					data->entity = entity;
					data->remove = (mask & 0x00000040) ? 1 : 0;
					if (mask & 0x00000800) data->modelindex = ReadByte(block);
					if (mask & 0x00100000) data->modelindex2 = ReadByte(block);
					if (mask & 0x00200000) data->modelindex3 = ReadByte(block);
					if (mask & 0x00400000) data->modelindex4 = ReadByte(block);
					if (mask & 0x00000010) data->frame = ReadByte(block);
					if (mask & 0x00020000) data->frame = ReadShort(block);
					if (mask & 0x00010000)
					{
						if (mask & 0x02000000) data->skin = ReadLong(block);
						else data->skin = ReadByte(block);
					}
					else
					{
						if (mask & 0x02000000) data->skin = ReadShort(block);
					}
					if (mask & 0x00004000)
					{
						if (mask & 0x00080000) data->effects = ReadLong(block);
						else data->effects = ReadByte(block);
					}
					else
					{
						if (mask & 0x00080000) data->effects = ReadShort(block);
					}
					if (mask & 0x00001000)
					{
						if (mask & 0x00040000) data->renderfx = ReadLong(block);
						else data->renderfx = ReadByte(block);
					}
					else
					{
						if (mask & 0x00040000) data->renderfx = ReadShort(block);
					}
					if (mask & 0x00000001) data->origin[0] = ReadCoord(block);
					if (mask & 0x00000002) data->origin[1] = ReadCoord(block);
					if (mask & 0x00000200) data->origin[2] = ReadCoord(block);
					if (mask & 0x00000400) data->angles[0] = ReadAngle(block);
					if (mask & 0x00000004) data->angles[1] = ReadAngle(block);
					if (mask & 0x00000008) data->angles[2] = ReadAngle(block);
					if (mask & 0x01000000) ReadPosition(block, data->old_origin);
					if (mask & 0x04000000) data->sound = ReadByte(block);
					data->event = (mask & 0x00000020) ? ReadByte(block) : 0;
					if (mask & 0x08000000) data->solid = ReadShort(block);

					block->entityupdated[data->entity] = true;
					msg.id = svc_packetentities;
					msg.data = (void *)data;
					if (change)
						block->message[block->num_messages++] = msg;
					else
//						gi.TagFree(msg.data);
						free(msg.data);
				}
				WriteLog("packentities\n");
			}
			break;
		case svc_deltapacketentities:
			break;
		case svc_frame:
			{
//				frame_t *data = gi.TagMalloc(sizeof(frame_t), TAG_LEVEL);
				frame_t *data = malloc(sizeof(frame_t));
				int i;

				framenum = data->seq1 = ReadLong(block);
				if (p->isdemo != RECORD_SERVER)
				{
					data->seq2 = ReadLong(block);
					if (p->version != 26)
					{
						ReadByte(block);
					}
					data->count = ReadByte(block);
					for (i = 0; i < data->count; i++)
						data->areas[i] = ReadByte(block);
				}
				else
					data->seq2 = 0;
				
				WriteLog("frame: %d\n", data->seq1);
				msg.data = (void *)data;
			}
			break;
		default:
			gi.error ("ParseBlock: Unknown message type: %d\n", id);
			break;
		} // end switch
		

		// svc_packetentities has its own routine
		if (id != svc_packetentities && msg.data != NULL)
		{
			if (change)
				block->message[block->num_messages++] = msg;
			else if (msg.data)
//				gi.TagFree(msg.data);
				free(msg.data);
		}

	} // end while
	WriteLog("End ParseBlock\n");
	return framenum;
}
