/*

MIPINDEX.CPP

Miptexture management utility for BSP / PAK / WAD2

Introducing two new file formats: DST (distributable BSP / WAD2) and IDX
(miptex index into other files)

by Ed Kiser (edkiser@jaxnet.com) Copyright (c) 1996

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

This program is always under construction.

History moved to history.txt.

*/

#ifndef __MIPINDEX_H
#include "mipindex.h"
#endif

indexer::indexer(int argc, char * argv[]): _argc(argc)
{ _argv=new (char*)[argc];
  for (int i=0; i<_argc; i++) _argv[i]=argv[i];
  action=error; what_error=unknownopt;
  if (_argc<2)
  { what_error=nooption;
    return;
  }
  if (stricmp(_argv[1],"-list")==0)
  { if (_argc>=3) action=list;
    else what_error=toofewparms;
  } else
  if (stricmp(_argv[1],"-wad2")==0)
  { if (_argc>=4) action=wad2;
    else what_error=toofewparms;
  } else
  if (stricmp(_argv[1],"-idx")==0)
  { if (_argc>=4) action=saveidx;
    else what_error=toofewparms;
  } else
  if (stricmp(_argv[1],"-dst")==0)
  { if (_argc>=4) action=dst;
    else what_error=toofewparms;
  } else
  if (stricmp(_argv[1],"-bsp")==0 || stricmp(_argv[1],"-undst")==0)
  { if (_argc>=4) action=undst;
    else what_error=toofewparms;
  } else
  if (stricmp(_argv[1],"-gpl")==0)
  { if (_argc==2) action=gpl;
    else what_error=toomanyparms;
  }
};

indexer::~indexer()
{ delete _argv;
};

void indexer::explain(ostream & cout)
{ char p[260];
  char * pp= p;
  switch (what_error)
  { case unknownopt:
      strcpy(p,"Unknown option ");
      strcat(p,_argv[1]);
      strcat(p,".");
      break;
    case toofewparms:
      pp="Too few parameters.";
      break;
    case nooption:
      pp="No option specified.";
      break;
    case toomanyparms:
      pp="Too many parameters.";
      break;
    default:
      pp="Unknown error.";
      break;
  }
  cout << "Option parsing failed: " << pp << "\n\n"
"Options and syntaxes are: <required> [optional] <1 or more or wildcard>+\n\
* List miptextures.         mipindex -list <src>+\n\
* Compile a WAD2.           mipindex -wad2 <wad2file> <source>+\n\
* Compile an IDX.           mipindex -idx <idxfile> <source>+\n\
* Convert BSP's to DST's.   mipindex -dst <index> <bspfile>+\n\
* Convert DST's to BSP's.   mipindex -undst <index> <dstfile>+\n\
* View licensing info       mipindex -gpl\n\
\n\
Sources and indeces can be .idx .pak .bsp .wad\n\
File extensions must always be explicitly specified!\n\
Compiling an .IDX is recommended for faster BSP / DST conversion.\n\
\n\
For information about this program, view the README.\n"
  ;
};

void indexer::run(ostream & cout)
{ switch (action)
  { case list: print_list(cout); break;
    case wad2: make_wad2(cout); break;
    case saveidx: make_idx(cout); break;
    case dst: bsp_dst(cout,true); break;
    case undst: bsp_dst(cout,false); break;
    case gpl: print_gpl(cout); break;
    default: cout << "Unimplemented.\n"; break;
  }
};

void indexer::print_list(ostream & cout)
{ for (int i=2; i<_argc; i++) try
  { idx.read_file(_argv[i],cout);
  }
  catch (char * P)
  { cout << "File error: " << P << "\n";
  }
  idx.write_text(cout);
};

void indexer::make_wad2(ostream & cout)
{ cout << "Making WAD2: " << _argv[2] << "\n";
  bool problems=false;
  try
  { idx.set_visitor(new wad2_writer(_argv[2]));
    for (int i=3; i<_argc; i++) try
    { idx.read_file(_argv[i],cout);
    }
    catch (char * P)
    { cout << "File error: " << P << "\n";
    }
  }
  catch (char * P)
  { cout << "Process error: " << P << "\n";
    problems=true;
  }
  if (!problems) cout << "Done.\n";
};

void indexer::make_idx(ostream & cout)
{ cout << "Making IDX: " << _argv[2] << "\n";
  ofstream idxfile(_argv[2],ios::out | ios::binary | ios::noreplace);
  if (idxfile.fail())
  { cout << "Error: File " << _argv[2] << " already exists.\n";
    return;
  }
  for (int i=3; i<_argc; i++) try
  { idx.read_file(_argv[i],cout);
  }
  catch (char * P)
  { cout << "File error: " << P << "\n";
  }
  idx.write_idx(idxfile,cout);
  cout << "Done.\n";
};

void indexer::print_gpl(ostream & cout)
{ cout << "\
This program is free software; you can redistribute it and/or modify\n\
it under the terms of the GNU General Public License as published by\n\
the Free Software Foundation; either version 2 of the License, or\n\
(at your option) any later version.\n\
\n\
This program is distributed in the hope that it will be useful,\n\
but WITHOUT ANY WARRANTY; without even the implied warranty of\n\
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n\
GNU General Public License for more details.\n\
\n\
You should have received a copy of the GNU General Public License\n\
along with this program; if not, write to the Free Software\n\
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n"
  ;
};

/***** main *****/

int main(int argc, char * argv[])
{ cout << "MIPINDEX Miptexture Indexer, version 1.2\n"
          "Copyright (c) " __DATE__ " " __TIME__ " by Edward Kiser.\n"
  "For latest information, visit http://www.jaxnet.com/~edkiser/mipindex/\n"
          "\n";
  indexer mip(argc,argv);
  if (mip.is_valid()) mip.run(cout);
  else mip.explain(cout);
};

