/***************************************************************************

 bmp2map.c - A conversion utility to convert 256 color .BMPs into
             Quake .MAPs.

             It's intended use is to quickly convert 2D floor/street
             plans into 3D maps.

             Pixel colors are used to create boxes textured as solid walls,
             doors, windows or windows filled with sky (for outside walls).

   Author - Jack Perdue (aka Silicon Slick) - si_slick@cy-net.net
     Date - May 10th, 1997
 Compiler - Borland TurboC 2.0 for MS-DOS

 Copyright(C) 1997 - Silicon Slick's Software, Supplies and Support Services
         http://www2.cy-net.net/~si_slick  si_slick@cy-net.net

	You are welcome to copy and modify this code at will as long as you
       include this copyright statement in all derivative works.

                 "Quake" is copyright id Software

(tabs=4)
***************************************************************************/

#include    <stdio.h>
#include    <stdlib.h>
#include    <dos.h>
#include    <mem.h>
#include    <string.h>
#include    <alloc.h>
#include    <conio.h>   /* for getch() */

typedef unsigned char  BYTE;
typedef unsigned short WORD;
typedef unsigned long  DWORD;

#define	TRUE	1
#define FALSE   0
#define	BMPEXT	".bmp"
#define MP1EXT	".mp1"
#define MP2EXT	".mp2"
#define MAPEXT	".map"

#define MUSTBETYPE  	(((WORD) 'M' << 8) | 'B') /* 1st 2bytes must=='BM' */
#define	BI_RGB			0L  /* uncompressed 256 color .BMPs only */
#define	BI_RLE8			1L	/* not supported */
#define	BI_RLE4			2L	/* not supported */

#define DEFQUNITSPERPEL	4		/* Quake units per pixel (box thickness) */
#define	DEFFLOOR		-64		/* z for floor */
#define	DEFCEILING 		64 		/* z for ceiling */
#define DEFWINBOTTOM	-24 	/* z for bottom of windows */
#define DEFWINTOP		32 		/* z for top of windows */
#define	DEFDOORTOP		32 		/* z for top of doors */
#define DEFWALLVAL		0x00	/* Palette index of color representing walls */
#define	DEFWINVAL		0x40	/* "" windows */
#define DEFDOORVAL		0x80	/* "" doors */
#define DEFSKYWINVAL	0xC0	/* "" windows to the outside */
#define	DEFAUTOLIGHT	TRUE	/* automatically add lights? */
#define	DEFAUTOLIGHTQU	128		/* interval (in Quake units) for lights */
#define	DEFWALLTEXT		"tech03_2"		/* texture name for walls/ceilings/floors */
#define DEFDOORTEXT		"door02_3"		/* "" doors */
#define DEFSKYTEXT		"sky4"			/* "" sky for windows to the outside */
#define	DEFFLOORTEXT	"afloor1_8"     /* "" the floor */
#define DEFCEILINGTEXT	"ceiling1_3"    /* "" the ceiling */
#define	DEFWADNAME		"c:\\quake\\wads\\quake101.wad"  /*default .WAD name */

#define CFGNAME			"bmp2map.cfg"  /* name of configuration file */
#define CFGQUNITSPERPEL	"qunitsperpel"
#define	CFGFLOOR		"floor"
#define	CFGCEILING		"ceiling"
#define	CFGWINBOTTOM	"windowbottom"
#define	CFGWINTOP		"windowtop"
#define	CFGDOORTOP		"doortop"
#define	CFGWALLVAL		"wallcolor"     /* see comments in #define DEF... section */
#define	CFGWINVAL		"wincolor"
#define	CFGDOORVAL		"doorcolor"
#define CFGSKYWINVAL	"skywincolor"
#define	CFGWALLTEXT		"walltexture"
#define	CFGSKYTEXT		"skytexture"
#define	CFGDOORTEXT		"doortexture"
#define	CFGFLOORTEXT	"floortexture"
#define	CFGCEILINGTEXT	"ceilingtexture"
#define	CFGWADNAME		"wadname"
#define	CFGAUTOLIGHT	"autolight"
#define	CFGAUTOLIGHTQU  "autolightqunits"

#define	TEXTURENAMESIZE	20
#define	WADNAMESIZE		60
#define	CFGLINESIZE		(WADNAMESIZE+20)

#define	EXTREME 		8192	/* for min/max tracking */

#define	ABSORBED		(-1)	/* brush has been combined with another */
#define	WALLBRUSH		1
#define	DOORBRUSH		2
#define	SKYBRUSH		3
#define	FLOORBRUSH		4
#define CEILINGBRUSH	5

#define MINBRUSH		1
#define MAXBRUSH		5 		/* change this when adding brush types */


typedef struct cfgstruct
{
	int floor;
	int ceiling;
	int winbottom;
	int wintop;
	int doortop;
	BYTE wallval;
	BYTE winval;		/* see comments in #define DEF... section */
	BYTE doorval;
	BYTE skywinval;
	BYTE qunitsperpel;
	BYTE autolight;
	int  autolightqunits;
	char walltext[ TEXTURENAMESIZE];
	char doortext[ TEXTURENAMESIZE];
	char skytext[ TEXTURENAMESIZE];
	char floortext[ TEXTURENAMESIZE];
	char ceilingtext[ TEXTURENAMESIZE];
	char wadname[ WADNAMESIZE];

} CFG;


typedef struct SIXSIDES
{
	int x1, x2, y1, y2, z1, z2;     /* Quake coordinates of brush */
	int texture; 					/* wall, door, sky, etc. */
} BRUSH;

typedef struct EXTREMES
{
	int xbase, ybase;
	int minx, maxx;
	int miny, maxy;
} QEXTREMES;


typedef struct tagBITMAPFILEHEADER
{
	WORD bfType;
	DWORD bfSize;
	WORD bfReserved1;
	WORD bfReserved2;
	DWORD bfOffBits;

} BITMAPFILEHEADER;


typedef struct tagBITMAPINFOHEADER
{
	DWORD biSize;
	DWORD biWidth;
	DWORD biHeight;
	WORD  biPlanes;
	WORD  biBitCount;
	DWORD biCompression;
	DWORD biSizeImage;
	DWORD biXPelsPerMeter;
	DWORD biYPelsPerMeter;
	DWORD biClrUsed;
	DWORD biClrImportant;

} BITMAPINFOHEADER;


typedef struct tagRGBQUAD
{
	BYTE rgbBlue;
	BYTE rgbGreen;
	BYTE rgbRed;
	BYTE rgbReserved;

} RGBQUAD;


/*** initcfg() - initialize configuration
***/

void initcfg( CFG *cfg)
{
	FILE *fp;
	char line[ CFGLINESIZE], *opt, *val;
	int ival;

	printf("Initializing configuration from %s.\n", CFGNAME);
	cfg->wallval = DEFWALLVAL;
	cfg->winval = DEFWINVAL;
	cfg->doorval = DEFDOORVAL;
	cfg->skywinval = DEFSKYWINVAL;
	cfg->qunitsperpel = DEFQUNITSPERPEL;
	cfg->floor = DEFFLOOR;
	cfg->ceiling = DEFCEILING;
	cfg->winbottom = DEFWINBOTTOM;
	cfg->wintop = DEFWINTOP;
	cfg->doortop = DEFDOORTOP;
	cfg->autolight = DEFAUTOLIGHT;
	cfg->autolightqunits = DEFAUTOLIGHTQU;
	strcpy( cfg->walltext, DEFWALLTEXT);
	strcpy( cfg->doortext, DEFDOORTEXT);
	strcpy( cfg->skytext, DEFSKYTEXT);
	strcpy( cfg->floortext, DEFFLOORTEXT);
	strcpy( cfg->ceilingtext, DEFCEILINGTEXT);
	strcpy( cfg->wadname, DEFWADNAME);

	if( ( fp = fopen( CFGNAME, "r")) == NULL)
	{
		printf("Unable to open %d for configuration info.\n");
		printf("Using program defaults...\n");
	}
	else
	{
		while( fgets( line, CFGLINESIZE, fp))
		{
			opt = strtok( line, "=");
			val = strtok( NULL, "\n");
			ival = atoi( val);
#ifdef DEBUG
			printf("Option: %s (%02x)  Value: %s (%02x)", opt,val, *opt, *val);
#endif

			if( ! strcmp( opt, CFGQUNITSPERPEL))
				cfg->qunitsperpel = ival;
			else if( ! strcmp( opt, CFGFLOOR))
				cfg->floor = ival;
			else if( ! strcmp( opt, CFGCEILING))
				cfg->ceiling = ival;
			else if( ! strcmp( opt, CFGWINBOTTOM))
				cfg->winbottom = ival;
			else if( ! strcmp( opt, CFGWINTOP))
					cfg->wintop = ival;
			else if( ! strcmp( opt, CFGDOORTOP))
				cfg->doortop = ival;
			else if( ! strcmp( opt, CFGWALLVAL))
				cfg->wallval = ival;
			else if( ! strcmp( opt, CFGWINVAL))
				cfg->winval = ival;
			else if( ! strcmp( opt, CFGDOORVAL))
				cfg->doorval = ival;
			else if( ! strcmp( opt, CFGSKYWINVAL))
				cfg->skywinval = ival;
			else if( ! strcmp( opt, CFGAUTOLIGHT))
				cfg->autolight = ival;
			else if( ! strcmp( opt, CFGAUTOLIGHTQU))
				cfg->autolightqunits = ival;
			else if( ! strcmp( opt, CFGWALLTEXT))
				strncpy( cfg->walltext, val, TEXTURENAMESIZE);
			else if( ! strcmp( opt, CFGSKYTEXT))
				strncpy( cfg->skytext, val, TEXTURENAMESIZE);
			else if( ! strcmp( opt, CFGDOORTEXT))
				strncpy( cfg->doortext, val, TEXTURENAMESIZE);
			else if( ! strcmp( opt, CFGFLOORTEXT))
				strncpy( cfg->floortext, val, TEXTURENAMESIZE);
			else if( ! strcmp( opt, CFGCEILINGTEXT))
				strncpy( cfg->ceilingtext, val, TEXTURENAMESIZE);
			else
			{
				printf("Unknown option ignored-> %s=%s\n", opt, val);
				printf("\nPress any key to continue...\n");
				getch();
			}
		}

	}

	printf("          Quake units per pixel: %d\n", cfg->qunitsperpel);
	printf("                          Floor: %d\n", cfg->floor);
	printf("                        Ceiling: %d\n", cfg->ceiling);
	printf("                  Window bottom: %d\n", cfg->winbottom);
	printf("                     Window top: %d\n", cfg->wintop);
	printf("                       Door top: %d\n\n",cfg->doortop);

	printf("        Palette index for walls: %d\n",cfg->wallval);
	printf("Palette index for blank windows: %d\n",cfg->winval);
	printf("        Palette index for doors: %d\n",cfg->doorval);
	printf("Palette index for skyed windows: %d\n",cfg->skywinval);
	printf("     Wall/Ceiling/Floor texture: %s\n",cfg->walltext);
	printf("                    Sky texture: %s\n",cfg->skytext);
	printf("                   Door texture: %s\n\n",cfg->doortext);
	printf("                  Auto lighting: %d\n",cfg->autolight);
	printf("  Auto light interval in Qunits: %d\n\n", cfg->autolightqunits);

	/* do some basic checking on options selected */
	if( cfg->qunitsperpel < 2)
	{
		printf("\n Error: Minimum of two quake units per pixel. \n");
		exit( 1);
	}


	if( cfg->floor >= cfg->winbottom)
	{
		printf("\n Error: Floor is above window bottom. \n");
		exit( 1);
	}
	else if( cfg->winbottom >= cfg->wintop)
	{
		printf("\n Error: Window bottom is above window top. \n");
		exit( 1);
	}
	else if( cfg->wintop >= cfg->ceiling)
	{
		printf("\n Error: Window top is above ceiling. \n");
		exit( 1);
	}
	else if( cfg->floor >= cfg->doortop)
	{
		printf("\n Error: Floor is above door top. \n");
		exit( 1);
	}
	else if( cfg->doortop >= cfg->ceiling)
	{
		printf("\n Error: Doortop is above ceiling. \n");
		exit( 1);
	}
	else if( cfg->floor >= cfg->ceiling)
	{
		printf("Don't know how you got past the window checks,but...\n");
		printf("\n Error: Floor is above the ceiling\n");
		exit( 1);
	}
	fclose( fp);
	return;
} /*** initcfg() ***/


/*** savebrush() - save coorinates and texture for brush
***/
void savebrush( FILE *fp, QEXTREMES *qe, int *brushcnt,
				int x1, int x2, int y1, int y2,
				int z1, int z2, int texture)
{
	BRUSH b;

	/* track extremes */
	if( x1 < qe->minx) qe->minx = x1;
	if( x2 > qe->maxx) qe->maxx = x2;
	if( y1 < qe->miny) qe->miny = y1;
	if( y2 > qe->maxy) qe->maxy = y1;

	b.x1 = x1; b.x2 = x2;
	b.y1 = y1; b.y2 = y2;
	b.z1 = z1; b.z2 = z2;
	b.texture = texture;

	fwrite( &b, sizeof(BRUSH), 1, fp);
	++(*brushcnt);
	return;
} /*** savebrush() ***/


/*** boundmap() - write brushes for box around map (make map leak-roof)
***/
void boundmap( CFG *cfg, FILE *mp1fp, QEXTREMES *qe, int *brushcnt)
{
	/* write floor and ceiling brushes */
	/* Z */
	savebrush( mp1fp, qe, brushcnt,
		qe->minx, qe->maxx,
		qe->miny, qe->maxy,
		cfg->ceiling, cfg->ceiling + cfg->qunitsperpel,
		CEILINGBRUSH);

	savebrush( mp1fp, qe, brushcnt,
		qe->minx, qe->maxx,
		qe->miny, qe->maxy,
		cfg->floor - cfg->qunitsperpel, cfg->floor,
		FLOORBRUSH);

	/* aahhh, heck... surround the whole damn thing (leakproof) */
	/* X */
	savebrush( mp1fp, qe, brushcnt,
		qe->minx - cfg->qunitsperpel, qe->minx,
		qe->miny, qe->maxy,
		cfg->floor - cfg->qunitsperpel, cfg->ceiling + cfg->qunitsperpel,
		SKYBRUSH);

	savebrush( mp1fp, qe, brushcnt,
		qe->maxx, qe->maxx + cfg->qunitsperpel,
		qe->miny, qe->maxy,
		cfg->floor - cfg->qunitsperpel, cfg->ceiling + cfg->qunitsperpel,
		SKYBRUSH);


	/* Y */
	savebrush( mp1fp, qe, brushcnt,
		qe->minx, qe->maxx,
		qe->miny - cfg->qunitsperpel, qe->miny,
		cfg->floor - cfg->qunitsperpel, cfg->ceiling + cfg->qunitsperpel,
		SKYBRUSH);

	savebrush( mp1fp, qe, brushcnt,
		qe->minx, qe->maxx,
		qe->maxy, qe->maxy + cfg->qunitsperpel,
		cfg->floor - cfg->qunitsperpel, cfg->ceiling + cfg->qunitsperpel,
		SKYBRUSH);

	return;
} /*** boundmap() ***/


/*** pel2brush() - calculate Quake units given pixels location in .BMP
***/

void pel2brush( CFG *cfg, FILE *fp, QEXTREMES *qe, int *brushcnt,
				int x, int y, int zbot, int ztop, int texture)
{
	int qx1, qx2, qy1, qy2;

	/* calculate quake coordinates from bitmap coordinates */
	qx1 = qe->xbase + cfg->qunitsperpel * x;
	qx2 = qe->xbase + cfg->qunitsperpel * (x+1);
	qy1 = qe->ybase + cfg->qunitsperpel * y;
	qy2 = qe->ybase + cfg->qunitsperpel * (y+1);

	savebrush( fp, qe, brushcnt, qx1, qx2, qy1, qy2, zbot, ztop, texture);
	return;
} /*** pel2brush() ***/



/*** initbmp() - read bitmap header and palette
***/
void initbmp( FILE *bmpfp, BITMAPINFOHEADER *bmih)
{
	BITMAPFILEHEADER bmfh; /* bit map file header */
	RGBQUAD rgbq;          /* RGBx foursome for palette */
	int i;

	/* read file header */
	if( fread( &bmfh, sizeof( BITMAPFILEHEADER), 1, bmpfp) != 1)
	{
		printf("Error reading .BMP.\n");
		exit(1);
	}

	/* check to see if it is a valid bitmap file */
	if( bmfh.bfType != MUSTBETYPE)
	{
		printf(".BMP is not a valid bitmap.\n");
		exit(1);
	}

	if( bmfh.bfReserved1 || bmfh.bfReserved2)
	{
		printf("Reserved fields are not 0?\n");
		exit(1);

	}
	printf(".BMP is %ld bytes long.  Image offset is %ld.\n",
		bmfh.bfSize, bmfh.bfOffBits);

	/* read info header */
	if( fread( bmih, sizeof( BITMAPINFOHEADER), 1, bmpfp) != 1)
	{
		printf("Error reading .BMP\n");
		exit(1);
	}

	if( bmih->biPlanes != 1)
	{
		printf(".BMP has incorrect value (%d) for biPlanes.\n",
			bmih->biPlanes);
		exit(1);
	}

	if( bmih->biBitCount != 8)
	{
		printf("Not a 8-bit (256 color) .BMP\n");
		exit(1);
	}
	if( bmih->biCompression != BI_RGB)
	{
		printf("Only uncompressed .BMPs allowed.\n");
		exit(1);
	}

	printf("Header: %ld bytes   Image: %ld bytes\n",
		bmih->biSize, bmih->biSizeImage);
	printf("%ld x %ld pixels w/ %ld colors, %ld of which are \"important\"\n",
		bmih->biWidth, bmih->biHeight, bmih->biClrUsed, bmih->biClrImportant);
	printf("Pixels/Meter = %ld x %ld\n",
		bmih->biXPelsPerMeter, bmih->biYPelsPerMeter);

	for( i = 0; i < bmih->biClrUsed; i++)
	{
		if(	fread( &rgbq, sizeof(RGBQUAD), 1, bmpfp) != 1)
		{
			printf("Error reading palette information.\n");
			exit(1);
		}
#ifdef DEBUG
		printf("Index: %02x   Red: %02x  Green: %02x  Blue:%02x\n",
			i, rgbq.rgbRed, rgbq.rgbGreen, rgbq.rgbBlue);
#endif
	}
	return;
} /*** initbmp() ***/


/*** bmp2mp1() - read bitmap recording brushes encountered
				 returns the number of brushes recorded
***/
int bmp2mp1( char *filename, CFG *cfg, QEXTREMES *qe)
{
	FILE *bmpfp, *mp1fp;
	int x, y;
	BYTE pel;
	WORD xWidth;
	char bmpfile[80], mp1file[80];
	BITMAPINFOHEADER bmih; /* bit map information header */
	int brushcnt = 0; 	/* initialize counter for counting brushes */

	/* prepare file names */
	strcpy( bmpfile, filename);
	strcat( bmpfile, BMPEXT);
	strcpy( mp1file, filename);
	strcat( mp1file, MP1EXT);

	/* open the file */
	printf("Reading file %s\n", bmpfile);
	if ((bmpfp = fopen(bmpfile,"rb")) == NULL)
	{
		printf("Error opening file %s.\n",bmpfile);
		exit(1);
	}

	/* read bitmap information header */
	initbmp( bmpfp, &bmih);

	/* open output file */
	if( ( mp1fp = fopen( mp1file, "wb")) == NULL)
	{
		printf("Unable to open file %s for output.\n", mp1file);
		exit(1);
	}

	/* calculate base quake coordinates so map is centered at origin */
	qe->xbase = ((int) bmih.biWidth/2) * cfg->qunitsperpel;
	qe->xbase += cfg->qunitsperpel - (qe->xbase%cfg->qunitsperpel);
	qe->xbase = - (qe->xbase);

	qe->ybase = ((int) bmih.biHeight/2) * cfg->qunitsperpel;
	qe->ybase += cfg->qunitsperpel - (qe->ybase%cfg->qunitsperpel);
	qe->ybase = - (qe->ybase);

	qe->minx = EXTREME;		/* Smallest X (in Quake units) encountered */
	qe->maxx = -EXTREME;		/* Largest "" */
	qe->miny = EXTREME;		/* Like above for Y */
	qe->maxy = -EXTREME;		/* "" */


	/* calculate number of bytes for each line (padding to `long's) */
	xWidth = bmih.biWidth;
	if( xWidth % 4) xWidth += 4 - (xWidth%4);
	for( y = 0; y < bmih.biHeight; y++)
	{
		printf(".");  /* progress indicator - one per line read */
		for( x = 0; x < xWidth; x++)
		{
			pel = (BYTE) fgetc(bmpfp);

			/* if not padding */
			if( x < bmih.biWidth)
			{
				/* write brush depending upon pixels color */
				if( pel == cfg->wallval)
				{
					pel2brush( cfg, mp1fp, qe, &brushcnt,
						x, y, cfg->floor, cfg->ceiling, WALLBRUSH);
				}
				else if( pel == cfg->winval)
				{
					pel2brush( cfg, mp1fp, qe, &brushcnt,
						x, y, cfg->floor, cfg->winbottom, WALLBRUSH);
					pel2brush( cfg, mp1fp, qe, &brushcnt,
						x, y, cfg->wintop, cfg->ceiling, WALLBRUSH);
				}
				else if( pel == cfg->doorval)
				{
					pel2brush( cfg, mp1fp, qe, &brushcnt,
						x, y, cfg->floor, cfg->doortop, DOORBRUSH);
					pel2brush( cfg, mp1fp, qe, &brushcnt,
						x, y, cfg->doortop, cfg->ceiling, WALLBRUSH);
				}
				else if( pel == cfg->skywinval)
				{
					pel2brush( cfg, mp1fp, qe, &brushcnt,
						x, y, cfg->floor, cfg->winbottom, WALLBRUSH);
					pel2brush( cfg, mp1fp, qe, &brushcnt,
						x, y, cfg->winbottom, cfg->wintop, SKYBRUSH);
					pel2brush( cfg, mp1fp, qe, &brushcnt,
						x, y, cfg->wintop, cfg->ceiling, WALLBRUSH);
				}
			}
		}
	}

	printf("\nX: %d to %d  by Y: %d to %d\n",
		qe->minx, qe->maxx, qe->miny, qe->maxy);
	fclose( bmpfp);
	boundmap( cfg, mp1fp, qe, &brushcnt); /* seal map in a box (leak-proof) */
	fclose( mp1fp);
	printf("%d brushes written to %s.\n", brushcnt, mp1file);
	return( brushcnt);
} /*** bmp2mp1() ***/


/*** printbrush() - for debugging purposes
***/
void printbrush( int seq, BRUSH *b)
{
	printf("%d - X: %d to %d  Y: %d to %d  Z: %d to %d  Texture: %d\n",
		seq, b->x1, b->x2, b->y1, b->y2, b->z1, b->z2, b->texture);
	return;
} /*** printbrush() ***/


/*** mp12mp2() - join similarly aligned brushes and write to new temp file
***/
void mp12mp2( char *filename, int brushcnt)
{
	FILE *mp1fp, *mp2fp;
	char mp1file[80], mp2file[80];
	register BRUSH huge *b;			/* must be HUGE on 8088 */
	int newcnt;				/* number of brushes after last reduction */
	unsigned long bsize;
	int i, j;
	int done;
	int savetexture; /* to check for memory segment overlap */

	/* prepare filenames */
	strcpy( mp1file, filename);
	strcat( mp1file, MP1EXT);
	strcpy( mp2file, filename);
	strcat( mp2file, MP2EXT);

	/* open the file */
	printf("Reading file %s\n", mp1file);
	if ((mp1fp = fopen(mp1file,"rb")) == NULL)
	{
		printf("Error opening file %s.\n",mp1file);
		exit(1);
	}

	/* let user know of memory requirements */
	printf("%d brushes\n", brushcnt);
	bsize = brushcnt;
	bsize *= sizeof(BRUSH *);

	printf("%ld bytes of available RAM.\n", farcoreleft());
	printf("Brush structure is %d bytes\n", sizeof(BRUSH));
	printf("%ld bytes needed for brush data.\n",
		(long) brushcnt*sizeof(BRUSH));

	/* allocate memory from far heap */
	if( (b = (BRUSH huge *) farmalloc( (long) brushcnt*sizeof(BRUSH))) == NULL)
	{
		printf("Unable to allocate memory for brushes.\n");
		printf("%ld bytes of available RAM.\n", farcoreleft());
		exit(1);
	}
	printf("%ld bytes of available RAM.\n", farcoreleft());

	/* load brushes into memory */
	for( i = 0; i < brushcnt; i++)
	{
		fread( (void *)(b+i), sizeof(BRUSH), 1, mp1fp);

		/*** debug code to detect 64K segment lapping ***/
		if( !i)
		{
			savetexture = b[0].texture;
		}
		else
		{
			if( b[0].texture != savetexture)
			{
				printf("First brush was trashed on reading brush %d\n", i);
				exit(1);
			}
		}
#ifdef DEBUG
		if( !(i%1000)) printbrush( i, b+i);
#endif
	}
	fclose( mp1fp);
	printf("Deleting temporary file %s.\n", mp1file);
	remove( mp1file);  /* delete temp file */

	printf("Combining brushes.\n");
	done = FALSE;
	while( ! done)
	{
		for( i = 0; i < (brushcnt-1); i++)
		{
#ifdef DEBUG
			if( !(i%1000)) printbrush( i, b+i);
#endif
			if( ! (i%100)) printf("."); /* progress indicator */
			if( b[i].texture > 0)  /* if brush not combined already */
			{
				for( j = i+1; j < brushcnt; j++)
				{

					/* if y's and z's line up and same texture */
					if( (b[i].y1 == b[j].y1) && (b[i].y2 == b[j].y2)
						&& (b[i].z1 == b[j].z1) && (b[i].z2 == b[j].z2)
						&& (b[i].texture == b[j].texture))
					{
						/* if common x's then combine */
						if( b[i].x1 == b[j].x2)
						{
							b[i].x1 = b[j].x1;
							b[j].texture = ABSORBED;
						}
						else if( b[i].x2 == b[j].x1)
						{
							b[i].x2 = b[j].x2;
							b[j].texture = ABSORBED;
						}
					}
					/* else if x's and z's line up and same texture */
					else if( (b[i].x1 == b[j].x1) && (b[i].x2 == b[j].x2)
						&& (b[i].z1 == b[j].z1) && (b[i].z2 == b[j].z2)
						&& (b[i].texture == b[j].texture))
					{
						/* if common y's then combine */
						if( b[i].y1 == b[j].y2)
						{
							b[i].y1 = b[j].y1;
							b[j].texture = ABSORBED;
						}
						else if( b[i].y2 == b[j].y1)
						{
							b[i].y2 = b[j].y2;
							b[j].texture = ABSORBED;
						}
					}

				}
			}
			/* quick check for memory segment overlap */
			if( (b[i].texture != ABSORBED) &&
				( ( b[i].texture < MINBRUSH) || ( b[i].texture > MAXBRUSH)))
			{
				printf("Invalid texture at brush %d\n", i);
				exit( 1);
			}
		}
		printf("\n"); /* end of progress indicator */

		/* compress and count */
		newcnt = 0;
		for( i = 0; i < brushcnt; i++)
		{
#ifdef DEBUG
		if( !(i%1000)) printbrush( i, b+i);
#endif

			if( b[i].texture != ABSORBED)
			{
				if( i != newcnt)
				{
					memcpy( ( void *)( b + newcnt), ( void *) ( b + i),
						sizeof(BRUSH));
				}
				newcnt++;
			}
		}
		printf("Brush count has been reduced to %d.\n", newcnt);
		if( newcnt == brushcnt) done = TRUE;
		brushcnt = newcnt;

	}

	if( (mp2fp = fopen(mp2file,"wb")) == NULL)
	{
		printf("Unable to open file %s for output.\n", mp2file);
		exit(1);
	}


	for( i = 0; i < brushcnt; i++)
	{
		if( b[i].texture != ABSORBED)
		{
#ifdef DEBUG
		if( !(i %1000)) printbrush( i, b+i);
#endif

			fwrite( ( void *) ( b + i), sizeof(BRUSH), 1, mp2fp);
		}
	}
	printf("%d brushes written to file %s.\n", brushcnt, mp2file);
	fclose( mp2fp);
	farfree( ( void *) b);
	return;
} /*** mp12mp2() ***/


/*** writebrush() - given coorinates bounding a box, write
					a brush for the box using texture specified.
***/
void writebrush( FILE *fp, int x1, int x2, int y1, int y2,
			int z1, int z2, char *texture)
{
	/* begin brush */
	fprintf( fp, "{\n");

	/* top */
	fprintf( fp,
		"( %d %d %d ) ( %d %d %d ) ( %d %d %d ) %s 0 0 0 1.000000 1.000000\n",
		x1, y2, z2, x2, y2, z2, x2, y1, z2, strupr( texture));

	/* bottom */
	fprintf( fp,
		"( %d %d %d ) ( %d %d %d ) ( %d %d %d ) %s 0 0 0 1.000000 1.000000\n",
		x1, y1, z1, x2, y1, z1, x2, y2, z1, strupr( texture));

	/* left */
	fprintf( fp,
		"( %d %d %d ) ( %d %d %d ) ( %d %d %d ) %s 0 0 0 1.000000 1.000000\n",
		x1, y2, z2, x1, y1, z2, x1, y1, z1, strupr( texture));

	/* right */
	fprintf( fp,
		"( %d %d %d ) ( %d %d %d ) ( %d %d %d ) %s 0 0 0 1.000000 1.000000\n",
		x2, y2, z1, x2, y1, z1, x2, y1, z2, strupr( texture));

	/* back */
	fprintf( fp,
		"( %d %d %d ) ( %d %d %d ) ( %d %d %d ) %s 0 0 0 1.000000 1.000000\n",
		x2, y2, z2, x1, y2, z2, x1, y2, z1, strupr( texture));

	/* front */
	fprintf( fp,
		"( %d %d %d ) ( %d %d %d ) ( %d %d %d ) %s 0 0 0 1.000000 1.000000\n",
		x2, y1, z1, x1, y1, z1, x1, y1, z2, strupr( texture));

	/* finish brush */
	fprintf( fp, "}\n");
	return;
} /*** writebrush() ***/


/*** mp22map() - convert compressed brushes into Quake .MAP file
***/
void mp22map( char *filename, CFG *cfg, QEXTREMES *qe)
{
	FILE *mp2fp, *mapfp;
	int brushcnt, lightcnt;
	char mp2file[80], mapfile[80];
	BRUSH b;
	int x,y;


	strcpy( mp2file, filename);
	strcat( mp2file, MP2EXT);
	strcpy( mapfile, filename);
	strcat( mapfile, MAPEXT);


	/* open the file */
	printf("Reading file %s\n", mp2file);
	if ((mp2fp = fopen(mp2file,"rb")) == NULL)
	{
		printf("Error opening file %s.\n",mp2file);
		exit(1);
	}


	/* open output file */
	if( ( mapfp = fopen( mapfile, "w")) == NULL)
	{
		printf("Unable to open file %s for output.\n", mapfile);
		exit(1);
	}
	/* prepare to write map */
	fprintf( mapfp, "{\n\"classname\" \"worldspawn\"\n\"wad\" \"%s\"\n",
		cfg->wadname);


	brushcnt = 0;
	while( fread( &b, sizeof(BRUSH),1,mp2fp))
	{
#ifdef DEBUG
		if(!( brushcnt%100)) printbrush( brushcnt, &b);
#endif
		switch( b.texture)
		{
			case WALLBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, cfg->walltext);
				break;
			case DOORBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, cfg->doortext);
				break;
			case SKYBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, cfg->skytext);
				break;
			case FLOORBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, cfg->floortext);
				break;
			case CEILINGBRUSH:
				writebrush( mapfp, b.x1, b.x2, b.y1, b.y2, b.z1, b.z2, cfg->ceilingtext);
				break;
			default:
				printf("Unknown texture (%d) encountered... brush ignored\n",
					b.texture);
				brushcnt--;
		}
		brushcnt++;
	}

	fclose( mp2fp);
	printf("Deleting temporary file %s.\n", mp2file);
	remove( mp2file); /* delete temporary file */

	/* finish brush section */
	fprintf( mapfp, "}\n");

	/* put start at origin (more entities later?) */
	fprintf( mapfp,
	"{\n\"classname\" \"info_player_start\"\n\"origin\" \"0 0 0\"\n}\n");

	/* if autolight write light entities using interval specified */
	lightcnt=0;
	if( cfg->autolight)
	{
		for( x = qe->minx+cfg->autolightqunits; x < qe->maxx; x += cfg->autolightqunits)
		{
			for( y = qe->miny+cfg->autolightqunits; y < qe->maxy; y+=cfg->autolightqunits)
			{
				fprintf( mapfp,
				"{\n\"classname\" \"light\"\n\"origin\" \"%d %d %d\"\n}\n",
				x,y,cfg->doortop);
				lightcnt++;
			}
		}
	}

	fclose( mapfp);
	printf("\nDone... %d brushes and %d lights written to %s.\n\n",
		brushcnt, lightcnt, mapfile);
	return;
} /*** mp22map() ***/


/*** main() - convert 256 color .BMP to Quake .MAP file
***/
void main( int argc, char **argv)
{
	CFG cfg;
	char filename[80];
	int brushcnt;
	QEXTREMES qe;

	/* initialize configuration */
	initcfg( &cfg);

	/* prepare file names */
	if( argc < 2)
	{
		printf("No .BMP file specified on command line\n");
		exit(1);
	}

	strcpy( filename, argv[1]);
	strtok( filename, ".\n");

	brushcnt = bmp2mp1( filename, &cfg, &qe); /* convert pixels to brushes */
	mp12mp2( filename, brushcnt); /* combine similarly aligned brushes */
	mp22map( filename, &cfg, &qe); /* write brushes in .MAP format */
	exit(0);
} /*** main() ***/
/*** EOF ***/
