// 
// maplist.c -- server maplist rotation/manipulation routines 
// 
// 
// 1/98 - L. Allan Campbell (Geist) 
// 

// INCLUDES ///////////////////////////////////////////////// 

#include "g_local.h" 
void WFFillMapNames();
       


// FUNCTIONS //////////////////////////////////////////////// 

void fixline(char *str)
{
	int i;
	i = 0;
	while (str[i] != 0)
	{
		if (str[i] >=128) str[i] = 0;
		if (str[i] < 32 && str[i] != '[' && str[i] != ']') str[i] = 0;
//gi.dprintf("<%d>", str[i]);
		++i;
	}
}

//Read a line from a file
void readline(FILE *file, char *str, int max)
{
	int i;
	int enough;
	unsigned char c;

	i = 0;
	enough = 0;
	while (i < max && !feof(file) && enough == 0)
	{
		c = fgetc(file);
		if (c >= ' ') str[i++] = c;
		if (c == 13 || c == 10) enough = 1;
//gi.dprintf("{%d}",c);

	}
	str[i] = 0;

	--i;
	//now trim back the white space
	while (i >= 0 && str[i] == ' ')
	{
		str[i] = 0;
		--i;
	}
}

// 
// LoadMapList 
// 
// Opens the specified file and scans/loads the maplist names 
// from the file's [maplist] section. (list is terminated with 
// "###") 
// 
// Args: 
//   ent      - entity (client) to print diagnostic messages to. 
//   filename - name of file containing maplist. 
// 
// Return: 0 = normal exit, maplist loaded 
//         1 = abnormal exit 
// 
int LoadMapList(edict_t *ent, char *filename) 
{ 
   FILE *fp; 
   int  i=0; 
   char szLineIn[80];
   char sMapName[80];
   char sVoteOnly[80];

    fp = OpenFile(ent, filename); 
    if (fp)  // opened successfully? 
   { 
      // scan for [maplist] section 
      do 
      { 
         fscanf(fp, "%s", szLineIn); 
      } while (!feof(fp) && (Q_stricmp(szLineIn, "[maplist]") != 0)); 
       if (feof(fp)) 
      { 
         // no [maplist] section 
         gi.dprintf ("-------------------------------------\n"); 
         gi.dprintf ("ERROR - No [maplist] section in \"%s\".\n", filename); 
         gi.dprintf ("-------------------------------------\n"); 
      } 
      else 
      { 
         gi.dprintf ("-------------------------------------\n"); 
  
         // read map names into array 
         while ((!feof(fp)) && (i<MAX_MAPS)) 
         { 
			readline(fp, szLineIn, 80);
			fixline(szLineIn);
            sMapName[0] = 0;
            sVoteOnly[0] = 0;
            sscanf(szLineIn, "%s %s", sMapName, sVoteOnly ); 
			if (szLineIn[0] == 0)
				continue;

//gi.dprintf("Map In: %s [%s]\n", sMapName, sVoteOnly);

            if (Q_stricmp(sMapName, "###") == 0)  // terminator is "###" 
               break;
			if (sMapName[0] == 0)
				continue;

			// TODO: check that maps exist before adding to list 
			//       (might be difficult to search a .pak file for these) 
			strncpy(maplist.mapnames[i], sMapName, MAX_MAPNAME_LEN);
			if (sVoteOnly[0] == 'v' || sVoteOnly[0] == 'V')
				maplist.voteonly[i] = true;
			else
				maplist.voteonly[i] = false;
			gi.dprintf("...%s\n", maplist.mapnames[i]); 
            i++; 
         } 
      } 
       CloseFile(ent, fp); 
       if (i == 0) 
      { 
         gi.dprintf ("No maps listed in [maplist] section of %s\n", filename);
         gi.dprintf ("-------------------------------------\n"); 
         return 0;  // abnormal exit -- no maps in file 
      } 
      gi.dprintf ("%i map(s) loaded.\n", i); 
      gi.dprintf ("-------------------------------------\n"); 
      maplist.nummaps = i; 
      return 1;     // normal exit 
   } 
  
   return 0;  // abnormal exit -- couldn't open file 
} 
  
//Clear the map votes
void ClearMapVotes() 
{ 
	int i;
	for (i=0; i < MAX_MAPS; ++i)
		maplist.votes[i] = 0;

} 

//Find highest voted map
/*
  Returns
    -1  No votes
	0-31 Index to selected map  
*/
int MapMaxVotes() 
{ 
	int i;
	int numvotes;
	int index;

	numvotes = 0;
	index = -1;
	i = 0;

	while (i < maplist.nummaps)
	{
		if (maplist.votes[i] > numvotes)
		{
		numvotes = maplist.votes[i];
		index = i;
		}
		++i;
	}
	return(index);
} 

void VoteForMap(int i)
{
	if (i >= 0 && i < maplist.nummaps)
		++maplist.votes[i];
}

void DumpMapVotes()
{
	int i;
	for (i = 0; i < maplist.nummaps; ++i)
		gi.dprintf("%d. %s (%d votes)\n",
		   i, maplist.mapnames[i], maplist.votes[i]);
}


 // 
// ClearMapList 
// 
// Clears/invalidates maplist. Might add more features in the future, 
// but resetting .nummaps to 0 will suffice for now. 
// 
// Args: 
//   ent      - entity (client) to print diagnostic messages to (future development). 
// 
// Return: (none) 
// 
void ClearMapList() 
{ 
   maplist.nummaps = 0; 
   ClearMapVotes();
} 
  

 // 
// DisplayMaplistUsage 
// 
// Displays current command options for maplists. 
// 
// Args: 
//   ent      - entity (client) to display help screen (usage) to. 
// 
// Return: (none) 
// 
void DisplayMaplistUsage(edict_t *ent) 
{ 
   gi.dprintf ("-------------------------------------\n"); 
   gi.dprintf ("usage:\n"); 
   gi.dprintf ("SV MAPLIST <filename> [<rot_flag>]\n"); 
   gi.dprintf ("  <filename> - server ini file\n"); 
   gi.dprintf ("  <rot_flag> - 0 = sequential (def)\n"); 
   gi.dprintf ("               1 = random\n"); 
   gi.dprintf ("SV MAPLIST START to go to 1st map\n"); 
   gi.dprintf ("SV MAPLIST NEXT to go to next map\n"); 
   gi.dprintf ("SV MAPLIST to display current list\n"); 
   gi.dprintf ("SV MAPLIST OFF to clear/disable\n"); 
   gi.dprintf ("SV MAPLIST HELP for this screen\n"); 
   gi.dprintf ("-------------------------------------\n"); 
} 

// MaplistNextMap
// Choose the next map in the list, or use voting system
void MaplistNextMap(edict_t *ent)
{ 
	int votemap;
	int i;
	int j;

	DumpMapVotes();

// Jason - j is so we alter the struct member currentmaps in the loops.

	j = maplist.currentmap;

	switch (maplist.rotationflag)        // choose next map in list 
	{ 
	case ML_ROTATE_SEQ:        // sequential rotation

// Jason - If there is only one map don't infinate loop

		if (maplist.nummaps > 1)
		{

// Jason - do while loops chooses next map and obeys the voteonly array
		
			do
			{
				i = (j + 1) % maplist.nummaps; 
				j++;
			}
			while (maplist.voteonly[i] == true);
		}
		else
			i = 0;
		break; 

	case ML_ROTATE_RANDOM:     // random rotation 
		if (maplist.nummaps > 1)
		{
			do
			{
				i = (int) (random() * maplist.nummaps); 
			}
			while ((maplist.voteonly[i] == true) || (i == j));
		}
		else
			i = 0;
		break; 

	default:       // should never happen, but set to first map if it does 
		i=0; 
	} // end switch 

	//See if map voting is on
	if ((int)wfflags->value & WF_MAP_VOTE)
	{
		votemap = MapMaxVotes();
		if (votemap >= 0)	//Yes there was one picked
			i = votemap;
		ClearMapVotes();
	}

	maplist.currentmap = i; 
	ent->map = maplist.mapnames[i]; 
} 


  
// 
// Cmd_Maplist_f 
// 
// Main command line parsing function. Enables/parses/diables maplists. 
// 
// Args: 
//   ent      - entity (client) to display messages to, if necessary. 
// 
// Return: (none) 
// 
// TODO: change "client 0" privs to be for server only, if dedicated. 
//       only allow other clients to view list and see HELP screen. 
//       (waiting for point release for this feature) 
// 
#define MLIST_ARG_1	2
#define MLIST_ARG_2	3
#define MLIST_ARG_3	4


void Cmd_Maplist_f (edict_t *ent) 
{ 
   int  i;    // looping and temp variable 
   int argcount;
   char *filename; 

   argcount = gi.argc() - 1;
//gi.dprintf("argcount = %d, arg1 = %s, arg2 = %s\n", argcount, gi.argv(MLIST_ARG_1),
//		   gi.argv(MLIST_ARG_2)); 

   switch (argcount) 
   { 
   case 2:  // various commands, or enable and assume rotationflag default 
//gi.dprintf("arg count 2\n");
      if (Q_stricmp(gi.argv(MLIST_ARG_1), "HELP") == 0) 
      { 
         DisplayMaplistUsage(ent); 
         break; 
      } 
       if (Q_stricmp(gi.argv(MLIST_ARG_1), "START") == 0) 
      { 
		if (maplist.nummaps > 0)  // does a maplist exist? 
			EndDMLevel(); 
		else 
		{
			gi.dprintf("Can't start - No maps in current list\n");
			DisplayMaplistUsage(ent); 
		}
		break; 
      } 
      else if (Q_stricmp(gi.argv(MLIST_ARG_1), "NEXT") == 0) 
      { 
         if (maplist.nummaps > 0)  // does a maplist exist? 
            EndDMLevel(); 
         else 
		 {
			gi.dprintf("Can't do next - No maps in current list\n");
            DisplayMaplistUsage(ent); 
		 }
          break; 
      } 
      else if (Q_stricmp(gi.argv(MLIST_ARG_1), "OFF") == 0) 
      { 
         if (maplist.nummaps > 0)  // does a maplist exist? 
         { 
            ClearMapList(); 
            wfadmin->value = (int) wfadmin->value & ~WF_ADMIN_MAP_LIST; 
            gi.dprintf ("Maplist cleared/disabled.\n"); 
         } 
         else 
         { 
            // maplist doesn't exist, so display usage 
			gi.dprintf("No maps in current list\n");
            DisplayMaplistUsage(ent); 
         } 
          break; 
      } 
      else 
	  {
         maplist.rotationflag = 0; 
	  }
  
        // no break here is intentional;  supposed to fall though to case 3 
    case 3:  // enable maplist - all args explicitly stated on command line 
//gi.dprintf("arg count 3\n");
	  if (gi.argc() == 3)  // this is required, because it can still = 2 
	  { 
		i = atoi(gi.argv(MLIST_ARG_2)); 
		if (Q_stricmp(gi.argv(MLIST_ARG_1), "GOTO") == 0)
		{ 
            // user trying to goto specified map # in list 
            if ((i<1) || (i>maplist.nummaps)) 
               DisplayMaplistUsage(ent); 
            else 
            { 
               ent = G_Spawn (); 
               ent->classname = "target_changelevel"; 
               ent->map = maplist.mapnames[i-1]; 
               maplist.currentmap = i-1; 
               BeginIntermission(ent); 
            } 
             break; 
         } 
         else 
         { 
            // user trying to specify new maplist 
            if ((i<0) || (i>=ML_ROTATE_NUM_CHOICES))  // check for valid rotationflag 
            {         
               // outside acceptable values for rotationflag 
               DisplayMaplistUsage(ent); 
               break; 
            } 
            else 
            { 
               maplist.rotationflag = atoi(gi.argv(MLIST_ARG_2)); 
            } 
         } 
      } 
      filename = gi.argv(MLIST_ARG_1);   // get filename from command line 
      if ((int) wfadmin->value & WF_ADMIN_MAP_LIST) 
      { 
         // tell user to cancel current maplist before starting new maplist 
         gi.dprintf ("You must disable current maplist first. (SV MAPLIST OFF)\n"); 
      } 
      else 
      { 
         // load new maplist 
		 ClearMapList(); 
         maplist.rotationflag = atoi(gi.argv(MLIST_ARG_2)); 
         if (LoadMapList(ent, filename))  // return 1 = success 
         { 
            wfadmin->value = (int) wfadmin->value | WF_ADMIN_MAP_LIST; 
            gi.dprintf ("Maplist created/enabled.\n"); 
//            gi.cprintf (ent, PRINT_HIGH, "Maplist created/enabled. You can now use START or NEXT.\n"); 
            maplist.currentmap = -1; 
			WFFillMapNames(); //prepare menu
         } 
      } 
       break; 
    case 1:  // display current maplist 
//gi.dprintf("arg count 1\n");
      if (maplist.nummaps > 0)  // does a maplist exist? 
      { 
         gi.dprintf ("-------------------------------------\n"); 
         for (i=0; i<maplist.nummaps; i++) 
         { 
            gi.dprintf ("#%2d \"%s\" (%d votes)\n", i+1, maplist.mapnames[i], maplist.votes[i]); 
         } 
          gi.dprintf ("%i map(s) in list.\n", i); 
          gi.dprintf ("Rotation flag = %i ", maplist.rotationflag); 
         switch (maplist.rotationflag) 
         { 
         case ML_ROTATE_SEQ: 
            gi.dprintf ("\"sequential\"\n"); 
            break; 
         case ML_ROTATE_RANDOM: 
            gi.dprintf ("\"random\"\n"); 
            break; 
  
         default: 
            gi.dprintf ("(ERROR)\n"); 
         } // end switch 
          if (maplist.currentmap == -1) 
         { 
            gi.dprintf ("Current map = #-1 (not started)\n"); 
         } 
         else 
         { 
            gi.dprintf ("Current map = #%i \"%s\"\n", 
               maplist.currentmap+1, maplist.mapnames[maplist.currentmap]); 
         } 
  
         gi.dprintf ("-------------------------------------\n"); 
         break; 
      } 
       // this is when the command is "sv maplist", but no maplist exists to display 
      DisplayMaplistUsage(ent); 
      break; 
    default: 
      DisplayMaplistUsage(ent); 
   }  // end switch 
}
