/*
 * java QuakeC compiler
 *
 * 12 Sepetember 1998   jet@poboxes.com
 */

import java.io.*;
import java.util.*;

class Parser {

// -- qcc.h --
	private final static int MAX_PARMS = 8;
	private final static int MAX_ERRORS = 10;
	private final static int MAX_NAME = 64;
	private final static int MAX_REGS = 16384;

	private final static int tt_eof = 1;		// end of file reached
	private final static int tt_name = 2;		// an alphanumeric name token
	private final static int tt_punct = 3;		// code punctuation
	private final static int tt_immediate = 4;

	private final static int OFS_NULL = 0;
	private final static int OFS_RETURN = 1;
	private final static int OFS_PARM0 = 4;	// leave 3 ofs for each parm to hold vectors
	private final static int OFS_PARM1 = 7;
	private final static int OFS_PARM2 = 10;
	private final static int OFS_PARM3 = 13;
	private final static int OFS_PARM4 = 16;
	private final static int OFS_PARM5 = 19;
	private final static int OFS_PARM6 = 22;
	private final static int OFS_PARM7 = 25;
	private final static int RESERVED_OFS = 28;

	private final static int MAX_STRINGS = 500000;
	private final static int MAX_GLOBALS = 16384;
	private final static int MAX_STATEMENTS = 65536;
	private final static int MAX_FUNCTIONS = 8192;

	private final static int MAX_SOUNDS = 1024;
	private final static int MAX_MODELS = 1024;
	private final static int MAX_FIELDS = 1024;
	private final static int MAX_FILES = 1024;
	private final static int MAX_DATA_PATH = 64;

// -- qcc.c --
	private static String sourcedir;	// char sourcedir[1024]
	private static String destfile;		// char destfile[1024]

	private static Object pr_globals[];	// float pr_globals[MAX_REGS]
	private static int numpr_globals;	// int numpr_globals

	private static String strings[];	// char strings[MAX_STRINGS]
	private static int strofs;		// int strofs

	private static dstatement_t statements[];	// dstatement_t statements[MAX_STATEMENTS]
	private static int numstatements;		// int numstatements
	private static int statement_linenums[];	// int statement_linenums[MAX_STATEMENTS]

	private static dfunction_t functions[];	// dfunction_t functions[MAX_FUNCTIONS]
	private static int numfunctions;	// int numfunctions

	private static ddef_t globals[];	// ddef_t globals[MAX_GLOBALS]
	private static int numglobaldefs;	// int numglobaldefs

	private static ddef_t fields[];		// ddef_t fields[MAX_FIELDS]
	private static int numfielddefs;	// int numfielddefs;

	private static String precache_sounds[];	// char precache_sounds[MAX_SOUNDS][MAX_DATA_PATH]
	private static int precache_sounds_block[];	// int precache_sounds_block[MAX_SOUNDS]
	private static int numsounds;			// int numsounds

	private static String precache_models[];	// char precache_models[MAX_SOUNDS][MAX_DATA_PATH]
	private static int precache_models_block[];	// int precache_models_block[MAX_SOUNDS]
	private static int nummodels;			// int nummodels

	private static String precache_files[];		// char precache_files[MAX_SOUNDS][MAX_DATA_PATH]
	private static int precache_files_block[];	// int precache_files_block[MAX_SOUNDS]
	private static int numfiles;			// int numfiles

// -- pr_comp.h --
	private final static int ev_void = 1;
	private final static int ev_string = 2;
	private final static int ev_float = 3;
	private final static int ev_vector = 4;
	private final static int ev_entity = 5;
	private final static int ev_field = 6;
	private final static int ev_function = 7;
	private final static int ev_pointer = 8;

	private final static int OP_DONE = 0;
	private final static int OP_MUL_F = 1;
	private final static int OP_MUL_V = 2;
	private final static int OP_MUL_FV = 3;
	private final static int OP_MUL_VF = 4;
	private final static int OP_DIV_F = 5;
	private final static int OP_ADD_F = 6;
	private final static int OP_ADD_V = 7;
	private final static int OP_SUB_F = 8;
	private final static int OP_SUB_V = 9;
	
	private final static int OP_EQ_F = 10;
	private final static int OP_EQ_V = 11;
	private final static int OP_EQ_S = 12;
	private final static int OP_EQ_E = 13;
	private final static int OP_EQ_FNC = 14;
	
	private final static int OP_NE_F = 15;
	private final static int OP_NE_V = 16;
	private final static int OP_NE_S = 17;
	private final static int OP_NE_E = 18;
	private final static int OP_NE_FNC = 19;
	
	private final static int OP_LE = 20;
	private final static int OP_GE = 21;
	private final static int OP_LT = 22;
	private final static int OP_GT = 23;

	private final static int OP_LOAD_F = 24;	
	private final static int OP_LOAD_V = 25;
	private final static int OP_LOAD_S = 26;
	private final static int OP_LOAD_ENT = 27;
	private final static int OP_LOAD_FLD = 28;
	private final static int OP_LOAD_FNC = 29;

	private final static int OP_ADDRESS = 30;

	private final static int OP_STORE_F = 31;
	private final static int OP_STORE_V = 32;
	private final static int OP_STORE_S = 33;
	private final static int OP_STORE_ENT = 34;
	private final static int OP_STORE_FLD = 35;
	private final static int OP_STORE_FNC = 36;

	private final static int OP_STOREP_F = 37;
	private final static int OP_STOREP_V = 38;
	private final static int OP_STOREP_S = 39;
	private final static int OP_STOREP_ENT = 40;
	private final static int OP_STOREP_FLD = 41;
	private final static int OP_STOREP_FNC = 42;

	private final static int OP_RETURN = 43;
	private final static int OP_NOT_F = 44;
	private final static int OP_NOT_V = 45;
	private final static int OP_NOT_S = 46;
	private final static int OP_NOT_ENT = 47;
	private final static int OP_NOT_FNC = 48;
	private final static int OP_IF = 49; 
	private final static int OP_IFNOT = 50;
	private final static int OP_CALL0 = 51;
	private final static int OP_CALL1 = 52;
	private final static int OP_CALL2 = 53;
	private final static int OP_CALL3 = 54;
	private final static int OP_CALL4 = 55;
	private final static int OP_CALL5 = 56;
	private final static int OP_CALL6 = 57;
	private final static int OP_CALL7 = 58;
	private final static int OP_CALL8 = 59;
	private final static int OP_STATE = 60;
	private final static int OP_GOTO = 61;
	private final static int OP_AND = 62;
	private final static int OP_OR = 63;
	
	private final static int OP_BITAND = 64;
	private final static int OP_BITOR = 65;

	private final static int DEF_SAVEGLOBAL = (1<<15);

	private final static int PROG_VERSION = 6;

// -- pr_lex.c --
	private static int pr_source_line;		// int pr_source_line

	private static PushbackInputStream pr_file_p;	// char *pr_file_p
// start of current source line
	private static String pr_line_start;		// char *pr_line_start

	private static int pr_bracelevel;		// int pr_bracelevel

	private static String pr_token;			// char pr_token[2048]
	private static int pr_token_type;		// token_type_t pr_token_type
	private static type_t pr_immediate_type;	// type_t *pr_immediate_type
	private static eval_t pr_immediate;		// eval_t pr_immediate

	private static String pr_immediate_string;	// char pr_immediate_string[2048]

	private static int pr_error_count;		// int pr_error_count

	private static String pr_punctuation[] = 
// longer symbols must be before a shorter partial match
{"&&", "||", "<=", ">=","==", "!=", ";", ",", "!", "*", "/", "(", ")", "-", "+", "=", "[", "]", "{", "}", "...", ".", "<", ">" , "#" , "&" , "|" , null};

// simple types.  function types are dynamically allocated
	private static type_t type_void;
	private static type_t type_string;
	private static type_t type_float;
	private static type_t type_vector;
	private static type_t type_entity;
	private static type_t type_field;
	private static type_t type_function;
// type_function is a void() function used for state defs
	private static type_t type_pointer;
	
	private static type_t type_floatfield;

	private static int type_size[] = {1,1,1,3,1,1,1,1};

	private static def_t def_void = new def_t (type_void, "temp");
	private static def_t def_string = new def_t (type_string, "temp");
	private static def_t def_float = new def_t (type_float, "temp");
	private static def_t def_vector = new def_t (type_vector, "temp");
	private static def_t def_entity = new def_t (type_entity, "temp");
	private static def_t def_field = new def_t (type_field, "temp");
	private static def_t def_function = new def_t (type_function, "temp");
	private static def_t def_pointer = new def_t (type_pointer, "temp");

	private static def_t def_ret, def_parms[];	// def_t def_parms[MAX_PARMS]

	private static def_t def_for_type[] = {def_void, def_string, def_float, def_vector, def_entity, def_field, def_function, def_pointer};

	private final static int MAX_FRAMES = 256;
	private static String pr_framemacros[];		// char pr_framemacros[MAX_FRAMES][16]
	private static int pr_nummacros;		// int pr_nummacros

	private static String pr_parm_names[];		// char pr_parm_names[MAX_PARMS][MAX_NAME];

// -- pr_comp.c --
	private static pr_info_t pr;		// pr_info_t pr
// to find def for a global variable
	private static def_t pr_global_defs[];	// def_t *pr_global_defs[MAX_REGS]
	private static int pr_edict_size;	// int pr_edict_size

// the function being parsed, or null
	private static def_t pr_scope;		// def_t *pr_scope
	private static boolean pr_dumpasm;	// boolean pr_dumpasm
// filename for function definition
	private static int s_file;		// string_t s_file

// for tracking local variables vs temps
	private static int locals_end;		// int locals_end

	private static opcode_t pr_opcodes[] = {
		new opcode_t ("<DONE>", "DONE", -1, false, def_entity, def_field, def_void),
		new opcode_t ("*", "MUL_F", 2, false, def_float, def_float, def_float),
		new opcode_t ("*", "MUL_V", 2, false, def_vector, def_vector, def_float),
		new opcode_t ("*", "MUL_FV", 2, false, def_float, def_vector, def_vector),
		new opcode_t ("*", "MUL_VF", 2, false, def_vector, def_float, def_vector),
 
		new opcode_t ("/", "DIV", 2, false, def_float, def_float, def_float),

		new opcode_t ("+", "ADD_F", 3, false, def_float, def_float, def_float),
		new opcode_t ("+", "ADD_V", 3, false, def_vector, def_vector, def_vector),
  
		new opcode_t ("-", "SUB_F", 3, false, def_float, def_float, def_float),
		new opcode_t ("-", "SUB_V", 3, false, def_vector, def_vector, def_vector),

		new opcode_t ("==", "EQ_F", 4, false, def_float, def_float, def_float),
		new opcode_t ("==", "EQ_V", 4, false, def_vector, def_vector, def_float),
		new opcode_t ("==", "EQ_S", 4, false, def_string, def_string, def_float),
		new opcode_t ("==", "EQ_E", 4, false, def_entity, def_entity, def_float),
		new opcode_t ("==", "EQ_FNC", 4, false, def_function, def_function, def_float),
 
		new opcode_t ("!=", "NE_F", 4, false, def_float, def_float, def_float),
		new opcode_t ("!=", "NE_V", 4, false, def_vector, def_vector, def_float),
		new opcode_t ("!=", "NE_S", 4, false, def_string, def_string, def_float),
		new opcode_t ("!=", "NE_E", 4, false, def_entity, def_entity, def_float),
		new opcode_t ("!=", "NE_FNC", 4, false, def_function, def_function, def_float),
 
		new opcode_t ("<=", "LE", 4, false, def_float, def_float, def_float),
		new opcode_t (">=", "GE", 4, false, def_float, def_float, def_float),
		new opcode_t ("<", "LT", 4, false, def_float, def_float, def_float),
		new opcode_t (">", "GT", 4, false, def_float, def_float, def_float),

		new opcode_t (".", "INDIRECT", 1, false, def_entity, def_field, def_float),
		new opcode_t (".", "INDIRECT", 1, false, def_entity, def_field, def_vector),
		new opcode_t (".", "INDIRECT", 1, false, def_entity, def_field, def_string),
		new opcode_t (".", "INDIRECT", 1, false, def_entity, def_field, def_entity),
		new opcode_t (".", "INDIRECT", 1, false, def_entity, def_field, def_field),
		new opcode_t (".", "INDIRECT", 1, false, def_entity, def_field, def_function),

		new opcode_t (".", "ADDRESS", 1, false, def_entity, def_field, def_pointer),

		new opcode_t ("=", "STORE_F", 5, true, def_float, def_float, def_float),
		new opcode_t ("=", "STORE_V", 5, true, def_vector, def_vector, def_vector),
		new opcode_t ("=", "STORE_S", 5, true, def_string, def_string, def_string),
		new opcode_t ("=", "STORE_ENT", 5, true, def_entity, def_entity, def_entity),
		new opcode_t ("=", "STORE_FLD", 5, true, def_field, def_field, def_field),
		new opcode_t ("=", "STORE_FNC", 5, true, def_function, def_function, def_function),

		new opcode_t ("=", "STOREP_F", 5, true, def_pointer, def_float, def_float),
		new opcode_t ("=", "STOREP_V", 5, true, def_pointer, def_vector, def_vector),
		new opcode_t ("=", "STOREP_S", 5, true, def_pointer, def_string, def_string),
		new opcode_t ("=", "STOREP_ENT", 5, true, def_pointer, def_entity, def_entity),
		new opcode_t ("=", "STOREP_FLD", 5, true, def_pointer, def_field, def_field),
		new opcode_t ("=", "STOREP_FNC", 5, true, def_pointer, def_function, def_function),

		new opcode_t ("<RETURN>", "RETURN", -1, false, def_void, def_void, def_void),
  
		new opcode_t ("!", "NOT_F", -1, false, def_float, def_void, def_float),
		new opcode_t ("!", "NOT_V", -1, false, def_vector, def_void, def_float),
		new opcode_t ("!", "NOT_S", -1, false, def_vector, def_void, def_float),
		new opcode_t ("!", "NOT_ENT", -1, false, def_entity, def_void, def_float),
		new opcode_t ("!", "NOT_FNC", -1, false, def_function, def_void, def_float),
  
		new opcode_t ("<IF>", "IF", -1, false, def_float, def_float, def_void),
		new opcode_t ("<IFNOT>", "IFNOT", -1, false, def_float, def_float, def_void),
  
// calls returns REG_RETURN
		new opcode_t ("<CALL0>", "CALL0", -1, false, def_function, def_void, def_void),
		new opcode_t ("<CALL1>", "CALL1", -1, false, def_function, def_void, def_void),
		new opcode_t ("<CALL2>", "CALL2", -1, false, def_function, def_void, def_void),
		new opcode_t ("<CALL3>", "CALL3", -1, false, def_function, def_void, def_void),
		new opcode_t ("<CALL4>", "CALL4", -1, false, def_function, def_void, def_void),
		new opcode_t ("<CALL5>", "CALL5", -1, false, def_function, def_void, def_void),
		new opcode_t ("<CALL6>", "CALL6", -1, false, def_function, def_void, def_void),
		new opcode_t ("<CALL7>", "CALL7", -1, false, def_function, def_void, def_void),
		new opcode_t ("<CALL8>", "CALL8", -1, false, def_function, def_void, def_void),
  
		new opcode_t ("<STATE>", "STATE", -1, false, def_float, def_float, def_void),
  
		new opcode_t ("<GOTO>", "GOTO", -1, false, def_float, def_void, def_void),
  
		new opcode_t ("&&", "AND", 6, false, def_float, def_float, def_float),
		new opcode_t ("||", "OR", 6, false, def_float, def_float, def_float),

		new opcode_t ("&", "BITAND", 2, false, def_float, def_float, def_float),
		new opcode_t ("|", "BITOR", 2, false, def_float, def_float, def_float),
		null
	};

	private final static int TOP_PRIORITY = 6;
	private final static int NOT_PRIORITY = 4;

	private static def_t junkdef;			// def_t junkdef

// -- cmdlib.c --
	private final static char PATHSEPERATOR = '/';

// set these before calling CheckParm
	private static int myargc;
	private static String myargv[];

	private static String com_token;		// char com_token[1024]
	private static int com_eof;			// int com_eof

// -- fini --

	private static BufferedInputStream pr_mark_file_p;	// used for mark & reset


// ****************************************************************************
// initialize stuff

	public Parser () {
		System.out.println ("Parser ()");

// -- qcc.c --
		pr_globals = new Object[MAX_REGS];
		strings = new String[MAX_STRINGS];
		statements = new dstatement_t[MAX_STATEMENTS];
		statement_linenums = new int[MAX_STATEMENTS];
		functions = new dfunction_t[MAX_FUNCTIONS];
		globals = new ddef_t[MAX_GLOBALS];
		fields = new ddef_t[MAX_FIELDS];
		precache_sounds = new String[MAX_SOUNDS];
		precache_sounds_block = new int[MAX_SOUNDS];
		precache_models = new String[MAX_SOUNDS];
		precache_models_block = new int[MAX_SOUNDS];
		precache_files = new String[MAX_SOUNDS];
		precache_files_block = new int[MAX_SOUNDS];

// -- pr_comp.c --
		pr = new pr_info_t ();
		pr_global_defs = new def_t[MAX_REGS];

// -- pr_lex.c --
		type_void = new type_t (ev_void, def_void);
		type_string = new type_t (ev_string, def_string);
		type_float = new type_t (ev_float, def_float);
		type_vector = new type_t (ev_vector, def_vector);
		type_entity = new type_t (ev_entity, def_entity);
		type_field = new type_t (ev_field, def_field);
		type_function = new type_t (ev_function, def_function, null, type_void);
		type_pointer = new type_t (ev_pointer, def_pointer);
	
		type_floatfield = new type_t (ev_field, def_field, null, type_float);

		def_parms = new def_t[MAX_PARMS];
		pr_framemacros = new String[MAX_FRAMES];

		pr_parm_names = new String[MAX_PARMS];
// -- fini --
	}

// ****************************************************************************
// functions and stuff

// -- qcc.h --
	float G_FLOAT (int o) {
		System.out.println ("= G_FLOAT (" + o + ")");
		return ((Float)pr_globals[o]).floatValue ();
	}

	void G_FLOAT (int o, float a) {
		System.out.println ("G_FLOAT (" + o + ") = " + a);
		pr_globals[o] = new Float (a);
	}

	int G_INT (int o) {
		System.out.println ("= G_INT (" + o + ")");
		return ((Integer)pr_globals[o]).intValue ();
	}

	void G_INT (int o, int a) {
		System.out.println ("G_INT (" + o + ") = " + a);
		pr_globals[o] = new Integer (a);
	}

	float[] G_VECTOR (int o) {
		System.out.println ("= G_VECTOR (" + o + ")");
		float f[] = {
			((Float)pr_globals[o]).floatValue (),
			((Float)pr_globals[o+1]).floatValue (),
			((Float)pr_globals[o+2]).floatValue ()};

		return f;
	}

	void G_VECTOR (int o, float[] a) {
		System.out.println ("G_VECTOR ("+o+") = "+a[0]+", "+a[1]+", "+a[2]);
		pr_globals[o] = new Float (a[0]);
		pr_globals[o+1] = new Float (a[1]);
		pr_globals[o+2] = new Float (a[2]);
	}


	String G_STRING (int o) {
		System.out.println ("= G_STRING (" + o + ")");
		return strings[((Integer)pr_globals[o]).intValue ()];
	}

	void G_STRING (int o, String a) {
		System.out.println ("G_STRING (" + o + ") = " + a);
		int j = ((Integer)pr_globals[o]).intValue ();
		strings[j] = a;
	}

	int G_FUNCTION (int o) {
		System.out.println ("= G_FUNCTION (" + o + ")");
		return ((Integer)pr_globals[o]).intValue ();
	}

	void G_FUNCTION (int o, int a) {
		System.out.println ("G_FUNCTION (" + o + ") =");
		pr_globals[o] = new Integer (a);
	}

// -- qcc.c --

	void writeFiles () {
		System.out.println ("writeFiles ()");

		PrintWriter f = null;
		String filename = sourcedir + "files.dat";
		try {
			f = new PrintWriter (new BufferedWriter (new FileWriter (filename)));
		} catch (IOException e) {
			System.out.println ("Error opening " + filename);
			return;
		}

		f.println (numsounds);
		for (int i = 0; i < numsounds; i++) {
			f.println (precache_sounds_block[i] + " " + precache_sounds[i]);
		}

		f.println (nummodels);
		for (int i = 0; i < nummodels; i++) {
			f.println (precache_models_block[i] + " " + precache_models[i]);
		}

		f.println (numfiles);
		for (int i = 0; i < numfiles; i++) {
			f.println (precache_files_block[i] + " " + precache_files[i]);
		}

		f.close ();
	}

// CopyString returns an offset from the string heap
	int CopyString (String str) {
		System.out.println ("CopyString (" + str + ")");

		int old = strofs;
		strings[strofs++] = str;
		return old;
	}

	void PrintStrings () {
		System.out.println ("PrintStrings ()");
	
		for (int i = 0; i < strofs; i++) {
			System.out.println ("" + i + " : " + strings[i]);
		}
		System.out.println ("");
	}

	void PrintFunctions () {
		System.out.println ("PrintFunctions ()");

		for (int i = 0; i < numfunctions; i++) {
			dfunction_t d = functions[i];
			System.out.print (strings[d.s_file] + " : " + 
					  strings[d.s_name] + " : " +
					  d.first_statement + d.parm_start);
			for (int j = 0; j < d.numparms; j++)
				System.out.print ("" + d.parm_size[j] + " ");
			System.out.println (")");
		}
	}

	void PrintFields () {
		System.out.println ("PrintFields ()");
	
		for (int i = 0; i < numfielddefs; i++) {
			ddef_t d = fields[i];
			System.out.println ("" + d.ofs + " : (" + d.type + ") " + strings[d.s_name]);
		}
	}

	void PrintGlobals () {
		System.out.println ("PrintGlobals ()");
	
		for (int i = 0; i < numglobaldefs; i++) {
			ddef_t d = globals[i];
			System.out.println ("" + d.ofs + " : (" + d.type + ") " + strings[d.s_name]);
		}
	}

	void InitData () {
		System.out.println ("InitData ()");

		numstatements = 1;
		strofs = 1;
		numfunctions = 1;
		numglobaldefs = 1;
		numfielddefs = 1;
	
		def_ret.ofs = OFS_RETURN;
		for (int i = 0; i < MAX_PARMS; i++) {
			def_parms[i].ofs = OFS_PARM0 + 3*i;
		}
	}

	void WriteData (int crc) {
		System.out.println ("WriteData (" + crc + ")");

		for (def_t def = pr.def_head.next; def != null; def = def.next) {
//		ListIterator li = pr.def.listIterator (0);
//		while (li.hasNext ()) {
//			def_t def = (def_t)li.next ();			

			ddef_t dd;
			if (def.type.type == ev_function) {
//				df = &functions[numfunctions];
//				numfunctions++;

			} else if (def.type.type == ev_field) {
				dd = fields[numfielddefs];
				numfielddefs++;
				dd.type = def.type.aux_type.type;
				dd.s_name = CopyString (def.name);
				dd.ofs = G_INT(def.ofs);
			}
			dd = globals[numglobaldefs];
			numglobaldefs++;
			dd.type = def.type.type;
			if (!def.initialized && 
			    def.type.type != ev_function && 
			    def.type.type != ev_field && 
			    def.scope == null)
			{
				dd.type |= DEF_SAVEGLOBAL;
			}
			dd.s_name = CopyString (def.name);
			dd.ofs = def.ofs;
		}

PrintStrings ();
PrintFunctions ();
PrintFields ();
PrintGlobals ();

		System.out.println (strofs + " strofs");
		System.out.println (numstatements + " numstatements");
		System.out.println (numfunctions + " numfunctions");
		System.out.println (numglobaldefs + " numglobaldefs");
		System.out.println (numfielddefs + " numfielddefs");
		System.out.println (numpr_globals + " numpr_globals");
	}

/*
===============
PR_String

Returns a string suitable for printing (no newlines, max 60 chars length)
===============
*/
	String PR_String (String string) {
		System.out.println ("PR_String (" + string + ")");

		char buf[] = new char[80];
		int s = 0;
		int i = 0;
		int len = string.length ();
	
		buf[s++] = '"';
		while (string != null && i < len) {
			if (s == 80 - 2)
				break;
			if (string.charAt (i) == '\n') {
				buf[s++] = '\\';
				buf[s++] = 'n';
			} else if (string.charAt (i) == '"') {
				buf[s++] = '\\';
				buf[s++] = '"';
			} else
				buf[s++] = string.charAt (i);
			i++;
			if (s > 60) {
				buf[s++] = '.';
				buf[s++] = '.';
				buf[s++] = '.';
				break;
			}
		}
		buf[s++] = '"';
		buf[s++] = 0;
		return new String (buf);
	}

	def_t PR_DefForFieldOfs (int ofs) throws Exception {
		System.out.println ("PR_DefForFieldOfs (" + ofs + ")");

		for (def_t d = pr.def_head.next; d != null; d = d.next) {
//		ListIterator li = pr.def.listIterator (0);
//		while (li.hasNext  ()) {
//			def_t d = (def_t)li.next ();			

			if (d.type.type != ev_field)
				continue;
			if (((Integer)pr_globals[d.ofs]).intValue () == ofs)
				return d;
		}
		throw new Exception ("PR_DefForFieldOfs: couldn't find " + ofs);
	}

/*
============
PR_ValueString

Returns a string describing *data in a type specific manner
=============
*/
	String PR_ValueString (int type, Object val) throws Exception {
		System.out.println ("PR_ValueString (" + type + ", Object)");

		String line;
		def_t def;
		dfunction_t f;
	
		switch (type) {
		case ev_string:
			line = PR_String (strings[((Integer)val).intValue ()]);
			break;

		case ev_entity:	
			line = "entity " + ((Integer)val).intValue ();
			break;

		case ev_function:
			f = functions[((Integer)val).intValue ()];
			if (f == null)
				line = "undefined function";
			else
				line = strings[f.s_name] + "()";
			break;

		case ev_field:
			def = PR_DefForFieldOfs (((Integer)val).intValue ());
			line = "." + def.name;
			break;

		case ev_void:
			line = "void";
			break;

		case ev_float:
			line = "" + ((Float)val).floatValue ();
			break;

		case ev_vector:
			line = "'" + ((Float [])val)[0].floatValue () +
                               " " + ((Float [])val)[1].floatValue () +
			       " " + ((Float [])val)[2].floatValue () +"'";
			break;

		case ev_pointer:
			line = "pointer";
			break;
	
		default:
			line = "bad type " + type;
			break;
		}
	
		return line;
	}

/*
============
PR_GlobalString

Returns a string with a description and the contents of a global,
padded to 20 field width
============
*/
	String PR_GlobalStringNoContents (int ofs) {
		System.out.println ("PR_GlobalStringNoContents (" + ofs + ")");

		String line;
	
		Object val = pr_globals[ofs];
		def_t def = pr_global_defs[ofs];
		if (def == null)
			line = "" + ofs + "(???)";
		else
			line = "" + ofs + "(" + def.name + ")";
	
		int i = line.length ();
		for (; i < 16; i++)
			line += " ";
		line += " ";
		
		return line;
	}

	String PR_GlobalString (int ofs) throws Exception {
		System.out.println ("PR_GlobalString (" + ofs + ")");

		String line;
		Object val = pr_globals[ofs];
		def_t def = pr_global_defs[ofs];
		if (def == null)
			return PR_GlobalStringNoContents (ofs);
		if (def.initialized && def.type.type != ev_function) {
			String s = PR_ValueString (def.type.type, pr_globals[ofs]);
			line = "" + ofs + "(" + s + ")";
		} else
			line = "" + ofs + "(" + def.name + ")";
	
		int i = line.length ();
		for (; i < 16; i++)
			line += " ";
		line += " ";
		
		return line;
	}

/*
============
PR_PrintOfs
============
*/
	void PR_PrintOfs (int ofs) throws Exception {
		System.out.println ("PR_PrintOfs (" + ofs + ")");

		System.out.println (PR_GlobalString (ofs));
	}

/*
=================
PR_PrintStatement
=================
*/
	void PR_PrintStatement (int is) throws Exception {
		System.out.println ("PR_PrintStatement (" + is + ")");

		dstatement_t s = statements[is];
		System.out.print ("" + is + " : " +
				  statement_linenums[is] + " : " +
				  pr_opcodes[s.op].opname + " ");
		int i = pr_opcodes[s.op].opname.length ();
		for (; i < 10; i++)
			System.out.print (" ");
		
		if (s.op == OP_IF || s.op == OP_IFNOT)
			System.out.print (PR_GlobalString(s.a) + "branch " + s.b);
		else if (s.op == OP_GOTO) {
			System.out.print ("branch " + s.a);
		} else if ((s.op - OP_STORE_F) < 6) {
			System.out.print (PR_GlobalString(s.a));
			System.out.print (PR_GlobalStringNoContents(s.b));
		} else {
			if (s.a != 0)
				System.out.print (PR_GlobalString(s.a));
			if (s.b != 0)
				System.out.print (PR_GlobalString(s.b));
			if (s.c != 0)
				System.out.print (PR_GlobalStringNoContents(s.c));
		}
		System.out.println ("");
	}

/*
============
PR_PrintDefs
============
*/
	void PR_PrintDefs () throws Exception {
		System.out.println ("PR_PrintDefs ()");

		for (def_t d = pr.def_head.next; d != null; d = d.next) {
//		ListIterator li = pr.def.listIterator (0);
//		while (li.hasNext ()) {
//			def_t d = (def_t)li.next ();			
			PR_PrintOfs (d.ofs);
		}
	}

/*
==============
PR_BeginCompilation

called before compiling a batch of files, clears the pr struct
==============
*/
	void PR_BeginCompilation () {
		System.out.println ("PR_BeginCompilation ()");

//		pr.memory = memory;
//		pr.max_memory = memsize;
	
		numpr_globals = RESERVED_OFS;
	
		for (int i = 0; i < RESERVED_OFS; i++)
			pr_global_defs[i] = def_void;
		
// link the function type in so state forward declarations match proper type
		pr.types = type_function;
		type_function.next = null;
		pr_error_count = 0;
	}

/*
==============
PR_FinishCompilation

called after all files are compiled to check for errors
Returns false if errors were detected.
==============
*/
	boolean PR_FinishCompilation () {
		System.out.println ("PR_FinishCompilation ()");

		boolean errors = false;
	
// check to make sure all functions prototyped have code
		for (def_t d = pr.def_head.next; d != null; d = d.next) {
//		ListIterator li = pr.def.listIterator (0);
//		while (li.hasNext ()) {
//			def_t d = (def_t)li.next ();			

			if (d.type.type == ev_function && d.scope == null) { // function parms are ok
				if (!d.initialized) {
					System.out.println ("function " + d.name + " was not defined");
					errors = true;
				}
			}
		}

		return !errors;
	}

/*
============
PR_WriteProgdefs

Writes the global and entity structures out
Returns a crc of the header, to be stored in the progs file for comparison
at load time.
============
*/
	int PR_WriteProgdefs (String filename) {
		System.out.println ("PR_WriteProgdefs (" + filename + ")");

		return 0;
	}

	void PrintFunction (String name) throws Exception {
		System.out.println ("PrintFunction (" + name + ")");

		int i;
	
		for (i = 0; i < numfunctions; i++)
			if (name.equals (strings[functions[i].s_name]))
				break;
	
		if (i == numfunctions)
			throw new Exception ("No function names \"" + name + "\"");

		dfunction_t df = functions[i];
		int j = df.first_statement;
		System.out.println ("Statements for " + name + ":\n");

		while (true) {
			PR_PrintStatement (j);
			dstatement_t ds = statements[j++];
			if (ds.op == 0) break;
		}
	}

// -- pr-comp.c --

/*
============
PR_Statement

Emits a primitive statement, returning the var it places it's value in
============
*/
	def_t PR_Statement (int io, def_t var_a, def_t var_b) {
		System.out.println ("PR_Statement (" + io + ", def_t, def_t)");

		dstatement_t statement;
		def_t var_c;
		opcode_t op = pr_opcodes[io];
	
		statement = statements[numstatements];
		statement_linenums[numstatements] = pr_source_line;
		statement.op = io;
		statement.a = (var_a != null) ? var_a.ofs : 0;
		statement.b = (var_b != null) ? var_b.ofs : 0;
		numstatements++;
	
	 	if (op.type_c == def_void || op.right_associative) {
			var_c = null;
			statement.c = 0;	// ifs, gotos, and assignments
						// don't need vars allocated
		} else {			// allocate result space
			var_c = new def_t ();
			var_c.ofs = numpr_globals;
			var_c.type = op.type_c.type;

			statement.c = numpr_globals;
			numpr_globals += type_size[op.type_c.type.type];
		}

		if (op.right_associative)
			return var_a;
		return var_c;
	}

/*
============
PR_ParseImmediate

Looks for a preexisting constant
============
*/
	def_t	PR_ParseImmediate () throws IOException, ParseException {
		System.out.println ("PR_ParseImmediate ()");

		def_t cn;

// check for a constant with the same value
		for (cn = pr.def_head.next; cn != null; cn = cn.next) {
//		ListIterator li = pr.def.listIterator (0);
//		while (li.hasNext ()) {
//			cn = (def_t)li.next ();			

			if (!cn.initialized) continue;
			if (cn.type != pr_immediate_type) continue;

			if (pr_immediate_type == type_string) {
				if (G_STRING(cn.ofs).equals (pr_immediate_string)) {
					PR_Lex ();
					return cn;
				}
			} else if (pr_immediate_type == type_float) {
				if (G_FLOAT(cn.ofs) == pr_immediate._float) {
					PR_Lex ();
					return cn;
				}
			} else if (pr_immediate_type == type_vector) {
				if ((G_FLOAT(cn.ofs) == pr_immediate.vector[0]) && 
				    (G_FLOAT(cn.ofs+1) == pr_immediate.vector[1]) && 
				    (G_FLOAT(cn.ofs+2) == pr_immediate.vector[2]))
				{
					PR_Lex ();
					return cn;
				}
			} else			
				throw new ParseException ("weird immediate type");		
		}
	
// allocate a new one
		cn = new def_t ();
//		cn.next = null;
//		pr.def_tail.next = cn;
//		pr.def_tail = cn;
		pr.def.add (cn);
		cn.type = pr_immediate_type;
		cn.name = "IMMEDIATE";
		cn.initialized = true;
		cn.scope = null;		// always share immediates

// copy the immediate to the global area
		cn.ofs = numpr_globals;
		pr_global_defs[cn.ofs] = cn;
		numpr_globals += type_size[pr_immediate_type.type];
		if (pr_immediate_type == type_string)
			pr_immediate.string = CopyString (pr_immediate_string);
	
		pr_globals[cn.ofs] = pr_immediate;
	
		PR_Lex ();

		return cn;
	}

	void PrecacheSound (def_t e, char ch) throws Exception {
		System.out.println ("PrecacheSound (def_t, "+ch+")");

		String n;
		int i;
	
		if (e.ofs != 0) return;

		n = G_STRING(e.ofs);
		for (i = 0; i < numsounds; i++)
			if (n.equals (precache_sounds[i])) return;

		if (numsounds == MAX_SOUNDS)
			throw new Exception ("PrecacheSound: numsounds == MAX_SOUNDS");

		precache_sounds[i] = n;
		if (ch >= '1'  && ch <= '9')
			precache_sounds_block[i] = ch - '0';
		else
			precache_sounds_block[i] = 1;
		numsounds++;
	}

	void PrecacheModel (def_t e, char ch) throws Exception {
		System.out.println ("PrecacheModel (def_t, "+ch+")");

		String  n;
		int i;
	
		if (e.ofs != 0) return;

		n = G_STRING(e.ofs);
		for (i = 0; i < nummodels; i++)
			if (n.equals (precache_models[i])) return;

		if (numsounds == MAX_SOUNDS)
			throw new Exception ("PrecacheModels: numsounds == MAX_SOUNDS");

		precache_models[i] = n;
		if (ch >= '1'  && ch <= '9')
			precache_models_block[i] = ch - '0';
		else
			precache_models_block[i] = 1;
		nummodels++;
	}

	void PrecacheFile (def_t e, char ch) throws Exception {
		System.out.println ("PrecacheFile (def_t, "+ch+")");

		String n;
		int i;
	
		if (e.ofs != 0) return;

		n = G_STRING(e.ofs);
		for (i = 0; i < numfiles; i++)
			if (n.equals (precache_files[i])) return;

		if (numfiles == MAX_FILES)
			throw new Exception ("PrecacheFile: numfiles == MAX_FILES");

		precache_files[i] = n;
		if (ch >= '1'  && ch <= '9')
			precache_files_block[i] = ch - '0';
		else
			precache_files_block[i] = 1;
		numfiles++;
	}

/*
============
PR_ParseFunctionCall
============
*/
	def_t PR_ParseFunctionCall (def_t func) throws ParseException, Exception {
		System.out.println ("PR_ParseFunctionCall (def_t)");

		def_t e;
		int arg;
		type_t t;
	
		t = func.type;

		if (t.type != ev_function)
			throw new ParseException ("not a function");
	
// copy the arguments to the global parameter variables
		arg = 0;
		if (!PR_Check(")")) {
			do {
				if (t.num_parms != -1 && arg >= t.num_parms)
					throw new ParseException ("too many parameters");
		
				e = PR_Expression (TOP_PRIORITY);

				if (arg == 0 && func.name.length () > 0) {
// save information for model and sound caching
					if (func.name.startsWith ("precache_sound"))
						PrecacheSound (e, func.name.charAt (14));
					else if (func.name.startsWith ("precache_model"))
						PrecacheModel (e, func.name.charAt (14));
					else if (func.name.startsWith ("precache_file"))
						PrecacheFile (e, func.name.charAt (13));
				}
						
				if (t.num_parms != -1 && (e.type != t.parm_types[arg]))
					throw new ParseException ("type mismatch on parm " + arg);

// a vector copy will copy everything
				def_parms[arg].type = t.parm_types[arg];
				PR_Statement (OP_STORE_V, e, def_parms[arg]);
				arg++;
			} while (PR_Check (","));
	
			if (t.num_parms != -1 && arg != t.num_parms)
				throw new ParseException ("too few parameters");
			PR_Expect (")");
		}

		if (arg >8)	
			throw new ParseException ("More than eight parameters");
		
		PR_Statement (OP_CALL0+arg, func, null);
	
		def_ret.type = t.aux_type;
		return def_ret;
	}

/*
============
PR_ParseValue

Returns the global ofs for the current token
============
*/
	def_t PR_ParseValue () throws IOException, ParseException {
		System.out.println ("PR_ParseValue ()");

		def_t d;
		String name;
	
// if the token is an immediate, allocate a constant for it
		if (pr_token_type == tt_immediate)
			return PR_ParseImmediate ();
	
		name = PR_ParseName ();
	
// look through the defs
		d = PR_GetDef (null, name, pr_scope, false);
		if (d != null)
			throw new ParseException ("Unknown value \"" + name + "\"");	
		return d;
	}

/*
============
PR_Term
============
*/
	def_t PR_Term () throws ParseException, Exception {
		System.out.println ("PR_Term ()");

		def_t e, e2;
		int	t;
	
		if (PR_Check ("!")) {
			e = PR_Expression (NOT_PRIORITY);
			t = e.type.type;
			if (t == ev_float)
				e2 = PR_Statement (OP_NOT_F, e, null);
			else if (t == ev_string)
				e2 = PR_Statement (OP_NOT_S, e, null);
			else if (t == ev_entity)
				e2 = PR_Statement (OP_NOT_ENT, e, null);
			else if (t == ev_vector)
				e2 = PR_Statement (OP_NOT_V, e, null);
			else if (t == ev_function)
				e2 = PR_Statement (OP_NOT_FNC, e, null);
			else {
				e2 = null;		// shut up compiler warning;
				throw new ParseException ("type mismatch for !");
			}
			return e2;
		}
	
		if (PR_Check ("(")) {
			e = PR_Expression (TOP_PRIORITY);
			PR_Expect (")");
			return e;
		}
	
		return PR_ParseValue ();
	}

/*
==============
PR_Expression
==============
*/

	def_t PR_Expression (int priority) throws ParseException, Exception {
		System.out.println ("PR_Expression ("+priority+")");

		opcode_t op = null, oldop;
		def_t e, e2;
		int type_a, type_b, type_c;
	
		if (priority == 0)
			return PR_Term ();
		
		e = PR_Expression (priority-1);
		
		while (true) {
			if (priority == 1 && PR_Check ("(") )
				return PR_ParseFunctionCall (e);

			for (int i = 0; op.name.length () > 0; i++) {
				op = pr_opcodes[i];

				if (op.priority != priority)
					continue;
				if (!PR_Check (op.name))
					continue;
				if ( op.right_associative ) {
// if last statement is an indirect, change it to an address of

					if ((statements[numstatements-1].op - OP_LOAD_F) < 6 ) {
						statements[numstatements-1].op = OP_ADDRESS;
						def_pointer.type.aux_type = e.type;
						e.type = def_pointer.type;
					}
					e2 = PR_Expression (priority);
				} else
					e2 = PR_Expression (priority-1);
				
// type check
				type_a = e.type.type;
				type_b = e2.type.type;

				if (op.name.charAt (0) == '.') { // field access gets type from field
					if (e2.type.aux_type != null)
						type_c = e2.type.aux_type.type;
					else
						type_c = -1;	// not a field
				} else
					type_c = ev_void;
				
				oldop = op;
				while (type_a != op.type_a.type.type || 
				       type_b != op.type_b.type.type || 
				      (type_c != ev_void && type_c != op.type_c.type.type))
				{
					op = pr_opcodes[++i];
					if (op.name.length () > 0 || !op.name.equals (oldop.name))
					throw new ParseException ("type mismatch for " + oldop.name);
				}
			
				if (type_a == ev_pointer && type_b != e.type.aux_type.type)
					throw new ParseException ("type mismatch for " + op.name);
			
			
				if (op.right_associative)
					e = PR_Statement (i, e2, e);
				else
					e = PR_Statement (i, e, e2);
			
				if (type_c != ev_void)	// field access gets type from field
					e.type = e2.type.aux_type;
			
				break;
			}
			if (op.name.length () > 0)
				break;	// next token isn't at this priority level
		}
	
		return e;
	}

/*
============
PR_ParseStatement

============
*/
	void PR_ParseStatement () throws ParseException, Exception {
		System.out.println ("PR_ParseStatement ()");

		def_t e;
		int patch1, patch2;
	
		if (PR_Check ("{")) {
			do {
				PR_ParseStatement ();
			} while (!PR_Check ("}"));
			return;
		}
	
		if (PR_Check("return")) {
			if (PR_Check (";")) {
				PR_Statement (OP_RETURN, null, null);
				return;
			}
			e = PR_Expression (TOP_PRIORITY);
			PR_Expect (";");
			PR_Statement (OP_RETURN, e, null);
			return;		
		}
	
		if (PR_Check("while")) {
			PR_Expect ("(");
			patch2 = numstatements;
			e = PR_Expression (TOP_PRIORITY);
			PR_Expect (")");
			patch1 = numstatements;
			PR_Statement (OP_IFNOT, e, null);
			PR_ParseStatement ();
			junkdef.ofs = patch2 - numstatements;
			PR_Statement (OP_GOTO, junkdef, null);
			statements[patch1].b = numstatements - patch1;
			return;
		}
	
		if (PR_Check("do")) {
			patch1 = numstatements;
			PR_ParseStatement ();
			PR_Expect ("while");
			PR_Expect ("(");
			e = PR_Expression (TOP_PRIORITY);
			PR_Expect (")");
			PR_Expect (";");
			junkdef.ofs = patch1 - numstatements;
			PR_Statement (OP_IF, e, junkdef);
			return;
		}
	
		if (PR_Check("local")) {
			PR_ParseDefs ();
			locals_end = numpr_globals;
			return;
		}
	
		if (PR_Check("if")) {
			PR_Expect ("(");
			e = PR_Expression (TOP_PRIORITY);
			PR_Expect (")");
			
			patch1 = numstatements;
			PR_Statement (OP_IFNOT, e, null);
		
			PR_ParseStatement ();
		
			if (PR_Check ("else")) {
				patch2 = numstatements;
				PR_Statement (OP_GOTO, null, null);
				statements[patch1].b = numstatements - patch1;
				PR_ParseStatement ();
				statements[patch2].a = numstatements - patch2;
			} else
				statements[patch1].b = numstatements - patch1;
		
			return;
		}
	
		PR_Expression (TOP_PRIORITY);
		PR_Expect (";");
	}

/*
==============
PR_ParseState

States are special functions made for convenience.  They automatically
set frame, nextthink (implicitly), and think (allowing forward definitions).

// void() name = [framenum, nextthink] {code}
// expands to:
// function void name ()
// {
//		self.frame=framenum;
//		self.nextthink = time + 0.1;
//		self.think = nextthink
//		<code>
// };
==============
*/
	void PR_ParseState () throws IOException, ParseException {
		System.out.println ("PR_ParseState ()");

		String name;
		def_t s1, def;
	
		if (pr_token_type != tt_immediate || 
		    pr_immediate_type != type_float)
			throw new ParseException ("state frame must be a number");
		s1 = PR_ParseImmediate ();
	
		PR_Expect (",");

		name = PR_ParseName ();
		def = PR_GetDef (type_function, name, null, true);
		
		PR_Expect ("]");
	
		PR_Statement (OP_STATE, s1, def);
	}

/*
============
PR_ParseImmediateStatements

Parse a function body
============
*/
	function_t PR_ParseImmediateStatements (type_t type) throws ParseException, Exception {
		System.out.println ("PR_ParseImmediateStatements (type_t)");

		function_t f;
		def_t defs[] = new def_t[MAX_PARMS];
	
		f = new function_t ();

// check for builtin function definition #1, #2, etc
		if (PR_Check ("#")) {
			if (pr_token_type != tt_immediate || 
			    pr_immediate_type != type_float || 
			    pr_immediate._float != (int)pr_immediate._float)
				throw new ParseException ("Bad builtin immediate");

			f.builtin = (int)pr_immediate._float;
			PR_Lex ();
			return f;
		}
	
		f.builtin = 0;

// define the parms
		for (int i = 0; i < type.num_parms; i++) {
			defs[i] = PR_GetDef (type.parm_types[i], pr_parm_names[i], pr_scope, true);
			f.parm_ofs[i] = defs[i].ofs;
			if (i > 0 && f.parm_ofs[i] < f.parm_ofs[i-1])
				throw new Exception ("bad parm order");
		}
	
		f.code = numstatements;

// check for a state opcode
		if (PR_Check ("["))
			PR_ParseState ();
		
// parse regular statements
		PR_Expect ("{");

		while (!PR_Check("}"))
			PR_ParseStatement ();
	
// emit an end of statements opcode
		PR_Statement (0, null, null);

		return f;
	}

/*
============
PR_GetDef

If type is NULL, it will match any type
If allocate is true, a new def will be allocated if it can't be found
============
*/
	def_t PR_GetDef (type_t type, String name, def_t scope, boolean allocate) throws ParseException {
		System.out.println ("PR_GetDef (type_t, " + name + ", def_t, " + allocate + ")");

		def_t def = null;
		String element;

System.out.println ("searching ...");
System.out.println ("def "+def);
System.out.println ("def.next "+def.next);
System.out.println ("head "+pr.def_head);
System.out.println ("head.next "+pr.def_head.next);
System.out.println ("tail "+pr.def_tail);
System.out.println ("tail.next "+pr.def_tail.next);

// see if the name is already in use
		for (def = pr.def_head.next; def != null;def = def.next) {
//		ListIterator li = pr.def.listIterator (0);
//		while (li.hasNext ()) {
//			def = (def_t)li.next ();			
System.out.println ("cf. " + def.name + " [" + def + "]");
			if (def.name.equals (name)) {
				if (def.scope != null && def.scope != scope)
					continue;		// in a different function
			
				if (type != null && def.type != type)
					throw new ParseException ("Type mismatch on redeclaration of " + name);
				return def;
			}
		}
	
		if (!allocate)
			return null;
		
// allocate a new def
		def = new def_t ();

System.out.println ("before");
System.out.println ("def "+def);
System.out.println ("def.next "+def.next);
System.out.println ("head "+pr.def_head);
System.out.println ("head.next "+pr.def_head.next);
System.out.println ("tail "+pr.def_tail);
System.out.println ("tail.next "+pr.def_tail.next);

		pr.def_tail.next = def;
		pr.def_tail = pr.def_tail.next;
//		pr.def.add (def);

System.out.println ("after");
System.out.println ("def "+def);
System.out.println ("def.next "+def.next);
System.out.println ("head "+pr.def_head);
System.out.println ("head.next "+pr.def_head.next);
System.out.println ("tail "+pr.def_tail);
System.out.println ("tail.next "+pr.def_tail.next);

		pr.def_tail.next = null;

System.out.println ("null");
System.out.println ("def "+def);
System.out.println ("def.next "+def.next);
System.out.println ("head "+pr.def_head);
System.out.println ("head.next "+pr.def_head.next);
System.out.println ("tail "+pr.def_tail);
System.out.println ("tail.next "+pr.def_tail.next);

		def.name = name;
		def.type = type;
		def.scope = scope;
		def.ofs = numpr_globals;
		pr_global_defs[numpr_globals] = def;
	
//
// make automatic defs for the vectors elements
// .origin can be accessed as .origin_x, .origin_y, and .origin_z
//
		if (type.type == ev_vector) {
			element = name + "_x";
			PR_GetDef (type_float, element, scope, true);
		
			element = name + "_y";
			PR_GetDef (type_float, element, scope, true);
		
			element = name + "_z";
			PR_GetDef (type_float, element, scope, true);
		} else
			numpr_globals += type_size[type.type];

		if (type.type == ev_field) {
			pr_globals[def.ofs] = new Integer (pr.size_fields);
		
			if (type.aux_type.type == ev_vector) {
				element = name + "_x";
				PR_GetDef (type_floatfield, element, scope, true);
			
				element = name + "_y";
				PR_GetDef (type_floatfield, element, scope, true);
				
				element = name + "_z";
				PR_GetDef (type_floatfield, element, scope, true);
			} else
				pr.size_fields += type_size[type.aux_type.type];
		}

int i = 0;
for (def_t t = pr.def_head; t != null; t = t.next) {
	System.out.println ("" + i + ": " + t.name + " [" + t + "]");
	i++;
}

		return def;
	}

/*
================
PR_ParseDefs

Called at the outer layer and when a local statement is hit
================
*/
	void PR_ParseDefs () throws ParseException, Exception {
		System.out.println ("PR_ParseDefs ()");

		String name;
		type_t type;
		def_t def;
		function_t f;
		dfunction_t df;
		int locals_start;

		type = PR_ParseType ();
	
		if (pr_scope != null && (type.type == ev_field || type.type == ev_function))
			throw new ParseException ("Fields and functions must be global");
		
		do {
			name = PR_ParseName ();

			def = PR_GetDef (type, name, pr_scope, true);
		
// check for an initialization
			if (PR_Check ("=")) {
				if (def.initialized)
					throw new ParseException (name + " redeclared");
	
				if (type.type == ev_function) {
					locals_start = locals_end = numpr_globals;
					pr_scope = def;
					f = PR_ParseImmediateStatements (type);
					pr_scope = null;
					def.initialized = true;
					G_FUNCTION(def.ofs, numfunctions);
					f.def = def;

// fill in the dfunction
					df = functions[numfunctions];
					numfunctions++;
					if (f.builtin > 0)
						df.first_statement = -f.builtin;
					else
						df.first_statement = f.code;
					df.s_name = CopyString (f.def.name);
					df.s_file = s_file;
					df.numparms =  f.def.type.num_parms;
					df.locals = locals_end - locals_start;
					df.parm_start = locals_start;
					for (int i = 0; i < df.numparms ; i++)
						df.parm_size[i] = type_size[f.def.type.parm_types[i].type];
				
					continue;
				}
				else if (pr_immediate_type != type)
					throw new ParseException ("wrong immediate type for " + name);
	
				def.initialized = true;
				pr_globals[def.ofs] = pr_immediate;
				PR_Lex ();
			}
		
		} while (PR_Check (","));

		PR_Expect (";");
	}

/*
============
PR_CompileFile

compiles the 0 terminated text, adding defintions to the pr structure
============
*/
	boolean	PR_CompileFile (String filename, PushbackInputStream pis, BufferedInputStream bis) throws Exception {
		System.out.println ("PR_CompileFile ("+filename+")");

		if (pr.memory != null)
			throw new Exception ("PR_CompileFile: Didn't clear");

		PR_ClearGrabMacros ();	// clear the frame macros

		pr_mark_file_p = bis;		
		pr_file_p = pis;
		s_file = CopyString (filename);

		pr_source_line = 0;
	
		PR_NewLine ();

		PR_Lex ();	// read first token

		while (pr_token_type != tt_eof) {
			try {
				pr_scope = null;	// outside all functions
		
				PR_ParseDefs ();
			} catch (ParseException e) {
				if (++pr_error_count > MAX_ERRORS)
					return false;
				PR_SkipToSemicolon ();
				if (pr_token_type == tt_eof)
					return false;
			}
		}
	
		return (pr_error_count == 0);
	}

// -- me --

	char pr_file_p_read () throws IOException {
		return (char)(new Integer (pr_file_p.read ())).byteValue ();
	}

// -- pr_lex.c --

/*
==============
PR_PrintNextLine
==============
*/
	void PR_PrintNextLine () throws IOException {
		System.out.println ("PR_PrintNextLine ()");

		char c;
		System.out.print ("" + pr_source_line);
		pr_mark_file_p.reset ();
		for (c = pr_file_p_read (); c != '\n'; c = pr_file_p_read ())
			System.out.print ("" + c);
		pr_file_p.unread (c);
		System.out.println ("");
	}

/*
==============
PR_NewLine

Call at start of file and when *pr_file_p == '\n'
==============
*/
	void PR_NewLine () throws IOException {
		System.out.println ("PR_NewLine ()");

		boolean	m;
	
		char c = pr_file_p_read ();
		if (c == '\n') {
			m = true;
		} else {
			pr_file_p.unread (c);
			m = false;
		}
		pr_source_line++;
		pr_mark_file_p.mark (1024);
		if (m)
			pr_file_p.unread (c);
	}

/*
==============
PR_LexString

Parses a quoted string
==============
*/
	void PR_LexString () throws IOException, ParseException {
		System.out.println ("PR_LexString ()");

		char c;
		pr_token = "";
	
		pr_file_p.read ();
		do {
			try {
				c = pr_file_p_read ();
			} catch (IOException e) {
				throw new ParseException ("EOF inside quote");
			}

			if (c == '\n')
				throw new ParseException ("newline inside quote");

			if (c == '\\') {	// escape char

				try {
					c = pr_file_p_read ();
				} catch (IOException e) {
					throw new ParseException ("EOF inside quote");
				}

				if (c == 'n')
					c = '\n';
				else if (c == '"')
					c = '"';
				else
					throw new ParseException ("Unknown escape char");
			} else if (c == '\"') {
				pr_token_type = tt_immediate;
				pr_immediate_type = type_string;
				pr_immediate_string = pr_token;
				return;
			}
			pr_token += c;
		} while (true);
	}

/*
==============
PR_LexNumber
==============
*/
	float PR_LexNumber () throws IOException {
		System.out.println ("PR_LexNumber ()");

		pr_token = "";
		char c = pr_file_p_read ();	
		do {
			pr_token += c;
			c = pr_file_p_read ();
		} while ((c >= '0' && c <= '9') || c == '.');
		pr_file_p.unread (c);

		return Float.valueOf (pr_token).floatValue ();
	}

/*
==============
PR_LexVector

Parses a single quoted vector
==============
*/
	void PR_LexVector () throws IOException, ParseException {
		System.out.println ("PR_LexVector ()");

		pr_file_p.read ();
		pr_token_type = tt_immediate;
		pr_immediate_type = type_vector;

		for (int i = 0; i < 3 ;i++) {
			pr_immediate.vector[i] = PR_LexNumber ();
			PR_LexWhitespace ();
		}

		char c = pr_file_p_read ();		
		if (c != '\'')
			throw new ParseException ("Bad vector");
	}


/*
==============
PR_LexName

Parses an identifier
==============
*/
	void PR_LexName () throws IOException {
		System.out.println ("PR_LexName ()");

		char c = pr_file_p_read ();
		pr_token = "";	
		do {
			pr_token += c;
			c = pr_file_p_read ();
		} while ((c >= 'a' && c <= 'z') || 
			 (c >= 'A' && c <= 'Z') || 
			 (c == '_') || 
			 (c >= '0' && c <= '9'));
		pr_file_p.unread (c);
		pr_token_type = tt_name;
	}

/*
==============
PR_LexPunctuation
==============
*/
	void PR_LexPunctuation () throws IOException, ParseException {
		System.out.println ("PR_LexPunctuation ()");

		String p;
	 	pr_token_type = tt_punct;	
		for (int i = 0; (p = pr_punctuation[i]) != null; i++) {
			int len = p.length ();
			pr_token = "";
			while (len > 0) {
				pr_token += pr_file_p_read ();
				len--;
			}
			if (pr_token.equals (p)) {
				if (p.charAt (0) == '{')
					pr_bracelevel++;
				else if (p.charAt (0) == '}')
					pr_bracelevel--;
				return;
			} else {
				while (len < p.length ()) {
					pr_file_p.unread (pr_token.charAt (len++));
				}
			}
		}
		throw new ParseException ("Unknown punctuation");
	}
		
/*
==============
PR_LexWhitespace
==============
*/
	void PR_LexWhitespace () throws IOException {
		System.out.println ("PR_LexWhitespace ()");

		char c;
	 	while (true) {
// skip whitespace
			while ((c = pr_file_p_read ()) <= ' ') {
				if (c == '\n') {
					pr_file_p.unread (c);
					PR_NewLine ();
					pr_file_p.read ();
				}
			}

//		System.out.println ("[mid]real character \""+c+"\"");
//		if (c <= ' ') System.out.println ("problem");

		
// skip // comments
			if (c == '/') {
				c = pr_file_p_read ();
				if (c == '/') {
					while ((c = pr_file_p_read ()) != '\n');
					pr_file_p.unread (c);
					PR_NewLine();
					pr_file_p.read ();
					continue;
				}
		
// skip /* */ comments
				else if (c == '*') {
					char d;
					c = ' ';
					do {
						d = c;
						c = pr_file_p_read ();
						if (c == '\n') {
							pr_file_p.unread (c);
							PR_NewLine();
							pr_file_p.read ();
						}
					} while (d != '*' || c != '/');
					continue;
				}
				else 
					pr_file_p.unread (c);
			} else
				pr_file_p.unread (c);
			break;		// a real character has been found
		}

//		c = pr_file_p_read ();
//		pr_file_p.unread (c);
//		System.out.println ("[end]real character \""+c+"\"");
//		if (c <= ' ') System.out.println ("problem");
	}

	void PR_ClearGrabMacros () {
		System.out.println ("PR_ClearGrabMacros ()");

		pr_nummacros = 0;
	}

	void PR_FindMacro () throws ParseException {
		System.out.println ("PR_FindMacro ()");

		for (int i = 0 ; i<pr_nummacros ; i++)
			if (pr_token.equals (pr_framemacros[i])) {
				pr_token = "" + i;
				pr_token_type = tt_immediate;
				pr_immediate_type = type_float;
				pr_immediate._float = i;
				return;
			}

		throw new ParseException ("Unknown frame macro $" + pr_token);
	}

// just parses text, returning false if an eol is reached
	boolean PR_SimpleGetToken () throws IOException {
		System.out.println ("PR_SimpleGetToken ()");

		char c;
	
// skip whitespace
		while ((c = pr_file_p_read ()) <= ' ') {
			if (c == '\n') {
				pr_file_p.unread (c);
				return false;
			}
		}

		pr_token = "";	
		while ((c = pr_file_p_read ()) > ' ' && c != ',' && c != ';') {
			pr_token += c;
		}
		pr_file_p.unread (c);
		return true;
	}

	void PR_ParseFrame () throws IOException {
		System.out.println ("PR_ParseFrame ()");

		while (PR_SimpleGetToken ()) {
			pr_framemacros[pr_nummacros] = pr_token;
			pr_nummacros++;
		}
	}

/*
==============
PR_LexGrab

Deals with counting sequence numbers and replacing frame macros
==============
*/
	void PR_LexGrab () throws IOException, ParseException {
		System.out.println ("PR_LexGrab ()");

		pr_file_p.read ();	// skip the $
		
		if (!PR_SimpleGetToken ())
			throw new ParseException ("hanging $");
	
// check for $frame
		if (pr_token.equals ("frame")) {
			PR_ParseFrame ();
			PR_Lex ();
		}
// ignore other known $commands
		else if (pr_token.equals ("cd") || 
			 pr_token.equals ("origin") || 
			 pr_token.equals ("base") || 
			 pr_token.equals ("flags") || 
			 pr_token.equals ("scale") || 
			 pr_token.equals ("skin"))
		{	// skip to end of line
			while (PR_SimpleGetToken ());
			PR_Lex ();
		}
// look for a frame name macro
		else
			PR_FindMacro ();
	}

/*
==============
PR_Lex

Sets pr_token, pr_token_type, and possibly pr_immediate and pr_immediate_type
==============
*/
	void PR_Lex () throws IOException, ParseException {
		System.out.println ("PR_Lex ()");

		pr_token = "";
		PR_LexWhitespace ();
	
		char c;
		try {
			c = pr_file_p_read ();
		} catch (IOException e) {
			pr_token_type = tt_eof;
			return;
		}
		pr_file_p.unread (c);

// handle quoted strings as a unit
		if (c == '\"') {
			PR_LexString ();
			return;
		}

// handle quoted vectors as a unit
		if (c == '\'') {
			PR_LexVector ();
			return;
		}

		if ((c >= 'a' && c <= 'z') || 
		    (c >= 'A' && c <= 'Z') || 
		     c == '_')
		{
			PR_LexName ();
			return;
		}
	
		if (c == '$') {
			PR_LexGrab ();
			return;
		}

// if the first character is a valid identifier, parse until a non-id
// character is reached
		if (c >= '0' && c <= '9') {
			pr_token_type = tt_immediate;
			pr_immediate_type = type_float;
			pr_immediate._float = PR_LexNumber ();
			return;
		}

		if (c == '-') {
			char d = pr_file_p_read ();
			pr_file_p.unread (d);
			if (d >= '0' && d <= '9') {
				pr_token_type = tt_immediate;
				pr_immediate_type = type_float;
				pr_immediate._float = PR_LexNumber ();
				return;
			}
		}
	
// parse symbol strings until a non-symbol is found
		PR_LexPunctuation ();
	}

/*
=============
PR_Expect

Issues an error if the current token isn't equal to string
Gets the next token
=============
*/
	void PR_Expect (String string) throws IOException, ParseException {
		System.out.println ("PR_Check (" + string + ")");

		if (! string.equals (pr_token))
			throw new ParseException ("expected " + string + ", found " + pr_token);
		PR_Lex ();
	}

/*
=============
PR_Check

Returns true and gets the next token if the current token equals string
Returns false and does nothing otherwise
=============
*/
	boolean PR_Check (String string) throws IOException, ParseException {
		System.out.println ("PR_Check (" + string + ")");

		if (! string.equals (pr_token))
			return false;
		
		PR_Lex ();
		return true;
	}

/*
============
PR_ParseName

Checks to see if the current token is a valid name
============
*/
	String PR_ParseName () throws IOException, ParseException {
		System.out.println ("PR_ParseName ()");

		String ident;
	
		if (pr_token_type != tt_name)
			throw new ParseException ("not a name");
		if (pr_token.length () >= MAX_NAME-1)
			throw new ParseException ("name too long");
		ident = new String (pr_token);
		PR_Lex ();
	
		return ident;
	}

/*
============
PR_FindType

Returns a preexisting complex type that matches the parm, or allocates
a new one and copies it out.
============
*/
	type_t PR_FindType (type_t type) {
		System.out.println ("PR_FindType ()");

		def_t def;
		type_t check;
		int i;
	
		for (check = pr.types; check != null; check = check.next) {
			if (check.type != type.type ||
			    check.aux_type != type.aux_type ||
			    check.num_parms != type.num_parms)
				continue;
	
			for (i = 0; i < type.num_parms; i++)
				if (check.parm_types[i] != type.parm_types[i])
					break;
			
			if (i == type.num_parms)
				return check;	
		}
	
// allocate a new one
		check = new type_t (type);
		check.next = pr.types;
		pr.types = check;
	
// allocate a generic def for the type, so fields can reference it
		def = new def_t ();
		def.name = "COMPLEX TYPE";
		def.type = check;
		check.def = def;
		return check;
	}

/*
============
PR_SkipToSemicolon

For error recovery, also pops out of nested braces
============
*/
	void PR_SkipToSemicolon () throws IOException, ParseException {
		System.out.println ("PR_SkipToSemicolon ()");

		do {
			if (pr_bracelevel != 0 && PR_Check (";"))
				return;
			PR_Lex ();
		} while (pr_token.charAt (0) != '\0');	// eof will return a null token
	}

/*
============
PR_ParseType

Parses a variable type, including field and functions types
============
*/
	type_t PR_ParseType () throws IOException, ParseException {
		System.out.println ("PR_ParseType ()");

		type_t newby;
		type_t type;
		String name;
	
		if (PR_Check (".")) {
			newby = new type_t ();
			newby.type = ev_field;
			newby.aux_type = PR_ParseType ();
			return PR_FindType (newby);
		}
	
		if (pr_token.equals ("float"))
			type = type_float;
		else if (pr_token.equals ("vector"))
			type = type_vector;
		else if (pr_token.equals ("float"))
			type = type_float;
		else if (pr_token.equals ("entity"))
			type = type_entity;
		else if (pr_token.equals ("string"))
			type = type_string;
		else if (pr_token.equals ("void"))
			type = type_void;
		else {
			throw new ParseException ("\"" + pr_token + "\" is not a type");
//			type = type_float;	// shut up compiler warning
		}
		PR_Lex ();
	
		if (!PR_Check ("("))
			return type;
	
// function type
		newby = new type_t ();
		newby.type = ev_function;
		newby.aux_type = type;	// return type
		newby.num_parms = 0;
		if (!PR_Check (")")) {
			if (PR_Check ("..."))
				newby.num_parms = -1;	// variable args
			else
				do {
					type = PR_ParseType ();
					name = PR_ParseName ();
					pr_parm_names[newby.num_parms] = name;
					newby.parm_types[newby.num_parms] = type;
					newby.num_parms++;
				} while (PR_Check (","));
	
			PR_Expect (")");
		}
	
		return PR_FindType (newby);
	}

// -- fini --
}

/* end of file */
